#pragma once

namespace Mix{

	template<typename T>
	class Queue
	{
	private:
		T* m_pBuffer;
		UInt32 m_Capacity;
		UInt32 m_ResizeStep;
		UInt32 m_HeadIndex;
		UInt32 m_Count;

#ifdef _DEBUG
		Mix::StringW m_DebugName;
#endif //_DEBUG

	public:
		Queue( void ) :
		m_pBuffer( NULL ),
		m_Capacity( 0 ),
		m_ResizeStep( 0 ),
		m_HeadIndex( 0 ),
		m_Count( 0 )
		{
#ifdef _DEBUG
			m_DebugName = L"";
#endif //_DEBUG
		}

		Queue( UInt32 defCapacity, UInt32 resizeStep, const wchar_t* pDebugName = NULL ) :
		m_pBuffer( NULL ),
		m_Capacity( 0 ),
		m_ResizeStep( 0 ),
		m_HeadIndex( 0 ),
		m_Count( 0 )
		{
#ifdef _DEBUG
			m_DebugName = L"";
#endif //_DEBUG

			Initialize( defCapacity, resizeStep, pDebugName );
		}

		~Queue( void )
		{
#ifdef _DEBUG
			MIX_LOG_DEV_INFO( L"Mix::Queue :  : Name[%s] Capacity[%d]", m_DebugName.GetConstPtr(), m_Capacity );
#endif //_DEBUG

			MIX_DELETE_ARRAY( m_pBuffer );
		}

		void Initialize( UInt32 defCapacity, UInt32 resizeStep, const wchar_t* pDebugName = NULL )
		{
#ifdef _DEBUG
			m_DebugName = MIX_SAFE_NAME( pDebugName );

			MIX_ASSERT_EX( defCapacity >= 2, L"Mix::Queue : LpVeBɂQȏw肷Kv܂ : Name[%s]", m_DebugName.GetConstPtr() );
			MIX_ASSERT_EX( m_pBuffer == NULL, L"Mix::Queue : łɏĂ܂ : Name[%s]", m_DebugName.GetConstPtr() );
#endif //_DEBUG

			m_Capacity = defCapacity;
			m_ResizeStep = resizeStep;

			m_pBuffer = new T[m_Capacity];
			MIX_ASSERT_EX( m_pBuffer != NULL, L"Mix::Queue : VXesĂ܂ : Name[%s]", m_DebugName.GetConstPtr() );

#ifdef _DEBUG

			MIX_LOG_INFO( L"Mix::Queue :  : Name[%s] Capacity[%d] ResizeStep[%d]",
				m_DebugName.GetConstPtr(),
				m_Capacity,
				m_ResizeStep );
#endif //_DEBUG
		}

		void Enqueue( const T& data )
		{
			MIX_ASSERT_EX( m_Capacity > 0, L"Mix::Queue : Ă܂ : Name[%s]", m_DebugName.GetConstPtr() );

			if( m_Count == m_Capacity )
			{
				MIX_ASSERT_EX( m_ResizeStep > 0, L"Mix::Queue : TCYXebvw肳ĂȂ߁ALpVeB𒴂܂ : Name[%s]", m_DebugName.GetConstPtr() );

				UInt32 nextCapacity = m_Capacity + m_ResizeStep;
				T* pNextBuffer = new T[nextCapacity];

				MIX_ASSERT_EX( pNextBuffer != NULL, L"Mix::Queue : VXesĂ܂ : Name[%s]", m_DebugName.GetConstPtr() );

				UInt32 srcIndex = m_HeadIndex;
				UInt32 dstIndex = m_HeadIndex;

				for( UInt32 i = 0; i < m_Count; i++ )
				{
					pNextBuffer[dstIndex] = m_pBuffer[srcIndex];

					srcIndex = ( srcIndex + 1 ) % m_Capacity;
					dstIndex = ( dstIndex + 1 ) % nextCapacity;
				}

				MIX_DELETE_ARRAY( m_pBuffer );
				m_pBuffer = pNextBuffer;
				m_Capacity = nextCapacity;

#ifdef _DEBUG
				MIX_LOG_INFO( L"Mix::Queue : TCY : Name[%s] Capacity[%d]",
					m_DebugName.GetConstPtr(),
					m_Capacity );
#endif //_DEBUG
			}

			m_pBuffer[( m_HeadIndex + m_Count ) % m_Capacity] = data;
			m_Count++;
		}

		const T& Dequeue( void )
		{
			MIX_ASSERT_EX( m_Capacity > 0, L"Mix::Queue : Ă܂ : Name[%s]", m_DebugName.GetConstPtr() );
			MIX_ASSERT_EX( m_Count > 0, L"Mix::Queue : L[̏ԂŃf[^oƂ܂(2) : Name[%s]", m_DebugName.GetConstPtr() );

			UInt32 preHeadIndex = m_HeadIndex;

			m_HeadIndex = ( m_HeadIndex + 1 ) % m_Capacity;
			m_Count--;

			return m_pBuffer[preHeadIndex];
		}

		void Drain( void )
		{
			m_Count = 0;
		}

		const T& GetHead( void ) const
		{
			MIX_ASSERT_EX( m_Capacity > 0, L"Mix::Queue : Ă܂ : Name[%s]", m_DebugName.GetConstPtr() );
			MIX_ASSERT( m_Count > 0 );

			return m_pBuffer[m_HeadIndex];
		}

		const T& GetTail( void ) const
		{
			MIX_ASSERT_EX( m_Capacity > 0, L"Mix::Queue : Ă܂ : Name[%s]", m_DebugName.GetConstPtr() );
			MIX_ASSERT( m_Count > 0 );

			return m_pBuffer[( m_HeadIndex + m_Count - 1 ) % m_Capacity];
		}

		UInt32 GetHeadIndex( void ) const
		{
			MIX_ASSERT_EX( m_Capacity > 0, L"Mix::Queue : Ă܂ : Name[%s]", m_DebugName.GetConstPtr() );
			MIX_ASSERT( m_Count > 0 );

			return m_HeadIndex;
		}

		UInt32 GetTailIndex( void ) const
		{
			MIX_ASSERT_EX( m_Capacity > 0, L"Mix::Queue : Ă܂ : Name[%s]", m_DebugName.GetConstPtr() );
			MIX_ASSERT( m_Count > 0 );

			return ( m_HeadIndex + m_Count - 1 ) % m_Capacity;
		}

		UInt32 GetCount( void ) const
		{
			return m_Count;
		}

		Boolean IsEmpty( void ) const
		{
			return ( m_Count == 0 );
		}

		T& operator [] ( UInt32 index )
		{
			MIX_ASSERT_EX( m_Capacity > 0, L"Mix::Queue : Ă܂ : Name[%s]", m_DebugName.GetConstPtr() );
			MIX_ASSERT( m_Capacity > index );

			return m_pBuffer[index];
		}
	};

}
