#include "Mix/Class/Network/TcpServer.h"
#include "Mix/Class/Network/Manager.h"
#include "Mix/Class/Network/Session.h"
#include "Mix/Class/Network/PacketTypes.h"
#include "Mix/Class/Network/PacketSender.h"
#include "Mix/Class/Network/PacketReceiver.h"
#include "Mix/Network/ITcpServerCallback.h"

namespace Mix{ namespace Network{

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// TcpServer::EventQueue
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

TcpServer::EventQueue::EventQueue( void ) :
m_SemaEmpty( EVENTQUEUE_MAX, EVENTQUEUE_MAX ),
m_SemaCount( 0, EVENTQUEUE_MAX ),
m_ReadPos( 0 ),
m_WritePos( 0 )
{
	m_Empty.type = EVENTTYPE_NONE;
	m_Empty.conn.pSession = NULL;
	m_Empty.conn.pPacketSender = NULL;
	m_Empty.conn.pPacketReceiver = NULL;
	m_Empty.err.pSession = NULL;
	m_Empty.err.ret = Mix::Network::OK;
}

void TcpServer::EventQueue::Push( const EVENTARG& arg )
{
	m_SemaEmpty.Reduce();
	m_Sync.Enter();

	m_Events[m_WritePos] = arg;
	m_WritePos = ( m_WritePos + 1 ) % EVENTQUEUE_MAX;

	m_Sync.Leave();
	m_SemaCount.Increase();
}

TcpServer::EVENTARG& TcpServer::EventQueue::Pop( void )
{
	EVENTARG* arg;

	if( m_SemaCount.Reduce( 0 ) == True )
	{
		m_Sync.Enter();

		arg = &( m_Events[m_ReadPos] );
		m_ReadPos = ( m_ReadPos + 1 ) % EVENTQUEUE_MAX;

		m_Sync.Leave();
		m_SemaEmpty.Increase();
	}
	else
	{
		arg = &m_Empty;
	}

	return *arg;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Host
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

TcpServer* TcpServer::CreateInstance( Mix::Network::Manager* pMgr, const wchar_t* name )
{
	return new TcpServer( pMgr, name );
}

TcpServer::TcpServer( Mix::Network::Manager* pMgr, const wchar_t* name ) :
m_pMgr( pMgr ),
m_Port( 0 ),
m_MaxConnection( 0 ),
m_pCallback( NULL ),
m_EventDestroy( True, False ),
m_bInitialize( False ),
m_Name( name )
{
	MIX_ADD_REF( m_pMgr );
}

TcpServer::~TcpServer( void )
{
	if( ( m_bInitialize == True ) &&
		( m_EventDestroy.Wait( 0 ) == False ) )
	{
		MIX_LOG_WARNING( L"TCPT[o[ : O Destroy ĂяoƂ𐄏܂ : Name[%s]", GetName() );
		Destroy();
	}

	if( m_pMgr != NULL )
	{
		m_pMgr->Remove( m_Port, Manager::PORTARGS::TCP_SERVER );
		MIX_RELEASE( m_pMgr );
	}
}

Boolean TcpServer::Initialize( const Mix::Network::TCP_SERVERCONFIG& cfg )
{
	UInt32 i;
	UInt32 j;

	m_Port = cfg.port;
	m_MaxConnection = cfg.maxConnection;
	m_pCallback = cfg.pCallback;
	m_SleepTime = cfg.sleepTime;
	m_EchoInterval = cfg.echoInterval;
	m_TimeOut = cfg.timeOut;
	m_SendBuffSize = cfg.sendBuffSize;
	m_ReceiveBuffSize = cfg.receiveBuffSize;
	m_ReceiveTempBuffSize = cfg.receiveTempBuffSize;

	//bXCxg쐬
	for( i = 0; i < LISTENEVENT_MAX; i++ )
	{
		m_hListenEventArray[i] = ::WSACreateEvent();
		if( m_hListenEventArray[i] == NULL )
		{
			for( j = 0; j < i; j++ )
			{
				::WSACloseEvent( m_hListenEventArray[j] );
			}

			MIX_LOG_ERROR( L"TCPT[o[̏Ɏs : Cxg쐬ł܂" );

			return False;
		}
	}

	//XbhJn
	if( m_Thread.Start( TcpServer::ThreadEntry, this ) == False )
	{
		for( i = 0; i < LISTENEVENT_MAX; i++ )
		{
			::WSACloseEvent( m_hListenEventArray[i] );
		}

		MIX_LOG_ERROR( L"TCPT[o[̏Ɏs : XbhJnł܂" );

		return False;
	}

	//ς
	m_bInitialize = True;

	return True;
}

const wchar_t* TcpServer::GetName( void ) const
{
	return m_Name.GetConstPtr();
}

void TcpServer::Poll( void )
{
	DoEvent();
}

void TcpServer::Destroy( void )
{
	UInt32 i;

	//jmF
	if( m_EventDestroy.Wait( 0 ) == True )
	{
		return;
	}

	//Iʒm
	if( m_hListenEventArray[LISTENEVENT_EXIT] != NULL )
	{
		::WSASetEvent( m_hListenEventArray[LISTENEVENT_EXIT] );
	}

	//j܂Ń|[O
	while( m_EventDestroy.Wait( 0 ) == False )
	{
		DoEvent();
		::Sleep( 0 );
	}

	//Xbh̏I҂
	m_Thread.Join();

	//c̃Cxg
	while( DoEvent() == True )
	{
		::Sleep( 0 );
	}

	//bXCxg
	for( i = 0; i < LISTENEVENT_MAX; i++ )
	{
		if( m_hListenEventArray[i] != NULL )
		{
			::WSACloseEvent( m_hListenEventArray[i] );
			m_hListenEventArray[i] = NULL;
		}
	}
}

Boolean TcpServer::IsDestroy( void ) const
{
	return m_EventDestroy.Wait( 0 );
}

UInt32 TcpServer::SendAll( const void* pData, UInt32 dataSize )
{
	PollObjectList::iterator it = m_PollObjectList.begin();
	PollObjectList::iterator it_end = m_PollObjectList.end();
	UInt32 ret = 0;

	while( it != it_end )
	{
		if( it->pPacketSender->Write( pData, dataSize ) == True )
		{
			ret++;
		}
		it++;
	}

	return ret;
}

UInt32 TcpServer::CloseAll( void )
{
	PollObjectList::iterator it = m_PollObjectList.begin();
	PollObjectList::iterator it_end = m_PollObjectList.end();

	while( it != it_end )
	{
		it->pSession->Close();
		it++;
	}

	return static_cast<UInt32>( m_PollObjectList.size() );
}

UInt32 TcpServer::GetCount( void ) const
{
	return static_cast<UInt32>( m_PollObjectList.size() );
}

Boolean TcpServer::DoEvent( void )
{
	EVENTARG eventArg;
	UInt32 packetType;
	void* pData;
	UInt32 dataSize;
	UInt32 nowTime;
	UInt32 elapseTime;
	PollObjectList::iterator it;
	POLLOBJECT* pPollObj;

	//Cxg擾
	eventArg = m_EventQue.Pop();

	//Cxg
	switch( eventArg.type )
	{
	//ڑCxg
	case TcpServer::EVENTTYPE_CONNECT:
		DoEvent_Connect( eventArg );
		break;

	//ؒfCxg
	case TcpServer::EVENTTYPE_DISCONNECT:
		DoEvent_Disconnect( eventArg );
		break;

	//G[Cxg
	case TcpServer::EVENTTYPE_ERROR:
		DoEvent_Error( eventArg );
		break;
	}

	//݂̎Ԃ擾
	nowTime = ::GetTickCount();

	//|[IuWFNg̏
	for( it = m_PollObjectList.begin(); it != m_PollObjectList.end(); ++it )
	{
		pPollObj = &( *it );


		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		// M
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

		pPollObj->pPacketReceiver->BeginRead();
		while( pPollObj->pPacketReceiver->Read( packetType, &pData, dataSize ) == True )
		{
			switch( packetType )
			{
			//GR[
			case Mix::Network::PKTYPE_ECHO:
				pPollObj->bRecvEcho = True;
				pPollObj->echoTime = nowTime;
				break;

			//f[^
			case Mix::Network::PKTYPE_DATA:
				m_pCallback->OnRecive( pPollObj->pSession, pData, dataSize );
				break;
			}
		}
		pPollObj->pPacketReceiver->EndRead();


		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		// GR[
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

		elapseTime = ( pPollObj->echoTime > nowTime )? ( ( 0xFFFFFFFF - pPollObj->echoTime ) + nowTime ) : ( nowTime - pPollObj->echoTime );
		if( pPollObj->bRecvEcho == True )
		{
			if( elapseTime >= m_EchoInterval )
			{
				pPollObj->pPacketSender->WriteEcho();
				pPollObj->bRecvEcho = False;
			}
		}
		else
		{
			if( elapseTime >= m_TimeOut )
			{
				m_pCallback->OnError( pPollObj->pSession, Mix::Network::ERR_TIMEOUT );
				pPollObj->pSession->Close();
			}
		}
	}

	return ( eventArg.type != TcpServer::EVENTTYPE_NONE );
}

void TcpServer::DoEvent_Connect( EVENTARG& eventArg )
{
	TcpServer::POLLOBJECT pollObj;

	//Xgɒǉ
	pollObj.bRecvEcho = True;
	pollObj.echoTime = ::GetTickCount();
	pollObj.pSession = eventArg.conn.pSession;
	pollObj.pPacketSender = eventArg.conn.pPacketSender;
	pollObj.pPacketReceiver = eventArg.conn.pPacketReceiver;
	m_PollObjectList.push_back( pollObj );

	//ڑʒm
	m_pCallback->OnOpenSession( eventArg.conn.pSession, eventArg.conn.pPacketSender );
}

void TcpServer::DoEvent_Disconnect( EVENTARG& eventArg )
{
	PollObjectList::iterator it;
	UInt32 packetType;
	UInt32 dataSize;
	void* pData;

	//|[IuWFNg擾
	it = FindPollObject( eventArg.disconn.pSession );
	if( it == m_PollObjectList.end() )
	{
		return;
	}

	//c̎Mʒm
	it->pPacketReceiver->BeginRead();
	while( it->pPacketReceiver->Read( packetType, &pData, dataSize ) == True )
	{
		if( packetType == Mix::Network::PKTYPE_DATA )
		{
			m_pCallback->OnRecive( ( *it ).pSession, pData, dataSize );
		}
	}
	it->pPacketReceiver->EndRead();

	//ؒfʒm
	m_pCallback->OnCloseSession( it->pSession );

	//
	MIX_RELEASE( it->pSession );
	MIX_RELEASE( it->pPacketSender );
	MIX_RELEASE( it->pPacketReceiver );
	MIX_RELEASE( eventArg.disconn.pSession );

	//Xg폜
	m_PollObjectList.erase( it );
}

void TcpServer::DoEvent_Error( EVENTARG& eventArg )
{
	if( eventArg.err.pSession == NULL )
	{
		//zXg̃G[
		m_pCallback->OnError( eventArg.err.ret );
	}
	else
	{
		//ZbṼG[
		m_pCallback->OnError( eventArg.err.pSession, eventArg.err.ret );
		MIX_RELEASE( eventArg.err.pSession );
	}
}

TcpServer::PollObjectList::iterator TcpServer::FindPollObject( Mix::Network::Session* pSession )
{
	PollObjectList::iterator it;

	for( it = m_PollObjectList.begin(); it != m_PollObjectList.end(); ++it )
	{
		if( ( *it ).pSession == pSession )
		{
			return it;
		}
	}

	return m_PollObjectList.end();
}

void TcpServer::ThreadMain( void )
{
	Boolean bRunning = True;
	UInt32 ret;
	UInt32 eventIndex;
	ConnectionList::iterator it;
	CONNECTION* pConn;
	Int8* recvBuff;
	UInt32 recvBuffSize;
	Int32 recvSize;
	Int8* sendBuff;
	UInt32 sendBuffSize;
	Int32 sendSize;
	UInt32 nowTime;
	UInt32 elapseTime;
	TcpServer::EVENTARG eventArg;
	WSANETWORKEVENTS events;

	if( ThreadMain_Initialize() == True )
	{
		while( bRunning == True )
		{
			/////////////////////////////////////////////////////////////////////////////////////////////////////////////
			// Cxg
			/////////////////////////////////////////////////////////////////////////////////////////////////////////////

			ret = ::WSAWaitForMultipleEvents( TcpServer::LISTENEVENT_MAX, m_hListenEventArray, FALSE, 0, FALSE );
			if( ret == WSA_WAIT_FAILED )
			{
				//G[ : bXCxg̑ҋ@Ɏs
				bRunning = False;
				ThreadMain_Error( NULL, Mix::Network::ERR_WAITEVENT );
			}
			else if( ( eventIndex = ( ret - WSA_WAIT_EVENT_0 ) ) < TcpServer::LISTENEVENT_MAX )
			{
				switch( eventIndex )
				{
				//Cxg
				case TcpServer::LISTENEVENT_ACCEPT:
					ThreadMain_NewConnection();
					break;

				//ICxg
				case TcpServer::LISTENEVENT_EXIT:
					bRunning = False;
					break;
				}

				::WSAResetEvent( m_hListenEventArray[eventIndex] );
			}


			/////////////////////////////////////////////////////////////////////////////////////////////////////////////
			// ZbV̑M
			/////////////////////////////////////////////////////////////////////////////////////////////////////////////

			it = m_ConnectionList.begin();
			while( it != m_ConnectionList.end() )
			{
				pConn = &( *it );

				/////////////////////////////////////////////////////////////////////////////////////////////////////////////
				// ZbV̏
				/////////////////////////////////////////////////////////////////////////////////////////////////////////////

				if( pConn->pSession->IsClose() == True )
				{
					MIX_SETBIT( pConn->flags, TcpServer::CF_CLOSE );
				}


				/////////////////////////////////////////////////////////////////////////////////////////////////////////////
				// Cxg
				/////////////////////////////////////////////////////////////////////////////////////////////////////////////

				ret = ::WSAWaitForMultipleEvents( 1, &( pConn->sockEvent ), FALSE, 0, FALSE );
				if( ret == WSA_WAIT_FAILED )
				{
					//G[ : ZbṼCxgҋ@Ɏs
					MIX_SETBIT( pConn->flags, TcpServer::CF_CLOSE );
					ThreadMain_Error( pConn->pSession, Mix::Network::ERR_WAITEVENT );
				}
				else
				{
					if( ::WSAEnumNetworkEvents( pConn->sock, pConn->sockEvent, &events ) != SOCKET_ERROR )
					{
						if( ( events.lNetworkEvents & FD_CLOSE ) != 0 )
						{
							//N[YCxg
							MIX_SETBIT( pConn->flags, TcpServer::CF_CLOSE );
						}
						else if( ( events.lNetworkEvents & FD_READ ) != 0 )
						{
							//[hCxg
							MIX_SETBIT( pConn->flags, TcpServer::CF_RECV_READ );
						}

						::WSAResetEvent( pConn->sockEvent );
					}
					else
					{
						//G[ : ZbṼCxg^Cv̓Ɏs
						MIX_SETBIT( pConn->flags, TcpServer::CF_CLOSE );
						ThreadMain_Error( pConn->pSession, Mix::Network::ERR_ENUMEVENT );
					}
				}


				/////////////////////////////////////////////////////////////////////////////////////////////////////////////
				// M
				/////////////////////////////////////////////////////////////////////////////////////////////////////////////

				if( MIX_TESTBIT( pConn->flags, TcpServer::CF_RECV_READ ) == TcpServer::CF_RECV_READ )
				{
					recvBuff = pConn->pPacketReceiver->GetWrite( recvBuffSize );
					if( recvBuff != NULL )
					{
						MIX_RESETBIT( pConn->flags, TcpServer::CF_NOCAPACITY_TEMPBUFF );

						recvSize = ::recv( pConn->sock, recvBuff, recvBuffSize, 0 );
						if( recvSize != SOCKET_ERROR )
						{
							//񂾃TCYʒm
							pConn->pPacketReceiver->Write( recvSize );
							pConn->receiveBytesPerSec += recvSize;

							MIX_RESETBIT( pConn->flags, TcpServer::CF_RECV_READ );
							MIX_SETBIT( pConn->flags, TcpServer::CF_RECV_PARSE );
						}
						else
						{
							//G[ : recv Ɏs
							MIX_SETBIT( pConn->flags, TcpServer::CF_CLOSE );
							ThreadMain_Error( pConn->pSession, Mix::Network::ERR_RECIVE );
						}
					}
					else
					{
						//x : e|Mobt@̗eʂs
						if( MIX_TESTBIT( pConn->flags, TcpServer::CF_NOCAPACITY_TEMPBUFF ) != TcpServer::CF_NOCAPACITY_TEMPBUFF )
						{
							MIX_SETBIT( pConn->flags, TcpServer::CF_NOCAPACITY_TEMPBUFF );
							MIX_LOG_WARNING( L"TCPT[o[̃e|Mobt@̋󂫗eʂsĂ܂ : Mɒx : Name[%s]", GetName() );
						}

						//ꉞp[XtO𗧂ĂĂ
						MIX_SETBIT( pConn->flags, TcpServer::CF_RECV_PARSE );
					}
				}

				if( MIX_TESTBIT( pConn->flags, TcpServer::CF_RECV_PARSE ) == TcpServer::CF_RECV_PARSE )
				{
					switch( pConn->pPacketReceiver->Parse() )
					{
					//
					case PacketReceiver::OK:
						MIX_RESETBIT( pConn->flags, TcpServer::CF_NOCAPACITY_BUFF );
						MIX_RESETBIT( pConn->flags, TcpServer::CF_RECV_PARSE );
						break;

					//x : Mobt@̋󂫗eʂsĂ
					case PacketReceiver::WARNING_BUFF_CAPACITY:
						if( MIX_TESTBIT( pConn->flags, TcpServer::CF_NOCAPACITY_BUFF ) != TcpServer::CF_NOCAPACITY_BUFF )
						{
							MIX_SETBIT( pConn->flags, TcpServer::CF_NOCAPACITY_BUFF );
							MIX_LOG_WARNING( L"TCPT[o[̎Mobt@̋󂫗eʂsĂ܂ : Mɒx : Name[%s]", GetName() );
						}
						break;

					//G[ : pPbgjĂ
					case PacketReceiver::ERROR_PACKET_BLOKEN:
						MIX_SETBIT( pConn->flags, TcpServer::CF_CLOSE );
						ThreadMain_Error( pConn->pSession, Mix::Network::ERR_BLOKENPACKET );
						break;
					//G[ : pPbg̃TCY傫
					case PacketReceiver::ERROR_PACKET_LARGE:
						MIX_SETBIT( pConn->flags, TcpServer::CF_CLOSE );
						ThreadMain_Error( pConn->pSession, Mix::Network::ERR_LARGEPACKETSIZE );
						break;
					}
				}


				/////////////////////////////////////////////////////////////////////////////////////////////////////////////
				// M
				/////////////////////////////////////////////////////////////////////////////////////////////////////////////

				sendBuff = pConn->pPacketSender->BeginRead( sendBuffSize );
				if( sendBuff != NULL )
				{
					while( sendBuffSize > 0 )
					{
						sendSize = ::send( pConn->sock, sendBuff, sendBuffSize, 0 );

						if( sendSize != SOCKET_ERROR )
						{
							sendBuffSize -= sendSize;
							sendBuff += sendSize;
							pConn->sendBytesPerSec += sendSize;
						}
						else
						{
							//G[ : send Ɏs
							MIX_SETBIT( pConn->flags, TcpServer::CF_CLOSE );
							ThreadMain_Error( pConn->pSession, Mix::Network::ERR_SEND );
							break;
						}
					}

					pConn->pPacketSender->EndRead();
				}


				/////////////////////////////////////////////////////////////////////////////////////////////////////////////
				// v
				/////////////////////////////////////////////////////////////////////////////////////////////////////////////

				nowTime = ::GetTickCount();
				elapseTime = ( pConn->updateTime > nowTime )? ( ( 0xFFFFFFFF - pConn->updateTime ) + nowTime ) : ( nowTime - pConn->updateTime );
				if( elapseTime >= TcpServer::ONSEC )
				{
					pConn->pSession->SetPerSec( pConn->sendBytesPerSec, pConn->receiveBytesPerSec );
					pConn->sendBytesPerSec = 0;
					pConn->receiveBytesPerSec = 0;
					pConn->updateTime = nowTime;
				}


				/////////////////////////////////////////////////////////////////////////////////////////////////////////////
				// N[Y
				/////////////////////////////////////////////////////////////////////////////////////////////////////////////

				if( MIX_TESTBIT( pConn->flags, TcpServer::CF_CLOSE ) == TcpServer::CF_CLOSE )
				{
					ThreadMain_CloseConnection( it );
				}
				else
				{
					it++;
				}
			}

			::Sleep( m_SleepTime );
		}
	}

	ThreadMain_Finalize();
}

Boolean TcpServer::ThreadMain_Initialize( void )
{
	UInt32 i;
	Boolean bResumeAddrYes = True;
	struct sockaddr_in sockAddr;

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// CxgZbg
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	for( i = 0; i < LISTENEVENT_MAX; i++ )
	{
		::WSAResetEvent( m_hListenEventArray[i] );
	}


	//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// bX\Pbg쐬
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	m_ListenSocket = ::socket( AF_INET, SOCK_STREAM, 0 );
	if( m_ListenSocket == INVALID_SOCKET )
	{
		return False;
	}

	if( ::setsockopt( m_ListenSocket, SOL_SOCKET, SO_REUSEADDR, reinterpret_cast<const char*>( &bResumeAddrYes ), sizeof( bResumeAddrYes ) ) == SOCKET_ERROR )
	{
		return False;
	}

	sockAddr.sin_family = AF_INET;
	sockAddr.sin_port = ::htons( m_Port );
	sockAddr.sin_addr.S_un.S_addr = INADDR_ANY;
	if( ::bind( m_ListenSocket, reinterpret_cast<SOCKADDR*>( &( sockAddr ) ), sizeof( sockAddr ) ) != 0 )
	{
		return False;
	}

	if( ::WSAEventSelect( m_ListenSocket, m_hListenEventArray[LISTENEVENT_ACCEPT], FD_ACCEPT ) == SOCKET_ERROR )
	{
		return False;
	}

	if( ::listen( m_ListenSocket, SOMAXCONN ) == SOCKET_ERROR )
	{
		return False;
	}

	return True;
}

void TcpServer::ThreadMain_Finalize( void )
{
	ConnectionList::iterator it;

	//ׂẴZbVI
	it = m_ConnectionList.begin();
	while( it != m_ConnectionList.end() )
	{
		ThreadMain_CloseConnection( it );
	}

	//bX\Pbg
	if( m_ListenSocket != INVALID_SOCKET )
	{
		::shutdown( m_ListenSocket, SD_BOTH );
		::closesocket( m_ListenSocket );
		m_ListenSocket = INVALID_SOCKET;
	}

	//jʒm
	m_EventDestroy.Set();
}

void TcpServer::ThreadMain_NewConnection( void )
{
	TcpServer::CONNECTION conn;
	int sockAddrLen = sizeof( conn.sockAddr );
	TcpServer::EVENTARG eventArg;

	//tO
	conn.flags = 0;
	conn.sendBytesPerSec = 0;
	conn.receiveBytesPerSec = 0;

	//\Pbg쐬
	conn.sock = ::accept( m_ListenSocket, reinterpret_cast<SOCKADDR*>( &conn.sockAddr ), &sockAddrLen );
	if( conn.sock == INVALID_SOCKET )
	{
		return;
	}

	//\PbgCxg쐬
	conn.sockEvent = ::WSACreateEvent();
	if( conn.sockEvent == NULL )
	{
		::closesocket( conn.sock );
		return;
	}
	if( ::WSAEventSelect( conn.sock, conn.sockEvent, FD_CLOSE | FD_READ ) == SOCKET_ERROR )
	{
		::WSACloseEvent( conn.sockEvent );
		::closesocket( conn.sock );
		return;
	}

	//pPbgZ_[쐬
	conn.pPacketSender = Mix::Network::PacketSender::CreateInstance( m_SendBuffSize );
	if( ( conn.pPacketSender != NULL ) &&
		( conn.pPacketSender->Initialize() == True ) )
	{
		;
	}
	else
	{
		::WSACloseEvent( conn.sockEvent );
		::closesocket( conn.sock );
		return;
	}

	//pPbgV[o쐬
	conn.pPacketReceiver = Mix::Network::PacketReceiver::CreateInstance( m_ReceiveBuffSize, m_ReceiveTempBuffSize );
	if( ( conn.pPacketReceiver != NULL ) &&
		( conn.pPacketReceiver->Initialize() == True ) )
	{
		;
	}
	else
	{
		MIX_RELEASE( conn.pPacketSender );
		::WSACloseEvent( conn.sockEvent );
		::closesocket( conn.sock );
		return;
	}

	//ZbV쐬
	conn.pSession = Mix::Network::Session::CreateInstance( Mix::StringW( ::inet_ntoa( conn.sockAddr.sin_addr ) ).GetConstPtr(), conn.pPacketSender );
	if( conn.pSession == NULL )
	{
		MIX_RELEASE( conn.pPacketReceiver );
		MIX_RELEASE( conn.pPacketSender );
		::WSACloseEvent( conn.sockEvent );
		::closesocket( conn.sock );
		return;
	}

	//ɏ
	conn.updateTime = ::GetTickCount();

	//ڑXgɒǉ
	m_ConnectionList.push_back( conn );

	//ڑCxg
	eventArg.type = TcpServer::EVENTTYPE_CONNECT;
	eventArg.conn.pSession = conn.pSession;
	eventArg.conn.pSession->AddRef();
	eventArg.conn.pPacketSender = conn.pPacketSender;
	eventArg.conn.pPacketSender->AddRef();
	eventArg.conn.pPacketReceiver = conn.pPacketReceiver;
	eventArg.conn.pPacketReceiver->AddRef();
	m_EventQue.Push( eventArg );
}

void TcpServer::ThreadMain_CloseConnection( ConnectionList::iterator& it )
{
	CONNECTION* pConn = &( *it );
	Boolean bContinueRecv = True;
	Boolean bNoCapacityTempBuff = False;
	Boolean bNoCapacityBuff = False;
	PacketReceiver::RESULT recvRes;
	Int8* recvBuff;
	UInt32 recvBuffSize;
	Int32 recvSize;
	EVENTARG eventArg;

	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// cM
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	::shutdown( pConn->sock, SD_SEND );

	while( bContinueRecv == True )
	{
		recvRes = PacketReceiver::OK;

		recvBuff = pConn->pPacketReceiver->GetWrite( recvBuffSize );
		if( recvBuff != NULL )
		{
			bNoCapacityTempBuff = False;

			recvSize = ::recv( pConn->sock, recvBuff, recvBuffSize, 0 );
			if( recvSize > 0 )
			{
				pConn->pPacketReceiver->Write( recvSize );
				recvRes = pConn->pPacketReceiver->Parse();
			}
			else
			{
				//SĂ̎M
				bContinueRecv = False;
			}
		}
		else
		{
			//e|sĂ邽߁A͂A󂯂
			//ł󂩂Ȃꍇ́APoll sĂȂ

			if( bNoCapacityTempBuff == False )
			{
				bNoCapacityTempBuff = True;
				MIX_LOG_WARNING( L"TCPT[o[̎Mɒx : ZbṼN[YɃe|Mobt@̋󂫗eʂs : Name[%s]", GetName() );
			}

			recvRes = pConn->pPacketReceiver->Parse();
		}

		switch( recvRes )
		{
		//
		case PacketReceiver::OK:
			bNoCapacityBuff = False;
			break;

		//x : Mobt@̋󂫗eʂsĂ
		case PacketReceiver::WARNING_BUFF_CAPACITY:
			if( bNoCapacityBuff == False )
			{
				bNoCapacityBuff = True;
				MIX_LOG_WARNING( L"TCPT[o[̎Mɒx : ZbṼN[YɎMobt@̋󂫗eʂs : Name[%s]", GetName() );
			}
			break;

		//G[ : pPbgjĂ
		case PacketReceiver::ERROR_PACKET_BLOKEN:
			ThreadMain_Error( pConn->pSession, Mix::Network::ERR_BLOKENPACKET );
			bContinueRecv = False;
			break;
		//G[ : pPbg̃TCY傫
		case PacketReceiver::ERROR_PACKET_LARGE:
			ThreadMain_Error( pConn->pSession, Mix::Network::ERR_LARGEPACKETSIZE );
			bContinueRecv = False;
			break;
		}

		::Sleep( 0 );
	}


	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Cxg
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	eventArg.type = TcpServer::EVENTTYPE_DISCONNECT;
	eventArg.disconn.pSession = pConn->pSession;
	eventArg.disconn.pSession->AddRef();


	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	::shutdown( pConn->sock, SD_BOTH );
	::closesocket( pConn->sock );
	::WSACloseEvent( pConn->sockEvent );

	if( pConn->pSession != NULL )
	{
		pConn->pSession->SetPerSec( 0, 0 );
		pConn->pSession->ForceClose();
	}

	MIX_RELEASE( pConn->pPacketSender );
	MIX_RELEASE( pConn->pPacketReceiver );
	MIX_RELEASE( pConn->pSession );

	it = m_ConnectionList.erase( it );


	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Cxg𑗐M
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	m_EventQue.Push( eventArg );
}

void TcpServer::ThreadMain_Error( Mix::Network::Session* pSession, Mix::Network::RESULT ret )
{
	TcpServer::EVENTARG eventArg;

	eventArg.type = TcpServer::EVENTTYPE_ERROR;
	eventArg.err.ret = ret;
	eventArg.err.pSession = pSession;
	if( eventArg.err.pSession != NULL )
	{
		eventArg.err.pSession->AddRef();
	}

	m_EventQue.Push( eventArg );
}

void __cdecl TcpServer::ThreadEntry( void* pArg )
{
	TcpServer* pTcpServer = static_cast<TcpServer*>( pArg );
	pTcpServer->ThreadMain();
}

}}
