#include "Mix/Class/Network/TcpClient.h"
#include "Mix/Class/Network/Manager.h"
#include "Mix/Class/Network/PacketTypes.h"
#include "Mix/Class/Network/PacketSender.h"
#include "Mix/Class/Network/PacketReceiver.h"
#include "Mix/Network/ITcpClientCallback.h"
#include <list>

namespace Mix{ namespace Network{

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// TcpClient::EventQueue
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

TcpClient::EventQueue::EventQueue( void ) :
m_SemaEmpty( EVENTQUEUE_MAX, EVENTQUEUE_MAX ),
m_SemaCount( 0, EVENTQUEUE_MAX ),
m_ReadPos( 0 ),
m_WritePos( 0 )
{
}

void TcpClient::EventQueue::Push( UInt32 type, Int32 opt )
{
	m_SemaEmpty.Reduce();
	m_Sync.Enter();

	m_Events[m_WritePos].type = type;
	m_Events[m_WritePos].opt = opt;
	m_WritePos = ( m_WritePos + 1 ) % TcpClient::EVENTQUEUE_MAX;

	m_Sync.Leave();
	m_SemaCount.Increase();
}

UInt32 TcpClient::EventQueue::Poll( Int32* opt )
{
	UInt32 ret = EVENTTYPE_NONE;

	if( m_SemaCount.Reduce( 0 ) == True )
	{
		m_Sync.Enter();

		ret = m_Events[m_ReadPos].type;
		if( opt != NULL )
		{
			*opt = m_Events[m_ReadPos].opt;
		}

		m_ReadPos = ( m_ReadPos + 1 ) % TcpClient::EVENTQUEUE_MAX;

		m_Sync.Leave();
		m_SemaEmpty.Increase();
	}

	return ret;
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Client
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

TcpClient* TcpClient::CreateInstance( Mix::Network::Manager* pMgr, const wchar_t* name )
{
	return new TcpClient( pMgr, name );
}

TcpClient::TcpClient( Mix::Network::Manager* pMgr, const wchar_t* name ) :
m_pMgr( pMgr ),
m_Port( 0 ),
m_Addr( L"" ),
m_pCallback( NULL ),
m_pPacketSender( NULL ),
m_pPacketReceiver( NULL ),
m_bConnected( False ),
m_bInitialize( False ),
m_EchoTime( 0 ),
m_EventDestroy( True, False ),
m_Name( name )
{
	MIX_ADD_REF( m_pMgr );

	m_ShareData.sendBytesPerSec = 0;
	m_ShareData.receiveBytesPerSec = 0;
}

TcpClient::~TcpClient( void )
{
	if( ( m_bInitialize == True ) &&
		( m_EventDestroy.Wait( 0 ) == False ) )
	{
		MIX_LOG_WARNING( L"TCPNCAg : O Destroy ĂяoƂ𐄏܂ : Name[%s] : ", GetName() );
		Destroy();
	}

	if( m_pMgr != NULL )
	{
		m_pMgr->Remove( m_Port, Manager::PORTARGS::TCP_CLIENT );
		MIX_RELEASE( m_pMgr );
	}
}

Boolean TcpClient::Initialize( const Mix::Network::TCP_CLIENTCONFIG& cfg )
{
	m_Port = cfg.port;
	m_Addr = cfg.pServerIPAddr;
	m_pCallback = cfg.pCallback;
	m_bUseNagle = cfg.bUseNagle;
	m_SleepTime = cfg.sleepTime;
	m_TimeOut = cfg.timeOut;

	m_pPacketSender = Mix::Network::PacketSender::CreateInstance( cfg.sendBuffSize );
	if( ( m_pPacketSender != NULL ) &&
		( m_pPacketSender->Initialize() == True ) )
	{
		m_pPacketSender->Disable();
	}
	else
	{
		MIX_RELEASE( m_pPacketSender );
		MIX_LOG_ERROR( L"TCPNCAg̏Ɏs : VXes : Name[%s]", GetName() );
		return False;
	}

	m_pPacketReceiver = Mix::Network::PacketReceiver::CreateInstance( cfg.receiveBuffSize, cfg.receiveTempBuffSize );
	if( ( m_pPacketReceiver != NULL ) &&
		( m_pPacketReceiver->Initialize() == True ) )
	{
		;
	}
	else
	{
		MIX_RELEASE( m_pPacketSender );
		MIX_RELEASE( m_pPacketReceiver );
		MIX_LOG_ERROR( L"TCPNCAg̏Ɏs : VXes : Name[%s]", GetName() );
		return False;
	}

	if( m_Thread.Start( TcpClient::ThreadEntry, this ) == False )
	{
		MIX_RELEASE( m_pPacketSender );
		MIX_RELEASE( m_pPacketReceiver );
		MIX_LOG_ERROR( L"TCPNCAg̏Ɏs : XbhJnł܂ : Name[%s]", GetName() );
		return False;
	}

	m_bInitialize = True;

	return True;
}

const wchar_t* TcpClient::GetName( void ) const
{
	return m_Name.GetConstPtr();
}

void TcpClient::Poll( void )
{
	DoEvent();
}

void TcpClient::Destroy( void )
{
	//jmF
	if( m_EventDestroy.Wait( 0 ) == True )
	{
		MIX_LOG_WARNING( L"TCPNCAg̔jɎs : ɔjĂ܂ : Name[%s]", GetName() );
		return;
	}

	//M𖳌ɂ
	if( m_pPacketSender != NULL )
	{
		m_pPacketSender->Disable();
	}

	//ICxg𑗂
	m_EventToThread.Push( TcpClient::EVENTTYPE_EXIT );

	//jCxg܂őҋ@
	while( m_EventDestroy.Wait( 0 ) == False )
	{
		DoEvent();
		::Sleep( 0 );
	}

	//c̃Cxg
	while( DoEvent() == True )
	{
		::Sleep( 0 );
	}

	//Xbh̏I҂
	m_Thread.Join();

	//
	MIX_RELEASE( m_pPacketSender );
	MIX_RELEASE( m_pPacketReceiver );
}

Boolean TcpClient::IsDestroy( void ) const
{
	return m_EventDestroy.Wait( 0 );
}

void TcpClient::Connect( void )
{
	if( m_EventDestroy.Wait( 0 ) == True )
	{
		MIX_LOG_ERROR( L"TCPNCAg͐ڑɎs܂ : ɔjĂ܂ : Name[%s]", GetName() );
		return;
	}

	m_EventToThread.Push( EVENTTYPE_CONNECT );
}

void TcpClient::Disconnect( void )
{
	if( m_EventDestroy.Wait( 0 ) == True )
	{
		MIX_LOG_ERROR( L"TCPNCAg͐ؒfɎs܂ : ɔjĂ܂ ] : Name[%s]", GetName() );
		return;
	}

	m_EventToThread.Push( EVENTTYPE_DISCONNECT );
}

Boolean TcpClient::IsConnected( void ) const
{
	return m_bConnected;
}

Boolean TcpClient::Send( const void* pData, UInt32 dataSize )
{
	Boolean ret;

	if( ( m_EventDestroy.Wait( 0 ) == True ) ||
		( ( m_pPacketSender == NULL ) ) )
	{
		MIX_LOG_ERROR( L"TCPNCAg͑MɎs܂ : ɔjĂ܂ : Name[%s]", GetName() );
		ret = False;
	}

	return m_pPacketSender->Write( pData, dataSize );
}

UInt32 TcpClient::GetSendBytesPerSec( void ) const
{
	UInt32 ret;

	m_ShareData.sync.Enter();
	ret = m_ShareData.sendBytesPerSec;
	m_ShareData.sync.Leave();

	return ret;
}

UInt32 TcpClient::GetReceiverBytesPerSec( void ) const
{
	UInt32 ret;

	m_ShareData.sync.Enter();
	ret = m_ShareData.receiveBytesPerSec;
	m_ShareData.sync.Leave();

	return ret;
}

Boolean TcpClient::DoEvent( void )
{
	Boolean ret = True;
	UInt32 nowTime;
	UInt32 elapseTime;
	UInt32 eventType;
	Int32 eventOpt;
	UInt32 packetType;
	void* pData;
	UInt32 dataSize;


	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Cxg
	/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	eventType = m_EventToCurrent.Poll( &eventOpt );

	switch( eventType )
	{
	//ڑCxg
	case EVENTTYPE_CONNECT:
		DoEvent_Connect();
		break;
	//ؒfCxg
	case EVENTTYPE_DISCONNECT:
		DoEvent_Disconnect();
		break;
	//G[Cxg
	case EVENTTYPE_ERROR:
		m_pCallback->OnError( static_cast<Mix::Network::RESULT>( eventOpt ) );
		break;

	//Cxg
	case EVENTTYPE_NONE:
		ret = False;
		break;
	}

	if( m_bConnected == True )
	{
		nowTime = ::GetTickCount();

		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		// M
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

		m_pPacketReceiver->BeginRead();
		while( m_pPacketReceiver->Read( packetType, &pData, dataSize ) == True )
		{
			switch( packetType )
			{
			//GR[
			case Mix::Network::PKTYPE_ECHO:
				m_EchoTime = nowTime;
				m_pPacketSender->WriteEcho();
				break;
			//f[^
			case Mix::Network::PKTYPE_DATA:
				m_pCallback->OnRecive( pData, dataSize );
				break;
			}
		}
		m_pPacketReceiver->EndRead();


		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		// GR[
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

		elapseTime = ( m_EchoTime > nowTime )? ( ( 0xFFFFFFFF - m_EchoTime ) + nowTime ) : ( nowTime - m_EchoTime );
		if( elapseTime >= m_TimeOut )
		{
			m_pCallback->OnError( Mix::Network::ERR_TIMEOUT );
			Disconnect();
		}
	}

	return ret;
}

void TcpClient::DoEvent_Connect( void )
{
	//ݒ
	m_bConnected = True;
	m_EchoTime = ::GetTickCount();

	//MLɂ
	m_pPacketSender->Enable();

	//ڑʒm
	m_pCallback->OnConnect();
}

void TcpClient::DoEvent_Disconnect( void )
{
	UInt32 packetType;
	UInt32 dataSize;
	void* pData;

	//ݒ
	m_bConnected = False;

	//M𖳌ɂ
	m_pPacketSender->Disable();

	//cM
	m_pPacketReceiver->BeginRead();
	while( m_pPacketReceiver->Read( packetType, &pData, dataSize ) == True )
	{
		if( packetType == Mix::Network::PKTYPE_DATA )
		{
			m_pCallback->OnRecive( pData, dataSize );
		}
	}
	m_pPacketReceiver->EndRead();

	//ؒfʒm
	m_pCallback->OnDisconnect();
}

void TcpClient::ThreadMain( void )
{
	Boolean bRunning = True;
	UInt32 ret;
	UInt32 recvBuffSize;
	Int32 recvSize;
	Int8* recvBuff;
	UInt32 sendBuffSize;
	Int8* sendBuff;
	Int32 sendSize;
	UInt32 nowTime;
	UInt32 elapseTime;
	THREADCONTEXT ctx;
	WSANETWORKEVENTS netEvents;

	//
	ctx.sock = INVALID_SOCKET;
	ctx.sockEvent = NULL;
	ctx.sockFlags = 0;
	ctx.bConnected = False;
	ctx.updateTime = ::GetTickCount();
	ctx.sendBytesPerSec = 0;
	ctx.receiveBytesPerSec = 0;

	while( bRunning == True )
	{
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////
		// Cxg
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////

		switch( m_EventToThread.Poll() )
		{
		//ڑCxg
		case EVENTTYPE_CONNECT:
			ThreadMain_Connect( ctx );
			break;
		//ؒfCxg
		case EVENTTYPE_DISCONNECT:
			MIX_SETBIT( ctx.sockFlags, TcpClient::SF_CLOSE );
			break;
		//ICxg
		case EVENTTYPE_EXIT:
			MIX_SETBIT( ctx.sockFlags, TcpClient::SF_CLOSE );
			bRunning = False;
			break;
		}


		/////////////////////////////////////////////////////////////////////////////////////////////////////////////
		// ڑĂԂł̏
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////

		if( ctx.bConnected == True )
		{
			/////////////////////////////////////////////////////////////////////////////////////////////////////////////
			// Cxg
			/////////////////////////////////////////////////////////////////////////////////////////////////////////////

			ret = ::WSAWaitForMultipleEvents( 1, &ctx.sockEvent, FALSE, 0, FALSE );
			if( ret == WSA_WAIT_FAILED )
			{
				//G[ : \PbgCxg̏Ɏs
				MIX_SETBIT( ctx.sockFlags, TcpClient::SF_CLOSE );
				ThreadMain_Error( Mix::Network::ERR_WAITEVENT );
			}
			else
			{
				::WSAEnumNetworkEvents( ctx.sock, ctx.sockEvent, &netEvents );
				if( ( ( netEvents.lNetworkEvents & FD_READ ) == FD_READ ) &&
					( netEvents.iErrorCode[FD_READ_BIT] == 0 ) )
				{
					//M
					MIX_SETBIT( ctx.sockFlags, TcpClient::SF_RECV_WRITE );
				}
				if( ( ( netEvents.lNetworkEvents & FD_CLOSE ) == FD_CLOSE ) &&
					( netEvents.iErrorCode[FD_CLOSE_BIT] == 0 ) )
				{
					//ؒf
					MIX_SETBIT( ctx.sockFlags, TcpClient::SF_CLOSE );
				}

				::WSAResetEvent( ctx.sockEvent );
			}


			/////////////////////////////////////////////////////////////////////////////////////////////////////////////
			// M
			/////////////////////////////////////////////////////////////////////////////////////////////////////////////
			
			if( MIX_TESTBIT( ctx.sockFlags, TcpClient::SF_RECV_WRITE ) == TcpClient::SF_RECV_WRITE )
			{
				recvBuff = m_pPacketReceiver->GetWrite( recvBuffSize );
				if( recvBuff != NULL )
				{
					MIX_RESETBIT( ctx.sockFlags, TcpClient::SF_NOCAPACITY_TEMPBUFF );

					recvSize = ::recv( ctx.sock, recvBuff, recvBuffSize, 0 );
					if( recvSize != SOCKET_ERROR )
					{
						m_pPacketReceiver->Write( recvSize );

						MIX_RESETBIT( ctx.sockFlags, TcpClient::SF_RECV_WRITE );
						MIX_SETBIT( ctx.sockFlags, TcpClient::SF_RECV_PARSE );

						ctx.receiveBytesPerSec += recvSize;
					}
					else
					{
						//G[ : recv Ɏs
						MIX_SETBIT( ctx.sockFlags, TcpClient::SF_CLOSE );
						ThreadMain_Error( Mix::Network::ERR_RECIVE );
					}
				}
				else
				{
					//x : e|Mobt@̋󂫗eʂs
					if( MIX_TESTBIT( ctx.sockFlags, TcpClient::SF_NOCAPACITY_TEMPBUFF ) != TcpClient::SF_NOCAPACITY_TEMPBUFF )
					{
						MIX_SETBIT( ctx.sockFlags, TcpClient::SF_NOCAPACITY_TEMPBUFF );
						MIX_LOG_WARNING( L"TCPNCAg̎Mɒx : e|Mobt@̋󂫗eʂs : Name[%s]", GetName() );
					}

					//ꉞp[XtO𗧂ĂĂ
					MIX_SETBIT( ctx.sockFlags, TcpClient::SF_RECV_PARSE );
				}
			}

			if( MIX_TESTBIT( ctx.sockFlags, TcpClient::SF_RECV_PARSE ) == TcpClient::SF_RECV_PARSE )
			{
				switch( m_pPacketReceiver->Parse() )
				{
				//
				case PacketReceiver::OK:
					MIX_RESETBIT( ctx.sockFlags, TcpClient::SF_NOCAPACITY_BUFF );
					MIX_RESETBIT( ctx.sockFlags, TcpClient::SF_RECV_PARSE );
					break;

				//x : Mobt@̋󂫗eʂsĂ
				case PacketReceiver::WARNING_BUFF_CAPACITY:
					if( MIX_TESTBIT( ctx.sockFlags, TcpClient::SF_NOCAPACITY_BUFF ) != TcpClient::SF_NOCAPACITY_BUFF )
					{
						MIX_SETBIT( ctx.sockFlags, TcpClient::SF_NOCAPACITY_BUFF );
						MIX_LOG_WARNING( L"TCPNCAg̎Mɒx : Mobt@̋󂫗eʂs : Name[%s]", GetName() );
					}
					break;

				//G[ : pPbgjĂ
				case PacketReceiver::ERROR_PACKET_BLOKEN:
					MIX_SETBIT( ctx.sockFlags, TcpClient::SF_CLOSE );
					ThreadMain_Error( Mix::Network::ERR_BLOKENPACKET );
					break;
				//G[ : pPbgTCY傫
				case PacketReceiver::ERROR_PACKET_LARGE:
					MIX_SETBIT( ctx.sockFlags, TcpClient::SF_CLOSE );
					ThreadMain_Error( Mix::Network::ERR_LARGEPACKETSIZE );
					break;
				}
			}


			/////////////////////////////////////////////////////////////////////////////////////////////////////////////
			// M
			/////////////////////////////////////////////////////////////////////////////////////////////////////////////

			sendBuff = m_pPacketSender->BeginRead( sendBuffSize );
			if( sendBuff != NULL )
			{
				while( sendBuffSize > 0 )
				{
					sendSize = ::send( ctx.sock, sendBuff, sendBuffSize, 0 );

					if( sendSize != SOCKET_ERROR )
					{
						sendBuffSize -= sendSize;
						sendBuff += sendSize;

						ctx.sendBytesPerSec += sendSize;
					}
					else
					{
						//G[ : send Ɏs
						ThreadMain_Error( Mix::Network::ERR_SEND );
						break;
					}
				}

				m_pPacketSender->EndRead();
			}

		
			/////////////////////////////////////////////////////////////////////////////////////////////////////////////
			// N[Y
			/////////////////////////////////////////////////////////////////////////////////////////////////////////////

			if( MIX_TESTBIT( ctx.sockFlags, TcpClient::SF_CLOSE ) == TcpClient::SF_CLOSE )
			{
				ThreadMain_Disconnect( ctx );
			}
		}


		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
		// v
		/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

		nowTime = ::GetTickCount();
		elapseTime = ( ctx.updateTime > nowTime )? ( ( 0xFFFFFFFF - ctx.updateTime ) + nowTime ) : ( nowTime - ctx.updateTime );
		if( elapseTime >= TcpClient::ONSEC )
		{
			m_ShareData.sync.Enter();
			m_ShareData.sendBytesPerSec = ctx.sendBytesPerSec;
			m_ShareData.receiveBytesPerSec = ctx.receiveBytesPerSec;
			m_ShareData.sync.Leave();

			ctx.sendBytesPerSec = 0;
			ctx.receiveBytesPerSec = 0;
			ctx.updateTime = nowTime;
		}

		::Sleep( m_SleepTime );
	}

	//jCxgVOiԂɂ
	m_EventDestroy.Set();
}

void TcpClient::ThreadMain_Connect( THREADCONTEXT& ctx )
{
	if( ctx.bConnected == True )
	{
		return;
	}

	WSAEVENT hConnEvent = NULL;
	WSANETWORKEVENTS connEvents;
	SOCKADDR_IN hostAddr;
	UInt32 ret;
	Int32 ret_i;
	Mix::StringA addrAnsi = m_Addr.GetConstPtr();
	Boolean bConnect = False;
	std::list<UInt32> addrList;

	// 
	ctx.sockFlags = 0;

	//\Pbg쐬
	ctx.sock = ::socket( AF_INET, SOCK_STREAM, 0 );
	if( ctx.sock == INVALID_SOCKET )
	{
		ThreadMain_Error( Mix::Network::ERR_FATAL );
		return;
	}

	//\PbgIvVݒ
	if( ::setsockopt( ctx.sock, IPPROTO_TCP, TCP_NODELAY, reinterpret_cast<Int8*>( &m_bUseNagle ), sizeof( m_bUseNagle ) ) == SOCKET_ERROR )
	{
		::closesocket( ctx.sock );
		ThreadMain_Error( Mix::Network::ERR_FATAL );
		return;
	}

	//\PbgCxg쐬
	ctx.sockEvent = ::WSACreateEvent();
	if( ctx.sockEvent == NULL )
	{
		::closesocket( ctx.sock );
		ThreadMain_Error( Mix::Network::ERR_FATAL );
		return;
	}

	//ڑCxg쐬
	hConnEvent = ::WSACreateEvent();
	if( hConnEvent == NULL )
	{
		::WSACloseEvent( ctx.sockEvent );
		::closesocket( ctx.sock );
		ThreadMain_Error( Mix::Network::ERR_FATAL );
		return;
	}
	if( ::WSAEventSelect( ctx.sock, hConnEvent, FD_CONNECT ) == SOCKET_ERROR )
	{
		::WSACloseEvent( hConnEvent );
		::WSACloseEvent( ctx.sockEvent );
		::closesocket( ctx.sock );
		ThreadMain_Error( Mix::Network::ERR_FATAL );
		return;
	}

	//AhXꗗ擾
	ret = ::inet_addr( addrAnsi.GetConstPtr() );
	if( ret == 0xFFFFFFFF )
	{
		struct hostent* pHostent = ::gethostbyname( addrAnsi.GetConstPtr() );
		if( pHostent != NULL )
		{
			UInt32** ppAddrList = reinterpret_cast<UInt32**>( pHostent->h_addr_list );
			while( *ppAddrList != NULL )
			{
				addrList.push_back( *( *ppAddrList ) );
				ppAddrList++;
			}
		}
	}
	else
	{
		addrList.push_back( ret );
	}
	if( addrList.size() <= 0 )
	{
		::WSACloseEvent( hConnEvent );
		::WSACloseEvent( ctx.sockEvent );
		::closesocket( ctx.sock );
		ThreadMain_Error( Mix::Network::ERR_FATAL );
		return;
	}

	//ڑJn
	::ZeroMemory( &hostAddr, sizeof( hostAddr ) );
	hostAddr.sin_family = AF_INET;
	hostAddr.sin_port = ::htons( m_Port );
	for( std::list<UInt32>::iterator it = addrList.begin(); ( it != addrList.end() ) && ( bConnect == False ); ++it )
	{
		hostAddr.sin_addr.s_addr = ( *it );

		if( ( ::connect( ctx.sock, reinterpret_cast<SOCKADDR*>( &hostAddr ), sizeof( hostAddr ) ) == SOCKET_ERROR ) &&
			( ::WSAGetLastError() != WSAEWOULDBLOCK ) )
		{
			continue;
		}

		ret = ::WSAWaitForMultipleEvents( 1, &hConnEvent, FALSE, 1000, FALSE );
		if( ret != WSA_WAIT_EVENT_0 )
		{
			continue;
		}

		ret_i = ::WSAEnumNetworkEvents( ctx.sock, hConnEvent, &connEvents );
		if( ret_i == SOCKET_ERROR )
		{
			continue;
		}

		if( ( connEvents.lNetworkEvents & FD_CONNECT ) &&
			( connEvents.iErrorCode[FD_CONNECT_BIT] == 0 ) )
		{
			bConnect = True;
		}
	}

	//RlNgCxg
	::WSACloseEvent( hConnEvent );

	if( bConnect == False )
	{
		//ڑs
		::WSACloseEvent( ctx.sockEvent );
		::closesocket( ctx.sock );
		ThreadMain_Error( Mix::Network::ERR_SERVER_NOTFOUND );
		return;
	}

	//Cxgݒ
	if( ::WSAEventSelect( ctx.sock, ctx.sockEvent, FD_READ | FD_CLOSE ) == SOCKET_ERROR )
	{
		::WSACloseEvent( ctx.sockEvent );
		::closesocket( ctx.sock );
		ThreadMain_Error( Mix::Network::ERR_FATAL );
		return;
	}

	//ڑς
	ctx.bConnected = True;

	//ڑʒm
	m_EventToCurrent.Push( EVENTTYPE_CONNECT );
}

void TcpClient::ThreadMain_Disconnect( THREADCONTEXT& ctx )
{
	//\Pbg
	if( ctx.sock != INVALID_SOCKET )
	{
		//Vbg_E
		::shutdown( ctx.sock, SD_BOTH );

		//N[Y
		::closesocket( ctx.sock );
		ctx.sock = INVALID_SOCKET;
	}

	//\PbgCxg
	if( ctx.sockEvent != NULL )
	{
		::WSACloseEvent( ctx.sockEvent );
		ctx.sockEvent = NULL;
	}

	if( ctx.bConnected == True )
	{
		//M𖳌ɂ
		m_pPacketSender->Disable( True );
		//ؒfς
		MIX_RESETBIT( ctx.sockFlags, TcpClient::SF_CLOSE );
		ctx.bConnected = False;
		//CxgXbhɐؒfʒm
		m_EventToCurrent.Push( EVENTTYPE_DISCONNECT );
	}

}

void TcpClient::ThreadMain_Error( Mix::Network::RESULT ret )
{
	m_EventToCurrent.Push( EVENTTYPE_ERROR, ret );
}

void __cdecl TcpClient::ThreadEntry( void* pArg )
{
	TcpClient* pTcpClient = static_cast<TcpClient*>( pArg );
	pTcpClient->ThreadMain();
}

}}
