#include "Mix/Interpolation.h"

namespace Mix{

////////////////////////////////////////////////////////////////////////////////////////////////////
// `
////////////////////////////////////////////////////////////////////////////////////////////////////

//`f[^
struct LINEAR_DATA
{
	UInt32 count;
	UInt32 numDimension;

	UInt32 secCount;

	Float32 firstPoint;
	Float32 lastPoint;

	std::vector<Float32> points;
	std::vector<Float32> values;
	std::vector<Float32> scales;

	std::vector<Float32> result;
};

//`ԃf[^̍\z
Boolean Linear_Build( LINEAR_DATA& data, const Float32* pPoints, const Float32* pValues, UInt32 numDimension, UInt32 count )
{
	if( ( pPoints == NULL ) ||
		( pValues == NULL ) ||
		( numDimension == 0 ) ||
		( count == 0 ) )
	{
		return False;
	}

	data.count = count;
	data.numDimension = numDimension;
	data.secCount = count - 1;
	data.firstPoint = pPoints[0];
	data.lastPoint = pPoints[count - 1];
	data.points.resize( count );
	data.values.resize( numDimension * count );
	data.scales.resize( count - 1 );
	data.result.resize( numDimension );

	Mix::Memory::Copy( &( data.points[0] ), pPoints, sizeof( Float32 ) * count );
	Mix::Memory::Copy( &( data.values[0] ), pValues, sizeof( Float32 ) * numDimension * count );

	for( UInt32 i = 0; i < data.secCount; i++ )
	{
		data.scales[i] = MIX_FLOAT_RECIPROCAL( pPoints[i + 1] - pPoints[i] );
	}

	return True;
}

Float32 Linear_GetPoint( LINEAR_DATA& data, UInt32 index )
{
	if( data.count <= index )
	{
		index = data.count - 1;
	}

	return data.points[index];
}

const Float32* Linear_GetValue( LINEAR_DATA& data, UInt32 index )
{
	if( data.count <= index )
	{
		index = data.count - 1;
	}

	return &( data.values[data.numDimension * index] );
}

const Float32* Linear_Calculate( LINEAR_DATA& data, Float32 point )
{
	const Float32* points = &( data.points[0] );
	const Float32* values = &( data.values[0] );
	const Float32* scales = &( data.scales[0] );
	Float32* result = &( data.result[0] );

	UInt32 i;

	if( data.secCount > 0 )
	{
		UInt32 first = 0;
		UInt32 last = data.secCount;
		UInt32 middle = ( first + last ) >> 1;

		UInt32 cur;
		UInt32 next;
		UInt32 index;
		Float32 ratio;
		const Float32* cv;
		const Float32* nv;

		while( first != middle )
		{
			if( points[middle] > point )
			{
				last = middle;
			}
			else
			{
				first = middle;
			}

			middle = ( first + last ) >> 1;
		}

		index = last - 1;
		cur = data.numDimension * index;
		next = data.numDimension * ( index + 1 );
		ratio = ( MIX_CLAMP( point, data.firstPoint, data.lastPoint ) - points[index] ) * scales[index];

		cv = values + cur;
		nv = values + next;

		for( i = 0; i < data.numDimension; i++ )
		{
			result[i] = *cv + ( *nv - *cv ) * ratio;

			cv++;
			nv++;
		}
	}
	else
	{
		for( i = 0; i < data.numDimension; i++ )
		{
			result[i] = values[i];
		}
	}

	return &( result[0] );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// XvC
////////////////////////////////////////////////////////////////////////////////////////////////////

//XvCf[^\
struct SPLINE_DATA
{
	UInt32 count;
	UInt32 numDimension;

	Int32 secCount;

	Float32 firstPoint;
	Float32 lastPoint;

	std::vector<Float32> points;
	std::vector<Float32> values;

	std::vector<Float32> q;
	std::vector<Float32> s;
	std::vector<Float32> r;

	std::vector<Float32> result;
};

//XvCf[^\z
Boolean Spline_Build( SPLINE_DATA& data, const Float32* pPoints, const Float32* pValues, UInt32 numDimension, UInt32 count )
{
	if( ( pPoints == NULL ) ||
		( pValues == NULL ) ||
		( numDimension == 0 ) ||
		( count == 0 ) )
	{
		return False;
	}

	UInt32 secCount = count - 1;

	UInt32 i;
	UInt32 j;
	Float32* points;
	Float32* values;

	std::vector<Float32> h;
	std::vector<Float32> b;
	std::vector<Float32> d;
	std::vector<Float32> g;
	std::vector<Float32> u;

	Float32* q;
	Float32* s;
	Float32* r;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	data.count = count;
	data.numDimension = numDimension;

	data.secCount = static_cast<Int32>( count ) - 1;

	data.firstPoint = pPoints[0];
	data.lastPoint = pPoints[count - 1];

	data.points.resize( count );
	data.values.resize( numDimension * count );

	data.q.resize( numDimension * secCount );
	data.s.resize( numDimension * secCount );
	data.r.resize( numDimension * count );

	data.result.resize( numDimension );

	Mix::Memory::Copy( &( data.points[0] ), pPoints, sizeof( Float32 ) * count );
	Mix::Memory::Copy( &( data.values[0] ), pValues, sizeof( Float32 ) * numDimension * count );

	points = &( data.points[0] );
	values = &( data.values[0] );
	q = &( data.q[0] );
	s = &( data.s[0] );
	r = &( data.r[0] );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 1
	////////////////////////////////////////////////////////////////////////////////////////////////////

	h.resize( secCount );
	b.resize( secCount );
	d.resize( numDimension * secCount );

	for( i = 0; i < secCount; i++ )
	{
		h[i] = points[i + 1] - points[i];
	}

	for( i = 1; i < secCount; i++ )
	{
		UInt32 pre = numDimension * ( i - 1 );
		UInt32 cur = numDimension * i;
		UInt32 next = numDimension * ( i + 1 );

		b[i] = 2.0f * ( h[i] + h[i - 1] );

		for( j = 0; j < numDimension; j++ )
		{
			d[cur + j] = 3.0f * ( MIX_FLOAT_DIV( ( values[next + j] - values[cur + j] ), h[i] ) - MIX_FLOAT_DIV( ( values[cur + j] - values[pre + j] ), h[i - 1] ) );
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 2
	////////////////////////////////////////////////////////////////////////////////////////////////////

	g.resize( secCount );
	u.resize( secCount * numDimension );

	g[1] = h[1] / b[1];
	for( i = 2; i < ( secCount - 1 ); i++ )
	{
		g[i] = MIX_FLOAT_DIV( h[i], ( b[i] - h[i - 1] * g[i - 1] ) );
	}

	for( j = 0; j < numDimension; j++ )
	{
		u[numDimension * 1 + j] = MIX_FLOAT_DIV( d[numDimension * 1 + j], b[1] );
	}

	for( i = 2; i < secCount; i++ )
	{
		UInt32 pre = numDimension * ( i - 1 );
		UInt32 cur = numDimension * i;

		for( j = 0; j < numDimension; j++ )
		{
			u[cur + j] = MIX_FLOAT_DIV( ( d[cur + j] - h[i - 1] * u[pre + j] ), ( b[i] - h[i - 1] * g[i - 1] ) );
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 3
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( j = 0; j < numDimension; j++ )
	{
		r[numDimension * 0 + j] = 0.0f;
		r[numDimension * secCount + j] = 0.0f;
		r[numDimension * ( secCount - 1 ) + j] = u[numDimension * ( secCount - 1 ) + j];
	}

	for( i = ( secCount - 2 ); i >= 1; i-- )
	{
		UInt32 cur = numDimension * i;
		UInt32 next = numDimension * ( i + 1 );

		for( j = 0; j < numDimension; j++ )
		{
			r[cur + j] = u[cur + j] - g[i] * r[next + j];
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 4
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( i = 0; i < secCount; i++ )
	{
		UInt32 cur = numDimension * i;
		UInt32 next = numDimension * ( i + 1 );

		for( j = 0; j < numDimension; j++ )
		{
			q[cur + j] = MIX_FLOAT_DIV( ( values[next + j] - values[cur + j] ), h[i] ) - h[i] * ( r[next + j] + 2.0f * r[cur + j] ) / 3.0f;
			s[cur + j] = MIX_FLOAT_DIV( ( r[next + j] - r[cur + j] ), ( 3.0f * h[i] ) );
		}
	}

	return True;
}

//w肵CfbNẌʒu擾
Float32 Spline_GetPoint( SPLINE_DATA& data, UInt32 index )
{
	if( data.count <= index )
	{
		index = ( data.count - 1 );
	}

	return data.points[index];
}

//w肵CfbNX̒l̃|C^擾
const Float32* Spline_GetValue( SPLINE_DATA& data, UInt32 index )
{
	if( data.count <= index )
	{
		index = ( data.count - 1 );
	}

	return &( data.values[data.numDimension * index] );
}

//w肵ʒu̕Ԃꂽl擾
const Float32* Spline_Calculate( SPLINE_DATA& data, Float32 point )
{
	const Float32* points = &( data.points[0] );
	const Float32* values = &( data.values[0] );
	const Float32* q = &( data.q[0] );
	const Float32* r = &( data.r[0] );
	const Float32* s = &( data.s[0] );

	UInt32 i;
	UInt32 cur;
	UInt32 index;
	Float32 ratio;
	Float32* result = &( data.result[0] );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ԂJnCfbNX
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( data.secCount > 0 )
	{
		UInt32 first = 0;
		UInt32 last = data.secCount;
		UInt32 middle = ( first + last ) >> 1;

		while( first != middle )
		{
			if( points[middle] > point )
			{
				last = middle;
			}
			else
			{
				first = middle;
			}

			middle = ( first + last ) >> 1;
		}

		index = last - 1;
	}
	else
	{
		index = 0;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	ratio = MIX_CLAMP( point, data.firstPoint, data.lastPoint ) - points[index];
	cur = data.numDimension * index;
	values += cur;
	q += cur;
	r += cur;
	s += cur;

	for( i = 0; i < data.numDimension; i++ )
	{
		result[i] = *values + ratio * ( *q + ratio * ( *r + *s * ratio ) );

		values++;
		q++;
		r++;
		s++;

//		result[i] = values[cur + i] + ratio * ( q[cur + i] + ratio * ( r[cur + i]  + s[cur + i] * ratio ) );
	}

	return &result[0];
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// LinearF
////////////////////////////////////////////////////////////////////////////////////////////////////

class LinearF::Impl
{
public:
	LINEAR_DATA data;
};

LinearF::LinearF( void ) : pImpl( new LinearF::Impl() )
{
}

LinearF::~LinearF( void )
{
	MIX_DELETE( pImpl );
}

Boolean LinearF::Build( const Float32* pPoints, const Float32* pValues, UInt32 count )
{
	return Linear_Build( pImpl->data, pPoints, pValues, 1, count );
}

Float32 LinearF::GetPoint( UInt32 index ) const
{
	return Linear_GetPoint( pImpl->data, index );
}

Float32 LinearF::GetValue( UInt32 index ) const
{
	return *Linear_GetValue( pImpl->data, index );
}

UInt32 LinearF::GetCount( void ) const
{
	return pImpl->data.count;
}

Float32 LinearF::Calculate( Float32 point ) const
{
	return *Linear_Calculate( pImpl->data, point );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// LinearV2
////////////////////////////////////////////////////////////////////////////////////////////////////

class LinearV2::Impl
{
public:
	LINEAR_DATA data;
	Mix::Vector2 result;

public:
	Boolean Build( const Float32* pPoints, const Mix::Vector2* pValues, UInt32 count )
	{
		return Linear_Build( data, pPoints, reinterpret_cast<const Float32*>( pValues ), 2, count );
	}

	void GetValue( UInt32 index )
	{
		const Float32* pResult = Linear_GetValue( data, index );

		result.x = pResult[0];
		result.y = pResult[1];
	}

	void Calculate( Float32 point )
	{
		const Float32* pResult = Linear_Calculate( data, point );

		result.x = pResult[0];
		result.y = pResult[1];
	}
};

LinearV2::LinearV2( void ) : pImpl( new LinearV2::Impl() )
{
}

LinearV2::~LinearV2( void )
{
	MIX_DELETE( pImpl );
}

Boolean LinearV2::Build( const Float32* pPoints, const Mix::Vector2* pValues, UInt32 count )
{
	return pImpl->Build( pPoints, pValues, count );
}

Float32 LinearV2::GetPoint( UInt32 index ) const
{
	return Linear_GetPoint( pImpl->data, index );
}

Mix::Vector2 LinearV2::GetValue( UInt32 index ) const
{
	pImpl->GetValue( index );

	return pImpl->result;
}

UInt32 LinearV2::GetCount( void ) const
{
	return pImpl->data.count;
}

Mix::Vector2 LinearV2::Calculate( Float32 point ) const
{
	pImpl->Calculate( point );

	return pImpl->result;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// LinearV3
////////////////////////////////////////////////////////////////////////////////////////////////////

class LinearV3::Impl
{
public:
	LINEAR_DATA data;
	Mix::Vector3 result;

public:
	Boolean Build( const Float32* pPoints, const Mix::Vector3* pValues, UInt32 count )
	{
		return Linear_Build( data, pPoints, reinterpret_cast<const Float32*>( pValues ), 3, count );
	}

	void GetValue( UInt32 index )
	{
		const Float32* pResult = Linear_GetValue( data, index );

		result.x = pResult[0];
		result.y = pResult[1];
		result.z = pResult[2];
	}

	void Calculate( Float32 point )
	{
		const Float32* pResult = Linear_Calculate( data, point );

		result.x = pResult[0];
		result.y = pResult[1];
		result.z = pResult[2];
	}
};

LinearV3::LinearV3( void ) : pImpl( new LinearV3::Impl() )
{
}

LinearV3::~LinearV3( void )
{
	MIX_DELETE( pImpl );
}

Boolean LinearV3::Build( const Float32* pPoints, const Mix::Vector3* pValues, UInt32 count )
{
	return pImpl->Build( pPoints, pValues, count );
}

Float32 LinearV3::GetPoint( UInt32 index ) const
{
	return Linear_GetPoint( pImpl->data, index );
}

Mix::Vector3 LinearV3::GetValue( UInt32 index ) const
{
	pImpl->GetValue( index );

	return pImpl->result;
}

UInt32 LinearV3::GetCount( void ) const
{
	return pImpl->data.count;
}

Mix::Vector3 LinearV3::Calculate( Float32 point ) const
{
	pImpl->Calculate( point );

	return pImpl->result;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// LinearV4
////////////////////////////////////////////////////////////////////////////////////////////////////

class LinearV4::Impl
{
public:
	LINEAR_DATA data;
	Mix::Vector4 result;

public:
	Boolean Build( const Float32* pPoints, const Mix::Vector4* pValues, UInt32 count )
	{
		return Linear_Build( data, pPoints, reinterpret_cast<const Float32*>( pValues ), 4, count );
	}

	void GetValue( UInt32 index )
	{
		const Float32* pResult = Linear_GetValue( data, index );

		result.x = pResult[0];
		result.y = pResult[1];
		result.z = pResult[2];
		result.w = pResult[3];
	}

	void Calculate( Float32 point )
	{
		const Float32* pResult = Linear_Calculate( data, point );

		result.x = pResult[0];
		result.y = pResult[1];
		result.z = pResult[2];
		result.w = pResult[3];
	}
};

LinearV4::LinearV4( void ) : pImpl( new LinearV4::Impl() )
{
}

LinearV4::~LinearV4( void )
{
	MIX_DELETE( pImpl );
}

Boolean LinearV4::Build( const Float32* pPoints, const Mix::Vector4* pValues, UInt32 count )
{
	return pImpl->Build( pPoints, pValues, count );
}

Float32 LinearV4::GetPoint( UInt32 index ) const
{
	return Linear_GetPoint( pImpl->data, index );
}

Mix::Vector4 LinearV4::GetValue( UInt32 index ) const
{
	pImpl->GetValue( index );

	return pImpl->result;
}

UInt32 LinearV4::GetCount( void ) const
{
	return pImpl->data.count;
}

Mix::Vector4 LinearV4::Calculate( Float32 point ) const
{
	pImpl->Calculate( point );

	return pImpl->result;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// SplineF
////////////////////////////////////////////////////////////////////////////////////////////////////

class SplineF::Impl
{
public:
	SPLINE_DATA data;
};

SplineF::SplineF( void ) : pImpl( new SplineF::Impl() )
{
}

SplineF::~SplineF( void )
{
	MIX_DELETE( pImpl );
}

Boolean SplineF::Build( const Float32* pPoints, const Float32* pValues, UInt32 count )
{
	return Spline_Build( pImpl->data, pPoints, pValues, 1, count );
}

Float32 SplineF::GetPoint( UInt32 index ) const
{
	return Spline_GetPoint( pImpl->data, index );
}

Float32 SplineF::GetValue( UInt32 index ) const
{
	return *Spline_GetValue( pImpl->data, index );
}

UInt32 SplineF::GetCount( void ) const
{
	return pImpl->data.count;
}

Float32 SplineF::Calculate( Float32 point ) const
{
	return *Spline_Calculate( pImpl->data, point );
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// SplineV2
////////////////////////////////////////////////////////////////////////////////////////////////////

class SplineV2::Impl
{
public:
	SPLINE_DATA data;
	Mix::Vector2 result;

public:
	Boolean Build( const Float32* pPoints, const Mix::Vector2* pValues, UInt32 count )
	{
		return Spline_Build( data, pPoints, reinterpret_cast<const Float32*>( pValues ), 2, count );
	}

	void GetValue( UInt32 index )
	{
		const Float32* pResult = Spline_GetValue( data, index );

		result.x = pResult[0];
		result.y = pResult[1];
	}

	void Calculate( Float32 point )
	{
		const Float32* pResult = Spline_Calculate( data, point );

		result.x = pResult[0];
		result.y = pResult[1];
	}
};

SplineV2::SplineV2( void ) : pImpl( new SplineV2::Impl() )
{
}

SplineV2::~SplineV2( void )
{
	MIX_DELETE( pImpl );
}

Boolean SplineV2::Build( const Float32* pPoints, const Mix::Vector2* pValues, UInt32 count )
{
	return pImpl->Build( pPoints, pValues, count );
}

Float32 SplineV2::GetPoint( UInt32 index ) const
{
	return Spline_GetPoint( pImpl->data, index );
}

Mix::Vector2 SplineV2::GetValue( UInt32 index ) const
{
	pImpl->GetValue( index );

	return pImpl->result;
}

UInt32 SplineV2::GetCount( void ) const
{
	return pImpl->data.count;
}

Mix::Vector2 SplineV2::Calculate( Float32 point ) const
{
	pImpl->Calculate( point );

	return pImpl->result;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// SplineV3
////////////////////////////////////////////////////////////////////////////////////////////////////

class SplineV3::Impl
{
public:
	SPLINE_DATA data;
	Mix::Vector3 result;

public:
	Boolean Build( const Float32* pPoints, const Mix::Vector3* pValues, UInt32 count )
	{
		return Spline_Build( data, pPoints, reinterpret_cast<const Float32*>( pValues ), 3, count );
	}

	void GetValue( UInt32 index )
	{
		const Float32* pResult = Spline_GetValue( data, index );

		result.x = pResult[0];
		result.y = pResult[1];
		result.z = pResult[2];
	}

	void Calculate( Float32 point )
	{
		const Float32* pResult = Spline_Calculate( data, point );

		result.x = pResult[0];
		result.y = pResult[1];
		result.z = pResult[2];
	}
};

SplineV3::SplineV3( void ) : pImpl( new SplineV3::Impl() )
{
}

SplineV3::~SplineV3( void )
{
	MIX_DELETE( pImpl );
}

Boolean SplineV3::Build( const Float32* pPoints, const Mix::Vector3* pValues, UInt32 count )
{
	return pImpl->Build( pPoints, pValues, count );
}

Float32 SplineV3::GetPoint( UInt32 index ) const
{
	return Spline_GetPoint( pImpl->data, index );
}

Mix::Vector3 SplineV3::GetValue( UInt32 index ) const
{
	pImpl->GetValue( index );

	return pImpl->result;
}

UInt32 SplineV3::GetCount( void ) const
{
	return pImpl->data.count;
}

Mix::Vector3 SplineV3::Calculate( Float32 point ) const
{
	pImpl->Calculate( point );

	return pImpl->result;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// SplineV4
////////////////////////////////////////////////////////////////////////////////////////////////////

class SplineV4::Impl
{
public:
	SPLINE_DATA data;
	Mix::Vector4 result;

public:
	Boolean Build( const Float32* pPoints, const Mix::Vector4* pValues, UInt32 count )
	{
		return Spline_Build( data, pPoints, reinterpret_cast<const Float32*>( pValues ), 4, count );
	}

	void GetValue( UInt32 index )
	{
		const Float32* pResult = Spline_GetValue( data, index );

		result.x = pResult[0];
		result.y = pResult[1];
		result.z = pResult[2];
		result.w = pResult[3];
	}

	void Calculate( Float32 point )
	{
		const Float32* pResult = Spline_Calculate( data, point );

		result.x = pResult[0];
		result.y = pResult[1];
		result.z = pResult[2];
		result.w = pResult[3];
	}
};

SplineV4::SplineV4( void ) : pImpl( new SplineV4::Impl() )
{
}

SplineV4::~SplineV4( void )
{
	MIX_DELETE( pImpl );
}

Boolean SplineV4::Build( const Float32* pPoints, const Mix::Vector4* pValues, UInt32 count )
{
	return pImpl->Build( pPoints, pValues, count );
}

Float32 SplineV4::GetPoint( UInt32 index ) const
{
	return Spline_GetPoint( pImpl->data, index );
}

Mix::Vector4 SplineV4::GetValue( UInt32 index ) const
{
	pImpl->GetValue( index );

	return pImpl->result;
}

UInt32 SplineV4::GetCount( void ) const
{
	return pImpl->data.count;
}

Mix::Vector4 SplineV4::Calculate( Float32 point ) const
{
	pImpl->Calculate( point );

	return pImpl->result;
}

}
