#include "Mix/Class/Graphics/Utility/Common/TextPrinter.h"

#include "Mix/Graphics/IDevice.h"
#include "Mix/Graphics/ITexture.h"

namespace Mix{ namespace Graphics{ namespace Utility{ namespace Common{

const wchar_t* TextPrinter::FAILED_INIT = L"eLXgv^[̏Ɏs";
const wchar_t* TextPrinter::FAILED_PROC = L"eLXgv^[̏Ɏs";

TextPrinter* TextPrinter::CreateInstance( Mix::Graphics::IDevice* pDevice, const wchar_t* pName )
{
	return new TextPrinter( pDevice, pName );
}

TextPrinter::TextPrinter( Mix::Graphics::IDevice* pDevice, const wchar_t* pName ) :
m_Name( MIX_SAFE_NAME( pName ) ),
m_pDevice( pDevice ),
m_TexSize( 0 ),
m_TexSizeInv( 0.0f ),
m_TexPos( TextPrinter::TEX_MARGIN, TextPrinter::TEX_MARGIN ),
m_TexLine( TextPrinter::TEX_MARGIN ),
m_PageCount( 0 )
{
	Mix::StringW tempStr;

	MIX_ADD_REF( m_pDevice );

	tempStr.Sprintf( L"%s/PageList", m_Name.GetConstPtr() );
	m_PageList.Initialize( 8, 2, tempStr.GetConstPtr() );

	tempStr.Sprintf( L"%s/SectList", m_Name.GetConstPtr() );
	m_SectList.Initialize( 8, 1, tempStr.GetConstPtr() );

	tempStr.Sprintf( L"%s/CharList", m_Name.GetConstPtr() );
	m_CharList.Initialize( 1024, 64, tempStr.GetConstPtr() );

	tempStr.Sprintf( L"%s/QuadGroupList", m_Name.GetConstPtr() );
	m_QuadGroupList.Initialize( 2, 1, tempStr.GetConstPtr() );

	tempStr.Sprintf( L"%s/QuadList", m_Name.GetConstPtr() );
	m_QuadList.Initialize( 512, 32, tempStr.GetConstPtr() );
}

TextPrinter::~TextPrinter( void )
{
	for( UInt32 i = 0; i < m_PageList.GetCount(); i++ )
	{
		MIX_RELEASE( m_PageList[i].pTexture );
	}

	for( UInt32 i = 0; i < m_SectList.GetCount(); i++ )
	{
		MIX_RELEASE( m_SectList[i].pFont );
	}

	m_CharList.Clear();
	m_QuadGroupList.Clear();
	m_QuadList.Clear();

	MIX_RELEASE( m_pDevice );
}

Boolean TextPrinter::Initialize( Int32 texSize, UInt32 defTexCount = 1 )
{
	//łɏĂ
	MIX_ASSERT( m_TexSize == 0 );

	//p[^s
	MIX_ASSERT( texSize >= 256 );
	MIX_ASSERT( defTexCount >= 1 );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// eNX`TCY
	////////////////////////////////////////////////////////////////////////////////////////////////////

	m_TexSize = texSize;

	switch( m_pDevice->GetShaderModel() )
	{
	case Mix::Graphics::SHADER_MODEL_3:
		m_TexSizeInv = MIX_FLOAT_RECIPROCAL( static_cast<Float32>( m_TexSize - 1 ) );
		break;
	case Mix::Graphics::SHADER_MODEL_4:
	case Mix::Graphics::SHADER_MODEL_5:
		m_TexSizeInv = MIX_FLOAT_RECIPROCAL( static_cast<Float32>( m_TexSize ) );
		break;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ŏ玝ĂeNX`쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( UInt32 i = 0; i < defTexCount; i++ )
	{
		Mix::Graphics::ITexture* pTexture = CreateTexture();
		if( pTexture == NULL )
		{
			MIX_LOG_ERROR( L"%s : y[W̃eNX`쐬ł܂ł : Name[%s]", TextPrinter::FAILED_INIT, m_Name.GetConstPtr() );
			return False;
		}

		TextPrinter::PAGE* pPage = m_PageList.Add();
		if( pPage == NULL )
		{
			MIX_LOG_ERROR( L"%s : %s : Name[%s]", TextPrinter::FAILED_INIT,  Mix::STR_OUTOFMEMORY, m_Name.GetConstPtr() );
			MIX_RELEASE( pTexture );
			return False;
		}

		pPage->pTexture = pTexture;
		pPage->sectStart = 0;
		pPage->sectCount = 0;
	}

	return True;
}

Boolean TextPrinter::Append( Mix::Graphics::Utility::IFont* pFont, const Mix::Vector2& basePos, const wchar_t* pText )
{
	MIX_ASSERT( m_TexSize > 0 );

	if( ( pFont == NULL ) ||
		( pText == NULL ) ||
		( ::wcslen( pText ) == 0 ) )
	{
		return False;
	}

	TextPrinter::PAGE* pPage = NULL;
	TextPrinter::SECTION* pSect = NULL;
	TextPrinter::QUAD_GROUP* pQuadGroup = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pPage = GetCurrentPage();
	if( pPage == NULL )
	{
		pPage = AddPage( pFont );
		if( pPage == NULL )
		{
			return False;
		}
	}

	pSect = AddSection( pFont );
	if( pSect == NULL )
	{
		return False;
	}

	m_QuadGroupList.Clear();
	m_QuadList.Clear();

	pQuadGroup = m_QuadGroupList.Add();
	if( pQuadGroup != NULL )
	{
		pQuadGroup->pTexture = pPage->pTexture;
		pQuadGroup->quadStart = 0;
		pQuadGroup->quadCount = 0;
	}
	else
	{
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ŏ̃y[W擾
	////////////////////////////////////////////////////////////////////////////////////////////////////

	UInt32 fontHeightI = pFont->GetHeight();
	Float32 fontHeightF = static_cast<Float32>( fontHeightI );
	Float32 fontHalfHeightF = fontHeightF * 0.5f;
	Float32 tabF = static_cast<Float32>( fontHalfHeightF * Mix::Graphics::Utility::STR_TAB_COUNT );

	Mix::Vector2 pos = basePos;

	for( UInt32 i = 0; pText[i] != L'\0'; i++ )
	{
		wchar_t code = pText[i];
		Mix::Graphics::Utility::IFont::GLYPH fontGlyph;

		if( code == L' ' )
		{
			pos.x += fontHalfHeightF;
		}
		else if( code == L'@' )
		{
			pos.x += fontHeightF;
		}
		else if( code == L'\t' )
		{
			pos.x += tabF;
		}
		else if( code == L'\r' )
		{
			;
		}
		else if( code == L'\n' )
		{
			pos.x = basePos.x;
			pos.y += fontHeightF;
		}
		else if( pFont->GetGlyph( code, fontGlyph ) == True )
		{
			TextPrinter::CHARACTER* pChar;
			TextPrinter::QUAD* pQuad;
			Float32 dl, dt, dr, db;
			Float32 tl, tt, tr, tb;
			Mix::Vector2* points;
			Mix::Vector2* texCoords;

			if( ( m_TexPos.x + fontGlyph.width + TextPrinter::TEX_MARGIN ) >= m_TexSize )
			{
				//͂ݏo
				m_TexPos.x = TextPrinter::TEX_MARGIN;
				m_TexPos.y = m_TexLine;
			}

			if( ( m_TexPos.y + fontGlyph.height + TextPrinter::TEX_MARGIN ) >= m_TexSize )
			{
				//c͂ݏo
				m_TexPos.x = TextPrinter::TEX_MARGIN;
				m_TexPos.y = TextPrinter::TEX_MARGIN;
				m_TexLine = TextPrinter::TEX_MARGIN;

				pPage = AddPage( pFont );
				if( pPage == NULL )
				{
					return False;
				}

				pSect = AddSection( pFont );
				if( pSect == NULL )
				{
					return False;
				}

				pQuadGroup = m_QuadGroupList.Add();
				if( pQuadGroup != NULL )
				{
					pQuadGroup->pTexture = pPage->pTexture;
					pQuadGroup->quadStart = m_QuadList.GetCount();
					pQuadGroup->quadCount = 0;
				}
				else
				{
					return False;
				}
			}

			//LN^[ǉ
			pChar = m_CharList.Add();
			pChar->texPos = m_TexPos;
			pChar->fontGlyph = fontGlyph;
			pSect->charCount++;

			//Nbhǉ
			pQuad = m_QuadList.Add();
			pQuadGroup->quadCount++;

			//`W
			dl = pos.x + static_cast<Float32>( fontGlyph.x );
			dt = pos.y + static_cast<Float32>( fontGlyph.y );
			dr = pos.x + static_cast<Float32>( fontGlyph.x + fontGlyph.width );
			db = pos.y + static_cast<Float32>( fontGlyph.y + fontGlyph.height );

			points = pQuad->points;

			points[0].Set( dl, dt );
			points[1].Set( dr, dt );
			points[2].Set( dr, db );
			points[3].Set( dl, db );

			//eNX`W
			tl = static_cast<Float32>( m_TexPos.x ) * m_TexSizeInv;
			tt = static_cast<Float32>( m_TexPos.y ) * m_TexSizeInv;
			tr = static_cast<Float32>( m_TexPos.x + fontGlyph.width ) * m_TexSizeInv;
			tb = static_cast<Float32>( m_TexPos.y + fontGlyph.height ) * m_TexSizeInv;

			texCoords = pQuad->texCoords;

			texCoords[0].Set( tl, tt );
			texCoords[1].Set( tr, tt );
			texCoords[2].Set( tr, tb );
			texCoords[3].Set( tl, tb );

			//`Wi߂
			pos.x += fontGlyph.cellWidth;

			//eNX`Wi߂
			m_TexPos.x += ( fontGlyph.width + TextPrinter::TEX_MARGIN );
			m_TexLine = max( m_TexLine, ( m_TexPos.y + fontGlyph.height + TextPrinter::TEX_MARGIN ) );
		}
	}

	return True;
}

Boolean TextPrinter::AppendString( Mix::Graphics::Utility::IFont* pFont, const wchar_t* pStr, UInt32 count )
{
	MIX_ASSERT( m_TexSize > 0 );

	if( ( pFont == NULL ) ||
		( pStr == NULL ) ||
		( ::wcslen( pStr ) == 0 ) ||
		( count == 0 ) )
	{
		return False;
	}

	TextPrinter::PAGE* pPage = NULL;
	TextPrinter::SECTION* pSect = NULL;
	TextPrinter::QUAD_GROUP* pQuadGroup = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pPage = GetCurrentPage();
	if( pPage == NULL )
	{
		pPage = AddPage( pFont );
		if( pPage == NULL )
		{
			return False;
		}
	}

	pSect = AddSection( pFont );
	if( pSect == NULL )
	{
		return False;
	}

	m_QuadGroupList.Clear();
	m_QuadList.Clear();

	pQuadGroup = m_QuadGroupList.Add();
	if( pQuadGroup != NULL )
	{
		pQuadGroup->pTexture = pPage->pTexture;
		pQuadGroup->quadStart = 0;
		pQuadGroup->quadCount = 0;
	}
	else
	{
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( UInt32 i = 0; ( i < count ) && ( pStr[i] != L'\0' ); i++ )
	{
		wchar_t code = pStr[i];

		Mix::Graphics::Utility::IFont::GLYPH fontGlyph;
		TextPrinter::QUAD* pQuad;
		Float32 tl, tt, tr, tb;
		Mix::Vector2* points;
		Mix::Vector2* texCoords;

		if( ( code == L' ' ) ||
			( code == L'@' ) ||
			( code == L'\t' ) ||
			( code == L'\r' ) ||
			( code == L'\n' ) )
		{
			tl = 0.0f;
			tt = 0.0f;
			tr = 0.0f;
			tb = 0.0f;
		}
		else if( pFont->GetGlyph( code, fontGlyph ) == True )
		{
			TextPrinter::CHARACTER* pChar;

			if( ( m_TexPos.x + fontGlyph.width + TextPrinter::TEX_MARGIN ) >= m_TexSize )
			{
				//͂ݏo
				m_TexPos.x = TextPrinter::TEX_MARGIN;
				m_TexPos.y = m_TexLine;
			}

			if( ( m_TexPos.y + fontGlyph.height + TextPrinter::TEX_MARGIN ) >= m_TexSize )
			{
				//c͂ݏo
				m_TexPos.x = TextPrinter::TEX_MARGIN;
				m_TexPos.y = TextPrinter::TEX_MARGIN;
				m_TexLine = TextPrinter::TEX_MARGIN;

				pPage = AddPage( pFont );
				if( pPage == NULL )
				{
					return False;
				}

				pSect = AddSection( pFont );
				if( pSect == NULL )
				{
					return False;
				}

				pQuadGroup = m_QuadGroupList.Add();
				if( pQuadGroup != NULL )
				{
					pQuadGroup->pTexture = pPage->pTexture;
					pQuadGroup->quadStart = m_QuadList.GetCount();
					pQuadGroup->quadCount = 0;
				}
				else
				{
					return False;
				}
			}

			//LN^[ǉ
			pChar = m_CharList.Add();
			pChar->texPos = m_TexPos;
			pChar->fontGlyph = fontGlyph;
			pSect->charCount++;

			//eNX`W
			tl = static_cast<Float32>( m_TexPos.x ) * m_TexSizeInv;
			tt = static_cast<Float32>( m_TexPos.y ) * m_TexSizeInv;
			tr = static_cast<Float32>( m_TexPos.x + fontGlyph.width ) * m_TexSizeInv;
			tb = static_cast<Float32>( m_TexPos.y + fontGlyph.height ) * m_TexSizeInv;

			//eNX`Wi߂
			m_TexPos.x += ( fontGlyph.width + TextPrinter::TEX_MARGIN );
			m_TexLine = max( m_TexLine, ( m_TexPos.y + fontGlyph.height + TextPrinter::TEX_MARGIN ) );
		}
		else
		{
			tl = 0.0f;
			tt = 0.0f;
			tr = 0.0f;
			tb = 0.0f;
		}

		//Nbhǉ
		pQuad = m_QuadList.Add();
		pQuadGroup->quadCount++;

		points = pQuad->points;
		texCoords = pQuad->texCoords;

		points[0].Set( 0.0f, 0.0f );
		points[1].Set( 0.0f, 0.0f );
		points[2].Set( 0.0f, 0.0f );
		points[3].Set( 0.0f, 0.0f );

		texCoords[0].Set( tl, tt );
		texCoords[1].Set( tr, tt );
		texCoords[2].Set( tr, tb );
		texCoords[3].Set( tl, tb );
	}

	return True;
}

const Mix::Container<TextPrinter::QUAD_GROUP>& TextPrinter::GetQuadGroups( void ) const
{
	return m_QuadGroupList;
}

const Mix::Container<TextPrinter::QUAD>& TextPrinter::GetQuads( void ) const
{
	return m_QuadList;
}

void TextPrinter::Execute( void )
{
	if( m_PageCount == 0 )
	{
		return;
	}

	const TextPrinter::PAGE* pPage = m_PageList.GetConstBeginPtr();
	const TextPrinter::PAGE* pPageEnd = pPage + m_PageCount;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// eNX`ɏ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	while( pPage != pPageEnd )
	{
		Mix::Graphics::TEXTURE_LOCK_INFO lockInfo;

		if( pPage->pTexture->Lock( lockInfo ) == True )
		{
			TextPrinter::SECTION* pSect = &( m_SectList[pPage->sectStart] );
			TextPrinter::SECTION* pSectEnd = pSect + pPage->sectCount;

			UInt32* pTexBits = static_cast<UInt32*>( lockInfo.bits );
			UInt32 dstPitch = ( lockInfo.pitch >> 2 );

			while( pSect != pSectEnd )
			{
				const TextPrinter::CHARACTER* pChar = &( m_CharList[pSect->charStart] );
				const TextPrinter::CHARACTER* pCharEnd = pChar + pSect->charCount;

				while( pChar != pCharEnd )
				{
					const Mix::Point& texPos = pChar->texPos;
					const Mix::Graphics::Utility::IFont::GLYPH& fontGlyph = pChar->fontGlyph;

					UInt32* pDstBase;
					UInt32* pDst;
					UInt32* pDstEnd;

					const UInt8* pSrcBase;
					const UInt8* pSrc;
					UInt32 srcPitch;

					Int32 y;

					//NA
					pDstBase = pTexBits + ( ( texPos.y - TextPrinter::TEX_MARGIN ) * dstPitch ) + ( texPos.x - TextPrinter::TEX_MARGIN );
					for( y = 0; y < ( fontGlyph.height + TextPrinter::TEX_MARGIN2 ); y++ )
					{
						pDst = pDstBase;
						pDstEnd = ( pDst + ( fontGlyph.width + TextPrinter::TEX_MARGIN2 ) );

						while( pDst != pDstEnd ) { *pDst++ = 0x00000000; }

						pDstBase += dstPitch;
					}

					//
					pSrcBase = static_cast<const UInt8*>( fontGlyph.pImg );
					srcPitch = fontGlyph.imgPitch;
					pDstBase = pTexBits + ( texPos.y * dstPitch ) + texPos.x;
					for( y = 0; y < fontGlyph.height; y++ )
					{
						pSrc = pSrcBase;
						pDst = pDstBase;
						pDstEnd = ( pDst + fontGlyph.width );

#if 1
						//LoX̃uh^CvΉ
						while( pDst != pDstEnd )
						{
							if( *pSrc > 0 ) { *pDst = ( ( ( *pSrc ) << 24 ) | 0x00FFFFFF ); }

							pSrc++;
							pDst++;
						}
#else
						while( pDst != pDstEnd ) { *pDst++ = ( ( ( *pSrc++ ) << 24 ) | 0x00FFFFFF ); }
#endif

						pSrcBase += srcPitch;
						pDstBase += dstPitch;
					}

					pChar++;
				}

				MIX_RELEASE( pSect->pFont );

				pSect++;
			}

			pPage->pTexture->Unlock();
		}

		pPage++;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// NA
	////////////////////////////////////////////////////////////////////////////////////////////////////

	m_TexPos = Mix::Point( TextPrinter::TEX_MARGIN, TextPrinter::TEX_MARGIN );
	m_TexLine = TextPrinter::TEX_MARGIN;

	m_PageCount = 0;
	m_SectList.Clear();
	m_CharList.Clear();

	m_QuadGroupList.Clear();
	m_QuadList.Clear();
}

UInt32 TextPrinter::GetTextureCount( void ) const
{
	return m_PageList.GetCount();
}

Mix::Graphics::ITexture* TextPrinter::GetTexturePtr( UInt32 index ) const
{
	if( m_PageList.GetCount() <= index )
	{
		return NULL;
	}

	return m_PageList[index].pTexture;
}

TextPrinter::PAGE* TextPrinter::GetCurrentPage( void )
{
	if( m_PageCount == 0 )
	{
		return NULL;
	}

	return &( m_PageList[m_PageCount - 1] );
}

TextPrinter::PAGE* TextPrinter::AddPage( Mix::Graphics::Utility::IFont* pFont )
{
	TextPrinter::PAGE* pPage;
	TextPrinter::SECTION* pSect;

	if( m_PageCount == m_PageList.GetCount() )
	{
		Mix::Graphics::ITexture* pTexture = CreateTexture();
		if( pTexture == NULL )
		{
			MIX_LOG_ERROR( L"%s : y[W̃eNX`쐬ł܂ł : Name[%s]", TextPrinter::FAILED_PROC, m_Name.GetConstPtr() );
			return NULL;
		}

		pPage = m_PageList.Add();
		if( pPage == NULL )
		{
			MIX_LOG_ERROR( L"%s : %s : Name[%s]", TextPrinter::FAILED_PROC, Mix::STR_OUTOFMEMORY, m_Name.GetConstPtr() );
			MIX_RELEASE( pTexture );
			return NULL;
		}

		pPage->pTexture = pTexture;
	}
	else
	{
		pPage = &( m_PageList[m_PageCount] );
	}

	//y[W̏
	pPage->sectStart = m_SectList.GetCount();
	pPage->sectCount = 0;
	m_PageCount++;

	//ZNV̒ǉ
	pSect = m_SectList.Add();
	if( pSect != NULL )
	{
		MIX_ADD_REF( pFont );

		pSect->pFont = pFont;
		pSect->charStart = m_CharList.GetCount();
		pSect->charCount = 0;

		pPage->sectCount = 1;
	}

	return pPage;
}

TextPrinter::SECTION* TextPrinter::AddSection( Mix::Graphics::Utility::IFont* pFont )
{
	MIX_ASSERT( pFont != NULL );
	MIX_ASSERT( m_PageCount > 0 );

	if( ( m_SectList.GetCount() > 0 ) &&
		( m_SectList[m_SectList.GetCount() - 1].pFont == pFont ) )
	{
		return &( m_SectList[m_SectList.GetCount() - 1] );
	}

	TextPrinter::PAGE* pPage = &( m_PageList[m_PageCount - 1] );

	TextPrinter::SECTION* pSect = m_SectList.Add();
	if( pSect == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", TextPrinter::FAILED_PROC, Mix::STR_OUTOFMEMORY, m_Name.GetConstPtr() );
		return NULL;
	}

	//ZNVCNg
	pPage->sectCount++;

	//ZNV
	MIX_ADD_REF( pFont );
	pSect->pFont = pFont;
	pSect->charStart = m_CharList.GetCount();
	pSect->charCount = 0;

	return pSect;
}

Mix::Graphics::ITexture* TextPrinter::CreateTexture( void )
{
	Mix::StringW tempStr;
	Mix::Graphics::ITexture* pTexture = NULL;

	tempStr.Sprintf( L"TextPrinter\\%s\\Texture%d", m_Name.GetConstPtr(), m_PageList.GetCount() );
	if( m_pDevice->CreateDynamicTexture( m_TexSize, m_TexSize, Mix::Graphics::FMT_R8G8B8A8, &pTexture, tempStr.GetConstPtr() ) == False )
	{
		return NULL;
	}

	return pTexture;
}

}}}}
