#include "Mix/Class/Graphics/DX9/VertexBuffer.h"
#include "Mix/Class/Graphics/DX9/Device.h"

namespace Mix{ namespace Graphics{ namespace DX9{

VertexBuffer* VertexBuffer::CreateInstance( Mix::Graphics::DX9::Device* pDevice, UInt32 count, UInt32 stride, UInt32 resizeStep, Boolean bDynamic, IDirect3DVertexBuffer9* pVertexBuffer, const wchar_t* pName )
{
	return new VertexBuffer( pDevice, count, stride, resizeStep, bDynamic, pVertexBuffer, pName );
}

VertexBuffer::VertexBuffer( Mix::Graphics::DX9::Device* pDevice, UInt32 count, UInt32 stride, UInt32 resizeStep, Boolean bDynamic, IDirect3DVertexBuffer9* pVertexBuffer, const wchar_t* pName ) :
DeviceObject( pDevice, Mix::Graphics::Common::DOT_VERTEXBUFFER, False, pName ),
m_pDevice( pDevice ),
m_pVertexBuffer( pVertexBuffer ),
m_pBuffer( NULL ),
m_Stride( stride ),
m_Capacity( count ),
m_Count( 0 ),
m_ResizeStep( resizeStep ),
m_bDynamic( bDynamic ),
m_LockFlags( 0 ),
m_pSource( NULL ),
m_bResize( False )
{
	MIX_ADD_REF( m_pDevice );

	m_LockFlags = ( m_bDynamic == True )? D3DLOCK_DISCARD : 0;
}

VertexBuffer::~VertexBuffer( void )
{
	Mix::Memory::Free( m_pBuffer );
	MIX_RELEASE( m_pVertexBuffer );
	MIX_RELEASE( m_pDevice );
}

HRESULT VertexBuffer::Initialize( const void* pInitialData )
{
	UInt32 buffSize = m_Stride * m_Capacity;

	//obt@쐬
	m_pBuffer = static_cast<UInt8*>( Mix::Memory::Allocate( buffSize ) );
	if( m_pBuffer == NULL )
	{
		return E_OUTOFMEMORY;
	}

	//obt@
	if( pInitialData != NULL )
	{
		HRESULT ret;
		void* pDst;

		Mix::Memory::Copy( m_pBuffer, pInitialData, buffSize );

		ret = m_pVertexBuffer->Lock( 0, 0, &pDst, m_LockFlags );
		if( ret == D3D_OK )
		{
			Mix::Memory::Copy( pDst, m_pBuffer, buffSize );
			m_pVertexBuffer->Unlock();
		}
		else
		{
			return ret;
		}

		m_Count = m_Capacity;
	}
	else
	{
		Mix::Memory::Zero( m_pBuffer, buffSize );
	}

	return D3D_OK;
}

IDirect3DVertexBuffer9* VertexBuffer::GetInterface( void )
{
	return m_pVertexBuffer;
}

Boolean VertexBuffer::Lock( void )
{
	if( ( m_pVertexBuffer == NULL ) ||
		( m_bDynamic == False ) )
	{
		return False;
	}

	m_pSource = &( m_pBuffer[0] );
	m_Count = 0;

	return True;
}

void VertexBuffer::Unlock( void )
{
	if( ( m_pVertexBuffer == NULL ) ||
		( m_pSource == NULL ) )
	{
		return;
	}

	void* pDst = NULL;

	if( m_bResize == True )
	{
		//_obt@̃TCY

		IDirect3DVertexBuffer9* pTempVB = NULL;
		HRESULT hRet;

		hRet = m_pDevice->CreateVertexBuffer( m_Capacity, m_Stride, m_bDynamic, &pTempVB );
		if( hRet == D3D_OK )
		{
			MIX_RELEASE( m_pVertexBuffer );
			m_pVertexBuffer = pTempVB;

			MIX_LOG_INFO( L"o[ebNXobt@̃TCYύX : Name[%s] Count[%d] Size[%dByte]",
				GetName(),
				m_Capacity,
				( m_Stride * m_Capacity ) );
		}
		else
		{
			MIX_LOG_ERROR( L"o[ebNXobt@̃TCY̕ύXɎs : Name[%s] %s",
				GetName(),
				m_pDevice->GetD3DResultText( hRet ) );
		}

		m_bResize = False;
	}

	//Rs[
	if( m_pVertexBuffer->Lock( 0, 0, &pDst, m_LockFlags ) == D3D_OK )
	{
		UInt32 copyCount = ( m_Capacity > m_Count )? m_Count : m_Capacity;
		Mix::Memory::Copy( pDst, m_pBuffer, ( m_Stride * copyCount ) );
		m_pVertexBuffer->Unlock();
	}

	m_pSource = NULL;
}

Boolean VertexBuffer::Push( const void* pData, UInt32 count )
{
	void* pDst = Push( count );
	if( pDst == NULL )
	{
		return False;
	}

	::CopyMemory( pDst, pData, ( m_Stride * count ) );

	return True;
}

void* VertexBuffer::Push( UInt32 count )
{
	if( m_pSource == NULL )
	{
		return NULL;
	}

	UInt32 size;
	UInt32 preCur;

	size = ( m_Count + count );

	if( m_ResizeStep > 0 )
	{
		if( size <= m_Capacity )
		{
			preCur = m_Count;
			m_Count += count;
		}
		else
		{
			UInt32 newCount;
			
			newCount = ( size / m_ResizeStep );
			if( ( size % m_ResizeStep ) > 0 )
			{
				newCount++;
			}

			newCount *= m_ResizeStep;

			if( Mix::Memory::Reallocate( reinterpret_cast<void**>( &m_pBuffer ), ( newCount * m_Stride ) ) == True )
			{
				m_pSource = &( m_pBuffer[0] );
				m_Capacity = newCount;

				preCur = m_Count;
				m_Count += count;

				m_bResize = True;
			}
			else
			{
				return NULL;
			}
		}
	}
	else
	{
		if( size <= m_Capacity )
		{
			preCur = m_Count;
			m_Count += count;
		}
		else
		{
			return NULL;
		}
	}

	return ( m_pSource + ( m_Stride * preCur ) );
}

void VertexBuffer::Clear( void )
{
	m_Count = 0;
}

UInt32 VertexBuffer::GetStride( void ) const
{
	return m_Stride;
}

UInt32 VertexBuffer::GetCount( void ) const
{
	return m_Count;
}

UInt32 VertexBuffer::GetCapacity( void ) const
{
	return m_Capacity;
}

void VertexBuffer::Invalidate( void )
{
	MIX_RELEASE( m_pVertexBuffer );
}

void VertexBuffer::Validate( void )
{
	if( ( m_pDevice == NULL ) ||
		( m_pVertexBuffer != NULL ) )
	{
		return;
	}

	HRESULT ret;

	//_obt@쐬
	ret = m_pDevice->CreateVertexBuffer( m_Capacity, m_Stride, m_bDynamic, &m_pVertexBuffer );
	if( ret == D3D_OK )
	{
		void* pDst = NULL;

		if( m_pVertexBuffer->Lock( 0, 0, &pDst, m_LockFlags ) == D3D_OK )
		{
			::CopyMemory( pDst, m_pBuffer, ( m_Stride * m_Capacity ) );
			m_pVertexBuffer->Unlock();
		}
	}
	else
	{
		MIX_LOG_ERROR( L"o[ebNXobt@̕AɎs : Name[%s] Result[%s]", GetName(), m_pDevice->GetD3DResultText( ret ) );
	}
}

}}}
