#include "Mix/Class/Graphics/DX9/IndexBuffer.h"
#include "Mix/Class/Graphics/DX9/Device.h"

namespace Mix{ namespace Graphics{ namespace DX9{

IndexBuffer* IndexBuffer::CreateInstance(	Mix::Graphics::DX9::Device* pDevice,
											Mix::Graphics::INDEX_TYPE type,
											D3DFORMAT format,
											UInt32 count,
											UInt32 resizeStep,
											Boolean bDynamic,
											IDirect3DIndexBuffer9* pIndexBuffer,
											const wchar_t* pName )
{
	return new IndexBuffer( pDevice, type, format, count, resizeStep, bDynamic, pIndexBuffer, pName );
}

IndexBuffer::IndexBuffer(	Mix::Graphics::DX9::Device* pDevice,
							Mix::Graphics::INDEX_TYPE type,
							D3DFORMAT format,
							UInt32 count,
							UInt32 resizeStep,
							Boolean bDynamic,
							IDirect3DIndexBuffer9* pIndexBuffer,
							const wchar_t* pName ) :
DeviceObject( pDevice, Mix::Graphics::Common::DOT_INDEXBUFFER, False, pName ),
m_pDevice( pDevice ),
m_pIndexBuffer( pIndexBuffer ),
m_pBuffer( NULL ),
m_Type( type ),
m_Format( format ),
m_Stride( ( type == Mix::Graphics::INDEX_USHORT )? 2 : 4 ),
m_Capacity( count ),
m_ResizeStep( resizeStep ),
m_bDynamic( bDynamic ),
m_LockFlags( 0 ),
m_Count( 0 ),
m_pSource( NULL ),
m_bResize( False )
{
	MIX_ADD_REF( m_pDevice );

	m_LockFlags = ( m_bDynamic == True )? D3DLOCK_DISCARD : 0;
}

IndexBuffer::~IndexBuffer( void )
{
	Mix::Memory::Free( m_pBuffer );
	MIX_RELEASE( m_pIndexBuffer );
	MIX_RELEASE( m_pDevice );
}

HRESULT IndexBuffer::Initialize( const void* pInitialData )
{
	UInt32 buffSize = m_Stride * m_Capacity;

	m_pBuffer = static_cast<UInt8*>( Mix::Memory::Allocate( buffSize ) );
	if( m_pBuffer == NULL )
	{
		return E_OUTOFMEMORY;
	}

	if( pInitialData != NULL )
	{
		HRESULT ret;
		void* pDst;

		Mix::Memory::Copy( m_pBuffer, pInitialData, buffSize );

		ret = m_pIndexBuffer->Lock( 0, 0, &pDst, m_LockFlags );
		if( ret == D3D_OK )
		{
			Mix::Memory::Copy( pDst, m_pBuffer, buffSize );
			m_pIndexBuffer->Unlock();
		}
		else
		{
			return ret;
		}

		m_Count = m_Capacity;
	}
	else
	{
		Mix::Memory::Zero( m_pBuffer, buffSize );
	}

	return D3D_OK;
}

IDirect3DIndexBuffer9* IndexBuffer::GetInterface( void )
{
	return m_pIndexBuffer;
}

void IndexBuffer::Invalidate( void )
{
	MIX_RELEASE( m_pIndexBuffer );
}

void IndexBuffer::Validate( void )
{
	if( ( m_pDevice == NULL ) ||
		( m_pIndexBuffer != NULL ) )
	{
		return;
	}

	HRESULT ret;

	//CfbNXobt@쐬
	ret = m_pDevice->CreateIndexBuffer( m_Format, m_Capacity, m_bDynamic, &m_pIndexBuffer );
	if( ret == D3D_OK )
	{
		void* pDst = NULL;

		if( m_pIndexBuffer->Lock( 0, 0, &pDst, m_LockFlags ) == D3D_OK )
		{
			::CopyMemory( pDst, m_pBuffer, ( m_Stride * m_Capacity ) );
			m_pIndexBuffer->Unlock();
		}
	}
	else
	{
		MIX_LOG_ERROR( L"CfbNXobt@̕AɎs : Name[%s] Result[%s]", GetName(), m_pDevice->GetD3DResultText( ret ) );
	}
}

Boolean IndexBuffer::Lock( void )
{
	if( ( m_bDynamic == False ) ||
		( m_pIndexBuffer == NULL ) )
	{
		return False;
	}

	m_pSource = &( m_pBuffer[0] );
	m_Count = 0;

	return True;
}

void IndexBuffer::Unlock( void )
{
	if( ( m_pIndexBuffer == NULL ) ||
		( m_pSource == NULL ) )
	{
		return;
	}

	void* pDst = NULL;

	if( m_bResize == True )
	{
		HRESULT ret;
		IDirect3DIndexBuffer9* pTempIB = NULL;

		ret = m_pDevice->CreateIndexBuffer( m_Format, m_Capacity, m_bDynamic, &pTempIB );
		if( ret == D3D_OK )
		{
			MIX_RELEASE( m_pIndexBuffer );
			m_pIndexBuffer = pTempIB;

			MIX_LOG_INFO( L"CfbNXobt@̃TCYύX : Name[%s] Count[%d] Size[%dByte]",
				GetName(),
				m_Capacity,
				( m_Stride * m_Capacity ) );
		}
		else
		{
			MIX_LOG_ERROR( L"CfbNXobt@̃TCYύXɎs : Name[%s] Result[%s]",
				GetName(),
				m_pDevice->GetD3DResultText( ret ) );
		}

		m_bResize = False;
	}

	if( m_pIndexBuffer->Lock( 0, 0, &pDst, m_LockFlags ) == D3D_OK )
	{
		UInt32 copyCount = ( m_Capacity > m_Count )? m_Count : m_Capacity;
		Mix::Memory::Copy( pDst, m_pBuffer, ( m_Stride * copyCount ) );
		m_pIndexBuffer->Unlock();
	}

	m_pSource = NULL;
}

Boolean IndexBuffer::Push( const void* pData, UInt32 count )
{
	void* pDst = Push( count );
	if( pDst == NULL )
	{
		return False;
	}

	::CopyMemory( pDst, pData, ( m_Stride * count ) );

	return True;
}

void* IndexBuffer::Push( UInt32 count )
{
	if( m_pSource == NULL )
	{
		return NULL;
	}

	UInt32 size;
	UInt32 preCur;

	size = ( m_Count + count );

	if( m_ResizeStep > 0 )
	{
		if( size <= m_Capacity )
		{
			preCur = m_Count;
			m_Count += count;
		}
		else
		{
			UInt32 newCount;
			
			newCount = ( size / m_ResizeStep );
			if( ( size % m_ResizeStep ) > 0 )
			{
				newCount++;
			}

			newCount *= m_ResizeStep;

			if( Mix::Memory::Reallocate( reinterpret_cast<void**>( &m_pBuffer ), ( newCount * 2 ) ) == True )
			{
				m_pSource = &( m_pBuffer[0] );
				m_Capacity = newCount;

				preCur = m_Count;
				m_Count += count;

				m_bResize = True;
			}
			else
			{
				return NULL;
			}
		}
	}
	else
	{
		if( size <= m_Capacity )
		{
			preCur = m_Count;
			m_Count += count;
		}
		else
		{
			return NULL;
		}
	}

	return ( m_pSource + ( preCur * m_Stride ) );
}

void IndexBuffer::Clear( void )
{
	m_Count = 0;
}

Mix::Graphics::INDEX_TYPE IndexBuffer::GetType( void ) const
{
	return m_Type;
}

UInt32 IndexBuffer::GetCount( void ) const
{
	return m_Count;
}

UInt32 IndexBuffer::GetCapacity( void ) const
{
	return m_Capacity;
}

}}}
