#include "Mix/Class/Graphics/DX9/Device.h"

#include "Mix/ScopedLock.h"
#include "Mix/File/IManager.h"
#include "Mix/File/IReader.h"
#include "Mix/File/IWriter.h"
#include "Mix/File/ILoader.h"
#include "Mix/Memory/IBuffer.h"

#include "Mix/Class/UserFile.h"
#include "Mix/Class/Graphics/DX9/VertexLayout.h"
#include "Mix/Class/Graphics/DX9/VertexShader.h"
#include "Mix/Class/Graphics/DX9/PixelShader.h"
#include "Mix/Class/Graphics/DX9/VertexBuffer.h"
#include "Mix/Class/Graphics/DX9/IndexBuffer.h"
#include "Mix/Class/Graphics/DX9/Texture2D.h"
#include "Mix/Class/Graphics/DX9/TextureCube.h"
#include "Mix/Class/Graphics/DX9/TextureTarget.h"
#include "Mix/Class/Graphics/DX9/TextureDepth.h"
#include "Mix/Class/Graphics/DX9/Query.h"

namespace Mix{ namespace Graphics{ namespace DX9{

const wchar_t* Device::FAILED_INITIALIZE            = L"OtBbNXfoCX( DirectX9 )̏Ɏs";
const wchar_t* Device::FAILED_COMPILEVERTEXSHADER   = L"o[ebNXVF[_[̃RpCɎs";
const wchar_t* Device::FAILED_COMPILEPIXELSHADER    = L"sNZVF[_[̃RpCɎs";
const wchar_t* Device::FAILED_CREATEVERTEXLAYOUT    = L"o[ebNXCAEg̍쐬Ɏs";
const wchar_t* Device::FAILED_CREATEVERTEXSHADER    = L"o[ebNXVF[_[̍쐬Ɏs";
const wchar_t* Device::FAILED_CREATEPIXELSHADER     = L"sNZVF[_[̍쐬Ɏs";
const wchar_t* Device::FAILED_CREATEVERTEXBUFFER    = L"o[ebNXobt@̍쐬Ɏs";
const wchar_t* Device::FAILED_CREATEINDEXBUFFER     = L"CfbNXobt@̍쐬Ɏs";
const wchar_t* Device::FAILED_CREATETEXTURE         = L"eNX`̍쐬Ɏs";
const wchar_t* Device::FAILED_CREATECUBETEXTURE     = L"L[ueNX`̍쐬Ɏs";
const wchar_t* Device::FAILED_CREATEDYNAMICTEXTURE  = L"_Ci~bNeNX`̍쐬Ɏs";
const wchar_t* Device::FAILED_CREATETARGETTEXTURE   = L"^[QbgeNX`̍쐬Ɏs";
const wchar_t* Device::FAILED_CREATEDEPTHTEXTURE    = L"fvXeNX`̍쐬Ɏs";
const wchar_t* Device::FAILED_CREATEQUERY           = L"NG̍쐬Ɏs";
const wchar_t* Device::FAILED_DRAW                  = L"`( Draw )Ɏs";
const wchar_t* Device::FAILED_DRAWINDEXED           = L"`( DrawIndexed )Ɏs";
const wchar_t* Device::FAILED_SAVESCREENSHOT        = L"XN[Vbg̕ۑɎs";
const wchar_t* Device::FAILED_SAVETEXTURE           = L"eNX`̕ۑɎs";

//D3D : o[ebNXCAEg̃Z}eBbN^Cve[u
const UInt8 Device::D3DVLSemanticTypeTable[Mix::Graphics::VERTEX_ELEMENT_SEMANTIC_MAX] =
{
	D3DDECLUSAGE_POSITION,
	D3DDECLUSAGE_BLENDWEIGHT,
	D3DDECLUSAGE_BLENDINDICES,
	D3DDECLUSAGE_NORMAL,
	D3DDECLUSAGE_PSIZE,
	D3DDECLUSAGE_TEXCOORD,
	D3DDECLUSAGE_TANGENT,
	D3DDECLUSAGE_BINORMAL,
	D3DDECLUSAGE_COLOR,
};

//D3D : o[ebNXCXg̃tH[}bge[u
const UInt8 Device::D3DVLFormatTable[Mix::Graphics::VERTEX_ELEMENT_FORMAT_MAX] =
{
	D3DDECLTYPE_FLOAT1,
	D3DDECLTYPE_FLOAT2,
	D3DDECLTYPE_FLOAT3,
	D3DDECLTYPE_FLOAT4,
	D3DDECLTYPE_UBYTE4,
	D3DDECLTYPE_SHORT2,
	D3DDECLTYPE_SHORT4,
	D3DDECLTYPE_UBYTE4N,
	D3DDECLTYPE_SHORT2N,
	D3DDECLTYPE_SHORT4N,
	D3DDECLTYPE_USHORT2N,
	D3DDECLTYPE_USHORT4N,
	D3DDECLTYPE_FLOAT16_2,
	D3DDECLTYPE_FLOAT16_4,
};

//D3D : tH[}bgeLXg
const wchar_t* Device::D3DFormatTextArray[20] =
{
	L"UNKNOWN",
	L"X8R8G8B8",
	L"A8R8G8B8",
	L"X1R5G5B5",
	L"R5G6B5",
	L"A8",
	L"R16F",
	L"R32F",
	L"G16R16F",
	L"G32R32F",
	L"A16B16G16R16F",
	L"A32B32G32R32F",
	L"DXT1",
	L"DXT2",
	L"DXT3",
	L"DXT4",
	L"DXT5",
	L"D16",
	L"D32",
	L"D24S8",
};

//D3D : UgeLXgz
const wchar_t* Device::D3DResultTextArray[7] =
{
	L"sȃG[܂",
	L"TuVXeŌs̃G[܂",
	L"foCX́AƉꂽeNjbNT|[gĂ܂",
	L"rfIsĂ܂",
	L"\bȟĂяoł( p[^ɖȒlݒ肳Ă\܂ )",
	L"f[^ł",
	L"VXesĂ܂",
};

//D3D : JO[he[u
const UInt32 Device::D3DCullModeTable[] =
{
	D3DCULL_NONE,
	D3DCULL_CW,
	D3DCULL_CCW,
};

//D3D : Zt@NVe[u
const UInt32 Device::D3DZFuncTable[Mix::Graphics::ZFUNC_MAX] =
{
	D3DCMP_ALWAYS,
	D3DCMP_NEVER,
	D3DCMP_LESS,
	D3DCMP_EQUAL,
	D3DCMP_LESSEQUAL,
	D3DCMP_GREATER,
	D3DCMP_NOTEQUAL,
	D3DCMP_GREATEREQUAL,
};

//D3D : v~eBu^Cve[u
const D3DPRIMITIVETYPE Device::D3DPrimitiveTypeTable[Mix::Graphics::PT_MAX] =
{
	D3DPT_POINTLIST,
	D3DPT_LINELIST,
	D3DPT_LINESTRIP,
	D3DPT_TRIANGLELIST,
	D3DPT_TRIANGLESTRIP,
};

//D3D : NG^Cve[u
const D3DQUERYTYPE Device::D3DQueryTypeTable[Mix::Graphics::QT_MAX] =
{
	D3DQUERYTYPE_EVENT,
	D3DQUERYTYPE_OCCLUSION,
};

//D3D : NGe[u
const wchar_t* Device::D3DQueryTextTable[Mix::Graphics::QT_MAX] =
{
	L"EVENT",
	L"OCCLUSION",
};

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Device
/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Device* Device::CreateInstance( Boolean bWaitVSync )
{
	return new Device( bWaitVSync );
}

Device::Device( Boolean bWaitVSync ) : Common::Device( bWaitVSync ),
m_pD3D( NULL ),
m_pD3DDevice( NULL ),
m_pD3DBackBuffer( NULL ),
m_Status( STATUS_OK ),
m_DrawPrimitiveCount( 0 )
{
	UInt32 i;

	::ZeroMemory( &m_Caps, sizeof( m_Caps ) );

	//r[|[g
	m_Viewport.X = 0;
	m_Viewport.Y = 0;
	m_Viewport.Width = 0;
	m_Viewport.Height = 0;
	m_Viewport.MinZ = 0.0f;
	m_Viewport.MaxZ = 1.0f;

	//X^CUXe[g
	m_RasterizeDesc.fillMode = Mix::Graphics::FILL_SOLID;
	m_RasterizeDesc.cullMode = Mix::Graphics::CULL_BACK;
	m_RasterizeDesc.bScissoring = False;
	m_RasterizeDesc.bMultisample = True;

	//fvXXe[g
	m_DepthDesc.bTest = True;
	m_DepthDesc.bWrite = True;
	m_DepthDesc.func = Mix::Graphics::ZFUNC_LESSEQUAL;

	//uhXe[g
	m_BlendDesc.type = Mix::Graphics::BLEND_COPY;
	m_BlendDesc.colorWriteMask = Mix::Graphics::COLOR_WRITE_RGBA;

	//^[QbgXe[g
	m_TargetState.rect = Mix::Rectangle( 0, 0, 0, 0 );
	for( i = 0; i < Mix::Graphics::TARGET_MAX; i++ )
	{
		m_TargetState.pTex[i] = NULL;
	}
	m_TargetState.pDSTex = NULL;
}

Device::~Device( void )
{
}

HRESULT Device::CreateVertexBuffer( UInt32 count, UInt32 stride, Boolean bDynamic, IDirect3DVertexBuffer9** ppVertexBuffer )
{
	UInt32 size = ( stride * count );
	UInt32 usage = ( bDynamic == True )? D3DUSAGE_DYNAMIC : 0;
	D3DPOOL poolArray[] = { D3DPOOL_DEFAULT, D3DPOOL_SYSTEMMEM, };
	UInt32 numPool = ( sizeof( poolArray ) / sizeof( D3DPOOL ) );
	UInt32 i;
	HRESULT hRet;

	for( i = 0; i < numPool; i++ )
	{
		hRet = m_pD3DDevice->CreateVertexBuffer( size, D3DUSAGE_WRITEONLY | usage, 0, poolArray[i], ppVertexBuffer, NULL );
		if( hRet == D3D_OK )
		{
			break;
		}
	}

	return hRet;
}

HRESULT Device::CreateIndexBuffer( D3DFORMAT format, UInt32 count, Boolean bDynamic, IDirect3DIndexBuffer9** ppIndexBuffer )
{
	if( ( format != D3DFMT_INDEX16 ) &&
		( format != D3DFMT_INDEX32 ) )
	{
		return D3DERR_INVALIDCALL;
	}

	UInt32 stride = ( format == D3DFMT_INDEX16 )? 2 : 4;
	UInt32 size = ( stride * count );
	UInt32 usage = ( bDynamic == True )? D3DUSAGE_DYNAMIC : 0;
	D3DPOOL poolArray[] = { D3DPOOL_DEFAULT, D3DPOOL_SYSTEMMEM, };
	UInt32 numPool = ( sizeof( poolArray ) / sizeof( D3DPOOL ) );
	UInt32 i;
	HRESULT hRet;

	for( i = 0; i < numPool; i++ )
	{
		hRet = m_pD3DDevice->CreateIndexBuffer( size, D3DUSAGE_WRITEONLY | usage, format, poolArray[i], ppIndexBuffer, NULL );
		if( hRet == D3D_OK )
		{
			break;
		}
	}

	return hRet;
}

HRESULT Device::CreateDynamicTexture( UInt32& width, UInt32& height, D3DFORMAT format, IDirect3DTexture9** ppTexture )
{
	D3DPOOL poolArray[] = { D3DPOOL_DEFAULT, D3DPOOL_MANAGED, };
	UInt32 poolNum = ( sizeof( poolArray ) / sizeof( D3DPOOL ) );
	UInt32 usage;
	UInt32 i;
	HRESULT hRet;

	if( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_POW2 ) == D3DPTEXTURECAPS_POW2 )
	{
		//2ׂ̂ɏC
		AdjustTexturePow2( width, height );
	}
	if( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_SQUAREONLY ) == D3DPTEXTURECAPS_SQUAREONLY )
	{
		//`ɏC
		AdjustTextureSquare( width, height );
	}

	for( i = 0; i < poolNum; i++ )
	{
		usage = ( poolArray[i] != D3DPOOL_MANAGED )? D3DUSAGE_DYNAMIC : 0;

		hRet = m_pD3DDevice->CreateTexture(	width,
											height,
											1,
											usage,
											format,
											poolArray[i],
											ppTexture,
											NULL );
		if( hRet == D3D_OK )
		{
			break;
		}
	}

	return hRet;
}

HRESULT Device::CreateTexture2D( const void* pBuffer, UInt32 size, D3DSURFACE_DESC& desc, IDirect3DTexture9** ppTexture )
{
	if( ( pBuffer == NULL ) ||
		( size == 0 ) ||
		( ppTexture == NULL ) )
	{
		return D3DERR_INVALIDCALL;
	}

	HRESULT hRet;
	D3DXIMAGE_INFO info;
	D3DXGetImageInfoFromFileInMemoryFunc D3DXGetImageInfoFromFileInMemory;
	D3DXCreateTextureFromFileInMemoryExFunc D3DXCreateTextureFromFileInMemoryEx;
	IDirect3DTexture9* pD3DTexture;

	D3DXGetImageInfoFromFileInMemory = static_cast<D3DXGetImageInfoFromFileInMemoryFunc>( m_D3DX9Module.GetFunction( L"D3DXGetImageInfoFromFileInMemory" ) );
	D3DXCreateTextureFromFileInMemoryEx = static_cast<D3DXCreateTextureFromFileInMemoryExFunc>( m_D3DX9Module.GetFunction( L"D3DXCreateTextureFromFileInMemoryEx" ) );

	hRet = D3DXGetImageInfoFromFileInMemory( pBuffer, size, &info );
	if( hRet != D3D_OK )
	{
		return hRet;
	}

	hRet = D3DXCreateTextureFromFileInMemoryEx( m_pD3DDevice, pBuffer, size, info.Width, info.Height, info.MipLevels, 0, D3DFMT_UNKNOWN, D3DPOOL_MANAGED, D3DX_FILTER_NONE, D3DX_FILTER_NONE, 0, NULL, NULL, &pD3DTexture );
	if( hRet != D3D_OK )
	{
		return hRet;
	}

	hRet = pD3DTexture->GetLevelDesc( 0, &desc );
	if( hRet != D3D_OK )
	{
		MIX_RELEASE( pD3DTexture );
		return hRet;
	}

	( *ppTexture ) = pD3DTexture;

	return D3D_OK;
}

HRESULT Device::CreateCubeTexture( UInt32 edgeLength, D3DFORMAT format, IDirect3DCubeTexture9** ppTexture )
{
	return m_pD3DDevice->CreateCubeTexture( edgeLength, 1, D3DUSAGE_RENDERTARGET, format, D3DPOOL_DEFAULT, ppTexture, NULL );
}

HRESULT Device::CreateTargetTexture( UInt32& width, UInt32& height, D3DFORMAT format, IDirect3DTexture9** ppTexture )
{
	if( ( m_Caps.MaxTextureWidth < width ) ||
		( m_Caps.MaxTextureHeight < height ) )
	{
		//eNX`TCYOa( AXyNgێ )
		Float32 ratio;

		if( ( m_Caps.MaxTextureWidth < width ) &&
			( m_Caps.MaxTextureHeight >= height ) )
		{
			ratio = static_cast<Float32>( m_Caps.MaxTextureWidth ) / static_cast<Float32>( width );
		}
		else if( ( m_Caps.MaxTextureWidth >= width ) &&
				( m_Caps.MaxTextureHeight < height ) )
		{
			ratio = static_cast<Float32>( m_Caps.MaxTextureHeight ) / static_cast<Float32>( height );
		}
		else
		{
			if( ( width - m_Caps.MaxTextureWidth ) >= ( height - m_Caps.MaxTextureHeight ) )
			{
				ratio = static_cast<Float32>( m_Caps.MaxTextureWidth ) / static_cast<Float32>( width );
			}
			else
			{
				ratio = static_cast<Float32>( m_Caps.MaxTextureHeight ) / static_cast<Float32>( height );
			}
		}

		width = static_cast<UInt32>( static_cast<Float32>( width ) * ratio );
		height = static_cast<UInt32>( static_cast<Float32>( height ) * ratio );
	}

	if( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_POW2 ) == D3DPTEXTURECAPS_POW2 )
	{
		//2ׂ̂ɏC
		AdjustTexturePow2( width, height );
	}
	if( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_SQUAREONLY ) == D3DPTEXTURECAPS_SQUAREONLY )
	{
		//`ɏC
		AdjustTextureSquare( width, height );
	}

	return m_pD3DDevice->CreateTexture( width, height, 1, D3DUSAGE_RENDERTARGET, format, D3DPOOL_DEFAULT, ppTexture, NULL );
}

HRESULT Device::CreateDepthTexture( UInt32 width, UInt32 height, D3DFORMAT format, IDirect3DSurface9** ppSurface )
{
	return m_pD3DDevice->CreateDepthStencilSurface( width, height, format, D3DMULTISAMPLE_NONE, 0, FALSE, ppSurface, NULL );
}

HRESULT Device::CreateQuery( Mix::Graphics::QUERY_TYPE type, IDirect3DQuery9** ppQuery )
{
	return m_pD3DDevice->CreateQuery( Device::D3DQueryTypeTable[type], ppQuery );
}

Mix::Graphics::ITexture* Device::CreateTexture( const void* pBuffer, UInt32 size, const wchar_t* pName, const wchar_t* pNameLabel, D3DSURFACE_DESC& desc )
{
	D3DXGetImageInfoFromFileInMemoryFunc D3DXGetImageInfoFromFileInMemory = static_cast<D3DXGetImageInfoFromFileInMemoryFunc>( m_D3DX9Module.GetFunction( L"D3DXGetImageInfoFromFileInMemory" ) );

	HRESULT hRet;
	D3DXIMAGE_INFO info;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// eNX`̏擾
	////////////////////////////////////////////////////////////////////////////////////////////////////

	hRet = D3DXGetImageInfoFromFileInMemory( pBuffer, size, &info );
	if( hRet != D3D_OK )
	{
		return NULL;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( info.ResourceType == D3DRTYPE_TEXTURE )
	{
		D3DXCreateTextureFromFileInMemoryExFunc D3DXCreateTextureFromFileInMemoryEx = static_cast<D3DXCreateTextureFromFileInMemoryExFunc>( m_D3DX9Module.GetFunction( L"D3DXCreateTextureFromFileInMemoryEx" ) );
		IDirect3DTexture9* pD3DTexture = NULL;
		Mix::Graphics::DX9::Texture2D* pTexture = NULL;

		hRet = D3DXCreateTextureFromFileInMemoryEx( m_pD3DDevice,
													pBuffer,
													size,
													info.Width,
													info.Height,
													info.MipLevels,
													0,
													D3DFMT_UNKNOWN,
													D3DPOOL_MANAGED,
													D3DX_FILTER_NONE,
													D3DX_FILTER_NONE,
													0,
													NULL,
													NULL,
													&pD3DTexture );
		if( hRet != D3D_OK )
		{
			MIX_LOG_ERROR( L"%s : D3DXCreateTextureFromFileInMemoryEx %s : %s[%s] Result[%s]", FAILED_CREATETEXTURE, Mix::STR_RETERROR, pNameLabel,
				pName,
				GetD3DResultText( hRet ) );

			return NULL;
		}

		hRet = pD3DTexture->GetLevelDesc( 0, &desc );
		if( hRet != D3D_OK )
		{
			MIX_LOG_ERROR( L"%s : IDirect3DTexture9::GetLevelDesc %s : %s[%s] Result[%s]", FAILED_CREATETEXTURE, Mix::STR_RETERROR, pNameLabel,
				pName,
				GetD3DResultText( hRet ) );

			MIX_RELEASE( pD3DTexture );
			return NULL;
		}

		pTexture = Mix::Graphics::DX9::Texture2D::CreateInstance(	this,
																	False, desc.Width, desc.Height, Device::ConvertTextureFormat( desc.Format ),
																	desc.Format, pD3DTexture,
																	True, pName );
		if( pTexture == NULL )
		{
			MIX_LOG_ERROR( L"%s : %s : %s[%s]", FAILED_CREATETEXTURE, Mix::STR_OUTOFMEMORY, pNameLabel, pName );
			MIX_RELEASE( pD3DTexture );
			return NULL;
		}

		return pTexture;
	}
	else if( info.ResourceType == D3DRTYPE_VOLUMETEXTURE )
	{
		MIX_LOG_ERROR( L"%s : {[eNX`̓T|[gĂ܂ : %s[%s]", FAILED_CREATETEXTURE, pNameLabel, pName );
		return NULL;
	}
	else if( info.ResourceType == D3DRTYPE_CUBETEXTURE )
	{
		D3DXCreateCubeTextureFromFileInMemoryExFunc D3DXCreateCubeTextureFromFileInMemoryEx = static_cast<D3DXCreateCubeTextureFromFileInMemoryExFunc>( m_D3DX9Module.GetFunction( L"D3DXCreateCubeTextureFromFileInMemoryEx" ) );
		IDirect3DCubeTexture9* pD3DTexture = NULL;
		Mix::Graphics::DX9::TextureCube* pTexture = NULL;

		hRet = D3DXCreateCubeTextureFromFileInMemoryEx( m_pD3DDevice,
														pBuffer,
														size,
														0,
														info.MipLevels,
														0,
														D3DFMT_UNKNOWN,
														D3DPOOL_MANAGED,
														D3DX_FILTER_NONE,
														D3DX_FILTER_NONE,
														0,
														NULL,
														NULL,
														&pD3DTexture );
		if( hRet != D3D_OK )
		{
			MIX_LOG_ERROR( L"%s : D3DXCreateCubeTextureFromFileInMemoryEx %s : %s[%s] Result[%s]", FAILED_CREATETEXTURE, Mix::STR_RETERROR, pNameLabel,
				pName,
				GetD3DResultText( hRet ) );

			return NULL;
		}

		hRet = pD3DTexture->GetLevelDesc( 0, &desc );
		if( hRet != D3D_OK )
		{
			MIX_LOG_ERROR( L"%s : IDirect3DCubeTexture9::GetLevelDesc %s : %s[%s] Result[%s]", FAILED_CREATETEXTURE, Mix::STR_RETERROR, pNameLabel,
				pName,
				GetD3DResultText( hRet ) );

			MIX_RELEASE( pD3DTexture );
			return NULL;
		}

		pTexture = Mix::Graphics::DX9::TextureCube::CreateInstance( this,
																	True, desc.Width, Device::ConvertTextureFormat( desc.Format ),
																	desc.Format, pD3DTexture,
																	pName );
		if( pTexture == NULL )
		{
			MIX_LOG_ERROR( L"%s : %s : %s[%s]", FAILED_CREATETEXTURE, Mix::STR_OUTOFMEMORY, pNameLabel, pName );
			MIX_RELEASE( pD3DTexture );
			return False;
		}

		return pTexture;
	}

	return NULL;
}

Mix::Graphics::FORMAT Device::ConvertTextureFormat( D3DFORMAT d3dFormat )
{
	switch( d3dFormat )
	{
	case D3DFMT_D16:
		return Mix::Graphics::FMT_D16;
	case D3DFMT_D32:
		return Mix::Graphics::FMT_D32;
	case D3DFMT_D24S8:
		return Mix::Graphics::FMT_D24S8;

	case D3DFMT_A8R8G8B8:
		return Mix::Graphics::FMT_R8G8B8A8;
	case D3DFMT_A8:
		return Mix::Graphics::FMT_A8;
	case D3DFMT_R16F:
		return Mix::Graphics::FMT_R16F;
	case D3DFMT_R32F:
		return Mix::Graphics::FMT_R32F;
	case D3DFMT_G16R16F:
		return Mix::Graphics::FMT_R16G16F;
	case D3DFMT_G32R32F:
		return Mix::Graphics::FMT_R32G32F;
	case D3DFMT_A16B16G16R16F:
		return Mix::Graphics::FMT_R16G16B16A16F;
	case D3DFMT_A32B32G32R32F:
		return Mix::Graphics::FMT_R32G32B32A32F;

	case D3DFMT_DXT1:
		return Mix::Graphics::FMT_DXT1;
	case D3DFMT_DXT2:
		return Mix::Graphics::FMT_DXT2;
	case D3DFMT_DXT3:
		return Mix::Graphics::FMT_DXT3;
	case D3DFMT_DXT4:
		return Mix::Graphics::FMT_DXT4;
	case D3DFMT_DXT5:
		return Mix::Graphics::FMT_DXT5;
	}

	return Mix::Graphics::FMT_UNKNOWN;
}

const wchar_t* Device::GetD3DFormatText( D3DFORMAT fmt )
{
	switch( fmt )
	{
		case D3DFMT_X8R8G8B8:
			return Device::D3DFormatTextArray[1];
		case D3DFMT_A8R8G8B8:
			return Device::D3DFormatTextArray[2];
		case D3DFMT_X1R5G5B5:
			return Device::D3DFormatTextArray[3];
		case D3DFMT_R5G6B5:
			return Device::D3DFormatTextArray[4];
		case D3DFMT_A8:
			return Device::D3DFormatTextArray[5];
		case D3DFMT_R16F:
			return Device::D3DFormatTextArray[6];
		case D3DFMT_R32F:
			return Device::D3DFormatTextArray[7];
		case D3DFMT_G16R16F:
			return Device::D3DFormatTextArray[8];
		case D3DFMT_G32R32F:
			return Device::D3DFormatTextArray[9];
		case D3DFMT_A16B16G16R16F:
			return Device::D3DFormatTextArray[10];
		case D3DFMT_A32B32G32R32F:
			return Device::D3DFormatTextArray[11];
		case D3DFMT_DXT1:
			return Device::D3DFormatTextArray[12];
		case D3DFMT_DXT2:
			return Device::D3DFormatTextArray[13];
		case D3DFMT_DXT3:
			return Device::D3DFormatTextArray[14];
		case D3DFMT_DXT4:
			return Device::D3DFormatTextArray[15];
		case D3DFMT_DXT5:
			return Device::D3DFormatTextArray[16];
		case D3DFMT_D16:
			return Device::D3DFormatTextArray[17];
		case D3DFMT_D32:
			return Device::D3DFormatTextArray[18];
		case D3DFMT_D24S8:
			return Device::D3DFormatTextArray[19];
	}

	return Device::D3DFormatTextArray[0];
}

const wchar_t* Device::GetD3DResultText( HRESULT result )
{
	switch( result )
	{
	case E_FAIL:
		return Device::D3DResultTextArray[1];
	case D3DERR_NOTAVAILABLE:
		return Device::D3DResultTextArray[2];
	case D3DERR_OUTOFVIDEOMEMORY:
		return Device::D3DResultTextArray[3];
	case D3DERR_INVALIDCALL:
		return Device::D3DResultTextArray[4];
	case D3DXERR_INVALIDDATA:
		return Device::D3DResultTextArray[5];
	case E_OUTOFMEMORY:
		return Device::D3DResultTextArray[6];
	}

	return Device::D3DResultTextArray[0];
}

const wchar_t* Device::GetQueryText( Mix::Graphics::QUERY_TYPE type )
{
	return Device::D3DQueryTextTable[type];
}

Boolean Device::Initialize( Mix::Graphics::SHADER_MODEL shaderModel, Boolean bFullscreen, Mix::UserFile* pSysReport )
{
	MIX_ASSERT( shaderModel == Mix::Graphics::SHADER_MODEL_3 );

	//_[Xe[g^Cve[u(D3D)
	const D3DRENDERSTATETYPE renderStateTypeTable[Device::RENDERSTATETYPE_MAX] =
	{
		D3DRS_FILLMODE,
		D3DRS_CULLMODE,
		D3DRS_SCISSORTESTENABLE,
		D3DRS_MULTISAMPLEANTIALIAS,
		D3DRS_ZENABLE,
		D3DRS_ZWRITEENABLE,
		D3DRS_ZFUNC,
		D3DRS_ALPHABLENDENABLE,
		D3DRS_SRCBLEND,
		D3DRS_DESTBLEND,
		D3DRS_BLENDOP,
		D3DRS_SRCBLENDALPHA,
		D3DRS_DESTBLENDALPHA,
		D3DRS_BLENDOPALPHA,
		D3DRS_COLORWRITEENABLE,
	};

	//Tv[Xe[g^Cve[u(D3D)
	const D3DSAMPLERSTATETYPE samplerStateTypeTable[Device::SAMPLERSTATETYPE_MAX] =
	{
		D3DSAMP_MAGFILTER,
		D3DSAMP_MINFILTER,
		D3DSAMP_MIPFILTER,
		D3DSAMP_ADDRESSU,
		D3DSAMP_ADDRESSV,
		D3DSAMP_ADDRESSW,
		D3DSAMP_MAXANISOTROPY,
	};

	UInt32 i;
	UInt32 j;
	HRESULT hRet;
	Boolean bCreateDevice = False;
	D3DXGetVertexShaderProfileFunc D3DXGetVertexShaderProfile;
	D3DXGetPixelShaderProfileFunc D3DXGetPixelShaderProfile;
	Mix::String moduleName;

	const UInt32 vpTable[] =
	{
		D3DCREATE_HARDWARE_VERTEXPROCESSING,
		D3DCREATE_MIXED_VERTEXPROCESSING,
		D3DCREATE_SOFTWARE_VERTEXPROCESSING,
	};
	const UInt32 MAX_VP = ( sizeof( vpTable ) / sizeof( UInt32 ) );

	const Mix::Point& backBuffSize = m_pWindow->GetBaseSize();

#ifdef _DEBUG
	//DirectX9W[̃[h
	if( m_D3D9Module.Load( L"d3d9d.dll" ) == True )
	{
	}
	else if( m_D3D9Module.Load( L"d3d9.dll" ) == True )
	{
		MIX_LOG_INFO( L"SDKCXg[ĂȂ߁A[XpDirectX9W[gp܂" );
	}
	else
	{
		::MessageBox( m_pWindow->GetHandle(), L"DirectX9CXg[Ă܂(1)", L"G[", MB_OK | MB_ICONSTOP );
		return False;
	}
#else //_DEBUG
	//DirectX9W[̃[h
	if( m_D3D9Module.Load( L"d3d9.dll" ) == False )
	{
		::MessageBox( m_pWindow->GetHandle(), L"DirectX9CXg[Ă܂(1)", L"G[", MB_OK | MB_ICONSTOP );
		return False;
	}
#endif //_DEBUG

	//D3DXW[̃[h
	moduleName.Sprintf( L"d3dx9_%d.dll", D3D_SDK_VERSION );
	if( m_D3DX9Module.Load( moduleName.GetConstPtr() ) == False )
	{
		::MessageBox( m_pWindow->GetHandle(), L"DirectX9CXg[Ă܂(2)", L"G[", MB_OK | MB_ICONSTOP );
		return False;
	}

	//Direct3D9̃t@NVo^
	if( m_D3D9Module.RegisterFunction( L"Direct3DCreate9" ) == False )
	{
		::MessageBox( m_pWindow->GetHandle(), L"DirectX9CXg[Ă܂(3)", L"G[", MB_OK | MB_ICONSTOP );
		return False;
	}

	//D3DX̃t@NVo^
	if( ( m_D3DX9Module.RegisterFunction( L"D3DXGetImageInfoFromFileInMemory" ) == False ) ||
		( m_D3DX9Module.RegisterFunction( L"D3DXCreateTextureFromFileInMemoryEx" ) == False ) ||
		( m_D3DX9Module.RegisterFunction( L"D3DXCreateCubeTextureFromFileInMemoryEx" ) == False ) ||
		( m_D3DX9Module.RegisterFunction( L"D3DXCompileShader" ) == False ) ||
		( m_D3DX9Module.RegisterFunction( L"D3DXGetVertexShaderProfile" ) == False ) ||
		( m_D3DX9Module.RegisterFunction( L"D3DXGetPixelShaderProfile" ) == False ) ||
		( m_D3DX9Module.RegisterFunction( L"D3DXSaveSurfaceToFileW" ) == False ) ||
		( m_D3DX9Module.RegisterFunction( L"D3DXSaveTextureToFileW" ) == False ) )
	{
		::MessageBox( m_pWindow->GetHandle(), L"DirectX9CXg[Ă܂(4)", L"G[", MB_OK | MB_ICONSTOP );
		return False;
	}

	//D3DX֐̎擾
	D3DXGetVertexShaderProfile = static_cast<D3DXGetVertexShaderProfileFunc>( m_D3DX9Module.GetFunction( L"D3DXGetVertexShaderProfile" ) );
	D3DXGetPixelShaderProfile = static_cast<D3DXGetPixelShaderProfileFunc>( m_D3DX9Module.GetFunction( L"D3DXGetPixelShaderProfile" ) );

	//Direct3D̍쐬
	typedef IDirect3D9 * ( WINAPI *Direct3DCreate9Function )( UINT SDKVersion );
	Direct3DCreate9Function Direct3DCreate9 = static_cast<Direct3DCreate9Function>(m_D3D9Module.GetFunction( L"Direct3DCreate9" ) );
	m_pD3D = Direct3DCreate9( D3D_SDK_VERSION );
	if( m_pD3D == NULL )
	{
		MIX_LOG_ERROR( L"%s : Direct3DCreate9 %s", FAILED_INITIALIZE, Mix::STR_RETERROR );
		return False;
	}

	//foCXŗL̎擾
	hRet = m_pD3D->GetDeviceCaps( 0, D3DDEVTYPE_HAL, &m_Caps );
	if( hRet != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : GetDeviceCaps %s : Result[%s]", FAILED_INITIALIZE, Mix::STR_RETERROR, GetD3DResultText( hRet ) );
		return False;
	}

	//foCX|[g
	if( pSysReport->Open() == True )
	{
		D3DADAPTER_IDENTIFIER9 adapIdent;

		pSysReport->WriteLine( L"////////////////////////////////////////////////////////////////////////////////////////////////////" );
		pSysReport->WriteLine( L"// OtBbNX( DirectX9 )                                                                     //" );
		pSysReport->WriteLine( L"////////////////////////////////////////////////////////////////////////////////////////////////////" );
		pSysReport->WriteLine( L"" );

		//fBXvCA_v^
		::ZeroMemory( &adapIdent, sizeof( adapIdent ) );
		if( m_pD3D->GetAdapterIdentifier( 0, 0, &adapIdent ) == D3D_OK )
		{
			pSysReport->WriteLine( L"[fBXvCA_v^]" );
			pSysReport->WriteLine( L"{" );
			pSysReport->WriteLine( L"    hCo : \"%s\"", Mix::String( adapIdent.Driver ).GetConstPtr() );
			pSysReport->WriteLine( L"    hCo̐ : \"%s\"", Mix::String( adapIdent.Description ).GetConstPtr() );
			pSysReport->WriteLine( L"    hCõo[W : 0x%08x%08x", adapIdent.DriverVersion.HighPart, adapIdent.DriverVersion.LowPart );
			pSysReport->WriteLine( L"    foCX : \"%s\"", Mix::String( adapIdent.DeviceName ) );
			pSysReport->WriteLine( L"    foCXID : 0x%08x", adapIdent.DeviceId );
			pSysReport->WriteLine( L"    x_[ID : 0x%08x", adapIdent.VendorId );
			pSysReport->WriteLine( L"    TuVXeID : 0x%08x", adapIdent.SubSysId );
			pSysReport->WriteLine( L"    rW : 0x%08x", adapIdent.Revision );
			pSysReport->WriteLine( L"    GUID : %08x-%04x-%04x-%02x-%02x-%02x-%02x-%02x-%02x-%02x-%02x", adapIdent.DeviceIdentifier.Data1, adapIdent.DeviceIdentifier.Data2, adapIdent.DeviceIdentifier.Data3, adapIdent.DeviceIdentifier.Data4[0], adapIdent.DeviceIdentifier.Data4[1], adapIdent.DeviceIdentifier.Data4[2], adapIdent.DeviceIdentifier.Data4[3], adapIdent.DeviceIdentifier.Data4[4], adapIdent.DeviceIdentifier.Data4[5], adapIdent.DeviceIdentifier.Data4[6], adapIdent.DeviceIdentifier.Data4[7] );
			pSysReport->WriteLine( L"    WHQL : 0x%08x", adapIdent.WHQLLevel );
			pSysReport->WriteLine( L"{" );
		}
		else
		{
			pSysReport->WriteLine( L"[fBXvCA_v^]" );
			pSysReport->WriteLine( L"{" );
			pSysReport->WriteLine( L"    擾Ɏs܂!" );
			pSysReport->WriteLine( L"{" );
		}

		pSysReport->WriteLine( L"" );

		pSysReport->WriteLine( L"[fBXvCA_v^̌ŗL]" );
		pSysReport->WriteLine( L"{" );
		pSysReport->WriteLine( L"    ~bv}bv̎[%s]", ( ( m_Caps.Caps2 & D3DCAPS2_CANAUTOGENMIPMAP ) == D3DCAPS2_CANAUTOGENMIPMAP )? L"" : L"~" );
		pSysReport->WriteLine( L"    IeNX`[%s]", ( ( m_Caps.Caps2 & D3DCAPS2_DYNAMICTEXTURES ) == D3DCAPS2_DYNAMICTEXTURES )? L"" : L"~" );

		pSysReport->WriteLine( L"    tXN[ł̃At@uhT|[g[%s]", ( ( m_Caps.Caps3 & D3DCAPS3_ALPHA_FULLSCREEN_FLIP_OR_DISCARD ) == D3DCAPS3_ALPHA_FULLSCREEN_FLIP_OR_DISCARD )? L"" : L"~" );

		pSysReport->WriteLine( L"    v[e[VXbvԊuT|[g[%s]", ( ( m_Caps.PresentationIntervals & D3DPRESENT_INTERVAL_IMMEDIATE ) == D3DPRESENT_INTERVAL_IMMEDIATE )? L"" : L"~" );

		pSysReport->WriteLine( L"    eNX`̃VXe񃍁[JrfIւ̃ubg[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_CANBLTSYSTONONLOCAL ) == D3DDEVCAPS_CANBLTSYSTONONLOCAL )? L"" : L"~" );
		pSysReport->WriteLine( L"    VXeɂsobt@gpł[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_EXECUTESYSTEMMEMORY ) == D3DDEVCAPS_EXECUTESYSTEMMEMORY )? L"" : L"~" );
		pSysReport->WriteLine( L"    rfIɂsobt@gpł[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_EXECUTEVIDEOMEMORY ) == D3DDEVCAPS_EXECUTEVIDEOMEMORY )? L"" : L"~" );
		pSysReport->WriteLine( L"    V[̃X^p̃n[hEFAANZ[V[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_HWRASTERIZATION ) == D3DDEVCAPS_HWRASTERIZATION )? L"" : L"~" );
		pSysReport->WriteLine( L"    n[hEFAŃgXtH[ACeBO[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_HWTRANSFORMANDLIGHT ) == D3DDEVCAPS_HWTRANSFORMANDLIGHT )? L"" : L"~" );
		pSysReport->WriteLine( L"    Npb`̃T|[g[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_NPATCHES ) == D3DDEVCAPS_NPATCHES )? L"" : L"~" );
		pSysReport->WriteLine( L"    n[hEFAŃX^AgXtH[ACeBOAVF[fBO[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_PUREDEVICE ) == D3DDEVCAPS_PUREDEVICE )? L"" : L"~" );
		pSysReport->WriteLine( L"    5xWFȐABXvC[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_QUINTICRTPATCHES ) == D3DDEVCAPS_QUINTICRTPATCHES )? L"" : L"~" );
		pSysReport->WriteLine( L"    `AOp`pb`T|[g[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_RTPATCHES ) == D3DDEVCAPS_RTPATCHES )? L"" : L"~" );
		pSysReport->WriteLine( L"    Ɨv[eNX`sĂ[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_SEPARATETEXTUREMEMORIES ) == D3DDEVCAPS_SEPARATETEXTUREMEMORIES )? L"" : L"~" );
		pSysReport->WriteLine( L"    񃍁[J rfI eNX`擾ł[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_TEXTURENONLOCALVIDMEM ) == D3DDEVCAPS_TEXTURENONLOCALVIDMEM )? L"" : L"~" );
		pSysReport->WriteLine( L"    VXeeNX`擾ł[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_TEXTURESYSTEMMEMORY ) == D3DDEVCAPS_TEXTURESYSTEMMEMORY )? L"" : L"~" );
		pSysReport->WriteLine( L"    rfIeNX`擾ł[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_TEXTUREVIDEOMEMORY ) == D3DDEVCAPS_TEXTUREVIDEOMEMORY )? L"" : L"~" );
		pSysReport->WriteLine( L"    gXtH[ACeBOς݂̒_ɁAVXẽobt@gpł[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_TLVERTEXSYSTEMMEMORY ) == D3DDEVCAPS_TLVERTEXSYSTEMMEMORY )? L"" : L"~" );
		pSysReport->WriteLine( L"    gXtH[ACeBOς݂̒_ɁArfĨobt@gpł[%s]", ( ( m_Caps.DevCaps & D3DDEVCAPS_TLVERTEXVIDEOMEMORY ) == D3DDEVCAPS_TLVERTEXVIDEOMEMORY )? L"" : L"~" );

		pSysReport->WriteLine( L"    ٕtB^O[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_ANISOTROPY ) == D3DPRASTERCAPS_ANISOTROPY )? L"" : L"~" );
		pSysReport->WriteLine( L"    F̃p[XyNeBu𐳂Ԃ[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_COLORPERSPECTIVE ) == D3DPRASTERCAPS_COLORPERSPECTIVE )? L"" : L"~" );
		pSysReport->WriteLine( L"    fBUOgpėǎȐFČł[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_DITHER ) == D3DPRASTERCAPS_DITHER )? L"" : L"~" );
		pSysReport->WriteLine( L"    ȑO̐[xoCAXT|[g[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_DEPTHBIAS ) == D3DPRASTERCAPS_DEPTHBIAS )? L"" : L"~" );
		pSysReport->WriteLine( L"    sNZ̐[wtHOl܂ގQƃe[uƏƍāAtHOlZo[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_FOGTABLE ) == D3DPRASTERCAPS_FOGTABLE )? L"" : L"~" );
		pSysReport->WriteLine( L"    CeBO̍ۂɃtHOlvZAX^̍ۂɃtHOlԂ[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_FOGVERTEX ) == D3DPRASTERCAPS_FOGVERTEX )? L"" : L"~" );
		pSysReport->WriteLine( L"    LODoCAXT|[g[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_MIPMAPLODBIAS ) == D3DPRASTERCAPS_MIPMAPLODBIAS )? L"" : L"~" );
		pSysReport->WriteLine( L"    }`TvÕIƃIt̐؂ւ[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_MULTISAMPLE_TOGGLE ) == D3DPRASTERCAPS_MULTISAMPLE_TOGGLE )? L"" : L"~" );
		pSysReport->WriteLine( L"    VU[eXgT|[g[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_SCISSORTEST ) == D3DPRASTERCAPS_SCISSORTEST )? L"" : L"~" );
		pSysReport->WriteLine( L"    ^̃X[vXP[x[X̐[xoCAXs[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_SLOPESCALEDEPTHBIAS ) == D3DPRASTERCAPS_SLOPESCALEDEPTHBIAS )? L"" : L"~" );
		pSysReport->WriteLine( L"    w gp[xobt@O[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_WBUFFER ) == D3DPRASTERCAPS_WBUFFER )? L"" : L"~" );
		pSysReport->WriteLine( L"    w x[X̃tHOT|[g[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_WFOG ) == D3DPRASTERCAPS_WFOG )? L"" : L"~" );
		pSysReport->WriteLine( L"    HRSsłÂƂ|S̃\[g[xobt@̊蓖ĂKvƂȂ[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_ZBUFFERLESSHSR ) == D3DPRASTERCAPS_ZBUFFERLESSHSR )? L"" : L"~" );
		pSysReport->WriteLine( L"    z x[X̃tHOT|[g[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_ZFOG ) == D3DPRASTERCAPS_ZFOG )? L"" : L"~" );
		pSysReport->WriteLine( L"    z eXgsł[%s]", ( ( m_Caps.RasterCaps & D3DPRASTERCAPS_ZTEST ) == D3DPRASTERCAPS_ZTEST )? L"" : L"~" );

		pSysReport->WriteLine( L"    eNX`́AׂĐ`łȂ΂ȂȂ[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_SQUAREONLY ) == D3DPTEXTURECAPS_SQUAREONLY )? L"" : L"~" );
		pSysReport->WriteLine( L"    ~bv}bveNX`[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_MIPMAP ) == D3DPTEXTURECAPS_MIPMAP )? L"" : L"~" );
		pSysReport->WriteLine( L"    eNX`̕ƍ́A2 ̗ݏŎw肷[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_POW2 ) == D3DPTEXTURECAPS_POW2 )? L"" : L"~" );
		pSysReport->WriteLine( L"    {[eNX`[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_VOLUMEMAP ) == D3DPTEXTURECAPS_VOLUMEMAP )? L"" : L"~" );
		pSysReport->WriteLine( L"    ~bv}bv{[eNX`[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_MIPVOLUMEMAP ) == D3DPTEXTURECAPS_MIPVOLUMEMAP )? L"" : L"~" );
		pSysReport->WriteLine( L"    ~bv}bvL[ueNX`[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_MIPCUBEMAP ) == D3DPTEXTURECAPS_MIPCUBEMAP )? L"" : L"~" );
		pSysReport->WriteLine( L"    {[eNX`̃fBW 2 ̗ݏŎw肷[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_VOLUMEMAP_POW2 ) == D3DPTEXTURECAPS_VOLUMEMAP_POW2 )? L"" : L"~" );
		pSysReport->WriteLine( L"    L[ueNX`[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_CUBEMAP ) == D3DPTEXTURECAPS_CUBEMAP )? L"" : L"~" );
		pSysReport->WriteLine( L"    L[ueNX`̃fBW 2 ̗ݏŎw肷[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_CUBEMAP_POW2 ) == D3DPTEXTURECAPS_CUBEMAP_POW2 )? L"" : L"~" );
		pSysReport->WriteLine( L"    eNX`sNZɂẴAt@[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_ALPHA ) == D3DPTEXTURECAPS_ALPHA )? L"" : L"~" );
		pSysReport->WriteLine( L"    eNX`pbgAt@`ł[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_ALPHAPALETTE ) == D3DPTEXTURECAPS_ALPHAPALETTE )? L"" : L"~" );
		pSysReport->WriteLine( L"    t 2 ̗ݏłȂfBWł2DeNX`[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_NONPOW2CONDITIONAL ) == D3DPTEXTURECAPS_NONPOW2CONDITIONAL )? L"" : L"~" );
		pSysReport->WriteLine( L"    vO\ьŒ@\VF[_ł̎ˉeovbNAbvZT|[gȂ[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_NOPROJECTEDBUMPENV ) == D3DPTEXTURECAPS_NOPROJECTEDBUMPENV )? L"" : L"~" );
		pSysReport->WriteLine( L"    p[XyNeBu␳̃eNX`O[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_PERSPECTIVE ) == D3DPTEXTURECAPS_PERSPECTIVE )? L"" : L"~" );
		pSysReport->WriteLine( L"    eNX`CfbNX́AԎsOɃeNX`TCYɍ킹ăXP[OȂ[%s]", ( ( m_Caps.TextureCaps & D3DPTEXTURECAPS_TEXREPEATNOTSCALEDBYSIZE ) == D3DPTEXTURECAPS_TEXREPEATNOTSCALEDBYSIZE )? L"" : L"~" );

		pSysReport->WriteLine( L"    RTpb`̓K^eZ[V[%s]", ( ( m_Caps.DevCaps2 & D3DDEVCAPS2_ADAPTIVETESSRTPATCH ) == D3DDEVCAPS2_ADAPTIVETESSRTPATCH )? L"" : L"~" );
		pSysReport->WriteLine( L"    Npb`̓K^eZ[V[%s]", ( ( m_Caps.DevCaps2 & D3DDEVCAPS2_ADAPTIVETESSNPATCH ) == D3DDEVCAPS2_ADAPTIVETESSNPATCH )? L"" : L"~" );
		pSysReport->WriteLine( L"    eNX`\[XƂĎg IDirect3DDevice9::StretchRect T|[g[%s]", ( ( m_Caps.DevCaps2 & D3DDEVCAPS2_CAN_STRETCHRECT_FROM_TEXTURES ) == D3DDEVCAPS2_CAN_STRETCHRECT_FROM_TEXTURES )? L"" : L"~" );
		pSysReport->WriteLine( L"    Npb`̃fBXv[Xg}bv[%s]", ( ( m_Caps.DevCaps2 & D3DDEVCAPS2_DMAPNPATCH ) == D3DDEVCAPS2_DMAPNPATCH )? L"" : L"~" );
		pSysReport->WriteLine( L"    Npb`̎OɃTvOꂽfBXv[Xg}bv[%s]", ( ( m_Caps.DevCaps2 & D3DDEVCAPS2_PRESAMPLEDDMAPNPATCH ) == D3DDEVCAPS2_PRESAMPLEDDMAPNPATCH )? L"" : L"~" );
		pSysReport->WriteLine( L"    Xg[ItZbg[%s]", ( ( m_Caps.DevCaps2 & D3DDEVCAPS2_STREAMOFFSET ) == D3DDEVCAPS2_STREAMOFFSET )? L"" : L"~" );
		pSysReport->WriteLine( L"    tŕ̒_vf 1 ̃Xg[̓ItZbgLł[%s]", ( ( m_Caps.DevCaps2 & D3DDEVCAPS2_VERTEXELEMENTSCANSHARESTREAMOFFSET ) == D3DDEVCAPS2_VERTEXELEMENTSCANSHARESTREAMOFFSET )? L"" : L"~" );
		pSysReport->WriteLine( L"" );

		pSysReport->WriteLine( L"    J[obt@ւ̃`lƂ݂̏T|[g : [%s]", MIX_LOG_BOOLEAN( ( m_Caps.PrimitiveMiscCaps & D3DPMISCCAPS_COLORWRITEENABLE ) == D3DPMISCCAPS_COLORWRITEENABLE ) );
		pSysReport->WriteLine( L"    J[obt@ʂ̏݃}XN̎gpT|[g : [%s]", MIX_LOG_BOOLEAN( ( m_Caps.PrimitiveMiscCaps & D3DPMISCCAPS_INDEPENDENTWRITEMASKS ) == D3DPMISCCAPS_INDEPENDENTWRITEMASKS ) );
		pSysReport->WriteLine( L"" );

		pSysReport->WriteLine( L"    eNX`̍̕ől : %d", m_Caps.MaxTextureWidth );
		pSysReport->WriteLine( L"    eNX`̍̍ől :%d", m_Caps.MaxTextureHeight );
		pSysReport->WriteLine( L"    {[eNX`̃fBW̍ől : %d", m_Caps.MaxVolumeExtent );
		pSysReport->WriteLine( L"    K̃eNX`W̐rbg̍ő͈ : %d", m_Caps.MaxTextureRepeat );
		pSysReport->WriteLine( L"    eNX`̍őAXyNg : %d", m_Caps.MaxTextureAspectRatio );
		pSysReport->WriteLine( L"    ٕ̍ől : %d", m_Caps.MaxAnisotropy );
		pSysReport->WriteLine( L"    Wx[X̍ő[xl %f", m_Caps.MaxVertexW );
		pSysReport->WriteLine( L"    eNX`ufBOXe[W̍ő吔 : %d", m_Caps.MaxTextureBlendStages );
		pSysReport->WriteLine( L"    eNX`ufBOXe[WɓɃoChłeNX`̍ő吔 : %d", m_Caps.MaxSimultaneousTextures );
		pSysReport->WriteLine( L"    |Cgv~eBu̍őTCY : %f", m_Caps.MaxPointSize );
		pSysReport->WriteLine( L"    őv~eBu : %d", m_Caps.MaxPrimitiveCount );
		pSysReport->WriteLine( L"    _ɑ΂ăT|[gCfbNX̍őTCY : %d", m_Caps.MaxVertexIndex );
		pSysReport->WriteLine( L"    f[^Xg[̍ő吔 : %d", m_Caps.MaxStreams );
		pSysReport->WriteLine( L"    f[^Xg[̍őXgCh : %d", m_Caps.MaxStreamStride );
		pSysReport->WriteLine( L"    o[ebNXVF[_[̃o[W : %d.%d", D3DSHADER_VERSION_MAJOR( m_Caps.VertexShaderVersion ), D3DSHADER_VERSION_MINOR( m_Caps.VertexShaderVersion ) );
		pSysReport->WriteLine( L"    o[ebNXVF[_[̒萔pɗ\񂳂Ă郌WX^ - vs_1_1 ̐ : %d", m_Caps.MaxVertexShaderConst );
		pSysReport->WriteLine( L"    sNZVF[_[̃o[W : %d.%d", D3DSHADER_VERSION_MAJOR( m_Caps.PixelShaderVersion ), D3DSHADER_VERSION_MINOR( m_Caps.PixelShaderVersion ) );
		pSysReport->WriteLine( L"    sNZVF[_̎ZpR|[lg̍ől :%f", m_Caps.PixelShader1xMaxValue );
		pSysReport->WriteLine( L"" );

		pSysReport->WriteLine( L"    Npb`̕x̍ő吔 : %f", m_Caps.MaxNpatchTessellationLevel );
		pSysReport->WriteLine( L"    _O^[Qbg̐ : %d", m_Caps.NumSimultaneousRTs );
		pSysReport->WriteLine( L"    o[ebNXVF[_̎sł閽߂̍ő吔 : %d", m_Caps.MaxVShaderInstructionsExecuted );
		pSysReport->WriteLine( L"    sNZVF[_̎sł閽߂̍ő吔 : %d", m_Caps.MaxPShaderInstructionsExecuted );
		pSysReport->WriteLine( L"    o[ebNXVF[_̖߃Xbg̍ő吔 : %d", m_Caps.MaxVertexShader30InstructionSlots );
		pSysReport->WriteLine( L"    sNZVF[_̖߃Xbg̍ő吔 : %d", m_Caps.MaxPixelShader30InstructionSlots );
		pSysReport->WriteLine( L"}" );
		pSysReport->WriteLine( L"" );

		pSysReport->Close();
	}

	//VF[_[f
	if( ( m_Caps.VertexShaderVersion < D3DVS_VERSION( 3, 0 ) ) ||
		( m_Caps.PixelShaderVersion < D3DPS_VERSION( 3, 0 ) ) )
	{
		MIX_LOG_ERROR( L"%s : VF[_[f3ȏłKv܂", FAILED_INITIALIZE );
		return False;
	}

	//fBXvC
	if( EnumDisplay( pSysReport ) == False )
	{
		MIX_LOG_ERROR( L"%s : gpłfBXvC܂", FAILED_INITIALIZE );
		return False;
	}

	//foCXR{|[g
	if( pSysReport->Open() == True )
	{
		Device::DeviceComboList::iterator it_dc;
		Device::DeviceComboList::iterator it_dc_begin = m_DeviceComboList.begin();
		Device::DeviceComboList::iterator it_dc_end = m_DeviceComboList.end();

		for( it_dc = it_dc_begin; it_dc != it_dc_end; ++it_dc )
		{
			Device::DEVICECOMBO* pdb = &( *it_dc );

			pSysReport->WriteLine( L"[foCX̐ݒ]" );
			pSysReport->WriteLine( L"{" );
			pSysReport->WriteLine( L"    A_v^tH[}bg[%s]", GetD3DFormatText( pdb->adapterFormat ) );
			pSysReport->WriteLine( L"    obNobt@tH[}bg[%s]", GetD3DFormatText( pdb->backBufferFormat ) );
			pSysReport->WriteLine( L"" );
			pSysReport->WriteLine( L"    EBhE[h[%s]", ( ( pdb->flags & Device::FLAG_MODE_WINDOW ) == FLAG_MODE_WINDOW )? L"" : L"~" );
			pSysReport->WriteLine( L"    tXN[[h[%s]", ( ( pdb->flags & Device::FLAG_MODE_FULLSCREEN ) == FLAG_MODE_FULLSCREEN )? L"" : L"~" );
			pSysReport->WriteLine( L"" );

			////////////////////////////////////////////////////////////////////////////////////////////////////

			pSysReport->WriteLine( L"    [2DeNX`]" );
			pSysReport->WriteLine( L"    {" );

			for( i = 0 ; i < Mix::Graphics::FMT_MAX; i++ )
			{
				if( ( pdb->formatTable[i].flags & FLAG_RESOURCE_TEXTURE ) == FLAG_RESOURCE_TEXTURE )
				{
					pSysReport->WriteLine( L"        %s", GetD3DFormatText( pdb->formatTable[i].format ) );
				}
			}

			pSysReport->WriteLine( L"    }" );
			pSysReport->WriteLine( L"" );

			////////////////////////////////////////////////////////////////////////////////////////////////////

			pSysReport->WriteLine( L"    [L[ueNX`]" );
			pSysReport->WriteLine( L"    {" );

			for( i = 0 ; i < Mix::Graphics::FMT_MAX; i++ )
			{
				if( ( pdb->formatTable[i].flags & FLAG_RESOURCE_CUBE ) == FLAG_RESOURCE_CUBE )
				{
					pSysReport->WriteLine( L"        %s", GetD3DFormatText( pdb->formatTable[i].format ) );
				}
			}

			pSysReport->WriteLine( L"    }" );
			pSysReport->WriteLine( L"" );

			////////////////////////////////////////////////////////////////////////////////////////////////////

			pSysReport->WriteLine( L"    [_Ci~bNeNX`]" );
			pSysReport->WriteLine( L"    {" );

			for( i = 0 ; i < Mix::Graphics::FMT_MAX; i++ )
			{
				if( ( pdb->formatTable[i].flags & FLAG_RESOURCE_TEXTURE ) == FLAG_RESOURCE_TEXTURE )
				{
					pSysReport->WriteLine( L"        %s", GetD3DFormatText( pdb->formatTable[i].format ) );
				}
			}

			pSysReport->WriteLine( L"    }" );
			pSysReport->WriteLine( L"" );

			////////////////////////////////////////////////////////////////////////////////////////////////////

			pSysReport->WriteLine( L"    [^[QbgeNX`]" );
			pSysReport->WriteLine( L"    {" );

			for( i = 0 ; i < Mix::Graphics::FMT_MAX; i++ )
			{
				if( ( pdb->formatTable[i].flags & FLAG_RESOURCE_TARGET ) == FLAG_RESOURCE_TARGET )
				{
					pSysReport->WriteLine( L"        %s", GetD3DFormatText( pdb->formatTable[i].format ) );
				}
			}

			pSysReport->WriteLine( L"    }" );
			pSysReport->WriteLine( L"" );

			////////////////////////////////////////////////////////////////////////////////////////////////////

			pSysReport->WriteLine( L"    [fvXeNX`]" );
			pSysReport->WriteLine( L"    {" );

			for( i = 0 ; i < Mix::Graphics::FMT_MAX; i++ )
			{
				if( ( pdb->formatTable[i].flags & FLAG_RESOURCE_DEPTHSTENCIL ) == FLAG_RESOURCE_DEPTHSTENCIL )
				{
					pSysReport->WriteLine( L"        %s", GetD3DFormatText( pdb->formatTable[i].format ) );
				}
			}

			pSysReport->WriteLine( L"    }" );

			pSysReport->WriteLine( L"}" );
			pSysReport->WriteLine( L"" );
		}

		pSysReport->Close();
	}

	//|[gI
	pSysReport->WriteLine( L"" );

	//foCXp[^쐬
	if( CreatePresentation( True ) == True )
	{
		m_DeviceParam.current = m_DeviceParam.next;
	}
	else
	{
		MIX_LOG_ERROR( L"%s : OtBbNfoCX( DirectX9 )̃p[^쐬ł܂ł", FAILED_INITIALIZE );
		return False;
	}

	//foCX쐬
	for( i = 0; ( i < MAX_VP ) && ( bCreateDevice == False ); i++ )
	{
		hRet = m_pD3D->CreateDevice(	0,
										D3DDEVTYPE_HAL,
										m_pWindow->GetHandle(),
										vpTable[i],
										&( m_DeviceParam.next.params ),
										&m_pD3DDevice );
		if( hRet == D3D_OK )
		{
			bCreateDevice = True;
		}
	}
	if( bCreateDevice == False )
	{
		MIX_LOG_ERROR( L"%s : CreateDevice %s : Result[%s]", FAILED_INITIALIZE, Mix::STR_RETERROR, GetD3DResultText( hRet ) );
		return False;
	}

	//_[Xe[ge[u
	for( i = 0; i < Device::RENDERSTATETYPE_MAX; i++ )
	{
		D3DRENDERSTATETYPE type = renderStateTypeTable[i];
		DWORD value = 0;

		m_pD3DDevice->GetRenderState( type, &value );

		m_RenderStateTable[i].type = type;
		m_RenderStateTable[i].defValue = value;
		m_RenderStateTable[i].value = value;
	}

	//Tv[Xe[ge[u
	for( i = 0; i < Mix::Graphics::TEXTURESTAGE_MAX; i++ )
	{
		for( j = 0; j < Device::SAMPLERSTATETYPE_MAX; j++ )
		{
			D3DSAMPLERSTATETYPE type = samplerStateTypeTable[j];
			DWORD value = 0;

			m_pD3DDevice->GetSamplerState( i, type, &value );

			m_SamplerStateTable[i][j].type = type;
			m_SamplerStateTable[i][j].defValue = value;
			m_SamplerStateTable[i][j].value = value;
		}
	}

	//`Xe[g
	ResetDrawState();

	//VF[_[vt@C̎擾
	m_VertexShaderProfile = D3DXGetVertexShaderProfile( m_pD3DDevice );
	m_PixelShaderProfile = D3DXGetPixelShaderProfile( m_pD3DDevice );

	//obNobt@擾
	hRet = m_pD3DDevice->GetBackBuffer( 0, 0, D3DBACKBUFFER_TYPE_MONO, &m_pD3DBackBuffer );
	if( hRet != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : GetBackBuffer %s : Result[%s]", FAILED_INITIALIZE, Mix::STR_RETERROR, GetD3DResultText( hRet ) );
		return False;
	}

	//^[Qbg̋`
	m_TargetState.rect = Mix::Rectangle( 0, 0, backBuffSize.x, backBuffSize.y );

	//r[|[gݒ
	SetViewport( 0, 0, backBuffSize.x, backBuffSize.y );

	//VU[ݒ
	SetScissor( 0, 0, backBuffSize.x, backBuffSize.y );

	if( bFullscreen == True )
	{
		//tXN[ɂ
		m_Status = Device::STATUS_CHANGE_FULLSCREEN;
	}

	TracePresentation( L"OtBbNfoCX( DirectX9 )" );

	return True;
}

Boolean Device::Update( void )
{
	switch( m_Status )
	{
	//EBhE[h֕ύX
	case Device::STATUS_CHANGE_WINDOW:
		ChangePresentation( True );
		break;

	//tXN[[h[h֕ύX
	case Device::STATUS_CHANGE_FULLSCREEN:
		ChangePresentation( False );
		break;

	//Xg
	case Device::STATUS_LOST:
		ResetPresentation();
		break;
	}

	return ( m_Status == Device::STATUS_OK );
}

Boolean Device::MessageProc( UInt32 msg, WPARAM wParam, LPARAM lParam )
{
	switch( msg )
	{
	case WM_SYSKEYDOWN:
		if( wParam == VK_RETURN )
		{
			ChangeWindowMode( !GetWindowMode() );
		}
		break;
	}

	return True;
}

Boolean Device::CompileShader(	const void* pSrc, UInt32 srcSize,
								Mix::Graphics::SHADER_MACRO* pMacros,
								const char* pFuncName,
								Mix::Graphics::SHADER_TARGET target,
								Mix::Memory::IBuffer** ppBuffer )
{
	D3DXCompileShaderFunc D3DXCompileShader = static_cast<D3DXCompileShaderFunc>( m_D3DX9Module.GetFunction( L"D3DXCompileShader" ) );

	HRESULT ret;
	std::vector<D3DXMACRO> d3dMacros;
	LPD3DXBUFFER pShaderBuffer = NULL;
	LPD3DXBUFFER pShaderError = NULL;
	Mix::Memory::IBuffer* pDst = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// }NXg̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pMacros != NULL )
	{
		D3DXMACRO temp;

		for( UInt32 i = 0; ( ( pMacros[i].pNameA != NULL ) && ( pMacros[i].pDefA != NULL ) ); i++ )
		{
			temp.Name = pMacros[i].pNameA;
			temp.Definition = pMacros[i].pDefA;
			d3dMacros.push_back( temp );
		}

		temp.Name = NULL;
		temp.Definition = NULL;
		d3dMacros.push_back( temp );
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// RpC
	////////////////////////////////////////////////////////////////////////////////////////////////////

	ret = D3DXCompileShader(	reinterpret_cast<LPCSTR>( pSrc ),
								srcSize,
								( d3dMacros.size() > 0  )? &( d3dMacros[0] ) : NULL,
								NULL,
								pFuncName,
								Mix::Graphics::SHADER_TARGET_TABLE[target],
								0,
								&pShaderBuffer,
								&pShaderError,
								NULL );
	if( ret != D3D_OK )
	{
		Mix::StringW errMsg = ( pShaderError != NULL )? reinterpret_cast<char*>( pShaderError->GetBufferPointer() ) : "???";

		MIX_LOG_ERROR( L"D3DXCompileShader %s : %s", Mix::STR_RETERROR, errMsg.GetConstPtr() );

		MIX_RELEASE( pShaderBuffer );
		MIX_RELEASE( pShaderError );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// obt@쐬ăRs[
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( Mix::Memory::CreateBuffer( pShaderBuffer->GetBufferSize(), pShaderBuffer->GetBufferPointer(), &pDst ) == False )
	{
		MIX_LOG_ERROR( L"%s : Size[%dByte]", Mix::STR_OUTOFMEMORY, pShaderBuffer->GetBufferSize() );

		MIX_RELEASE( pShaderBuffer );
		MIX_RELEASE( pShaderError );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ㏈
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_RELEASE( pShaderBuffer );
	MIX_RELEASE( pShaderError );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppBuffer ) = pDst;

	return True;
}

void Device::OnDispose( void )
{
	UInt32 i;

	for( i = 0; i < Mix::Graphics::TARGET_MAX; i++ )
	{
		MIX_RELEASE( m_TargetState.pTex[i] );
	}

	MIX_RELEASE( m_TargetState.pDSTex );

	MIX_RELEASE( m_pD3DBackBuffer );
	MIX_RELEASE( m_pD3DDevice );
	MIX_RELEASE( m_pD3D );
}

void Device::ResetTargetState( void )
{
	UInt32 i;

	m_TargetState.rect = Mix::Rectangle( 0, 0, m_DeviceParam.current.screenSize.x, m_DeviceParam.current.screenSize.y );

	m_pD3DDevice->SetRenderTarget( 0, m_pD3DBackBuffer );
	MIX_RELEASE( m_TargetState.pTex[0] );

	for( i = 1; i < Mix::Graphics::TARGET_MAX; i++ )
	{
		m_pD3DDevice->SetRenderTarget( i, NULL );
		MIX_RELEASE( m_TargetState.pTex[i] );
	}

	m_pD3DDevice->SetDepthStencilSurface( NULL );
	MIX_RELEASE( m_TargetState.pDSTex );
}

void Device::FlushTargetState( void )
{
	UInt32 i;
	
	for( i = 0; i < Mix::Graphics::TARGET_MAX; i++ )
	{
		IDirect3DSurface9* pD3DSurface = NULL;

		if( m_TargetState.pTex[i] != NULL )
		{
			pD3DSurface = static_cast<Mix::Graphics::DX9::Texture*>( m_TargetState.pTex[i] )->GetSurface();
		}
		else if( i == 0 )
		{
			pD3DSurface = m_pD3DBackBuffer;
		}

		m_pD3DDevice->SetRenderTarget( i, pD3DSurface );
	}

	if( m_TargetState.pDSTex != NULL )
	{
		m_pD3DDevice->SetDepthStencilSurface( static_cast<Mix::Graphics::DX9::TextureDepth*>( m_TargetState.pDSTex )->GetSurface( 0 ) );
	}
	else
	{
		m_pD3DDevice->SetDepthStencilSurface( NULL );
	}
}

void Device::ResetRenderState( void )
{
	UInt32 i;
	DWORD value;

	for( i = 0; i < Device::RENDERSTATETYPE_MAX; i++ )
	{
		m_RenderStateTable[i].value = m_RenderStateTable[i].defValue;
	}

	if( ( m_pD3DDevice->GetRenderState( D3DRS_LIGHTING, &value ) == D3D_OK ) &&
		( value == TRUE ) )
	{
		m_pD3DDevice->SetRenderState( D3DRS_LIGHTING, FALSE );
	}

	m_RasterizeDesc.fillMode = Mix::Graphics::FILL_SOLID;
	m_RasterizeDesc.cullMode = Mix::Graphics::CULL_BACK;
	m_RasterizeDesc.bScissoring = False;
	m_RasterizeDesc.bMultisample = True;

	m_DepthDesc.bTest = False;
	m_DepthDesc.bWrite = True;
	m_DepthDesc.func = Mix::Graphics::ZFUNC_LESSEQUAL;

	m_BlendDesc.type = Mix::Graphics::BLEND_COPY;
	m_BlendDesc.colorWriteMask = Mix::Graphics::COLOR_WRITE_RGBA;
}

void Device::ResetRenderState( RENDERSTATETYPE type )
{
	RENDERSTATE* pRenderState = &( m_RenderStateTable[type] );

	if( ( pRenderState->value != pRenderState->defValue ) &&
		( m_pD3DDevice->SetRenderState( pRenderState->type, pRenderState->defValue ) == D3D_OK ) )
	{
		pRenderState->value = pRenderState->defValue;
	}
}

void Device::SetRenderState( RENDERSTATETYPE type, UInt32 value )
{
	RENDERSTATE* pRenderState;
	
	pRenderState = &( m_RenderStateTable[type] );

	if( ( pRenderState->value != value ) &&
		( m_pD3DDevice->SetRenderState( pRenderState->type, value ) == D3D_OK ) )
	{
		pRenderState->value = value;
	}
}

void Device::ResetSamplerState( void )
{
	UInt32 i;
	UInt32 j;

	for( i = 0; i < Mix::Graphics::TEXTURESTAGE_MAX; i++ )
	{
		for( j = 0; j < Device::SAMPLERSTATETYPE_MAX; j++ )
		{
			m_SamplerStateTable[i][j].value = m_SamplerStateTable[i][j].defValue;
		}
	}
}

void Device::SetSamplerState( UInt32 slot, SAMPLERSTATETYPE type, UInt32 value )
{
	SAMPLERSTATE* pSamplerState;

	pSamplerState = &( m_SamplerStateTable[slot][type] );

	if( ( pSamplerState->value != value ) &&
		( m_pD3DDevice->SetSamplerState( slot, pSamplerState->type, value ) == D3D_OK ) )
	{
		pSamplerState->value = value;
	}
}

void Device::ResetDrawState( void )
{
	UInt32 i;

	m_DrawState.current.pVertexLayout = NULL;
	m_DrawState.current.pVertexShader = NULL;
	m_DrawState.current.pPixelShader = NULL;
	m_DrawState.current.pVertexBuffer = NULL;
	m_DrawState.current.pIndexBuffer = NULL;

	m_DrawState.next.pVertexLayout = NULL;
	m_DrawState.next.pVertexShader = NULL;
	m_DrawState.next.pPixelShader = NULL;
	m_DrawState.next.pVertexBuffer = NULL;
	m_DrawState.next.pIndexBuffer = NULL;

	if( m_pD3DDevice != NULL )
	{
		for( i = 0; i < Mix::Graphics::TEXTURESTAGE_MAX; i++ )
		{
			m_DrawState.textureTable[i] = NULL;
			m_pD3DDevice->SetTexture( i, NULL );
		}
	}
	else
	{
		for( i = 0; i < Mix::Graphics::TEXTURESTAGE_MAX; i++ )
		{
			m_DrawState.textureTable[i] = NULL;
		}
	}
}

void Device::FlushDrawState( void )
{
	//_CAEg̐ݒ
	if( ( m_DrawState.next.pVertexLayout != NULL ) &&
		( m_DrawState.current.pVertexLayout != m_DrawState.next.pVertexLayout ) )
	{
		m_pD3DDevice->SetVertexDeclaration( static_cast<Mix::Graphics::DX9::VertexLayout*>( m_DrawState.next.pVertexLayout )->GetInterface() );
		m_DrawState.current.pVertexLayout = m_DrawState.next.pVertexLayout;
	}

	//o[ebNXVF[_[̐ݒ
	if( ( m_DrawState.next.pVertexShader != NULL ) &&
		( m_DrawState.current.pVertexShader != m_DrawState.next.pVertexShader ) )
	{
		m_pD3DDevice->SetVertexShader( static_cast<Mix::Graphics::DX9::VertexShader*>( m_DrawState.next.pVertexShader )->GetInsterface() );
		m_DrawState.current.pVertexShader = m_DrawState.next.pVertexShader;
	}

	//sNZVF[_[̐ݒ
	if( ( m_DrawState.next.pPixelShader != NULL ) &&
		( m_DrawState.current.pPixelShader != m_DrawState.next.pPixelShader ) )
	{
		m_pD3DDevice->SetPixelShader( static_cast<Mix::Graphics::DX9::PixelShader*>( m_DrawState.next.pPixelShader )->GetInsterface() );
		m_DrawState.current.pPixelShader = m_DrawState.next.pPixelShader;
	}

	//_obt@̐ݒ
	if( ( m_DrawState.next.pVertexBuffer != NULL ) &&
		( m_DrawState.current.pVertexBuffer != m_DrawState.next.pVertexBuffer ) )
	{
		m_pD3DDevice->SetStreamSource( 0, static_cast<Mix::Graphics::DX9::VertexBuffer*>( m_DrawState.next.pVertexBuffer )->GetInterface(), 0, m_DrawState.next.pVertexBuffer->GetStride() );
		m_DrawState.current.pVertexBuffer = m_DrawState.next.pVertexBuffer;
	}

	//CfbNXobt@̐ݒ
	if( ( m_DrawState.next.pIndexBuffer != NULL ) &&
		( m_DrawState.current.pIndexBuffer != m_DrawState.next.pIndexBuffer ) )
	{
		m_pD3DDevice->SetIndices( static_cast<Mix::Graphics::DX9::IndexBuffer*>( m_DrawState.next.pIndexBuffer )->GetInterface() );
		m_DrawState.current.pIndexBuffer = m_DrawState.next.pIndexBuffer;
	}
}

UInt32 Device::ComputePrimitiveCount( Mix::Graphics::PRIMITIVE_TYPE primitiveType, UInt32 num )
{
	UInt32 count = 0;

	switch( primitiveType )
	{
	case PRIMITIVE_POINTLIST:
		count = num;
		break;
	case PRIMITIVE_LINELIST:
		count = ( num >> 1 );
		break;
	case PRIMITIVE_LINESTRIP:
		if( num >= 2 )
		{
			count = ( num - 1 );
		}
		break;
	case PRIMITIVE_TRIANGLELIST:
		count = ( num / 3 );
		break;
	case PRIMITIVE_TRIANGLESTRIP:
		if( num >= 3 )
		{
			count = ( num - 2 );
		}
		break;
	}

	return count;
}

void Device::AdjustTexturePow2( UInt32& width, UInt32& height )
{
	UInt32 i;

	//
	for( i = 1; i < m_Caps.MaxTextureWidth; i += i )
	{
		if( i >= width )
		{
			width = i;
			break;
		}
	}

	//
	for( i = 1; i < m_Caps.MaxTextureHeight; i += i )
	{
		if( i >= height )
		{
			height = i;
			break;
		}
	}
}

void Device::AdjustTextureSquare( UInt32& width, UInt32& height )
{
	//`ɏC
	UInt32 temp = max( width, height );
	width = temp;
	height = temp;
}

Boolean Device::CreateVertexLayout( const Mix::Graphics::VERTEX_ELEMENT* pElements, UInt32 elementCount, const Mix::Graphics::IVertexShader* pVertexShader, Mix::Graphics::IVertexLayout** ppVertexLayout, const wchar_t* pName )
{
	Mix::String name = MIX_SAFE_NAME( pName );

	if( ( pElements == NULL ) ||
		( elementCount == 0 ) ||
		( pVertexShader == NULL ) ||
		( ppVertexLayout == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : pElements[%s] elementCount[%d] pVertexShader[%s] ppVertexLayout[%s]",
			FAILED_CREATEVERTEXLAYOUT,
			Mix::STR_ILLEGALARG,
			name.GetConstPtr(),
			( pElements != NULL )? L"" : L"~",
			elementCount,
			( pVertexShader != NULL )? L"" : L"~",
			( ppVertexLayout != NULL )? L"" : L"~"
			);

		return False;
	}

	UInt32 i;
	UInt32 stride;
	HRESULT ret;
	Mix::Graphics::DX9::VertexLayout* pVertexLayout;
	D3DVERTEXELEMENT9* pD3DElements;
	IDirect3DVertexDeclaration9* pD3DVertexLayout;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// o[ebNXfN[V쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	stride = 0;

	pD3DElements = static_cast<D3DVERTEXELEMENT9*>( Mix::Memory::Allocate( sizeof( D3DVERTEXELEMENT9 ) * ( elementCount + 1 ) ) );
	if( pD3DElements == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", FAILED_CREATEVERTEXLAYOUT, Mix::STR_OUTOFMEMORY, name.GetConstPtr() );
		return False;
	}

	for( i = 0; i < elementCount; i++ )
	{
		UInt32 format = pElements[i].format;

		pD3DElements[i].Stream = 0;
		pD3DElements[i].Offset = static_cast<WORD>( pElements[i].offset );
		pD3DElements[i].Type = Device::D3DVLFormatTable[format];
		pD3DElements[i].Method = D3DDECLMETHOD_DEFAULT;
		pD3DElements[i].Usage = Device::D3DVLSemanticTypeTable[pElements[i].semanticType];
		pD3DElements[i].UsageIndex = static_cast<BYTE>( pElements[i].semanticIndex );

		stride += Device::VERTEX_ELEMENT_SIZE_TABLE[format];
	}
	
	pD3DElements[elementCount].Stream = 0xFF;
	pD3DElements[elementCount].Offset = 0;
	pD3DElements[elementCount].Type = D3DDECLTYPE_UNUSED;
	pD3DElements[elementCount].Method = 0;
	pD3DElements[elementCount].Usage = 0;
	pD3DElements[elementCount].UsageIndex = 0;

	ret = m_pD3DDevice->CreateVertexDeclaration( &( pD3DElements[0] ), &pD3DVertexLayout );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : CreateVertexDeclaration %s : Name[%s] Result[%s]", FAILED_CREATEVERTEXLAYOUT, Mix::STR_RETERROR, name.GetConstPtr(), GetD3DResultText( ret ) );
		Mix::Memory::Free( pD3DElements );
		return False;
	}

	Mix::Memory::Free( pD3DElements );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pVertexLayout = Mix::Graphics::DX9::VertexLayout::CreateInstance(	this,
																		pD3DVertexLayout,
																		pElements,
																		elementCount,
																		stride,
																		name.GetConstPtr() );
	if( pVertexLayout != NULL )
	{
		( *ppVertexLayout ) = pVertexLayout;
	}
	else
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", FAILED_CREATEVERTEXLAYOUT, Mix::STR_OUTOFMEMORY, name.GetConstPtr() );
		MIX_RELEASE( pD3DVertexLayout );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO( L"o[ebNXCAEg쐬 : Name[%s] Stride[%d]", name.GetConstPtr(), stride );

	return True;
}

Boolean Device::CreateHullShaderFromFile( const wchar_t* pFileName, Mix::Graphics::IHullShader** ppHullShader )
{
	MIX_ERROR( L"nVF[_[̍쐬Ɏs : VF[_[fR CreateHullShaderFromFile ͎gpł܂" );
	return False;
}

Boolean Device::CreateHullShaderFromMemory( const void* pSrc, UInt32 srcSize, Mix::Graphics::IHullShader** ppHullShader, const wchar_t* pName )
{
	MIX_ERROR( L"nVF[_[̍쐬Ɏs : VF[_[fR CreateHullShaderFromMemory ͎gpł܂" );
	return False;
}

Boolean Device::CreateDomainShaderFromFile( const wchar_t* pFileName, Mix::Graphics::IDomainShader** ppDomainShader )
{
	MIX_ERROR( L"hCVF[_[̍쐬Ɏs : VF[_[fR CreateDomainShaderFromFile ͎gpł܂" );
	return False;
}

Boolean Device::CreateDomainShaderFromMemory( const void* pSrc, UInt32 srcSize, Mix::Graphics::IDomainShader** ppDomainShader, const wchar_t* pName )
{
	MIX_ERROR( L"hCVF[_[̍쐬Ɏs : VF[_[fR CreateDomainShaderFromMemory ͎gpł܂" );
	return False;
}

Boolean Device::CreateGeometryShaderFromFile( const wchar_t* pFileName, Mix::Graphics::IGeometryShader** ppGeometryShader )
{
	MIX_ERROR( L"WIgVF[_[̍쐬Ɏs : VF[_[fR CreateGeometryShaderFromFile ͎gpł܂" );
	return False;
}

Boolean Device::CreateGeometryShaderFromMemory( const void* pSrc, UInt32 srcSize, Mix::Graphics::IGeometryShader** ppGeometryShader, const wchar_t* pName )
{
	MIX_ERROR( L"WIgVF[_[̍쐬Ɏs : VF[_[fR CreateGeometryShaderFromMemory ͎gpł܂" );
	return False;
}

Boolean Device::CreateVertexShaderFromMemory( const void* pSrcData, UInt32 srcDataSize, Mix::Graphics::IVertexShader** ppVertexShader, const wchar_t* pName )
{
	Mix::String name = MIX_SAFE_NAME( pName );

	if( ( pSrcData == NULL ) ||
		( srcDataSize == 0 ) ||
		( ppVertexShader == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : pSrcData[%s] srcDatSize[%d] ppVertexShader[%s]",
			FAILED_CREATEVERTEXSHADER,
			Mix::STR_ILLEGALARG,
			name.GetConstPtr(),
			( pSrcData != NULL )? L"" : L"~",
			srcDataSize,
			( ppVertexShader != NULL )? L"" : L"~"
			);

		return False;
	}

	HRESULT ret;
	Mix::Graphics::DX9::VertexShader* pVertexShader;
	IDirect3DVertexShader9* pD3DVertexShader;
	
	ret = m_pD3DDevice->CreateVertexShader( static_cast<const DWORD*>( pSrcData ), &pD3DVertexShader );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : CreateVertexShader %s : Name[%s]", FAILED_CREATEVERTEXSHADER, Mix::STR_RETERROR, name.GetConstPtr() );
		return False;
	}

	pVertexShader = Mix::Graphics::DX9::VertexShader::CreateInstance( this, pD3DVertexShader, name.GetConstPtr() );
	if( pVertexShader != NULL )
	{
		( *ppVertexShader ) = pVertexShader;
	}
	else
	{
		MIX_RELEASE( pD3DVertexShader );
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", FAILED_CREATEVERTEXSHADER, Mix::STR_OUTOFMEMORY, name.GetConstPtr() );
		return False;
	}

	MIX_LOG_INFO( L"o[ebNXVF[_[쐬 : Name[%s]", name.GetConstPtr() );

	return True;
}

Boolean Device::CreateVertexShaderFromFile( const wchar_t* pFileName, Mix::Graphics::IVertexShader** ppVertexShader )
{
	if( ( pFileName == NULL ) ||
		( ::wcslen( pFileName ) == 0 ) ||
		( ppVertexShader == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pFileName[%s] ppVertexShader[%s]",
			FAILED_CREATEVERTEXSHADER,
			Mix::STR_ILLEGALARG,
			( pFileName != NULL )? pFileName : L"NULL",
			( ppVertexShader != NULL )? L"" : L"~"
			);

		return False;
	}

	HRESULT ret;
	Mix::Memory::IBuffer* pBuffer = NULL;
	Mix::Graphics::DX9::VertexShader* pVertexShader;
	IDirect3DVertexShader9* pD3DVertexShader = NULL;

	if( m_pFileMgr->CreateBufferFromFile( pFileName, &pBuffer ) == False )
	{
		MIX_LOG_ERROR( L"%s : t@C܂ : File[%s]", FAILED_CREATEVERTEXSHADER, pFileName );
		return False;
	}

	ret = m_pD3DDevice->CreateVertexShader( static_cast<const DWORD*>( pBuffer->GetConstPointer() ), &pD3DVertexShader );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : CreateVertexShader %s : File[%s]", FAILED_CREATEVERTEXSHADER, Mix::STR_RETERROR, pFileName );
		MIX_RELEASE( pBuffer );
		return False;
	}

	MIX_RELEASE( pBuffer );

	pVertexShader = Mix::Graphics::DX9::VertexShader::CreateInstance( this, pD3DVertexShader, pFileName );
	if( pVertexShader != NULL )
	{
		( *ppVertexShader ) = pVertexShader;
	}
	else
	{
		MIX_RELEASE( pD3DVertexShader );
		MIX_LOG_ERROR( L"%s : %s : File[%s]", FAILED_CREATEVERTEXSHADER, Mix::STR_OUTOFMEMORY, pFileName );
		return False;
	}

	MIX_LOG_INFO( L"o[ebNXVF[_[쐬 : File[%s]", pFileName );

	return True;
}

Boolean Device::CreatePixelShaderFromMemory( const void* pSrcData, UInt32 srcDataSize, Mix::Graphics::IPixelShader** ppPixelShader, const wchar_t* pName )
{
	Mix::String name = MIX_SAFE_NAME( pName );

	if( ( pSrcData == NULL ) ||
		( srcDataSize == 0 ) ||
		( ppPixelShader == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : pSrcData[%s] srcDataSize[%d] ppPixelShader[%s]",
			FAILED_CREATEPIXELSHADER,
			Mix::STR_ILLEGALARG,
			name.GetConstPtr(),
			( pSrcData != NULL )? L"" : L"~",
			srcDataSize,
			( ppPixelShader != NULL )? L"" : L"~"
			);

		return False;
	}

	HRESULT ret;
	Mix::Graphics::DX9::PixelShader* pPixelShader;
	IDirect3DPixelShader9* pD3DPixelShader;
	
	ret = m_pD3DDevice->CreatePixelShader( static_cast<const DWORD*>( pSrcData ), &pD3DPixelShader );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : CreatePixelShader %s : Name[%s]", FAILED_CREATEPIXELSHADER, Mix::STR_RETERROR, name.GetConstPtr() );
		return False;
	}

	pPixelShader = Mix::Graphics::DX9::PixelShader::CreateInstance( this, pD3DPixelShader, name.GetConstPtr() );
	if( pPixelShader != NULL )
	{
		( *ppPixelShader ) = pPixelShader;
	}
	else
	{
		MIX_RELEASE( pD3DPixelShader );
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", FAILED_CREATEPIXELSHADER, Mix::STR_OUTOFMEMORY, name.GetConstPtr() );
		return False;
	}

	MIX_LOG_INFO( L"sNZVF[_[쐬 : Name[%s]", name.GetConstPtr() );

	return True;
}

Boolean Device::CreatePixelShaderFromFile( const wchar_t* pFileName, Mix::Graphics::IPixelShader** ppPixelShader )
{
	if( ( pFileName == NULL ) ||
		( ::wcslen( pFileName ) == 0 ) ||
		( ppPixelShader == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pFileName[%s] ppPixelShader[%s]",
			FAILED_CREATEPIXELSHADER,
			Mix::STR_ILLEGALARG,
			( pFileName != NULL )? pFileName : L"NULL",
			( ppPixelShader != NULL )? L"" : L"~"
			);

		return False;
	}

	HRESULT ret;
	Mix::Memory::IBuffer* pBuffer = NULL;
	Mix::Graphics::DX9::PixelShader* pPixelShader;
	IDirect3DPixelShader9* pD3DPixelShader = NULL;

	if( m_pFileMgr->CreateBufferFromFile( pFileName, &pBuffer ) == False )
	{
		MIX_LOG_ERROR( L"%s : t@C܂ : File[%s]", FAILED_CREATEPIXELSHADER, pFileName );
		return False;
	}

	ret = m_pD3DDevice->CreatePixelShader( static_cast<const DWORD*>( pBuffer->GetConstPointer() ), &pD3DPixelShader );
	if( ret != D3D_OK )
	{
		MIX_RELEASE( pBuffer );
		MIX_LOG_ERROR( L"%s : CreatePixelShader %s : File[%s]", FAILED_CREATEPIXELSHADER, Mix::STR_RETERROR, pFileName );
		return False;
	}

	MIX_RELEASE( pBuffer );

	pPixelShader = Mix::Graphics::DX9::PixelShader::CreateInstance( this, pD3DPixelShader, pFileName );
	if( pPixelShader != NULL )
	{
		( *ppPixelShader ) = pPixelShader;
	}
	else
	{
		MIX_RELEASE( pD3DPixelShader );
		MIX_LOG_ERROR( L"%s : %s : File[%s]", FAILED_CREATEPIXELSHADER, Mix::STR_OUTOFMEMORY, pFileName );
		return False;
	}

	MIX_LOG_INFO( L"sNZVF[_[쐬 : File[%s]", pFileName );

	return True;
}

Boolean Device::CreateShaderConstant( UInt32 size, Boolean bDynamic, const void* pInitialData, Mix::Graphics::IShaderConstant** ppShaderConstant, const wchar_t* pName )
{
	MIX_ERROR( L"VF[_[RX^g̍쐬Ɏs : VF[_[fR CreateShaderConstant ͎gpł܂" );
	return False;
}

Boolean Device::CreateVertexBuffer( UInt32 count, UInt32 stride, UInt32 resizeStep, Boolean bDynamic, const void* pInitialData, Mix::Graphics::IVertexBuffer** ppVertexBuffer, const wchar_t* pName )
{
	Mix::String name = MIX_SAFE_NAME( pName );

	if( ( count == 0 ) ||
		( stride == 0 ) ||
		( ppVertexBuffer == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : count[%d] stride[%d] ppVertexBuffer[%s]",
			FAILED_CREATEVERTEXBUFFER,
			Mix::STR_ILLEGALARG,
			name.GetConstPtr(),
			count,
			stride,
			( ppVertexBuffer != NULL )? L"" : L"~"
			);

		return False;
	}

	if( ( resizeStep > 0 ) &&
		( bDynamic == False ) )
	{
		MIX_LOG_WARNING( L"%s : Ił͂Ȃ̂ɃTCYXebvw肳Ă܂ : Name[%s]", FAILED_CREATEVERTEXBUFFER, name.GetConstPtr() );
	}

	HRESULT ret;
	IDirect3DVertexBuffer9* pD3DVertexBuffer;

	//D3Do[ebNXobt@̍쐬
	ret = CreateVertexBuffer( count, stride, bDynamic, &pD3DVertexBuffer );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : CreateVertexBuffer %s : Name[%d] Result[%s]", FAILED_CREATEVERTEXBUFFER, Mix::STR_RETERROR, name.GetConstPtr(), GetD3DResultText( ret ) );
		return False;
	}

	//o[ebNXobt@C^[tF[X쐬
	Mix::Graphics::DX9::VertexBuffer* pVertexBuffer = Mix::Graphics::DX9::VertexBuffer::CreateInstance( this, count, stride, resizeStep, bDynamic, pD3DVertexBuffer, name.GetConstPtr() );
	if( pVertexBuffer == NULL )
	{
		MIX_RELEASE( pD3DVertexBuffer );
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", FAILED_CREATEVERTEXBUFFER, Mix::STR_OUTOFMEMORY, name.GetConstPtr() );
		return False;
	}

	//o[ebNXobt@
	ret = pVertexBuffer->Initialize( pInitialData );
	if( ret == D3D_OK )
	{
		( *ppVertexBuffer ) = pVertexBuffer;
	}
	else
	{
		MIX_RELEASE( pVertexBuffer );
		MIX_LOG_ERROR( L"%s : Initialize %s : Name[%s] Result[%s]", FAILED_CREATEVERTEXBUFFER, Mix::STR_RETERROR, name.GetConstPtr(), GetD3DResultText( ret ) );
		return False;
	}

	MIX_LOG_INFO( L"o[ebNXobt@쐬 : Name[%s] Count[%d] Stride[%d] Dynamic[%s] ResizeStep[%d]",
		name.GetConstPtr(),
		count,
		stride,
		( bDynamic == True )? L"" : L"~",
		resizeStep
		);

	return True;
}

Boolean Device::CreateIndexBuffer( Mix::Graphics::INDEX_TYPE type, UInt32 count, UInt32 resizeStep, Boolean bDynamic, const void* pInitialData, Mix::Graphics::IIndexBuffer** ppIndexBuffer, const wchar_t* pName )
{
	Mix::String name = MIX_SAFE_NAME( pName );

	if( ( count == 0 ) ||
		( ppIndexBuffer == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : count[%d] ppIndexBuffer[%s]",
			FAILED_CREATEINDEXBUFFER,
			Mix::STR_ILLEGALARG,
			name.GetConstPtr(),
			count,
			( ppIndexBuffer != NULL )? L"" : L"~"
			);

		return False;
	}

	HRESULT ret;
	D3DFORMAT format;
	IDirect3DIndexBuffer9* pD3DIndexBuffer;

	//tH[}bg
	switch( type )
	{
	case Mix::Graphics::INDEX_USHORT:
		format = D3DFMT_INDEX16;
		break;
	case Mix::Graphics::INDEX_UINT:
		format = D3DFMT_INDEX32;
		break;
	}

	//D3DCfbNXobt@쐬
	ret = CreateIndexBuffer( format, count, bDynamic, &pD3DIndexBuffer );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : CreateIndexBuffer %s : Name[%s] Result[%s]", FAILED_CREATEINDEXBUFFER, Mix::STR_RETERROR, name.GetConstPtr(), GetD3DResultText( ret ) );
		return False;
	}

	//CfbNXobt@C^[tF[X쐬
	Mix::Graphics::DX9::IndexBuffer* pIndexBuffer = Mix::Graphics::DX9::IndexBuffer::CreateInstance( this, type, format, count, resizeStep, bDynamic, pD3DIndexBuffer, name.GetConstPtr() );
	if( pIndexBuffer == NULL )
	{
		MIX_RELEASE( pD3DIndexBuffer );
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", FAILED_CREATEINDEXBUFFER, Mix::STR_OUTOFMEMORY, name.GetConstPtr() );
		return False;
	}

	//CfbNXobt@
	ret = pIndexBuffer->Initialize( pInitialData );
	if( ret == D3D_OK )
	{
		( *ppIndexBuffer ) = pIndexBuffer;
	}
	else
	{
		MIX_RELEASE( pIndexBuffer );
		MIX_LOG_ERROR( L"%s : Initialize %s : Name[%s] Result[%s]", FAILED_CREATEINDEXBUFFER, Mix::STR_RETERROR, name.GetConstPtr(), GetD3DResultText( ret ) );
		return False;
	}

	MIX_LOG_INFO( L"CfbNXobt@쐬 : Name[%s] Count[%d] Dynamic[%s] ResizeStep[%d]",
		name.GetConstPtr(),
		count,
		( bDynamic == True )? L"" : L"~",
		resizeStep
		);

	return True;
}

Boolean Device::CreateTextureFromFile( const wchar_t* pFileName, Mix::Graphics::ITexture** ppTexture )
{
	if( ( pFileName == NULL ) ||
		( ppTexture == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pFileName[%s] ppTexture[%s]",
			FAILED_CREATETEXTURE,
			Mix::STR_ILLEGALARG,
			( pFileName != NULL )? pFileName : L"NULL",
			( ppTexture != NULL )? L"" : L"~"
			);

		return False;
	}

	Mix::Graphics::Common::DeviceObject* pDeviceObject = FindDeviceObject( pFileName );

	//RNVɓo^Ăꍇ́A擾
	if( pDeviceObject != NULL )
	{
		switch( pDeviceObject->GetType() )
		{
		case Mix::Graphics::Common::DOT_TEXTURE_2D:
		case Mix::Graphics::Common::DOT_TEXTURE_CUBE:
			( *ppTexture ) = dynamic_cast<Mix::Graphics::ITexture*>( pDeviceObject );
			MIX_ADD_REF( ( *ppTexture ) );
			MIX_LOG_INFO( L"eNX`쐬 : LbV擾 : File[%s]", pFileName );
			break;
		default:
			MIX_LOG_ERROR( L"%s : w肳ꂽt@C̓eNX`ł͂܂ : File[%s]", FAILED_CREATETEXTURE, pFileName );
			return False;
		}
	}
	else
	{
		D3DSURFACE_DESC desc;
		Mix::Memory::IBuffer* pBuffer;
		Mix::Graphics::ITexture* pTexture = NULL;

		//obt@쐬
		if( m_pFileMgr->CreateBufferFromFile( pFileName, &pBuffer ) == False )
		{
			return False;
		}

		//C^[tF[X쐬
		pTexture = CreateTexture( pBuffer->GetConstPointer(), pBuffer->GetSize(), pFileName, L"File", desc );
		if( pTexture != NULL )
		{
			MIX_RELEASE( pBuffer );
			( *ppTexture ) = pTexture;
		}
		else
		{
			MIX_RELEASE( pBuffer );
			return False;
		}

		MIX_LOG_INFO( L"eNX`쐬 : File[%s] Width[%d] Height[%d] Format[%s]", pFileName, desc.Width, desc.Height, GetD3DFormatText( desc.Format ) );
	}

	return True;
}

Boolean Device::CreateTextureFromMemory( const wchar_t* pName, const void* pSrc, UInt32 srcSize, Mix::Graphics::ITexture** ppTexture )
{
	if( ( pName == NULL ) ||
		( ::wcslen( pName ) == 0 ) ||
		( pSrc == NULL ) ||
		( srcSize == 0 ) ||
		( ppTexture == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pName[%s] pSrc, srcSize[%d] ppTexture[%s]",
			FAILED_CREATETEXTURE,
			Mix::STR_ILLEGALARG,
			( pName != NULL )? pName : L"NULL",
			( pSrc != NULL )? L"" : L"~",
			srcSize,
			( ppTexture != NULL )? L"" : L"~"
			);

		return False;
	}

	Mix::Graphics::Common::DeviceObject* pDeviceObject = FindDeviceObject( pName );

	//RNVɓo^Ăꍇ́A擾
	if( pDeviceObject != NULL )
	{
		switch( pDeviceObject->GetType() )
		{
		case Mix::Graphics::Common::DOT_TEXTURE_2D:
		case Mix::Graphics::Common::DOT_TEXTURE_CUBE:
			( *ppTexture ) = dynamic_cast<Mix::Graphics::ITexture*>( pDeviceObject );
			MIX_ADD_REF( ( *ppTexture ) );
			MIX_LOG_INFO( L"eNX`쐬 : LbV擾 : Name[%s]", pName );
			break;
		default:
			MIX_LOG_ERROR( L"%s : w肳ꂽÕ\[X̓eNX`ł͂܂ : Name[%s]", FAILED_CREATETEXTURE, pName );
			return False;
		}
	}
	else
	{
		D3DSURFACE_DESC desc;
		Mix::Graphics::ITexture* pTexture;

		//C^[tF[X쐬
		pTexture = CreateTexture( pSrc, srcSize, pName, L"Name", desc );
		if( pTexture != NULL )
		{
			( *ppTexture ) = pTexture;
		}
		else
		{
			return False;
		}

		MIX_LOG_INFO( L"eNX`쐬 : Name[%s] Width[%d] Height[%d] Format[%s]", pName, desc.Width, desc.Height, GetD3DFormatText( desc.Format ) );
	}

	return True;
}

Boolean Device::CreateCubeTexture( UInt32 edgeLength, Mix::Graphics::FORMAT format, Mix::Graphics::ITexture** ppTexture, const wchar_t* pName )
{
	Mix::String name = MIX_SAFE_NAME( pName );

	if( ( edgeLength == 0 ) ||
		( ppTexture == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : edgeLength[%d] ppTexture[%s]",
			FAILED_CREATECUBETEXTURE,
			Mix::STR_ILLEGALARG,
			name.GetConstPtr(),
			edgeLength,
			MIX_LOG_PTR( ppTexture ) );

		return False;
	}

	HRESULT ret;
	const FORMATINFO* pFmtInfo;
	IDirect3DCubeTexture9* pD3DTexture;
	Mix::Graphics::DX9::TextureCube* pTexture;

	pFmtInfo = &( m_DeviceParam.current.formatTable[format] );
	if( MIX_TESTBIT( pFmtInfo->flags, Device::FLAG_RESOURCE_CUBE) != Device::FLAG_RESOURCE_CUBE )
	{
		MIX_LOG_ERROR( L"%s : T|[gȂtH[}bg : Name[%s] : Format[%s]", FAILED_CREATECUBETEXTURE,
			name.GetConstPtr(),
			GetD3DFormatText( pFmtInfo->format ) );

		return False;
	}

	ret = CreateCubeTexture( edgeLength, pFmtInfo->format, &pD3DTexture );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : CreateCubeTexture %s : Name[%s] Result[%s]", FAILED_CREATECUBETEXTURE, Mix::STR_RETERROR,
			name.GetConstPtr(),
			GetD3DResultText( ret ) );

		return False;
	}

	pTexture = Mix::Graphics::DX9::TextureCube::CreateInstance(	this,
																False, edgeLength, Device::ConvertTextureFormat( pFmtInfo->format ),
																pFmtInfo->format, pD3DTexture,
																name.GetConstPtr() );
	if( pTexture == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", FAILED_CREATECUBETEXTURE, Mix::STR_OUTOFMEMORY, name.GetConstPtr() );
		MIX_RELEASE( pD3DTexture );
		return False;
	}

	( *ppTexture ) = pTexture;

	MIX_LOG_INFO( L"L[ueNX`쐬 : Name[%s] EdgeLength[%d] Format[%s]", name.GetConstPtr(), edgeLength, GetD3DFormatText( pFmtInfo->format ) );

	return True;
}

Boolean Device::CreateDynamicTexture( UInt32 width, UInt32 height, Mix::Graphics::FORMAT format, Mix::Graphics::ITexture** ppTexture, const wchar_t* pName )
{
	Mix::String name = MIX_SAFE_NAME( pName );

	if( ( width == 0 ) ||
		( height == 0 ) ||
		( ppTexture == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : width[%d] height[%d] ppTexture[%s]",
			FAILED_CREATEDYNAMICTEXTURE,
			Mix::STR_ILLEGALARG,
			name.GetConstPtr(),
			width,
			height,
			( ppTexture != NULL )? L"" : L"~"
			);

		return False;
	}

	HRESULT ret;
	const FORMATINFO* pFmtInfo;
	IDirect3DTexture9* pD3DTexture;
	Mix::Graphics::DX9::Texture2D* pTexture;

	//D3DtH[}bg擾
	pFmtInfo = &( m_DeviceParam.current.formatTable[format] );

	//D3DeNX`쐬
	ret = CreateDynamicTexture( width, height, pFmtInfo->format, &pD3DTexture );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : CreateDynamicTexture %s : Name[%s] Result[%s]", FAILED_CREATEDYNAMICTEXTURE, Mix::STR_RETERROR, name.GetConstPtr(), GetD3DResultText( ret ) );
		return False;
	}

	//eNX`C^[tF[X쐬
	pTexture = Mix::Graphics::DX9::Texture2D::CreateInstance(	this,
																True, width, height, Device::ConvertTextureFormat( pFmtInfo->format ),
																pFmtInfo->format, pD3DTexture,
																False, name.GetConstPtr() );
	if( pTexture != NULL )
	{
		( *ppTexture ) = pTexture;
	}
	else
	{
		MIX_RELEASE( pD3DTexture );
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", FAILED_CREATEDYNAMICTEXTURE, Mix::STR_OUTOFMEMORY, name.GetConstPtr() );
		return False;
	}

	MIX_LOG_INFO( L"_Ci~bNeNX`쐬 : Name[%s] Width[%d] Height[%d] Format[%s]",
		name.GetConstPtr(),
		width,
		height,
		GetD3DFormatText( pFmtInfo->format )
		);

	return True;
}

Boolean Device::CreateTargetTexture( UInt32 width, UInt32 height, Mix::Graphics::FORMAT format, Mix::Graphics::ITexture** ppTexture, const wchar_t* pName )
{
	Mix::String name = MIX_SAFE_NAME( pName );

	if( ( width == 0 ) ||
		( height == 0 ) ||
		( ppTexture == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : width[%d] height[%d] ppTexture[%s]",
			FAILED_CREATETARGETTEXTURE,
			Mix::STR_ILLEGALARG,
			name.GetConstPtr(),
			width,
			height,
			( ppTexture != NULL )? L"" : L"~"
			);

		return False;
	}

	HRESULT ret;
	const FORMATINFO* pFmtInfo;
	IDirect3DTexture9* pD3DTexture;
	IDirect3DSurface9* pD3DSurface;
	Mix::Graphics::DX9::TextureTarget* pTexture;

	//tH[}bg擾
	pFmtInfo = &( m_DeviceParam.current.formatTable[format] );

	//tH[}bg`FbN
	if( MIX_TESTBIT( pFmtInfo->flags, FLAG_RESOURCE_TARGET ) != FLAG_RESOURCE_TARGET )
	{
		MIX_LOG_ERROR( L"%s : T|[gȂtH[}bg : Name[%s] Format[%s]", FAILED_CREATETARGETTEXTURE, name.GetConstPtr(), GetD3DFormatText( pFmtInfo->format ) );
		return False;
	}

	//D3DeNX`쐬
	ret = CreateTargetTexture( width, height, pFmtInfo->format, &pD3DTexture );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : CreateTargetTexture %s : Name[%s] Rsult[%s]", FAILED_CREATETARGETTEXTURE, Mix::STR_RETERROR, name.GetConstPtr(), GetD3DResultText( ret ) );
		return False;
	}

	//D3DT[tFCX擾
	ret = pD3DTexture->GetSurfaceLevel( 0, &pD3DSurface );
	if( ret != D3D_OK )
	{
		MIX_RELEASE( pD3DTexture );
		MIX_LOG_ERROR( L"%s : GetSurfaceLevel %s : Name[%s] Result[%s]", FAILED_CREATETARGETTEXTURE, Mix::STR_RETERROR, name.GetConstPtr(), GetD3DResultText( ret ) );
		return False;
	}

	//eNX`C^[tF[X쐬
	pTexture = Mix::Graphics::DX9::TextureTarget::CreateInstance(	this,
																	width, height, Device::ConvertTextureFormat( pFmtInfo->format ),
																	pD3DTexture, pD3DSurface, pFmtInfo->format,
																	name.GetConstPtr() );
	if( pTexture != NULL )
	{
		( *ppTexture ) = pTexture;
	}
	else
	{
		MIX_RELEASE( pD3DSurface );
		MIX_RELEASE( pD3DTexture );
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", FAILED_CREATETARGETTEXTURE, Mix::STR_OUTOFMEMORY, name.GetConstPtr() );
		return False;
	}

	MIX_LOG_INFO( L"^[QbgeNX`쐬 : Name[%s] Width[%d] Height[%d] Format[%s]",
		name.GetConstPtr(),
		width,
		height,
		GetD3DFormatText( pFmtInfo->format )
		);

	return True;
}

Boolean Device::CreateDepthTexture( UInt32 width, UInt32 height, Mix::Graphics::FORMAT format, Mix::Graphics::ITexture** ppTexture, const wchar_t* pName )
{
	Mix::String name = MIX_SAFE_NAME( pName );

	if( ( width == 0 ) ||
		( height == 0 ) ||
		( ppTexture == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : width[%d] height[%d] ppTexture[%s]",
			FAILED_CREATEDEPTHTEXTURE,
			Mix::STR_ILLEGALARG,
			name.GetConstPtr(),
			width,
			height,
			( ppTexture != NULL )? L"" : L"~"
			);

		return False;
	}

	HRESULT ret;
	const FORMATINFO* pFmtInfo;
	IDirect3DSurface9* pD3DSurface;
	Mix::Graphics::DX9::TextureDepth* pTexture;

	pFmtInfo = &( m_DeviceParam.current.formatTable[format] );

	if( MIX_TESTBIT( pFmtInfo->flags, FLAG_RESOURCE_DEPTHSTENCIL ) != FLAG_RESOURCE_DEPTHSTENCIL )
	{
		MIX_LOG_ERROR( L"%s : T|[gȂtH[}bg : Name[%s] Format[%s]", FAILED_CREATEDEPTHTEXTURE, name.GetConstPtr(), GetD3DFormatText( pFmtInfo->format ) );
		return False;
	}

	ret = CreateDepthTexture( width, height, pFmtInfo->format, &pD3DSurface );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : CreateDepthTexture %s : Name[%s] Result[%s]", FAILED_CREATEDEPTHTEXTURE, Mix::STR_RETERROR, name.GetConstPtr(), GetD3DResultText( ret ) );
		return False;
	}

	pTexture = Mix::Graphics::DX9::TextureDepth::CreateInstance(	this,
																	width, height, Device::ConvertTextureFormat( pFmtInfo->format ),
																	pFmtInfo->format, pD3DSurface,
																	name.GetConstPtr() );
	if( pTexture != NULL )
	{
		( *ppTexture ) = pTexture;
	}
	else
	{
		MIX_RELEASE( pD3DSurface );
		MIX_LOG_ERROR( L"%s : %s : Name[%s]", FAILED_CREATEDEPTHTEXTURE, Mix::STR_OUTOFMEMORY, name.GetConstPtr() );
		return False;
	}

	MIX_LOG_INFO( L"fvXeNX`쐬 : Name[%s] Width[%d] Height[%d] Format[%s]",
		name.GetConstPtr(),
		width,
		height,
		GetD3DFormatText( pFmtInfo->format )
		);

	return True;
}

Boolean Device::CreateQuery( Mix::Graphics::QUERY_TYPE type, Mix::Graphics::IQuery** ppQuery )
{
	if( ppQuery == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : ppQuery[%s]",
			FAILED_CREATEQUERY,
			Mix::STR_ILLEGALARG,
			( ppQuery != NULL )? L"" : L"~"
			);

		return False;
	}

	HRESULT ret;
	IDirect3DQuery9* pD3DQuery;
	Mix::Graphics::DX9::Query* pQuery; 

	ret = CreateQuery( type, &pD3DQuery );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : CreateQuery %s : Type[%s] Result[%s]", FAILED_CREATEQUERY, Mix::STR_RETERROR, Device::D3DQueryTextTable[type], GetD3DResultText( ret ) );
		return False;
	}

	pQuery = Mix::Graphics::DX9::Query::CreateInstance( this, type, pD3DQuery, NULL );
	if( pQuery != NULL )
	{
		( *ppQuery ) = pQuery;
	}
	else
	{
		MIX_RELEASE( pD3DQuery );
		MIX_LOG_ERROR( L"%s : %s : Type[%s]", FAILED_CREATEQUERY, Mix::STR_OUTOFMEMORY, Device::D3DQueryTextTable[type] );
		return False;
	}

	MIX_LOG_INFO( L"NG쐬 : Type[%s]", Device::D3DQueryTextTable[type] );

	return True;
}

Boolean Device::CheckTextureFormat( Mix::Graphics::FORMAT format )
{
	return ( ( m_DeviceParam.current.formatTable[format].flags & FLAG_RESOURCE_TEXTURE ) == FLAG_RESOURCE_TEXTURE );
}

Boolean Device::CheckCubeTextureFormat( Mix::Graphics::FORMAT format )
{
	return ( ( m_DeviceParam.current.formatTable[format].flags & FLAG_RESOURCE_CUBE ) == FLAG_RESOURCE_CUBE );
}

Boolean Device::CheckDynamicTextureFormat( Mix::Graphics::FORMAT format )
{
	return ( ( m_DeviceParam.current.formatTable[format].flags & FLAG_RESOURCE_TEXTURE ) == FLAG_RESOURCE_TEXTURE );
}

Boolean Device::CheckTargetTextureFormat( Mix::Graphics::FORMAT format )
{
	return ( ( m_DeviceParam.current.formatTable[format].flags & FLAG_RESOURCE_TARGET ) == FLAG_RESOURCE_TARGET );
}

Boolean Device::CheckDepthTextureFormat( Mix::Graphics::FORMAT format )
{
	return ( ( m_DeviceParam.current.formatTable[format].flags & FLAG_RESOURCE_DEPTHSTENCIL ) == FLAG_RESOURCE_DEPTHSTENCIL );
}

Boolean Device::CheckIndexType( Mix::Graphics::INDEX_TYPE type )
{
	Boolean ret = False;

	switch( type )
	{
	case Mix::Graphics::INDEX_USHORT:
		ret = ( m_Caps.MaxVertexIndex >= 0x0000FFFF );
		break;

	case Mix::Graphics::INDEX_UINT:
		ret = ( m_Caps.MaxVertexIndex > 0x0000FFFF );
		break;
	}

	return ret;
}

void Device::ChangeWindowMode( Boolean bWindowed )
{
	if( m_Status == Device::STATUS_OK )
	{
		m_Status = ( GetWindowMode() == True )? Device::STATUS_CHANGE_FULLSCREEN : Device::STATUS_CHANGE_WINDOW;
	}
}

Boolean Device::GetWindowMode( void ) const
{
	return m_DeviceParam.current.params.Windowed;
}

Boolean Device::Begin( void )
{
	//v~eBu`搔Zbg
	m_DrawPrimitiveCount = 0;

	//`Xe[gZbg
//	ResetDrawState();

	//V[Jn
	return ( m_pD3DDevice->BeginScene() == D3D_OK );
}

void Device::End( void )
{
	if( m_pD3DDevice->EndScene() != D3D_OK )
	{
		MIX_LOG_ERROR( L"Mix::Graphics::IDevice::End : IDirect3DDevice9::EndScene %s", Mix::STR_RETERROR );
	}

	ResetTargetState();
	ResetDrawState();
}

void Device::Present( void )
{
	switch( m_pD3DDevice->Present( NULL, NULL, NULL, NULL ) )
	{
	case D3DERR_DEVICELOST:
		m_Status = Device::STATUS_LOST;
		break;
	case D3DERR_INVALIDCALL:
		MIX_LOG_ERROR( L"Mix::Graphics::IDevice::Present : IDirect3DDevice9::Present %s", Mix::STR_RETERROR );
		break;
	case D3DERR_DRIVERINTERNALERROR:
		m_pWindow->SendMessage( WM_MIX_INTERNALERROR, Mix::ERR_GRAPHICS_DRIVERERROR, 0 );
		break;
	}
}

void Device::SetTarget( Mix::Graphics::ITexture* pT0, Mix::Graphics::ITexture* pT1, Mix::Graphics::ITexture* pT2, Mix::Graphics::ITexture* pT3, Mix::Graphics::ITexture* pDS )
{
	if( ( ( pT0 != NULL ) && ( pT0->GetType() != Mix::Graphics::TEXTURE_TARGET ) ) ||
		( ( pT1 != NULL ) && ( pT1->GetType() != Mix::Graphics::TEXTURE_TARGET ) ) ||
		( ( pT2 != NULL ) && ( pT2->GetType() != Mix::Graphics::TEXTURE_TARGET ) ) ||
		( ( pT3 != NULL ) && ( pT3->GetType() != Mix::Graphics::TEXTURE_TARGET ) ) ||
		( ( pDS != NULL ) && ( pDS->GetType() != Mix::Graphics::TEXTURE_DEPTH ) ) )
	{
		MIX_LOG_ERROR( L"^[Qbg̐ݒɎs : pT0[%s] pT1[%s] pT2[%s] pT3[%s] pDS[%s] : ȃeNX`w肳Ă܂",
			( ( pT0 == NULL ) || ( pT0->GetType() == Mix::Graphics::TEXTURE_TARGET ) )? L"" : L"~",
			( ( pT1 == NULL ) || ( pT1->GetType() == Mix::Graphics::TEXTURE_TARGET ) )? L"" : L"~",
			( ( pT2 == NULL ) || ( pT2->GetType() == Mix::Graphics::TEXTURE_TARGET ) )? L"" : L"~",
			( ( pT3 == NULL ) || ( pT3->GetType() == Mix::Graphics::TEXTURE_TARGET ) )? L"" : L"~",
			( ( pDS == NULL ) || ( pDS->GetType() == Mix::Graphics::TEXTURE_DEPTH  ) )? L"" : L"~" );

		return;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// eNX`̔j( DirectX11 Ɠ𓯂ɂ邽 )
	////////////////////////////////////////////////////////////////////////////////////////////////////

	for( UInt32 i = 0; i < Mix::Graphics::TEXTURESTAGE_MAX; i++ )
	{
		m_pD3DDevice->SetTexture( i, NULL );
		m_DrawState.textureTable[i] = NULL;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ApɃ^[Qbgۑ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pT0 != NULL )
	{
		m_TargetState.rect = Mix::Rectangle( 0, 0, pT0->GetWidth(), pT0->GetHeight() );
	}
	else
	{
		m_TargetState.rect = Mix::Rectangle( 0, 0, m_DeviceParam.current.screenSize.x, m_DeviceParam.current.screenSize.y );
	}

	MIX_RELEASE( m_TargetState.pTex[0] );
	MIX_ADD_REF( pT0 );
	m_TargetState.pTex[0] = pT0;

	MIX_RELEASE( m_TargetState.pTex[1] );
	MIX_ADD_REF( pT1 );
	m_TargetState.pTex[1] = pT1;

	MIX_RELEASE( m_TargetState.pTex[2] );
	MIX_ADD_REF( pT2 );
	m_TargetState.pTex[2] = pT2;

	MIX_RELEASE( m_TargetState.pTex[3] );
	MIX_ADD_REF( pT3 );
	m_TargetState.pTex[3] = pT3;

	MIX_RELEASE( m_TargetState.pDSTex );
	MIX_ADD_REF( pDS );
	m_TargetState.pDSTex = pDS;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// ^[QbgtbV
	////////////////////////////////////////////////////////////////////////////////////////////////////

	FlushTargetState();
}

void Device::GetTarget( Mix::Graphics::ITexture** ppT0, Mix::Graphics::ITexture** ppT1, Mix::Graphics::ITexture** ppT2, Mix::Graphics::ITexture** ppT3, Mix::Graphics::ITexture** ppDS )
{
	if( ppT0 != NULL )
	{
		MIX_ADD_REF( m_TargetState.pTex[0] );
		( *ppT0 ) = m_TargetState.pTex[0];
	}

	if( ppT1 != NULL )
	{
		MIX_ADD_REF( m_TargetState.pTex[1] );
		( *ppT1 ) = m_TargetState.pTex[1];
	}

	if( ppT2 != NULL )
	{
		MIX_ADD_REF( m_TargetState.pTex[2] );
		( *ppT2 ) = m_TargetState.pTex[2];
	}

	if( ppT3 != NULL )
	{
		MIX_ADD_REF( m_TargetState.pTex[3] );
		( *ppT3 ) = m_TargetState.pTex[3];
	}

	if( ppDS != NULL )
	{
		MIX_ADD_REF( m_TargetState.pDSTex );
		( *ppDS ) = m_TargetState.pDSTex;
	}
}

void Device::SetViewport( Int32 x, Int32 y, Int32 width, Int32 height )
{
	if( m_TargetState.rect.Contains( Mix::Rectangle( x, y, width, height ), &m_ViewportRect ) == False )
	{
		m_ViewportRect.x = 0;
		m_ViewportRect.y = 0;
		m_ViewportRect.width = 1;
		m_ViewportRect.height = 1;
	}

	m_Viewport.X = m_ViewportRect.x;
	m_Viewport.Y = m_ViewportRect.y;
	m_Viewport.Width = m_ViewportRect.width;
	m_Viewport.Height = m_ViewportRect.height;

	m_pD3DDevice->SetViewport( &m_Viewport );
}

void Device::SetViewport( const Mix::Rectangle& rect )
{
	SetViewport( rect.x, rect.y, rect.width, rect.height );
}

const Mix::Rectangle& Device::GetViewport( void ) const
{
	return m_ViewportRect;
}

Mix::Matrix4x4 Device::GetScreenTransformMatrix( void ) const
{
//	Float32 x = static_cast<Float32>( m_ViewportRect.x );
//	Float32 y = static_cast<Float32>( m_ViewportRect.y );
	Float32 w = static_cast<Float32>( m_ViewportRect.width );
	Float32 h = static_cast<Float32>( m_ViewportRect.height );

	Mix::Matrix4x4 mat;

	mat.m00 = MIX_FLOAT_DIV( +2.0f, w );
	mat.m11 = MIX_FLOAT_DIV( -2.0f, h );
	mat.m22 = 0.0f;
	mat.m30 = -1.0f;
	mat.m31 = +1.0f;
//	mat.m30 = MIX_FLOAT_RECIPROCAL( -1.0f + ( x * 2.0f / w ) );
//	mat.m31 = MIX_FLOAT_RECIPROCAL( +1.0f - ( y * 2.0f / h ) );

	return mat;
}

void Device::SetScissor( Int32 x, Int32 y, Int32 width, Int32 height )
{
	RECT rect;

	if( m_TargetState.rect.Contains( Mix::Rectangle( x, y, width, height ), &m_ScissorRect ) == False )
	{
		m_ScissorRect.x = 0;
		m_ScissorRect.y = 0;
		m_ScissorRect.width = 1;
		m_ScissorRect.height = 1;
	}

	rect.left = m_ScissorRect.x;
	rect.top = m_ScissorRect.y;
	rect.right = m_ScissorRect.x + m_ScissorRect.width;
	rect.bottom = m_ScissorRect.y + m_ScissorRect.height;

	m_pD3DDevice->SetScissorRect( &rect );
}

void Device::SetScissor( const Mix::Rectangle& rect )
{
	SetScissor( rect.x, rect.y, rect.width, rect.height );
}

const Mix::Rectangle& Device::GetScissor( void ) const
{
	return m_ScissorRect;
}

void Device::Clear( Boolean clearTarget, Boolean clearDepth, const Mix::Vector4& color, Float32 z )
{
	DWORD flags = 0;

	if( clearTarget == True ) { flags |= D3DCLEAR_TARGET; }
	if( clearDepth == True ) { flags |= D3DCLEAR_ZBUFFER; }

	if( flags != 0 )
	{
		UInt32 colorI;

		colorI = MIX_CLAMP( static_cast<UInt32>( color.a * 255.0f ), 0, 255 ) << 24;
		colorI |= MIX_CLAMP( static_cast<UInt32>( color.r * 255.0f ), 0, 255 ) << 16;
		colorI |= MIX_CLAMP( static_cast<UInt32>( color.g * 255.0f ), 0, 255 ) << 8;
		colorI |= MIX_CLAMP( static_cast<UInt32>( color.b * 255.0f ), 0, 255 );

		m_pD3DDevice->Clear( 0, NULL, flags, colorI, z, 0L );
	}
}

void Device::SetRasterizerState( Mix::Graphics::FILL_TYPE fillMode, Mix::Graphics::CULL_TYPE cullMode, Boolean bScissoring, Boolean bMultisample )
{
	Mix::Graphics::RASTERIZER_DESC desc;

	desc.fillMode = fillMode;
	desc.cullMode = cullMode;
	desc.bScissoring = bScissoring;
	desc.bMultisample = bMultisample;

	SetRasterizerState( desc );
}

void Device::SetRasterizerState( const Mix::Graphics::RASTERIZER_DESC& desc )
{
	m_RasterizeDesc = desc;

	SetRenderState( Device::RENDERSTATETYPE_FILLMODE, ( m_RasterizeDesc.fillMode == Mix::Graphics::FILL_WIREFRAME )? D3DFILL_WIREFRAME : D3DFILL_SOLID );
	SetRenderState( Device::RENDERSTATETYPE_CULLMODE, Device::D3DCullModeTable[m_RasterizeDesc.cullMode] );
	SetRenderState( Device::RENDERSTATETYPE_SCISSORTESTENABLE, m_RasterizeDesc.bScissoring );
}

const Mix::Graphics::RASTERIZER_DESC& Device::GetRasterizerState( void ) const
{
	return m_RasterizeDesc;
}

void Device::SetDepthState( Boolean bTest, Boolean bWrite, Mix::Graphics::ZFUNC_TYPE func )
{
	Mix::Graphics::DEPTH_DESC desc;

	desc.bTest = bTest;
	desc.bWrite = bWrite;
	desc.func = func;

	SetDepthState( desc );
}

void Device::SetDepthState( const Mix::Graphics::DEPTH_DESC& desc )
{
	m_DepthDesc = desc;

	SetRenderState( Device::RENDERSTATETYPE_ZENABLE, ( m_DepthDesc.bTest == True )? D3DZB_TRUE : D3DZB_FALSE );
	SetRenderState( Device::RENDERSTATETYPE_ZWRITEENABLE, m_DepthDesc.bWrite );
	SetRenderState( Device::RENDERSTATETYPE_ZFUNC, Device::D3DZFuncTable[m_DepthDesc.func] );
}

const Mix::Graphics::DEPTH_DESC& Device::GetDepthState( void ) const
{
	return m_DepthDesc;
}

void Device::SetBlendState( Mix::Graphics::BLEND_TYPE type, Mix::Graphics::COLOR_WRITE_MASK colorWriteMask )
{
	Mix::Graphics::BLEND_DESC desc;

	desc.type = type;
	desc.colorWriteMask = colorWriteMask;

	SetBlendState( desc );
}

void Device::SetBlendState( const Mix::Graphics::BLEND_DESC& desc )
{
	UInt32 mask;

	m_BlendDesc = desc;

	switch( m_BlendDesc.type )
	{
	case Mix::Graphics::BLEND_COPY:
		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND, D3DBLEND_ONE );
		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND, D3DBLEND_ZERO );
		SetRenderState( Device::RENDERSTATETYPE_BLENDOP, D3DBLENDOP_ADD );
		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND_ALPHA, D3DBLEND_ONE );
		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND_ALPHA, D3DBLEND_ZERO );
		SetRenderState( Device::RENDERSTATETYPE_BLENDOP_ALPHA, D3DBLENDOP_ADD );
		SetRenderState( Device::RENDERSTATETYPE_ALPHABLENDENABLE, FALSE );
		break;
	case Mix::Graphics::BLEND_NORMAL:
		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND, D3DBLEND_SRCALPHA );
		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND, D3DBLEND_INVSRCALPHA );
		SetRenderState( Device::RENDERSTATETYPE_BLENDOP, D3DBLENDOP_ADD );
		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND_ALPHA, D3DBLEND_SRCALPHA );
		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND_ALPHA, D3DBLEND_INVSRCALPHA );
		SetRenderState( Device::RENDERSTATETYPE_BLENDOP_ALPHA, D3DBLENDOP_ADD );
		SetRenderState( Device::RENDERSTATETYPE_ALPHABLENDENABLE, TRUE );
		break;
	case Mix::Graphics::BLEND_ADD:
		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND, D3DBLEND_SRCALPHA );
		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND, D3DBLEND_ONE );
		SetRenderState( Device::RENDERSTATETYPE_BLENDOP, D3DBLENDOP_ADD );
		SetRenderState( Device::RENDERSTATETYPE_ALPHABLENDENABLE, TRUE );
//		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND_ALPHA, D3DBLEND_ONE );
//		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND_ALPHA, D3DBLEND_ZERO );
//		SetRenderState( Device::RENDERSTATETYPE_BLENDOP_ALPHA, D3DBLENDOP_ADD );
		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND_ALPHA, D3DBLEND_SRCALPHA );
		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND_ALPHA, D3DBLEND_ONE );
		SetRenderState( Device::RENDERSTATETYPE_BLENDOP_ALPHA, D3DBLENDOP_ADD );
		SetRenderState( Device::RENDERSTATETYPE_ALPHABLENDENABLE, TRUE );
		break;
	case Mix::Graphics::BLEND_SUB:
		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND, D3DBLEND_SRCALPHA );
		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND, D3DBLEND_ONE );
		SetRenderState( Device::RENDERSTATETYPE_BLENDOP, D3DBLENDOP_REVSUBTRACT );
		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND_ALPHA, D3DBLEND_SRCALPHA );
		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND_ALPHA, D3DBLEND_ONE );
		SetRenderState( Device::RENDERSTATETYPE_BLENDOP_ALPHA, D3DBLENDOP_REVSUBTRACT );
		SetRenderState( Device::RENDERSTATETYPE_ALPHABLENDENABLE, TRUE );
		break;
	case Mix::Graphics::BLEND_MUL:
		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND, D3DBLEND_ZERO );
		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND, D3DBLEND_SRCCOLOR );
		SetRenderState( Device::RENDERSTATETYPE_BLENDOP, D3DBLENDOP_ADD );
		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND_ALPHA, D3DBLEND_ZERO );
		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND_ALPHA, D3DBLEND_SRCALPHA );
		SetRenderState( Device::RENDERSTATETYPE_BLENDOP_ALPHA, D3DBLENDOP_ADD );
		SetRenderState( Device::RENDERSTATETYPE_ALPHABLENDENABLE, TRUE );
		break;
	case Mix::Graphics::BLEND_SCREEN:
		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND, D3DBLEND_INVDESTCOLOR );
		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND, D3DBLEND_ONE );
		SetRenderState( Device::RENDERSTATETYPE_BLENDOP, D3DBLENDOP_ADD );
		SetRenderState( Device::RENDERSTATETYPE_SRCBLEND_ALPHA, D3DBLEND_INVDESTALPHA );
		SetRenderState( Device::RENDERSTATETYPE_DESTBLEND_ALPHA, D3DBLEND_ONE );
		SetRenderState( Device::RENDERSTATETYPE_BLENDOP_ALPHA, D3DBLENDOP_ADD );
		SetRenderState( Device::RENDERSTATETYPE_ALPHABLENDENABLE, TRUE );
		break;
	}

	switch( m_BlendDesc.colorWriteMask )
	{
	case Mix::Graphics::COLOR_WRITE_RGBA:
		mask = D3DCOLORWRITEENABLE_RED | D3DCOLORWRITEENABLE_GREEN | D3DCOLORWRITEENABLE_BLUE | D3DCOLORWRITEENABLE_ALPHA;
		break;
	case Mix::Graphics::COLOR_WRITE_RGB:
		mask = D3DCOLORWRITEENABLE_RED | D3DCOLORWRITEENABLE_GREEN | D3DCOLORWRITEENABLE_BLUE;
		break;
	case Mix::Graphics::COLOR_WRITE_A:
		mask = D3DCOLORWRITEENABLE_ALPHA;
		break;
	}

	SetRenderState( Device::RENDERSTATETYPE_WRITECOLOR, mask );
}

const Mix::Graphics::BLEND_DESC& Device::GetBlendState( void ) const
{
	return m_BlendDesc;
}

void Device::SetVertexLayout( Mix::Graphics::IVertexLayout* pVertexLayout )
{
	m_DrawState.next.pVertexLayout = pVertexLayout;
}

void Device::SetHullShader( Mix::Graphics::IHullShader* pHullShader )
{
	MIX_ERROR( L"nVF[_[̐ݒɎs : VF[_[fR SetHullShader ͎gpł܂" );
}

void Device::SetDomainShader( Mix::Graphics::IDomainShader* pDomainShader )
{
	MIX_ERROR( L"hCVF[_[̐ݒɎs : VF[_[fR SetDomainShader ͎gpł܂" );
}

void Device::SetGeometryShader( Mix::Graphics::IGeometryShader* pGeometryShader )
{
	MIX_ERROR( L"WIgVF[_[̐ݒɎs : VF[_[fR SetGeometryShader ͎gpł܂" );
}

void Device::SetVertexShader( Mix::Graphics::IVertexShader* pVertexShader )
{
	m_DrawState.next.pVertexShader = pVertexShader;
}

void Device::SetPixelShader( Mix::Graphics::IPixelShader* pPixelShader )
{
	m_DrawState.next.pPixelShader = pPixelShader;
}

void Device::SetVertexShaderConstantB( UInt32 startRegister, const Boolean* data, UInt32 count )
{
	MIX_ASSERT( ( data != NULL ) && ( count > 0 ) );

	m_pD3DDevice->SetVertexShaderConstantB( startRegister, data, count );
}

void Device::SetVertexShaderConstantI( UInt32 startRegister, const Int32* data, UInt32 count )
{
	MIX_ASSERT( ( data != NULL ) && ( count > 0 ) );

	m_pD3DDevice->SetVertexShaderConstantI( startRegister, data, count );
}

void Device::SetVertexShaderConstantF( UInt32 startRegister, const Float32* data, UInt32 count )
{
	MIX_ASSERT( ( data != NULL ) && ( count > 0 ) );

	m_pD3DDevice->SetVertexShaderConstantF( startRegister, data, count );
}

void Device::SetVertexShaderConstantF( UInt32 startRegister, const Mix::Vector4* data, UInt32 count )
{
	MIX_ASSERT( ( data != NULL ) && ( count > 0 ) );

	m_pD3DDevice->SetVertexShaderConstantF( startRegister, reinterpret_cast<const Float32*>( data ), count );
}

void Device::SetVertexShaderConstantF( UInt32 startRegister, const Mix::Matrix4x4* data, UInt32 count )
{
	MIX_ASSERT( ( data != NULL ) && ( count > 0 ) );

	m_pD3DDevice->SetVertexShaderConstantF( startRegister, reinterpret_cast<const Float32*>( data ), 4 * count );
}

void Device::SetPixelShaderConstantB( UInt32 startRegister, const Boolean* data, UInt32 count )
{
	MIX_ASSERT( ( data != NULL ) && ( count > 0 ) );

	m_pD3DDevice->SetPixelShaderConstantB( startRegister, data, count );
}

void Device::SetPixelShaderConstantI( UInt32 startRegister, const Int32* data, UInt32 count )
{
	MIX_ASSERT( ( data != NULL ) && ( count > 0 ) );

	m_pD3DDevice->SetPixelShaderConstantI( startRegister, data, count );
}

void Device::SetPixelShaderConstantF( UInt32 startRegister, const Float32* data, UInt32 count )
{
	MIX_ASSERT( ( data != NULL ) && ( count > 0 ) );

	m_pD3DDevice->SetPixelShaderConstantF( startRegister, data, count );
}

void Device::SetPixelShaderConstantF( UInt32 startRegister, const Mix::Vector4* data, UInt32 count )
{
	MIX_ASSERT( ( data != NULL ) && ( count > 0 ) );

	m_pD3DDevice->SetPixelShaderConstantF( startRegister, reinterpret_cast<const Float32*>( data ), count );
}

void Device::SetPixelShaderConstantF( UInt32 startRegister, const Mix::Matrix4x4* data, UInt32 count )
{
	MIX_ASSERT( ( data != NULL ) && ( count > 0 ) );

	m_pD3DDevice->SetPixelShaderConstantF( startRegister, reinterpret_cast<const Float32*>( data ), 4 * count );
}

void Device::SetHullShaderConstant( UInt32 slot, Mix::Graphics::IShaderConstant* pShaderConstant )
{
	MIX_ERROR( L"nVF[_RX^g̐ݒɎs : VF[_[fR SetHullShaderConstant ͎gpł܂" );
}

void Device::SetDomainShaderConstant( UInt32 slot, Mix::Graphics::IShaderConstant* pShaderConstant )
{
	MIX_ERROR( L"hCVF[_RX^g̐ݒɎs : VF[_[fR SetDomainShaderConstant ͎gpł܂" );
}

void Device::SetGeometryShaderConstant( UInt32 slot, Mix::Graphics::IShaderConstant* pShaderConstant )
{
	MIX_ERROR( L"WIgVF[_RX^g̐ݒɎs : VF[_[fR SetGeometryShaderConstant ͎gpł܂" );
}

void Device::SetVertexShaderConstant( UInt32 slot, Mix::Graphics::IShaderConstant* pShaderConstant )
{
	MIX_ERROR( L"o[ebNXVF[_RX^g̐ݒɎs : VF[_[fR SetVertexShaderConstant ͎gpł܂" );
}

void Device::SetPixelShaderConstant( UInt32 slot, Mix::Graphics::IShaderConstant* pShaderConstant )
{
	MIX_ERROR( L"sNZVF[_RX^g̐ݒɎs : VF[_[fR SetPixelShaderConstant ͎gpł܂" );
}

void Device::SetVertexBuffer( Mix::Graphics::IVertexBuffer* pVertexBuffer )
{
	m_DrawState.next.pVertexBuffer = pVertexBuffer;
}

void Device::SetIndexBuffer( Mix::Graphics::IIndexBuffer* pIndexBuffer )
{
	m_DrawState.next.pIndexBuffer = pIndexBuffer;
}

UInt32 Device::GetTextureStageMax( void ) const
{
	return m_Caps.MaxTextureBlendStages;
}

void Device::SetTexture( UInt32 stage, Mix::Graphics::TEXTURE_FILTER_TYPE filter, Mix::Graphics::TEXTURE_ADDRESS_TYPE address, Mix::Graphics::ITexture* pTexture )
{
	if( GetTextureStageMax() <= stage )
	{
		MIX_LOG_ERROR( L"eNX`̐ݒɎs : %s : staget( 0`7 )", Mix::STR_ILLEGALARG, stage );
		return;
	}

	UInt32 d3dMinFilter;
	UInt32 d3dMagFilter;
	UInt32 d3dMipFilter;
	UInt32 d3dAddress;
	UInt32 d3dAnisotropy;

	switch( filter )
	{
		case Mix::Graphics::TEXTURE_FILTER_POINT:
			d3dMinFilter = D3DTEXF_POINT;
			d3dMagFilter = D3DTEXF_POINT;
			d3dMipFilter = D3DTEXF_POINT;
			d3dAnisotropy = 1;
			break;
		case Mix::Graphics::TEXTURE_FILTER_LINEAR:
			d3dMinFilter = D3DTEXF_LINEAR;
			d3dMagFilter = D3DTEXF_LINEAR;
			d3dMipFilter = D3DTEXF_LINEAR;
			d3dAnisotropy = 1;
			break;
		case Mix::Graphics::TEXTURE_FILTER_ANISOTROPIC:
			d3dMinFilter = D3DTEXF_ANISOTROPIC;
			d3dMagFilter = D3DTEXF_LINEAR;
			d3dMipFilter = D3DTEXF_LINEAR;
			d3dAnisotropy = 8;
			break;

		default:
			d3dMinFilter = D3DTEXF_POINT;
			d3dAnisotropy = 1;
	}

	switch( address )
	{
	case Mix::Graphics::TEXTURE_ADDRESS_WRAP:
		d3dAddress = D3DTADDRESS_WRAP;
		break;
	case Mix::Graphics::TEXTURE_ADDRESS_CLAMP:
		d3dAddress = D3DTADDRESS_CLAMP;
		break;

	default:
		d3dAddress = D3DTADDRESS_WRAP;
	}

	SetSamplerState( stage, Device::SAMPLERSTATETYPE_MINFILTER, d3dMinFilter );
	SetSamplerState( stage, Device::SAMPLERSTATETYPE_MAGFILTER, d3dMagFilter );
	SetSamplerState( stage, Device::SAMPLERSTATETYPE_MIPFILTER, d3dMipFilter );
	SetSamplerState( stage, Device::SAMPLERSTATETYPE_ANISOTROPY, d3dAnisotropy );
	SetSamplerState( stage, Device::SAMPLERSTATETYPE_ADDRESSU, d3dAddress );
	SetSamplerState( stage, Device::SAMPLERSTATETYPE_ADDRESSV, d3dAddress );
	SetSamplerState( stage, Device::SAMPLERSTATETYPE_ADDRESSW, D3DTADDRESS_CLAMP );

	if( m_DrawState.textureTable[stage] != pTexture )
	{
		if( pTexture != NULL )
		{
			m_pD3DDevice->SetTexture( stage, static_cast<Mix::Graphics::DX9::Texture*>( pTexture )->GetInterface() );
		}
		else
		{
			m_pD3DDevice->SetTexture( stage, NULL );
		}

		m_DrawState.textureTable[stage] = pTexture;
	}
}

void Device::ResetTexture( UInt32 stage )
{
	if( GetTextureStageMax() <= stage )
	{
		MIX_LOG_ERROR( L"eNX`̃ZbgɎs : %s : stage[ 0 <= %d >= 7 ]", Mix::STR_ILLEGALARG, stage );
		return;
	}

	if( m_DrawState.textureTable[stage] != NULL )
	{
		m_pD3DDevice->SetTexture( stage, NULL );
		m_DrawState.textureTable[stage] = NULL;
	}
}

void Device::Draw( Mix::Graphics::PRIMITIVE_TYPE primitiveType, UInt32 startVertex, UInt32 numVertex )
{
	UInt32 numPrimitive;

	if( numVertex == 0 )
	{
		MIX_LOG_ERROR( L"%s : _ 0 ł", FAILED_DRAW );
		return;
	}

	if( ( m_DrawState.next.pVertexLayout == NULL ) ||
		( m_DrawState.next.pVertexShader == NULL ) ||
		( m_DrawState.next.pPixelShader == NULL ) ||
		( m_DrawState.next.pVertexBuffer == NULL ) )
	{
		MIX_LOG_WARNING( L"%s : KvȃfoCXIuWFNgݒ肳Ă܂\n{\n  IVertexLayout %s\n  IVertexShader %s \n  IPixelShader  %s\n  IVertexBuffer %s\n}",
			FAILED_DRAW,
			( m_DrawState.next.pVertexLayout != NULL )? L"" : L"~",
			( m_DrawState.next.pVertexShader != NULL )? L"" : L"~",
			( m_DrawState.next.pPixelShader != NULL )? L"" : L"~",
			( m_DrawState.next.pVertexBuffer != NULL )? L"" : L"~" );

		return;
	}

	numPrimitive = ComputePrimitiveCount( primitiveType, numVertex );
	if( numPrimitive == 0 )
	{
		MIX_LOG_ERROR( L"%s : v~eBu^Cvƒ_v܂", FAILED_DRAW );
		return;
	}

	FlushDrawState();

	m_pD3DDevice->DrawPrimitive( Device::D3DPrimitiveTypeTable[primitiveType], startVertex, numPrimitive );

	m_DrawPrimitiveCount += numPrimitive;
}

void Device::DrawIndexed( Mix::Graphics::PRIMITIVE_TYPE primitiveType, UInt32 startVertex, UInt32 numVertex, UInt32 startIndex, UInt32 numIndex )
{
	UInt32 numPrimitive;

	if( ( numVertex == 0 ) ||
		( numIndex == 0 ) )
	{
		MIX_LOG_ERROR( L"%s : _ACfbNX 0 ł", FAILED_DRAWINDEXED );
		return;
	}

	if( ( m_DrawState.next.pVertexLayout == NULL ) ||
		( m_DrawState.next.pVertexShader == NULL ) ||
		( m_DrawState.next.pPixelShader == NULL ) ||
		( m_DrawState.next.pVertexBuffer == NULL ) ||
		( m_DrawState.next.pIndexBuffer == NULL ) )
	{
		MIX_LOG_WARNING( L"%s : `ɕKvȃfoCXIuWFNgݒ肳Ă܂\n{\n  IVertexLayout %s\n  IVertexShader %s \n  IPixelShader  %s\n  IVertexBuffer %s \n  IIndexBuffer  %s\n}",
			FAILED_DRAWINDEXED,
			( m_DrawState.next.pVertexLayout != NULL )? L"" : L"~",
			( m_DrawState.next.pVertexShader != NULL )? L"" : L"~",
			( m_DrawState.next.pPixelShader != NULL )? L"" : L"~",
			( m_DrawState.next.pVertexBuffer != NULL )? L"" : L"~",
			( m_DrawState.next.pIndexBuffer != NULL )? L"" : L"~" );
	}

	numPrimitive = ComputePrimitiveCount( primitiveType, numIndex );
	if( numPrimitive == 0 )
	{
		MIX_LOG_ERROR( L"%s : v~eBu^Cvƒ_v܂", FAILED_DRAWINDEXED );
		return;
	}

	FlushDrawState();

	m_pD3DDevice->DrawIndexedPrimitive( Device::D3DPrimitiveTypeTable[primitiveType], 0, startVertex, numVertex, startIndex, numPrimitive );

	m_DrawPrimitiveCount += numPrimitive;
}

Boolean Device::SaveScreenshot( const wchar_t* pFileName )
{
	if( ( pFileName == NULL ) ||
		( ::wcslen( pFileName ) < 4 ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pFileName[%s]", FAILED_SAVESCREENSHOT, Mix::STR_ILLEGALARG, ( pFileName != NULL )? pFileName : L"NULL" );
		return False;
	}

	wchar_t fileExt[5];
	HRESULT ret;
	D3DXIMAGE_FILEFORMAT format;
	D3DXSaveSurfaceToFileWFunc D3DXSaveSurfaceToFileW;
	LPDIRECT3DSURFACE9 pBackBuffer = NULL;

	D3DXSaveSurfaceToFileW = static_cast<D3DXSaveSurfaceToFileWFunc>( m_D3DX9Module.GetFunction( L"D3DXSaveSurfaceToFileW" ) );

	::wcscpy_s( fileExt, 5, &( pFileName[( ::wcslen( pFileName ) - 4 )] ) );
	::_wcsupr_s( fileExt, 5 );

	if( ::wcscmp( fileExt, L".BMP" ) == 0 )
	{
		format = D3DXIFF_BMP;
	}
	else if( ::wcscmp( fileExt, L".JPG" ) == 0 )
	{
		format = D3DXIFF_JPG;
	}
	else if( ::wcscmp( fileExt, L".PNG" ) == 0 )
	{
		format = D3DXIFF_PNG;
	}
	else
	{
		MIX_LOG_ERROR( L"%s : T|[gĂȂtH[}bg( BMP JPG PNG ) : File[%s]", FAILED_SAVESCREENSHOT, pFileName );
		return False;
	}

	ret = m_pD3DDevice->GetBackBuffer( 0, 0, D3DBACKBUFFER_TYPE_MONO, &pBackBuffer );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : IDirect3DDevice9::GetBackBuffer %s : File[%s] Result[%s]", FAILED_SAVESCREENSHOT, Mix::STR_RETERROR, pFileName, GetD3DResultText( ret )  );
		return False;
	}

	ret = D3DXSaveSurfaceToFileW( pFileName, format, pBackBuffer, NULL, NULL );
	if( ret != D3D_OK )
	{
		MIX_LOG_ERROR( L"%s : D3DXSaveSurfaceToFileW %s : File[%s] Result[%s]", FAILED_SAVESCREENSHOT, Mix::STR_RETERROR, pFileName, GetD3DResultText( ret )  );
		MIX_RELEASE( pBackBuffer );
		return False;
	}

	MIX_RELEASE( pBackBuffer );

	MIX_LOG_INFO( L"XN[Vbgۑ : File[%s]", pFileName );

	return True;
}

Boolean Device::SaveTexture( Mix::Graphics::ITexture* pTexture, const wchar_t* pFileName )
{
	if( ( pTexture == NULL ) ||
		( pFileName == NULL ) ||
		( ::wcslen( pFileName ) == 0 ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pTexture[%s] pFileName[%s]",
			FAILED_SAVETEXTURE,
			Mix::STR_ILLEGALARG,
			( pTexture != NULL )? L"" : L"~",
			( pFileName != NULL )? pFileName : L"NULL" );

		return False;
	}

	wchar_t fileExt[5];
	HRESULT ret;
	D3DXIMAGE_FILEFORMAT format;
	Mix::Graphics::DX9::Texture* pTextureDX9 = static_cast<Mix::Graphics::DX9::Texture*>( pTexture );
	Mix::Graphics::TEXTURE_TYPE textureType = pTexture->GetType();

	::wcscpy_s( fileExt, 5, &( pFileName[( ::wcslen( pFileName ) - 4 )] ) );
	::_wcsupr_s( fileExt, 5 );

	if( ::wcscmp( fileExt, L".BMP" ) == 0 )
	{
		format = D3DXIFF_BMP;
	}
	else if( ::wcscmp( fileExt, L".JPG" ) == 0 )
	{
		format = D3DXIFF_JPG;
	}
	else if( ::wcscmp( fileExt, L".PNG" ) == 0 )
	{
		format = D3DXIFF_PNG;
	}
	else
	{
		MIX_LOG_ERROR( L"%s : T|[gĂȂtH[}bg( BMP JPG PNG ) : File[%s]", FAILED_SAVETEXTURE, pFileName );
		return False;
	}

	if( ( textureType == Mix::Graphics::TEXTURE_2D ) ||
		( textureType == Mix::Graphics::TEXTURE_TARGET ) )
	{
		D3DXSaveTextureToFileWFunc D3DXSaveTextureToFileW;

		D3DXSaveTextureToFileW = static_cast<D3DXSaveTextureToFileWFunc>( m_D3DX9Module.GetFunction( L"D3DXSaveTextureToFileW" ) );

		ret = D3DXSaveTextureToFileW( pFileName, format, pTextureDX9->GetInterface(), NULL );
		if( ret != D3D_OK )
		{
			MIX_LOG_ERROR( L"%s : D3DXSaveTextureToFileW %s : File[%s] Result[%s]", FAILED_SAVETEXTURE, Mix::STR_RETERROR, pFileName, GetD3DResultText( ret )  );
			return False;
		}
	}
	else if(	( textureType == Mix::Graphics::TEXTURE_CUBE ) ||
				( textureType == Mix::Graphics::TEXTURE_DEPTH ) )
	{
		D3DXSaveSurfaceToFileWFunc D3DXSaveSurfaceToFileW;

		D3DXSaveSurfaceToFileW = static_cast<D3DXSaveSurfaceToFileWFunc>( m_D3DX9Module.GetFunction( L"D3DXSaveSurfaceToFileW" ) );

		ret = D3DXSaveSurfaceToFileW( pFileName, format, pTextureDX9->GetSurface(), NULL, NULL );
		if( ret != D3D_OK )
		{
			MIX_LOG_ERROR( L"%s : D3DXSaveSurfaceToFileW %s : File[%s] Result[%s]", FAILED_SAVETEXTURE, Mix::STR_RETERROR, pFileName, GetD3DResultText( ret )  );
			return False;
		}
	}

	MIX_LOG_INFO( L"eNX`ۑ : File[%s]", pFileName );

	return True;
}

const Mix::Point& Device::GetScreenSize( void ) const
{
	return m_DeviceParam.current.screenSize;
}

UInt32 Device::GetDrawPrimitiveCount( void ) const
{
	return m_DrawPrimitiveCount;
}

Mix::Graphics::SHADER_MODEL Device::GetShaderModel( void ) const
{
	return Mix::Graphics::SHADER_MODEL_3;
}

Boolean Device::EnumDisplay( Mix::UserFile* pSysReport )
{
	Mix::String strTemp;
	std::list<D3DFORMAT> adapFormatList;
	UInt32 adapModeCount;
	UInt32 i;
	UInt32 j;
	UInt32 k;

	//A_v^AobNobt@tH[}bge[u
	const D3DFORMAT abFormatTable[] =
	{
		D3DFMT_X8R8G8B8,
		D3DFMT_X1R5G5B5,
		D3DFMT_R5G6B5,
	};
	const UInt32 MAX_ABFORMAT = ( sizeof( abFormatTable ) / sizeof( D3DFORMAT ) );

	//eNX`tH[}bge[u
	const D3DFORMAT texFormatTable[] =
	{
		D3DFMT_A8R8G8B8,
		D3DFMT_A8,
		D3DFMT_R16F,
		D3DFMT_R32F,
		D3DFMT_G16R16F,
		D3DFMT_G32R32F,
		D3DFMT_A16B16G16R16F,
		D3DFMT_A32B32G32R32F,
		D3DFMT_DXT1,
		D3DFMT_DXT2,
		D3DFMT_DXT3,
		D3DFMT_DXT4,
		D3DFMT_DXT5,
	};
	const UInt32 MAX_TEXFORMAT = ( sizeof( texFormatTable ) / sizeof( D3DFORMAT ) );

	//^[QbgeNX`tH[}bgXg
	const D3DFORMAT targetFormatTable[] =
	{
		D3DFMT_A8R8G8B8,
		D3DFMT_R16F,
		D3DFMT_R32F,
		D3DFMT_G16R16F,
		D3DFMT_G32R32F,
		D3DFMT_A16B16G16R16F,
		D3DFMT_A32B32G32R32F,
	};
	const UInt32 MAX_TARGETFORMAT = ( sizeof( targetFormatTable ) / sizeof( D3DFORMAT ) );

	//L[ueNX`tH[}bgXg
	const D3DFORMAT cubeTexFormatTable[] =
	{
		D3DFMT_A8R8G8B8,
		D3DFMT_R16F,
		D3DFMT_R32F,
		D3DFMT_G16R16F,
		D3DFMT_G32R32F,
		D3DFMT_A16B16G16R16F,
		D3DFMT_A32B32G32R32F,
	};
	const UInt32 MAX_CUBETEXFORMAT = ( sizeof( cubeTexFormatTable ) / sizeof( D3DFORMAT ) );

	//fvXXeVtH[}bge[u
	const D3DFORMAT dsFormatTable[] =
	{
		D3DFMT_D16,			//fvX(16rbg) XeV()
		D3DFMT_D32,			//fvX(32rbg) XeV()
		D3DFMT_D24S8,		//fvX(24rbg) XeV(8rbg)
	};
	const UInt32 MAX_DSFORMAT = ( sizeof( dsFormatTable ) / sizeof( D3DFORMAT ) );

	const Mix::Point& backBuffSize = m_pWindow->GetBaseSize();

	//fBXvC[h
	for( i = 0; i < MAX_ABFORMAT; i++ )
	{
		D3DFORMAT eAdapterFormat = abFormatTable[i];

		adapModeCount = m_pD3D->GetAdapterModeCount( 0, eAdapterFormat );
		for( j = 0; j < adapModeCount; j++ )
		{
			D3DDISPLAYMODE sDispMode;

			//fBXvC[h擾
			::ZeroMemory( &sDispMode, sizeof( sDispMode ) );
			if( m_pD3D->EnumAdapterModes( 0, eAdapterFormat, j, &sDispMode ) != D3D_OK )
			{
				continue;
			}

			//ʃTCỸ`FbN
			if( ( static_cast<UInt32>( backBuffSize.x ) > sDispMode.Width ) ||
				( static_cast<UInt32>( backBuffSize.y ) > sDispMode.Height ) )
			{
				continue;
			}

			//A_v^[tH[}bg
			std::list<D3DFORMAT>::iterator it;
			for( it = adapFormatList.begin(); it != adapFormatList.end(); ++it )
			{
				if( ( *it ) == sDispMode.Format )
				{
					break;
				}
			}
			if( it == adapFormatList.end() )
			{
				adapFormatList.push_back( sDispMode.Format );
			}

			//fBXvC[h
			m_DisplayModeList.push_back( sDispMode );
		}
	}

	//fBXvC[hXg\[g
	m_DisplayModeList.sort( SORT_DISPLAYMODE() );

	//fBXvC[hXg|[g
	if( pSysReport->Open() == True )
	{
		wchar_t formatStr[12];

		pSysReport->WriteLine( L"[fBXvC[hꗗ]" );
		pSysReport->WriteLine( L"{" );

		for( DisplayModeList::iterator it = m_DisplayModeList.begin(); it != m_DisplayModeList.end(); ++it )
		{
			switch( ( *it ).Format )
			{
			case D3DFMT_A2R10G10B10:
				::wcscpy_s( formatStr, L"A2R10G10B10" );
				break;
			case D3DFMT_X8R8G8B8:
				::wcscpy_s( formatStr, L"X8R8G8B8" );
				break;
			case D3DFMT_X1R5G5B5:
				::wcscpy_s( formatStr, L"X1R5G5B5" );
				break;
			case D3DFMT_R5G6B5:
				::wcscpy_s( formatStr, L"R5G6B5" );
				break;
			default:
				::wcscpy_s( formatStr, L"UNKNOWN" );
			}

			pSysReport->WriteLine( L"     Width[%d] Height[%d] RefreshRate[%d] Format[%s]", ( *it ).Width, ( *it ).Height, ( *it ).RefreshRate, formatStr );
		}

		pSysReport->WriteLine( L"}" );
		pSysReport->WriteLine( L"" );
		pSysReport->Close();
	}

	//foCXR{쐬
	for( std::list<D3DFORMAT>::iterator it = adapFormatList.begin(); it != adapFormatList.end(); ++it )
	{
		D3DFORMAT adapterFormat = ( *it );

		for( j = 0; j < MAX_ABFORMAT; j++ )
		{
			D3DFORMAT backBufferFormat = abFormatTable[j];
			DEVICECOMBO devCombo;

			//A_v^ƃobNobt@̃tH[}bgvȂ̂̓XLbv
			if( adapterFormat != backBufferFormat )
			{
				continue;
			}

			//
			devCombo.flags = 0;
			devCombo.adapterFormat = adapterFormat;
			devCombo.backBufferFormat = backBufferFormat;
			devCombo.formatTable[Mix::Graphics::FMT_UNKNOWN].format			= D3DFMT_UNKNOWN;
			devCombo.formatTable[Mix::Graphics::FMT_UNKNOWN].flags			= 0;
			devCombo.formatTable[Mix::Graphics::FMT_D16].format				= D3DFMT_D16;
			devCombo.formatTable[Mix::Graphics::FMT_D16].flags				= 0;
			devCombo.formatTable[Mix::Graphics::FMT_D32].format				= D3DFMT_D32;
			devCombo.formatTable[Mix::Graphics::FMT_D32].flags				= 0;
			devCombo.formatTable[Mix::Graphics::FMT_D24S8].format			= D3DFMT_D24S8;
			devCombo.formatTable[Mix::Graphics::FMT_D24S8].flags			= 0;
			devCombo.formatTable[Mix::Graphics::FMT_R8G8B8A8].format		= D3DFMT_A8R8G8B8;
			devCombo.formatTable[Mix::Graphics::FMT_R8G8B8A8].flags			= 0;
			devCombo.formatTable[Mix::Graphics::FMT_A8].format				= D3DFMT_A8;
			devCombo.formatTable[Mix::Graphics::FMT_A8].flags				= 0;
			devCombo.formatTable[Mix::Graphics::FMT_R16F].format			= D3DFMT_R16F;
			devCombo.formatTable[Mix::Graphics::FMT_R16F].flags				= 0;
			devCombo.formatTable[Mix::Graphics::FMT_R32F].format			= D3DFMT_R32F;
			devCombo.formatTable[Mix::Graphics::FMT_R32F].flags				= 0;
			devCombo.formatTable[Mix::Graphics::FMT_R16G16F].format			= D3DFMT_G16R16F;
			devCombo.formatTable[Mix::Graphics::FMT_R16G16F].flags			= 0;
			devCombo.formatTable[Mix::Graphics::FMT_R32G32F].format			= D3DFMT_G32R32F;
			devCombo.formatTable[Mix::Graphics::FMT_R32G32F].flags			= 0;
			devCombo.formatTable[Mix::Graphics::FMT_R16G16B16A16F].format	= D3DFMT_A16B16G16R16F;
			devCombo.formatTable[Mix::Graphics::FMT_R16G16B16A16F].flags	= 0;
			devCombo.formatTable[Mix::Graphics::FMT_R32G32B32A32F].format	= D3DFMT_A32B32G32R32F;
			devCombo.formatTable[Mix::Graphics::FMT_R32G32B32A32F].flags	= 0;
			devCombo.formatTable[Mix::Graphics::FMT_DXT1].format			= D3DFMT_DXT1;
			devCombo.formatTable[Mix::Graphics::FMT_DXT1].flags				= 0;
			devCombo.formatTable[Mix::Graphics::FMT_DXT2].format			= D3DFMT_DXT2;
			devCombo.formatTable[Mix::Graphics::FMT_DXT2].flags				= 0;
			devCombo.formatTable[Mix::Graphics::FMT_DXT3].format			= D3DFMT_DXT3;
			devCombo.formatTable[Mix::Graphics::FMT_DXT3].flags				= 0;
			devCombo.formatTable[Mix::Graphics::FMT_DXT4].format			= D3DFMT_DXT4;
			devCombo.formatTable[Mix::Graphics::FMT_DXT4].flags				= 0;
			devCombo.formatTable[Mix::Graphics::FMT_DXT5].format			= D3DFMT_DXT5;
			devCombo.formatTable[Mix::Graphics::FMT_DXT5].flags				= 0;

			//EBhE[h̃`FbN
			if( ( m_pD3D->CheckDeviceType(	0,
											D3DDEVTYPE_HAL,
											adapterFormat,
											backBufferFormat,
											True ) == D3D_OK ) )
			{
				devCombo.flags |= FLAG_MODE_WINDOW;
			}

			//tXN[[h̃`FbN
			if( ( m_pD3D->CheckDeviceType(	0,
											D3DDEVTYPE_HAL,
											adapterFormat,
											backBufferFormat,
											False ) == D3D_OK ) )
			{
				devCombo.flags |= FLAG_MODE_FULLSCREEN;
			}

			//fvXXeVtH[}bg̗
			for( k = 0; k < MAX_DSFORMAT; k++ )
			{
				D3DFORMAT dsFormat = dsFormatTable[k];

				if( m_pD3D->CheckDeviceFormat(	0,
												D3DDEVTYPE_HAL,
												adapterFormat,
												D3DUSAGE_DEPTHSTENCIL,
												D3DRTYPE_SURFACE,
												dsFormat ) != D3D_OK )
				{
					continue;
				}

				if( m_pD3D->CheckDepthStencilMatch(	0,
													D3DDEVTYPE_HAL,
													adapterFormat,
													backBufferFormat,
													dsFormat ) != D3D_OK )
				{
					continue;
				}

				switch( dsFormat )
				{
				case D3DFMT_D16:
					devCombo.formatTable[Mix::Graphics::FMT_D16].flags |= FLAG_RESOURCE_DEPTHSTENCIL;
					break;
				case D3DFMT_D32:
					devCombo.formatTable[Mix::Graphics::FMT_D32].flags |= FLAG_RESOURCE_DEPTHSTENCIL;
					break;
				case D3DFMT_D24S8:
					devCombo.formatTable[Mix::Graphics::FMT_D24S8].flags |= FLAG_RESOURCE_DEPTHSTENCIL;
					break;
				}
			}

			//eNX`tH[}bg̗
			for( k = 0; k < MAX_TEXFORMAT; k++ )
			{
				D3DFORMAT texFormat = texFormatTable[k];

				if( m_pD3D->CheckDeviceFormat(	0,
												D3DDEVTYPE_HAL,
												adapterFormat,
												0,
												D3DRTYPE_TEXTURE,
												texFormat ) != D3D_OK )
				{
					continue;
				}

				switch( texFormat )
				{
				case D3DFMT_A8R8G8B8:
					devCombo.formatTable[Mix::Graphics::FMT_R8G8B8A8].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				case D3DFMT_A8:
					devCombo.formatTable[Mix::Graphics::FMT_A8].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				case D3DFMT_R16F:
					devCombo.formatTable[Mix::Graphics::FMT_R16F].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				case D3DFMT_R32F:
					devCombo.formatTable[Mix::Graphics::FMT_R32F].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				case D3DFMT_G16R16F:
					devCombo.formatTable[Mix::Graphics::FMT_R16G16F].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				case D3DFMT_G32R32F:
					devCombo.formatTable[Mix::Graphics::FMT_R32G32F].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				case D3DFMT_A16B16G16R16F:
					devCombo.formatTable[Mix::Graphics::FMT_R16G16B16A16F].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				case D3DFMT_A32B32G32R32F:
					devCombo.formatTable[Mix::Graphics::FMT_R32G32B32A32F].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				case D3DFMT_DXT1:
					devCombo.formatTable[Mix::Graphics::FMT_DXT1].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				case D3DFMT_DXT2:
					devCombo.formatTable[Mix::Graphics::FMT_DXT2].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				case D3DFMT_DXT3:
					devCombo.formatTable[Mix::Graphics::FMT_DXT3].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				case D3DFMT_DXT4:
					devCombo.formatTable[Mix::Graphics::FMT_DXT4].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				case D3DFMT_DXT5:
					devCombo.formatTable[Mix::Graphics::FMT_DXT5].flags |= FLAG_RESOURCE_TEXTURE;
					break;
				}
			}

			//^[QbgtH[}bg̗
			for( k = 0; k < MAX_TARGETFORMAT; k++ )
			{
				D3DFORMAT format = targetFormatTable[k];

				if( m_pD3D->CheckDeviceFormat(	0,
												D3DDEVTYPE_HAL,
												adapterFormat,
												D3DUSAGE_RENDERTARGET,
												D3DRTYPE_TEXTURE,
												format ) != D3D_OK )
				{
					continue;
				}

				switch( format )
				{
				case D3DFMT_A8R8G8B8:
					devCombo.formatTable[Mix::Graphics::FMT_R8G8B8A8].flags |= FLAG_RESOURCE_TARGET;
					break;
				case D3DFMT_R16F:
					devCombo.formatTable[Mix::Graphics::FMT_R16F].flags |= FLAG_RESOURCE_TARGET;
					break;
				case D3DFMT_R32F:
					devCombo.formatTable[Mix::Graphics::FMT_R32F].flags |= FLAG_RESOURCE_TARGET;
					break;
				case D3DFMT_G16R16F:
					devCombo.formatTable[Mix::Graphics::FMT_R16G16F].flags |= FLAG_RESOURCE_TARGET;
					break;
				case D3DFMT_G32R32F:
					devCombo.formatTable[Mix::Graphics::FMT_R32G32F].flags |= FLAG_RESOURCE_TARGET;
					break;
				case D3DFMT_A16B16G16R16F:
					devCombo.formatTable[Mix::Graphics::FMT_R16G16B16A16F].flags |= FLAG_RESOURCE_TARGET;
					break;
				case D3DFMT_A32B32G32R32F:
					devCombo.formatTable[Mix::Graphics::FMT_R32G32B32A32F].flags |= FLAG_RESOURCE_TARGET;
					break;
				}
			}

			for( k = 0; k < MAX_CUBETEXFORMAT; k++ )
			{
				D3DFORMAT format = cubeTexFormatTable[k];

				if( m_pD3D->CheckDeviceFormat(	0,
												D3DDEVTYPE_HAL,
												adapterFormat,
												D3DUSAGE_RENDERTARGET,
												D3DRTYPE_CUBETEXTURE,
												format ) != D3D_OK )
				{
					continue;
				}

				switch( format )
				{
				case D3DFMT_A8R8G8B8:
					devCombo.formatTable[Mix::Graphics::FMT_R8G8B8A8].flags |= FLAG_RESOURCE_CUBE;
					break;
				case D3DFMT_R16F:
					devCombo.formatTable[Mix::Graphics::FMT_R16F].flags |= FLAG_RESOURCE_CUBE;
					break;
				case D3DFMT_R32F:
					devCombo.formatTable[Mix::Graphics::FMT_R32F].flags |= FLAG_RESOURCE_CUBE;
					break;
				case D3DFMT_G16R16F:
					devCombo.formatTable[Mix::Graphics::FMT_R16G16F].flags |= FLAG_RESOURCE_CUBE;
					break;
				case D3DFMT_G32R32F:
					devCombo.formatTable[Mix::Graphics::FMT_R32G32F].flags |= FLAG_RESOURCE_CUBE;
					break;
				case D3DFMT_A16B16G16R16F:
					devCombo.formatTable[Mix::Graphics::FMT_R16G16B16A16F].flags |= FLAG_RESOURCE_CUBE;
					break;
				case D3DFMT_A32B32G32R32F:
					devCombo.formatTable[Mix::Graphics::FMT_R32G32B32A32F].flags |= FLAG_RESOURCE_CUBE;
					break;
				}
			}

			m_DeviceComboList.push_back( devCombo );
		}
	}

	return True;
}

Boolean Device::CreatePresentation( Boolean bWindowed )
{
	const Mix::Point& backBuffSize = m_pWindow->GetBaseSize();

	DEVICECOMBO* pBestDevCombo = NULL;
	UInt32 width = 0;
	UInt32 height = 0;
	UInt32 refreshRate = 0;
	UInt32 interval = 0;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// œKȃfoCXR{擾
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( bWindowed == True )
	{
		for( DeviceComboList::iterator it = m_DeviceComboList.begin(); ( it != m_DeviceComboList.end() ) && ( pBestDevCombo == NULL ); ++it )
		{
			DEVICECOMBO* pDevCombo = &( *it );

			if( MIX_TESTBIT( pDevCombo->flags, FLAG_MODE_WINDOW ) == FLAG_MODE_WINDOW )
			{
				pBestDevCombo = pDevCombo;
			}
		}

		if( pBestDevCombo == NULL )
		{
			return False;
		}

		width = backBuffSize.x;
		height = backBuffSize.y;
		refreshRate = 0;
		interval = D3DPRESENT_INTERVAL_IMMEDIATE;
	}
	else
	{
		D3DDISPLAYMODE desktopMode;
		D3DDISPLAYMODE* pBestDispMode = NULL;
		Boolean bFindBestDevCombo = False;

		//fXNgbṽ[h擾
		if( m_pD3D->GetAdapterDisplayMode( 0, &desktopMode ) != D3D_OK )
		{
			return False;
		}

		//݂̃fBXvC[hƃ}b`foCXR{T
		for( DeviceComboList::iterator it = m_DeviceComboList.begin(); ( it != m_DeviceComboList.end() ) && ( bFindBestDevCombo == False ); ++it )
		{
			DEVICECOMBO* pDevCombo = &( *it );

			if( MIX_TESTBIT( pDevCombo->flags, FLAG_MODE_FULLSCREEN ) != FLAG_MODE_FULLSCREEN )
			{
				continue;
			}

			Boolean bMatchesBackBuffer = ( pDevCombo->backBufferFormat == pDevCombo->adapterFormat );
			Boolean bMatchesDesktop = ( pDevCombo->adapterFormat == desktopMode.Format );

			if( ( pBestDevCombo == NULL ) ||
				( ( pBestDevCombo->adapterFormat != desktopMode.Format ) && ( bMatchesDesktop == True ) ||
				( ( bMatchesBackBuffer == True ) && ( bMatchesDesktop == True ) ) ) )
			{
				pBestDevCombo = pDevCombo;
				if( ( bMatchesBackBuffer == True ) &&
					( bMatchesDesktop == True ) )
				{
					bFindBestDevCombo = True;
				}
			}
		}

		//œKȃfBXvC[hT
		for( DisplayModeList::iterator it = m_DisplayModeList.begin(); ( it != m_DisplayModeList.end() ) && ( pBestDispMode == NULL ); ++it )
		{
			D3DDISPLAYMODE* pDispMode = &( *it );

			//tH[}bg`FbN
			if( pDispMode->Format != pBestDevCombo->adapterFormat )
			{
				continue;
			}

			//tbV[g̃`FbN
			if( desktopMode.RefreshRate != pDispMode->RefreshRate )
			{
				continue;
			}

			//TCỸ`FbN
			if( ( static_cast<UInt32>( backBuffSize.x ) > pDispMode->Width ) ||
				( static_cast<UInt32>( backBuffSize.y ) > pDispMode->Height ) )
			{
				continue;
			}

			pBestDispMode = pDispMode;
		}

		if( pBestDispMode == NULL )
		{
			return False;
		}

		width = pBestDispMode->Width;
		height = pBestDispMode->Height;
		refreshRate = pBestDispMode->RefreshRate;
		interval = ( m_bWaitVSync == True )? D3DPRESENT_INTERVAL_DEFAULT : D3DPRESENT_INTERVAL_IMMEDIATE;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// v[e[Vp[^ݒ
	////////////////////////////////////////////////////////////////////////////////////////////////////

	m_DeviceParam.next.screenSize.x = width;
	m_DeviceParam.next.screenSize.y = height;
	m_DeviceParam.next.formatTable = &( pBestDevCombo->formatTable[0] );
	m_DeviceParam.next.params.Windowed = bWindowed;
	m_DeviceParam.next.params.BackBufferCount = 1;
	m_DeviceParam.next.params.SwapEffect = D3DSWAPEFFECT_DISCARD;
	m_DeviceParam.next.params.hDeviceWindow = m_pWindow->GetHandle();
	m_DeviceParam.next.params.EnableAutoDepthStencil = FALSE;
	m_DeviceParam.next.params.AutoDepthStencilFormat = D3DFMT_UNKNOWN;
	m_DeviceParam.next.params.Flags = 0;
	m_DeviceParam.next.params.MultiSampleType = D3DMULTISAMPLE_NONE;
	m_DeviceParam.next.params.MultiSampleQuality = 0;
	m_DeviceParam.next.params.BackBufferWidth = width;
	m_DeviceParam.next.params.BackBufferHeight = height;
	m_DeviceParam.next.params.BackBufferFormat = pBestDevCombo->backBufferFormat;
	m_DeviceParam.next.params.FullScreen_RefreshRateInHz = refreshRate;
	m_DeviceParam.next.params.PresentationInterval = interval;

	return True;
}

void Device::ChangePresentation( Boolean bWindowed )
{
	HRESULT hRet;

	if( CreatePresentation( bWindowed ) == True )
	{
		ResetDrawState();
		ResetSamplerState();
		ResetRenderState();

		//obNobt@
		MIX_RELEASE( m_pD3DBackBuffer );

		//foCXIuWFNg𖳌ɂ
		InvalidateDeviceObject();

		//EBhEtXN[̏ꍇ́AZbgOɃEBhEX^CύX
		if( m_DeviceParam.next.params.Windowed == False )
		{
			m_pWindow->SetStyle( m_DeviceParam.next.params.Windowed );
		}

		//Zbg
		hRet = m_pD3DDevice->Reset( &( m_DeviceParam.next.params ) );
		if( hRet == D3D_OK )
		{
			//tXN[EBhȄꍇ́AZbgɃEBhEX^CύX
			if( m_DeviceParam.next.params.Windowed == True )
			{
				m_pWindow->SetStyle( m_DeviceParam.next.params.Windowed );
			}

			m_DeviceParam.current = m_DeviceParam.next;

			TracePresentation( L"OtBbNfoCX̐ݒύX" );
		}
		else
		{
			//EBhEtXN[̏ꍇ́AZbgOɃEBhEX^CύX
			if( m_DeviceParam.current.params.Windowed == False )
			{
				m_pWindow->SetStyle( m_DeviceParam.current.params.Windowed );
			}

			//ăZbg
			hRet = m_pD3DDevice->Reset( &( m_DeviceParam.current.params ) );
			if( hRet == D3D_OK )
			{
				//tXN[EBhȄꍇ́AZbgɃEBhEX^CύX
				if( m_DeviceParam.current.params.Windowed == True )
				{
					m_pWindow->SetStyle( m_DeviceParam.current.params.Windowed );
				}
			}
		}

		if( hRet == D3D_OK )
		{
			//obNobt@Ď擾
			m_pD3DDevice->GetBackBuffer( 0, 0, D3DBACKBUFFER_TYPE_MONO, &m_pD3DBackBuffer );

			//foCXIuWFNgLɂ
			ValidateDeviceObject();

			//`^[QbgtbV
			FlushTargetState();

			m_Status = Device::STATUS_OK;
		}
		else
		{
			m_pWindow->SendMessage( WM_MIX_INTERNALERROR, Mix::ERR_GRAPHICS_CHANGEMODE, 0 );
		}
	}
}

void Device::ResetPresentation( void )
{
	HRESULT hRet;

	hRet = m_pD3DDevice->TestCooperativeLevel();
	if( hRet != D3D_OK )
	{
		if( hRet == D3DERR_DEVICENOTRESET )
		{
			ResetDrawState();
			ResetSamplerState();
			ResetRenderState();

			MIX_RELEASE( m_pD3DBackBuffer );

			InvalidateDeviceObject();

			hRet = m_pD3DDevice->Reset( &( m_DeviceParam.current.params ) );

			switch( hRet )
			{
			case D3D_OK:
				m_pD3DDevice->GetBackBuffer( 0, 0, D3DBACKBUFFER_TYPE_MONO, &m_pD3DBackBuffer );
				ValidateDeviceObject();
				FlushTargetState();
				m_Status = Device::STATUS_OK;
				break;

			case D3DERR_DEVICELOST:
				::SleepEx( 200, TRUE );
				break;
			}
		}
		else
		{
			::SleepEx( 200, TRUE );
		}
	}
	else
	{
		m_Status = Device::STATUS_OK;
	}
}

void Device::TracePresentation( const wchar_t* pTitle )
{
	MIX_ASSERT( pTitle != NULL );

	int refreshRate;

	if( m_DeviceParam.current.params.Windowed == True )
	{
		HDC hDC = ::GetDC( m_pWindow->GetHandle() );
		if( hDC != NULL )
		{
			refreshRate = ::GetDeviceCaps( hDC, VREFRESH );
			::ReleaseDC( m_pWindow->GetHandle(), hDC );
		}
		else
		{
			refreshRate = 0;
		}
	}
	else
	{
		refreshRate = static_cast<UINT>( m_DeviceParam.current.params.FullScreen_RefreshRateInHz );
	}

	MIX_LOG_INFO( L"%s : Mode[%s] Size[%dx%d] BackBufferFormat[%s] RefreshRate[%dHz]",
		pTitle,
		( m_DeviceParam.current.params.Windowed == True )? L"Window" : L"Fullscreen",
		m_DeviceParam.current.params.BackBufferWidth,
		m_DeviceParam.current.params.BackBufferHeight,
		GetD3DFormatText( m_DeviceParam.current.params.BackBufferFormat ),
		refreshRate );
}

}}}
