#include "Mix/Class/Graphics/DX11/Texture2D.h"
#include "Mix/Class/Graphics/DX11/Device.h"

namespace Mix{ namespace Graphics{ namespace DX11{

Texture2D* Texture2D::CreateInstance(	Mix::Graphics::DX11::Device* pDevice,
										Boolean bDynamic, UInt32 width, UInt32 height, Mix::Graphics::FORMAT format,
										ID3D11Texture2D* pTexture, ID3D11ShaderResourceView* pSRView,
										Boolean bRegisterMap, const wchar_t* pName )
{
	return new Texture2D( pDevice, bDynamic, width, height, format, pTexture, pSRView, bRegisterMap, pName );
}

Texture2D::Texture2D(	Mix::Graphics::DX11::Device* pDevice,
						Boolean bDynamic, UInt32 width, UInt32 height, Mix::Graphics::FORMAT format,
						ID3D11Texture2D* pTexture, ID3D11ShaderResourceView* pSRView,
						Boolean bRegisterMap, const wchar_t* pName ) :
DeviceObject( pDevice, Mix::Graphics::Common::DOT_TEXTURE_2D, bRegisterMap, pName ),
m_pDevice( pDevice ),
m_bDynamic( bDynamic ),
m_Width( width ),
m_Height( height ),
m_Format( format ),
m_pTexture( pTexture ),
m_pSRView( pSRView ),
m_bLock( False )
{
	MIX_ADD_REF( m_pDevice );
}

Texture2D::~Texture2D( void )
{
	MIX_RELEASE( m_pSRView );
	MIX_RELEASE( m_pTexture );
	MIX_RELEASE( m_pDevice );
}

void Texture2D::Attach( ID3D11Texture2D* pD3DTexture2D, ID3D11ShaderResourceView* pShaderResourceView )
{
	D3D11_TEXTURE2D_DESC desc;

	m_pTexture = pD3DTexture2D;
	m_pSRView = pShaderResourceView;

	m_pTexture->GetDesc( &desc );

	m_Width = desc.Width;
	m_Height = desc.Height;
}

ID3D11Resource* Texture2D::GetInterface( void )
{
	return m_pTexture;
}

ID3D11ShaderResourceView* Texture2D::GetSRView( void )
{
	return m_pSRView;
}

Boolean Texture2D::Lock( Mix::Graphics::TEXTURE_LOCK_INFO& lockInfo )
{
	if( ( m_bLock == True ) ||
		( m_bDynamic == False ) ||
		( m_pTexture == NULL ) )
	{
		return False;
	}

	D3D11_MAPPED_SUBRESOURCE mapped;

	if( m_pDevice->Map( m_pTexture, True, mapped ) != S_OK )
	{
		return False;
	}

	lockInfo.bits = mapped.pData;
	lockInfo.pitch = mapped.RowPitch;

	m_bLock = True;

	return True;
}

void Texture2D::Unlock( void )
{
	if( ( m_bLock == False ) ||
		( m_pTexture == NULL ) )
	{
		return;
	}

	m_pDevice->Unmap( m_pTexture );
	m_bLock = False;
}

UInt32 Texture2D::GetWidth( void ) const
{
	return m_Width;
}

UInt32 Texture2D::GetHeight( void ) const
{
	return m_Height;
}

Mix::Graphics::TEXTURE_TYPE Texture2D::GetType( void ) const
{
	return Mix::Graphics::TEXTURE_2D;
}

Mix::Graphics::FORMAT Texture2D::GetFormat( void ) const
{
	return m_Format;
}

Boolean Texture2D::IsAvailable( void ) const
{
	return ( ( m_pTexture != NULL ) && ( m_pSRView != NULL ) );
}

}}}
