#pragma once

#include <vector>
#include <list>
#include <map>

#include "Mix/Module.h"
#include "Mix/CriticalSection.h"
#include "Mix/Class/Graphics/Common/Device.h"

namespace Mix{
	class UserFile;
}

namespace Mix{ namespace Graphics{ namespace DX11{

	class DeviceObject;
	class VertexLayout;
	class VertexShader;
	class PixelShader;
	class VertexBuffer;
	class IndexBuffer;

	class Device : public Mix::Graphics::Common::Device
	{
	private:
		//D3DUgeLXgz
		static const wchar_t* D3DResultTextArray[12];
		//D3Do[ebNXCAEgtH[}bge[u
		static const DXGI_FORMAT D3DVLFormatTable[Mix::Graphics::VERTEX_ELEMENT_FORMAT_MAX];
		//D3Do[ebNXCAEgZ}eBbNe[u
		static const char* D3DVLSemanticTable[Mix::Graphics::VERTEX_ELEMENT_SEMANTIC_MAX];
		//D3Dv~eBug|W
		static const D3D11_PRIMITIVE_TOPOLOGY D3DPrimitiveTopologyTable[Mix::Graphics::PT_MAX];
		//D3DNGfXNvVe[u
		static const D3D11_QUERY_DESC D3DQueryDescTable[Mix::Graphics::QT_MAX];
		//D3DNGe[u
		static const wchar_t* D3DQueryTextTable[Mix::Graphics::QT_MAX];
		//D3DCfbNXtH[}bgeLXge[u
		static const wchar_t* D3DIndexFormatTextTable[Mix::Graphics::IT_MAX];
		//D3DCfbNXtH[}bge[u
		static const DXGI_FORMAT D3DIndexFormatTable[Mix::Graphics::IT_MAX];
		//CfbNXXgChe[u
		static const UInt32 D3DIndexStrideTable[Mix::Graphics::IT_MAX];

		//Cxg^Cv
		enum EVENTTYPE
		{
			EVENTTYPE_NONE				= 0x00,	//Cxg͖
			EVENTTYPE_CHANGEMODE		= 0x01,	//[hύXCxg
			EVENTTYPE_RESTOREWINDOW		= 0x02,	//EBhEXgACxg
			EVENTTYPE_INACTIVATEWINDOW	= 0x03,	//EBhEANeBuCxg
		};

		//Xe[^X
		enum STATUS
		{
			STATUS_OK				= 0x00,	//
			STATUS_CHANGEMODE		= 0x01,	//[hύX
		};

		//\[XtO
		enum RESOURCEFLAG
		{
			RS_TEXTURE2D		= 0x01,
			RS_CUBE				= 0x02,
			RS_TARGET			= 0x04,
			RS_DEPTH_STENCIL	= 0x08,
		};

		//tO
		enum FLAGS
		{
			GEOMETRY_SHADER			= 0x00000001,
			COMPUTE_SHADER			= 0x00000002,
			HAL_DOMAIN_SHADER		= 0x00000004,
			OUTPUTSTREAM			= 0x00000008,
			TEXTURE_ARRAY			= 0x00000010,
			CUBETEXTURE_ARRAY		= 0x00000020,
			TEXTURE_BC4_BC5			= 0x00000040,
			TEXTURE_BC6H_BC7		= 0x00000080,
			ALPHA_TO_COVERAGE		= 0x00000100,
			EXTEND_FORMAT			= 0x00000200,
			HIGHCOLOR_FORMAT_10XR	= 0x00000400,
			OCCLUSIONQUERY			= 0x00000800,
			SEPARATEALPHABLEND		= 0x00001000,
			MIRRORONE				= 0x00002000,
			OVERLAP_VERTEXELEMENT	= 0x00004000,
			INDEPENDENTWRITEMASKS	= 0x00008000,
			INSTANCING				= 0x00010000,
		};

		//\
		struct CAPS
		{
			wchar_t shaderModel[32];
			UInt32 flags;
			UInt32 maxTextureSize;
			UInt32 maxCubeTextureSize;
			UInt32 maxVolumeTextureRange;
			UInt32 maxTextureRepeat;
			UInt32 maxAnisotropy;
			UInt32 maxPrimitive;
			UInt32 maxTarget;
		};

		//\[XtH[}bg\
		struct RESOURCEFORMAT
		{
			DXGI_FORMAT format;
			UInt8 flags;
		};

		//^[QbgXe[g
		struct TARGETSTATE
		{
			Mix::Rectangle rect;
			Mix::Graphics::ITexture* pT0;
			Mix::Graphics::ITexture* pT1;
			Mix::Graphics::ITexture* pT2;
			Mix::Graphics::ITexture* pT3;
			Mix::Graphics::ITexture* pDS;
		};

		//`Xe[g
		struct DRAWSTATE
		{
			Mix::Graphics::IShaderConstant* hscTable[D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT];
			Mix::Graphics::IShaderConstant* dscTable[D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT];
			Mix::Graphics::IShaderConstant* gscTable[D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT];
			Mix::Graphics::IShaderConstant* vscTable[D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT];
			Mix::Graphics::IShaderConstant* pscTable[D3D11_COMMONSHADER_CONSTANT_BUFFER_API_SLOT_COUNT];

			ID3D11SamplerState* samplerTable[Mix::Graphics::TEXTURESTAGE_MAX];
			Mix::Graphics::ITexture* textureTable[Mix::Graphics::TEXTURESTAGE_MAX];

			struct
			{
				Mix::Graphics::IVertexLayout* pVertexLayout;

				Mix::Graphics::IHullShader* pHullShader;
				Mix::Graphics::IDomainShader* pDomainShader;
				Mix::Graphics::IGeometryShader* pGeometryShader;
				Mix::Graphics::IVertexShader* pVertexShader;
				Mix::Graphics::IPixelShader* pPixelShader;

				Mix::Graphics::IVertexBuffer* pVertexBuffer;
				Mix::Graphics::IIndexBuffer* pIndexBuffer;

			}current, next;
		};

		//fBXvC[h^
		typedef std::vector<DXGI_MODE_DESC> DisplayModeList;

		//D3D11֐` : D3D11CreateDevice
		typedef HRESULT ( WINAPI *D3D11CreateDeviceFunc )( __in_opt IDXGIAdapter* pAdapter,
			D3D_DRIVER_TYPE DriverType,
			HMODULE Software,
			UINT Flags,
			__in_ecount_opt( FeatureLevels ) CONST D3D_FEATURE_LEVEL* pFeatureLevels,
			UINT FeatureLevels,
			UINT SDKVersion,
			__out_opt ID3D11Device** ppDevice,
			__out_opt D3D_FEATURE_LEVEL* pFeatureLevel,
			__out_opt ID3D11DeviceContext** ppImmediateContext );

		//D3DX11֐` : D3DX11CompileFromMemory
		typedef HRESULT ( WINAPI *D3DX11CompileFromMemoryFunc )( LPCSTR pSrcData,
			SIZE_T SrcDataLen,
			LPCSTR pFileName,
			CONST D3D10_SHADER_MACRO* pDefines,
			LPD3D10INCLUDE pInclude,
			LPCSTR pFunctionName,
			LPCSTR pProfile,
			UINT Flags1,
			UINT Flags2,
			ID3DX11ThreadPump* pPump,
			ID3D10Blob** ppShader,
			ID3D10Blob** ppErrorMsgs,
			HRESULT* pHResult );

		//D3DX11֐` : D3DX11GetImageInfoFromMemory
		typedef HRESULT ( WINAPI *D3DX11GetImageInfoFromMemoryFunc )( LPCVOID pSrcData,
			SIZE_T SrcDataSize,
			ID3DX11ThreadPump* pPump,
			D3DX11_IMAGE_INFO* pSrcInfo,
			HRESULT* pHResult );

		//D3DX11֐` : D3DX11CreateTextureFromMemory
		typedef HRESULT ( WINAPI *D3DX11CreateTextureFromMemoryFunc )( ID3D11Device* pDevice,
			LPCVOID pSrcData,
			SIZE_T SrcDataSize,
			D3DX11_IMAGE_LOAD_INFO* pLoadInfo,
			ID3DX11ThreadPump* pPump,
			ID3D11Resource** ppTexture,
			HRESULT* pHResult );

		//D3DX11֐` : D3DX11SaveTextureToFileW
		typedef HRESULT ( WINAPI *D3DX11SaveTextureToFileWFunc )( ID3D11DeviceContext* pContext,
			ID3D11Resource* pSrcTexture,
			D3DX11_IMAGE_FILE_FORMAT DestFormat,
			LPCWSTR pDestFile );

		//DXGI֐` : CreateDXGIFactory
		typedef HRESULT ( WINAPI *CreateDXGIFactoryFunc )( REFIID riid, void **ppFactory );


	public:
		static Device* CreateInstance( Boolean bWaitVSync );

	private:
		Mix::Module						m_D3D11Module;
		Mix::Module						m_D3DX11Module;
		Mix::Module						m_DXGIModule;

		DisplayModeList					m_DisplayModeList;
		Boolean							m_IndexTypeSupportList[Mix::Graphics::IT_MAX];
		RESOURCEFORMAT					m_ResourceFmtList[Mix::Graphics::FMT_MAX];

		CAPS							m_Caps;

		D3D_FEATURE_LEVEL				m_FeatureLevel;
		DXGI_SWAP_CHAIN_DESC			m_SwapChainDesc;
		D3D11_VIEWPORT					m_Viewport;
		Mix::Rectangle					m_ViewportRect;
		Mix::Rectangle					m_ScissorRect;
		Mix::Graphics::RASTERIZER_DESC	m_RasterizeDesc;
		Mix::Graphics::DEPTH_DESC		m_DepthDesc;
		Mix::Graphics::BLEND_DESC		m_BlendDesc;

		IDXGIFactory*					m_pDXGIFactory;
		IDXGIAdapter*					m_pDXGIAdapter;
		IDXGIOutput*					m_pDXGIOutput;
		IDXGISwapChain*					m_pDXGISwapChain;
		ID3D11Texture2D*				m_pBackBuffer;
		ID3D11RenderTargetView*			m_pBackBufferView;
		ID3D11Device*					m_pD3DDevice;
		ID3D11DeviceContext*			m_pD3DDeviceContext;

		ID3D11RasterizerState*			m_pRasterizerStateArray[Mix::Graphics::FILL_MAX][Mix::Graphics::CULL_MAX][2];
		ID3D11DepthStencilState*		m_pDepthStateArray[2][2][Mix::Graphics::ZFUNC_MAX];
		ID3D11BlendState*				m_pBlendStateArray[Mix::Graphics::BLEND_MAX][Mix::Graphics::COLOR_WRITE_MASK_MAX];
		ID3D11SamplerState*				m_pSamplerArray[Mix::Graphics::TEXTUREFILTER_MAX][Mix::Graphics::TEXTUREADDRESS_MAX];

		HWND							m_hWnd;
		Boolean							m_bWindowed;

		TARGETSTATE						m_TargetState;
		DRAWSTATE						m_DrawState;

		Mix::Point						m_ScreenSize;
		UInt32							m_DrawPrimitiveCount;
		Mix::Graphics::SHADER_MODEL		m_ShaderModel;

	private:
		Device( Boolean bWaitVSync );
		virtual ~Device( void );

		HRESULT CreateBackBuffer( const wchar_t* failedMsg );
		Boolean ResizeBackBuffer( void );

		void ResetTargetState( void );

		void ResetDrawState( void );
		void FlushDrawState( UInt32 startVertex, UInt32 startIndex );

		Boolean CreateDeviceObject( const void* pBuff, UInt32 buffSize, D3D11_TEXTURE2D_DESC& desc, ID3D11Texture2D** ppD3DTexture, ID3D11ShaderResourceView** ppD3DView );

		UInt32 ComputePrimitiveCount( Mix::Graphics::PRIMITIVE_TYPE type, UInt32 num );

		Boolean CompileShaderFromMemory(	const wchar_t* pNameLabel, const wchar_t* pName,
											const void* pSrc, UInt32 srcSize,
											Mix::Graphics::SHADER_MACRO* pMacros,
											const char* pFuncName,
											Mix::Graphics::SHADER_TARGET target,
											Mix::Memory::IBuffer** ppBuffer );

	public:
		HRESULT CreateBuffer( D3D11_BIND_FLAG bindFlags, UInt32 size, Boolean bDynamic, const void* pInitialData, ID3D11Buffer** ppBuffer );
		HRESULT CreateVertexBuffer( UInt32 count, UInt32 stride, Boolean bDynamic, const void* pInitialData, ID3D11Buffer** ppBuffer );
		HRESULT CreateIndexBuffer( Mix::Graphics::INDEX_TYPE type, UInt32 count, Boolean bDynamic, const void* pInitialData, ID3D11Buffer** ppBuffer );

		HRESULT Map( ID3D11Resource* pResource, Boolean bDynamic, D3D11_MAPPED_SUBRESOURCE& mapped );
		void Unmap( ID3D11Resource* pResource );

		void BeginQuery( ID3D11Query* pQuery );
		void EndQuery( ID3D11Query* pQuery );
		HRESULT GetQueryData( ID3D11Query* pQuery, Boolean bFlush, UInt32& data );

		static Mix::Graphics::FORMAT ConvertTextureFormat( DXGI_FORMAT d3dFormat );
		const wchar_t* GetD3DResultText( HRESULT ret );

	protected:
		virtual void OnDispose( void );

	public:
		virtual Boolean Initialize( Mix::Graphics::SHADER_MODEL shaderModel, Boolean bFullscreen, Mix::UserFile* pSysReport );
		virtual Boolean MessageProc( UInt32 msg, WPARAM wParam, LPARAM lParam );

		virtual Boolean CompileShader(	const void* pSrc, UInt32 srcSize,
										Mix::Graphics::SHADER_MACRO* pMacros,
										const char* pFuncName,
										Mix::Graphics::SHADER_TARGET target,
										Mix::Memory::IBuffer** ppBuffer );

	public:
		virtual Boolean CreateVertexLayout( const Mix::Graphics::VERTEX_ELEMENT* pElements, UInt32 numElement, const Mix::Graphics::IVertexShader* pVertexShader, Mix::Graphics::IVertexLayout** ppVertexLayout, const wchar_t* pName );

		virtual Boolean CreateHullShaderFromFile( const wchar_t* pFileName, Mix::Graphics::IHullShader** ppHullShader );
		virtual Boolean CreateHullShaderFromMemory( const void* pSrc, UInt32 srcSize, Mix::Graphics::IHullShader** ppHullShader, const wchar_t* pName );
		virtual Boolean CreateDomainShaderFromFile( const wchar_t* pFileName, Mix::Graphics::IDomainShader** ppDomainShader );
		virtual Boolean CreateDomainShaderFromMemory( const void* pSrc, UInt32 srcSize, Mix::Graphics::IDomainShader** ppDomainShader, const wchar_t* pName );
		virtual Boolean CreateGeometryShaderFromFile( const wchar_t* pFileName, Mix::Graphics::IGeometryShader** ppGeometryShader );
		virtual Boolean CreateGeometryShaderFromMemory( const void* pSrc, UInt32 srcSize, Mix::Graphics::IGeometryShader** ppGeometryShader, const wchar_t* pName );
		virtual Boolean CreateVertexShaderFromFile( const wchar_t* pFileName, Mix::Graphics::IVertexShader** ppVertexShader );
		virtual Boolean CreateVertexShaderFromMemory( const void* pSrcData, UInt32 srcDataSize, Mix::Graphics::IVertexShader** ppVertexShader, const wchar_t* pName );
		virtual Boolean CreatePixelShaderFromFile( const wchar_t* pFileName, Mix::Graphics::IPixelShader** ppPixelShader );
		virtual Boolean CreatePixelShaderFromMemory( const void* pSrcData, UInt32 srcDataSize, Mix::Graphics::IPixelShader** ppPixelShader, const wchar_t* pName );

		virtual Boolean CreateShaderConstant( UInt32 size, Boolean bDynamic, const void* pInitialData, Mix::Graphics::IShaderConstant** ppShaderConstant, const wchar_t* pName = NULL );

		virtual Boolean CreateVertexBuffer( UInt32 count, UInt32 stride, UInt32 resizeStep, Boolean bDynamic, const void* pInitialData, Mix::Graphics::IVertexBuffer** ppVertexBuffer, const wchar_t* pName );
		virtual Boolean CreateIndexBuffer( Mix::Graphics::INDEX_TYPE type, UInt32 count, UInt32 resizeStep, Boolean bDynamic, const void* pInitialData, Mix::Graphics::IIndexBuffer** ppIndexBuffer, const wchar_t* pName );

		virtual Boolean CreateTextureFromFile( const wchar_t* pFileName, Mix::Graphics::ITexture** ppTexture );
		virtual Boolean CreateTextureFromMemory( const wchar_t* pName, const void* pSrc, UInt32 srcSize, Mix::Graphics::ITexture** ppTexture );
		virtual Boolean CreateCubeTexture( UInt32 edgeLength, Mix::Graphics::FORMAT format, Mix::Graphics::ITexture** ppTexture, const wchar_t* pName );
		virtual Boolean CreateDynamicTexture( UInt32 width, UInt32 height, Mix::Graphics::FORMAT format, Mix::Graphics::ITexture** ppTexture, const wchar_t* pName );
		virtual Boolean CreateTargetTexture( UInt32 width, UInt32 height, Mix::Graphics::FORMAT format, Mix::Graphics::ITexture** ppTexture, const wchar_t* pName );
		virtual Boolean CreateDepthTexture( UInt32 width, UInt32 height, Mix::Graphics::FORMAT format, Mix::Graphics::ITexture** ppTexture, const wchar_t* pName );

		virtual Boolean CreateQuery( Mix::Graphics::QUERY_TYPE type, Mix::Graphics::IQuery** ppQuery );

		virtual Boolean CheckTextureFormat( Mix::Graphics::FORMAT format );
		virtual Boolean CheckCubeTextureFormat( Mix::Graphics::FORMAT format );
		virtual Boolean CheckDynamicTextureFormat( Mix::Graphics::FORMAT format );
		virtual Boolean CheckTargetTextureFormat( Mix::Graphics::FORMAT format );
		virtual Boolean CheckDepthTextureFormat( Mix::Graphics::FORMAT format );
		virtual Boolean CheckIndexType( Mix::Graphics::INDEX_TYPE type );

		virtual void ChangeWindowMode( Boolean bWindowed );
		virtual Boolean GetWindowMode( void ) const;

		virtual Boolean Begin( void );
		virtual void End( void );
		virtual void Present( void );

		virtual void SetTarget( Mix::Graphics::ITexture* pT0, Mix::Graphics::ITexture* pT1, Mix::Graphics::ITexture* pT2, Mix::Graphics::ITexture* pT3, Mix::Graphics::ITexture* pDS );
		virtual void GetTarget( Mix::Graphics::ITexture** ppT0, Mix::Graphics::ITexture** ppT1, Mix::Graphics::ITexture** ppT2, Mix::Graphics::ITexture** ppT3, Mix::Graphics::ITexture** ppDS );

		virtual void Clear( Boolean clearTarget, Boolean clearDepth, const Mix::Vector4& color, Float32 z );

		virtual void SetViewport( Int32 x, Int32 y, Int32 width, Int32 height );
		virtual void SetViewport( const Mix::Rectangle& rect );
		virtual const Mix::Rectangle& GetViewport( void ) const;

		virtual Mix::Matrix4x4 GetScreenTransformMatrix( void ) const;

		virtual void SetScissor( Int32 x, Int32 y, Int32 width, Int32 height );
		virtual void SetScissor( const Mix::Rectangle& rect );
		virtual const Mix::Rectangle& GetScissor( void ) const;

		virtual void SetRasterizerState( Mix::Graphics::FILL_TYPE fillMode, Mix::Graphics::CULL_TYPE cullMode, Boolean bScissoring, Boolean bMultisample );
		virtual void SetRasterizerState( const Mix::Graphics::RASTERIZER_DESC& state );
		virtual const Mix::Graphics::RASTERIZER_DESC& GetRasterizerState( void ) const;

		virtual void SetDepthState( Boolean bTest, Boolean bWrite, Mix::Graphics::ZFUNC_TYPE func );
		virtual void SetDepthState( const Mix::Graphics::DEPTH_DESC& state );
		virtual const Mix::Graphics::DEPTH_DESC& GetDepthState( void ) const;

		virtual void SetBlendState( Mix::Graphics::BLEND_TYPE type, Mix::Graphics::COLOR_WRITE_MASK colorWriteMask );
		virtual void SetBlendState( const Mix::Graphics::BLEND_DESC& desc );
		virtual const Mix::Graphics::BLEND_DESC& GetBlendState( void ) const;

		virtual void SetVertexLayout( Mix::Graphics::IVertexLayout* pVertexLayout );

		virtual void SetHullShader( Mix::Graphics::IHullShader* pHullShader );
		virtual void SetDomainShader( Mix::Graphics::IDomainShader* pDomainShader );
		virtual void SetGeometryShader( Mix::Graphics::IGeometryShader* pGeometryShader );
		virtual void SetVertexShader( Mix::Graphics::IVertexShader* pVertexShader );
		virtual void SetPixelShader( Mix::Graphics::IPixelShader* pPixelShader );

		virtual void SetVertexShaderConstantB( UInt32 startRegister, const Boolean* data, UInt32 count );
		virtual void SetVertexShaderConstantI( UInt32 startRegister, const Int32* data, UInt32 count );
		virtual void SetVertexShaderConstantF( UInt32 startRegister, const Float32* data, UInt32 count );
		virtual void SetVertexShaderConstantF( UInt32 startRegister, const Mix::Vector4* data, UInt32 count );
		virtual void SetVertexShaderConstantF( UInt32 startRegister, const Mix::Matrix4x4* data, UInt32 count );
		virtual void SetPixelShaderConstantB( UInt32 startRegister, const Boolean* data, UInt32 count );
		virtual void SetPixelShaderConstantI( UInt32 startRegister, const Int32* data, UInt32 count );
		virtual void SetPixelShaderConstantF( UInt32 startRegister, const Float32* data, UInt32 count );
		virtual void SetPixelShaderConstantF( UInt32 startRegister, const Mix::Vector4* data, UInt32 count );
		virtual void SetPixelShaderConstantF( UInt32 startRegister, const Mix::Matrix4x4* data, UInt32 count );

		virtual void SetHullShaderConstant( UInt32 slot, Mix::Graphics::IShaderConstant* pShaderConstant );
		virtual void SetDomainShaderConstant( UInt32 slot, Mix::Graphics::IShaderConstant* pShaderConstant );
		virtual void SetGeometryShaderConstant( UInt32 slot, Mix::Graphics::IShaderConstant* pShaderConstant );
		virtual void SetVertexShaderConstant( UInt32 slot, Mix::Graphics::IShaderConstant* pShaderConstant );
		virtual void SetPixelShaderConstant( UInt32 slot, Mix::Graphics::IShaderConstant* pShaderConstant );

		virtual void SetVertexBuffer( Mix::Graphics::IVertexBuffer* pVertexBuffer );
		virtual void SetIndexBuffer( Mix::Graphics::IIndexBuffer* pIndexBuffer );

		virtual UInt32 GetTextureStageMax( void ) const;
		virtual void SetTexture( UInt32 stage, Mix::Graphics::TEXTURE_FILTER_TYPE filter, Mix::Graphics::TEXTURE_ADDRESS_TYPE address, Mix::Graphics::ITexture* pTexture );
		virtual void ResetTexture( UInt32 stage );

		virtual void Draw( Mix::Graphics::PRIMITIVE_TYPE primitiveType, UInt32 startVertex, UInt32 numVertex );
		virtual void DrawIndexed( Mix::Graphics::PRIMITIVE_TYPE primitiveType, UInt32 startVertex, UInt32 numVertex, UInt32 startIndex, UInt32 numIndex );

		virtual Boolean SaveScreenshot( const wchar_t* pFileName );
		virtual Boolean SaveTexture( Mix::Graphics::ITexture* pTexture, const wchar_t* pFileName );

		virtual const Mix::Point& GetScreenSize( void ) const;
		virtual UInt32 GetDrawPrimitiveCount( void ) const;
		virtual Mix::Graphics::SHADER_MODEL GetShaderModel( void ) const;

	private:
		static const wchar_t* FAILED_INITIALIZE;
		static const wchar_t* FAILED_COMPILEVERTEXSHADER;
		static const wchar_t* FAILED_COMPILEPIXELSHADER;
		static const wchar_t* FAILED_CREATEVERTEXLAYOUT;
		static const wchar_t* FAILED_CREATEHULLSHADER;
		static const wchar_t* FAILED_CREATEDOMAINSHADER;
		static const wchar_t* FAILED_CREATEGEOMETRYSHADER;
		static const wchar_t* FAILED_CREATEVERTEXSHADER;
		static const wchar_t* FAILED_CREATEPIXELSHADER;
		static const wchar_t* FAILED_CREATESHADERCONST;
		static const wchar_t* FAILED_CREATEVERTEXBUFFER;
		static const wchar_t* FAILED_CREATEINDEXBUFFER;
		static const wchar_t* FAILED_CREATETEXTURE;
		static const wchar_t* FAILED_CREATECUBETEXTURE;
		static const wchar_t* FAILED_CREATEDYNAMICTEXTURE;
		static const wchar_t* FAILED_CREATETARGETTEXTURE;
		static const wchar_t* FAILED_CREATEDEPTHTEXTURE;
		static const wchar_t* FAILED_CREATEQUERY;
		static const wchar_t* FAILED_RESIZEBACKBUFFER;
		static const wchar_t* FAILED_DRAW;
		static const wchar_t* FAILED_DRAWINDEXED;
		static const wchar_t* FAILED_SAVESCREENSHOT;
		static const wchar_t* FAILED_SAVETEXTURE;
	};

}}}
