#include "Mix/Class/Graphics/Common/Device.h"
#include "Mix/Memory/IBuffer.h"

namespace Mix{ namespace Graphics{ namespace Common{

const wchar_t* Device::FAILED_COMPILESHADER = L"VF[_[̃RpCɎs";

// tH[}bgeLXge[u
const wchar_t* Device::FORMAT_TEXT_TABLE[Mix::Graphics::FMT_MAX] =
{
	L"UNKNOWN",
	L"D16",
	L"D32",
	L"D24S8",
	L"R8G8B8A8",
	L"A8",
	L"R16F",
	L"R32F",
	L"R16G16F",
	L"R32G32F",
	L"R16G16B16A16F",
	L"R32G32B32A32F",
	L"DXT1",
	L"DXT2",
	L"DXT3",
	L"DXT4",
	L"DXT5",
};

// o[ebNXGg̃TCYe[u
UInt32 Device::VERTEX_ELEMENT_SIZE_TABLE[Mix::Graphics::VERTEX_ELEMENT_FORMAT_MAX] =
{
	4,	//VLF_FLOAT32_1
	8,	//VLF_FLOAT32_2
	12,	//VLF_FLOAT32_3
	16,	//VLF_FLOAT32_4
	4,	//VLF_UINT8_4
	4,	//VLF_INT16_2
	8,	//VLF_INT16_4
	4,	//VLF_UINT8_4N
	4,	//VLF_INT16_2N
	8,	//VLF_INT16_4N
	4,	//VLF_UINT16_2N
	8,	//VLF_UINT16_4N
	4,	//VLF_FLOAT16_2
	8,	//VLF_FLOAT16_4
};

Device::Device( Boolean bWaitVSync ) :
m_pWindow( NULL ),
m_pFileMgr( NULL ),
m_bWaitVSync( bWaitVSync )
{
	MIX_ASSERT( Mix::GetInternalWindowPtr() != NULL );
	MIX_ASSERT( Mix::File::GetManagerPtr() != NULL );

	Mix::Window* pWindow = Mix::GetInternalWindowPtr();
	Mix::File::IManager* pFileMgr = Mix::File::GetManagerPtr();

	MIX_ADD_REF( pWindow );
	m_pWindow = pWindow;

	MIX_ADD_REF( pFileMgr );
	m_pFileMgr = pFileMgr;
}

Device::~Device( void )
{
}

UInt32 Device::GetTextureDimension( const void* pSrc, UInt32 srcSize )
{
	if( ( 4 + sizeof( DDSURFACEDESC2 ) ) > srcSize )
	{
		return 2;
	}

	const UInt8* pAddr = static_cast<const UInt8*>( pSrc );
	UInt32 magicNumber = *reinterpret_cast<const UInt32*>( &( pAddr[0] ) );
	UInt32 dim = 2;

	if( magicNumber == 0x20534444/*DDS*/ )
	{
		const DDSURFACEDESC2* pDesc = reinterpret_cast<const DDSURFACEDESC2*>( &( pAddr[4] ) );

		if( pDesc->dwSize == 124 )
		{
			if( MIX_TESTBIT( pDesc->dwFlags, DDSD_DEPTH ) == DDSD_DEPTH )
			{
				//3DeNX`
				dim = 3;
			}
			else
			{
				//2DeNX`
				;
			}
		}
	}

	return dim;
}

Boolean Device::Initialize( Mix::Graphics::SHADER_MODEL shaderModel, Boolean bFullscreen, Mix::UserFile* pSysReport )
{
	return True;
}

Boolean Device::Update( void )
{
	return True;
}

Boolean Device::MessageProc( UInt32 msg, WPARAM wParam, LPARAM lParam )
{
	return True;
}

Boolean Device::CompileShaderFromFile(	const wchar_t* pFilePath,
										Mix::Graphics::SHADER_MACRO* pMacros,
										const char* pFuncName,
										Mix::Graphics::SHADER_TARGET target,
										Mix::Memory::IBuffer** ppBuffer )
{
	if( ( pFilePath == NULL ) ||
		( ::wcslen( pFilePath ) == 0 ) ||
		( pFuncName == NULL ) ||
		( ::strlen( pFuncName ) == 0 ) ||
		( ppBuffer == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : pFilePath[%s] pFuncName[%s] ppBuffer[%s]",
			Device::FAILED_COMPILESHADER,
			Mix::STR_ILLEGALARG,
			MIX_LOG_STR( pFilePath ),
			MIX_LOG_STR( Mix::StringW( pFuncName ).GetConstPtr() ),
			MIX_LOG_PTR( ppBuffer ) );

		return False;
	}

	Mix::Memory::IBuffer* pSrc = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"VF[_[̃RpC : FilePath[%s] FuncName[%s] Target[%s]",
		pFilePath,
		Mix::StringW( pFuncName ).GetConstPtr(),
		Mix::StringW( Mix::Graphics::SHADER_TARGET_TABLE[target] ).GetConstPtr() );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// s
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( m_pFileMgr->CreateBufferFromFile( pFilePath, &pSrc ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		return False;
	}

	if( CompileShader( pSrc->GetConstPointer(), pSrc->GetSize(), pMacros, pFuncName, target, ppBuffer ) == False )
	{
		MIX_LOG_INFO_SECT_END();
		MIX_RELEASE( pSrc );
		return False;
	}

	MIX_RELEASE( pSrc );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO( L" : Size[%dByte]", ( *ppBuffer )->GetSize() );
	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Device::CompileShaderFromMemory(	const wchar_t* pName,
											const void* pSrc, UInt32 srcSize,
											Mix::Graphics::SHADER_MACRO* pMacros,
											const char* pFuncName,
											Mix::Graphics::SHADER_TARGET target,
											Mix::Memory::IBuffer** ppBuffer )
{
	Mix::StringW safeName = MIX_SAFE_NAME( pName );

	if( ( pSrc == NULL ) ||
		( srcSize == 0 ) ||
		( pFuncName == NULL ) ||
		( ::strlen( pFuncName ) == 0 ) ||
		( ppBuffer == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : Name[%s] : pSrc[%s] srcSize[%d] pFuncName[%s] ppBuffer[%s]",
			Device::FAILED_COMPILESHADER,
			Mix::STR_ILLEGALARG,
			safeName.GetConstPtr(),
			MIX_LOG_PTR( pSrc ),
			srcSize,
			MIX_LOG_STR( Mix::StringW( pFuncName ).GetConstPtr() ),
			MIX_LOG_PTR( ppBuffer ) );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : Jn
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO_SECT_START( L"VF[_[̃RpC : Name[%s] FuncName[%s] Target[%s]",
		safeName.GetConstPtr(),
		Mix::StringW( pFuncName ).GetConstPtr(),
		Mix::StringW( Mix::Graphics::SHADER_TARGET_TABLE[target] ).GetConstPtr() );

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// s
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( CompileShader( pSrc, srcSize, pMacros, pFuncName, target, ppBuffer ) == False )
	{
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// O : I
	////////////////////////////////////////////////////////////////////////////////////////////////////

	MIX_LOG_INFO( L" : Size[%dByte]", ( *ppBuffer )->GetSize() );
	MIX_LOG_INFO_SECT_END();

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

void Device::Dispose( void )
{
	OnDispose();

	MIX_RELEASE( m_pWindow );
	MIX_RELEASE( m_pFileMgr );
}

void Device::OnDispose( void )
{
}

void Device::AppendDeviceObject( Mix::Graphics::Common::DeviceObject* pDeviceObject )
{
	if( pDeviceObject->IsRegisterMap() == True )
	{
		m_Map.insert( std::pair<Mix::String, Mix::Graphics::Common::DeviceObject*>( pDeviceObject->GetName(), pDeviceObject ) );
	}
	else
	{
		m_List.push_back( pDeviceObject );
	}

	m_SafeList.push_back( pDeviceObject );
}

void Device::RemoveDeviceObject( Mix::Graphics::Common::DeviceObject* pDeviceObject )
{
	if( pDeviceObject->IsRegisterMap() == True )
	{
		DeviceObjectMap::iterator it = m_Map.find( pDeviceObject->GetName() );
		if( it != m_Map.end() )
		{
			m_Map.erase( it );
		}
	}
	else
	{
		m_List.remove( pDeviceObject );
	}

	m_SafeList.remove( pDeviceObject );
}

Mix::Graphics::Common::DeviceObject* Device::FindDeviceObject( const wchar_t* pName )
{
	DeviceObjectMap::iterator it;
	
	it = m_Map.find( pName );
	if( it == m_Map.end() )
	{
		return NULL;
	}

	return it->second;
}

void Device::InvalidateDeviceObject( void )
{
	for( DeviceObjectList::iterator it = m_SafeList.begin(); it != m_SafeList.end(); ++it )
	{
		( *it )->Invalidate();
	}
}

void Device::ValidateDeviceObject( void )
{
	for( DeviceObjectList::iterator it = m_SafeList.begin(); it != m_SafeList.end(); ++it )
	{
		( *it )->Validate();
	}
}

const wchar_t* Device::GetFormatText( Mix::Graphics::FORMAT format ) const
{
	return Device::FORMAT_TEXT_TABLE[format];
}

}}}
