#include "Mix/Geometry.h"

namespace Mix{ namespace Geometry{

Float32 DistancePointPlane( const Mix::Vector3& p, const Mix::Geometry::Plane& plane )
{
	return ( Mix::Vector3::Dot( plane.normal, p ) + plane.d );
}

void ClosestPointSphere( const Mix::Vector3& p, const Mix::Geometry::Sphere& sphere, Mix::Vector3& q )
{
	Mix::Vector3 v = ( p - sphere.center );

	if( sphere.radius >= v.GetLength() )
	{
		q = p;
	}

	v.Normalize();
	q = ( sphere.center + ( v * sphere.radius ) );
}

void ClosestPointPlane( const Mix::Vector3& p, const Mix::Geometry::Plane& plane, Mix::Vector3& q )
{
	Float32 t = Mix::Vector3::Dot( plane.normal, p ) - plane.d;

	q = ( p - ( t * plane.normal ) );
}

void ClosestPointOBB( const Mix::Vector3& p, const Mix::Geometry::OBB& obb, Mix::Vector3& q )
{
	Int32 i;
	Mix::Vector3 d = ( p - obb.center );

	q = obb.center;

	for( i = 0; i < 3; i++ )
	{
		float dist = Mix::Vector3::Dot( d, obb.axis[i] );

		if( dist > obb.length.data[i] )
		{
			dist = obb.length.data[i];
		}

		if( dist < -obb.length.data[i] )
		{
			dist = -obb.length.data[i];
		}

		q += ( obb.axis[i] * dist );
	}
}

void ClosestPointAABB( const Mix::Vector3& p, const Mix::Geometry::AABB& aabb, Mix::Vector3& q )
{
	const Float32* point = p.data;
	const Float32* pointEnd = point + 3;
	const Float32* aabbMin = aabb.min.data;
	const Float32* aabbMax = aabb.max.data;

	Float32* out = q.data;

	Float32 elm;
	Float32 tmp;

	while( point != pointEnd )
	{
		elm = *point++;

		tmp = *aabbMin++;
		if( elm < tmp ) { elm = tmp; }

		tmp = *aabbMax++;
		if( elm > tmp ) { elm = tmp; }

		*out++ = elm;
	}
}

Boolean PointInTriangle( const Mix::Vector3& p, const Mix::Vector3& a, const Mix::Vector3& b, const Mix::Vector3& c )
{
	Mix::Vector3 aa = ( a - p );
	Mix::Vector3 bb = ( b - p );
	Mix::Vector3 cc = ( c - p );
	Mix::Vector3 u = Mix::Vector3::Cross( bb, cc );
	Mix::Vector3 v = Mix::Vector3::Cross( cc, aa );

	if( Mix::Vector3::Dot( u, v ) < MIX_FLOAT_EPSILON )
	{
		return False;
	}

	Mix::Vector3 w = Mix::Vector3::Cross( a, b );

	if( Mix::Vector3::Dot( u, w ) < MIX_FLOAT_EPSILON )
	{
		return False;
	}

	return True;
}

Boolean IntersectPointAABB( const Mix::Vector3& p, const Mix::Geometry::AABB& aabb )
{
	const Mix::Vector3& aabbMin = aabb.min;
	const Mix::Vector3& aabbMax = aabb.max;

	if( ( aabbMin.x >  p.x ) || ( aabbMin.y >  p.y ) || ( aabbMin.z >  p.z ) ||
		( aabbMax.x <= p.x ) || ( aabbMax.y <= p.y ) || ( aabbMax.z <= p.z ) )
	{
		return False;
	}

	return True;
}

Boolean IntersectSphereSphere( const Mix::Geometry::Sphere& sphere1, const Mix::Geometry::Sphere& sphere2, Float32* pDist )
{
	Mix::Vector3 vec = ( sphere1.center - sphere2.center );
	Float32 dist = vec.GetLength();

	if( pDist != NULL )
	{
		*pDist = dist;
	}

	return ( sphere1.radius + sphere2.radius ) >= dist;
}

Boolean IntersectSpherePlane( const Mix::Geometry::Sphere& sphere, const Mix::Geometry::Plane& plane, Float32* pDist )
{
	Float32 distance = Mix::Vector3::Dot( sphere.center, plane.normal ) - plane.d;

	if( pDist != NULL )
	{
		( *pDist ) = distance;
	}

	return ::fabs( distance ) <= sphere.radius;
}

Boolean IntersectSphereOBB( const Mix::Geometry::Sphere& sphere, const Mix::Geometry::OBB& obb, Mix::Vector3* pPos )
{
	Mix::Vector3 pos;
	Mix::Vector3 vec;

	ClosestPointOBB( sphere.center, obb, pos );

	if( pPos != NULL )
	{
		*pPos = pos;
	}

	vec = ( pos - sphere.center );

	return ( Mix::Vector3::Dot( vec, vec ) <= ( sphere.radius * sphere.radius ) );
}

Boolean IntersectSphereAABB( const Mix::Geometry::Sphere& sphere, const Mix::Geometry::AABB& aabb, Mix::Vector3* pPos )
{
	Mix::Vector3 pos;
	Mix::Vector3 vec;

	ClosestPointAABB( sphere.center, aabb, pos );

	if( pPos != NULL )
	{
		*pPos = pos;
	}

	vec = ( pos - sphere.center );

	return ( vec.GetLengthSqr() <= ( sphere.radius * sphere.radius ) );
}

Boolean IntersectPlanePlane( const Mix::Geometry::Plane& plane1, const Mix::Geometry::Plane& plane2, Mix::Vector3* pPos, Mix::Vector3* pDir )
{
	Mix::Vector3 d = Mix::Vector3::Cross( plane1.normal, plane2.normal );
	Float32 denom = Mix::Vector3::Dot( d, d );

	if( pDir != NULL )
	{
		*pDir = d;
	}

	if( denom < MIX_FLOAT_EPSILON )
	{
		return False;
	}

	if( pPos != NULL )
	{
		*pPos = ( Mix::Vector3::Cross( ( plane1.d * plane2.normal ) - ( plane2.d * plane1.normal ), d ) / denom );
	}

	return True;
}

Boolean IntersectPlaneOBB( const Mix::Geometry::Plane& plane, const Mix::Geometry::OBB& obb )
{
	Int32 i;
	Float32 r;
	Float32 s;

	r = 0;
	for( i = 0; i < 3; i++ )
	{
		r += ( obb.length.data[i] * ::fabs( Mix::Vector3::Dot( plane.normal, obb.axis[i] ) ) );
	}

	s = ( Mix::Vector3::Dot( plane.normal, obb.center ) - plane.d );

	return ( ::fabs( s ) <= r );
}

Boolean IntersectPlaneAABB( const Mix::Geometry::Plane& plane, const Mix::Geometry::AABB& aabb )
{
	Int32 i;
	Float32 r;
	Float32 s;
	Mix::Vector3 c = ( ( aabb.max + aabb.min ) * 0.5f );
	Mix::Vector3 e = ( aabb.max - c );

	r = 0;
	for( i = 0; i < 3; i++ )
	{
		r += e.data[i] * ::fabs( plane.normal.data[i] );
	}

	s = ( Mix::Vector3::Dot( plane.normal, c ) - plane.d );

	return ( ::fabs( s ) <= r );
}

Boolean IntersectOBBOBB( const Mix::Geometry::OBB& obb1, const Mix::Geometry::OBB& obb2 )
{
	static const Int32 INDEX_TABLE[9][18] = 
	{
		{ 1, 2, 0, 2, 1, 0, /**/ 1, 0, 2, 2, 0, 1, /**/ 2, 1, 0, 1, 2, 0, },
		{ 1, 2, 1, 2, 1, 1, /**/ 0, 0, 2, 2, 0, 0, /**/ 2, 1, 1, 1, 2, 1, },
		{ 1, 2, 2, 2, 1, 2, /**/ 0, 0, 1, 1, 0, 0, /**/ 2, 1, 2, 1, 2, 2, },
		{ 0, 2, 0, 2, 0, 0, /**/ 1, 1, 2, 2, 1, 1, /**/ 0, 2, 0, 2, 0, 0, },
		{ 0, 2, 1, 2, 0, 1, /**/ 0, 1, 2, 2, 1, 0, /**/ 0, 2, 1, 2, 0, 1, },
		{ 0, 2, 2, 2, 0, 2, /**/ 0, 1, 1, 1, 1, 0, /**/ 0, 2, 2, 2, 0, 2, },
		{ 0, 1, 0, 1, 0, 0, /**/ 1, 2, 2, 2, 2, 1, /**/ 1, 0, 0, 0, 1, 0, },
		{ 0, 1, 1, 1, 0, 1, /**/ 0, 2, 2, 2, 2, 0, /**/ 1, 0, 1, 0, 1, 1, },
		{ 0, 1, 2, 1, 0, 2, /**/ 0, 2, 1, 1, 2, 0, /**/ 1, 0, 2, 0, 1, 2, },
	};

	Int32 i;
	Int32 j;
	Float32 ra;
	Float32 rb;
	Mix::Matrix3x3 R;
	Mix::Matrix3x3 absR;

	for( i = 0; i < 3; i++ )
	{
		for( j = 0; j < 3; j++ )
		{
			R.m[i][j] = Mix::Vector3::Dot( obb1.axis[i], obb2.axis[j] );
		}
	}

	Mix::Vector3 d = ( obb2.center - obb1.center );
	Mix::Vector3 t;

	for( i = 0; i < 3; i++ )
	{
		t.data[i] = Mix::Vector3::Dot( d, obb1.axis[i] );
	}

	//ZG[łȂ悤ɃCvVǉ
	for( i = 0; i < 3; i++ )
	{
		for( j = 0; j < 3; j++ )
		{
			absR.m[i][j] = ( ::fabs( absR.m[i][j] ) + MIX_FLOAT_EPSILON );
		}
	}

	for( i = 0; i < 3; i++ )
	{
		ra = obb1.length.data[i];

		rb = 0;
		for( j = 0; j < 3; j++ )
		{
			rb += ( obb2.length.data[j] * absR.m[i][j] );
		}

		if( ::fabs( t.data[i] ) > ( ra + rb ) )
		{
			return False;
		}
	}

	for( i = 0; i < 3; i++ )
	{
		ra = 0;
		for( j = 0; j < 3; j++ )
		{
			ra += ( obb1.length.data[j] * absR.m[j][i] );
		}

		rb = obb2.length.data[i];

		if( ::fabs( t.data[0] * R.m[0][i] + t.data[1] * R.m[1][i] + t.data[2] * R.m[2][i] ) > ( ra + rb ) )
		{
			return False;
		}
	}

	for( i = 0; i < 9; i++ )
	{
		const Int32* p = &( INDEX_TABLE[i][0] );

		ra = ( ( obb1.length.data[p[0]] * absR.m[p[1]][p[2]] ) + ( obb1.length.data[p[3]] * absR.m[p[4]][p[5]] ) );
		rb = ( ( obb2.length.data[p[6]] * absR.m[p[7]][p[8]] ) + ( obb2.length.data[p[9]] * absR.m[p[10]][p[11]] ) );
		if( ::fabs( t.data[p[12]] * R.m[p[13]][p[14]] - t.data[p[15]] * R.m[p[16]][p[17]] ) > ( ra + rb ) )
		{
			return False;
		}
	}

	return True;
}

Boolean IntersectAABBAABB( const Mix::Geometry::AABB& aabb1, const Mix::Geometry::AABB& aabb2 )
{
	Int32 i;

	for( i = 0; i < 3; i++ )
	{
		if( ( aabb1.max.data[i] < aabb2.min.data[i] ) ||
			( aabb1.min.data[i] > aabb2.max.data[i] ) )
		{
			return False;
		}
	}

	return True;
}

Boolean IntersectRaySphere( const Mix::Vector3& rayPos, const Mix::Vector3& rayDir, const Mix::Geometry::Sphere& sphere, Float32* pDist, Mix::Vector3* pPos )
{
	Mix::Vector3 m = ( rayPos - sphere.center );
	float b = Mix::Vector3::Dot( m, rayDir );
	float c = ( Mix::Vector3::Dot( m, m ) - ( sphere.radius * sphere.radius ) );

	if( ( c > MIX_FLOAT_EPSILON ) &&
		( b > MIX_FLOAT_EPSILON ) )
	{
		return False;
	}

	float discr = ( ( b * b ) - c );

	if( discr < MIX_FLOAT_EPSILON )
	{
		return False;
	}

	float t = ( -b - ::sqrtf( discr ) );

	if( t < MIX_FLOAT_EPSILON )
	{
		t = 0.0f;
	}

	if( pDist != NULL )
	{
		*pDist = t;
	}

	Mix::Vector3 pos = ( rayPos + ( rayDir * t ) );

	if( pPos != NULL )
	{
		*pPos = pos;
	}

	return True;
}

Boolean IntersectRayPlane( const Mix::Vector3& rayPos, const Mix::Vector3& rayDir, const Mix::Geometry::Plane& plane, Float32* pDist, Mix::Vector3* pPos )
{
	Float32 t = ( ( plane.d - Mix::Vector3::Dot( rayPos, plane.normal ) ) / Mix::Vector3::Dot( rayDir, plane.normal ) );

	if( pDist != NULL )
	{
		*pDist = t;
	}

	if( MIX_FLOAT_MIN >= t )
	{
		return False;
	}

	if( pPos != NULL )
	{
		*pPos = ( rayPos + ( rayDir * t ) );
	}

	return True;
}

Boolean IntersectRayOBB( const Mix::Vector3& rayPos, const Mix::Vector3& rayDir, const Mix::Geometry::OBB& obb )
{
	Mix::Vector3 m = ( rayPos + rayDir ) * 0.5f;
	Mix::Vector3 d = ( rayDir - m );

	m = ( m - obb.center );
	m = Mix::Vector3( Mix::Vector3::Dot( obb.axis[0], m ), Mix::Vector3::Dot( obb.axis[1], m ), Mix::Vector3::Dot( obb.axis[2], m ) );
	d = Mix::Vector3( Mix::Vector3::Dot( obb.axis[0], d ), Mix::Vector3::Dot( obb.axis[1], d ), Mix::Vector3::Dot( obb.axis[2], d ) );

	Float32 adx = ::fabsf( d.x );
	if( ::fabsf( m.x ) > obb.length.x + adx )
	{
		return False;
	}

	Float32 ady = ::fabsf( d.y );
	if( ::fabsf( m.y ) > obb.length.y + ady )
	{
		return False;
	}

	Float32 adz = ::fabsf( d.z );
	if( ::fabsf( m.z ) > obb.length.z + adz )
	{
		return False;
	}

	adx += MIX_FLOAT_EPSILON;
	ady += MIX_FLOAT_EPSILON;
	adz += MIX_FLOAT_EPSILON;
        
	if( ::fabsf( m.y * d.z - m.z * d.y ) > obb.length.y * adz + obb.length.z * ady )
	{
		return False;
	}

	if( ::fabsf( m.z * d.x - m.x * d.z ) > obb.length.x * adz + obb.length.z * adx )
	{
		return False;
	}

	if( ::fabsf( m.x * d.y - m.y * d.x ) > obb.length.x * ady + obb.length.y * adx )
	{
		return False;
	}

	return True;
}

Boolean IntersectRayAABB( const Mix::Vector3& rayPos, const Mix::Vector3& rayDir, const Mix::Geometry::AABB& aabb, Float32* pDist, Mix::Vector3* pPos )
{
	Int32 i;
	Float32 tmin = 0.0f;
	Float32 tmax = 10000.0f;

	for( i = 0; i < 3; i++ )
	{
		if( ::fabs( rayDir.data[i] ) < MIX_FLOAT_EPSILON )
		{
			if( ( rayPos.data[i] < aabb.min.data[i] ) ||
				( rayPos.data[i] > aabb.max.data[i] ) )
			{
				return False;
			}
		}
		else
		{
			Float32 ood = ( 1.0f / rayDir.data[i] );
			Float32 t1 = ( ( aabb.min.data[i] - rayPos.data[i] ) * ood );
			Float32 t2 = ( ( aabb.max.data[i] - rayPos.data[i] ) * ood );

			if( t1 > t2 )
			{
				Float32 t = t1;
				t1 = t2;
				t2 = t;
			}

			if( t1 > tmin )
			{
				tmin = t1;
			}

			if( t2 > tmax )
			{
				tmax = t2;
			}

			if( tmin > tmax )
			{
				return False;
			}
		}
	}

	if( pDist != NULL )
	{
		*pDist = tmin;
	}

	if( pPos != NULL )
	{
		*pPos = ( rayPos + ( rayDir * tmin ) );
	}

	return True;
}

Boolean IntersectSegmentSphere( const Mix::Vector3& p1, const Mix::Vector3& p2, const Mix::Geometry::Sphere& sphere )
{
	Mix::Vector3 v = ( p2 - p1 );
	Mix::Vector3 c = ( sphere.center - p1 );
	Float32 d = Mix::Vector3::Dot( v, c );

	if( d < MIX_FLOAT_EPSILON )
	{
		return ( sphere.radius >= c.GetLength() );
	}

	Float32 r2 = ( sphere.radius * sphere.radius );

	if( v.GetLengthSqr() < d )
	{
		Mix::Vector3 ce = ( sphere.center - p2 );
		return ( r2 > ce.GetLengthSqr() );
	}

	Float32 ce2 = ( c.GetLengthSqr() - d / v.GetLengthSqr() );
	if( r2 > ce2 )
	{
		return True;
	}

	return False;
}

Boolean IntersectSegmentPlane( const Mix::Vector3& p1, const Mix::Vector3& p2, const Mix::Geometry::Plane& plane, Float32* pDist, Mix::Vector3* pPos )
{
	Mix::Vector3 ab = ( p2 - p1 );
	Float32 t = ( ( plane.d - Mix::Vector3::Dot( plane.normal, p1 ) ) / Mix::Vector3::Dot( plane.normal, ab ) );

	if( pDist != NULL )
	{
		*pDist = t;
	}

	if( ( t < MIX_FLOAT_EPSILON ) ||
		( t > 1.0f ) )
	{
		return False;
	}

	if( pPos != NULL )
	{
		*pPos = ( p1 + ( ab * t ) );
	}

	return True;
}

Boolean IntersectSegmentOBB( const Mix::Vector3& p1, const Mix::Vector3& p2, const Mix::Geometry::OBB& obb, Float32* pDist, Mix::Vector3* pPos )
{
	return False;
}

Boolean IntersectSegmentAABB( const Mix::Vector3& p1, const Mix::Vector3& p2, const Mix::Geometry::AABB& aabb )
{
	Mix::Vector3 e = ( aabb.max - aabb.min );
	Mix::Vector3 d = ( p2 - p1 );
	Mix::Vector3 m = ( p1 + p2 - aabb.min - aabb.max );

	Float32 adx = ::fabs( d.x );
	if( ::fabs( m.x ) > ( e.x + adx ) )
	{
		return False;
	}

	Float32 ady = ::fabs( d.y );
	if( ::fabs( m.y ) > ( e.y + ady ) )
	{
		return False;
	}

	Float32 adz = ::fabs( d.z );
	if( ::fabs( m.z ) > ( e.z + adz ) )
	{
		return False;
	}

	adx += MIX_FLOAT_EPSILON;
	ady += MIX_FLOAT_EPSILON;
	adz += MIX_FLOAT_EPSILON;

	if( ::fabs( m.y * d.z - m.z * d.y ) > ( e.y * adz * e.z * ady ) )
	{
		return False;
	}

	if( ::fabs( m.z * d.x - m.x * d.z ) > ( e.x * adz * e.z * adx ) )
	{
		return False;
	}

	if( ::fabs( m.x * d.y - m.y * d.x ) > ( e.x * ady * e.y * adx ) )
	{
		return False;
	}

	return True;
}

}}
