#pragma once

#include <vector>

#include "Mix/Dynamics/IWorld.h"

#ifdef _DEBUG
	#include "Mix/Timer.h"
#endif //_DEBUG

namespace Mix{ namespace Dynamics{

	class ObjectContext;
	class Object;

	class World : public Mix::Dynamics::IWorld
	{
	private:
		enum INTERNAL_VALUE
		{
			MANIFOLD_POINT_DEFAULT_CAPACITY = 128,
		};

		struct TestSegmentResultCallback : public btCollisionWorld::ClosestRayResultCallback
		{
			Int32 m_partId;
			Int32 m_triangleIndex;

			TestSegmentResultCallback( const btVector3&	rayFromWorld, const btVector3& rayToWorld );
			virtual btScalar addSingleResult( btCollisionWorld::LocalRayResult& rayResult, bool normalInWorldSpace );
		};

		struct TestSegmentCustomResultCallback : public World::TestSegmentResultCallback
		{
			Mix::Dynamics::IWorld::TestCallback* m_pCallback;

			TestSegmentCustomResultCallback( const btVector3& rayFromWorld, const btVector3& rayToWorld, Mix::Dynamics::IWorld::TestCallback* pCallback );
			virtual btScalar addSingleResult( btCollisionWorld::LocalRayResult& rayResult, bool normalInWorldSpace );
		};

		struct TestSweepResultCallback : public btCollisionWorld::ClosestConvexResultCallback
		{
			Int32 m_partId;
			Int32 m_triangleIndex;

			TestSweepResultCallback( const btVector3& convexFromWorld, const btVector3& convexToWorld );
			virtual	btScalar addSingleResult( btCollisionWorld::LocalConvexResult& convexResult, bool normalInWorldSpace );
		};

		struct TestSweepCustomResultCallback : public World::TestSweepResultCallback
		{
			Mix::Dynamics::IWorld::TestCallback* m_pCallback;

			TestSweepCustomResultCallback( const btVector3& convexFromWorld, const btVector3& convexToWorld, Mix::Dynamics::IWorld::TestCallback* pCallback );
			virtual	btScalar addSingleResult( btCollisionWorld::LocalConvexResult& convexResult, bool normalInWorldSpace );
		};

		typedef std::vector<Mix::Dynamics::IJoint*> JointList;
		typedef std::vector<Mix::Dynamics::IObject*> ObjectList;
		typedef std::vector<Mix::Dynamics::ObjectContext*> ObjectContextList;
		typedef std::vector<Mix::Dynamics::IRigidBody*> RigidBodyList;
		typedef std::vector<Mix::Dynamics::MANIFOLD_POINT> ManifoldPointList;

	public:
		static World* CreateInstance( const wchar_t* pName );

	private:
		Mix::Vector3 m_Gravity;
		World::JointList m_JointList;
		World::ObjectList m_ObjectList;
		World::ObjectContextList m_ObjectContextList;
		World::RigidBodyList m_RigidBodyList;
		UInt32 m_JointCountTable[Mix::Dynamics::JOINT_TYPE_MAX];
		UInt32 m_ObjectCountTable[Mix::Dynamics::OBJECT_TYPE_MAX];
		World::ManifoldPointList m_ManifoldPoints;

		btCollisionDispatcher* m_pDispatcher;
		btDbvtBroadphase* m_pBroadphas;
		btSequentialImpulseConstraintSolver* m_pSolver;
		btDefaultCollisionConfiguration* m_pConfiguration;
		btDiscreteDynamicsWorld* m_pWorld;
		btGhostPairCallback* m_pGhostPairCallback;

		Mix::Dynamics::DEBUG_WORLD_PROFILE m_Profile;

#ifdef _DEBUG
		Mix::Timer m_DebTimer;
#endif //_DEBUG

	private:
		World( const wchar_t* pName );
		virtual ~World( void );

		Boolean TestRay( World::TestSegmentResultCallback& callback, Mix::Dynamics::IWorld::TEST_RESULT& result );
		Boolean TestSweep( Mix::Dynamics::IShape* pShape, World::TestSweepResultCallback& callback, Mix::Dynamics::IWorld::TEST_RESULT& result );

		static bool ContactAddedCallback(	btManifoldPoint& cp,
											const btCollisionObjectWrapper* cow0, int partId0, int index0,
											const btCollisionObjectWrapper* cow1, int partId1, int index1 );

	public:
		Boolean Initialize( void );

		void BeginRefreshObject( Mix::Dynamics::Object* pObject );
		void EndRefreshObject( Mix::Dynamics::Object* pObject );

		void RefreshObject( Mix::Dynamics::Object* pObject );

		btDynamicsWorld* Bullet_GetDynamicsWorldPtr( void ) const;

	public:
		virtual Boolean ContainsJoint( Mix::Dynamics::IJoint* pJoint ) const;
		virtual Boolean AddJoint( Mix::Dynamics::IJoint* pJoint, Boolean bCollisionDisabled );
		virtual Boolean RemoveJoint( Mix::Dynamics::IJoint* pJoint );
		virtual UInt32 GetJointCount( Mix::Dynamics::IJoint::TYPE type ) const;
		virtual UInt32 GetTotalJointCount( void ) const;

		virtual Boolean ContainsObject( Mix::Dynamics::IObject* pObject ) const;
		virtual Boolean AddObject( Mix::Dynamics::IObject* pObject );
		virtual Boolean RemoveObject( Mix::Dynamics::IObject* pObject );
		virtual UInt32 GetObjectCount( Mix::Dynamics::IObject::TYPE type ) const;
		virtual UInt32 GetTotalObjectCount( void ) const;

		virtual void SetGravity( const Mix::Vector3& gravity );
		virtual const Mix::Vector3& GetGravity( void ) const;

		virtual UInt32 Activate( void );
		virtual UInt32 Deactivate( Boolean bForce );

		virtual void Update( Float32 dt, Float32 baseDt );

		virtual Boolean TestRay(	const Mix::Vector3& fromWorldPos,
									const Mix::Vector3& toWorldPos,
									UInt16 filterGroup,
									UInt16 filterMask,
									Mix::Dynamics::IWorld::TestCallback* pCallback,
									Mix::Dynamics::IWorld::TEST_RESULT& result );

		virtual Boolean TestSweep(	Mix::Dynamics::IShape* pShape,
									const Mix::Vector3& moveFrom,
									const Mix::Vector3& moveTo,
									UInt16 filterGroup,
									UInt16 filterMask,
									Mix::Dynamics::IWorld::TestCallback* pCallback,
									Mix::Dynamics::IWorld::TEST_RESULT& result );

		virtual const Mix::Dynamics::DEBUG_WORLD_PROFILE& Debug_GetProfile( void ) const;
		virtual void Debug_Draw( Mix::Graphics::Utility::ILineArt* pLineArt );

	private:
		static const wchar_t* FAILED_ADDCONSTRAINT;
		static const wchar_t* FAILED_REMOVECONSTRAINT;
		static const wchar_t* FAILED_ADDOBJECT;
		static const wchar_t* FAILED_REMOVEOBJECT;
		static const wchar_t* FAILED_TESTSWEEP;
	};

}}
