#include "Mix/Class/Dynamics/Manager.h"

#include "Mix/Class/Dynamics/Utility.h"
#include "Mix/Class/Dynamics/SphereShape.h"
#include "Mix/Class/Dynamics/BoxShape.h"
#include "Mix/Class/Dynamics/ConeShape.h"
#include "Mix/Class/Dynamics/CapsuleShape.h"
#include "Mix/Class/Dynamics/CylinderShape.h"
#include "Mix/Class/Dynamics/ConvexHullShape.h"
#include "Mix/Class/Dynamics/CompoundShape.h"
#include "Mix/Class/Dynamics/PointJoint.h"
#include "Mix/Class/Dynamics/HingeJoint.h"
#include "Mix/Class/Dynamics/SliderJoint.h"
#include "Mix/Class/Dynamics/DofJoint.h"
#include "Mix/Class/Dynamics/BallJoint.h"
#include "Mix/Class/Dynamics/World.h"
#include "Mix/Class/Dynamics/StaticPlane.h"
#include "Mix/Class/Dynamics/StaticMesh.h"
#include "Mix/Class/Dynamics/RigidBody.h"
#include "Mix/Class/Dynamics/Sensor.h"
#include "Mix/Class/Dynamics/KinematicCharacter.h"
#include "Mix/Class/Dynamics/Vehicle.h"
#include "Mix/Class/Dynamics/ObjectDragger.h"

#include "LinearMath/btPoolAllocator.h"

namespace Mix{ namespace Dynamics{

////////////////////////////////////////////////////////////////////////////////////////////////////
// Manager
////////////////////////////////////////////////////////////////////////////////////////////////////

const wchar_t* Manager::FAILED_CREATESPHERESHAPE         = L"XtBAVFCv̍쐬Ɏs";
const wchar_t* Manager::FAILED_CREATEBOXSHAPE            = L"{bNXVFCv̍쐬Ɏs";
const wchar_t* Manager::FAILED_CREATECONESHAPE           = L"R[VFCv̍쐬Ɏs";
const wchar_t* Manager::FAILED_CREATECAPSULESHAPE        = L"JvZVFCv̍쐬Ɏs";
const wchar_t* Manager::FAILED_CREATECYLINDERSHAPE       = L"V_[VFCv̍쐬Ɏs";
const wchar_t* Manager::FAILED_CREATECONVEXHULLSHAPE     = L"RxbNXnVFCv̍쐬Ɏs";
const wchar_t* Manager::FAILED_CREATECOMPOUNDSHAPE       = L"RpEhVFCv̍쐬Ɏs";
const wchar_t* Manager::FAILED_CREATEWORLD               = L"[h̍쐬Ɏs";
const wchar_t* Manager::FAILED_CREATE_OBJECT_PICKER      = L"IuWFNgsbJ[̍쐬Ɏs";

const wchar_t* Manager::AXISNAME_TABLE[3] =
{
	L"X",
	L"Y",
	L"Z",
};

Manager* Manager::CreateInstance( void )
{
	return new Manager();
}

Manager::Manager( void )
{
}

Manager::~Manager( void )
{
}

Boolean Manager::CreateSphereShape( Float32 radius, Mix::Dynamics::ISphereShape** ppSphereShape, const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( radius <= 0.0f ) ||
		( ppSphereShape == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : radius[%f] ppSphereShape[%s]",
			FAILED_CREATESPHERESHAPE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			radius,
			MIX_LOG_PTR( ppSphereShape ) );

		return False;
	}

	Mix::Dynamics::SphereShape* pSphereShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pSphereShape = Mix::Dynamics::SphereShape::CreateInstance();
	if( pSphereShape == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATESPHERESHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pSphereShape->Initialize( radius ) == False )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATESPHERESHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		MIX_RELEASE( pSphereShape );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppSphereShape ) = pSphereShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateBoxShape( Float32 x, Float32 y, Float32 z, Mix::Dynamics::IBoxShape** ppBoxShape, const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( x <= 0.0f ) ||
		( y <= 0.0f ) ||
		( z <= 0.0f ) ||
		( ppBoxShape == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : x[%f] y[%f] z[%f] ppBoxShape[%s]",
			FAILED_CREATEBOXSHAPE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			x,
			y,
			z,
			MIX_LOG_PTR( ppBoxShape ) );

		return False;
	}

	return CreateBoxShape( Mix::Vector3( x, y, z ), ppBoxShape, pDebugName );
}

Boolean Manager::CreateBoxShape( const Mix::Vector3& size, Mix::Dynamics::IBoxShape** ppBoxShape, const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( size.x <= 0.0f ) ||
		( size.y <= 0.0f ) ||
		( size.z <= 0.0f ) ||
		( ppBoxShape == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : size.x[%f] size.y[%f] size.z[%f] ppBoxShape[%s]",
			FAILED_CREATEBOXSHAPE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			size.x,
			size.y,
			size.z,
			MIX_LOG_PTR( ppBoxShape ) );

		return False;
	}

	Mix::Dynamics::BoxShape* pBoxShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pBoxShape = Mix::Dynamics::BoxShape::CreateInstance( size );
	if( pBoxShape == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATEBOXSHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pBoxShape->Initialize() == False )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATEBOXSHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		MIX_RELEASE( pBoxShape );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppBoxShape ) = pBoxShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateConeShape( Mix::Dynamics::AXIS_TYPE axis, Float32 axisLength, Float32 radius, Mix::Dynamics::IConeShape** ppConeShape, const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( axisLength < MIX_FLOAT_EPSILON ) ||
		( radius < MIX_FLOAT_EPSILON ) ||
		( ppConeShape == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : axisLength[%f] radius[%f] ppConeShape[%s]",
			FAILED_CREATECONESHAPE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			axisLength,
			radius,
			MIX_LOG_PTR( ppConeShape ) );

		return False;
	}

	Mix::Dynamics::ConeShape* pConeShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pConeShape = Mix::Dynamics::ConeShape::CreateInstance();
	if( pConeShape == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATECONESHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pConeShape->Initialize( axis, axisLength, radius ) == False )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATECONESHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		MIX_RELEASE( pConeShape );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppConeShape ) = pConeShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateCapsuleShape( Mix::Dynamics::AXIS_TYPE axis, Float32 height, Float32 radius, Mix::Dynamics::ICapsuleShape** ppCapsuleShape, const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( height <= 0.0f ) ||
		( radius < 0.0f ) ||
		( ppCapsuleShape == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : height[%f] radius[%f] ppBoxShape[%s]",
			FAILED_CREATECAPSULESHAPE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			height,
			radius,
			MIX_LOG_PTR( ppCapsuleShape ) );

		return False;
	}

	Mix::Dynamics::CapsuleShape* pCapsuleShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pCapsuleShape = Mix::Dynamics::CapsuleShape::CreateInstance();
	if( pCapsuleShape == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATECAPSULESHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pCapsuleShape->Initialize( axis, height, radius ) == False )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATECAPSULESHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		MIX_RELEASE( pCapsuleShape );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppCapsuleShape ) = pCapsuleShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateCylinderShape( Mix::Dynamics::AXIS_TYPE axis, Float32 height, Float32 radius, Mix::Dynamics::ICylinderShape** ppCylinderShape, const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( height < MIX_FLOAT_EPSILON ) ||
		( radius < MIX_FLOAT_EPSILON ) ||
		( ppCylinderShape == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : height[%f] radius[%f] ppCylinderShape[%s]",
			FAILED_CREATECYLINDERSHAPE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			height,
			radius,
			MIX_LOG_PTR( ppCylinderShape ) );

		return False;
	}

	Mix::Dynamics::CylinderShape* pCylinderShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pCylinderShape = Mix::Dynamics::CylinderShape::CreateInstance();
	if( pCylinderShape == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATECYLINDERSHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pCylinderShape->Initialize( axis, height, radius ) == False )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATECYLINDERSHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		MIX_RELEASE( pCylinderShape );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppCylinderShape ) = pCylinderShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateConvexHullShape( const Mix::Dynamics::MESH& mesh, Mix::Dynamics::IConvexHullShape** ppConvexHullShape, const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ppConvexHullShape == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppConvexHullShape[%s]",
			FAILED_CREATECONVEXHULLSHAPE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( ppConvexHullShape ) );

		return False;
	}

	Mix::Dynamics::ConvexHullShape* pConvexHullShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pConvexHullShape = Mix::Dynamics::ConvexHullShape::CreateInstance();
	if( pConvexHullShape == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATECONVEXHULLSHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pConvexHullShape->Initialize( mesh ) == False )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATECONVEXHULLSHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		MIX_RELEASE( pConvexHullShape );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppConvexHullShape ) = pConvexHullShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateCompoundShape( Mix::Dynamics::ICompoundShape** ppCompoundShape, const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ppCompoundShape == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppCompoundShape[%s]",
			FAILED_CREATECOMPOUNDSHAPE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( ppCompoundShape ) );

		return False;
	}

	Mix::Dynamics::CompoundShape* pCompoundShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pCompoundShape = Mix::Dynamics::CompoundShape::CreateInstance();
	if( pCompoundShape == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATECOMPOUNDSHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pCompoundShape->Initialize() == False )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATECOMPOUNDSHAPE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		MIX_RELEASE( pCompoundShape );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppCompoundShape ) = pCompoundShape;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreatePointJoint(	Mix::Dynamics::IRigidBody* pRigidBodyA,
									const Mix::Vector3& pivotA,
									Mix::Dynamics::IPointJoint** ppPointJoint,
									const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( pRigidBodyA == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pRigidBodyA[%s]",
			PointJoint::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( pRigidBodyA ) );

		return False;
	}

	Mix::Dynamics::PointJoint* pJoint = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pJoint = Mix::Dynamics::PointJoint::CreateInstance( pRigidBodyA, pivotA );
	if( pJoint == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", PointJoint::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pJoint->Initialize( safeDebugName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pJoint );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppPointJoint ) = pJoint;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreatePointJoint(	Mix::Dynamics::IRigidBody* pRigidBodyA,
									Mix::Dynamics::IRigidBody* pRigidBodyB,
									const Mix::Vector3& pivotA,
									const Mix::Vector3& pivotB,
									Mix::Dynamics::IPointJoint** ppPointJoint,
									const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( pRigidBodyA == NULL ) ||
		( pRigidBodyB == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pRigidBodyA[%s] pRigidBodyB[%s]",
			PointJoint::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( pRigidBodyA ),
			MIX_LOG_PTR( pRigidBodyB ) );

		return False;
	}

	Mix::Dynamics::PointJoint* pJoint = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pJoint = Mix::Dynamics::PointJoint::CreateInstance( pRigidBodyA, pRigidBodyB, pivotA, pivotB );
	if( pJoint == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", PointJoint::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pJoint->Initialize( safeDebugName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pJoint );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppPointJoint ) = pJoint;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateHingeJoint(	Mix::Dynamics::IRigidBody* pRigidBodyA,
									const Mix::Vector3& pivotA,
									const Mix::Vector3& axis,
									Mix::Dynamics::IHingeJoint** ppHingeJoint,
									const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( pRigidBodyA == NULL ) ||
		( ppHingeJoint == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pRigidBodyA[%s] ppHingeJoint[%s]",
			HingeJoint::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( pRigidBodyA ),
			MIX_LOG_PTR( ppHingeJoint ) );

		return False;
	}

	Mix::Dynamics::HingeJoint* pJoint = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pJoint = Mix::Dynamics::HingeJoint::CreateInstance( pRigidBodyA, pivotA, axis );
	if( pJoint == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", HingeJoint::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pJoint->Initialize( safeDebugName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pJoint );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppHingeJoint ) = pJoint;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateHingeJoint(	Mix::Dynamics::IRigidBody* pRigidBodyA,
									Mix::Dynamics::IRigidBody* pRigidBodyB,
									const Mix::Vector3& pivotA,
									const Mix::Vector3& pivotB,
									const Mix::Vector3& axis,
									Mix::Dynamics::IHingeJoint** ppHingeJoint,
									const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( pRigidBodyA == NULL ) ||
		( pRigidBodyB == NULL ) ||
		( ppHingeJoint == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pRigidBodyA[%s] pRigidBodyB[%s] ppHingeJoint[%s]",
			HingeJoint::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( pRigidBodyA ),
			MIX_LOG_PTR( pRigidBodyB ),
			MIX_LOG_PTR( ppHingeJoint ) );

		return False;
	}

	Mix::Dynamics::HingeJoint* pJoint = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pJoint = Mix::Dynamics::HingeJoint::CreateInstance( pRigidBodyA, pRigidBodyB, pivotA, pivotB, axis );
	if( pJoint == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", HingeJoint::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pJoint->Initialize( safeDebugName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pJoint );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppHingeJoint ) = pJoint;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateSliderJoint(	Mix::Dynamics::IRigidBody* pRigidBodyA,
									const Mix::Quaternion& rotA,
									const Mix::Vector3& pivotA,
									Mix::Dynamics::ISliderJoint** ppSliderJoint,
									const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( pRigidBodyA == NULL ) ||
		( ppSliderJoint == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pRigidBodyA[%s] ppSliderJoint[%s]",
			SliderJoint::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( pRigidBodyA ),
			MIX_LOG_PTR( ppSliderJoint ) );

		return False;
	}

	Mix::Dynamics::SliderJoint* pJoint = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pJoint = Mix::Dynamics::SliderJoint::CreateInstance( pRigidBodyA, rotA, pivotA );
	if( pJoint == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", SliderJoint::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pJoint->Initialize( safeDebugName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pJoint );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppSliderJoint ) = pJoint;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateSliderJoint(	Mix::Dynamics::IRigidBody* pRigidBodyA,
									Mix::Dynamics::IRigidBody* pRigidBodyB,
									const Mix::Quaternion& rotA,
									const Mix::Quaternion& rotB,
									const Mix::Vector3& pivotA,
									const Mix::Vector3& pivotB,
									Mix::Dynamics::ISliderJoint** ppSliderJoint,
									const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( pRigidBodyA == NULL ) ||
		( pRigidBodyB == NULL ) ||
		( ppSliderJoint == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pRigidBodyA[%s] pRigidBodyB[%s] ppSliderJoint[%s]",
			SliderJoint::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( pRigidBodyA ),
			MIX_LOG_PTR( pRigidBodyB ),
			MIX_LOG_PTR( ppSliderJoint ) );

		return False;
	}

	Mix::Dynamics::SliderJoint* pJoint = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pJoint = Mix::Dynamics::SliderJoint::CreateInstance(	pRigidBodyA,
															pRigidBodyB,
															rotA,
															rotB,
															pivotA,
															pivotB );
	if( pJoint == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", SliderJoint::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pJoint->Initialize( safeDebugName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pJoint );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppSliderJoint ) = pJoint;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateDofJoint(	Mix::Dynamics::IRigidBody* pRigidBodyA,
									const Mix::Quaternion& rotA,
									const Mix::Vector3& pivotA,
									Mix::Dynamics::IDofJoint** ppDofJoint,
									const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( pRigidBodyA == NULL ) ||
		( ppDofJoint == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pRigidBodyA[%s] ppDofJoint[%s]",
			DofJoint::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( pRigidBodyA ),
			MIX_LOG_PTR( ppDofJoint ) );

		return False;
	}

	Mix::Dynamics::DofJoint* pJoint = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pJoint = Mix::Dynamics::DofJoint::CreateInstance( pRigidBodyA, rotA, pivotA );
	if( pJoint == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", DofJoint::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pJoint->Initialize( safeDebugName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pJoint );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppDofJoint ) = pJoint;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateDofJoint(	Mix::Dynamics::IRigidBody* pRigidBodyA,
									Mix::Dynamics::IRigidBody* pRigidBodyB,
									const Mix::Quaternion& rotA,
									const Mix::Quaternion& rotB,
									const Mix::Vector3& pivotA,
									const Mix::Vector3& pivotB,
									Mix::Dynamics::IDofJoint** ppDofJoint,
									const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( pRigidBodyA == NULL ) ||
		( pRigidBodyB == NULL ) ||
		( ppDofJoint == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pRigidBodyA[%s] pRigidBodyB[%s] ppDofJoint[%s]",
			DofJoint::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( pRigidBodyA ),
			MIX_LOG_PTR( pRigidBodyB ),
			MIX_LOG_PTR( ppDofJoint ) );

		return False;
	}

	Mix::Dynamics::DofJoint* pJoint = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pJoint = Mix::Dynamics::DofJoint::CreateInstance(	pRigidBodyA,
														pRigidBodyB,
														rotA,
														rotB,
														pivotA,
														pivotB );
	if( pJoint == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", DofJoint::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pJoint->Initialize( safeDebugName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pJoint );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppDofJoint ) = pJoint;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateBallJoint(	Mix::Dynamics::IRigidBody* pRigidBodyA,
									const Mix::Vector3& pivotA,
									const Mix::Vector3& twistAxis,
									const Mix::Vector3& swingAxis,
									Mix::Dynamics::IBallJoint** ppBallJoint,
									const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( pRigidBodyA == NULL ) ||
		( ppBallJoint == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pRigidBodyA[%s] ppBallJoint[%s]",
			BallJoint::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( pRigidBodyA ),
			MIX_LOG_PTR( ppBallJoint ) );

		return False;
	}

	Mix::Dynamics::BallJoint* pJoint = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pJoint = Mix::Dynamics::BallJoint::CreateInstance( pRigidBodyA, pivotA, twistAxis, swingAxis );
	if( pJoint == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", BallJoint::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pJoint->Initialize( safeDebugName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pJoint );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppBallJoint ) = pJoint;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateBallJoint(	Mix::Dynamics::IRigidBody* pRigidBodyA,
									Mix::Dynamics::IRigidBody* pRigidBodyB,
									const Mix::Vector3& pivotA,
									const Mix::Vector3& pivotB,
									const Mix::Vector3& twistAxis,
									const Mix::Vector3& swingAxis,
									Mix::Dynamics::IBallJoint** ppBallJoint,
									const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( pRigidBodyA == NULL ) ||
		( pRigidBodyB == NULL ) ||
		( ppBallJoint == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pRigidBodyA[%s] pRigidBodyB[%s] ppBallJoint[%s]",
			BallJoint::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( pRigidBodyA ),
			MIX_LOG_PTR( pRigidBodyB ),
			MIX_LOG_PTR( ppBallJoint ) );

		return False;
	}

	Mix::Dynamics::BallJoint* pJoint = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pJoint = Mix::Dynamics::BallJoint::CreateInstance( pRigidBodyA, pRigidBodyB, pivotA, pivotB, twistAxis, swingAxis );
	if( pJoint == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", BallJoint::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pJoint->Initialize( safeDebugName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pJoint );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppBallJoint ) = pJoint;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateWorld( Mix::Dynamics::IWorld** ppWorld, const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ppWorld == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppWorld[%s]",
			FAILED_CREATEWORLD,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			MIX_LOG_PTR( ppWorld ) );

		return False;
	}

	Mix::Dynamics::World* pWorld = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pWorld = Mix::Dynamics::World::CreateInstance( safeDebugName.GetConstPtr() );
	if( pWorld == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATEWORLD, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pWorld->Initialize() == False )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", FAILED_CREATEWORLD, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		MIX_RELEASE( pWorld );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppWorld ) = pWorld;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateStaticPlane( const Mix::Vector3& normal, Float32 constant, Mix::Dynamics::IStaticPlane** ppStaticPlane, const wchar_t* pDebugName )
{
	Mix::StringW safeName = MIX_SAFE_NAME( pDebugName );

	if( ppStaticPlane == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ppStaticPlane[%s]",
			StaticPlane::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			MIX_LOG_PTR( ppStaticPlane ) );

		return False;
	}

	Mix::Dynamics::StaticPlane* pStaticPlane = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pStaticPlane = Mix::Dynamics::StaticPlane::CreateInstance( normal, constant );
	if( pStaticPlane == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", StaticPlane::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pStaticPlane->Initialize( safeName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pStaticPlane );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppStaticPlane ) = pStaticPlane;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateStaticMesh( const Mix::Dynamics::MESH* meshes, UInt32 meshNum, UInt32 materialNum, Mix::Dynamics::IStaticMesh** ppStaticMesh, const wchar_t* pDebugName )
{
	Mix::StringW safeName = MIX_SAFE_NAME( pDebugName );

	if( ( meshes == NULL ) ||
		( meshNum == 0 ) ||
		( ppStaticMesh == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] meshes[%s] meshNum[%d] ppStaticMesh[%s]",
			StaticMesh::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeName.GetConstPtr(),
			MIX_LOG_PTR( meshes ),
			meshNum,
			MIX_LOG_PTR( ppStaticMesh ) );

		return False;
	}

	Mix::Dynamics::StaticMesh* pStaticMesh = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pStaticMesh = Mix::Dynamics::StaticMesh::CreateInstance();
	if( pStaticMesh == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", StaticMesh::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pStaticMesh->Initialize( meshes, meshNum, materialNum, safeName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pStaticMesh );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppStaticMesh ) = pStaticMesh;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateStaticMesh( Mix::Memory::IBuffer* pBuffer, UInt32 materialNum, Mix::Dynamics::IStaticMesh** ppStaticMesh, const wchar_t* pDebugName )
{
	Mix::StringW safeName = MIX_SAFE_NAME( pDebugName );

	if( ( pBuffer == NULL ) ||
		( ppStaticMesh == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] pBuffer[%s] ppStaticMesh[%s]",
			StaticMesh::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeName.GetConstPtr(),
			MIX_LOG_PTR( pBuffer ),
			MIX_LOG_PTR( ppStaticMesh ) );

		return False;
	}

	Mix::Dynamics::StaticMesh* pStaticMesh = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pStaticMesh = Mix::Dynamics::StaticMesh::CreateInstance();
	if( pStaticMesh == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", StaticMesh::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pStaticMesh->Initialize( pBuffer->GetPointer(), pBuffer->GetSize(), materialNum, safeName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pStaticMesh );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppStaticMesh ) = pStaticMesh;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateRigidBody( Float32 mass, Mix::Dynamics::IShape* pShape, Mix::Dynamics::IRigidBody** ppRigidBody, const wchar_t* pDebugName )
{
	Mix::StringW safeName = MIX_SAFE_NAME( pDebugName );

	if( ( pShape == NULL ) ||
		( ppRigidBody == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pShape[%s] ppRigidBody[%s]",
			RigidBody::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeName.GetConstPtr(),
			MIX_LOG_PTR( pShape ),
			MIX_LOG_PTR( ppRigidBody ) );

		return False;
	}

	Mix::Dynamics::RigidBody* pRigidBody = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pRigidBody = Mix::Dynamics::RigidBody::CreateInstance( mass, pShape );
	if( pRigidBody == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", RigidBody::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pRigidBody->Initialize( safeName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pRigidBody );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppRigidBody ) = pRigidBody;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateSensor(	Mix::Dynamics::IShape* pShape, 
								Mix::Dynamics::ISensor** ppSensor,
								const wchar_t* pDebugName )
{
	Mix::StringW safeName = MIX_SAFE_NAME( pDebugName );

	if( ( pShape == NULL ) ||
		( ppSensor == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pShape[%s] ppSensor[%s]",
			Sensor::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeName.GetConstPtr(),
			MIX_LOG_PTR( pShape ),
			MIX_LOG_PTR( ppSensor ) );

		return False;
	}

	Mix::Dynamics::Sensor* pSensor = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pSensor = Mix::Dynamics::Sensor::CreateInstance( pShape );
	if( pSensor == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", Sensor::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pSensor->Initialize( safeName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pSensor );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppSensor ) = pSensor;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateKinematicCharacter(	Float32 height,
											Float32 radius,
											Float32 stepHeight,
											Mix::Dynamics::IKinematicCharacter** ppKinematicCharacter,
											const wchar_t* pDebugName )
{
	Mix::StringW safeName = MIX_SAFE_NAME( pDebugName );

	if( ( height <= 0.0f ) ||
		( radius <= 0.0f ) ||
		( height < radius ) ||
		( stepHeight <= 0.0f ) ||
		( ppKinematicCharacter == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : ( height[%f > 0.0f] >= radius[%f > 0.0f] ) stepHeight[%f > 0.0f] ppKinematicCharacter[%s]",
			KinematicCharacter::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeName.GetConstPtr(),
			height,
			radius,
			stepHeight,
			MIX_LOG_PTR( ppKinematicCharacter ) );

		return False;
	}

	Mix::Dynamics::KinematicCharacter* pKinematicCharacter = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pKinematicCharacter = Mix::Dynamics::KinematicCharacter::CreateInstance();
	if( pKinematicCharacter == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", KinematicCharacter::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pKinematicCharacter->Initialize( height, radius, stepHeight, safeName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pKinematicCharacter );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppKinematicCharacter ) = pKinematicCharacter;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateVehicle( Float32 mass, Mix::Dynamics::IShape* pShape, const Mix::Dynamics::VEHICLE_WHEEL_DESC* wheels, UInt32 wheelNum, Mix::Dynamics::IVehicle** ppVehicle, const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( mass <= 0.0f ) ||
		( wheels == NULL ) ||
		( wheelNum == 0 ) ||
		( ppVehicle == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : mass[%f > 0.0f] wheels[%s] wheelNum[%d > 0] ppVehicle[%s]",
			Vehicle::FAILED_CREATE,
			Mix::STR_ILLEGALARG,
			safeDebugName.GetConstPtr(),
			mass,
			MIX_LOG_PTR( wheels ),
			wheelNum,
			MIX_LOG_PTR( ppVehicle ) );

		return False;
	}

	Mix::Dynamics::Vehicle* pVehicle = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pVehicle = Mix::Dynamics::Vehicle::CreateInstance( mass, pShape );
	if( pVehicle == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]", Vehicle::FAILED_CREATE, Mix::STR_OUTOFMEMORY, safeDebugName.GetConstPtr() );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// 
	////////////////////////////////////////////////////////////////////////////////////////////////////

	if( pVehicle->Initialize( wheels, wheelNum, safeDebugName.GetConstPtr() ) == False )
	{
		MIX_RELEASE( pVehicle );
		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppVehicle ) = pVehicle;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

Boolean Manager::CreateObjectDragger( Mix::Dynamics::IWorld* pWorld, Mix::Dynamics::IObjectDragger** ppObjectDragger, const wchar_t* pDebugName )
{
	Mix::StringW safeDebugName = MIX_SAFE_NAME( pDebugName );

	if( ( pWorld == NULL ) ||
		( ppObjectDragger == NULL ) )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s] : pWorld[%s] ppObjectDragger[%s]",
			Manager::FAILED_CREATE_OBJECT_PICKER,
			Mix::STR_ILLEGALARG,
			safeDebugName,
			MIX_LOG_PTR( pWorld ),
			MIX_LOG_PTR( ppObjectDragger ) );

		return False;
	}

	Mix::Dynamics::ObjectDragger* pObjectDragger = NULL;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// C^[tF[X̍쐬
	////////////////////////////////////////////////////////////////////////////////////////////////////

	pObjectDragger = Mix::Dynamics::ObjectDragger::CreateInstance( pWorld );
	if( pObjectDragger == NULL )
	{
		MIX_LOG_ERROR( L"%s : %s : DebugName[%s]",
			Manager::FAILED_CREATE_OBJECT_PICKER,
			Mix::STR_OUTOFMEMORY,
			safeDebugName );

		return False;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	// n
	////////////////////////////////////////////////////////////////////////////////////////////////////

	( *ppObjectDragger ) = pObjectDragger;

	////////////////////////////////////////////////////////////////////////////////////////////////////

	return True;
}

const Mix::Vector4& Manager::Debug_GetDrawColor( Mix::Dynamics::DEBUG_DRAW_COLOR_TYPE type ) const
{
	return Mix::Dynamics::Debug::GetColor( type );
}

void Manager::Debug_SetDrawColor( Mix::Dynamics::DEBUG_DRAW_COLOR_TYPE type, const Mix::Vector4& color )
{
	Mix::Dynamics::Debug::SetColor( type, color );
}

}}
