#include "Mix/Common.h"

#include "Shlwapi.h"
#include "Mix/Class/CpuID.h"
#include "Mix/Class/Engine.h"
#include "SFMT.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace Mix{

	/*
		CX^X̃nh
	*/

	HINSTANCE GetInstanceHandle( void )
	{
		return GetModuleHandle( NULL );
	}

	/*
		CPU̎擾
	*/

	const Mix::CPU_INFO& GetCPUInfo( void )
	{
		return Mix::CpuID::GetInfo();
	}

	/*
		GW̎擾
	*/

	Mix::IEngine* GetEnginePtr( void )
	{
		return Mix::Engine::GetInstance();
	}

	Boolean GetEngine( Mix::IEngine** ppEngine )
	{
		Mix::Engine* pEngine = Mix::Engine::GetInstance();

		if( pEngine != NULL )
		{
			MIX_ADD_REF( pEngine );
			( *ppEngine ) = pEngine;
		}
		else
		{
			return False;
		}

		return True;
	}

	/*
		EBhE̎擾
	*/

	Mix::IWindow* GetWindowPtr( void )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );
		return Mix::GetInternalEnginePtr()->GetWindowPtr();
	}

	Boolean GetWindow( Mix::IWindow** ppWindow )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );

		Mix::IWindow* pWindow = Mix::GetInternalEnginePtr()->GetWindowPtr();

		if( pWindow != NULL )
		{
			MIX_ADD_REF( pWindow );
			( *ppWindow ) = pWindow;
		}
		else
		{
			return False;
		}

		return True;
	}

	/*
		̎擾
	*/

	Int32 Sign( Int32 value )
	{
		return ( value >= 0 )? +1 : -1;
	}

	Float32 SignF( Float32 value )
	{
		return ( value >= 0.0f )? +1.0f : -1.0f;
	}

	/*
		_̕
	*/

	Float32 SqrtF( const Float32& value )
	{
		Float32 xHalf = 0.5f * value;
		Int32 tmp = 0x5F3759DF - ( *(Int32*)&value >> 1 ); //initial guess
		Float32 xRes  = *( Float32* )&tmp;

		xRes *= ( 1.5f - ( xHalf * xRes * xRes ) );
		//xRes *= ( 1.5f - ( xHalf * xRes * xRes ) );//RgAEgOƐxオ

		return xRes * value;
	}

	/*
		_̋t
	*/

	Float32 InvF( Float32 value )
	{
		UInt32* i = ( UInt32* )&value;

//		*i = 0x7F000000 - *i;
		*i = 0x7EEEEEEE - *i;

		return value;
	}

	/*
		̋t
	*/

	Float32 InvSqrtF( Float32 value )
	{
		Float32 x2 = value * 0.5f;
		Float32 y = value;
		Long32 i = *( Long32* )&y;

		i = 0x5F3759DF - ( i >> 1 );
		y = *( Float32* )&i;
		y = y * ( 1.5f - ( x2 * y * y ) );

		return y;
	}

	/*
		
	*/

	static sfmt_t g_SFMT;

	void InitRand( void )
	{
		::sfmt_init_gen_rand( &g_SFMT, static_cast<uint32_t>( ::time( NULL ) ) );
	}

	void InitRand( UInt32 seed )
	{
		::sfmt_init_gen_rand( &g_SFMT, seed );
	}

	void InitRand( UInt32* seeds, UInt32 numSeed )
	{
		::sfmt_init_by_array( &g_SFMT, seeds, static_cast<int>( numSeed ) );
	}

	UInt32 Rand( void )
	{
		return ::sfmt_genrand_uint32( &g_SFMT );
	}

	UInt32 Rand( Int32 maxRnd )
	{
		return Mix::Rand() % maxRnd;
	}

	UInt32 Rand( UInt32 r0, UInt32 r1 )
	{
		if( r0 == r1 )
		{
			return r0;
		}

		UInt32 minRnd;
		UInt32 maxRnd;

		if( r0 < r1 )
		{
			minRnd = r0;
			maxRnd = r1;
		}
		else
		{
			minRnd = r1;
			maxRnd = r0;
		}

		return minRnd + Mix::Rand() * ( maxRnd - minRnd );
	}

	Float32 RandF( void )
	{
		#define RND_INNER_LOOP() \
			if( coinFlips & 1 ) \
			{ \
				break; \
			} \
			coinFlips >>= 1; \
			ret -= 0x800000;

		UInt32 ret = 0x3F000000 | ( 0x7FFFFF & ( ( rand() << 8 ) ^ rand() ) );
		UInt16 coinFlips;

		for( ;; )
		{
			coinFlips = Mix::Rand();

			RND_INNER_LOOP(); RND_INNER_LOOP(); RND_INNER_LOOP();

			if( ( ret & 0x3F800000 ) == 0  )
			{
				return 0.0f;
			}

			RND_INNER_LOOP(); RND_INNER_LOOP(); RND_INNER_LOOP();
			RND_INNER_LOOP(); RND_INNER_LOOP(); RND_INNER_LOOP();
			RND_INNER_LOOP(); RND_INNER_LOOP(); RND_INNER_LOOP();
			RND_INNER_LOOP(); RND_INNER_LOOP(); RND_INNER_LOOP();
		}

		return *( ( Float32* )( &ret ) );
	}

	Float32 RandF( Float32 maxRnd )
	{
		return Mix::RandF() * maxRnd;
	}

	Float32 RandF( Float32 r0, Float32 r1 )
	{
		if( r0 == r1 )
		{
			return r0;
		}

		Float32 minRnd;
		Float32 maxRnd;

		if( r0 < r1 )
		{
			minRnd = r0;
			maxRnd = r1;
		}
		else
		{
			minRnd = r1;
			maxRnd = r0;
		}

		return minRnd + Mix::RandF() * ( maxRnd - minRnd );
	}

	/*
		_̕
	*/

	Float32 LerpF( Float32 a, Float32 b, Float32 t )
	{
		return ( a * ( 1.0f - t ) ) + ( b * t );
	}

	/*
		w肳ꂽxNgQ{̐ȃxNg߂
	*/

	void PlaneSpace( const Mix::Vector3& n, Mix::Vector3& p, Mix::Vector3& q )
	{
		static Float32 SQRT12 = 0.7071067811865475244008443621048490f;

		if( ::fabs( n.z ) > SQRT12 )
		{
			Float32 a = n.y * n.y + n.z * n.z;
			Float32 k = MIX_FLOAT_RECIPROCAL( ::sqrtf( a ) );

			p.x = 0.0f;
			p.y = -n.z * k;
			p.z = n.y * k;

			q.x = a * k;
			q.y = -n.x * p.z;
			q.z = n.x * p.y;
		}
		else
		{
			Float32 a = n.x * n.x + n.y * n.y;
			Float32 k = MIX_FLOAT_RECIPROCAL( ::sqrtf( a ) );

			p.x = -n.y * k;
			p.y = n.x * k;
			p.z = 0.0f;

			q.x = -n.z * p.y;
			q.y = n.z * p.x;
			q.z = a * k;
		}
	}

	/*
		w肳ꂽsQ{̃xNgŒZ̉]߂
	*/

	Mix::Quaternion ShortestArc( const Mix::Vector3& v0, const Mix::Vector3& v1 )
	{
		Mix::Vector3 cv = Mix::Vector3::Cross( v0, v1 );
		Float32 d = Mix::Vector3::Dot( v0, v1 );

		Mix::Quaternion ret;

		if( d < ( -1.0f + MIX_FLOAT_EPSILON ) )
		{
			Mix::Vector3 av;
			Mix::Vector3 dummy;

			Mix::PlaneSpace( v0, av, dummy );

			ret.x = av.x;
			ret.y = av.y;
			ret.z = av.z;
			ret.w = 0.0f;
		}
		else
		{
			Float32 s = ::sqrtf( ( 1.0f + d ) * 2.0f );
			Float32 rs = MIX_FLOAT_RECIPROCAL( s );

			ret.x = cv.x * rs;
			ret.y = cv.y * rs;
			ret.z = cv.z * rs;
			ret.w = s * 0.5f;
		}

		return ret;
	}

	/*
		s4x4 : [hWXN[Wɕϊ
	*/

	Mix::Vector3 Project( const Mix::Matrix4x4& viewMat, const Mix::Matrix4x4& projMat, const Mix::Vector3& worldPos, const Mix::Vector2& screenSize )
	{
		Mix::Matrix4x4 unViewport;
		Mix::Matrix4x4 mat;
		Mix::Vector4 pos;
		Mix::Vector4 result;

		unViewport.m00 = ( screenSize.x * 0.5f );
		unViewport.m11 = -( screenSize.y * 0.5f );
		unViewport.m30 = unViewport.m00;
		unViewport.m31 = ( screenSize.y * 0.5f );

		mat = viewMat;
		mat *= projMat;

		pos = mat * worldPos;
		pos /= pos.w;

		result = unViewport * pos;

		return Mix::Vector3( result.x, result.y, result.z );
	}

	/*
		s4x4 : XN[W[hWɕϊ
	*/

	Mix::Vector3 Unproject( const Mix::Matrix4x4& viewMat, const Mix::Matrix4x4& projMat, const Mix::Vector3& screenPos, const Mix::Vector2& screenSize )
	{
		Mix::Matrix4x4 invViewMat;
		Mix::Matrix4x4 invProjMat;
		Mix::Matrix4x4 unViewport;
		Mix::Matrix4x4 temp;

		invViewMat = viewMat.ToInverse();
		invProjMat = projMat.ToInverse();

		unViewport.m00 = ( screenSize.x * 0.5f );
		unViewport.m11 = -( screenSize.y * 0.5f );
		unViewport.m30 = unViewport.m00;
		unViewport.m31 = ( screenSize.y * 0.5f );

		unViewport.Inverse();

		temp = unViewport;
		temp *= invProjMat;
		temp *= invViewMat;

		return temp * screenPos;
	}

	/*
		pX
	*/

	Boolean CombinePath( const wchar_t* pBaseDirPath, const wchar_t* pRelPath, Mix::StringW& outPath )
	{
		if( ( pBaseDirPath == NULL ) ||
			( pRelPath == NULL ) )
		{
			return False;
		}

		wchar_t tempPath[_MAX_PATH];

		Mix::Memory::Zero( &tempPath, sizeof( tempPath ) );

		if( PathCombineW( tempPath, pBaseDirPath, pRelPath ) != NULL )
		{
			outPath = tempPath;
		}
		else
		{
			return False;
		}

		return True;
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::File
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace Mix{ namespace File{

	Mix::File::IManager* GetManagerPtr( void )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );
		return Mix::GetInternalEnginePtr()->GetFileManagerPtr();
	}

	Boolean GetManager( Mix::File::IManager** ppManager )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );

		Mix::File::IManager* pManager = Mix::GetInternalEnginePtr()->GetFileManagerPtr();

		if( pManager != NULL )
		{
			MIX_ADD_REF( pManager );
			( *ppManager ) = pManager;
		}
		else
		{
			return False;
		}

		return True;
	}

}}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Dynamics
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace Mix{ namespace Dynamics{

	Mix::Dynamics::IManager* GetManagerPtr( void )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );
		return Mix::GetInternalEnginePtr()->GetDynamicsManagerPtr();
	}

	Boolean GetManager( Mix::Dynamics::IManager** ppManager )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );

		Mix::Dynamics::IManager* pManager = Mix::GetInternalEnginePtr()->GetDynamicsManagerPtr();

		if( pManager != NULL )
		{
			MIX_ADD_REF( pManager );
			( *ppManager ) = pManager;
		}
		else
		{
			return False;
		}

		return True;
	}

}}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Graphics
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace Mix{ namespace Graphics{

	Mix::Graphics::IManager* GetManagerPtr( void )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );
		return Mix::GetInternalEnginePtr()->GetGraphicsManagerPtr();
	}

	Boolean GetManager( Mix::Graphics::IManager** ppManager )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );

		Mix::Graphics::IManager* pManager = Mix::GetInternalEnginePtr()->GetGraphicsManagerPtr();

		if( pManager != NULL )
		{
			MIX_ADD_REF( pManager );
			( *ppManager ) = pManager;
		}
		else
		{
			return False;
		}

		return True;
	}

}}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Sound
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace Mix{ namespace Sound{

	Mix::Sound::IManager* GetManagerPtr( void )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );
		return Mix::GetInternalEnginePtr()->GetSoundManagerPtr();
	}

	Boolean GetManager( Mix::Sound::IManager** ppManager )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );

		Mix::Sound::IManager* pManager = Mix::GetInternalEnginePtr()->GetSoundManagerPtr();

		if( pManager != NULL )
		{
			MIX_ADD_REF( pManager );
			( *ppManager ) = pManager;
		}
		else
		{
			return False;
		}

		return True;
	}

}}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Input
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace Mix{ namespace Input{

	Mix::Input::IManager* GetManagerPtr( void )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );
		return Mix::GetInternalEnginePtr()->GetInputManagerPtr();
	}

	Boolean GetManager( Mix::Input::IManager** ppManager )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );

		Mix::Input::IManager* pManager = Mix::GetInternalEnginePtr()->GetInputManagerPtr();

		if( pManager != NULL )
		{
			MIX_ADD_REF( pManager );
			( *ppManager ) = pManager;
		}
		else
		{
			return False;
		}

		return True;
	}

}}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Input
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace Mix{ namespace Network{

	Mix::Network::IManager* GetManagerPtr( void )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );
		return Mix::GetInternalEnginePtr()->GetNetworkManagerPtr();
	}

	Boolean GetManager( Mix::Network::IManager** ppManager )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );

		Mix::Network::IManager* pManager = Mix::GetInternalEnginePtr()->GetNetworkManagerPtr();

		if( pManager != NULL )
		{
			MIX_ADD_REF( pManager );
			( *ppManager ) = pManager;
		}
		else
		{
			return False;
		}

		return True;
	}

}}

////////////////////////////////////////////////////////////////////////////////////////////////////
// Mix::Scene
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace Mix{ namespace Scene{

	/*
		u[\
	*/

	static const Int32 BLUR_FP_RES_I = 10000;
	static const Float32 BLUR_FP_RES_F = 10000.0f;
	static const Float32 BLUR_FP_RES_F_INV = 0.0001f;

	BLUR::BLUR( void ) :
	type( Mix::Scene::BLUR::T_NONE ),
	data( 0 )
	{
	}

	BLUR::TYPE BLUR::GetType( void ) const
	{
		return type;
	}

	Int32 BLUR::GetIntValue( void ) const
	{
		return data / BLUR_FP_RES_I;
	}

	Float32 BLUR::GetFloatValue( void ) const
	{
		return static_cast<Float32>( data ) * BLUR_FP_RES_F_INV;
	}

	Boolean BLUR::IsValid( void ) const
	{
		return ( type != Mix::Scene::BLUR::T_NONE );
	}

	const Mix::Scene::BLUR& BLUR::NONE( void )
	{
		static const Mix::Scene::BLUR none;
		return none;
	}

	Mix::Scene::BLUR BLUR::KS_3x3( void )
	{
		Mix::Scene::BLUR ret;

		ret.type = Mix::Scene::BLUR::T_KS_3x3;
		ret.data = Mix::Scene::KS_BLUR_MIN_ITERATION * BLUR_FP_RES_I;

		return ret;
	}

	Mix::Scene::BLUR BLUR::KS_3x3( Int32 iteration )
	{
		Mix::Scene::BLUR ret;

		ret.type = Mix::Scene::BLUR::T_KS_3x3;
		ret.data = MIX_CLAMP( iteration, Mix::Scene::KS_BLUR_MIN_ITERATION, Mix::Scene::KS_BLUR_MAX_ITERATION ) * BLUR_FP_RES_I;

		return ret;
	}

	Mix::Scene::BLUR BLUR::KS_5x5( void )
	{
		Mix::Scene::BLUR ret;

		ret.type = Mix::Scene::BLUR::T_KS_5x5;
		ret.data = Mix::Scene::KS_BLUR_MIN_ITERATION * BLUR_FP_RES_I;

		return ret;
	}

	Mix::Scene::BLUR BLUR::KS_5x5( Int32 iteration )
	{
		Mix::Scene::BLUR ret;

		ret.type = Mix::Scene::BLUR::T_KS_5x5;
		ret.data = MIX_CLAMP( iteration, Mix::Scene::KS_BLUR_MIN_ITERATION, Mix::Scene::KS_BLUR_MAX_ITERATION ) * BLUR_FP_RES_I;

		return ret;
	}

	Mix::Scene::BLUR BLUR::GAUSSIAN( void )
	{
		Mix::Scene::BLUR ret;

		ret.type = Mix::Scene::BLUR::T_GAUSSIAN;
		ret.data = static_cast<Int32>( Mix::Scene::GAUSSIAN_BLUR_MIN_DISPERSION * BLUR_FP_RES_F );

		return ret;
	}

	Mix::Scene::BLUR BLUR::GAUSSIAN( Float32 dispersion )
	{
		Mix::Scene::BLUR ret;

		ret.type = Mix::Scene::BLUR::T_GAUSSIAN;
		ret.data = static_cast<Int32>( MIX_CLAMP( dispersion, Mix::Scene::GAUSSIAN_BLUR_MIN_DISPERSION, Mix::Scene::GAUSSIAN_BLUR_MAX_DISPERSION ) * BLUR_FP_RES_F );

		return ret;
	}

	Mix::Scene::BLUR BLUR::GAUSSIAN_EX( void )
	{
		Mix::Scene::BLUR ret;

		ret.type = Mix::Scene::BLUR::T_GAUSSIAN_EX;
		ret.data = Mix::Scene::GAUSSIAN_BLUR_EX_MIN_RADIUS * BLUR_FP_RES_I;

		return ret;
	}

	Mix::Scene::BLUR BLUR::GAUSSIAN_EX( Int32 radius )
	{
		Mix::Scene::BLUR ret;

		ret.type = Mix::Scene::BLUR::T_GAUSSIAN_EX;
		ret.data = MIX_CLAMP( radius, Mix::Scene::GAUSSIAN_BLUR_EX_MIN_RADIUS, Mix::Scene::GAUSSIAN_BLUR_EX_MAX_RADIUS ) * BLUR_FP_RES_I;

		return ret;
	}

	/*
		[Vnh\
	*/

	MOTION_HANDLE::MOTION_HANDLE( void ) :
	ref0( NULL ),
	ref1( NULL )
	{
	}

	MOTION_HANDLE::MOTION_HANDLE( Int32 value ) :
	ref0( NULL ),
	ref1( NULL )
	{
	}

	MOTION_HANDLE::MOTION_HANDLE( const MOTION_HANDLE& handle ) :
	ref0( handle.ref0 ),
	ref1( handle.ref1 )
	{
	}

	MOTION_HANDLE::MOTION_HANDLE( const void* r0, const void* r1 ) :
	ref0( r0 ),
	ref1( r1 )
	{
	}

	MOTION_HANDLE& MOTION_HANDLE::operator = ( Int32 value )
	{
		*this = MOTION_HANDLE();

		return *this;
	}

	MOTION_HANDLE& MOTION_HANDLE::operator = ( const MOTION_HANDLE& handle )
	{
		ref0 = handle.ref0;
		ref1 = handle.ref1;

		return *this;
	}

	Boolean MOTION_HANDLE::operator == ( Int32 value ) const
	{
		if( ( ref0 != NULL ) ||
			( ref1 != NULL ) )
		{
			return False;
		}

		return True;
	}

	Boolean MOTION_HANDLE::operator != ( Int32 value ) const
	{
		if( ( ref0 != NULL ) ||
			( ref1 != NULL ) )
		{
			return True;
		}

		return False;
	}

	Boolean MOTION_HANDLE::operator != ( const MOTION_HANDLE& handle ) const
	{
		if( ( ref0 != handle.ref0 ) ||
			( ref1 != handle.ref1 ) )
		{
			return True;
		}

		return False;
	}

	Boolean MOTION_HANDLE::operator == ( const MOTION_HANDLE& handle ) const
	{
		if( ( ref0 != handle.ref0 ) ||
			( ref1 != handle.ref1 ) )
		{
			return False;
		}

		return True;
	}

	bool MOTION_HANDLE::operator < ( const MOTION_HANDLE& handle ) const
	{
		if( ref0 < handle.ref0 )
		{
			return true;
		}
		else if( ref0 > handle.ref0 )
		{
			return false;
		}

		if( ref1 < handle.ref1 )
		{
			return true;
		}
		else if( ref1 > handle.ref1 )
		{
			return false;
		}

		return false;
	}

	/*
		֐
	*/

	Mix::Scene::IManager* GetManagerPtr( void )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );
		return Mix::GetInternalEnginePtr()->GetSceneManagerPtr();
	}

	Boolean GetManager( Mix::Scene::IManager** ppManager )
	{
		MIX_ASSERT( Mix::GetInternalEnginePtr() != NULL );

		Mix::Scene::IManager* pManager = Mix::GetInternalEnginePtr()->GetSceneManagerPtr();

		if( pManager != NULL )
		{
			MIX_ADD_REF( pManager );
			( *ppManager ) = pManager;
		}
		else
		{
			return False;
		}

		return True;
	}

}}
