﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

using Vintagestory.API.Client;
using Vintagestory.API.Common;
using Vintagestory.API.Config;
using Vintagestory.API.MathTools;


namespace CarbonCopy
{
	public class CarbonCopyCommand : ClientChatCommand
	{
		private const int _highlightSlot = 42;
		private const string _exportPath = @"Exports";
		private const string _name = @"export";

		private ICoreClientAPI ClientAPI { get; set; }
		private ILogger Logger { get; set; }

		private BlockPos start;
		private BlockPos end;
		private bool validSelection;
		private List<BlockPos> markedBlocks = new List<BlockPos>( );
		private List<int> colorList = new List<int> { ColorUtil.ToRgba(180, 0, 195, 0) };

		internal ModInfo InternalVersion
		{
			get
			{
				var gameMod = ClientAPI.ModLoader.GetMod(@"survival");//TODO: Figure out which; Creative / Survival ?
			return gameMod.Info;
			}
		}


		public CarbonCopyCommand(ICoreClientAPI clientAPI)
		{		
		this.Command = _name;
		this.Syntax = @"mark[ start/end(x/y/z) ], save [name], clear; dump [items/blocks] [domain?]";
		this.Description = @"Export carbon-copies of block selections locally.(as json export schematic) - also Block info...";

		ClientAPI = clientAPI;
		Logger = clientAPI.Logger;
		}



		/* .export 
		 * mark
		 * mark start
		 * mark end
		 * mark start x+1
		 * mark end y-1
		 * mark start z+20
		 * mark clear
		 * save {NAME}
		 * dump blocks/items
		 *
		 */

		public override void CallHandler(IPlayer player, int groupId, CmdArgs args)
		{
		if (args.Length > 0) {
		string command = args.PopWord( );

		switch (command) {

		case "mark":
			Process_MarkSubcommand(args);
			break;

		case "save":
			if (validSelection) {
			var filename = args.PopWord("export_" + Math.Round(DateTime.UtcNow.TimeOfDay.TotalSeconds));
			Process_SaveCommand(filename);
			}
			break;

		case "clear":
			Process_ClearCommand( );
			break;

		case "dump":
			string ofType = args.PopWord("blocks");
			string domainFilter = null;
			if (args.Length == 1) 
			{
			domainFilter = args.PopWord( );
			}

			if (ofType == "items") { Dump_ItemList(domainFilter); } else { Dump_BlockList(domainFilter); }			
			break;

		default:
			ClientAPI.ShowChatMessage($"Unrecognised command: '{command}' - try 'mark/save/clear'...");
			break;
		}
		}
		else {
				ClientAPI.ShowChatMessage("Needs a command (parameter): mark/save/clear/dump");
		}
		}

		private void Process_MarkSubcommand(CmdArgs args)
		{
		if (args.Length > 0) {
		string subCommand = args.PopWord("?").ToLowerInvariant( );


		switch (subCommand) {
		case "start":
			if (this.start == null) {
			this.start = ClientAPI.World.Player.Entity.Pos.AsBlockPos.Copy( );
			}
			BendBlockPosition(this.start, args);

			ClientAPI.ShowChatMessage(string.Format("Start@ ({0}).", start));
			break;

		case "end":
			if (this.end == null) {
			this.end = ClientAPI.World.Player.Entity.Pos.AsBlockPos.Copy( );
			}
			BendBlockPosition(this.end, args);

			ClientAPI.ShowChatMessage(string.Format("End@ ({0}).", end));
			break;

		default:
			ClientAPI.ShowChatMessage("Mark synax: start or end");
			break;
		}

		this.validSelection = false;

		if (this.start != null &&
		this.end != null &&
		ClientAPI.World.BulkBlockAccessor.IsValidPos(this.start)
		&& ClientAPI.World.BulkBlockAccessor.IsValidPos(this.end)		
		   ) {

		//Calc. Vol:
		var size = start.DistanceTo(end);

		if (size > 1) {
		this.validSelection = true;
		this.markedBlocks.Clear( );

		ClientAPI.World.BulkBlockAccessor.WalkBlocks(start, end, addToMarked);

		ClientAPI.World.HighlightBlocks(ClientAPI.World.Player, _highlightSlot, this.markedBlocks, colorList, EnumHighlightBlocksMode.Absolute, EnumHighlightShape.Arbitrary);

		int X = Math.Abs(start.X - end.X) + 1;
		int Y = Math.Abs(start.Y - end.Y) + 1;
		int Z = Math.Abs(start.Z - end.Z) + 1;

		ClientAPI.ShowChatMessage($"Marked (X {X:#},Y {Y:#} ,Z {Z:#} ) {size:f1} diagonal = {markedBlocks.Count} cells");
		}
		else {
		ClientAPI.ShowChatMessage("Marked area: volume too small");
		}
		}		
		}
		else {
		ClientAPI.ShowChatMessage("(Needs args) Mark: start / end (x,y,z +-1#)");
		}

		}

		private void BendBlockPosition(BlockPos position, CmdArgs args)
		{
		if (args.Length == 0) return;
		char? axis = args.PopChar();
		int? offset = args.PopInt();

		if (axis.HasValue && offset.HasValue) {		
		switch (Char.ToLowerInvariant(axis.Value)) {

		case 'x':			
			position.Add(dx: offset.Value, dy: 0, dz: 0);
			ClientAPI.ShowChatMessage(string.Format("altered {0}, offset {1}", axis.Value, offset.Value));
			break;

		case 'y':			
			position.Add(dx: 0, dy: offset.Value, dz: 0);
			ClientAPI.ShowChatMessage(string.Format("altered {0}, offset {1}", axis.Value, offset.Value));
			break;

		case 'z':			
			position.Add(dx: 0, dy: 0, dz: offset.Value);
			ClientAPI.ShowChatMessage(string.Format("altered {0}, offset {1}", axis.Value, offset.Value));
			break;

		default:
			ClientAPI.ShowChatMessage("Can't apply offset; not in \"x, y, z  +1#\" form...");
			break;
		}
		}
		}

		private void addToMarked(Block aBlock, BlockPos aPos)
		{
		this.markedBlocks.Add(aPos.Copy( ));
		}

		private void Process_SaveCommand(string filename)
		{
		if (validSelection) {
		BlockSchematic blockExport = new BlockSchematic( );
		//Random Start / End

		//'Lower'-> South + Westmost corner of same volume
		BlockPos seCorner = new BlockPos(Math.Max(start.X, end.X), Math.Min(start.Y, end.Y), Math.Max(start.Z, end.Z));
		//'Upper'-> North + Eastmost corner of same volume		
		BlockPos nwCorner = new BlockPos(Math.Min(start.X, end.X), Math.Max(start.Y, end.Y), Math.Min(start.Z, end.Z));

		//diagonal 'left'; offset into 'upper' - adds corner
		nwCorner.Up();

		seCorner.X += 1;
		seCorner.Z += 1;	

		blockExport.Init(ClientAPI.World.BlockAccessor);
		#if DEBUG
		Logger.Debug("Start: {0}, End:{1}", start, end);				
		Logger.Debug("S.E C: {0}, N.W:{1} (offset)", seCorner, nwCorner);
		#endif
		blockExport.AddArea(ClientAPI.World, seCorner, nwCorner);//Off by 1 in XYZ

		if (blockExport.Pack(ClientAPI.World, seCorner))//?
		{
		ClientAPI.ShowChatMessage(string.Format("Export packed OK: {0} BlockIDs, {1}, Entities, {2} BlockEntities; ({3} total blocks)", blockExport.BlockIds.Count, blockExport.Entities.Count, blockExport.BlockEntities.Count, blockExport.Indices.Count));

		filename = Path.Combine(ClientAPI.GetOrCreateDataPath(_exportPath), filename);

		blockExport.Save(filename);

		Process_ClearCommand( );
		}
		else {
		ClientAPI.ShowChatMessage("Error; can't pack for export...");
		}
		}
		}


		private void Process_ClearCommand( )
		{
		this.start = null;
		this.end = null;
		this.validSelection = false;
		this.markedBlocks.Clear( );
		ClientAPI.World.HighlightBlocks(ClientAPI.World.Player, _highlightSlot, this.markedBlocks, EnumHighlightBlocksMode.Absolute, EnumHighlightShape.Arbitrary);

		ClientAPI.ShowChatMessage("Mark(s) Cleared.");
		}

		private void Dump_BlockList(string domainFilter )
		{
		var filename = Path.Combine(ClientAPI.GetOrCreateDataPath(_exportPath), "vs_block_list_export_" + InternalVersion.Version + ".tsv");

		IEnumerable<Block> bQuery;
		if (string.IsNullOrWhiteSpace(domainFilter)) {
		bQuery = ClientAPI.World.Blocks.Where(blk => blk.Code != null);
		}
		else {
		bQuery = ClientAPI.World.Blocks.Where(blk => blk.Code != null && blk.Code.Domain.Equals(domainFilter, StringComparison.OrdinalIgnoreCase));
		filename = Path.Combine(ClientAPI.GetOrCreateDataPath(_exportPath), "vs_block_list_export_" + InternalVersion.Version+ "_["+domainFilter +"]" + ".tsv");
		}

		using (StreamWriter reporter = new StreamWriter(filename, false)) {

		reporter.WriteLine("MC Block Name\tVS Block Name\tVS ID#\tDomain\tMaterial\t");
		Logger.VerboseDebug("Dumping {0} Blocks", ClientAPI.World.Blocks.Count);

		

		foreach (var block in bQuery) {
		if (block.Id != 0 && !block.IsMissing && block.Code.Path != "unknown" ) {
		reporter.WriteLine($"____\t{block.Code.Path}\t{block.BlockId}\t{block.Code.Domain}\t{block.BlockMaterial}");
		}

		}

		reporter.Flush( );
		}

		ClientAPI.ShowChatMessage("Created Blocks List....");
		}

		private void Dump_ItemList(string domainFilter )
		{
		var filename = Path.Combine(ClientAPI.GetOrCreateDataPath(_exportPath), "vs_item_list_export_" + InternalVersion.Version + ".tsv");

		IEnumerable<Item> iQuery;
		if (string.IsNullOrWhiteSpace(domainFilter)) {
		iQuery = ClientAPI.World.Items.Where(itm => itm.Code != null);
		}
		else {
		iQuery = ClientAPI.World.Items.Where(itm => itm.Code != null && itm.Code.Domain.Equals(domainFilter, StringComparison.OrdinalIgnoreCase));
		filename = Path.Combine(ClientAPI.GetOrCreateDataPath(_exportPath), "vs_item_list_export_" + InternalVersion.Version + "_[" + domainFilter + "]" + ".tsv");
		}

		using (StreamWriter reporter = new StreamWriter(filename, false)) {

		reporter.WriteLine("MC Item Name\tVS Item Name\tVS ID#\tDomain\tStorage Flags\tMax Stack Size\tC.I. Tabs\t");
		Logger.VerboseDebug("Dumping {0} Items", ClientAPI.World.Items.Count);

		

		foreach (var item in iQuery ) {
		if (item.Id != 0 && !item.IsMissing && item.Code.Path != "unknown") {
		reporter.WriteLine($"____\t{item.Code.Path}\t{item.ItemId}\t{item.Code.Domain}\t{String.Join(",", item.StorageFlags)}\t{item.MaxStackSize}\t{(item.CreativeInventoryTabs != null ? String.Join(",",item.CreativeInventoryTabs): " ")}");
		}

		}

		reporter.Flush( );
		}

		ClientAPI.ShowChatMessage("Created Items List....");
		}
	}
}

