﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Collections.ObjectModel;

using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Web.UI;

using Hjg.Pngcs;
using Hjg.Pngcs.Chunks;

using Vintagestory.API.Common;
using Vintagestory.API.MathTools;



namespace Automap
{
	public partial class AutomapMod
	{
		private Thread cartographer_thread;

		private const string _mapPath = @"Maps";
		private const string _chunkPath = @"Chunks";
		private const string _domain = @"automap";

		private const string chunkFile_filter = @"*_*.png";
		private static Regex chunkShardRegex = new Regex(@"(?<X>[\d]+)_(?<Z>[\d]+).png", RegexOptions.Singleline);

		private ConcurrentDictionary<Vec2i, uint> columnCounter = new ConcurrentDictionary<Vec2i, uint>(3, 150 );
		private ColumnsMetadata chunkTopMetadata;

		private PointsOfInterest POIs;
		private Dictionary<int, Designator> BlockID_Designators;


		private Vec2i startChunkColumn;
		private uint lastUpdate;

		private string path;
		private IAsset stylesFile;

		#region Internals
		private void StartAutomap( )
		{
		path = ClientAPI.GetOrCreateDataPath(_mapPath);
		path = ClientAPI.GetOrCreateDataPath(Path.Combine(path, "World_" + ClientAPI.World.Seed));//Add name of World too...'ServerApi.WorldManager.CurrentWorldName'

		stylesFile = ClientAPI.World.AssetManager.Get(new AssetLocation(_domain, "config/automap_format.css"));
		Logger.VerboseDebug("CSS loaded: {0} size: {1}",stylesFile.IsLoaded() ,stylesFile.ToText( ).Length);

		Prefill_POI_Designators( );
		startChunkColumn = new Vec2i((ClientAPI.World.Player.Entity.LocalPos.AsBlockPos.X / ClientAPI.World.BlockAccessor.ChunkSize), (ClientAPI.World.Player.Entity.LocalPos.AsBlockPos.Z / ClientAPI.World.BlockAccessor.ChunkSize));
		chunkTopMetadata = new ColumnsMetadata(startChunkColumn);

		Logger.Notification("AUTOMAP Start {0}", startChunkColumn);
		Reload_Metadata( );

		ClientAPI.Event.ChunkDirty += ChunkAChanging;

		cartographer_thread = new Thread(Cartographer);
		cartographer_thread.Name = "Cartographer";
		cartographer_thread.Priority = ThreadPriority.Lowest;
		cartographer_thread.IsBackground = true;

		ClientAPI.Event.RegisterGameTickListener(AwakenCartographer, 6000);
		}

		private void ChunkAChanging(Vec3i chunkCoord, IWorldChunk chunk, EnumChunkDirtyReason reason)
		{			
		Vec2i topPosition = new Vec2i(chunkCoord.X, chunkCoord.Z);

			columnCounter.AddOrUpdate(topPosition, 1, (key, colAct) => colAct + 1);
		}

		private void AwakenCartographer(float delayed)
		{

		if (ClientAPI.IsGamePaused != false || ClientAPI.IsShuttingDown != true) {
		#if DEBUG
		Logger.VerboseDebug("Cartographer re-trigger from [{0}]", cartographer_thread.ThreadState);
		#endif

		if (cartographer_thread.ThreadState.HasFlag(ThreadState.Unstarted)) {
		cartographer_thread.Start( );
		}
		else if (cartographer_thread.ThreadState.HasFlag(ThreadState.WaitSleepJoin)) {
		//Time to (re)write chunk shards
		cartographer_thread.Interrupt( );
		}
		ClientAPI.TriggerChatMessage($"Automap {lastUpdate} changes - MAX (N:{chunkTopMetadata.North_mostChunk},S:{chunkTopMetadata.South_mostChunk},E:{chunkTopMetadata.East_mostChunk}, W:{chunkTopMetadata.West_mostChunk} - TOTAL: {chunkTopMetadata.Count})");
		}

		}


		private void Cartographer( )
		{
	wake:
		Logger.VerboseDebug("Cartographer thread awoken");

		try {
		uint ejectedItem = 0;
		uint updatedChunks = 0;
						

		//-- Should dodge enumerator changing underfoot....at a cost.
		if (!columnCounter.IsEmpty) {
		var tempSet = columnCounter.ToArray( ).OrderByDescending(kvp => kvp.Value);
		foreach (var mostActiveCol in tempSet) {

		var mapChunk = ClientAPI.World.BlockAccessor.GetMapChunk(mostActiveCol.Key);

		if (mapChunk == null) {
		Logger.Warning("SKIP CHUNK: ({0}) - Map Chunk NULL!", mostActiveCol.Key);

		columnCounter.TryRemove(mostActiveCol.Key, out ejectedItem );
		continue;
		}
		
		ColumnMeta chunkMeta = UpdateColumnMetadata(mostActiveCol,mapChunk);
		PngWriter pngWriter = SetupPngImage(mostActiveCol.Key, chunkMeta);

		uint updatedPixels = 0;
		GenerateChunkImage(mostActiveCol.Key, mapChunk, pngWriter , out updatedPixels);
		
		if (updatedPixels > 0) {		
		
		#if DEBUG
		Logger.VerboseDebug("Wrote chunk shard: ({0}) - Edits#:{1}, Pixels#:{2}", mostActiveCol.Key, mostActiveCol.Value, updatedPixels);
		#endif
		updatedChunks++;
		chunkTopMetadata.Update(chunkMeta);
		columnCounter.TryRemove(mostActiveCol.Key, out ejectedItem);
		}
		else {
		columnCounter.TryRemove(mostActiveCol.Key, out ejectedItem);
		Logger.VerboseDebug("Un-painted chunk: ({0}) ", mostActiveCol.Key);
		}

		}
		}

		if (updatedChunks > 0) {
		//TODO: ONLY update if chunk bounds have changed!
		lastUpdate = updatedChunks;
		GenerateMapHTML( );
		updatedChunks = 0;
		}

		//Then sleep until interupted again, and repeat

		Logger.VerboseDebug("Thread '{0}' about to sleep indefinitely.", Thread.CurrentThread.Name);

		Thread.Sleep(Timeout.Infinite);

		} catch (ThreadInterruptedException) {

		Logger.VerboseDebug("Thread '{0}' interupted [awoken]", Thread.CurrentThread.Name);
		goto wake;

		} catch (ThreadAbortException) {
		Logger.VerboseDebug("Thread '{0}' aborted.", Thread.CurrentThread.Name);

		} finally {
		Logger.VerboseDebug("Thread '{0}' executing finally block.", Thread.CurrentThread.Name);
		}
		}

		#endregion


		private void Prefill_POI_Designators( )
		{
		this.POIs = new PointsOfInterest( );
		this.BlockID_Designators = new Dictionary<int, Designator>( );

		//Add special marker types for BlockID's of "Interest", overwrite colour, and method

		var theDesignators = new List<Designator>{
				DefaultDesignators.Roads,
                DefaultDesignators.GroundSigns,
                DefaultDesignators.WallSigns,
                DefaultDesignators.PostSigns,
				};

		Install_POI_Designators(theDesignators);
		}

		private void Install_POI_Designators(ICollection<Designator> designators)
		{
		Logger.VerboseDebug("Connecting {0} configured Designators", designators.Count);
		foreach (var designator in designators) {				
			var blockIDs = Helpers.ArbitrarytBlockIdHunter(ClientAPI, designator.Pattern, designator.Material);
				if (blockIDs.Count > 0) { Logger.VerboseDebug("Designator {0} has {1} associated blockIDs", designator.ToString( ), blockIDs.Count); }
			foreach (var entry in blockIDs) {
			BlockID_Designators.Add(entry.Key, designator);
			}
		}

		}



		#region COPYPASTA
		//TODO: rewrite - with vertical ray caster, down to bottom-most chunk (for object detection...)
		//A partly re-written; ChunkMapLayer :: public int[] GenerateChunkImage(Vec2i chunkPos, IMapChunk mc)
		internal void GenerateChunkImage(Vec2i chunkPos, IMapChunk mc, PngWriter pngWriter, out uint pixelCount)
		{
		pixelCount = 0;
		BlockPos tmpPos = new BlockPos( );
		Vec2i localpos = new Vec2i( );
		int chunkSize = ClientAPI.World.BlockAccessor.ChunkSize;
		var chunksColumn = new IWorldChunk[ClientAPI.World.BlockAccessor.MapSizeY / chunkSize];

		int topChunkY = mc.YMax / chunkSize;//Heywaitaminute -- this isn't a highest FEATURE, if Rainmap isn't accurate!
											//Metadata of DateTime chunk was edited, chunk coords.,world-seed? Y-Max feature height
											//Grab a chunk COLUMN... Topmost Y down...
		for (int chunkY = 0; chunkY <= topChunkY; chunkY++) {
		chunksColumn[chunkY] = ClientAPI.World.BlockAccessor.GetChunk(chunkPos.X, chunkY, chunkPos.Y);
		//What to do if chunk is a void? invalid?
		}

		// Prefetch map chunks, in pattern
		IMapChunk[ ] mapChunks = new IMapChunk[ ]
		{
			ClientAPI.World.BlockAccessor.GetMapChunk(chunkPos.X - 1, chunkPos.Y - 1),
			ClientAPI.World.BlockAccessor.GetMapChunk(chunkPos.X - 1, chunkPos.Y),
			ClientAPI.World.BlockAccessor.GetMapChunk(chunkPos.X, chunkPos.Y - 1)
		};

		//pre-create PNG line slices...
		ImageLine[ ] lines = Enumerable.Repeat(new object(), chunkSize).Select(l => new ImageLine(pngWriter.ImgInfo)).ToArray();  

		for (int posIndex = 0; posIndex < (chunkSize * chunkSize); posIndex++) {
		int mapY = mc.RainHeightMap[posIndex];
		int localChunkY = mapY / chunkSize;
		if (localChunkY >= (chunksColumn.Length)) continue;//Out of range!

		MapUtil.PosInt2d(posIndex, chunkSize, localpos);
		int localX = localpos.X;
		int localZ = localpos.Y;

		float b = 1;
		int leftTop, rightTop, leftBot;

		IMapChunk leftTopMapChunk = mc;
		IMapChunk rightTopMapChunk = mc;
		IMapChunk leftBotMapChunk = mc;

		int topX = localX - 1;
		int botX = localX;
		int leftZ = localZ - 1;
		int rightZ = localZ;

		if (topX < 0 && leftZ < 0) {
		leftTopMapChunk = mapChunks[0];
		rightTopMapChunk = mapChunks[1];
		leftBotMapChunk = mapChunks[2];
		}
		else {
		if (topX < 0) {
		leftTopMapChunk = mapChunks[1];
		rightTopMapChunk = mapChunks[1];
		}
		if (leftZ < 0) {
		leftTopMapChunk = mapChunks[2];
		leftBotMapChunk = mapChunks[2];
		}
		}

		topX = GameMath.Mod(topX, chunkSize);
		leftZ = GameMath.Mod(leftZ, chunkSize);

		leftTop = leftTopMapChunk == null ? 0 : Math.Sign(mapY - leftTopMapChunk.RainHeightMap[leftZ * chunkSize + topX]);
		rightTop = rightTopMapChunk == null ? 0 : Math.Sign(mapY - rightTopMapChunk.RainHeightMap[rightZ * chunkSize + topX]);
		leftBot = leftBotMapChunk == null ? 0 : Math.Sign(mapY - leftBotMapChunk.RainHeightMap[leftZ * chunkSize + botX]);

		float slopeness = (leftTop + rightTop + leftBot);

		if (slopeness > 0) b = 1.2f;
		if (slopeness < 0) b = 0.8f;

		b -= 0.15f; //Slope boost value 

		if (chunksColumn[localChunkY] == null) {

		continue;
		}

		chunksColumn[localChunkY].Unpack( );
		int blockId = chunksColumn[localChunkY].Blocks[MapUtil.Index3d(localpos.X, mapY % chunkSize, localpos.Y, chunkSize, chunkSize)];

		Block block = ClientAPI.World.Blocks[blockId];

		tmpPos.Set(chunkSize * chunkPos.X + localpos.X, mapY, chunkSize * chunkPos.Y + localpos.Y);

		int avgCol = block.GetColor(ClientAPI, tmpPos);
		int rndCol = block.GetRandomColor(ClientAPI, tmpPos, BlockFacing.UP);		
		int col = ColorUtil.ColorOverlay(avgCol, rndCol, 0.125f);
		var packedFormat = ColorUtil.ColorMultiply3Clamped(col, b);

		int red = ColorUtil.ColorB(packedFormat);
		int green = ColorUtil.ColorG(packedFormat);
        int blue = ColorUtil.ColorR(packedFormat);


		//============ POI Population =================
		if (BlockID_Designators.ContainsKey(blockId)) {
		var desig = BlockID_Designators[blockId];
		red = desig.OverwriteColor.R;
		green = desig.OverwriteColor.G;
		blue = desig.OverwriteColor.B;

		if (desig.SpecialAction != null) {		
		desig.SpecialAction(ClientAPI, this.POIs, tmpPos, block);
		}
		}
		
		ImageLineHelper.SetPixel(lines[localZ], localX, red, green, blue);

		//chunkImage.SetPixel(localX, localZ, pixelColor);
		pixelCount++;
		}

		for (int row = 0; row < pngWriter.ImgInfo.Rows; row++) {
		pngWriter.WriteRow(lines[row], row);
		}

		pngWriter.End( );		
		}
		#endregion


		private void GenerateMapHTML( )
		{
		string mapFilename = Path.Combine(path, "Automap.html");

		int TopNorth = chunkTopMetadata.North_mostChunk;
		int TopSouth = chunkTopMetadata.South_mostChunk;
		int TopEast = chunkTopMetadata.East_mostChunk;
		int TopWest = chunkTopMetadata.West_mostChunk;

		using (StreamWriter outputText = new StreamWriter(File.Open(mapFilename, FileMode.Create, FileAccess.Write, FileShare.ReadWrite))) {
		using (HtmlTextWriter tableWriter = new HtmlTextWriter(outputText)) {
		tableWriter.BeginRender( );
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Html);

		tableWriter.RenderBeginTag(HtmlTextWriterTag.Head);
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Title);
		tableWriter.WriteEncodedText("Generated Automap");
		tableWriter.RenderEndTag( );
		//CSS  style  here
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Style);
		tableWriter.Write(stylesFile.ToText( ));
		tableWriter.RenderEndTag( );//</style>

		tableWriter.RenderEndTag( );

		tableWriter.RenderBeginTag(HtmlTextWriterTag.Body);
		tableWriter.RenderBeginTag(HtmlTextWriterTag.P);
		tableWriter.WriteEncodedText($"Created {DateTimeOffset.UtcNow.ToString("u")}");
		tableWriter.RenderEndTag( );
		tableWriter.RenderBeginTag(HtmlTextWriterTag.P);
		tableWriter.WriteEncodedText($"W:{TopWest}, E: {TopEast}, N:{TopNorth}, S:{TopSouth} ");
		tableWriter.RenderEndTag( );
		tableWriter.WriteLine( );
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Table);
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Caption);
		tableWriter.WriteEncodedText($"Start: {startChunkColumn}, Seed: {ClientAPI.World.Seed}\n");		
		tableWriter.RenderEndTag( );

		//################ X-Axis <thead> #######################
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Thead);
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Tr);

		tableWriter.RenderBeginTag(HtmlTextWriterTag.Th);
		tableWriter.Write("N, W");
		tableWriter.RenderEndTag( );

		for (int xAxisT = TopWest; xAxisT <= TopEast; xAxisT++) {
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Th);
		tableWriter.Write(xAxisT);
		tableWriter.RenderEndTag( );
		}

		tableWriter.RenderBeginTag(HtmlTextWriterTag.Th);
		tableWriter.Write("N, E");
		tableWriter.RenderEndTag( );
		
		tableWriter.RenderEndTag( );
		tableWriter.RenderEndTag( );
		//###### </thead> ################################

		//###### <tbody> - Chunk rows & Y-axis cols
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Tbody);

		//######## <tr> for every vertical row
		for (int yAxis = TopNorth; yAxis <= TopSouth; yAxis++) {
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Tr);
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Td);
		tableWriter.Write(yAxis);//legend: Y-axis
		tableWriter.RenderEndTag( );

		for (int xAxis = TopWest; xAxis <= TopEast; xAxis++) {
		//###### <td>  #### for chunk shard 
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Td);
		var colLoc = new Vec2i(xAxis, yAxis);
		if (chunkTopMetadata.Contains( colLoc)){
		ColumnMeta meta = chunkTopMetadata[colLoc];
		//Tooltip first					
		tableWriter.AddAttribute(HtmlTextWriterAttribute.Class, "tooltip");
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Div);

		tableWriter.AddAttribute(HtmlTextWriterAttribute.Src, $"{xAxis}_{yAxis}.png");		
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Img);
		tableWriter.RenderEndTag( );
		// <span class="tooltiptext">Tooltip text
		tableWriter.AddAttribute(HtmlTextWriterAttribute.Class, "tooltiptext");
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Span);

		StringBuilder tooltipText = new StringBuilder( );
		tooltipText.Append($"{meta.Location.PrettyCoords(ClientAPI)} ");
		tooltipText.Append($" Max-Height: {meta.YMax}, Temp: {meta.Temperature.ToString("F1")} " );
		tooltipText.Append($" Rainfall: {meta.Rainfall.ToString("F1")}, ");
		tooltipText.Append($" Shrubs: {meta.ShrubDensity.ToString("F1")}, ");
		tooltipText.Append($" Forest: {meta.ForestDensity.ToString("F1")}, ");
		tooltipText.Append($" Fertility: {meta.Fertility.ToString("F1")}, ");

		if (meta.RockRatio != null) {
		foreach (KeyValuePair<int, uint> blockID in meta.RockRatio) {
		var block = ClientAPI.World.GetBlock(blockID.Key);
		tooltipText.AppendFormat(" {0} × {1},\t", block.Code.GetName( ), meta.RockRatio[blockID.Key]);
		}
		}

		tableWriter.WriteEncodedText(tooltipText.ToString() );
		
		tableWriter.RenderEndTag( );//</span>
										

		tableWriter.RenderEndTag( );//</div> --tooltip enclosure
		}
		else {
		tableWriter.Write("?");
		}	

		tableWriter.RenderEndTag( );
		}//############ </td> ###########

		tableWriter.RenderBeginTag(HtmlTextWriterTag.Td);
		tableWriter.Write(yAxis);//legend: Y-axis
		tableWriter.RenderEndTag( );

		tableWriter.RenderEndTag( );
		
		}
		tableWriter.RenderEndTag( );

		//################ X-Axis <tfoot> #######################
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Tfoot);
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Tr);

		tableWriter.RenderBeginTag(HtmlTextWriterTag.Td);
		tableWriter.Write("S, W");
		tableWriter.RenderEndTag( );

		for (int xAxisB = TopWest; xAxisB <= TopEast; xAxisB++) {
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Td);
		tableWriter.Write(xAxisB);
		tableWriter.RenderEndTag( );
		}

		tableWriter.RenderBeginTag(HtmlTextWriterTag.Td);
		tableWriter.Write("S, E");
		tableWriter.RenderEndTag( );

		tableWriter.RenderEndTag( );
		tableWriter.RenderEndTag( );
		//###### </tfoot> ################################


		tableWriter.RenderEndTag( );//</table>
		
		//############## POI list #####################
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Ul);
		foreach (var poi in this.POIs) {
		tableWriter.RenderBeginTag(HtmlTextWriterTag.Li);
		tableWriter.WriteEncodedText(poi.Timestamp.ToString("u"));
		tableWriter.WriteEncodedText(poi.Notes);
		tableWriter.WriteEncodedText(poi.Location.PrettyCoords(this.ClientAPI));
		tableWriter.RenderEndTag( );
		}

		tableWriter.RenderEndTag( );
		tableWriter.RenderEndTag( );
		

		tableWriter.EndRender( );
		tableWriter.Flush( );
		}
		outputText.Flush( );		
		}

		Logger.VerboseDebug("Generated HTML map");
		}



		private ColumnMeta UpdateColumnMetadata(KeyValuePair<Vec2i, uint> mostActiveCol, IMapChunk mapChunk)
		{
		ColumnMeta data = new ColumnMeta(mostActiveCol.Key.Copy());
		BlockPos equivBP = new BlockPos(mostActiveCol.Key.X * ClientAPI.World.BlockAccessor.ChunkSize,
										mapChunk.YMax,
										mostActiveCol.Key.Y * ClientAPI.World.BlockAccessor.ChunkSize);

		var climate = ClientAPI.World.BlockAccessor.GetClimateAt(equivBP);		
		data.Temperature = climate.Temperature;
		data.Fertility = climate.Fertility;
		data.ForestDensity = climate.ForestDensity;
		data.Rainfall = climate.Rainfall;
		data.ShrubDensity = climate.ShrubDensity;

		data.YMax = mapChunk.YMax;

		
		/* Only present on server....
		if (mapChunk.TopRockIdMap != null) {
		foreach (var topRockId in mapChunk.TopRockIdMap) {

		if (data.RockRatio.ContainsKey(topRockId)) { data.RockRatio[topRockId]++; }
		else { data.RockRatio.Add(topRockId, 1); }
		}
		}*/


		return data;
		}

		/// <summary>
		/// Reload chunk bounds from chunk shards
		/// </summary>
		/// <returns>The metadata.</returns>
		private void Reload_Metadata( )
		{	
		var worldmapDir = new DirectoryInfo(path);

		if (worldmapDir.Exists) {

		var files = worldmapDir.GetFiles(chunkFile_filter);

		if (files.Length > 0) {
		#if DEBUG
		Logger.VerboseDebug("{0} Existing world chunk shards", files.Length);
		#endif

		PngChunk.FactoryRegister(PngMetadataChunk.ID, typeof(PngMetadataChunk));

		foreach (var shardFile in files) {
		var result = chunkShardRegex.Match(shardFile.Name);
		if (result.Success) {
		int X_chunk_pos = int.Parse(result.Groups["X"].Value );
		int Z_chunk_pos = int.Parse(result.Groups["Z"].Value );
		
		//Parse PNG chunks for METADATA in shard
		using (var fileStream = shardFile.OpenRead( ))
		{
		PngReader pngRead = new PngReader(fileStream );
		pngRead.ReadSkippingAllRows( );
		pngRead.End( );

		PngMetadataChunk metadataFromPng = pngRead.GetChunksList( ).GetById1(PngMetadataChunk.ID) as PngMetadataChunk;

		chunkTopMetadata.Add(metadataFromPng.ChunkMetadata);
		}
		
		}
		}

		}
		}
		else {
		#if DEBUG
		Logger.VerboseDebug("Could not open world map directory");
		#endif
		}



		}

		private PngWriter SetupPngImage(Vec2i coord, ColumnMeta metadata)
		{
		ImageInfo imageInf = new ImageInfo(ClientAPI.World.BlockAccessor.ChunkSize, ClientAPI.World.BlockAccessor.ChunkSize, 8, false);
		
		string filename = $"{coord.X}_{coord.Y}.png";
		filename = Path.Combine(path, filename);

		PngWriter pngWriter = FileHelper.CreatePngWriter(filename, imageInf, true);
		PngMetadata meta = pngWriter.GetMetadata( );
		meta.SetTimeNow( );
		meta.SetText("Chunk_X", coord.X.ToString("D"));
		meta.SetText("Chunk_Y", coord.Y.ToString("D"));
		//Setup specialized meta-data PNG chunks here...
		PngMetadataChunk pngChunkMeta = new PngMetadataChunk(pngWriter.ImgInfo);
		pngChunkMeta.ChunkMetadata = metadata;		
		pngWriter.GetChunksList( ).Queue(pngChunkMeta);

		return pngWriter;
		}
	}

}