﻿using System;

using System.Linq;
using System.Text;
using System.Collections.Generic;

using Vintagestory.API.Common;
using Vintagestory.API.Server;
using Vintagestory.Server;
using Vintagestory.API.Common.Entities;

namespace AdminToolkit
{
	public class IdlePlayerControl : AdminModCommand
	{
		const string idleKick_Key = @"atk:idle-kick";

		uint kick_tally;
		long timerIdent;

		Dictionary<string, EntityPos> playerPositions;


		public IdlePlayerControl(ICoreServerAPI _serverAPI) : base(_serverAPI)
		{
		playerPositions = new Dictionary<string, EntityPos>(32);
		this.Command = "idlekick";
		this.Description = "Adjust Idle-ejector system settings.";
		this.Syntax = @" set {minutes} | stats ";
		this.RequiredPrivilege = Privilege.kick;
		this.handler += HandleIdlerCommand;
		ServerAPI.Event.PlayerLeave += PlayerLeft;
		ServerAPI.Event.PlayerDisconnect += PlayerLeft;
		
		if (this.CachedConfiguration.IdleTimeout.TotalMinutes >= 5f) {
		ServerAPI.Event.ServerRunPhase(EnumServerRunPhase.RunGame, RestartTimers);
		}

		}

		private void HandleIdlerCommand(IServerPlayer player, int groupId, CmdArgs args)
		{
		var command = args.PopWord(string.Empty).ToLowerInvariant();

		switch (command) {				
			case @"set":
				//Change default & next cycle check time
				var timeMinutes = args.PopInt(0);
				if (timeMinutes > 0) {
				TimeSpan newTimeout = new TimeSpan(0, timeMinutes.Value, 0);
				this.CachedConfiguration.IdleTimeout = newTimeout;
				player.SendMessage(groupId, $"Changed Idle timeout to: {newTimeout}", EnumChatType.CommandSuccess);
				RestartTimers();
				} else 
				{
				player.SendMessage(groupId, "Idle-check disabled.", EnumChatType.CommandError);
				HaltIdleCheck( );
				}
			break;

			case @"stats":
			if (this.CachedConfiguration.IdleTimeout.TotalMinutes > 0f) 
				{ player.SendMessage(groupId, $"Ejected: {kick_tally} player(s), Timeout = {this.CachedConfiguration.IdleTimeout}", EnumChatType.CommandSuccess); }
				else { player.SendMessage(groupId, "offline.", EnumChatType.CommandSuccess);	}
			break;

			default:			
				player.SendMessage(groupId, $"Unrecognised command: '{command}'", EnumChatType.CommandError);
			break;
			}
		}

		private void HaltIdleCheck( )
		{
		if (timerIdent != 0L) {
			ServerAPI.Event.UnregisterCallback(timerIdent);
			playerPositions.Clear( );
			}
		}

		private void RestartTimers( )
		{			
		if (timerIdent != 0L) 
			{
			ServerAPI.Event.UnregisterCallback(timerIdent);
			playerPositions.Clear( );			
			}
		FindIdlers(0f);
		}

		private void FindIdlers(float delta )
		{
		#if DEBUG
		Logger.VerboseDebug("Starting idle player check");
		#endif

		foreach (var srvPlayer in ServerAPI.Server.Players.Where(plr => plr.ConnectionState == EnumClientState.Playing)) {
			if (srvPlayer?.WorldData?.CurrentGameMode == EnumGameMode.Survival) {
				EntityPos oldPos;
				if (playerPositions.TryGetValue(srvPlayer.PlayerUID, out oldPos)) {
					if (oldPos.BasicallySameAsIgnoreAngles(srvPlayer.Entity.ServerPos, 0.01f)) {
							if ( srvPlayer.Entity.CurrentControls.HasFlag( EnumEntityActivity.Idle) ||
							    srvPlayer.Entity.CurrentControls.HasFlag(EnumEntityActivity.FloorSitting) ||
							    srvPlayer.Entity.CurrentControls.HasFlag(EnumEntityActivity.Dead) ||
							    srvPlayer.Entity.CurrentControls.HasFlag(EnumEntityActivity.None) )
							{
								//Eee's not movin' 
								KickIdler(srvPlayer);
							}

						}
				}
				else {
					playerPositions.Add(srvPlayer.PlayerUID, srvPlayer.Entity.ServerPos);
					}
				}
			}

			timerIdent = ServerAPI.Event.RegisterCallback(FindIdlers, ( int )this.CachedConfiguration.IdleTimeout.TotalMilliseconds);
		}


		private void KickIdler(IServerPlayer them)
		{
		kick_tally++;
		var message = MultiLang.Get(them.LanguageCode, idleKick_Key, them.PlayerName);
		them.Disconnect(message);		 
		//ServerAPI.BroadcastMessageToAllGroups(message, EnumChatType.Notification);
		}

		private void PlayerLeft(IServerPlayer byPlayer)
		{
		if (playerPositions.ContainsKey(byPlayer.PlayerUID)) playerPositions.Remove(byPlayer.PlayerUID);
		}
}
}

