package jp.co.tryjava.jaxb;

import static org.hamcrest.CoreMatchers.*;
import static org.junit.Assert.*;

import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.URL;

import javax.xml.bind.JAXB;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;

import org.apache.commons.io.FileUtils;
import org.junit.Test;
import org.junit.experimental.runners.Enclosed;
import org.junit.runner.RunWith;

/**
 * JAXB（Java Architecture for XML Binding）を使ってみる練習です。
 * <p>
 * ・JAXB（Java Architecture for XML Binding）<br/>
 * ・Java SE 6 以降で使えます。<br/>
 * </p>
 */
@RunWith(Enclosed.class)
public class JaxbJpTest {

    public static class 日本語タグ {
        /**
         * タグ名に日本語を使うXMLを出力する練習です。
         */
        @Test
        public void testMarshal() {
            // 入力オブジェクト
            PersonJp person = new PersonJp();
            person.name = "山田 太郎";
            person.age = 20;

            // 出力先
            StringWriter actual = new StringWriter();

            // 実行
            JAXB.marshal(person, actual);

            // 期待値
            String expected = "" //
                    + "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" //
                    + "<人データ>\n" //
                    + "    <名前>山田 太郎</名前>\n" //
                    + "    <年齢>20</年齢>\n" //
                    + "</人データ>\n" //
            ;

            // 検証
            assertThat(actual.toString(), is(expected));
        }

        /**
         * タグ名に日本語を使うXMLを読み込む練習です。
         */
        @Test
        public void testUnmarshal() {
            // 入力XML
            String input = "" //
                    + "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" //
                    + "<人データ>\n" //
                    + "    <名前>山田 太郎</名前>\n" //
                    + "    <年齢>20</年齢>\n" //
                    + "</人データ>\n" //
            ;
            StringReader sr = new StringReader(input);

            // 実行
            PersonJp actual = JAXB.unmarshal(sr, PersonJp.class);

            // 検証
            assertThat(actual.name, is("山田 太郎"));
            assertThat(actual.age, is(20));
        }
    }

    public static class 日本語タグ_文字コード指定あり {
        /**
         * タグ名に日本語を使うXMLを出力する練習です。
         * <p>
         * ・ソースの文字コード：UTF-8<br/>
         * ・XMLの文字コード：Shift_JIS<br/>
         * </p>
         *
         * @throws IOException
         * @throws JAXBException
         */
        @Test
        public void testMarshal() throws IOException, JAXBException {
            // 入力オブジェクト
            PersonJp person = new PersonJp();
            person.name = "山田 太郎";
            person.age = 20;

            // 出力先
            StringWriter actual = new StringWriter();

            // Marshallerインスタンス作成
            // 文字コードにシフトJISを指定。
            // XMLを人が読みやすいようにフォーマットする。
            // ※改行コードはLF（0x0A）のみの模様。
            JAXBContext context = JAXBContext.newInstance(person.getClass());
            Marshaller marshaller = context.createMarshaller();
            marshaller.setProperty(Marshaller.JAXB_ENCODING, "Shift_JIS");
            marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);

            // 実行
            marshaller.marshal(person, actual);

            // 期待値
            URL url = getClass().getResource(
                    "/jp/co/tryjava/jaxb/person_sjis.xml");
            File file = FileUtils.toFile(url);
            String expected = FileUtils.readFileToString(file, "Shift_JIS");

            // 検証
            assertThat(actual.toString(), is(expected));
        }

        /**
         * タグ名に日本語を使うXMLを読み込む練習です。
         * <p>
         * ・ソースの文字コード：UTF-8<br/>
         * ・XMLの文字コード：Shift_JIS<br/>
         * </p>
         *
         * @throws IOException
         */
        @Test
        public void testUnmarshal() throws IOException {
            // 入力XML
            URL url = getClass().getResource(
                    "/jp/co/tryjava/jaxb/person_sjis.xml");
            File file = FileUtils.toFile(url);
            String input = FileUtils.readFileToString(file, "Shift_JIS");
            StringReader sr = new StringReader(input);

            // 実行
            PersonJp actual = JAXB.unmarshal(sr, PersonJp.class);

            // 検証
            assertThat(actual.name, is("山田 太郎"));
            assertThat(actual.age, is(20));
        }
    }

}
