package com.tryjava.datetime;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;

import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.commons.lang.time.DateUtils;

import com.tryjava.util.AppUtil;

/**
 * 日時を扱う練習
 *
 * Apache Commons Lang について
 * <ul>
 * <li>https://commons.apache.org/proper/commons-lang/
 * <li>バージョン3.2では、Java6以上が必要。
 * <li>バージョン3.0では、Java5以上が必要。
 * <li>バージョン2.xは、Java1.4で使用可能。
 * </ul>
 *
 * 参考
 * <ul>
 * <li>Java日付処理メモ(Hishidama's Java Date Memo)<br>
 * http://www.ne.jp/asahi/hishidama/home/tech/java/date.html
 * <li>日時に関する情報を取得する(get) - Calendarクラス<br>
 * http://www.javadrive.jp/start/calendar/index2.html
 * </ul>
 *
 * @since 2016/5/2
 */
public class DateTime01 {
	DateFormat format = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS");
	Date date1 = new Date();
	Calendar cal1 = Calendar.getInstance();
	long msec1 = System.currentTimeMillis();

	public static void main(String[] args) throws ParseException {
		DateTime01 app = new DateTime01();
		app.run();
	}

	void run() throws ParseException {
		createDateTime();
		getDateTimePart();
		calcDateTime();
	}

	/**
	 * 現在/指定日時を作成する。
	 *
	 * Java1.4～7では、主に、Dateクラス、Calendarクラスを使う。
	 */
	void createDateTime() throws ParseException {
		System.out.println("--- " + AppUtil.getMethod() + " ---");

		// Dateクラス
		// 現在日時
		Date nowDate1 = new Date();
		// 文字列→Date
		DateFormat dateTimeFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
		Date date1 = dateTimeFormat.parse("2016/5/2 15:00:00");
		// 文字列→Date（Apache Commons Lang）
		Date date1b = DateUtils.parseDate("2016/5/2 15:00:00", new String[] { "yyyy/MM/dd HH:mm:ss" });
		Date date1c = DateUtils.parseDate("2016-5-2 15:00:00",
				new String[] { "yyyy/MM/dd HH:mm:ss", "yyyy-MM-dd HH:mm:ss" });
		// Calendar→Date
		Date date2 = new GregorianCalendar(2016, 5 - 1, 2, 15, 0, 0).getTime();
		// long→Date
		Date date3 = new Date(System.currentTimeMillis());

		System.out.println(nowDate1);
		System.out.println(date1);
		System.out.println(date1b);
		System.out.println(date1c);
		System.out.println(date2);
		System.out.println(date3);

		// Calendarクラス
		// 現在日時
		Calendar nowCal1 = Calendar.getInstance();
		// 指定日時
		Calendar cal1 = new GregorianCalendar(2016, 5 - 1, 2, 15, 0, 0);
		// Date→Calendar
		Calendar cal2 = Calendar.getInstance();
		cal2.setTime(date1);
		// Date→Calendar（Apache Commons Lang）
		Calendar cal2b = DateUtils.toCalendar(date1);
		// long→Calendar
		Calendar cal3 = Calendar.getInstance();
		cal3.setTimeInMillis(System.currentTimeMillis());

		System.out.println(format.format(nowCal1.getTime()));
		System.out.println(format.format(cal1.getTime()));
		System.out.println(format.format(cal2.getTime()));
		System.out.println(format.format(cal2b.getTime()));
		System.out.println(format.format(cal3.getTime()));

		// long
		// 現在日時
		long nowMsec1 = System.currentTimeMillis();
		// 指定日時
		long msec1 = System.currentTimeMillis() - 24 * 60 * 60 * 1000;
		// Date→long
		long msec2 = date1.getTime();
		// Calendar→long
		long msec3 = cal1.getTimeInMillis();

		System.out.println(nowMsec1);
		System.out.println(msec1);
		System.out.println(msec2);
		System.out.println(msec3);

		// 文字列
		// Date→文字列
		DateFormat timestampFormat1 = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS");
		String str1 = timestampFormat1.format(date1);
		// Date→文字列（Apache Commons Lang）
		String str1b = DateFormatUtils.format(date1, "yyyy/MM/dd HH:mm:ss.SSS");
		// Calendar→文字列
		String str2 = format.format(cal1.getTime());
		// long→文字列
		String str3 = format.format(new Date(msec1));

		System.out.println(str1);
		System.out.println(str1b);
		System.out.println(str2);
		System.out.println(str3);
	}

	/**
	 * 日時から一部を取得する。
	 */
	void getDateTimePart() {
		System.out.println("--- " + AppUtil.getMethod() + " ---");

		//
		// 日付
		//

		// Date 日時→日付（時刻ゼロ）
		Calendar tmpCal1 = Calendar.getInstance();
		tmpCal1.setTime(date1);
		Calendar tmpCal2 = new GregorianCalendar(tmpCal1.get(Calendar.YEAR), tmpCal1.get(Calendar.MONTH),
				tmpCal1.get(Calendar.DATE));
		Date ymdDate1 = tmpCal2.getTime();

		// Calendar 日時→日付（時刻ゼロ）
		Calendar ymdCal1 = new GregorianCalendar(cal1.get(Calendar.YEAR), cal1.get(Calendar.MONTH),
				cal1.get(Calendar.DATE));

		// Date 日時→日付（時刻ゼロ）（Apache Commons Lang）
		Date ymdDate2 = DateUtils.truncate(date1, Calendar.DATE);
		// Calendar 日時→日付（時刻ゼロ）（Apache Commons Lang）
		Calendar ymdCal2 = DateUtils.truncate(cal1, Calendar.DATE);

		// 上記以外に、書式"yyyy/MM/dd"のフォーマットで文字列として取得することも可能。

		System.out.println(format.format(ymdDate1));
		System.out.println(format.format(ymdCal1.getTime()));
		System.out.println(format.format(ymdDate2));
		System.out.println(format.format(ymdCal2.getTime()));

		//
		// 時刻
		//

		// 時刻のみを取り出すことは出来ない模様。
		// 必要なら、書式"HH:mm:ss"のフォーマットで文字列として取得する（？）

		//
		// 曜日
		//

		// Dateから曜日取得
		// ※ロケールによって曜日の表記が変わる。
		DateFormat weekFormat1 = new SimpleDateFormat("E", Locale.JAPANESE);
		DateFormat weekFormat2 = new SimpleDateFormat("EEEE", Locale.JAPANESE);
		String weekStr1 = weekFormat1.format(date1);
		String weekStr2 = weekFormat2.format(date1);

		// Calendarから曜日取得
		int weekInt1 = cal1.get(Calendar.DAY_OF_WEEK);
		String[] weekNames = new String[] { "日", "月", "火", "水", "木", "金", "土" };
		String weekStr3 = weekNames[weekInt1 - 1];

		// Dateから曜日取得（Apache Commons Lang）
		// ※ロケールによって曜日の表記が変わる。
		String weekStr6 = DateFormatUtils.format(date1, "E", Locale.JAPANESE);
		String weekStr7 = DateFormatUtils.format(date1, "EEEE", Locale.JAPANESE);
		// Calendarから曜日取得（Apache Commons Lang）
		// ※ロケールによって曜日の表記が変わる。
		String weekStr8 = DateFormatUtils.format(cal1, "E", Locale.JAPANESE);
		String weekStr9 = DateFormatUtils.format(cal1, "EEEE", Locale.JAPANESE);

		System.out.println(weekStr1);
		System.out.println(weekStr2);
		System.out.println(weekStr3);
		System.out.println(weekStr6);
		System.out.println(weekStr7);
		System.out.println(weekStr8);
		System.out.println(weekStr9);
	}

	/**
	 * 日時を計算する。
	 */
	void calcDateTime() {
		System.out.println("--- " + AppUtil.getMethod() + " ---");

		//
		// 計算
		//

		// 加算/減算

		// Date 1秒後、1日後
		Date nextSecDate = new Date(msec1 + 1 * 1000);
		Date nextDayDate = new Date(msec1 + 24 * 60 * 60 * 1000);
		// Calendar 1秒後、1日後、１ヶ月後
		Calendar nextSecCal = (Calendar) cal1.clone();
		nextSecCal.add(Calendar.SECOND, 1);
		Calendar nextDayCal = (Calendar) cal1.clone();
		nextDayCal.add(Calendar.DATE, 1);
		Calendar nextMonthCal = (Calendar) cal1.clone();
		nextMonthCal.add(Calendar.MONTH, 1);

		// Date 1秒後、1日後（Apache Commons Lang）
		Date nextSecDate2 = DateUtils.addSeconds(date1, 1);
		Date nextDayDate2 = DateUtils.addDays(date1, 1);
		Date nextMonthDate2 = DateUtils.addMonths(date1, 1);

		System.out.println("1秒後：" + format.format(nextSecDate));
		System.out.println("1日後：" + format.format(nextDayDate));
		System.out.println("1秒後：" + format.format(nextSecCal.getTime()));
		System.out.println("1日後：" + format.format(nextDayCal.getTime()));
		System.out.println("1ヶ月後：" + format.format(nextMonthCal.getTime()));
		System.out.println("1秒後：" + format.format(nextSecDate2));
		System.out.println("1日後：" + format.format(nextDayDate2));
		System.out.println("1ヶ月後：" + format.format(nextMonthDate2));

		// Calendar 月初、月末
		Calendar monthBeginning = (Calendar) cal1.clone();
		monthBeginning.set(Calendar.DATE, 1);
		Calendar monthEnd = (Calendar) cal1.clone();
		monthEnd.set(Calendar.DATE, cal1.getActualMaximum(Calendar.DATE));

		System.out.println("月初：" + format.format(monthBeginning.getTime()));
		System.out.println("月末：" + format.format(monthEnd.getTime()));

		//
		// 差
		//

		// Date 日時の差 ミリ秒単位、日数単位（切り捨て）
		// ※日付の差がほしいときは、日時→日付のみに変換して計算する。
		long dateDiffMsec = nextDayDate.getTime() - date1.getTime();
		long dateDiffDays = (nextDayDate.getTime() - date1.getTime()) / (24 * 60 * 60 * 1000);
		// Calendar 日時の差 ミリ秒単位、日数単位（切り捨て）
		// ※日付の差がほしいときは、日時→日付のみに変換して計算する。
		long calDiffMsec = nextDayCal.getTimeInMillis() - cal1.getTimeInMillis();
		long calDiffDays = (nextDayCal.getTimeInMillis() - cal1.getTimeInMillis()) / (24 * 60 * 60 * 1000);

		// Date 日付の差（Apache Commons Lang）
		long dateDiffDays2 = (DateUtils.truncate(nextDayDate, Calendar.DATE).getTime()
				- DateUtils.truncate(date1, Calendar.DATE).getTime()) / (24 * 60 * 60 * 1000);
		// Calendar 日付の差（Apache Commons Lang）
		long calDiffDays2 = (DateUtils.truncate(nextDayCal, Calendar.DATE).getTimeInMillis()
				- DateUtils.truncate(cal1, Calendar.DATE).getTimeInMillis()) / (24 * 60 * 60 * 1000);

		System.out.println(dateDiffMsec);
		System.out.println(dateDiffDays);
		System.out.println(calDiffMsec);
		System.out.println(calDiffDays);
		System.out.println(dateDiffDays2);
		System.out.println(calDiffDays2);

		//
		// 比較
		//

		// Dateの前後比較
		boolean dateBeforeFlag = date1.before(nextDayDate);
		boolean dateAfterFlag = nextDayDate.after(date1);
		// Calendarの前後比較
		boolean calBeforeFlag = cal1.before(nextDayCal);
		boolean calAfterFlag = nextDayCal.after(cal1);

		// Date 日付一致確認（Apache Commons Lang）
		// ※時刻は無視して、日付のみ比較される。
		boolean dateSameFlag = DateUtils.isSameDay(date1, nextSecDate);
		// Calendar 日付一致確認（Apache Commons Lang）
		// ※時刻は無視して、日付のみ比較される。
		boolean calSameFlag = DateUtils.isSameDay(cal1, nextSecCal);

		System.out.println("前後比較：" + dateBeforeFlag);
		System.out.println("前後比較：" + dateAfterFlag);
		System.out.println("前後比較：" + calBeforeFlag);
		System.out.println("前後比較：" + calAfterFlag);
		System.out.println("一致確認：" + dateSameFlag);
		System.out.println("一致確認：" + calSameFlag);
	}
}
