package jp.sourceforge.doclet.multilingualfilter;

import com.sun.javadoc.*;
import com.sun.tools.doclets.formats.html.ConfigurationImpl;
import com.sun.tools.doclets.formats.html.HtmlDoclet;
import com.sun.tools.doclets.internal.toolkit.Configuration;
import java.util.HashSet;

/**
 * <div lang="ja">
 * doclet - 多言語フィルタ。
 * <p>
 * javaソースコードのコメントに&lt;div lang=&amp;ja&amp;&gt;のようなタグを用いて
 * 記述することで、コメント欄を多言語化する。
 * 使用方法は、javadocのコマンドライン引数に
 * <div>
 * <code>-doclet このdocletクラス -docletpath このdocletクラスがアーカイブされているjarファイル</code>
 * </div>
 * および
 * <div>
 * <code>-sellang ja,en</code>
 * </div>
 * という<code>-sellang</code>パラメータを追加して続けてコンマ区切りで言語を指定する。
 * &quot;div lang=&quot;ブロックの抽出が終えた後は自動的に標準のjavadocで処理する。
 * これにより任意の言語のjavadocを作成することを目的とする。
 * </p>
 * <p>
 * 比較的小規模な個人の開発などのように独立して作業する他の翻訳者がいないとか、
 * 各国語ドキュメントを一つのソースに統合したいとかの要望に
 * 答える足がかりになるよう期待して公開する。
 * </p>
 * <p>
 * コマンド行オプションで、出力する言語を選択、並べ替えできる。
 * たとえば、
 * <pre>-sellang ja</pre>
 * ならば日本語ブロック以外を出力しない(つまり ="en" などのブロックは無視するが、
 * ブロックでない部分はそのまま通す)。
 * <pre>-sellang en,ja</pre>
 * とすれば、ブロックはenを先に、jaを後に、
 * それ以外の&lt;div lang=&quot;..&quot;&gt;ブロックは無視する。
 * </p>
 * </div>
 * <div lang="en">
 * <p>
 * The java documentation comment multilanguage filter doclet.
 * This doclet is a filter for multilanguage Java documentation comment
 * using custom tag {@literal <div lang="xx">..</div>}.
 * xx is language code which likes 'ja' and 'en' and other any string specified language.
 * The command line option :
 * <pre>
 * -sellang ja,en
 * </pre>
 * specify extract ja block and en block.
 * To use this doclet, you may add to javadoc command line option as follows
 * <pre>
 * javadoc -doclet 'this doclet class' -docletpath 'jar file included this doclet class'
 * </pre>
 * </p>
 * </div>
 * @author tsurutsuru <kiramik@users.sourceforge.jp>
 */
public class MultilingualFilter {

    public static final HtmlDoclet htmlDoclet = new HtmlDoclet();
    /**
     * <div lang="ja">
     * 言語指定リスト。
     * </div>
     */
    private static String[] _langs = new String[0];
    /**
     * <div lang="ja">
     * 同じドキュメントを重複処理しないためのフラグ。
     * </div>
     */
    private static HashSet<Doc> _marked = new HashSet<Doc>();

    /**
     * <div lang="ja">docletのスタートポイント。</div>
     * <div lang="en">Start point of doclet.</div>
     * @param root 
     * <span lang="ja>システムから渡される。</span>
     * <span lang="en">passed by system.</span>
     * @return 
     * <span lang="ja">首尾よく処理すればtrue</span>
     */
    public static boolean start(RootDoc root) {
        MultilingualFilter me = new MultilingualFilter();
        return me.start2(root);
    }

    public boolean start2(RootDoc root) {
        _langs = readOptions(root.options());
        if ( treeTraversal(root)) {
            return htmlDoclet.start(root);
        }
        return false;
    }

    /**
     * <div lang="ja">自動で呼ばれる必須のメソッド。このオプションに必要な長さを返す。</div>
     * @param option 
     * <span lang="ja">システムからコマンドラインオプションの一つが選択して渡されるらしい。</span>
     * <span lang="en">option is setting by system.</span>
     * @return 
     * <span lang="ja">2固定。-sellangしかチェックしてない。</span>
     */
    public static int optionLength(String option) {
        if (option.equals("-sellang")) {
            return 2;
        }
        return htmlDoclet.optionLength(option);
    }

    /**
     * <div lang="ja">-sellangオプションのパラメータを読み込むメソッド。</div>
     * @param options 
     * <span lang="ja">システムから渡されるコマンドラインオプションの表</span>
     * @return 
     * <span lang="ja">lang=xxの部分で、出力されるべきxxの一覧</span>
     */
    private static String[] readOptions(String[][] options) {
        String[] langs = null;
        for (int i = 0; i < options.length; i++) {
            String[] opt = options[i];
            if (opt[0].equals("-sellang")) {
                langs = opt[1].split(",");
            }
        }
        return langs;
    }

    /**
     * <div lang="ja">
     * オプションの形式チェック。docletシステムが自動で呼ぶ。
     * 今回はただ一回だけオプションが指定されたかどうかだけを調べている。
     * さらに詳細なチェックを盛り込んでも良い。
     * </div>
     * @param options <span lang="ja">コマンドラインオプション</span>
     * @param reporter <span lang="ja">エラーの時に利用されるオブジェクトらしい</span>
     * @return <span lang="ja">間違いを検出しなければtrue</span>
     */
    public static boolean validOptions(String options[][],
            DocErrorReporter reporter) {
        boolean foundTagOption = false;
        for (int i = 0; i < options.length; i++) {
            String[] opt = options[i];
            if (opt[0].equals("-sellang")) {
                if (foundTagOption) {
                    reporter.printError("Only one -sellang option allowed.\n-sellangオプションは一回だけ指定できます。");
                    return false;
                } else {
                    foundTagOption = true;
                }
            }
        }
        if (!foundTagOption) {
            reporter.printError("Usage: javadoc -sellang ja,en -doclet multilanguageFilter ...");
        }
        if (foundTagOption) {
            return htmlDoclet.validOptions(options, reporter);
        }
        return false;
    }

    /**
     * <div lang="ja">
     * このdocletか対応するjava言語バージョン。
     * 汎用型、注釈、enum、および varArgs が追加された版なら 1.5.
     * </div>
     * @return <span lang="ja">htmlDocletに依存</span>
     */
    public static LanguageVersion languageVersion() {
        return htmlDoclet.languageVersion();
    }

    /**
     * <div lang="ja">
     * getRawCommentText()/setRawCommentText()ができる要素を
     * 出来るだけたくさん探してくる。
     * </div>
     * @param root <span lang="ja">ルート要素</span>
     * @return <span lang="ja">常に真</span>
     */
    private boolean  treeTraversal(RootDoc root) {

        Configuration configuration = ConfigurationImpl.getInstance();
        PackageDoc[] packages = configuration.packages;
        if (packages != null) {
            for (int i = 0; i < packages.length; i++) {
                replaceComment(packages[i]);
            }
        }

        replaceComment(root);

        PackageDoc[] packagedoc = root.specifiedPackages();
        for (int i = 0; i < packagedoc.length; i++) {
            replaceComment(packagedoc[i]);
            ClassDoc[] classdoc = packagedoc[i].allClasses();
            for (int j = 0; j < classdoc.length; j++) {
                replaceComment(classdoc[j]);
            }
            AnnotationTypeDoc[] annotationtypedoc = packagedoc[i].annotationTypes();
            for (int j = 0; j < annotationtypedoc.length; j++) {
                replaceComment(annotationtypedoc[j]);
            }
        }


        ClassDoc[] classdocs = root.classes();
        for (int i = 0; i < classdocs.length; i++) {

            replaceComment(classdocs[i]);

            ConstructorDoc[] constructordoc = classdocs[i].constructors();
            for (int j = 0; j < constructordoc.length; j++) {
                replaceComment(constructordoc[j]);
            }

            FieldDoc[] enumConstants = classdocs[i].enumConstants();
            for (int j = 0; j < enumConstants.length; j++) {
                replaceComment(enumConstants[j]);
            }

            FieldDoc[] fields = classdocs[i].fields();
            for (int j = 0; j < fields.length; j++) {
                replaceComment(fields[j]);
            }

            ClassDoc[] innerclasses = classdocs[i].innerClasses();
            for (int j = 0; j < innerclasses.length; j++) {
                replaceComment(innerclasses[j]);
            }

            // interfaceTypesにしないとダメなんじゃないかと思うが、その場合戻り値がType[]となるので違うかも。
            ClassDoc[] interfaces = classdocs[i].interfaces();
            for (int j = 0; j < interfaces.length; j++) {
                replaceComment(interfaces[j]);
            }

            MethodDoc[] methoddoc = classdocs[i].methods();
            for (int j = 0; j < methoddoc.length; j++) {
                replaceComment(methoddoc[j]);
                Tag[] tags = methoddoc[j].tags();
                for (int k = 0; k < tags.length; k++) {
                    replaceComment(tags[k].holder());
                }
            }

            FieldDoc[] serializablefields = classdocs[i].serializableFields();
            for (int j = 0; j < serializablefields.length; j++) {
                replaceComment(serializablefields[j]);
            }

            MethodDoc[] serializationmethods = classdocs[i].serializationMethods();
            for (int j = 0; j < serializationmethods.length; j++) {
                replaceComment(serializationmethods[j]);
            }


//            TypeVariable[] typeparameters = classdocs[i].typeParameters();
//            for (int j = 0; j < typeparameters.length; j++) {
//                replaceComment(typeparameters[j]);
//            }
//

//            ParamTag[] typeparamtags = classdocs[i].typeParamTags();
//            for (int j = 0; j < typeparamtags.length; j++) {
//                // parameterComment()で参照は出来るが設定は出来ない模様。
//            }

//            ClassDoc[] xxx = classdocs[i].interfaces();
//            for (int j=0; j<xxx.length; j++) {
//                replaceComment(xxx[j]);
//            }
        }
        return true;
    }

    /**
     * <div lang="ja">
     * 処理を簡単に記述するための中間メソッド。
     * 引数のオブジェクトのコメントを置き換える。
     * </div>
     * @param doc 
     * <span lang="ja">Docオブジェクトなら、getRawCommentText()/setRawCommentText()ができる</span>
     */
    private void replaceComment(Doc doc) {
        if (_marked.add(doc)) {
            doc.setRawCommentText(languageTag(doc.getRawCommentText()));
        }
    }

    /**
     * <div lang="ja">
     * 言語タグの選択と並べ替えの本体。
     * </div>
     * @param javaDocComment <span lang="ja">ドキュメントコメント</span>
     * @return <span lang="ja">言語タグを並べ替え/選択抽出したドキュメントコメント</span>
     */
    private String languageTag(String javaDocComment) {
        try {
            LangBlockParser lbp = new LangBlockParser(javaDocComment);
            return lbp.getTextAll(_langs);
        } catch (ParseError e) {
            e.printStackTrace();
            return "--- ERROR ---";
        }
    }
}
