package jp.sourceforge.doclet.multilingualfilter;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * <div lang="ja">
 * div,spanタグにのみ注目した単純な構文解析クラス
 * </div>
 * @author tsurutsuru <kiramik@users.sourceforge.jp>
 */
public class LangTagParser {

    /**
     * <div lang="ja">入力ドキュメント</div>
     * <div lang="en">input document.</div>
     */
    private String _doc;
    /**
     * <div lang="en">current position in parseing input document.</div>
     * <div lang="ja">入力ドキュメントの解析現在位置</div>
     */
    private int _pos;
    private String _type;   // 戻り値のタイプ"div", "/div", "span", "/span", "text", "whiteText"
    private String _lang;   // 開始タグの場合、戻り値のlang=の値。lang=がなかったり、タイプがdiv以外の場合は""。
    private Pattern _langPattern = Pattern.compile("lang=\"?(\\w+)\"?");

    public LangTagParser(String document) {
        _doc = document;
        _pos = 0;
    }

    /**
     * <div lang="ja">
     * next()で得られた戻り値のタイプ。
     * </div>
     * @return "text", "div", "/div", "span", "/span", "whiteText" のいずれか。
     */
    public String type() {
        return _type;
    }

    /**
     * <div lang="ja">
     * next()で得られた戻り値がdiv開始タグの場合、lang=パラメータの値。
     * パラメータがなかったり、そもそもdiv/spanタグでなかったりした場合は""。
     * </div>
     * <div lang="en">
     * </div>
     * @return
     * <span lang="ja">lang=パラメータの値または""</span>
     * <span lang="en">"" or xx of lang=xx</span>
     */
    public String lang() {
        return _lang;
    }

    public boolean hasNext() {
        return _pos < _doc.length();
    }

    public String next() throws divTagSyntaxError {
        int i1 = _doc.indexOf("<div", _pos);
        int i2 = _doc.indexOf("</div>", _pos);
        int i3 = _doc.indexOf("<span", _pos);
        int i4 = _doc.indexOf("</span>", _pos);

        _lang = "";
        if (_pos == i1) {   // div開始タグを取り出す
            if ((i2 = _doc.indexOf(">", _pos)) >= _pos) {
                String result = _doc.substring(_pos, ++i2);
                Matcher match = _langPattern.matcher(result);
                if (match.find()) {
                    _lang = match.group(1);
                }
                _pos = i2;
                _type = "div";
                return result;
            } else {
                throw new divTagSyntaxError("'<div'に対する'>'がありません。検出文字位置" + _pos);
            }
        }
        if (_pos == i2) {            // </div>タグを取り出す
            _pos += "</div>".length();
            _type = "/div";
            return "</div>";
        }
        if (_pos == i3) {   // <span ... を取り出す
            if ((i4 = _doc.indexOf(">", _pos)) >= _pos) {
                String result = _doc.substring(_pos, ++i4);
                Matcher match = _langPattern.matcher(result);
                if (match.find()) {
                    _lang = match.group(1);
                }
                _pos = i4;
                _type = "span";
                return result;
            } else {
                throw new divTagSyntaxError("'<span'に対する'>'がありません。検出文字位置" + _pos);
            }
        }
        if (_pos == i4) {   // </span>タグを取り出す
            _pos += "</span>".length();
            _type = "/span";
            return "</span>";
        }


        // 以降はテキストを取り出す

        int p = _doc.length();  // テキストの終了位置
        int[] x = {i1, i2, i3, i4};
        for (int i = 0; i < x.length; i++) {
            if (x[i] >= 0 && x[i] < p) {
                p = x[i];
            }
        }

        String text = _doc.substring(_pos, p);
        _pos = p;
        if (text.matches("\\s+")) {
            _type = "whiteText";
        } else {
            _type = "text";
        }
        return text;
    }

    class divTagSyntaxError extends Exception {

        private static final long serialVersionUID = 2782793156882437392L;

        public divTagSyntaxError(String message) {
            super(message);
        }
    }

}
