package jp.sourceforge.doclet.multilingualfilter;

import java.util.ArrayList;
import java.util.EmptyStackException;
import java.util.Stack;

/**
 * <div lang="ja">
 * 言語ブロックに分割管理するクラス。
 * </div>
 * <div lang="en">
 * The class to manage language block.
 * </div>
 * @author turuturu <kiramik@users.sourceforge.jp>
 */
public class LangBlockParser {

    /**
     * <div lang="ja">タグ解析クラスのオブジェクトを保持するフィールド</div>
     * <div lang="en">tag parser</div>
     */
    LangTagParser _tagParser;
    /**
     * <div lang="ja">言語ブロックに分割したドキュメンテーションコメントを保持するフィールド</div>
     * <div lang="en">documentation comment break in language block</div>
     */
    ArrayList<LangBlock> _blocks = new ArrayList<LangBlock>();

    /**
     * <div lang="ja">
     * <p>
     * 引数のドキュメント中の&lt;div lang=&quot;xx&quot;&gt;...&lt;/div&gt;と
     * &lt;span lang=&quot;xx&quot;&gt;...&lt;/span&gt;の
     * 各区間を言語xxで示された&quot;言語ブロック&quot;として取り扱うクラス。
     * 上記タグによって明示された言語ブロックの外側の文字は、
     * どこにも属さない言語ブロックとして言語選択やソートに影響せず常に出力される。
     * もし隣り合う言語ブロックが同じ言語であるなら、これらは一つのブロックとして
     * 取り扱う。
     * つまりこのクラスは、隣り合うブロックは常に異なる言語であるようにブロックの整理をする。
     * </p>
     * 以下の制限事項がある。
     * <ul>
     * <li>言語ブロックのネストは正しく扱えない。</li>
     * <li>spanとdivのネストは区別していない。</li>
     * </ul>
     * </div>
     * <div lang="en">
     * Read documentation comment. And parse.
     * </div>
     * @param docment 
     * <span lang="ja">任意のテキスト</span>
     * <span lang="en">documentation comment (any text)</span>
     */
    public LangBlockParser(String docment) throws ParseError {
        Stack <String> currentLanguage = new Stack <String> ();
        currentLanguage.push("");   // base language
        _tagParser = new LangTagParser(docment);
        try {
            while (_tagParser.hasNext()) {
                String tagOrText = _tagParser.next();
                String tagType = _tagParser.type();
                String tagLang = _tagParser.lang();

                if (tagType.equals("div") || tagType.equals("span")) {
                    if (tagLang.isEmpty()) {
                        // タグに言語指定がなければ現在の言語を継続
                        currentLanguage.push(currentLanguage.peek());
                    } else {
                        // 言語指定があれば切り替える
                        currentLanguage.push(tagLang);
                    }
                }

                if (_blocks.isEmpty()) {
                    newLangBlock(tagOrText, currentLanguage.peek());
                } else if (lastBlockGetLang().equals(currentLanguage.peek())) {
                    appendLangBlock(tagOrText);
                } else if (tagType.equals("whiteText")) {  // </div>や</span>の後に来る
                    appendLangBlock(tagOrText);
                } else {
                    newLangBlock(tagOrText, currentLanguage.peek());
                }

                if (tagType.equals("/div") || tagType.equals("/span")) {
                    currentLanguage.pop();
                }
            }
            
            int stackDepth=0;
            while (!currentLanguage.empty()) {
                currentLanguage.pop();
                stackDepth++;
            }
            if (stackDepth == 0) {
                throw new ParseError("最後まで解析した結果、</div>または</span>が一つ多すぎます");
            } else if (stackDepth > 1) {
                throw new ParseError("最後まで解析した結果、<div>または<span>が" + (stackDepth-1) + "個多すぎます");
            }

        } catch (LangTagParser.divTagSyntaxError e) {
            System.out.println(e.getMessage());
            e.printStackTrace();
        } catch (EmptyStackException e) {
            throw new ParseError("</div>または</span>が2個以上多すぎます");
        }
    }

    /**
     * <div lang="ja">
     * 隣り合う言語指定のあるブロックを、引数で与えた言語リストの順にソートして返す。
     * 隣り合うということは、言語指定のないブロックをまたいでソートは行わないということ。
     * </div>
     * <div lang="en">
     * Get documentation comment which sorted in language list's order.
     * </div>
     * @param langList
     * <span lang="ja">言語リスト</span>
     * <span lang="en">language order list</span>
     * @return
     * <span lang="ja">ソート済みテキスト。</span>
     * <span lang="en">documentation comment which sorted in language list's order.</span>
     */
    public String getTextAll(String[] langList) {
        extractByLang(langList);
        sortByLang(langList);
        return getTextAll();
    }

    /**
     * <span lang="ja">内部配列へ、新たな言語ブロックを追加する。</span>
     * @param text <span lang="ja">テキスト</span>
     * @param lang <span lang="ja">言語</span>
     */
    private void newLangBlock(String text, String lang) {
        _blocks.add(new LangBlock(text, lang));
    }
    /**
     * <div lang="ja">
     * 内部配列の最後の言語ブロックにテキストを追加する。
     * </div>
     * @param text <span lang="ja">追加するテキスト</span>
     */
    private void appendLangBlock(String text) {
        _blocks.get(_blocks.size()-1).appendText(text);
    }

    /**
     * <div lang="ja">
     * 内部配列の最後の言語ブロックの言語を返す。
     * </div>
     * @return <span lang="ja">言語</span>
     */
    private String lastBlockGetLang() {
        return getLang(_blocks.size()-1);
    }

    private String getLang(int index) {
        return _blocks.get(index).getLang();
    }
    /**
     * <div lang="ja">
     * 内部配列を連結してテキストを返す。
     * </div>
     * <div lang="en">
     * Get documentation comment.
     * </div>
     * @return <span lang="ja">テキスト</span><span lang="en">documentation comment</span>
     */
    public String getTextAll() {
        StringBuilder result = new StringBuilder();
        for (LangBlock b : _blocks) {
            result.append(b.getText());
        }
        return result.toString();
    }

    /**
     * 引数の言語リストに含まれないブロックを削除する。
     * @param languageList 言語リスト
     */
    public void extractByLang(String[] languageList) {
        if (languageList.length == 0) {
            return;
        }

        for (int i = _blocks.size() - 1; i >= 0; i--) {
            if (!getLang(i).isEmpty() && _blocks.get(i).rank(languageList) < 0) {
                _blocks.remove(i);
            }
        }
    }

    /**
     * 引数の言語リストの並びにあわせて、隣り合う言語ブロックをソートする。
     * バブルソート。
     * @param languageList 言語リスト
     */
    public void sortByLang(String[] languageList) {
        boolean changed;
        do {
            changed = false;
            for (int i = 0; i < _blocks.size() - 1; i++) {
                if (!_blocks.get(i).getLang().isEmpty()
                        && !_blocks.get(i + 1).getLang().isEmpty()) {
                    int a = _blocks.get(i).rank(languageList);
                    int b = _blocks.get(i + 1).rank(languageList);
                    if (a >= 0 && b >= 0 && a > b) {
                        _blocks.set(i + 1, _blocks.set(i, _blocks.get(i + 1)));
                        changed = true;
                    }
                }
            }
        } while (changed);
    }

    /**
     * <div lang="ja">
     * 言語ブロッククラス。
     * 連続する同じ言語のブロックは一つのブロックにまとめるよう取り扱うためのクラス。
     * </div>
     */
    private class LangBlock {

        String _text;
        String _lang;

        LangBlock(String text, String lang) {
            _text = text;
            _lang = lang;
        }

        /**
         * <div lang="ja">
         * 引数のテキストをこのブロックに追加する。
         * </div>
         * @param text <span lang="ja">追加しようとするテキスト</span>
         * @return <span lang="ja">自分自身を返す</span>
         */
        public LangBlock appendText(String text) {
            _text = _text.concat(text);
            return this;
        }

        public String getText() {
            return _text;
        }

        public String getLang() {
            return _lang;
        }

        /**
         * <div lang="ja">
         * 言語リストの何番目にこのブロックの言語が現れるかを返す。
         * 言語リスト中に該当言語がなければ-1を返す。
         * </div>
         * @param list <span lang="ja">言語リスト</span>
         * @return <span lang="ja">言語リスト中の順位。なければ-1。</span>
         */
        public int rank(String[] list) {
            for (int i = 0; i < list.length; i++) {
                if (_lang.equals(list[i])) {
                    return i;
                }
            }
            return -1;
        }
    }

}
