// vim: foldmethod=marker commentstring=//%s
package mn.jp.kekkouyakan.collision;
import mn.jp.kekkouyakan.geom.KyVector2f;
import mn.jp.kekkouyakan.util.KyLinkedElement;
import mn.jp.kekkouyakan.util.KyCollectionUtil;
import mn.jp.kekkouyakan.util.KyFloatComparator;
import mn.jp.kekkouyakan.functional.IKyAction1;
import mn.jp.kekkouyakan.functional.IKyAction2;
import mn.jp.kekkouyakan.functional.KyBinder;
import java.util.LinkedList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Iterator;

public class ColProgressControl extends ColControl
{//{{{
	public static interface IHandler
	{//{{{
		void beginCollision( ColWorld world_ );
		void beginBodyCollision( ColBody body_, boolean temp_ );
		void collideWith( IKyAction1<Message> msgScan_, ColBody ball_, ColTransform delta_, Iterable<ColResult> colLs_, float time_, boolean temp_ );
		void outOfRange( ColSpace space_, ColBody bd_, ColShape shape_, boolean temp_ );
		void endBodyCollision( ColBody body_, boolean temp_ );
		void endCollision( ColWorld world_ );
	}//}}}
	public static class DefaultHandler implements IHandler
	{//{{{
		public void beginCollision( ColWorld world_ )
		{//{{{
		}//}}}
		public void beginBodyCollision( ColBody body_, boolean temp_ )
		{//{{{
		}//}}}
		public void collideWith( IKyAction1<Message> msgScan_, ColBody ball_, ColTransform delta_, Iterable<ColResult> colLs_, float time_, boolean temp_ )
		{//{{{
		}//}}}
		public void outOfRange( ColSpace space_, ColBody bd_, ColShape shape_, boolean temp_ )
		{//{{{
		}//}}}
		public void endBodyCollision( ColBody body_, boolean temp_ )
		{//{{{
		}//}}}
		public void endCollision( ColWorld world_ )
		{//{{{
		}//}}}
	}//}}}
	public static class Message
	{//{{{
		public Message( ColBody bd_ )
		{//{{{
			body = bd_;
		}//}}}
		ColBody body = null;
		ColTransform delta = null;
		ColTransform tempDelta = null;
		boolean tempClosing = true;

		static Message newSlip( KyFloatComparator cmp_, ColResult collision_, float frict_, KyVector2f ballVec_, float time_ )
		{//{{{
			KyVector2f slipVec_ = collision_.getSlipVector( cmp_, ballVec_ );
			if( slipVec_ == null ){
				return null;
			}
			slipVec_.mulLocal( frict_ );
			Message msg_ = new Message( collision_.ballShape1.body );
			msg_.tempDelta = new ColTransform();
			msg_.tempDelta.position.copy( slipVec_.mul( time_ ) );
			return msg_;
		}//}}}
		static Message newBound( KyFloatComparator cmp_, ColResult collision_, float coef_, KyVector2f ballVec_, float time_ )
		{//{{{
			KyVector2f vec_ = collision_.getBoundVector( cmp_, ballVec_, coef_ );
			if( vec_ == null ){
				return null;
			}
			Message msg_ = new Message( collision_.ballShape1.body );
			msg_.delta = new ColTransform();
			msg_.delta.position.copy( vec_ );
			msg_.tempDelta = new ColTransform();
			msg_.tempDelta.position.copy( vec_.mul( time_ ) );
			return msg_;
		}//}}}
	}//}}}

	IHandler handler;
	float progressRatio = 0.9f;
	float progressMinTimeRatio = 0.1f;
	KyFloatComparator floatComparator;

	public ColProgressControl( IHandler handler_, KyFloatComparator floatComparator_ )
	{//{{{
		handler = handler_;
		floatComparator = floatComparator_;
	}//}}}
	static class SpaceVisitor implements ColSpace.IVisitor
	{//{{{
		float minProgress = 1.0f;
		ColShape newShape;
		ColBody ballBody;
		LinkedList<ColResult> collisionList = new LinkedList<ColResult>();
		LinkedList<ColResult> unspecifiedList = new LinkedList<ColResult>();
		KyFloatComparator floatComparator;

		public SpaceVisitor( ColBody ballBody_, ColShape newShape_, KyFloatComparator floatComparator_ )
		{//{{{
			ballBody = ballBody_;
			newShape = newShape_;
			floatComparator = floatComparator_;
		}//}}}

		public Iterable<ColResult> getCollisionResults()
		{//{{{
			return KyCollectionUtil.combineIterable( collisionList, unspecifiedList );
		}//}}}

		IKyAction2<ColResult, ColBody> scan = new IKyAction2<ColResult, ColBody>()
		{//{{{
			public void call( ColResult res0_, ColBody wallBody_ ){
				if( res0_.isUnspecifiedProgress( floatComparator ) ){
					unspecifiedList.add( res0_.getClone() );
					return;
				}
				int cmp_ = floatComparator.compare( res0_.progress, minProgress );
				if( cmp_ > 0 ){
					return;
				}
				if( cmp_ < 0 ){
					collisionList.clear();
					minProgress = res0_.progress;
				}
				ColResult res_ = res0_.getClone();
				collisionList.add( res_ );
			}
		};//}}}
		public boolean hasCollision()
		{//{{{
			return !collisionList.isEmpty() || !unspecifiedList.isEmpty();
		}//}}}
		public float getMinProgress()
		{//{{{
			if( unspecifiedList.isEmpty() ){
				return minProgress;
			}
			else{
				return 0f;
			}
		}//}}}
		public boolean visit( ColSpace.Node node_ )
		{//{{{
			ColBody wallBody_ = (ColBody)node_.getValue();
			if( wallBody_ == ballBody ){
				return true;
			}
			IKyAction1<ColResult> scan_ = KyBinder.toAction1( scan, wallBody_ );
			ColResult.collideWithShape( floatComparator, scan_, ballBody.shape, newShape, wallBody_.shape );
			return true;
		}//}}}
	}//}}}

	public void update(
		ColWorld world_,
		final ColBody ballBody_,
		ColTransform delta0_,
		float minTime_,
		float maxTime_,
		HashSet<ColBody> tempBodySet_
	)
	{//{{{
		float progressMinTime_ = progressMinTimeRatio * world_.frameUnitTime;
		boolean tempMode_ = (tempBodySet_ != null);
		float timeSz_ = maxTime_ - minTime_;
		if( timeSz_ < progressMinTime_ ){
			return;
		}
		ColTransform maxDelta_ = delta0_.mul( timeSz_ );
		handler.beginBodyCollision( ballBody_, tempMode_ );

		//position action
		{//{{{
			ColTransform maxPosDelta_ = new ColTransform();
			maxPosDelta_.position = maxDelta_.position;
			ColShape maxPosShape_ = ballBody_.shape.generateNext( maxPosDelta_ );

			SpaceVisitor viPos_ = new SpaceVisitor( ballBody_, maxPosShape_, floatComparator );
			if( !world_.space.visitCollisionNodes( ballBody_.shape, maxPosShape_, viPos_ ) ){
				handler.outOfRange( world_.space, ballBody_, maxPosShape_, tempMode_ );
				handler.endBodyCollision( ballBody_, tempMode_ );
				return;
			}

			if( !viPos_.hasCollision() ){
				ballBody_.shape = maxPosShape_;
				world_.space.register( ballBody_.node );
			}
			else {
				float minProgress_ = viPos_.getMinProgress();
				assert( floatComparator.lessEquals( 0f, minProgress_ ) );
				assert( floatComparator.lessEquals( minProgress_, 1f ) );
				float prog_ = minProgress_ * progressRatio;

				if( !floatComparator.equals( prog_, 0f ) ){
					ColTransform posDelta_ = maxPosDelta_.mul( prog_ );
					ColShape posShape_ = ballBody_.shape.generateNext( posDelta_ );
					if( world_.space.isOutOfRange( posShape_ ) ){
						handler.outOfRange( world_.space, ballBody_, posShape_, tempMode_ );
						handler.endBodyCollision( ballBody_, tempMode_ );
						return;
					}
					ballBody_.shape = posShape_;
					world_.space.register( ballBody_.node );
				}

				float newMinTime_ = minTime_ + timeSz_*minProgress_;
				float leftTime_ = maxTime_ - newMinTime_;

				//temp transform action
				final LinkedList<Message> tempMsgLs_ = new LinkedList<Message>();
				IKyAction1<Message> msgScan_ = null;
				if( floatComparator.equals( leftTime_, 0f ) ||
					tempBodySet_ != null && tempBodySet_.contains( ballBody_ )
				){
					msgScan_ = new IKyAction1<Message>(){
						public void call( Message msg_ ){
							if( msg_.delta != null ){
								msg_.body.defaultTransform = msg_.delta.getClone();
							}
						}
					};
				}
				else{
					if( tempBodySet_ == null ){
						tempBodySet_ = new HashSet<ColBody>();
					}
					final HashSet<ColBody> tempBodySet__ = tempBodySet_;
					assert( !tempBodySet_.contains( ballBody_ ) );

					msgScan_ = new IKyAction1<Message>(){
						boolean closed_ = false;
						public void call( Message msg_ ){
							if( msg_.delta != null ){
								msg_.body.defaultTransform = msg_.delta.getClone();
							}
							if( msg_.tempDelta != null ){
								Message msgClone_ = new Message( msg_.body );
								msgClone_.tempDelta = msg_.tempDelta.getClone();
								tempMsgLs_.add( msgClone_ );
							}
							if( msg_.tempClosing && !closed_ ){
								closed_ = true;
								tempBodySet__.add( ballBody_ );
							}
						}
					};
				}
				handler.collideWith(
					msgScan_,
					ballBody_,
					maxPosDelta_,
					viPos_.getCollisionResults(),
					leftTime_,
					tempMode_
				);
				for( Message tempMsg_ : tempMsgLs_ ){
					update(
						world_,
						tempMsg_.body,
						tempMsg_.tempDelta, 
						newMinTime_,
						maxTime_,
						tempBodySet_
					);
				}
			}
		}
		//}}}
		//scale , rotation action
		if( !tempMode_
			&& ( !maxDelta_.scale.equals( floatComparator, 0f, 0f ) || !floatComparator.equals( maxDelta_.angle360, 0f ) )
		){//{{{
			ColTransform maxExDelta_ = new ColTransform();
			maxExDelta_.scale = maxDelta_.scale;
			maxExDelta_.angle360 = maxDelta_.angle360;
			ColShape maxExShape_ = ballBody_.shape.generateNext( maxExDelta_ );

			SpaceVisitor viEx_ = new SpaceVisitor( ballBody_, maxExShape_, floatComparator );
			if( !world_.space.visitCollisionNodes( ballBody_.shape, maxExShape_, viEx_ ) ){
				handler.outOfRange( world_.space, ballBody_, maxExShape_, tempMode_ );
				handler.endBodyCollision( ballBody_, tempMode_ );
				return;
			}

			if( !viEx_.hasCollision() ){
				ballBody_.shape = maxExShape_;
				world_.space.register( ballBody_.node );
			}
			else{
				float minProgress_ = viEx_.getMinProgress();
				assert( floatComparator.lessEquals( 0f, minProgress_ ) );
				assert( floatComparator.lessEquals( minProgress_, 1f ) );
				float newMinTime_ = minTime_ + timeSz_*minProgress_;
				float leftTime_ = maxTime_ - newMinTime_;

				if( floatComparator.more( minProgress_, 0f ) ){
					ColTransform exDelta_ = maxExDelta_.mul( minProgress_ );
					ballBody_.shape = ballBody_.shape.generateNext( exDelta_ );
					world_.space.register( ballBody_.node );
				}
				handler.collideWith(
					null,
					ballBody_,
					maxExDelta_,
					viEx_.getCollisionResults(),
					leftTime_,
					tempMode_
				);
			}
		}//}}}

		handler.endBodyCollision( ballBody_, tempMode_ );
	}//}}}
	@Override
	public void onUpdate( ColWorld world_, float vt_ )
	{//{{{
		handler.beginCollision( world_ );
		for( KyLinkedElement<ColBody>
			bdElem_ = world_.bodyHead.getNext();
			bdElem_ != null;
			bdElem_ = bdElem_.getNext()
		){
			ColBody bd_ = bdElem_.getValue();
			if( bd_.isDead() ){
				bdElem_.disconnect();
				continue;
			}
			if( bd_.defaultTransform == null ){
				continue;
			}
			update( world_, bd_, bd_.defaultTransform, 0f, vt_, null );
			if( bd_.isDead() ){
				bdElem_.disconnect();
				continue;
			}
		}
		handler.endCollision( world_ );
	}//}}}

	static void calcCollision( KyVector2f ov1_, KyVector2f ov2_, float elast_, KyVector2f iv1_, float m1_, KyVector2f iv2_, float m2_ )
	{//{{{
		//http://homepage2.nifty.com/eman/dynamics/collision.html
		assert( ov1_ != null );
		assert( ov2_ != null );
		assert( iv1_ != null );
		assert( iv2_ != null );
		float common1_ = (1f +elast_)/(m1_/m2_ + 1);
		float common2_ = (1f +elast_)/(m2_/m1_ + 1);
		float x1_ = (-iv1_.x+iv2_.x)*common1_ +iv1_.x;
		float y1_ = (-iv1_.y+iv2_.y)*common1_ +iv1_.y;
		float x2_ = (-iv2_.x+iv1_.x)*common2_ +iv2_.x;
		float y2_ = (-iv2_.y+iv1_.y)*common2_ +iv2_.y;
		ov1_.x = x1_;
		ov1_.y = y1_;
		ov2_.x = x2_;
		ov2_.y = y2_;
	}//}}}
	static void calcCollision( KyVector2f ov1_, KyVector2f ov2_, float elast_, KyVector2f iv1_, float m1_, float m2_ )
	{//{{{
		assert( ov1_ != null );
		assert( ov2_ != null );
		assert( iv1_ != null );
		float common1_ = (1f +elast_)/(m1_/m2_ + 1);
		float common2_ = (1f +elast_)/(m2_/m1_ + 1);
		float x1_ = (-iv1_.x)*common1_ +iv1_.x;
		float y1_ = (-iv1_.y)*common1_ +iv1_.y;
		float x2_ = iv1_.x*common2_;
		float y2_ = iv1_.y*common2_;
		ov1_.x = x1_;
		ov1_.y = y1_;
		ov2_.x = x2_;
		ov2_.y = y2_;
	}//}}}
	static boolean getBallVector( KyVector2f ov1_, KyVector2f ov2_, ColResult res_, ColShape shape1_, ColShape shape2_ )
	{//{{{
		assert( ov1_ != null );
		assert( ov2_ != null );

		if( shape1_ instanceof ColCircle ){
			assert( shape2_ instanceof ColCircle );
			ColCircle cir1_ = (ColCircle)shape1_;
			ColCircle cir2_ = (ColCircle)shape2_;
			ov1_.copy( cir2_.transform.position.sub( cir1_.transform.position ) );
			return false;
		}
		else if( shape1_ instanceof ColVectors ){
			assert( shape2_ instanceof ColVectors );
			ColVectors vecs1_ = (ColVectors)shape1_;
			ColVectors vecs2_ = (ColVectors)shape2_;
			KyVector2f[] vecLs1_ = vecs1_.currentPoints;
			KyVector2f[] vecLs2_ = vecs2_.currentPoints;
			int index1_ = res_.ballIndex1;
			int index2_ = res_.ballIndex2;
			assert( index1_ >= 0 );
			ov1_.copy( vecLs2_[ index1_ ].sub( vecLs1_[ index1_ ] ) );
			if( index2_ < 0 ){
				return false;
			}
			else{
				ov2_.copy( vecLs2_[ index2_ ].sub( vecLs1_[ index2_ ] ) );
				return true;
			}
		}
		else{
			throw new AssertionError();
		}
	}//}}}

}//}}}
