/*
 * Copyright (C) 2007 The Android Open Source Project
 * Copyright (C) 2011 Kouichi Machida (k01-dev@live.jp)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jp.gr.java_conf.k01.android.widgetpack.widget;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.LinearGradient;
import android.graphics.Paint;
import android.graphics.RectF;
import android.graphics.Shader;
import android.graphics.SweepGradient;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;

/**
 * J[sbJ[@\񋟂r[B
 * 
 * Android SDK Sample  ColorPickerDialog ɂĂ܂B
 * 
 * @author Kouichi Machida
 */
public class ColorPickerView extends View {

	/**
	 * J[sbJ[̐FύXꂽۂɌĂяo郊XiB
	 */
	public static interface OnColorChangedListener {

		/**
		 * FύXꂽۂɌĂяo܂B
		 * 
		 * @param color
		 *            ύXꂽFB
		 */
		void colorChanged(int color);

	}

	/**
	 * ~̑\܂B
	 */
	private static final int CENTER_RADIUS = 30;

	/**
	 * View ̉\܂B
	 */
	private static final int CENTER_X = 130;

	/**
	 * View ̍\܂B
	 */
	private static final int CENTER_Y = 130;

	/**
	 * J[o[̐Fێ܂B
	 */
	private static final int[] colors = new int[] { 0xffff0000, 0xffff00ff, 0xff0000ff, 0xff00ffff, 0xff00ff00, 0xffffff00, 0xffff0000 };

	/**
	 * ~\܂B
	 */
	private static final float PI = 3.1415927f;

	/**
	 * ̎lp̈ʒu\܂B
	 */
	private static final float X0 = -(ColorPickerView.CENTER_X - (ColorPickerView.CENTER_RADIUS / 2f)) / 2f;

	/**
	 * ̎lp̈ʒu\܂B
	 */
	private static final float X1 = (ColorPickerView.CENTER_X - (ColorPickerView.CENTER_RADIUS / 2f)) / 2f;

	/**
	 * IĂFێ܂B
	 */
	private int color;

	/**
	 * Paint IuWFNgێ܂B
	 */
	private final Paint colorsPaint;

	/**
	 * Fێ܂B
	 */
	private float hue;

	/**
	 * Paint IuWFNgێ܂B
	 */
	private final Paint huePaint;

	/**
	 * OnColorChangedListener IuWFNgێ܂B
	 */
	private OnColorChangedListener onColorChangedListener = null;

	/**
	 * ColorPickerView NX̐VCX^X܂B
	 * 
	 * @param context
	 */
	public ColorPickerView(Context context) {
		this(context, null);
	}

	/**
	 * ColorPickerView NX̐VCX^X܂B
	 * 
	 * @param context
	 * @param attrs
	 */
	public ColorPickerView(Context context, AttributeSet attrs) {
		this(context, attrs, 0);
	}

	/**
	 * ColorPickerView NX̐VCX^X܂B
	 * 
	 * @param context
	 * @param attrs
	 * @param defStyle
	 */
	public ColorPickerView(Context context, AttributeSet attrs, int defStyle) {
		super(context, attrs, defStyle);

		this.color = Color.WHITE;
		this.hue = ColorUtils.getHue(this.color);

		this.colorsPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
		this.colorsPaint.setStyle(Paint.Style.STROKE);
		this.colorsPaint.setShader(new SweepGradient(0, 0, ColorPickerView.colors, null));
		this.colorsPaint.setStrokeWidth(ColorPickerView.CENTER_RADIUS);

		this.huePaint = new Paint(Paint.ANTI_ALIAS_FLAG);
		this.huePaint.setStyle(Paint.Style.FILL);
		this.huePaint.setStrokeWidth(2);
	}

	/**
	 * J[sbJ[őIĂF擾܂B
	 * 
	 * @return ݑIĂFB
	 */
	public int getColor() {
		return this.color;
	}

	/**
	 * OnColorChangedListener IuWFNg擾܂B
	 * 
	 * @return OnColorChangedListener IuWFNgB
	 */
	public OnColorChangedListener getOnColorChangedListener() {
		return this.onColorChangedListener;
	}

	/**
	 * ĕ`ƃXiւ̒ʒms܂B
	 */
	private void onColorChanged() {
		this.invalidate();

		if (this.onColorChangedListener != null) {
			this.onColorChangedListener.colorChanged(this.color);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see android.view.View#onDraw(android.graphics.Canvas)
	 */
	@Override
	protected void onDraw(Canvas canvas) {
		float r = ColorPickerView.CENTER_X - this.colorsPaint.getStrokeWidth() * 0.5f;

		canvas.translate(ColorPickerView.CENTER_X, ColorPickerView.CENTER_X);
		canvas.drawOval(new RectF(-r, -r, r, r), this.colorsPaint);

		for (float y = 0; y < 1; y += 0.01f) {
			int[] mChroma = new int[10];
			int i = 0;

			for (float x = 0; i < 10; x += 0.1, i += 1) {
				mChroma[i] = ColorUtils.HSVToColor(this.hue, x, y);
			}

			this.huePaint.setShader(new LinearGradient(ColorPickerView.X0, 0, ColorPickerView.X1, 0, mChroma, null, Shader.TileMode.CLAMP));

			canvas.drawLine(ColorPickerView.X0, ColorPickerView.X0 + (r * y), ColorPickerView.X1, ColorPickerView.X0 + (r * y), this.huePaint);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see android.view.View#onMeasure(int, int)
	 */
	@Override
	protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
		this.setMeasuredDimension(ColorPickerView.CENTER_X * 2, ColorPickerView.CENTER_Y * 2);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see android.view.View#onTouchEvent(android.view.MotionEvent)
	 */
	@Override
	public boolean onTouchEvent(MotionEvent event) {
		float x = event.getX() - ColorPickerView.CENTER_X;
		float y = event.getY() - ColorPickerView.CENTER_Y;
		float r = (float) (Math.sqrt(x * x + y * y));

		switch (event.getAction()) {
		case MotionEvent.ACTION_DOWN:
			break;
		case MotionEvent.ACTION_MOVE:
			if (r <= ColorPickerView.CENTER_X) {
				if (ColorPickerView.CENTER_X - ColorPickerView.CENTER_RADIUS < r) {
					float angle = (float) Math.atan2(y, x);
					// need to turn angle [-PI ... PI] into unit [0....1]
					float unit = angle / (2 * ColorPickerView.PI);

					if (unit < 0) {
						unit += 1;
					}

					// Ώۈʒu̐F擾܂B
					this.color = ColorUtils.interpColor(ColorPickerView.colors, unit);

					// FF肵܂B
					this.hue = ColorUtils.getHue(this.color);

					// ĕ`ƃXiւ̒ʒms܂B
					this.onColorChanged();
				} else if ((ColorPickerView.X0 <= x) && (x < ColorPickerView.X1) && (ColorPickerView.X0 <= y) && (y < ColorPickerView.X1)) {
					// FF肵܂B
					this.color = ColorUtils.HSVToColor(this.hue, (x - ColorPickerView.X0) / ColorPickerView.CENTER_X, (y - ColorPickerView.X0) / ColorPickerView.CENTER_Y);

					// ĕ`ƃXiւ̒ʒms܂B
					this.onColorChanged();
				}
			}

			break;
		case MotionEvent.ACTION_UP:
			break;
		}

		return true;
	}

	/**
	 * Fݒ肵܂B
	 * 
	 * @param color
	 *            ݒ肷FB
	 */
	public void setColor(int color) {
		// ݂̒lƈقȂlmF܂B
		if (this.color != color) {
			// lXV܂B
			this.color = color;
			this.hue = ColorUtils.getHue(this.color);

			// ĕ`ƃXiւ̒ʒms܂B
			this.onColorChanged();
		}
	}

	/**
	 * OnColorChangedListener IuWFNgݒ肵܂B
	 * 
	 * @param onColorChangedListener
	 *            ݒ肷 OnColorChangedListener IuWFNgB
	 */
	public void setOnColorChangedListener(OnColorChangedListener onColorChangedListener) {
		this.onColorChangedListener = onColorChangedListener;
	}

}
