# -*- mode: perl; coding: utf-8 -*-
# Keitairc::Config
#
# Copyright (c) 2008 Jun Morimoto <morimoto@mrmt.net>
# This program is covered by the GNU General Public License 2

package Keitairc::Config;
use AppConfig qw(:argcount);
use Cwd;
use Encode;
use Encode::MIME::Name;
use strict;
use warnings;
our @ISA = qw(AppConfig);

################################################################
sub new{
	my $class = shift;
	my $arg = shift;
	my @argv = @{$arg->{argv}};
	my $me = $class->SUPER::new(
		{CASE => 1,
		 ERROR => \&on_error,
		 GLOBAL => {
			 ARGCOUNT => ARGCOUNT_ONE,
		 }});

	$me->define(
		# required
		'irc_nick',
		'irc_username',
		'irc_server',
		'web_password',
		'web_host',

		# optional
		'irc_password',
		'irc_desc' => {DEFAULT => 'keitairc'},
		'au_subscriber_id' => {DEFAULT => ''},
		'docomo_foma_icc' => {DEFAULT => ''},
		'docomo_imodeid' => {DEFAULT => ''},
		'softbank_serial_key' => {DEFAULT => ''},
		'emobile_userid' => {DEFAULT => ''},
		'irc_keyword',
		'web_listen_port',
		'web_title' => {DEFAULT => 'keitairc'},
		'common_header' => {DEFAULT => '
<meta name="Robots" content="noindex,nofollow" />
<meta name="Keywords" content="norobot" />
<meta http-equiv="pragma" content="no-cache" />
<meta http-equiv="cache-control" content="no-cache" />
<meta http-equiv="expires" content="-1" />'},
		'extra_header' => {DEFAULT => ''},
		'silent_config' => {DEFAULT => $arg->{silent}},
		'version' => {DEFAULT => $arg->{version}},
		'template_dir'  => {DEFAULT => getcwd() . '/data/templates:__KEITAIRC_DATA_DIR__/templates'},
		'plugin_dir' => {DEFAULT => getcwd() . '/data/plugins:__KEITAIRC_DATA_DIR__/plugins'},
		'public_dir' => {DEFAULT => getcwd() . '/data/public:__KEITAIRC_DATA_DIR__/public'},
		'url_redirect' => {DEFAULT => ''},
		'smtp_server' => {DEFAULT => ''},
		'smtp_from' => {DEFAULT => ''},
		'smtp_to' => {DEFAULT => ''},
		'rgeocode_server' => {DEFAULT => 'finds'},
		'pid_file' => {DEFAULT => 'keitairc.pid'},

		'web_root' => {DEFAULT => '/', VALIDATE => \&valid_web_root},
		'web_schema' => {DEFAULT => 'http', VALIDATE => \&valid_web_schema},
		'fontsize' => {DEFAULT => '+0', VALIDATE => \&valid_fontsize},
		'mobile_fontsize' => {DEFAULT => -1, VALIDATE => \&valid_fontsize},
		'irc_charset' => {DEFAULT => 'utf8', VALIDATE => \&valid_charset},
		'web_charset' => {DEFAULT => 'shiftjis', VALIDATE => \&valid_charset},
		'pid_dir' => {DEFAULT => $ENV{HOME} . '.keitairc_log', VALIDATE => \&valid_dir},
		'url_target' => {DEFAULT => '_self', VALIDATE => \&valid_url_target},
		'log' => {DEFAULT => 'file', VALIDATE => \&valid_log},

		# optional integer params
		'irc_port' => {DEFAULT => 6667, VALIDATE => \&valid_int},
		'cookie_ttl' => {DEFAULT => 86400 * 3, VALIDATE => \&valid_int},  # 3 days
		'session_ttl' => {DEFAULT => 60 * 30, VALIDATE => \&valid_int},  # 30 min
		'web_port' => {DEFAULT => 8080, VALIDATE => \&valid_int},
		'web_lines' => {DEFAULT => 100, VALIDATE => \&valid_int},
		'ping_delay' => {DEFAULT => 30, VALIDATE => \&valid_int},
		'reconnect_delay' => {DEFAULT => 10, VALIDATE => \&valid_int},

		# optional boolean params
		'show_joinleave' => {ARGCOUNT => ARGCOUNT_NONE, DEFAULT => 1},
		'show_console' => {ARGCOUNT => ARGCOUNT_NONE, DEFAULT => 0},
		'follow_nick' => {ARGCOUNT => ARGCOUNT_NONE, DEFAULT => 1},
		'debug' => {ARGCOUNT => ARGCOUNT_NONE, DEFAULT => 0},
		'daemonize' => {ARGCOUNT => ARGCOUNT_NONE, DEFAULT => 0},
		'reverse_message' => {ARGCOUNT => ARGCOUNT_NONE, DEFAULT => 1},
		'reverse_recent' => {ARGCOUNT => ARGCOUNT_NONE, DEFAULT => 1},
		'reverse_unread' => {ARGCOUNT => ARGCOUNT_NONE, DEFAULT => 1},
		'webkit_newui' => {ARGCOUNT => ARGCOUNT_NONE, DEFAULT => 1},

		# obsolates (ignored)
		'show_newmsgonly' => {VALIDATE => \&valid_obsolates},
		'web_username' => {VALIDATE => \&valid_obsolates},
		'use_cookie' => {VALIDATE => \&valid_obsolates},
		'au_pcsv' => {VALIDATE => \&valid_obsolates},
		);

	if(-r '/etc/keitairc'){
		$me->file('/etc/keitairc');
	}
	if(-r $ENV{HOME} . '/.keitairc'){
		$me->file($ENV{HOME} . '/.keitairc');
	}

	if(defined $argv[0]){
		if(-r $argv[0]){
			$me->file($argv[0]);
			shift(@argv);
		}
	}

	$me->args(\@argv);

	# check required parameters
	foreach my $n (qw(irc_nick irc_username irc_server web_host web_password)) {
		if(!defined($me->get($n)) || !length($me->get($n))) {
			die($n . ' does not specified');
		}
	}

	if(!defined($me->web_listen_port()) || !length($me->web_listen_port())){
		$me->web_listen_port($me->web_port());
	}

	$me;
}

################################################################
sub file{
	my $me = shift;
	my $file = shift;
	if(-r $file){
		$me->SUPER::file($file);
		print STDERR "Loaded configuration file: $file\n" unless $me->silent_config();
		return;
	}
	warn("$file does not exist");
}

################################################################
sub content_charset{
	my $me = shift;
	Encode::MIME::Name::get_mime_name(Encode::resolve_alias($me->web_charset()));
}

################################################################
sub dump {
	my $me = shift;
	my %list = $me->varlist('.*');
	my $ret = '';
	foreach my $k (sort keys %list) {
		if (defined $list{$k} && length($list{$k})) {
			my $value = $list{$k};
			$value =~ s/\x0D\x0A|\x0D|\x0A//g;
			$ret .= $k .' = '. $value . "\n"
		}
	}
	return $ret;
}

################################################################
# config params check utility functions
################################################################
sub on_error {
	my $msg = shift;
	die $msg;
}

sub valid_int {
	my ($name, $value) = @_;
	return 1 if ($value =~ /^\d+$/);

	if (my $val = eval $value) {
		return 1 if ($val =~ /^\d+$/);
	}

	return 0;
}

sub valid_web_root {
	my ($name, $value) = @_;
	return ($value =~ /^\//);
}

sub valid_web_schema {
	my ($name, $value) = @_;
	return ($value =~ /^https?$/);
}

sub valid_fontsize {
	my ($name, $value) = @_;
	return ($value =~ /^[+-]?[0-7]$/);
}

sub valid_charset {
	my ($name, $value) = @_;
	return Encode::resolve_alias($value);
}

sub valid_dir {
	my ($name, $value) = @_;
	if (-w $value) {
		return 1;
	} else {
		print STDERR "pid_dir $value is not writable\n";
		return 0;
	}
}

sub valid_url_target {
	my ($name, $value) = @_;
	return ($value =~ /^_(?:blank|self|top|parent)$/);
}

sub valid_log {
	my ($name, $value) = @_;
	return ($value =~ /^(?:(?:file|syslog|stdio)$|(?:file|syslog):)/);
}

sub valid_obsolates {
	my ($name, $value) = @_;
	warn($name . ' has obsoleted from keitairc 2.0, ignored');
	return 1;
}

1;
