window.addEventListener("load", event => {
  // 各種イベントハンドラを登録
  // 「インストール」ボタンをクリック→registerServiceWorker()を実行
  document.getElementById("install_svcworker_button")
    .addEventListener("click", registerServiceWorker);

  // 「インストール」ボタンをクリック→unregisterServiceWorker()を実行
  document.getElementById("uninstall_svcworker_button")
    .addEventListener("click", unregisterServiceWorker);

  // ページロード時にService Workerの登録状況をチェックする
  checkServiceWorkerRegistered();

  // メッセージ一覧を更新する
  reloadMessages();

  // 5秒おきにメッセージ一覧を更新する
  setInterval(reloadMessages, 5000);
});

// メッセージ一覧を更新する
function reloadMessages() {
  const elem = document.getElementById("messages");
  if (elem === null) {
    console.log("no #message element");
    return;
  }

  // 子ノードを全削除
  while (elem.firstChild) {
    elem.removeChild(elem.firstChild);
  }

  // メッセージを取得
  testDb.getMessages()
    .then(messages => {
      // 子ノードを作成
      for (const msg of messages) {
        const newElem = document.createElement("li");
        newElem.textContent = msg;
        elem.append(newElem);
      }
    })
    .catch(error => {
      console.log(error);
    });
}

// サーバー鍵をここで指定する
const VAP_APP_SERVER_KEY = "";

function registerServiceWorker() {
  // Service Workerを登録する
  if ("serviceWorker" in navigator) {
    navigator.serviceWorker.register("./svc_worker.js")
      .then(registration => {
        // 成功ログを出力する
        console.log("Service Worker registration succeed."
                    + `scope: ${registration.scope}`);
      })
      .catch(err => {
        // 失敗ログを出力する
        console.log(`Service Worker registration failed: ${err}`);
        checkServiceWorkerRegistered();
      });

    navigator.serviceWorker.ready.then(registration => {
      // Service Workerが利用可能になったら
      // Pushを受信する設定を行う
      const options = {
        userVisibleOnly: true,
        applicationServerKey: VAP_APP_SERVER_KEY,
      };
      registration.pushManager.subscribe(options)
        .then(pushSubscription => {
          console.log("subscribe succeed.");
          const prop = pushSubscription.toJSON();
          console.log("property: " + JSON.stringify(prop));
          
        })
        .catch(err => {
          console.log("subscribe failed: " + err);
        });
      checkServiceWorkerRegistered();
    });
  }
}

function unregisterServiceWorker() {
  // Service Workerの登録を解除する
  if ("serviceWorker" in navigator) {
    // Service Workerの登録情報を取得するために
    // まずServiceWorkerRegistrationオブジェクトを取得する
    navigator.serviceWorker.getRegistration()
      .then(registration => {
        // ServiceWorkerRegistrationオブジェクトを取得できた場合、
        // まずPush購読を停止する
        registration.pushManager.getSubscription()
          .then(subscription => {
            if (subscription) {
              subscription.unsubscribe()
                .then(result => {
                  console.log("unsubscribe succeed.");
                })
                .catch(err => {
                  console.log("unsubscribe error: " + err);
                });
            }
          })
          .catch(err => {
            console.log("getSubscription error: " + err);
          });
        // unregister()で登録解除を行う
        if (registration) {
          registration.unregister()
            .then(result => {
              if (result) {
                console.log("Service Worker unregister succeed.");
              }
              checkServiceWorkerRegistered();
            });
        }
      })
      .catch(err => {
        // 失敗ログを出力する
        console.log("Service Worker unregister failed.");
      });
  }
}

function checkServiceWorkerRegistered() {
  // Servie Workerが登録されているかチェックする
  navigator.serviceWorker.getRegistration()
    .then(registration => {
      // registrationが存在すれば登録済み、
      // undefinedなら未登録
      // 結果に応じてinstall-svcwもしくはuninstall-svcwを
      // 表示する
      const elInst = document.getElementById("install-svcw");
      const elUninst = document.getElementById("uninstall-svcw");
      if (registration) {
        elUninst.removeAttribute("style");
        elInst.setAttribute("style", "display:none;");
      } else {
        elInst.removeAttribute("style");
        elUninst.setAttribute("style", "display:none;");
      }
    });
}

