//
// database.js
//

var mongodb = require('mongodb');
var assert = require('assert');
var database = exports;

/**
 * データベースアクセス用基本クラス。
 * コレクションオブジェクトの作成など、基本的なデータベースアクセスメソッドを
 * 提供する
 */
var Database = function () {};

/**
 * データベースオブジェクトを取得する
 */
Database.prototype._open = function (database, callback) {
  var self = this;

  // すでにデータベースオブジェクトが作成されていた場合、再利用する
  if (this._db && !this._db.close) {
    callback(null, this._db);
    return;
  }
  var server = new mongodb.Server('127.0.0.1', 27017, {});
  var db_connector = new mongodb.Db(database, server, {safe:true});

  // データベースを開く
  db_connector.open(function (err, db) {
    if (err) {
      callback(err);
      return;
    }
    self._db = db;
    callback(null, db);
  });
}

/**
 * データベース接続をすべてクローズする
 */
Database.prototype.close = function() {
  if (this._db) {
    this._db.close();
    delete this._db;
  }
}

/**
 * コレクションにアクセスするためのオブジェクトを取得する
 */
Database.prototype._getCollection = function (collectionName, callback) {
  this._open('mobbs', function(err, db) {
    if (err) {
      callback(err);
      return;
    }
    db.createCollection(collectionName, callback);
  });
};

/**
 * トピックデータにアクセスするためのクラス
 */
var Topics = function () {};
Topics.prototype = new Database();

/**
 * Topicsクラスのインスタンスを生成する
 */
database.getTopics = function () {
  return new Topics();
}

/**
 * トピックを新しい順にソートし、start番目からend番目までのトピックを取得する
 */
Topics.prototype.getLatest = function (start, end, callback) {
  // topicsコレクションを取得する
  this._getCollection('topics', function (err, collection) {
    if (err) {
      callback(err);
      return;
    }
    var cursor = collection.find({});
    cursor.sort([['date', -1]]).limit(end - start).skip(start);
    cursor.toArray(callback);
  });
};

/**
 * トピックIDで指定されたトピックを返す
 */
Topics.prototype.findById = function (topicId, callback) {
  this._getCollection('topics', function (err, collection) {
    if (err) {
      callback(err);
      return;
    }
    collection.findOne({topicId: topicId}, callback);
  });
};

/**
 * トピックを新規作成する
 */
Topics.prototype.insert = function (topic, callback) {
  var self = this;

  // countersコレクションからカウンタの値を取得する
  self._getCollection('counters', function (err, collection) {
    if (err) {
      callback(err);
      return;
    }
    // counterの値を取得すると同時にインクリメントする
    collection.findAndModify({name:'topics'}, {}, {$inc: {count:1}}, createTopic);
  });

  // 取得したカウンタの値を作成するトピックのトピックIDとしてトピックを作成
  function createTopic(err, counter) {
    var nextId = counter.count;
    var newTopic = {
      topicId: nextId,
      title: topic.title,
      text: topic.text,
      date: new Date(),
      postBy: topic.postBy || '',
      relatedUrl: topic.relatedUrl || '',
      replies: []
    };
    self._getCollection('topics', function (err, collection) {
      if (err) {
        callback(err);
        return;
      }
      collection.insert(newTopic, function (err, obj) {
        if (err) {
          callback(err);
          return;
        }
        callback(err, obj[0]);
      });
    });
  }
};

/**
 * コメントデータにアクセスするためのオブジェクト
 */
var Comments = function () {};
Comments.prototype = new Database();

/**
 * Commentsクラスのインスタンスを生成する
 */
database.getComments = function () {
  return new Comments();
}

/**
 * コメントIDで指定したコメントを取得
 */
Comments.prototype.findById = function (commentId, callback) {
  this._getCollection('comments', function (err, collection) {
    if (err) {
      callback(err);
      return;
    }
    // commentId引数が数値の場合、コメントオブジェクトを返す
    if (typeof commentId === 'number') {
      collection.findOne({commentId: commentId}, callback);
    } else {
      // commentId引数が配列の場合、取得したコメントを配列で返す
      var cursor = collection.find({commentId: {$in: commentId}});
      cursor.sort([['date', -1]]);
      cursor.toArray(callback);
    }
  });
};

/**
 * コメントを新規作成してデータベースに挿入する
 */
Comments.prototype.insert = function (topicId, comment, callback) {
  var self = this;
  // countersコレクションからカウンタの値を取得する
  self._getCollection('counters', function (err, collection) {
    if (err) {
      callback(err);
      return;
    }
    collection.findAndModify({name:'comments'}, {}, {$inc: {count:1}}, createComment);
  });

  // 取得したカウンタの値をコメントIDとするコメントを作成
  function createComment(err, counter) {
    var nextId = counter.count;
    var newComment = {
      commentId: nextId,
      topicId: topicId,
      title: comment.title,
      text: comment.text,
      date: new Date(),
      postBy: comment.postBy || '',
      relatedUrl: comment.relatedUrl || '',
      replies: [],
      parentCommentId: comment.parentCommentId || null
    };

    // 親コメントが指定されていない場合、トピックのrepliesフィールドに
    // コメントIDを追加する
    if (newComment.parentCommentId === null) {
      appendReplyToTopic(newComment, function (err) {
        if (err) {
          callback(err);
          return;
        }
        insertComment(newComment);
      });
    } else {
      // 親コメントが指定されている場合は親コメントのrepliesフィールドに
      // コメントIDを追加する
      appendReplyToComment(newComment, function (err) {
        if (err) {
          callback(err);
          return;
        }
        insertComment(newComment);
      });
    }
  }

  // 指定したコメントのrepliesフィールドに指定したコメントIDを追加する
  function appendReplyToComment(comment, callback) { 
    var parentId = comment.parentCommentId;
    var childId = comment.commentId;
    self._getCollection('comments', function (err, collection) {
      if (err) {
        callback(err);
        return;
      }
      collection.findAndModify( {commentId:parentId},
                                {},
                                {$push: {replies:childId}},
                                callback);
    });
  };

  // 指定したトピックのreplesフィールドに指定したコメントIDを追加する
  function appendReplyToTopic(comment, callback) {
    var parentId = comment.topicId;
    var childId = comment.commentId;
    self._getCollection('topics', function (err, collection) {
      if (err) {
        callback(err);
        return;
      }
      collection.findAndModify( {topicId:parentId},
                                {},
                                {$push: {replies:childId}},
                                callback);
    });
  };

  // コメントをデータベースに挿入する
  function insertComment(newComment) {
    self._getCollection('comments', function (err, collection) {
      if (err) {
        callback(err);
        return;
      }
      collection.insert(newComment, {}, function (err, obj) {
        if (err) {
          callback(err);
          return;
        }
        // 挿入したオブジェクトを引数に与えてコールバック関数を実行する
        callback(err, obj[0]);
      });
    });
  }
};

/**
 * トピックに付けられたコメントをツリー状のオブジェクトで取得する
 */
Comments.prototype.getCommentTree = function (topicId, callback) {
  var self = this;

  var topics = database.getTopics();
  topics.findById(topicId, function (err, topic) {
    topics.close();
    if (err) {
      callback(err);
      return;
    }
    self._getCollection('comments', function (err, collection) {
      if (err) {
        callback(err);
        return;
      }
      var items = {};
      var cursor = collection.find({topicId:topicId});
      cursor.sort([['date', -1]]);
      cursor.each(function (err, comment) {
        if (err) {
          callback(err);
          return;
        }
        if (comment !== null) {
          items[comment.commentId] =  comment;
        } else {
          // ループ終了時の処理
          var tree = buildCommentTree(topic.replies, items, []);
          callback(null, tree);
        }
      });
    });
  });

  function buildCommentTree(parents, comments, tree) {
    var comment;
    for (var i = 0; i < parents.length; i++) {
      comment = comments[parents[i]];
      comment.children = [];
      buildCommentTree(comment.replies, comments, comment.children);
      tree.push(comment);
    }
    return tree;
  }
};
