#!/usr/local/bin/node

var readline = require('readline');

var auth = require('./auth');
var tweetWriter = require('./tweetwriter');

var rl = null;
var twitter = null;

// モードを表す疑似クラス
var mode = {
  current: null,
  start: function (mode) {
    if (this.current) {
      this.current.exit();
    }
    this.current = mode;
    this.current.start();
  }
}

// timelineモードを定義
var timeline = {
  start: function () {
    if (rl !== null) {
      rl.close();
      rl = null;
    }
    tweetWriter.resume();
    process.stdin.once('data', function () {
      // commandモードに遷移
      mode.start(command);
    });
    process.stdin.resume();
  },
  exit: function () {
    tweetWriter.pause();
  }
};

// commandモードを定義
var command = {
  start: function () {
    rl = readline.createInterface({
      input: process.stdin,
      output: process.stdout,
      completer: function (line) {
        var completions = ['tweet', 'quit'];
        var hits = completions.filter(function(c) {
          return c.indexOf(line) == 0;
        });
        // show all completions if none found
        return [hits.length ? hits : completions, line];
      }
    });
    rl.on('line', this.line);
    rl.on('SIGINT', this.SIGINT);
    rl.setPrompt('> ');
    rl.prompt();
  },
  exit: function () {
  },
  line: function (data) {
    // 入力された文字列に応じて処理を切り替える
    switch(data) {
    case 'tweet':
      // writeMessageモードを開始する
      mode.start(writeMessage);
      break;
    case 'quit':
      // アプリケーションを終了する
      process.exit(0);
    case '':
      // timelineモードに遷移する
      mode.start(timeline);
      break;
    default:
      // エラーメッセージを表示してtimelineモードに遷移する
      process.stdout.write('invalid command: ' + data + '\n');
      mode.start(timeline);
      break;
    }
  },
  SIGINT: function () {
    // Ctrl-Cを受け付けたらtimelineモードに遷移する
    mode.start(timeline);
  }
}

// writeMessageモードを定義
var newMessage = '';
var writeMessage = {
  start: function () {
    newMessage = '';

    // readlineインターフェイスのイベントハンドラの付け替えを行う
    rl.removeAllListeners('line');
    rl.removeAllListeners('SIGINT');
    rl.on('line', this.line);
    rl.on('SIGINT', this.SIGINT);

    rl.setPrompt('tweet> ');
    process.stdout.write('write tweet message and push Ctrl-C.\n');
    rl.prompt();
  },
  exit: function () {
  },
  line: function (data) {
    // 入力したメッセージを保存しておく
    if (newMessage === '') {
      newMessage = data;
    } else {
      newMessage = newMessage + '\n' + data;
    }
    rl.prompt();
  },
  SIGINT: function () {
    // Ctrl-Cを受け付けたらメッセージの入力を終了する
    process.stdout.write('\n----\n');
    process.stdout.write(newMessage + '\n');
    process.stdout.write('----\n');

    // メッセージを表示し、投稿するかどうかを確認する
    var message = 'tweet this message? (y:yes/n:no) ';
    rl.question(message, onAnswer);

    function onAnswer(answer) {
      if (answer === 'y') {
        twitter.updateStatus(newMessage, {}, function (err, data) {
        });
        process.stdout.write('tweet done.\n\n');
        mode.start(timeline);
      } else if (answer === 'n') {
        process.stdout.write('tweet canceled.\n\n');
        mode.start(timeline);
      } else {
        rl.question(message, onAnswer);
      }
    }

  }
}

// ntwitter.Twitterクラスのインスタンスを生成して処理を開始する
auth.getClient(function (err, twit) {
  twitter = twit;

  // タイムライン最新20件を取得する
  twitter.getHomeTimeline({}, function (err, data) {
    if (err) {
      throw err;
    }
    tweetWriter.write(data.reverse());
  });

  // User streamを使ったタイムライン受信を開始する
  twitter.stream('user', {}, function (stream) {
    stream.on('data', function (data) {
      tweetWriter.write(data);
    });
    stream.on('error', function(err) {
      console.log(err);
    });
  });

  // タイムラインモードを開始
  mode.start(timeline);
});

