#!/usr/bin/env python
# -*- coding: utf-8 -*-
import argparse
import os

import tensorflow as tf

def main():
    # 引数をパースする
    p = argparse.ArgumentParser(description='convert images to TFRecord format')
    p.add_argument('dimension',
                   type=lambda x: [int(r) for r in x.split("x")],
                   help='image dimension. example: 10x10')
    p.add_argument('label',
                   type=int,
                   help='label.')
    p.add_argument('output',
                   #type=argparse.FileType('w'),
                   help='output file')
    p.add_argument('target_dir',
                   nargs='+',
                   help='target directory')
    args = p.parse_args()
    if len(args.dimension) != 2:
        raise argparse.ArgumentTypeError("dimension must be <num>x<num>") 

    # 計算グラフを構築する
    (width, height) = args.dimension
    (filepath, padded_image) = _create_comp_graph(width, height)

    # 出力先の用意
    writer = tf.python_io.TFRecordWriter(args.output)
      
    # セッションを実行
    sess = tf.Session()

    # 指定したディレクトリ内のファイルを列挙する
    for dirname in args.target_dir:
        for filename in os.listdir(dirname):
            pathname = os.path.join(dirname, filename)
            print("process {}...".format(pathname))
            try:
                image = sess.run(padded_image, {filepath: pathname})
            except tf.errors.InvalidArgumentError:
                # 読み込みに失敗したらメッセージを出力して継続する
                print("{}: invalid jpeg file. ignored.".format(pathname))
                continue

            features = tf.train.Features(feature={
                'label':     tf.train.Feature(int64_list=tf.train.Int64List(value=[args.label])),
                'height':    tf.train.Feature(int64_list=tf.train.Int64List(value=[height])),
                'width':     tf.train.Feature(int64_list=tf.train.Int64List(value=[width])),
                'raw_image': tf.train.Feature(float_list=tf.train.FloatList(value=image.reshape(width*height*3))),
            })
            example = tf.train.Example(features=features)
            writer.write(example.SerializeToString())

    # 終了
    writer.close()
    print("done")

# 指定したファイルを読み出して指定したサイズにリサイズする
def _create_comp_graph(width, height):
    filepath = tf.placeholder(tf.string, name="pathname")
    content = tf.read_file(filepath)
    raw_image = tf.image.decode_jpeg(content, channels=3)
    # raw_imageのデータ型はuint8

    # 画像のサイズを[height, width, channels]の形で取得
    shape = tf.shape(raw_image)
    raw_height = tf.to_float(tf.slice(shape, [0], [1]))
    raw_width = tf.to_float(tf.slice(shape, [1], [1]))

    # 画像サイズと出力サイズのアスペクト比を比較し、
    # 画像サイズのほうが大きければ高さを、
    # 小さければ幅を出力サイズにそろえるよう
    # 拡大縮小比を求める
    # 幅/高さは整数（int）型データなので、適宜float型に変換する
    aspect_ratio = float(height) / width
    raw_aspect_ratio = raw_height / raw_width
    scale = tf.cond(tf.reduce_any(raw_aspect_ratio > [aspect_ratio]),
                     lambda: [height] / raw_height,
                     lambda: [width] / raw_width
    )

    # 求めた比率で画像をリサイズする
    # resized_imageはfloat32型となる
    new_size = tf.to_int32(tf.concat([raw_height, raw_width], 0) * scale)
    resized_image = tf.image.resize_images(raw_image, new_size)

    # 余白を0で埋めて指定したサイズにそろえる
    padded_image = tf.image.resize_image_with_crop_or_pad(
        resized_image,
        height,
        width
    )

    return (filepath, padded_image)


if __name__ == "__main__":
    main()


