/*
 * Created by Sebastian Bugiu on 4/9/23, 10:12 PM
 * sebastian.bugiu@headwayentertainment.net
 * Last modified 1/26/22, 6:23 PM
 * Copyright (c) 2023.
 * All rights reserved.
 */

package com.headwayent.blackholedarksun;

import org.robovm.apple.foundation.Foundation;
import org.robovm.apple.foundation.NSAutoreleasePool;
import org.robovm.apple.foundation.NSDictionary;
import org.robovm.apple.foundation.NSException;
import org.robovm.apple.foundation.NSThread;
import org.robovm.apple.glkit.GLKViewDrawableColorFormat;
import org.robovm.apple.glkit.GLKViewDrawableDepthFormat;
import org.robovm.apple.uikit.UIApplication;

import com.badlogic.gdx.backends.iosrobovm.IOSApplication;
import com.badlogic.gdx.backends.iosrobovm.IOSApplicationConfiguration;
import com.badlogic.gdx.graphics.glutils.HdpiMode;
import com.badlogic.gdx.pay.ios.apple.PurchaseManageriOSApple;

import java.io.PrintWriter;
import java.io.StringWriter;

import headwayent.blackholedarksun.BlackholeDarksunMain;
import headwayent.blackholedarksun.MainApp;
import headwayent.hotshotengine.ApplicationStartSettings;

public class IOSLauncher extends IOSApplication.Delegate {

//    static {
//        System.loadLibrary("native-lib");
//    }

    @Override
    protected IOSApplication createApplication() {
        IOSApplicationConfiguration config = new IOSApplicationConfiguration();
        MainApp.PLATFORM = MainApp.Platform.IOS;
//        config.r = 8;
//        config.g = 8;
//        config.b = 8;
//        config.a = 8;
//        config.depth = 24;
//        config.stencil = 8;
//        config.useWakelock = true;
//        config.hideStatusBar = true;
//        config.useImmersiveMode = true;
        config.colorFormat = GLKViewDrawableColorFormat.RGBA8888;
        config.depthFormat = GLKViewDrawableDepthFormat._24;
        config.hdpiMode = HdpiMode.Pixels;

        ApplicationStartSettings applicationStartSettings = new ApplicationStartSettings();
        // We don't use this on IOS.
//        applicationStartSettings.uncaughtExceptionHandler = new IOSUncaughtExceptionHandler();
        registerDefaultJavaUncaughtExceptionHandler();
        applicationStartSettings.applicationMode = MainApp.Mode.CLIENT;
        applicationStartSettings.iosConfig = config;

        PurchaseManageriOSApple purchaseManager = new PurchaseManageriOSApple();
        applicationStartSettings.purchaseManager = purchaseManager;

        IOSApplication iosApplication = new IOSApplication(new BlackholeDarksunMain(null, applicationStartSettings), config);
        return iosApplication;
    }

    /**
     * Registers a default java uncaught exception handler that forwards exceptions to RoboVM's signal handlers.
     * Use this if you want Java exceptions to be logged by crash reporters.
     */
    public static void registerDefaultJavaUncaughtExceptionHandler() {
        Thread.setDefaultUncaughtExceptionHandler(new java.lang.Thread.UncaughtExceptionHandler() {
            @Override
            public void uncaughtException (Thread thread, Throwable ex) {
                System.out.println("UncaughtExceptionHandler called");
                StringWriter sw = new StringWriter();
                PrintWriter pw = new PrintWriter(sw);
                ex.printStackTrace(pw);
                pw.flush();
                Foundation.log(sw.toString());
                NSException exception = new NSException(ex.getClass().getName(), sw.toString(), new NSDictionary<>());
                if (NSThread.getCurrentThread().isMainThread()) {
                    exception.raise();
                } else {
//                    long handler = getUncaughtExceptionHandler();
//                    callUncaughtExceptionHandler(handler, exception);
                    new IOSUncaughtExceptionHandler().uncaughtException(thread, ex);
                    // We should never get to this line!
                }
            }
        });
        System.out.println("UncaughtExceptionHandler set");
    }


    public static void main(String[] argv) {
        NSAutoreleasePool pool = new NSAutoreleasePool();
        UIApplication.main(argv, null, IOSLauncher.class);
        pool.close();
    }
}