/*
 * Created by Sebastian Bugiu on 4/9/23, 10:11 PM
 * sebastian.bugiu@headwayentertainment.net
 * Last modified 5/14/16, 11:47 AM
 * Copyright (c) 2023.
 * All rights reserved.
 */

package headwayent.hotshotengine.scriptcompiler;

import headwayent.hotshotengine.ENG_Bitmap;
import headwayent.hotshotengine.ENG_Utility;
import headwayent.hotshotengine.renderer.ENG_Image;
import headwayent.hotshotengine.renderer.ENG_PixelUtil.PixelFormat;
import headwayent.hotshotengine.resource.ENG_PVRTexture;
import headwayent.hotshotengine.resource.ENG_PVRTexture.PVRTextureFlagType;
import headwayent.hotshotengine.resource.ENG_Resource;

import java.io.File;
import java.nio.ByteBuffer;
import java.util.ArrayList;

import org.apache.commons.io.FilenameUtils;

import com.badlogic.gdx.graphics.Pixmap;
import com.badlogic.gdx.graphics.Pixmap.Format;

public class ENG_TextureCompiler {

    private static final String NAME_PVR = "pvr";
    private static final String NAME_JPG = "jpg";
    private static final String NAME_PNG = "png";

    public static void parsePVRTexture(ENG_Resource res, ArrayList<ENG_Image> imageList) {
        ENG_PVRTexture pvrTex = new ENG_PVRTexture(res.getId());
        ArrayList<ArrayList<ByteBuffer>> data = new ArrayList<>();
        data.add(pvrTex.getImageData());
        ENG_Image img = new ENG_Image();
        PixelFormat pf = null;
        if (pvrTex.getInternalFormat() ==
                PVRTextureFlagType.kPVRTextureFlagTypePVRTC_2.getType()) {
            if (pvrTex.hasAlpha()) {
                pf = PixelFormat.PF_PVRTC_RGBA2;
            } else {
                pf = PixelFormat.PF_PVRTC_RGB2;
            }
        } else if (pvrTex.getInternalFormat() ==
                PVRTextureFlagType.kPVRTextureFlagTypePVRTC_4.getType()) {
            if (pvrTex.hasAlpha()) {
                pf = PixelFormat.PF_PVRTC_RGBA4;
            } else {
                pf = PixelFormat.PF_PVRTC_RGB4;
            }
        }
        img.loadDynamicImage(data, pvrTex.getWidth(), pvrTex.getHeight(), 1, pf, false, 1, pvrTex.getImageData().size() - 1);
        imageList.add(img);
    }

    public static void parseBitmapTexture(ENG_Resource res, ArrayList<ENG_Image> imageList) {
        ENG_Image img = new ENG_Image();
        ENG_Bitmap bitmap;
        if (res.getFilename() != null && !res.getFilename().isEmpty()) {
            bitmap = new ENG_Bitmap(res.getName(), res.getFilename());
        } else {
            bitmap = new ENG_Bitmap(res.getId(), ENG_CompilerUtil.trimFile(res.getName()));
        }
        ArrayList<ArrayList<ByteBuffer>> data = new ArrayList<>();
        data.add(new ArrayList<ByteBuffer>());
//		Bitmap.Config cfg = bitmap.getBitmap().getConfig();
        Format format = bitmap.getBitmap().getFormat();
        //bitmap.getBitmap().getTextureData().getFormat();
        int pixelSize;
        PixelFormat srcPf = null;
        PixelFormat destPf;
        //Android is still fantastic! Bitmaps use ARGB while OpenGL uses RGBA!!!
        //Apparently it's just a mistake...Android does use RGBA even if written wrong
        switch (format) {
            case Alpha:
                pixelSize = 1;
                destPf = PixelFormat.PF_A8;
                break;
            case RGBA4444:
                pixelSize = 2;
                destPf = PixelFormat.PF_R8G8B8A8;
                break;
            case RGBA8888:
                pixelSize = 4;
                destPf = PixelFormat.PF_R8G8B8A8;
                break;
            case RGB565:
                pixelSize = 2;
                destPf = PixelFormat.PF_R5G6B5;
                break;
            case RGB888:
                pixelSize = 3;
                destPf = PixelFormat.PF_B8G8R8;
                break;
            default:
                //Should never get here???
                throw new IllegalArgumentException("Unknown pixel format");
        }
        int sizeInBytes = pixelSize * bitmap.getBitmap().getWidth() * bitmap.getBitmap().getHeight();
        ByteBuffer src = ENG_Utility.allocateDirect(sizeInBytes);
//		ByteBuffer dest = ENG_Utility.allocateDirect(sizeInBytes);
        Pixmap bmp = bitmap.getBitmap();
//        ENG_PixelBox srcBox = new ENG_PixelBox(bmp.getWidth(), bmp.getHeight(), 1, srcPf, src);
//		ENG_PixelBox destBox = new ENG_PixelBox(bmp.getWidth(), bmp.getHeight(),
//				1, destPf, dest);
//		bitmap.getBitmap().getTextureData().prepare();
//		src.put(bitmap.getBitmap().getTextureData().consumePixmap().getPixels());//copyPixelsToBuffer(src);
//		bitmap.getBitmap().getTextureData().consumePixmap().getPixels().rewind();
        src.put(bmp.getPixels());
        bmp.getPixels().rewind();
        src.flip();
        //	ENG_PixelUtil.bulkPixelConversion(srcBox, destBox);
//		dest.flip();
        //	src.flip();
        //Now we should have the converted bitmap (to RGBA)
        data.get(0).add(src);
//		for (int i = 0; i < sizeInBytes; ++i) {
//			System.out.println(src.get(i));
//		}
        img.loadDynamicImage(data, bitmap.getBitmap().getWidth(), bitmap.getBitmap().getHeight(), 1, destPf, false, 1, 0);
        imageList.add(img);
        bitmap.destroy();
    }

    public static ArrayList<ENG_Image> compileTexture(String fileName, String path, boolean fromSDCard) {
        ArrayList<ENG_Image> imageList = new ArrayList<>();
        if (fromSDCard) {
            // The resource needs the full path and filename
            // in order to extract its internal path
            parseTexture(imageList, new ENG_Resource(FilenameUtils.getName(fileName), path + File.separator + fileName));
        } else {
            throw new UnsupportedOperationException();
//			String file = ENG_CompilerUtil.trimFile(fileName);
            //	String ext = ENG_CompilerUtil.getExtension(file);
//			int fileHandle = ENG_Resource.getFileHandle(file, path);

//			parseTexture(imageList, new ENG_Resource(fileName, fileHandle));
        }
        return imageList;
    }

    /**
     * In the file the texture must have extension
     *
     * @param fileName
     * @param path
     * @return
     */
//	public static ArrayList<ENG_Image> compileTextureFromFile(
//			String fileName, String path) {
//		ArrayList<ENG_Resource> resList = ENG_Resource.getResourceID(fileName, path);
//		ArrayList<ENG_Image> imageList = new ArrayList<ENG_Image>();
//		for (ENG_Resource res : resList) {			
//			parseTexture(imageList, res);
//		}
//		return imageList;
//	}
    private static void parseTexture(ArrayList<ENG_Image> imageList, ENG_Resource res) {
        if (res.getName().endsWith(NAME_PVR)) {
            parsePVRTexture(res, imageList);
        } else if (res.getName().endsWith(NAME_JPG) || res.getName().endsWith(NAME_PNG)) {
            parseBitmapTexture(res, imageList);
        }
    }
}
