/*
 * Created by Sebastian Bugiu on 4/9/23, 10:11 PM
 * sebastian.bugiu@headwayentertainment.net
 * Last modified 5/17/21, 9:20 PM
 * Copyright (c) 2023.
 * All rights reserved.
 */

package headwayent.hotshotengine.scriptcompiler;

import headwayent.hotshotengine.basictypes.ENG_Integer;
import headwayent.hotshotengine.exception.ENG_InvalidFormatParsingException;
import headwayent.hotshotengine.exception.ENG_UndeclaredIdentifierException;
import headwayent.hotshotengine.renderer.ENG_BlendMode.SceneBlendFactor;
import headwayent.hotshotengine.renderer.ENG_BlendMode.SceneBlendOperation;
import headwayent.hotshotengine.renderer.ENG_ColorValue;
import headwayent.hotshotengine.renderer.ENG_PixelUtil.PixelFormat;
import headwayent.hotshotengine.renderer.ENG_TextureUnitState.TextureAddressingMode;
import headwayent.hotshotengine.resource.ENG_CompiledResource;
import headwayent.hotshotengine.resource.ENG_MaterialResource;
import headwayent.hotshotengine.resource.ENG_PassResource;
import headwayent.hotshotengine.resource.ENG_Resource;
import headwayent.hotshotengine.resource.ENG_Shader;
import headwayent.hotshotengine.resource.ENG_ShaderNamedParams;
import headwayent.hotshotengine.resource.ENG_ShaderParam;
import headwayent.hotshotengine.resource.ENG_TechniqueResource;
import headwayent.hotshotengine.resource.ENG_TextureUnitResource;
import headwayent.hotshotengine.resource.ENG_TextureUnitResource.BindingType;
import headwayent.hotshotengine.resource.ENG_TextureUnitResource.ColourOpType;
import headwayent.hotshotengine.resource.ENG_TextureUnitResource.ContentType;
import headwayent.hotshotengine.resource.ENG_TextureUnitResource.CubeType;
import headwayent.hotshotengine.resource.ENG_TextureUnitResource.EnvMapType;
import headwayent.hotshotengine.resource.ENG_TextureUnitResource.FilterType;
import headwayent.hotshotengine.resource.ENG_TextureUnitResource.TextureUsage;
import headwayent.hotshotengine.resource.shadertype.ENG_ShaderTypeBool;
import headwayent.hotshotengine.resource.shadertype.ENG_ShaderTypeCustom;
import headwayent.hotshotengine.resource.shadertype.ENG_ShaderTypeFloat;
import headwayent.hotshotengine.resource.shadertype.ENG_ShaderTypeInt;
import headwayent.hotshotengine.resource.shadertype.ENG_ShaderTypeStandard;
import headwayent.hotshotengine.resource.shadertype.ENG_ShaderTypeStandard.NamedParamType;

import java.io.DataInputStream;
import java.util.ArrayList;
import java.util.HashMap;

public class ENG_MaterialCompiler extends ENG_AbstractCompiler<ENG_CompiledResource> {

    protected static boolean init;

    private static final String VERTEX_PROGRAM = "vertex_program";
    private static final String FRAGMENT_PROGRAM = "fragment_program";
    private static final String VERTEX_PROGRAM_REF = "vertex_program_ref";
    private static final String FRAGMENT_PROGRAM_REF = "fragment_program_ref";
    private static final String TEXTURE_UNIT = "texture_unit";
    private static final String MATERIAL = "material";
    private static final String TECHNIQUE = "technique";
    private static final String PASS = "pass";
    //Shader variable types
    private static final String INT = "int";
    private static final String FLOAT = "float";
    private static final String BOOL = "bool";
    private static final String VEC2 = "vec2";
    private static final String VEC3 = "vec3";
    private static final String VEC4 = "vec4";
    private static final String IVEC2 = "ivec2";
    private static final String IVEC3 = "ivec3";
    private static final String IVEC4 = "ivec4";
    private static final String BVEC2 = "bvec2";
    private static final String BVEC3 = "bvec3";
    private static final String BVEC4 = "bvec4";
    private static final String MAT2 = "mat2";
    private static final String MAT3 = "mat3";
    private static final String MAT4 = "mat4";

    private static final int FLOAT_PARSE = 1;
    private static final int VEC2_PARSE = 2;
    private static final int VEC3_PARSE = 3;
    private static final int VEC4_PARSE = 4;
    private static final int BOOL_PARSE = 1;
    private static final int BVEC2_PARSE = 2;
    private static final int BVEC3_PARSE = 3;
    private static final int BVEC4_PARSE = 4;
    private static final int INT_PARSE = 1;
    private static final int IVEC2_PARSE = 2;
    private static final int IVEC3_PARSE = 3;
    private static final int IVEC4_PARSE = 4;
    private static final int MAT2_PARSE = 4;
    private static final int MAT3_PARSE = 9;
    private static final int MAT4_PARSE = 16;

    //Vertex and fragment program
    private static final String SOURCE = "source";
    private static final String DEFAULT_PARAMS = "default_params";
    private static final String PARAM_NAMED = "param_named";
    private static final String PARAM_NAMED_AUTO = "param_named_auto";
    private static final String CUSTOM = "custom";
    private static final String INCLUDES_SKELETAL_ANIMATION =
            "includes_skeletal_animation";
    private static final String INCLUDES_MORPH_ANIMATION =
            "includes_morph_animation";
    private static final String INCLUDES_POSE_ANIMATION =
            "includes_pose_animation";
    private static final String USES_VERTEX_TEXTURE_FETCH =
            "uses_vertex_texture_fetch";
    private static final String USES_ADJACENCY_INFORMATION =
            "uses_adjacency_information";

    private static final String ON = "on";
    private static final String OFF = "off";

    //Pass scene blending factors
    private static final String SB_ONE = "one";
    private static final String SB_ZERO = "zero";
    private static final String SB_DEST_COLOUR = "dest_colour";
    private static final String SB_SRC_COLOUR = "src_colour";
    private static final String SB_ONE_MINUS_DEST_COLOUR = "one_minus_dest_colour";
    private static final String SB_ONE_MINUS_SRC_COLOUR = "one_minus_src_colour";
    private static final String SB_DEST_ALPHA = "dest_alpha";
    private static final String SB_SRC_ALPHA = "src_alpha";
    private static final String SB_ONE_MINUS_DEST_ALPHA = "one_minus_dest_alpha";
    private static final String SB_ONE_MINUS_SRC_ALPHA = "one_minus_src_alpha";

    //Pass scene blending ops
    private static final String SB_OP_ADD = "add";
    private static final String SB_OP_SUBSTRACT = "substract";
    private static final String SB_OP_REVERSE_SUBSTRACT = "reverse_substract";
    private static final String SB_OP_MIN = "min";
    private static final String SB_OP_MAX = "max";

    //Texture unit
    private static final String TEXTURE_ALIAS = "texture_alias";
    private static final String TEXTURE = "texture";
    private static final String ANIM_TEXTURE = "anim_texture";
    private static final String CUBIC_TEXTURE = "cubic_texture";
    private static final String BINDING_TYPE = "binding_type";
    private static final String CONTENT_TYPE = "content_type";
    private static final String TEX_COORD_SET = "tex_coord_set";
    private static final String TEX_ADDRESS_MODE = "tex_address_mode";
    private static final String TEX_BORDER_COLOUR = "tex_border_colour";
    private static final String FILTERING = "filtering";
    private static final String MAX_ANISOTROPY = "max_anisotropy";
    private static final String MIPMAP_BIAS = "mipmap_bias";
    private static final String COLOUR_OP = "colour_op";
    private static final String COLOUR_OP_EX = "colour_op_ex";
    private static final String COLOUR_OP_MULTIPASS_FALLBACK = "colour_op_multipass_fallback";
    private static final String ALPHA_OP_EX = "alpha_op_ex";
    private static final String ENV_MAP = "env_map";
    private static final String SCROLL = "scroll";
    private static final String SCROLL_ANIM = "scroll_anim";
    private static final String ROTATE = "rotate";
    private static final String ROTATE_ANIM = "rotate_anim";
    private static final String SCALE = "scale";
    private static final String WAVE_XFORM = "wave_xform";
    private static final String TRANSFORM = "transform";

    // Texture unit -> texture
    private static final String TEXTURE_NAME_TYPE = "type";
    private static final String TEXTURE_NAME_NUM_MIPMAPS = "num_mipmaps";
    private static final String TEXTURE_NAME_NUM_MIPMAPS_UNLIMITED = "unlimited";
    private static final String TEXTURE_NAME_GAMMA = "gamma";
    private static final String TEXTURE_NAME_USE_SHADOW_BUFFER = "shadow_buffer";
    private static final String TEXTURE_NAME_TEXTURE_USAGE = "texture_usage";

    private static final String PF_L8 = "PF_L8";
    private static final String PF_L16 = "PF_L16";
    private static final String PF_A8 = "PF_A8";
    private static final String PF_A4L4 = "PF_A4L4";
    private static final String PF_BYTE_LA = "PF_BYTE_LA";
    private static final String PF_R5G6B5 = "PF_R5G6B5";
    private static final String PF_B5G6R5 = "PF_B5G6R5";
    private static final String PF_R3G3B2 = "PF_R3G3B2";
    private static final String PF_A4R4G4B4 = "PF_A4R4G4B4";
    private static final String PF_A1R5G5B5 = "PF_A1R5G5B5";
    private static final String PF_R8G8B8 = "PF_R8G8B8";
    private static final String PF_B8G8R8 = "PF_B8G8R8";
    private static final String PF_A8R8G8B8 = "PF_A8R8G8B8";
    private static final String PF_A8B8G8R8 = "PF_A8B8G8R8";
    private static final String PF_B8G8R8A8 = "PF_B8G8R8A8";
    private static final String PF_R8G8B8A8 = "PF_R8G8B8A8";
    private static final String PF_X8R8G8B8 = "PF_X8R8G8B8";
    private static final String PF_X8B8G8R8 = "PF_X8B8G8R8";
    private static final String PF_A2R10G10B10 = "PF_A2R10G10B10";
    private static final String PF_A2B10G10R10 = "PF_A2B10G10R10";
    private static final String PF_FLOAT16_R = "PF_FLOAT16_R";
    private static final String PF_FLOAT16_RGB = "PF_FLOAT16_RGB";
    private static final String PF_FLOAT16_RGBA = "PF_FLOAT16_RGBA";
    private static final String PF_FLOAT32_R = "PF_FLOAT32_R";
    private static final String PF_FLOAT32_RGB = "PF_FLOAT32_RGB";
    private static final String PF_FLOAT32_RGBA = "PF_FLOAT32_RGBA";
    private static final String PF_SHORT_RGBA = "PF_SHORT_RGBA";

    //Texture unit -> cubic
    private static final String CUBIC_COMBINEDUVW = "combinedUVW";
    private static final String CUBIC_SEPARATEUV = "separateUV";

    //Texture unit -> binding type
    private static final String BINDING_VERTEX = "vertex";
    private static final String BINDING_FRAGMENT = "fragment";

    //Texture unit -> content type
    private static final String CONTENT_NAMED = "named";
    private static final String CONTENT_SHADOW = "shadow";
    private static final String CONTENT_COMPOSITOR = "compositor";

    //Texture unit -> address mode
    private static final String WRAP_MODE_WRAP = "wrap";
    private static final String WRAP_MODE_CLAMP = "clamp";
    private static final String WRAP_MODE_MIRROR = "mirror";
    private static final String WRAP_MODE_BORDER = "border";

    //Texture unit -> filtering
    private static final String FILTERING_NONE = "none";
    private static final String FILTERING_POINT = "point";
    private static final String FILTERING_LINEAR = "linear";
    private static final String FILTERING_ANISOTROPIC = "anisotropic";

    //Texture unit -> colour op
    private static final String COLOUR_OP_REPLACE = "replace";
    private static final String COLOUR_OP_ADD = "add";
    private static final String COLOUR_OP_MODULATE = "modulate";
    private static final String COLOUR_OP_ALPHA_BLEND = "alpha_blend";

    //Texture unit -> env map
    private static final String ENV_MAP_OFF = "off";
    private static final String ENV_MAP_SPHERICAL = "spherical";
    private static final String ENV_MAP_PLANAR = "planar";
    private static final String ENV_MAP_CUBIC_REFLECTION = "cubic_reflection";
    private static final String ENV_MAP_CUBIC_NORMAL = "cubic_normal";

    //Texture unit -> texture usage
    private static final String TU_STATIC = "static";
    private static final String TU_DYNAMIC = "dynamic";
    private static final String TU_WRITE_ONLY = "write_only";
    private static final String TU_STATIC_WRITE_ONLY = "static_write_only";
    private static final String TU_DYNAMIC_WRITE_ONLY = "dynamic_write_only";
    private static final String TU_DYNAMIC_WRITE_ONLY_DISCARDABLE = "dynamic_write_only_discardable";
    private static final String TU_AUTOMIPMAP = "automipmap";
    private static final String TU_RENDERTARGET = "rendertarget";
    private static final String TU_DEFAULT = "default";


    //Param named auto attribute names
    private static final String ACT_WORLD_MATRIX = "world_matrix";
    private static final String ACT_INVERSE_WORLD_MATRIX = "inverse_world_matrix";
    private static final String ACT_TRANSPOSE_WORLD_MATRIX = "transpose_world_matrix";
    private static final String ACT_INVERSE_TRANSPOSE_WORLD_MATRIX =
            "inverse_transpose_world_matrix";
    private static final String ACT_VIEW_MATRIX = "view_matrix";
    private static final String ACT_INVERSE_VIEW_MATRIX = "inverse_view_matrix";
    private static final String ACT_TRANSPOSE_VIEW_MATRIX = "transpose_view_matrix";
    private static final String ACT_INVERSE_TRANSPOSE_VIEW_MATRIX =
            "inverse_transpose_view_matrix";
    private static final String ACT_PROJECTION_MATRIX = "projection_matrix";
    private static final String ACT_INVERSE_PROJECTION_MATRIX =
            "inverse_projection_matrix";
    private static final String ACT_TRANSPOSE_PROJECTION_MATRIX =
            "transpose_projection_matrix";
    private static final String ACT_INVERSE_TRANSPOSE_PROJECTION_MATRIX =
            "inverse_transpose_projection_matrix";
    private static final String ACT_WORLDVIEW_MATRIX = "worldview_matrix";
    private static final String ACT_INVERSE_WORLDVIEW_MATRIX =
            "inverse_worldview_matrix";
    private static final String ACT_TRANSPOSE_WORLDVIEW_MATRIX =
            "transpose_worldview_matrix";
    private static final String ACT_INVERSE_TRANSPOSE_WORLDVIEW_MATRIX =
            "inverse_transpose_worldview_matrix";
    private static final String ACT_VIEWPROJ_MATRIX = "viewproj_matrix";
    private static final String ACT_INVERSE_VIEWPROJ_MATRIX = "inverse_viewproj_matrix";
    private static final String ACT_TRANSPOSE_VIEWPROJ_MATRIX =
            "transpose_viewproj_matrix";
    private static final String ACT_INVERSE_TRANSPOSE_VIEWPROJ_MATRIX =
            "inverse_transpose_viewproj_matrix";
    private static final String ACT_WORLDVIEWPROJ_MATRIX = "worldviewproj_matrix";
    private static final String ACT_INVERSE_WORLDVIEWPROJ_MATRIX =
            "inverse_worldviewproj_matrix";
    private static final String ACT_TRANSPOSE_WORLDVIEWPROJ_MATRIX =
            "transpose_worldviewproj_matrix";
    private static final String ACT_INVERSE_TRANSPOSE_WORLDVIEWPROJ_MATRIX =
            "inverse_transpose_worldviewproj_matrix";

    private static final String DEPTH_CHECK = "depth_check";

    private static final String SCENE_BLEND = "scene_blend";

    private static final String SCENE_BLEND_OP = "scene_blend_op";

    private static final String DEPTH_WRITE = "depth_write";

    private static final HashMap<String, ENG_Integer> shaderParamType =
            new HashMap<>();

    private static int vertexName;
    private static int fragmentName;
    private static int materialName;
    private static int techniqueName;
    private static int passName;
    private static int textureUnitName;

    /**
     * Free memory by getting rid of the loaded chunk of strings
     */
    public static void emptyParametersList() {
        init = false;
        shaderParamType.clear();
    }

    private static void init() {
        init = true;
        shaderParamType.put("world_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_WORLD_MATRIX));
        shaderParamType.put("inverse_world_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_WORLD_MATRIX));
        shaderParamType.put("transpose_world_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TRANSPOSE_WORLD_MATRIX));
        shaderParamType.put("inverse_transpose_world_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_WORLD_MATRIX));
        shaderParamType.put("world_matrix_array_3x4", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_WORLD_MATRIX_ARRAY_3X4));
        shaderParamType.put("view_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_VIEW_MATRIX));
        shaderParamType.put("inverse_view_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_VIEW_MATRIX));
        shaderParamType.put("transpose_view_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TRANSPOSE_VIEW_MATRIX));
        shaderParamType.put("inverse_transpose_view_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_VIEW_MATRIX));
        shaderParamType.put("projection_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_PROJECTION_MATRIX));
        shaderParamType.put("inverse_projection_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_PROJECTION_MATRIX));
        shaderParamType.put("transpose_projection_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TRANSPOSE_PROJECTION_MATRIX));
        shaderParamType.put("inverse_transpose_projection_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_PROJECTION_MATRIX));
        shaderParamType.put("worldview_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_WORLDVIEW_MATRIX));
        shaderParamType.put("inverse_worldview_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_WORLDVIEW_MATRIX));
        shaderParamType.put("transpose_worldview_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TRANSPOSE_WORLDVIEW_MATRIX));
        shaderParamType.put("inverse_transpose_worldview_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_WORLDVIEW_MATRIX));
        shaderParamType.put("viewproj_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_VIEWPROJ_MATRIX));
        shaderParamType.put("inverse_viewproj_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_VIEWPROJ_MATRIX));
        shaderParamType.put("transpose_viewproj_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TRANSPOSE_VIEWPROJ_MATRIX));
        shaderParamType.put("inverse_transpose_viewproj_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_VIEWPROJ_MATRIX));
        shaderParamType.put("worldviewproj_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_WORLDVIEWPROJ_MATRIX));
        shaderParamType.put("inverse_worldviewproj_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_WORLDVIEWPROJ_MATRIX));
        shaderParamType.put("transpose_worldviewproj_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TRANSPOSE_WORLDVIEWPROJ_MATRIX));
        shaderParamType.put("inverse_transpose_worldviewproj_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_WORLDVIEWPROJ_MATRIX));
        shaderParamType.put("texture_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TEXTURE_MATRIX));
        shaderParamType.put("render_target_flipping", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_RENDER_TARGET_FLIPPING));
        shaderParamType.put("vertex_winding", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_VERTEX_WINDING));
        shaderParamType.put("light_diffuse_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_DIFFUSE_COLOUR));
        shaderParamType.put("light_specular_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_SPECULAR_COLOUR));
        shaderParamType.put("light_attenuation", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_ATTENUATION));
        shaderParamType.put("spotlight_params", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SPOTLIGHT_PARAMS));
        shaderParamType.put("light_position", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_POSITION));
        shaderParamType.put("light_direction", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION));
        shaderParamType.put("light_position_object_space", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_OBJECT_SPACE));
        shaderParamType.put("light_direction_object_space", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_OBJECT_SPACE));
        shaderParamType.put("light_distance_object_space", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_DISTANCE_OBJECT_SPACE));
        shaderParamType.put("light_position_view_space", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_VIEW_SPACE));
        shaderParamType.put("light_direction_view_space", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_VIEW_SPACE));
        shaderParamType.put("light_power", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_POWER));
        shaderParamType.put("light_diffuse_colour_power_scaled", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_DIFFUSE_COLOUR_POWER_SCALED));
        shaderParamType.put("light_specular_colour_power_scaled", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_SPECULAR_COLOUR_POWER_SCALED));
        shaderParamType.put("light_number", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_NUMBER));
        shaderParamType.put("light_diffuse_colour_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_DIFFUSE_COLOUR_ARRAY));
        shaderParamType.put("light_specular_colour_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_SPECULAR_COLOUR_ARRAY));
        shaderParamType.put("light_diffuse_colour_power_scaled_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_DIFFUSE_COLOUR_POWER_SCALED_ARRAY));
        shaderParamType.put("light_specular_colour_power_scaled_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_SPECULAR_COLOUR_POWER_SCALED_ARRAY));
        shaderParamType.put("light_attenuation_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_ATTENUATION_ARRAY));
        shaderParamType.put("spotlight_params_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SPOTLIGHT_PARAMS_ARRAY));
        shaderParamType.put("light_position_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_ARRAY));
        shaderParamType.put("light_direction_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_ARRAY));
        shaderParamType.put("light_position_object_space_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_OBJECT_SPACE_ARRAY));
        shaderParamType.put("light_direction_object_space_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_OBJECT_SPACE_ARRAY));
        shaderParamType.put("light_distance_object_space_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_DISTANCE_OBJECT_SPACE_ARRAY));
        shaderParamType.put("light_position_view_space_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_VIEW_SPACE_ARRAY));
        shaderParamType.put("light_direction_view_space_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_VIEW_SPACE_ARRAY));
        shaderParamType.put("light_power_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_POWER_ARRAY));
        shaderParamType.put("light_count", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_COUNT));
        shaderParamType.put("light_casts_shadows", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LIGHT_CASTS_SHADOWS));
        shaderParamType.put("ambient_light_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_AMBIENT_LIGHT_COLOUR));
        shaderParamType.put("surface_ambient_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SURFACE_AMBIENT_COLOUR));
        shaderParamType.put("surface_diffuse_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SURFACE_DIFFUSE_COLOUR));
        shaderParamType.put("surface_specular_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SURFACE_SPECULAR_COLOUR));
        shaderParamType.put("surface_emissive_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SURFACE_EMISSIVE_COLOUR));
        shaderParamType.put("surface_shininess", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SURFACE_SHININESS));
        shaderParamType.put("derived_ambient_light_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_DERIVED_AMBIENT_LIGHT_COLOUR));
        shaderParamType.put("derived_scene_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_DERIVED_SCENE_COLOUR));
        shaderParamType.put("derived_light_diffuse_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_DERIVED_LIGHT_DIFFUSE_COLOUR));
        shaderParamType.put("derived_light_specular_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_DERIVED_LIGHT_SPECULAR_COLOUR));
        shaderParamType.put("derived_light_diffuse_colour_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_DERIVED_LIGHT_DIFFUSE_COLOUR_ARRAY));
        shaderParamType.put("derived_light_specular_colour_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_DERIVED_LIGHT_SPECULAR_COLOUR_ARRAY));
        shaderParamType.put("fog_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_FOG_COLOUR));
        shaderParamType.put("fog_params", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_FOG_PARAMS));
        shaderParamType.put("camera_position", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_CAMERA_POSITION));
        shaderParamType.put("camera_position_object_space", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_CAMERA_POSITION_OBJECT_SPACE));
        shaderParamType.put("lod_camera_position", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LOD_CAMERA_POSITION));
        shaderParamType.put("lod_camera_position_object_space", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_LOD_CAMERA_POSITION_OBJECT_SPACE));
        shaderParamType.put("time", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TIME));
        shaderParamType.put("time_0_x", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TIME_0_X));
        shaderParamType.put("costime_0_x", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_COSTIME_0_X));
        shaderParamType.put("sintime_0_x", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SINTIME_0_X));
        shaderParamType.put("tantime_0_x", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TANTIME_0_X));
        shaderParamType.put("time_0_x_packed", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TIME_0_X_PACKED));
        shaderParamType.put("time_0_1", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TIME_0_1));
        shaderParamType.put("costime_0_1", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_COSTIME_0_1));
        shaderParamType.put("sintime_0_1", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SINTIME_0_1));
        shaderParamType.put("tantime_0_1", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TANTIME_0_1));
        shaderParamType.put("time_0_1_packed", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TIME_0_1_PACKED));
        shaderParamType.put("time_0_2pi", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TIME_0_2PI));
        shaderParamType.put("costime_0_2pi", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_COSTIME_0_2PI));
        shaderParamType.put("sintime_0_2pi", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SINTIME_0_2PI));
        shaderParamType.put("tantime_0_2pi", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TANTIME_0_2PI));
        shaderParamType.put("time_0_2pi_packed", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TIME_0_2PI_PACKED));
        shaderParamType.put("frame_time", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_FRAME_TIME));
        shaderParamType.put("fps", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_FPS));
        shaderParamType.put("viewport_width", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_VIEWPORT_WIDTH));
        shaderParamType.put("viewport_height", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_VIEWPORT_HEIGHT));
        shaderParamType.put("inverse_viewport_width", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_VIEWPORT_WIDTH));
        shaderParamType.put("inverse_viewport_height", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_VIEWPORT_HEIGHT));
        shaderParamType.put("viewport_size", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_VIEWPORT_SIZE));
        shaderParamType.put("texel_offsets", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TEXEL_OFFSETS));
        shaderParamType.put("view_direction", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_VIEW_DIRECTION));
        shaderParamType.put("view_side_vector", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_VIEW_SIDE_VECTOR));
        shaderParamType.put("view_up_vector", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_VIEW_UP_VECTOR));
        shaderParamType.put("fov", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_FOV));
        shaderParamType.put("near_clip_distance", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_NEAR_CLIP_DISTANCE));
        shaderParamType.put("far_clip_distance", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_FAR_CLIP_DISTANCE));
        shaderParamType.put("texture_viewproj_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TEXTURE_VIEWPROJ_MATRIX));
        shaderParamType.put("texture_viewproj_matrix_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TEXTURE_VIEWPROJ_MATRIX_ARRAY));
        shaderParamType.put("texture_worldviewproj_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TEXTURE_WORLDVIEWPROJ_MATRIX));
        shaderParamType.put("texture_worldviewproj_matrix_array", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TEXTURE_WORLDVIEWPROJ_MATRIX_ARRAY));
        shaderParamType.put("spotlight_viewproj_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SPOTLIGHT_VIEWPROJ_MATRIX));
        shaderParamType.put("spotlight_worldviewproj_matrix", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SPOTLIGHT_WORLDVIEWPROJ_MATRIX));
        shaderParamType.put("scene_depth_range", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SCENE_DEPTH_RANGE));
        shaderParamType.put("shadow_scene_depth_range", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SHADOW_SCENE_DEPTH_RANGE));
        shaderParamType.put("shadow_colour", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SHADOW_COLOUR));
        shaderParamType.put("shadow_extrusion_distance", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_SHADOW_EXTRUSION_DISTANCE));
        shaderParamType.put("texture_size", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_TEXTURE_SIZE));
        shaderParamType.put("inverse_texture_size", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_INVERSE_TEXTURE_SIZE));
        shaderParamType.put("packed_texture_size", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_PACKED_TEXTURE_SIZE));
        shaderParamType.put("pass_number", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_PASS_NUMBER));
        shaderParamType.put("pass_iteration_number", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_PASS_ITERATION_NUMBER));
        shaderParamType.put("animation_parametric", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_ANIMATION_PARAMETRIC));
        shaderParamType.put("custom", new ENG_Integer(
                ENG_ShaderTypeStandard.ACT_CUSTOM));

    }

    private static void checkShaderType(String type) {
        if ((!type.equalsIgnoreCase(FLOAT)) &&
                (!type.equalsIgnoreCase(INT)) &&
                (!type.equalsIgnoreCase(BOOL)) &&
                (!type.equalsIgnoreCase(VEC2)) &&
                (!type.equalsIgnoreCase(VEC3)) &&
                (!type.equalsIgnoreCase(VEC4)) &&
                (!type.equalsIgnoreCase(IVEC2)) &&
                (!type.equalsIgnoreCase(IVEC3)) &&
                (!type.equalsIgnoreCase(IVEC4)) &&
                (!type.equalsIgnoreCase(BVEC2)) &&
                (!type.equalsIgnoreCase(BVEC3)) &&
                (!type.equalsIgnoreCase(BVEC4)) &&
                (!type.equalsIgnoreCase(MAT2)) &&
                (!type.equalsIgnoreCase(MAT3)) &&
                (!type.equalsIgnoreCase(MAT4))) {
            throw new ENG_InvalidFormatParsingException();
        }
    }

    private static float parseFloat(DataInputStream fp0) {
        String s = ENG_CompilerUtil.getNextWord(fp0);
        float f;
        try {
            f = Float.parseFloat(s);
        } catch (NumberFormatException e) {
            throw new ENG_InvalidFormatParsingException();
        }
        return f;
    }

    private static int parseInt(DataInputStream fp0) {
        String s = ENG_CompilerUtil.getNextWord(fp0);
        int f;
        try {
            f = Integer.parseInt(s);
        } catch (NumberFormatException e) {
            throw new ENG_InvalidFormatParsingException();
        }
        return f;
    }

    private static boolean parseBoolean(DataInputStream fp0) {
        String s = ENG_CompilerUtil.getNextWord(fp0);
        boolean f;
        try {
            f = Boolean.parseBoolean(s);
        } catch (NumberFormatException e) {
            throw new ENG_InvalidFormatParsingException();
        }
        return f;
    }

    private static float[] parseFloatVec(int numToParse, DataInputStream fp0) {
        float[] f = new float[numToParse];
        for (int i = 0; i < numToParse; ++i) {
            f[i] = parseFloat(fp0);
        }
        return f;
    }

    private static int[] parseIntVec(int numToParse, DataInputStream fp0) {
        int[] f = new int[numToParse];
        for (int i = 0; i < numToParse; ++i) {
            f[i] = parseInt(fp0);
        }
        return f;
    }

    private static boolean[] parseBooleanVec(int numToParse, DataInputStream fp0) {
        boolean[] f = new boolean[numToParse];
        for (int i = 0; i < numToParse; ++i) {
            f[i] = parseBoolean(fp0);
        }
        return f;
    }

    private static ENG_ShaderTypeFloat addFloatToShader(int numToParse,
                                                        String paramName, DataInputStream fp0) {
        ENG_ShaderTypeFloat t = new ENG_ShaderTypeFloat();
        t.name = paramName;
        t.value = parseFloatVec(numToParse, fp0);
        return t;
    }

    private static ENG_ShaderTypeInt addIntToShader(int numToParse,
                                                    String paramName, DataInputStream fp0) {
        ENG_ShaderTypeInt t = new ENG_ShaderTypeInt();
        t.name = paramName;
        t.value = parseIntVec(numToParse, fp0);
        return t;
    }

    private static ENG_ShaderTypeBool addBooleanToShader(int numToParse,
                                                         String paramName, DataInputStream fp0) {
        ENG_ShaderTypeBool t = new ENG_ShaderTypeBool();
        t.name = paramName;
        t.value = parseBooleanVec(numToParse, fp0);
        return t;
    }

    private static void parseShaderType(DataInputStream fp0, ENG_Shader shader,
                                        boolean defaultParam) {
        //	checkShaderType(type);
        //	String s = null;
        String paramName = ENG_CompilerUtil.getNextWord(fp0);
        String type = ENG_CompilerUtil.getNextWord(fp0);
        if ((paramName == null) || (type == null)) {
            throw new ENG_InvalidFormatParsingException();
        }
        ENG_ShaderNamedParams namedParams;
        if (defaultParam) {
            if (shader.defaultParam.namedParam == null) {
                shader.defaultParam.namedParam = new ENG_ShaderNamedParams();
            }
            namedParams = shader.defaultParam.namedParam;
        } else {

            if (shader.paramList.get(shader.paramList.size() - 1).namedParam == null) {
                shader.paramList.get(shader.paramList.size() - 1).namedParam =
                        new ENG_ShaderNamedParams();
            }
            namedParams = shader.paramList.get(shader.paramList.size() - 1).namedParam;
        }
        if (type.equalsIgnoreCase(FLOAT)) {

		/*	ENG_ShaderTypeFloat t = new ENG_ShaderTypeFloat();
            t.name = paramName;
			t.value = parseFloatVec(FLOAT_PARSE, fp0);*/
            namedParams.FloatList.add(addFloatToShader(FLOAT_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(VEC2)) {
		/*	ENG_ShaderTypeFloat t = new ENG_ShaderTypeFloat();
			t.name = paramName;
			t.value = parseFloatVec(VEC2_PARSE, fp0);*/
            namedParams.Vec2List.add(addFloatToShader(VEC2_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(VEC3)) {
            namedParams.Vec3List.add(addFloatToShader(VEC3_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(VEC4)) {
            namedParams.Vec4List.add(addFloatToShader(VEC4_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(MAT2)) {
            namedParams.Mat2List.add(addFloatToShader(MAT2_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(MAT3)) {
            namedParams.Mat3List.add(addFloatToShader(MAT3_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(MAT4)) {
            namedParams.Mat4List.add(addFloatToShader(MAT4_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(INT)) {
            namedParams.IntList.add(addIntToShader(INT_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(IVEC2)) {
            namedParams.IVec2List.add(addIntToShader(IVEC2_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(IVEC3)) {
            namedParams.IVec3List.add(addIntToShader(IVEC3_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(IVEC4)) {
            namedParams.IVec4List.add(addIntToShader(IVEC4_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(BOOL)) {
            namedParams.BoolList.add(addBooleanToShader(BOOL_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(BVEC2)) {
            namedParams.BVec2List.add(addBooleanToShader(BVEC2_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(BVEC3)) {
            namedParams.BVec3List.add(addBooleanToShader(BVEC3_PARSE, paramName, fp0));
        } else if (type.equalsIgnoreCase(BVEC4)) {
            namedParams.BVec4List.add(addBooleanToShader(BVEC4_PARSE, paramName, fp0));
        } else {
            throw new ENG_InvalidFormatParsingException();
        }
    }


    private static NamedParamType getExtraParamType(int type) {
        switch (type) {
            case ENG_ShaderTypeStandard.ACT_WORLD_MATRIX:
            case ENG_ShaderTypeStandard.ACT_INVERSE_WORLD_MATRIX:
            case ENG_ShaderTypeStandard.ACT_TRANSPOSE_WORLD_MATRIX:
            case ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_WORLD_MATRIX:
            case ENG_ShaderTypeStandard.ACT_WORLD_MATRIX_ARRAY_3X4:
            case ENG_ShaderTypeStandard.ACT_VIEW_MATRIX:
            case ENG_ShaderTypeStandard.ACT_INVERSE_VIEW_MATRIX:
            case ENG_ShaderTypeStandard.ACT_TRANSPOSE_VIEW_MATRIX:
            case ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_VIEW_MATRIX:
            case ENG_ShaderTypeStandard.ACT_PROJECTION_MATRIX:
            case ENG_ShaderTypeStandard.ACT_INVERSE_PROJECTION_MATRIX:
            case ENG_ShaderTypeStandard.ACT_TRANSPOSE_PROJECTION_MATRIX:
            case ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_PROJECTION_MATRIX:
            case ENG_ShaderTypeStandard.ACT_WORLDVIEW_MATRIX:
            case ENG_ShaderTypeStandard.ACT_INVERSE_WORLDVIEW_MATRIX:
            case ENG_ShaderTypeStandard.ACT_TRANSPOSE_WORLDVIEW_MATRIX:
            case ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_WORLDVIEW_MATRIX:
            case ENG_ShaderTypeStandard.ACT_VIEWPROJ_MATRIX:
            case ENG_ShaderTypeStandard.ACT_INVERSE_VIEWPROJ_MATRIX:
            case ENG_ShaderTypeStandard.ACT_TRANSPOSE_VIEWPROJ_MATRIX:
            case ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_VIEWPROJ_MATRIX:
            case ENG_ShaderTypeStandard.ACT_WORLDVIEWPROJ_MATRIX:
            case ENG_ShaderTypeStandard.ACT_INVERSE_WORLDVIEWPROJ_MATRIX:
            case ENG_ShaderTypeStandard.ACT_TRANSPOSE_WORLDVIEWPROJ_MATRIX:
            case ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_WORLDVIEWPROJ_MATRIX:
                return NamedParamType.NONE;
            case ENG_ShaderTypeStandard.ACT_TEXTURE_MATRIX:
                return NamedParamType.INT;
            case ENG_ShaderTypeStandard.ACT_RENDER_TARGET_FLIPPING:
            case ENG_ShaderTypeStandard.ACT_VERTEX_WINDING:
                return NamedParamType.NONE;
            case ENG_ShaderTypeStandard.ACT_LIGHT_DIFFUSE_COLOUR:
            case ENG_ShaderTypeStandard.ACT_LIGHT_SPECULAR_COLOUR:
            case ENG_ShaderTypeStandard.ACT_LIGHT_ATTENUATION:
            case ENG_ShaderTypeStandard.ACT_SPOTLIGHT_PARAMS:
            case ENG_ShaderTypeStandard.ACT_LIGHT_POSITION:
            case ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION:
            case ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_OBJECT_SPACE:
            case ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_OBJECT_SPACE:
            case ENG_ShaderTypeStandard.ACT_LIGHT_DISTANCE_OBJECT_SPACE:
            case ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_VIEW_SPACE:
            case ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_VIEW_SPACE:
            case ENG_ShaderTypeStandard.ACT_LIGHT_POWER:
                return NamedParamType.INT;
            case ENG_ShaderTypeStandard.ACT_LIGHT_DIFFUSE_COLOUR_POWER_SCALED:
            case ENG_ShaderTypeStandard.ACT_LIGHT_SPECULAR_COLOUR_POWER_SCALED:
            case ENG_ShaderTypeStandard.ACT_LIGHT_NUMBER:
                return NamedParamType.NONE;
            case ENG_ShaderTypeStandard.ACT_LIGHT_DIFFUSE_COLOUR_ARRAY:
            case ENG_ShaderTypeStandard.ACT_LIGHT_SPECULAR_COLOUR_ARRAY:
            case ENG_ShaderTypeStandard.ACT_LIGHT_DIFFUSE_COLOUR_POWER_SCALED_ARRAY:
            case ENG_ShaderTypeStandard.ACT_LIGHT_SPECULAR_COLOUR_POWER_SCALED_ARRAY:
            case ENG_ShaderTypeStandard.ACT_LIGHT_ATTENUATION_ARRAY:
            case ENG_ShaderTypeStandard.ACT_SPOTLIGHT_PARAMS_ARRAY:
            case ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_ARRAY:
            case ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_ARRAY:
            case ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_OBJECT_SPACE_ARRAY:
            case ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_OBJECT_SPACE_ARRAY:
            case ENG_ShaderTypeStandard.ACT_LIGHT_DISTANCE_OBJECT_SPACE_ARRAY:
            case ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_VIEW_SPACE_ARRAY:
            case ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_VIEW_SPACE_ARRAY:
            case ENG_ShaderTypeStandard.ACT_LIGHT_POWER_ARRAY:
                return NamedParamType.INT;
            case ENG_ShaderTypeStandard.ACT_LIGHT_COUNT:
            case ENG_ShaderTypeStandard.ACT_LIGHT_CASTS_SHADOWS:
            case ENG_ShaderTypeStandard.ACT_AMBIENT_LIGHT_COLOUR:
            case ENG_ShaderTypeStandard.ACT_SURFACE_AMBIENT_COLOUR:
            case ENG_ShaderTypeStandard.ACT_SURFACE_DIFFUSE_COLOUR:
            case ENG_ShaderTypeStandard.ACT_SURFACE_SPECULAR_COLOUR:
            case ENG_ShaderTypeStandard.ACT_SURFACE_EMISSIVE_COLOUR:
            case ENG_ShaderTypeStandard.ACT_SURFACE_SHININESS:
            case ENG_ShaderTypeStandard.ACT_DERIVED_AMBIENT_LIGHT_COLOUR:
            case ENG_ShaderTypeStandard.ACT_DERIVED_SCENE_COLOUR:
                return NamedParamType.NONE;
            case ENG_ShaderTypeStandard.ACT_DERIVED_LIGHT_DIFFUSE_COLOUR:
            case ENG_ShaderTypeStandard.ACT_DERIVED_LIGHT_SPECULAR_COLOUR:
            case ENG_ShaderTypeStandard.ACT_DERIVED_LIGHT_DIFFUSE_COLOUR_ARRAY:
            case ENG_ShaderTypeStandard.ACT_DERIVED_LIGHT_SPECULAR_COLOUR_ARRAY:
                return NamedParamType.INT;
            case ENG_ShaderTypeStandard.ACT_FOG_COLOUR:
            case ENG_ShaderTypeStandard.ACT_FOG_PARAMS:
            case ENG_ShaderTypeStandard.ACT_CAMERA_POSITION:
            case ENG_ShaderTypeStandard.ACT_CAMERA_POSITION_OBJECT_SPACE:
            case ENG_ShaderTypeStandard.ACT_LOD_CAMERA_POSITION:
            case ENG_ShaderTypeStandard.ACT_LOD_CAMERA_POSITION_OBJECT_SPACE:
                return NamedParamType.NONE;
            case ENG_ShaderTypeStandard.ACT_TIME:
            case ENG_ShaderTypeStandard.ACT_TIME_0_X:
            case ENG_ShaderTypeStandard.ACT_COSTIME_0_X:
            case ENG_ShaderTypeStandard.ACT_SINTIME_0_X:
            case ENG_ShaderTypeStandard.ACT_TANTIME_0_X:
            case ENG_ShaderTypeStandard.ACT_TIME_0_X_PACKED:
            case ENG_ShaderTypeStandard.ACT_TIME_0_1:
            case ENG_ShaderTypeStandard.ACT_COSTIME_0_1:
            case ENG_ShaderTypeStandard.ACT_SINTIME_0_1:
            case ENG_ShaderTypeStandard.ACT_TANTIME_0_1:
            case ENG_ShaderTypeStandard.ACT_TIME_0_1_PACKED:
            case ENG_ShaderTypeStandard.ACT_TIME_0_2PI:
            case ENG_ShaderTypeStandard.ACT_COSTIME_0_2PI:
            case ENG_ShaderTypeStandard.ACT_SINTIME_0_2PI:
            case ENG_ShaderTypeStandard.ACT_TANTIME_0_2PI:
            case ENG_ShaderTypeStandard.ACT_TIME_0_2PI_PACKED:
                return NamedParamType.FLOAT;
            case ENG_ShaderTypeStandard.ACT_FRAME_TIME:
            case ENG_ShaderTypeStandard.ACT_FPS:
            case ENG_ShaderTypeStandard.ACT_VIEWPORT_WIDTH:
            case ENG_ShaderTypeStandard.ACT_VIEWPORT_HEIGHT:
            case ENG_ShaderTypeStandard.ACT_INVERSE_VIEWPORT_WIDTH:
            case ENG_ShaderTypeStandard.ACT_INVERSE_VIEWPORT_HEIGHT:
            case ENG_ShaderTypeStandard.ACT_VIEWPORT_SIZE:
            case ENG_ShaderTypeStandard.ACT_TEXEL_OFFSETS:
            case ENG_ShaderTypeStandard.ACT_VIEW_DIRECTION:
            case ENG_ShaderTypeStandard.ACT_VIEW_SIDE_VECTOR:
            case ENG_ShaderTypeStandard.ACT_VIEW_UP_VECTOR:
            case ENG_ShaderTypeStandard.ACT_FOV:
            case ENG_ShaderTypeStandard.ACT_NEAR_CLIP_DISTANCE:
            case ENG_ShaderTypeStandard.ACT_FAR_CLIP_DISTANCE:
                return NamedParamType.NONE;
            case ENG_ShaderTypeStandard.ACT_TEXTURE_VIEWPROJ_MATRIX:
            case ENG_ShaderTypeStandard.ACT_TEXTURE_VIEWPROJ_MATRIX_ARRAY:
                return NamedParamType.INT;
            case ENG_ShaderTypeStandard.ACT_TEXTURE_WORLDVIEWPROJ_MATRIX:
                return NamedParamType.NONE;
            case ENG_ShaderTypeStandard.ACT_TEXTURE_WORLDVIEWPROJ_MATRIX_ARRAY:
            case ENG_ShaderTypeStandard.ACT_SPOTLIGHT_VIEWPROJ_MATRIX:
                return NamedParamType.INT;
            case ENG_ShaderTypeStandard.ACT_SPOTLIGHT_WORLDVIEWPROJ_MATRIX:
            case ENG_ShaderTypeStandard.ACT_SCENE_DEPTH_RANGE:
            case ENG_ShaderTypeStandard.ACT_SHADOW_SCENE_DEPTH_RANGE:
            case ENG_ShaderTypeStandard.ACT_SHADOW_COLOUR:
            case ENG_ShaderTypeStandard.ACT_SHADOW_EXTRUSION_DISTANCE:
                return NamedParamType.NONE;
            case ENG_ShaderTypeStandard.ACT_TEXTURE_SIZE:
            case ENG_ShaderTypeStandard.ACT_INVERSE_TEXTURE_SIZE:
            case ENG_ShaderTypeStandard.ACT_PACKED_TEXTURE_SIZE:
            case ENG_ShaderTypeStandard.ACT_PASS_NUMBER:
            case ENG_ShaderTypeStandard.ACT_PASS_ITERATION_NUMBER:
            case ENG_ShaderTypeStandard.ACT_ANIMATION_PARAMETRIC:
                return NamedParamType.NONE;
            case ENG_ShaderTypeStandard.ACT_CUSTOM:
                return NamedParamType.INT;
            default:
                // Should never get here
                throw new IllegalArgumentException(type + " is an invalid shader type");
        }
    }

    private static void parseAttributeName(DataInputStream fp0, ENG_Shader shader,
                                           boolean defaultParam) {
        String paramName = ENG_CompilerUtil.getNextWord(fp0);
        String type = ENG_CompilerUtil.getNextWord(fp0);
        if ((paramName == null) || (type == null)) {
            throw new ENG_InvalidFormatParsingException();
        }


        if (!init) {
            init();
        }
        ENG_Integer t = shaderParamType.get(type);
        if (t != null) {
            int shaderType = t.getValue();
            if (shaderType == ENG_ShaderTypeStandard.ACT_CUSTOM) {
                ENG_ShaderTypeCustom c = new ENG_ShaderTypeCustom();
                c.name = paramName;
                String customParamNum = ENG_CompilerUtil.getNextWord(fp0);
                if (customParamNum != null) {
                    c.id = Integer.parseInt(customParamNum);
                } else {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (defaultParam) {
                    shader.defaultParam.customType.add(c);
                } else {
                    shader.paramList.get(
                            shader.paramList.size() - 1).customType.add(c);
                }
            } else {
                ENG_ShaderTypeStandard s = new ENG_ShaderTypeStandard();
                s.name = paramName;
                s.type = shaderType;
                s.extraParamType = getExtraParamType(shaderType);
                switch (s.extraParamType) {
                    case FLOAT:
                        s.extraParamFloat = getFloat(fp0);
                        break;
                    case INT:
                        s.extraParamInt = getInt(fp0);
                        break;
                    case NONE:
                    default:
                        // Do nothing
                }
                if (defaultParam) {
                    shader.defaultParam.standardType.add(s);
                } else {
                    shader.paramList.get(
                            shader.paramList.size() - 1).standardType.add(s);

                }
            }
        } else {
            throw new ENG_InvalidFormatParsingException();
        }
	/*	if (type.equalsIgnoreCase(ACT_WORLD_MATRIX)) {
			
		} else if (type.equalsIgnoreCase(ACT_INVERSE_WORLD_MATRIX)) {
			
		}*/
    }

    private static void parseParams(DataInputStream fp0, ENG_Shader shader,
                                    boolean defaultParam) {
        incrementBracketLevel();
        String s;
        while (true) {
            s = ENG_CompilerUtil.getNextWord(fp0);
            if (s == null) {
                throw new ENG_InvalidFormatParsingException();
            }
            if (s.equalsIgnoreCase(PARAM_NAMED)) {
                parseShaderType(fp0, shader, defaultParam);
            } else if (s.equalsIgnoreCase(PARAM_NAMED_AUTO)) {
                parseAttributeName(fp0, shader, defaultParam);
            } else if (s.equalsIgnoreCase(BRACKET_CLOSE)) {
                decrementBracketLevel();
                break;
            }
        }
    }

    private static void parseShader(DataInputStream fp0, ENG_Shader shader,
                                    String currentDir, boolean fromSDCard) {
        //	String path = null;
        //	String fileName = null;

        if (ENG_CompilerUtil.getNextWord(fp0).equals(BRACKET_OPEN)) {
            incrementBracketLevel();
            String s;
            while (true) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(SOURCE)) {
                    String p = ENG_CompilerUtil.getNextWord(fp0);
                    if (p != null) {
                        p = currentDir + p;
                        String[] ret = ENG_CompilerUtil.getPathAndFileName(p, fromSDCard);
                        shader.path = ret[0];
                        shader.fileName = ret[1];
                    } else {
                        throw new ENG_InvalidFormatParsingException();
                    }
                } else if (s.equalsIgnoreCase(DEFAULT_PARAMS)) {
                    String bracket = ENG_CompilerUtil.getNextWord(fp0);
                    if (bracket != null) {
                        if (bracket.equals(BRACKET_OPEN)) {
                            shader.defaultParam = new ENG_ShaderParam();
                            parseParams(fp0, shader, true);
                        }
                    } else {
                        throw new ENG_InvalidFormatParsingException();
                    }
                } else if (s.equalsIgnoreCase(INCLUDES_SKELETAL_ANIMATION)) {
                    shader.skeletalAnimation = getBoolean(fp0);
                } else if (s.equalsIgnoreCase(INCLUDES_MORPH_ANIMATION)) {
                    shader.morphAnimation = getBoolean(fp0);
                } else if (s.equalsIgnoreCase(INCLUDES_POSE_ANIMATION)) {
                    shader.poseAnimationNum = getInt(fp0);
                } else if (s.equalsIgnoreCase(USES_VERTEX_TEXTURE_FETCH)) {
                    shader.useVertexTextureFetch = getBoolean(fp0);
                } else if (s.equalsIgnoreCase(USES_ADJACENCY_INFORMATION)) {
                    shader.useAdjacencyInfo = getBoolean(fp0);
                } else if (s.equalsIgnoreCase(BRACKET_CLOSE)) {
                    decrementBracketLevel();
                    break;
                }
            }
        } else {
            throw new ENG_InvalidFormatParsingException();
        }
    }

    private static ENG_Shader parseProgram(DataInputStream fp0,
                                           String currentDir, int programType, boolean fromSDCard) {
        String name = ENG_CompilerUtil.getNextWord(fp0);

        ENG_Shader shader = new ENG_Shader();
        if (name == null) {
            //Assign a default name
            if (programType == ENG_Shader.VERTEX_PROGRAM) {
                name = String.valueOf(vertexName++);
            } else if (programType == ENG_Shader.FRAGMENT_PROGRAM) {
                name = String.valueOf(fragmentName++);
            } else {
                throw new IllegalArgumentException();
            }
        } else {
            shader.name = name;
        }
        shader.type = programType;
        parseShader(fp0, shader, currentDir, fromSDCard);
        return shader;
    }

    private static int parseProgramParams(DataInputStream fp0, HashMap<String, ENG_Shader> shaderList, ENG_PassResource pass, int shaderType) {
        String s = ENG_CompilerUtil.getNextWord(fp0);
        if (s == null) {
            throw new ENG_InvalidFormatParsingException();
        }
	/*	int size = list.size();
		ENG_Shader shader = null;
		boolean found = false;
		for (int i = 0; i < size; ++i) {
			shader = list.get(i);
			if (shader.name.equals(s)) {
				found = true;
				break;
			}
		}
		if (!found) {
			throw new ENG_UndeclaredIdentifierException();
		}*/
        ENG_Shader shader = shaderList.get(s);
        if (shader == null) {
            throw new ENG_UndeclaredIdentifierException("shader: " + s);
        }
        s = ENG_CompilerUtil.getNextWord(fp0);
        if ((s != null) && (s.equalsIgnoreCase(BRACKET_OPEN))) {
            incrementBracketLevel();
        } else {
            throw new ENG_InvalidFormatParsingException();
        }
        s = ENG_CompilerUtil.getNextWord(fp0);
        if (s == null) {
            throw new ENG_InvalidFormatParsingException();
        }
        if (s.equalsIgnoreCase(BRACKET_CLOSE)) {
            switch (shaderType) {
                case ENG_Shader.VERTEX_PROGRAM:
                    pass.vertexProgramID = shader.name;
                    break;
                case ENG_Shader.FRAGMENT_PROGRAM:
                    pass.fragmentProgramID = shader.name;
                    break;
                default:

            }
            return -1;
        } else {
            ENG_CompilerUtil.setLookAhead(s);
        }
        int shaderParam = shader.paramList.size();
        switch (shaderType) {
            case ENG_Shader.VERTEX_PROGRAM:
                pass.vertexProgramID = shader.name;
                pass.vertexProgramParamListID = shaderParam;
                break;
            case ENG_Shader.FRAGMENT_PROGRAM:
                pass.fragmentProgramID = shader.name;
                pass.fragmentProgramParamListID = shaderParam;
                break;
            default:

        }
        shader.paramList.add(new ENG_ShaderParam());
        parseParams(fp0, shader, false);
        return shaderParam;
    }

    private static ENG_TextureUnitResource parseTextureUnit(DataInputStream fp0) {
        
        String s = ENG_CompilerUtil.getNextWord(fp0);
        ENG_TextureUnitResource tex = new ENG_TextureUnitResource();
        boolean defName = false;
        if (s.equalsIgnoreCase(BRACKET_OPEN)) {
            tex.name = String.valueOf(passName++);
            tex.textureAlias = "";
            defName = true;
        } else {
            tex.name = s;
            tex.textureAlias = s;
        }
        checkNameParsed(fp0, defName);
        boolean textureAliasSet = false;
        boolean textureSet = false;
        boolean animTexSet = false;
        boolean cubicSet = false;
        boolean bindingSet = false;
        boolean contentSet = false;
        boolean coordSet = false;
        boolean addressModeSet = false;
        boolean borderColourSet = false;
        boolean filteringSet = false;
        boolean maxAnisotropySet = false;
        boolean mipmapBiasSet = false;
        boolean colourOpSet = false;
        boolean envMapSet = false;
        boolean ignoreMaxAni = true;
        boolean textureUsage = false;
        while (true) {
            s = ENG_CompilerUtil.getNextWord(fp0);
            if (s == null) {
                throw new ENG_InvalidFormatParsingException();
            }
            if (s.equalsIgnoreCase(TEXTURE_ALIAS)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                tex.textureAlias = s;
            } else if (s.equalsIgnoreCase(TEXTURE)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (tex.textureName == null) {
                    tex.textureName = new ArrayList<>();
                }
                tex.textureName.add(s);
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(TEXTURE_NAME_NUM_MIPMAPS)) {
                    s = ENG_CompilerUtil.getNextWord(fp0);
                    if (s == null) {
                        throw new ENG_InvalidFormatParsingException();
                    }
                    if (s.equalsIgnoreCase(TEXTURE_NAME_NUM_MIPMAPS_UNLIMITED)) {
                        tex.numMipmaps = 0;
                    } else {
                        try {
                            tex.numMipmaps = Integer.parseInt(s);
                        } catch (NumberFormatException e) {
                            throw new ENG_InvalidFormatParsingException();
                        }
                    }
                }
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(TEXTURE_NAME_TYPE)) {
                    s = ENG_CompilerUtil.getNextWord(fp0);
                    if (s == null) {
                        throw new ENG_InvalidFormatParsingException();
                    }
                    if (s.equalsIgnoreCase(PF_L8)) {
                        tex.pf = PixelFormat.PF_L8;
                    } else if (s.equalsIgnoreCase(PF_L16)) {
                        tex.pf = PixelFormat.PF_L16;
                    } else if (s.equalsIgnoreCase(PF_A8)) {
                        tex.pf = PixelFormat.PF_A8;
                    } else if (s.equalsIgnoreCase(PF_A4L4)) {
                        tex.pf = PixelFormat.PF_A4L4;
                    } else if (s.equalsIgnoreCase(PF_BYTE_LA)) {
                        tex.pf = PixelFormat.PF_BYTE_LA;
                    } else if (s.equalsIgnoreCase(PF_R5G6B5)) {
                        tex.pf = PixelFormat.PF_R5G6B5;
                    } else if (s.equalsIgnoreCase(PF_B5G6R5)) {
                        tex.pf = PixelFormat.PF_B5G6R5;
                    } else if (s.equalsIgnoreCase(PF_R3G3B2)) {
                        tex.pf = PixelFormat.PF_R3G3B2;
                    } else if (s.equalsIgnoreCase(PF_A4R4G4B4)) {
                        tex.pf = PixelFormat.PF_A4R4G4B4;
                    } else if (s.equalsIgnoreCase(PF_A1R5G5B5)) {
                        tex.pf = PixelFormat.PF_A1R5G5B5;
                    } else if (s.equalsIgnoreCase(PF_R8G8B8)) {
                        tex.pf = PixelFormat.PF_R8G8B8;
                    } else if (s.equalsIgnoreCase(PF_B8G8R8)) {
                        tex.pf = PixelFormat.PF_B8G8R8;
                    } else if (s.equalsIgnoreCase(PF_A8R8G8B8)) {
                        tex.pf = PixelFormat.PF_A8R8G8B8;
                    } else if (s.equalsIgnoreCase(PF_A8B8G8R8)) {
                        tex.pf = PixelFormat.PF_A8B8G8R8;
                    } else if (s.equalsIgnoreCase(PF_B8G8R8A8)) {
                        tex.pf = PixelFormat.PF_B8G8R8A8;
                    } else if (s.equalsIgnoreCase(PF_R8G8B8A8)) {
                        tex.pf = PixelFormat.PF_R8G8B8A8;
                    } else if (s.equalsIgnoreCase(PF_X8R8G8B8)) {
                        tex.pf = PixelFormat.PF_X8R8G8B8;
                    } else if (s.equalsIgnoreCase(PF_X8B8G8R8)) {
                        tex.pf = PixelFormat.PF_X8B8G8R8;
                    } else if (s.equalsIgnoreCase(PF_A2R10G10B10)) {
                        tex.pf = PixelFormat.PF_A2R10G10B10;
                    } else if (s.equalsIgnoreCase(PF_A2B10G10R10)) {
                        tex.pf = PixelFormat.PF_A2B10G10R10;
                    } else if (s.equalsIgnoreCase(PF_FLOAT16_R)) {
                        tex.pf = PixelFormat.PF_FLOAT16_R;
                    } else if (s.equalsIgnoreCase(PF_FLOAT16_RGB)) {
                        tex.pf = PixelFormat.PF_FLOAT16_RGB;
                    } else if (s.equalsIgnoreCase(PF_FLOAT16_RGBA)) {
                        tex.pf = PixelFormat.PF_FLOAT16_RGBA;
                    } else if (s.equalsIgnoreCase(PF_FLOAT32_R)) {
                        tex.pf = PixelFormat.PF_FLOAT32_R;
                    } else if (s.equalsIgnoreCase(PF_FLOAT32_RGB)) {
                        tex.pf = PixelFormat.PF_FLOAT32_RGB;
                    } else if (s.equalsIgnoreCase(PF_FLOAT32_RGBA)) {
                        tex.pf = PixelFormat.PF_FLOAT32_RGBA;
                    } else if (s.equalsIgnoreCase(PF_SHORT_RGBA)) {
                        tex.pf = PixelFormat.PF_SHORT_RGBA;
                    } else {
                        throw new ENG_InvalidFormatParsingException();
                    }
                }
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(TEXTURE_NAME_GAMMA)) {
                    s = ENG_CompilerUtil.getNextWord(fp0);
                    if (s == null) {
                        throw new ENG_InvalidFormatParsingException();
                    }
                    int gamma;
                    try {
                        gamma = Integer.parseInt(s);
                    } catch (NumberFormatException e) {
                        throw new ENG_InvalidFormatParsingException();
                    }
                    tex.gamma = (gamma != 0);
                }
                s = ENG_CompilerUtil.getNextWord(fp0);
                checkNull(s);
                if (s.equalsIgnoreCase(TEXTURE_NAME_USE_SHADOW_BUFFER)) {
                    tex.useShadowBuffer = getBoolean(fp0);
                } else {
                    // Couldn't find it so pass it to the next read.
                    ENG_CompilerUtil.setLookAhead(s);
                }
                s = ENG_CompilerUtil.getNextWord(fp0);
                checkNull(s);
                if (s.equalsIgnoreCase(TEXTURE_NAME_TEXTURE_USAGE)) {
                    s = ENG_CompilerUtil.getNextWord(fp0);
                    checkNull(s);
                    if (s.equalsIgnoreCase(TU_STATIC)) {
                        tex.usage = TextureUsage.TU_STATIC;
                    } else if (s.equalsIgnoreCase(TU_DYNAMIC)) {
                        tex.usage = TextureUsage.TU_DYNAMIC;
                    } else if (s.equalsIgnoreCase(TU_WRITE_ONLY)) {
                        tex.usage = TextureUsage.TU_WRITE_ONLY;
                    } else if (s.equalsIgnoreCase(TU_STATIC_WRITE_ONLY)) {
                        tex.usage = TextureUsage.TU_STATIC_WRITE_ONLY;
                    } else if (s.equalsIgnoreCase(TU_DYNAMIC_WRITE_ONLY)) {
                        tex.usage = TextureUsage.TU_DYNAMIC_WRITE_ONLY;
                    } else if (s.equalsIgnoreCase(TU_DYNAMIC_WRITE_ONLY_DISCARDABLE)) {
                        tex.usage = TextureUsage.TU_DYNAMIC_WRITE_ONLY_DISCARDABLE;
                    } else if (s.equalsIgnoreCase(TU_AUTOMIPMAP)) {
                        tex.usage = TextureUsage.TU_AUTOMIPMAP;
                    } else if (s.equalsIgnoreCase(TU_RENDERTARGET)) {
                        tex.usage = TextureUsage.TU_RENDERTARGET;
                    } else if (s.equalsIgnoreCase(TU_DEFAULT)) {
                        tex.usage = TextureUsage.TU_DEFAULT;
                    } else {
                        throw new ENG_InvalidFormatParsingException();
                    }
                    textureUsage = true;
                } else {
                    ENG_CompilerUtil.setLookAhead(s);
                }
                textureSet = true;
            } else if (s.equalsIgnoreCase(ANIM_TEXTURE)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                tex.animTexture = s;
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                int numFrames;
                try {
                    numFrames = Integer.parseInt(s);
                } catch (NumberFormatException e) {
                    throw new ENG_InvalidFormatParsingException();
                }
                tex.numFrames = numFrames;
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                float duration;
                try {
                    duration = Float.parseFloat(s);
                } catch (NumberFormatException e) {
                    throw new ENG_InvalidFormatParsingException();
                }
                tex.duration = duration;
                animTexSet = true;
            } else if (s.equalsIgnoreCase(CUBIC_TEXTURE)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                tex.cubeName = s;
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(CUBIC_COMBINEDUVW)) {
                    tex.cubeType = CubeType.combinedUVW;
                } else if (s.equalsIgnoreCase(CUBIC_SEPARATEUV)) {
                    tex.cubeType = CubeType.separateUV;
                } else {
                    throw new ENG_InvalidFormatParsingException();
                }
                cubicSet = true;
            } else if (s.equalsIgnoreCase(BINDING_TYPE)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(BINDING_VERTEX)) {
                    tex.bindingType = headwayent.hotshotengine.resource
                            .ENG_TextureUnitResource.BindingType.vertex;
                } else if (s.equalsIgnoreCase(BINDING_FRAGMENT)) {
                    tex.bindingType = headwayent.hotshotengine.resource
                            .ENG_TextureUnitResource.BindingType.fragment;
                } else {
                    throw new ENG_InvalidFormatParsingException();
                }
                bindingSet = true;
            } else if (s.equalsIgnoreCase(CONTENT_TYPE)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(CONTENT_NAMED)) {
                    tex.contentType = ContentType.named;
                } else if (s.equalsIgnoreCase(CONTENT_SHADOW)) {
                    tex.contentType = ContentType.shadow;
                } else if (s.equalsIgnoreCase(CONTENT_COMPOSITOR)) {
                    tex.contentType = ContentType.compositor;
                    s = ENG_CompilerUtil.getNextWord(fp0);
                    if (s == null) {
                        throw new ENG_InvalidFormatParsingException();
                    }
                    tex.referencedCompositorName = s;
                    s = ENG_CompilerUtil.getNextWord(fp0);
                    if (s == null) {
                        throw new ENG_InvalidFormatParsingException();
                    }
                    tex.referencedTextureName = s;
                    s = ENG_CompilerUtil.getNextWord(fp0);
                    if (s == null) {
                        throw new ENG_InvalidFormatParsingException();
                    }
                    int mrtIndex;
                    try {
                        mrtIndex = Integer.parseInt(s);
                    } catch (NumberFormatException e) {
                        throw new ENG_InvalidFormatParsingException();
                    }
                    tex.mrtTextureIndex = mrtIndex;
                } else {
                    throw new ENG_InvalidFormatParsingException();
                }
                contentSet = true;
            } else if (s.equalsIgnoreCase(TEX_COORD_SET)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                int texCoord;
                try {
                    texCoord = Integer.parseInt(s);
                } catch (NumberFormatException e) {
                    throw new ENG_InvalidFormatParsingException();
                }
                tex.texCoordSet = texCoord;
                coordSet = true;
            } else if (s.equalsIgnoreCase(TEX_ADDRESS_MODE)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(WRAP_MODE_WRAP)) {
                    tex.addresingMode = TextureAddressingMode.TAM_WRAP;
                } else if (s.equalsIgnoreCase(WRAP_MODE_MIRROR)) {
                    tex.addresingMode = TextureAddressingMode.TAM_MIRROR;
                } else if (s.equalsIgnoreCase(WRAP_MODE_CLAMP)) {
                    tex.addresingMode = TextureAddressingMode.TAM_CLAMP;
                } else if (s.equalsIgnoreCase(WRAP_MODE_BORDER)) {
                    tex.addresingMode = TextureAddressingMode.TAM_BORDER;
                } else {
                    throw new ENG_InvalidFormatParsingException();
                }
                addressModeSet = true;
            } else if (s.equalsIgnoreCase(TEX_BORDER_COLOUR)) {
                float[] color = new float[4];
                for (int i = 0; i < 4; ++i) {
                    s = ENG_CompilerUtil.getNextWord(fp0);
                    if (s == null) {
                        throw new ENG_InvalidFormatParsingException();
                    }
                    try {
                        color[i] = Float.parseFloat(s);
                    } catch (NumberFormatException e) {
                        throw new ENG_InvalidFormatParsingException();
                    }
                }
                tex.textureBorder = new ENG_ColorValue(color[0], color[1], color[2],
                        color[3]);
                borderColourSet = true;
            } else if (s.equalsIgnoreCase(FILTERING)) {

                FilterType[] ft = new FilterType[3];
                for (int i = 0; i < 3; ++i) {
                    s = ENG_CompilerUtil.getNextWord(fp0);
                    if (s == null) {
                        throw new ENG_InvalidFormatParsingException();
                    }
                    if (s.equalsIgnoreCase(FILTERING_NONE)) {
                        ft[i] = FilterType.none;
                    } else if (s.equalsIgnoreCase(FILTERING_POINT)) {
                        ft[i] = FilterType.point;
                    } else if (s.equalsIgnoreCase(FILTERING_LINEAR)) {
                        ft[i] = FilterType.linear;
                    } else if (s.equalsIgnoreCase(FILTERING_ANISOTROPIC)) {
                        ft[i] = FilterType.anisotropic;
                        ignoreMaxAni = false;
                    } else {
                        throw new ENG_InvalidFormatParsingException();
                    }
                }

                tex.filterTypeMin = ft[0];
                tex.filterTypeMag = ft[1];
                tex.filterTypeMip = ft[2];
                //Check if valid settings
                if (tex.filterTypeMin == FilterType.none) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (tex.filterTypeMag == FilterType.none) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (tex.filterTypeMip == FilterType.anisotropic) {
                    throw new ENG_InvalidFormatParsingException();
                }
                filteringSet = true;
            } else if (s.equalsIgnoreCase(MAX_ANISOTROPY)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                int maxAni;
                try {
                    maxAni = Integer.parseInt(s);
                } catch (NumberFormatException e) {
                    throw new ENG_InvalidFormatParsingException();
                }
                tex.maxAnisotropy = maxAni;
                maxAnisotropySet = true;
            } else if (s.equalsIgnoreCase(MIPMAP_BIAS)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                int mipmap;
                try {
                    mipmap = Integer.parseInt(s);
                } catch (NumberFormatException e) {
                    throw new ENG_InvalidFormatParsingException();
                }
                tex.mipmapBias = mipmap;
                mipmapBiasSet = true;
            } else if (s.equalsIgnoreCase(COLOUR_OP)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(COLOUR_OP_REPLACE)) {
                    tex.colourOpType = ColourOpType.replace;
                } else if (s.equalsIgnoreCase(COLOUR_OP_ADD)) {
                    tex.colourOpType = ColourOpType.add;
                } else if (s.equalsIgnoreCase(COLOUR_OP_MODULATE)) {
                    tex.colourOpType = ColourOpType.modulate;
                } else if (s.equalsIgnoreCase(COLOUR_OP_ALPHA_BLEND)) {
                    tex.colourOpType = ColourOpType.alpha_blend;
                } else {
                    throw new ENG_InvalidFormatParsingException();
                }
                colourOpSet = true;
            } else if (s.equalsIgnoreCase(ENV_MAP)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(ENV_MAP_OFF)) {
                    tex.envMapType = EnvMapType.off;
                } else if (s.equalsIgnoreCase(ENV_MAP_SPHERICAL)) {
                    tex.envMapType = EnvMapType.spherical;
                } else if (s.equalsIgnoreCase(ENV_MAP_PLANAR)) {
                    tex.envMapType = EnvMapType.planar;
                } else if (s.equalsIgnoreCase(ENV_MAP_CUBIC_REFLECTION)) {
                    tex.envMapType = EnvMapType.cubic_reflection;
                } else if (s.equalsIgnoreCase(ENV_MAP_CUBIC_NORMAL)) {
                    tex.envMapType = EnvMapType.cubic_normal;
                } else {
                    throw new ENG_InvalidFormatParsingException();
                }
                envMapSet = true;
            } else if (s.equalsIgnoreCase(BRACKET_CLOSE)) {
                decrementBracketLevel();
                break;
            }
        }
        if (!textureSet) {
            // DONT ADD ANYTHING or else if we have an animated texture
            // we will load the texture frames plus one "" as the first frame
            // killing the mFrames in TextureUnitState
            //	tex.textureName.add("");
        }
        if (!animTexSet) {
            tex.animTexture = "";
        }
        if (!cubicSet) {
            tex.cubeName = "";
        }
        if ((!cubicSet) && (!animTexSet) && (!textureSet)) {
            throw new ENG_InvalidFormatParsingException("You must set at least one name for the texture " +
                    "unit to make any sense");
        }
        if (!bindingSet) {
            tex.bindingType = BindingType.fragment;
        }
        if (!contentSet) {
            tex.contentType = ContentType.named;
        }
        if (!coordSet) {
            tex.texCoordSet = 0;
        }
        if (!addressModeSet) {
            tex.addresingMode = TextureAddressingMode.TAM_WRAP;
        }
        if (!borderColourSet) {
            tex.textureBorder = new ENG_ColorValue(ENG_ColorValue.BLACK);
        }
        if (!filteringSet) {
            tex.filterTypeMin = FilterType.linear;
            tex.filterTypeMag = FilterType.linear;
            tex.filterTypeMip = FilterType.point;
        }
        if (!maxAnisotropySet) {
            tex.maxAnisotropy = 1;
        }
        if (filteringSet && !maxAnisotropySet && !ignoreMaxAni) {
            throw new ENG_InvalidFormatParsingException();
        }
        if (!mipmapBiasSet) {
            tex.mipmapBias = 0;
        }
        if (!colourOpSet) {
            tex.colourOpType = ColourOpType.modulate;
        }
        if (!envMapSet) {
            tex.envMapType = EnvMapType.off;
        }
        if (!textureUsage) {
            tex.usage = TextureUsage.TU_DEFAULT;
        }
        return tex;
    }

    private static ENG_PassResource parsePass(DataInputStream fp0, HashMap<String, ENG_Shader> shaderList) {
        String s = ENG_CompilerUtil.getNextWord(fp0);
        ENG_PassResource pass = new ENG_PassResource();
        boolean defName = false;
        if (s.equalsIgnoreCase(BRACKET_OPEN)) {
            pass.name = String.valueOf(passName++);
            defName = true;
        } else {
            pass.name = s;
        }
        checkNameParsed(fp0, defName);
        boolean depthChecked = false;
        boolean sceneBlended = false;
        boolean sceneBlendedSeparate = false;
        boolean sceneBlendOp = false;
        boolean depthWrite = false;
        while (true) {
            s = ENG_CompilerUtil.getNextWord(fp0);
            if (s == null) {
                throw new ENG_InvalidFormatParsingException();
            }
            if (s.equalsIgnoreCase(VERTEX_PROGRAM_REF)) {
                parseProgramParams(fp0, shaderList, pass, ENG_Shader.VERTEX_PROGRAM);
            } else if (s.equalsIgnoreCase(FRAGMENT_PROGRAM_REF)) {
                parseProgramParams(fp0, shaderList, pass, ENG_Shader.FRAGMENT_PROGRAM);
            } else if (s.equalsIgnoreCase(TEXTURE_UNIT)) {
                pass.textureUnits.add(parseTextureUnit(fp0));
            } else if (s.equalsIgnoreCase(DEPTH_CHECK)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                int depthCheck;
                try {
                    depthCheck = Integer.parseInt(s);
                } catch (NumberFormatException e) {
                    throw new ENG_InvalidFormatParsingException();
                }
                pass.depthCheck = (depthCheck != 0);
                depthChecked = true;
            } else if (s.equalsIgnoreCase(SCENE_BLEND)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(SB_ONE)) {
                    pass.sourceBlendFactor = SceneBlendFactor.SBF_ONE;
                } else if (s.equalsIgnoreCase(SB_ZERO)) {
                    pass.sourceBlendFactor = SceneBlendFactor.SBF_ZERO;
                } else if (s.equalsIgnoreCase(SB_DEST_COLOUR)) {
                    pass.sourceBlendFactor = SceneBlendFactor.SBF_DEST_COLOUR;
                } else if (s.equalsIgnoreCase(SB_SRC_COLOUR)) {
                    pass.sourceBlendFactor = SceneBlendFactor.SBF_SOURCE_COLOUR;
                } else if (s.equalsIgnoreCase(SB_ONE_MINUS_DEST_COLOUR)) {
                    pass.sourceBlendFactor = SceneBlendFactor.SBF_ONE_MINUS_DEST_COLOUR;
                } else if (s.equalsIgnoreCase(SB_ONE_MINUS_SRC_COLOUR)) {
                    pass.sourceBlendFactor = SceneBlendFactor.SBF_ONE_MINUS_SOURCE_COLOUR;
                } else if (s.equalsIgnoreCase(SB_DEST_ALPHA)) {
                    pass.sourceBlendFactor = SceneBlendFactor.SBF_DEST_ALPHA;
                } else if (s.equalsIgnoreCase(SB_SRC_ALPHA)) {
                    pass.sourceBlendFactor = SceneBlendFactor.SBF_SOURCE_ALPHA;
                } else if (s.equalsIgnoreCase(SB_ONE_MINUS_DEST_ALPHA)) {
                    pass.sourceBlendFactor = SceneBlendFactor.SBF_ONE_MINUS_DEST_ALPHA;
                } else if (s.equalsIgnoreCase(SB_ONE_MINUS_SRC_ALPHA)) {
                    pass.sourceBlendFactor = SceneBlendFactor.SBF_ONE_MINUS_SOURCE_ALPHA;
                } else {
                    throw new ENG_InvalidFormatParsingException();
                }
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(SB_ONE)) {
                    pass.destBlendFactor = SceneBlendFactor.SBF_ONE;
                } else if (s.equalsIgnoreCase(SB_ZERO)) {
                    pass.destBlendFactor = SceneBlendFactor.SBF_ZERO;
                } else if (s.equalsIgnoreCase(SB_DEST_COLOUR)) {
                    pass.destBlendFactor = SceneBlendFactor.SBF_DEST_COLOUR;
                } else if (s.equalsIgnoreCase(SB_SRC_COLOUR)) {
                    pass.destBlendFactor = SceneBlendFactor.SBF_SOURCE_COLOUR;
                } else if (s.equalsIgnoreCase(SB_ONE_MINUS_DEST_COLOUR)) {
                    pass.destBlendFactor = SceneBlendFactor.SBF_ONE_MINUS_DEST_COLOUR;
                } else if (s.equalsIgnoreCase(SB_ONE_MINUS_SRC_COLOUR)) {
                    pass.destBlendFactor = SceneBlendFactor.SBF_ONE_MINUS_SOURCE_COLOUR;
                } else if (s.equalsIgnoreCase(SB_DEST_ALPHA)) {
                    pass.destBlendFactor = SceneBlendFactor.SBF_DEST_ALPHA;
                } else if (s.equalsIgnoreCase(SB_SRC_ALPHA)) {
                    pass.destBlendFactor = SceneBlendFactor.SBF_SOURCE_ALPHA;
                } else if (s.equalsIgnoreCase(SB_ONE_MINUS_DEST_ALPHA)) {
                    pass.destBlendFactor = SceneBlendFactor.SBF_ONE_MINUS_DEST_ALPHA;
                } else if (s.equalsIgnoreCase(SB_ONE_MINUS_SRC_ALPHA)) {
                    pass.destBlendFactor = SceneBlendFactor.SBF_ONE_MINUS_SOURCE_ALPHA;
                } else {
                    throw new ENG_InvalidFormatParsingException();
                }
                sceneBlended = true;
            } else if (s.equalsIgnoreCase(SCENE_BLEND_OP)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(SB_OP_ADD)) {
                    pass.blendOperation = SceneBlendOperation.SBO_ADD;
                } else if (s.equalsIgnoreCase(SB_OP_SUBSTRACT)) {
                    pass.blendOperation = SceneBlendOperation.SBO_SUBTRACT;
                } else if (s.equalsIgnoreCase(SB_OP_REVERSE_SUBSTRACT)) {
                    pass.blendOperation = SceneBlendOperation.SBO_REVERSE_SUBTRACT;
                } else if (s.equalsIgnoreCase(SB_OP_MAX)) {
                    pass.blendOperation = SceneBlendOperation.SBO_MAX;
                } else if (s.equalsIgnoreCase(SB_OP_MIN)) {
                    pass.blendOperation = SceneBlendOperation.SBO_MIN;
                } else {
                    throw new ENG_InvalidFormatParsingException();
                }
                sceneBlendOp = true;
            } else if (s.equalsIgnoreCase(DEPTH_WRITE)) {
                s = ENG_CompilerUtil.getNextWord(fp0);
                if (s == null) {
                    throw new ENG_InvalidFormatParsingException();
                }
                if (s.equalsIgnoreCase(ON)) {
                    pass.depthWrite = true;
                } else if (s.equalsIgnoreCase(OFF)) {
                    pass.depthWrite = false;
                } else {
                    throw new ENG_InvalidFormatParsingException();
                }
                depthWrite = true;
            } else if (s.equalsIgnoreCase(BRACKET_CLOSE)) {
                decrementBracketLevel();
                break;
            }
        }
        if (!depthChecked) {
            //Enable by default
            pass.depthCheck = true;
        }
        if (!sceneBlended) {
            pass.sourceBlendFactor = SceneBlendFactor.SBF_ONE;
            pass.destBlendFactor = SceneBlendFactor.SBF_ZERO;
        }
        if (!sceneBlendOp) {
            pass.blendOperation = SceneBlendOperation.SBO_ADD;
        }
        if (!depthWrite) {
            pass.depthWrite = true;
        }
        return pass;
    }


    private static ENG_TechniqueResource parseTechnique(DataInputStream fp0, HashMap<String, ENG_Shader> shaderList) {
        String s = ENG_CompilerUtil.getNextWord(fp0);
        ENG_TechniqueResource tech = new ENG_TechniqueResource();
        boolean defName = false;
        if (s.equalsIgnoreCase(BRACKET_OPEN)) {
            tech.name = String.valueOf(techniqueName++);
            defName = true;
        } else {
            tech.name = s;
        }
        checkNameParsed(fp0, defName);
        while (true) {
            s = ENG_CompilerUtil.getNextWord(fp0);
            if (s == null) {
                throw new ENG_InvalidFormatParsingException();
            }
            if (s.equalsIgnoreCase(PASS)) {
                passName = 0;
                tech.passList.add(parsePass(fp0, shaderList));
            } else if (s.equalsIgnoreCase(BRACKET_CLOSE)) {
                decrementBracketLevel();
                break;
            }
        }
        return tech;
    }

    private static ENG_MaterialResource parseMaterial(DataInputStream fp0, HashMap<String, ENG_Shader> shaderList) {
        String s = ENG_CompilerUtil.getNextWord(fp0);
        ENG_MaterialResource mat = new ENG_MaterialResource();
        boolean defName = false;
        if (s.equalsIgnoreCase(BRACKET_OPEN)) {
            mat.name = String.valueOf(materialName++);
            defName = true;
        } else {
            mat.name = s;
        }
        checkNameParsed(fp0, defName);
        while (true) {
            s = ENG_CompilerUtil.getNextWord(fp0);
            if (s == null) {
                throw new ENG_InvalidFormatParsingException();
            }
            if (s.equalsIgnoreCase(TECHNIQUE)) {
                techniqueName = 0;
                mat.techniqueList.add(parseTechnique(fp0, shaderList));
            } else if (s.equalsIgnoreCase(BRACKET_CLOSE)) {
                decrementBracketLevel();
                break;
            }
        }
        return mat;
    }

    private void putInShaderList(ENG_Shader shader) {
        ENG_Shader put = shaderList.put(shader.name, shader);
        if (put != null) {
            throw new IllegalArgumentException(shader.name + " already loaded");
        }
    }

    public static class MaterialResult {
        public ENG_CompiledResource compiledResource;
        public HashMap<String, ENG_Shader> shaderList;
    }

    private final HashMap<String, ENG_Shader> shaderList = new HashMap<>();

    public HashMap<String, ENG_Shader> getShaderList() {
        return shaderList;
    }

    public void addToShaderList(HashMap<String, ENG_Shader> shaderList) {
        this.shaderList.putAll(shaderList);
    }

//    public void setShaderList(HashMap<String, ENG_Shader> shaderList) {
//        this.shaderList = shaderList;
//    }

    public ENG_CompiledResource compileImpl(String fileName, String path, boolean fromSDCard) {
        DataInputStream fp0 = null;
        try {
            fp0 = ENG_Resource.getFileAsStream(
                    fileName, path, fromSDCard);
            String s;
            String dir;
            String currentDir = "";
//            MaterialResult result = new MaterialResult();
//            result.shaderList = new HashMap<>();
            ArrayList<ENG_Shader> vertexProgram = new ArrayList<>();
            ArrayList<ENG_Shader> fragmentProgram = new ArrayList<>();
            ArrayList<ENG_MaterialResource> matRes = new ArrayList<>();
            while ((s = ENG_CompilerUtil.getNextWord(fp0)) != null) {
                dir = ENG_CompilerUtil.checkDirChange(s, fp0);
                if (dir != null) {
                    currentDir = dir;
                }
                if (s.equalsIgnoreCase(VERTEX_PROGRAM)) {
                    ENG_Shader program = parseProgram(fp0, currentDir, ENG_Shader.VERTEX_PROGRAM, fromSDCard);
                    putInShaderList(program);
                    vertexProgram.add(program);
                    //	vertexProgram.get(vertexProgram.size() - 1).id =
                    //		vertexProgram.size() - 1;
                } else if (s.equalsIgnoreCase(FRAGMENT_PROGRAM)) {
                    ENG_Shader program = parseProgram(fp0, currentDir, ENG_Shader.FRAGMENT_PROGRAM, fromSDCard);
                    putInShaderList(program);
                    fragmentProgram.add(program);
                                /*	fragmentProgram.get(fragmentProgram.size() - 1).id =
                                                fragmentProgram.size() - 1;*/
                } else if (s.equalsIgnoreCase(MATERIAL)) {
                    matRes.add(parseMaterial(fp0, shaderList));
                }
            }
            ENG_CompiledResource cr = new ENG_CompiledResource();
            cr.vertexProgramList = vertexProgram;
            cr.fragmentProgramList = fragmentProgram;
            cr.materialResourceList = matRes;
            return cr;
//            result.compiledResource = cr;
//            return result;
        } finally {
            ENG_CompilerUtil.close(fp0);
        }
    }
}
