/*
 * Created by Sebastian Bugiu on 4/9/23, 10:11 PM
 * sebastian.bugiu@headwayentertainment.net
 * Last modified 8/8/21, 5:10 AM
 * Copyright (c) 2023.
 * All rights reserved.
 */

package headwayent.hotshotengine.audio;

import headwayent.blackholedarksun.gamestatedebugger.FrameInterval;
import headwayent.hotshotengine.ENG_Math;
import headwayent.hotshotengine.ENG_Utility;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import com.badlogic.gdx.Audio;
import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.audio.Sound;

/**
 * This class loads all the sounds into a HashMap and communicates directly with
 * Android sound classes
 *
 * @author Alexandru
 */

public class ENG_SoundManager {

    private static class SoundInternal {

        private final String name; // For debugging
        private final Sound soundID;
        private final long duration;
        private final int priority; // 0 is the highest priority. 1, 2, 3 are lower priorities.
//        public float leftVolume = 1.0f, rightVolume = 1.0f; // These concepts are now obsolete.
//        public float volume; // 0.0f - 1.0f.

        public SoundInternal(String name, Sound soundID, long duration, int priority) {
            this.name = name;
            this.soundID = soundID;
            this.duration = duration;
            this.priority = priority;
        }

        public String getName() {
            return name;
        }

        public Sound getSoundID() {
            return soundID;
        }

        public long getDuration() {
            return duration;
        }

        public int getPriority() {
            return priority;
        }
    }

    private static class ActiveSound {
        private final long playId;
        private final long startTime;
        private final SoundInternal sound;
        private final int currentPriority; // This is derived based on SoundInternal priority and volume at sound start.
        private final boolean looping;

        public ActiveSound(long playId, long startTime, SoundInternal sound, int currentPriority, boolean isLooping) {
            this.playId = playId;
            this.startTime = startTime;
            this.sound = sound;
            this.currentPriority = currentPriority;
            this.looping = isLooping;
        }

        public long getPlayId() {
            return playId;
        }

        public long getStartTime() {
            return startTime;
        }

        public SoundInternal getSound() {
            return sound;
        }

        public int getCurrentPriority() {
            return currentPriority;
        }

        public boolean isLooping() {
            return looping;
        }
    }

    public static final int MAX_CONCURRENT_SOUNDS = 16;
    public static final int MAX_SOUND_VOLUME = 100;
    public static final int SOUND_LOWEST_PRIORITY = 999;
    private static final boolean DEBUG = false;
    public static final boolean IGNORE_SOUND = false; // For faster loading on mobile
    public static final boolean FORCE_PLAY = true;
    private final HashMap<String, SoundInternal> sounds = new HashMap<>();
    private final ArrayList<ActiveSound> activeSounds = new ArrayList<>(MAX_CONCURRENT_SOUNDS);
    //	private SoundPool player;
    private final Audio audioManager;
//	private Context context;

    private String path = "";
    private final boolean soundAvailable = true;

    /**
     * The constructor with arguments instantiates the audioManager object
     *
     * @param aContext The Context in which the stream will be played
     */
    public ENG_SoundManager(/*Context aContext*/) {

//		context = aContext;
        audioManager = Gdx.audio;
//				(AudioManager) context
//				.getSystemService(Context.AUDIO_SERVICE);
    }


    /**
     * Instantiates the HashMap and the SoundPool fields Loads all the sounds
     * into the HashMap
     */
    public void loadSounds() {

	/*	player = new SoundPool(6, AudioManager.STREAM_MUSIC, 0);
        if (player == null) {
			if (MainActivity.isDebugmode()) {
				throw new NullPointerException(
						"Sound player could not be initialized");
			} else {
				// Set sound not supported
				soundAvailable = false;
			}
		}*/
        //	sounds = new HashMap<String, Integer>();

        // sounds.put("explosion", player.load(context, R.raw.dulce, 1));
        // sounds.put("collision", player.load(context, R.raw.omfg, 1));
        // sounds.put("launch", player.load(context, R.raw.sursa, 1));

    }

    public long playSound(String name, int volume) {
        return playSound(name, volume, false);
    }

    /**
     * @param name Name of the stream to be played
     */
    public long playSound(String name) {
		
	/*	if (soundAvailable) {

		//	float volume = (float) getVolume(name) / 10;
			SoundInternal sound = sounds.get(name);
			if (sound == null) {
				throw new IllegalArgumentException(name + " is not a valid sound name");
			}
	
			player.play(sound.soundID, sound.leftVolume, sound.rightVolume, 10, 0, 1);
		}*/
        return playSound(name, 100, false);

    }

    /**
     * @param name Name of the stream to be played
     * @param loop If is -1 the stream will be looping
     *             returning -1 means invalid sound
     */
    public long playSound(String name, int volume, boolean isLooping) {

        return playSound(name, volume, 0.0f, isLooping);

    }

    public long playSound(String name, int volume, float pan, boolean isLooping) {
        if (IGNORE_SOUND) {
            return -1;
        }

        if (volume == 0) {
            return -1;
        }

        if (volume < 0 || volume > 100) {
            throw new IllegalArgumentException("volume sent is: " + volume);
        }
        long playID = -1;
        if (soundAvailable) {
            //	float volume = (float) getVolume(name) / 10;
            SoundInternal sound = sounds.get(name);
            if (sound == null) {
                throw new IllegalArgumentException(name + " is not a valid sound name");
            }
            if (!shouldPlaySound(sound)) {
                return playID;
            }
		/*	player.play(sound.soundID, sound.leftVolume, sound.rightVolume, 10,
					isLooping, 1);*/
            float vol = (float) volume / (float) MAX_SOUND_VOLUME;
            vol = ENG_Math.clamp(vol, 0.0f, 1.0f);

            pan = ENG_Math.clamp(pan, -1.0f, 1.0f);
            if (isLooping) {
                playID = sound.getSoundID().loop(vol, 1.0f, pan);
            } else {
                playID = sound.getSoundID().play(vol, 1.0f, pan);
            }
            if (playID == -1) {
//				throw new ENG_SoundException("Could not play sound: " + name);
                System.out.println("Could not play sound: " + name);
            } else {
                activeSounds.add(new ActiveSound(playID, ENG_Utility.currentTimeMillis(),
                        sound, volume == 0 ? SOUND_LOWEST_PRIORITY : sound.getPriority(), isLooping));
            }
            if (DEBUG) {
                System.out.println("Starting sound " + name + " playId " + playID + " sound volume: " + volume + " pan: " + pan);
            }

        }
        return playID;
    }

    private boolean shouldPlaySound(SoundInternal soundToBePlayed) {
        // Remove sounds that have finished playing from the active sounds list.
        removeFinishedSounds();

        if (DEBUG) {
            System.out.println("Current active sounds num: " + activeSounds.size());
            System.out.println("Attempting to play " + soundToBePlayed.getName() + " with default priority " + soundToBePlayed.getPriority());
        }

        // If we still have all sounds playing we need to try to make room for the new
        // sound based on priority.
        if (activeSounds.size() == MAX_CONCURRENT_SOUNDS) {
            if (DEBUG) {
                System.out.println("Max concurrent sounds " + MAX_CONCURRENT_SOUNDS + " reached");
            }
            ActiveSound activeSoundWithLowestPriority = null;
            for (ActiveSound activeSound : activeSounds) {
                if (activeSoundWithLowestPriority == null ||
                        activeSound.getCurrentPriority() > activeSoundWithLowestPriority.getCurrentPriority()) {
                    activeSoundWithLowestPriority = activeSound;
                }
            }
            // If there were more with the lowest priority in the active sounds list
            // then we selected the one longest playing. If the new sound has the same
            // priority then we evict the one with the same priority but longest playing.
            if (activeSoundWithLowestPriority.getCurrentPriority() >= soundToBePlayed.getPriority()) {
                if (DEBUG) {
                    System.out.println("Evicting sound " +
                            activeSoundWithLowestPriority.getSound().getName() +
                            " with lowest priority " +
                            activeSoundWithLowestPriority.getCurrentPriority() +
                            " to make room for " + soundToBePlayed.getName() +
                            " with priority " + soundToBePlayed.getPriority());
                }
                activeSounds.remove(activeSoundWithLowestPriority);
                stopSound(activeSoundWithLowestPriority.getSound(), activeSoundWithLowestPriority.getPlayId());
            } else {
                if (DEBUG) {
                    System.out.println("Sound " + soundToBePlayed.getName() + " with default priority " + soundToBePlayed.getPriority() + " could not be played");
                }
                return false;
            }
        }
        return true;
    }

    private void removeFinishedSounds() {
        for (Iterator<ActiveSound> iterator = activeSounds.iterator(); iterator.hasNext(); ) {
            ActiveSound activeSound = iterator.next();
            if (!activeSound.isLooping() && ENG_Utility.hasTimePassed(FrameInterval.SOUND_SHOULD_PLAY_SOUND +
                    activeSound.getSound().getName() + " " +
                    activeSound.getPlayId(),
                    activeSound.getStartTime(),
                    activeSound.getSound().getDuration())) {
//                System.out.println("Removing sound that finished playing " + activeSound.getSound().getName() +
//                        " with play id " + activeSound.getPlayId() +
//                        " and priority " + activeSound.getCurrentPriority());
                iterator.remove();
            }
        }
    }

    private boolean isSoundActive(long playId) {
        for (ActiveSound activeSound : activeSounds) {
            if (activeSound.getPlayId() == playId) {
                return true;
            }
        }
        return false;
    }

    public long getSoundDuration(String name) {
        SoundInternal sound = sounds.get(name);
        if (sound == null) {
            throw new IllegalArgumentException(name + " is not a valid sound name");
        }
        return sound.getDuration();
    }

    /**
     * Loads a sound into the HashMap and to the SoundPool itself
     *  @param name Name of the sound to be loaded for a specified path The path
     *             is specified in the class field "path"
     * @param duration
     * @param priority
     */
    public void loadSound(String name, String filename, long duration, int priority) {

//		throw new UnsupportedOperationException("Must use the id version");

//		String baseName = FilenameUtils.getBaseName(name);
        if (IGNORE_SOUND) {
            return;
        }
        sounds.put(name, new SoundInternal(name, Gdx.audio.newSound(Gdx.files.local(filename)), duration, priority));
    }

    public void loadSound(String name, int handle) {
//		sounds.put(name, new SoundInternal(player.load(context, handle, 0)));
        throw new UnsupportedOperationException();
    }

    /**
     * Pauses a stream from playing
     *
     * @param name Name of the sound to be paused
     */
    public void pauseSound(String name) {
        if (IGNORE_SOUND) {
            return;
        }

        if (soundAvailable) {
            // it will have no effect if the stream is not playing
            SoundInternal sound = sounds.get(name);
            if (sound == null) {
                throw new IllegalArgumentException(name + " is not a valid sound name");
            }
//			player.pause(sound.soundID);
//			sound.soundID.
//			throw new UnsupportedOperationException();
            sound.getSoundID().pause();
        }

    }

    public void pauseSound(String name, long id) {
        if (IGNORE_SOUND) {
            return;
        }
        if (soundAvailable) {
            SoundInternal sound = sounds.get(name);
            if (sound == null) {
                throw new IllegalArgumentException(name + " is not a valid sound name");
            }
            sound.getSoundID().pause(id);
        }
    }

    public void resumeSound(String name) {
        if (IGNORE_SOUND) {
            return;
        }
        if (soundAvailable) {
            SoundInternal sound = sounds.get(name);
            if (sound == null) {
                throw new IllegalArgumentException(name + " is not a valid sound name");
            }
            sound.getSoundID().resume();
        }
    }

    public void resumeSound(String name, long id) {
        if (IGNORE_SOUND) {
            return;
        }
        if (soundAvailable) {
            SoundInternal sound = sounds.get(name);
            if (sound == null) {
                throw new IllegalArgumentException(name + " is not a valid sound name");
            }
            sound.getSoundID().resume(id);
        }
    }

    /**
     * Stops a stream from playing
     *
     * @param name Name of the sound to be stopped
     */
    public void stopSound(String name) {

        if (IGNORE_SOUND) {
            return;
        }

        if (soundAvailable) {
            // it will have no effect if the stream is not playing
            SoundInternal sound = sounds.get(name);
            if (sound == null) {
                throw new IllegalArgumentException(name + " is not a valid sound name");
            }
//			player.stop(sound.soundID);
            sound.getSoundID().stop();
        }

    }

    public void stopSound(String name, long id) {
        if (IGNORE_SOUND) {
            return;
        }
        if (soundAvailable) {
            SoundInternal sound = sounds.get(name);
            if (sound == null) {
                throw new IllegalArgumentException(name + " is not a valid sound name");
            }
            for (Iterator<ActiveSound> iterator = activeSounds.iterator(); iterator.hasNext(); ) {
                ActiveSound activeSound = iterator.next();
                if (activeSound.getPlayId() == id) {
                    if (DEBUG) {
                        System.out.println("Removing sound by stopping " + activeSound.getSound().getName() +
                                " with play id " + activeSound.getPlayId() +
                                " and" +
                                " priority " + activeSound.getCurrentPriority());
                    }
                    iterator.remove();
                }
            }

            stopSound(sound, id);
        }
    }

    private void stopSound(SoundInternal sound, long id) {
        sound.getSoundID().stop(id);
    }

    public void setVolume(String name, long id, int volume, boolean vibrate) {
        if (IGNORE_SOUND) {
            return;
        }
        volume = ENG_Math.clamp(volume, 0, MAX_SOUND_VOLUME);
        SoundInternal sound = sounds.get(name);
        if (sound == null) {
            throw new IllegalArgumentException(name + " is not a valid sound name");
        }
        float vol = volume * 0.01f;
        vol = ENG_Math.clamp(vol, 0.0f, 1.0f);
        if (DEBUG) {
            System.out.println(name + " playId " + id + " sound volume: " + volume);
        }
        sound.getSoundID().setVolume(id, vol);
//        sound.leftVolume = vol;
//        sound.rightVolume = vol;
    }

    /**
     * Sets the Volume for a specified stream
     *
     * @param name   The name of the stream
     * @param volume The volume
     */
    public void setVolume(String name, int volume, boolean vibrate) {
        if (IGNORE_SOUND) {
            return;
        }
        volume = ENG_Math.clamp(volume, 0, MAX_SOUND_VOLUME);

        SoundInternal sound = sounds.get(name);
        if (sound == null) {
            throw new IllegalArgumentException(name + " is not a valid sound name");
        }
	/*	int streamMaxVolume = audioManager.getStreamMaxVolume(integer);
		// Map volume between 0 - 100 to 0 - streamMaxVolume
		float mapping = ((float) volume) * 0.01f;
		audioManager.setStreamVolume(integer, (int) (mapping * streamMaxVolume),
				AudioManager.FLAG_VIBRATE);
				*/
        float vol = volume * 0.01f;
        vol = ENG_Math.clamp(vol, 0.0f, 1.0f);
        if (DEBUG) {
            System.out.println(name + " sound volume: " + volume);
        }
//		player.setVolume(sound.soundID, vol, vol);
//        sound.leftVolume = vol;
//        sound.rightVolume = vol;
    }

    public void setVolume(String name, int leftVolume, int rightVolume,
                          boolean vibrate) {
        if (IGNORE_SOUND) {
            return;
        }
        leftVolume = ENG_Math.clamp(leftVolume, 0, MAX_SOUND_VOLUME);
        rightVolume = ENG_Math.clamp(rightVolume, 0, MAX_SOUND_VOLUME);

        SoundInternal sound = sounds.get(name);
        if (sound == null) {
            throw new IllegalArgumentException(name + " is not a valid sound name");
        }
	/*	int streamMaxVolume = audioManager.getStreamMaxVolume(integer);
		// Map volume between 0 - 100 to 0 - streamMaxVolume
		float mapping = ((float) volume) * 0.01f;
		audioManager.setStreamVolume(integer, (int) (mapping * streamMaxVolume),
				AudioManager.FLAG_VIBRATE);
				*/
        float lvol = leftVolume * 0.01f;
        float rvol = rightVolume * 0.01f;
        if (DEBUG) {
            System.out.println(name + " sound volume: " + leftVolume);
            System.out.println(name + " sound volume: " + rightVolume);
        }
//		player.setVolume(sound.soundID, lvol, rvol);
//        sound.leftVolume = lvol;
//        sound.rightVolume = rvol;
    }

    /**
     * Returns an int representing the volume of a specified stream @ name The
     * name of the stream
     */
    public int getVolume(String name) {
        return 0;
//        if (IGNORE_SOUND) {
//            return 0;
//        }
//
//        SoundInternal sound = sounds.get(name);
//        if (sound == null) {
//            throw new IllegalArgumentException(name + " is not a valid sound name");
//        }
//        //	int volume = audioManager.getStreamVolume(integer);
//
//        return (int) ((sound.leftVolume + sound.rightVolume) * 0.5f * 100.0f);
    }

    public int getLeftVolume(String name) {
        return 0;
//        if (IGNORE_SOUND) {
//            return 0;
//        }
//        SoundInternal sound = sounds.get(name);
//        if (sound == null) {
//            throw new IllegalArgumentException(name + " is not a valid sound name");
//        }
//        //	int volume = audioManager.getStreamVolume(integer);
//
//        return (int) (sound.leftVolume * 100.0f);
    }

    public int getRightVolume(String name) {
        return 0;
//        if (IGNORE_SOUND) {
//            return 0;
//        }
//        SoundInternal sound = sounds.get(name);
//        if (sound == null) {
//            throw new IllegalArgumentException(name + " is not a valid sound name");
//        }
//        //	int volume = audioManager.getStreamVolume(integer);
//
//        return (int) (sound.rightVolume * 100.0f);
    }

    /**
     * @param name The name of the stream
     * @return If true, it means the stream exists in the hashMap
     */

    public boolean fileExists(String name) {

        return sounds.containsKey(name);

    }

    /**
     * Initiates the path variable
     *
     * @param name The path itself
     */
    public void setPath(String name) {

        path = name;
    }


    public void disposeSound(String name) {
        
        if (IGNORE_SOUND) {
            return;
        }
        SoundInternal sound = sounds.remove(name);
        if (sound == null) {
            throw new IllegalArgumentException(name + " is not a valid sound name");
        }
        sound.getSoundID().dispose();
    }


    public void setPan(String name, long id, float pan, int volume) {
        
        if (IGNORE_SOUND) {
            return;
        }

        // Optimization to no longer pan and set volume for stopped sounds.
        removeFinishedSounds();
        if (!isSoundActive(id)) {
            return;
        }

        SoundInternal sound = sounds.get(name);
        if (sound == null) {
            throw new IllegalArgumentException(name + " is not a valid sound name");
        }
        float vol = volume * 0.01f;
        vol = ENG_Math.clamp(vol, 0.0f, 1.0f);
        pan = ENG_Math.clamp(pan, -1.0f, 1.0f);
        if (DEBUG) {
//            System.out.println(name + " playId " + id + " sound volume: " + volume + " pan: " + pan);
        }
//		sound.soundID.setVolume(id, vol);
//        sound.leftVolume = vol;
//        sound.rightVolume = vol;
        pan = ENG_Math.clamp(pan, -1.0f, 1.0f);
        sound.getSoundID().setPan(id, pan, vol);
    }


    public void disposeOfAllSounds() {
        
        if (IGNORE_SOUND) {
            return;
        }
        for (SoundInternal sound : sounds.values()) {
            sound.getSoundID().dispose();
        }
        sounds.clear();
    }

}
