/*
 * Created by Sebastian Bugiu on 4/9/23, 10:11 PM
 * sebastian.bugiu@headwayentertainment.net
 * Last modified 5/17/21, 9:20 PM
 * Copyright (c) 2023.
 * All rights reserved.
 */

package headwayent.hotshotengine.audio;


/**
 * @author Alexandru This class implements the sound engine interface and makes
 *         calls to a ENG_SoundManager object
 */

public class ENG_Sound implements ENG_ISoundRoot {

    /**
     * soundManger - a SoundManger object context - the Context passed in the
     * constructor
     */
    private final ENG_SoundManager soundManager;
    // private Context context;
    private boolean soundsActive = true;

    /**
     * The constructor with arguments instantiates the soundManager object and
     * calls the loadSounds() method
     *
     * @param aContext The Context in which the stream will be played
     */
    public ENG_Sound(/* Context aContext */) {

        // context = aContext;
        soundManager = new ENG_SoundManager(/* context */);
        soundManager.loadSounds();
    }

    public long playSound(String name) {
        return playSound(name, 100);
    }

    // ENG_Sound() { } //TODO: trebuie completat cu un context default

    /**
     * @param name Name of the stream to be played
     */
    public long playSound(String name, int volume) {

        if (soundManager.fileExists(name)) {

            long id = -1;
            if (soundsActive || ENG_SoundManager.FORCE_PLAY) {
                id = soundManager.playSound(name, volume);
            }

            return id;
        } else {

            if (!ENG_SoundManager.IGNORE_SOUND) {
                throw new IllegalArgumentException("File name not found.");
            } else {
                return -1;
            }
        }

    }

    public long playSound(String name, PlayType playType) {
        return playSound(name, 100, playType);
    }

    /**
     * @param name     Name of the stream to be played
     * @param playType The PlayType value can be PLAY_LOOP or PLAY_ONCE
     */
    public long playSound(String name, int volume, PlayType playType) {
        return playSound(name, volume, 0.0f, playType);

    }

    public long playSound(String name, int volume, float pan, PlayType playType){
        if (soundManager.fileExists(name)) {

            boolean isLooping = playType == PlayType.PLAY_LOOP;

//			if (playType == PlayType.PLAY_LOOP) {
//
//				isLooping = true;
//
//			} else {
//
//				isLooping = 0;
//			}

            long id = -1;
            if (soundsActive || ENG_SoundManager.FORCE_PLAY) {
                id = soundManager.playSound(name, volume, pan, isLooping);
            }
            return id;

        } else {

            if (!ENG_SoundManager.IGNORE_SOUND) {
                throw new IllegalArgumentException("File name not found.");
            } else {
                return -1;
            }

        }
    }

    public long getSoundDuration(String name) {
        return soundManager.getSoundDuration(name);
    }

    /**
     * Loads a sound into the HashMap and to the SoundPool itself
     *
     * @param name Name of the sound to be loaded for a specified path The path
     *             must be specified using setPath()
     */
    public void loadSound(String name, String filename, long duration, int priority) {

        soundManager.loadSound(name, filename, duration, priority);
    }

    public void loadSound(String name, int handle) {
        soundManager.loadSound(name, handle);
    }

    /**
     * Pauses a stream from playing
     *
     * @param name Name of the sound to be paused
     */
    public void pauseSound(String name) {

        // it will have no effect if the stream is not playing
        soundManager.pauseSound(name);

    }

    public void resumeSound(String name) {
        soundManager.resumeSound(name);
    }

    /**
     * Stops a stream from playing
     *
     * @param name Name of the sound to be stopped
     */
    public void stopSound(String name) {

        // it will have no effect if the stream is not playing
        soundManager.stopSound(name);

    }

    /**
     * This method changes the volume of a specified stream
     *
     * @param name
     *            The name of the stream
     * @param volume
     *            The new value ( must be between 0-7 )
     */
//	public void setVolume(String name, int volume) {
//
//		soundManager.setVolume(name, volume, false);
//
//	}
//
//	public void setVolume(String name, int leftVolume, int rightVolume) {
//
//		soundManager.setVolume(name, leftVolume, rightVolume, false);
//
//	}

    /**
     * This method changes the volume of a specified stream
     *
     * @param name
     *            The name of the stream
     * @param volume
     *            The new value ( must be between 0-7 )
     */
    /*
	 * public void setVolume(String name, int volume, boolean vibrate) {
	 * 
	 * soundManager.setVolume(name, volume, vibrate);
	 * 
	 * }
	 */

    /**
     * Returns an int representing the volume of a specified stream @ name The
     * name of the stream
     */
    public int getVolume(String name) {

        if (soundManager.fileExists(name)) {

            return soundManager.getVolume(name);

        } else {

            if (!ENG_SoundManager.IGNORE_SOUND) {
                throw new IllegalArgumentException("File name not found.");
            } else {
                return 0;
            }
        }

    }

    /**
     * Sets the path or the sounds
     *
     * @param name The path itself
     */
    public void setPath(String name) {

        soundManager.setPath(name);
    }

    @Override
    public void enableSounds() {
        
        soundsActive = true;
    }

    @Override
    public void disableSounds() {
        
        soundsActive = false;
    }

    @Override
    public boolean isEnabled() {
        
        return soundsActive;
    }

    @Override
    public void pauseSound(String name, long id) {
        
        soundManager.pauseSound(name, id);
    }

    @Override
    public void resumeSound(String name, long id) {
        
        soundManager.resumeSound(name, id);
    }

    @Override
    public void stopSound(String name, long id) {
        
        soundManager.stopSound(name, id);
    }

    @Override
    public void setVolume(String name, long id, int volume) {
        
        soundManager.setVolume(name, id, volume, false);
    }

    @Override
    public void disposeSound(String name) {
        
        soundManager.disposeSound(name);
    }

    @Override
    public void setPan(String name, long id, float pan, int volume) {
        
        soundManager.setPan(name, id, pan, volume);
    }

    @Override
    public void disposeOfAllSounds() {
        
        soundManager.disposeOfAllSounds();
    }

}
