/*
 * Created by Sebastian Bugiu on 4/9/23, 10:11 PM
 * sebastian.bugiu@headwayentertainment.net
 * Last modified 8/6/21, 4:51 PM
 * Copyright (c) 2023.
 * All rights reserved.
 */

package headwayent.hotshotengine;

import headwayent.blackholedarksun.APP_Game;
import headwayent.blackholedarksun.MainApp;
import headwayent.hotshotengine.basictypes.ENG_Integer;
import headwayent.hotshotengine.exception.ENG_InvalidFieldStateException;
import headwayent.hotshotengine.exception.ENG_InvalidPathException;
import headwayent.hotshotengine.renderer.ENG_BlendMode.LayerBlendOperation;
import headwayent.hotshotengine.renderer.ENG_Common.FilterOptions;
import headwayent.hotshotengine.renderer.ENG_GpuProgram.GpuProgramType;
import headwayent.hotshotengine.renderer.ENG_GpuProgramParameters;
import headwayent.hotshotengine.renderer.ENG_GpuProgramParameters.AutoConstantType;
import headwayent.hotshotengine.renderer.ENG_HighLevelGpuProgram;
import headwayent.hotshotengine.renderer.ENG_HighLevelGpuProgramManager;
import headwayent.hotshotengine.renderer.ENG_Material;
import headwayent.hotshotengine.renderer.ENG_MaterialManager;
import headwayent.hotshotengine.renderer.ENG_Pass;
import headwayent.hotshotengine.renderer.ENG_Technique;
import headwayent.hotshotengine.renderer.ENG_Texture.TextureMipmap;
import headwayent.hotshotengine.renderer.ENG_Texture.TextureUsage;
import headwayent.hotshotengine.renderer.ENG_TextureUnitState;
import headwayent.hotshotengine.renderer.ENG_TextureUnitState.BindingType;
import headwayent.hotshotengine.renderer.ENG_TextureUnitState.ContentType;
import headwayent.hotshotengine.renderer.ENG_TextureUnitState.UVWAddressingMode;
import headwayent.hotshotengine.renderer.nullrendersystem.NullProgramFactory;
import headwayent.hotshotengine.renderer.opengles.glsl.GLSLProgramFactory;
import headwayent.hotshotengine.resource.ENG_CompiledResource;
import headwayent.hotshotengine.resource.ENG_MaterialResource;
import headwayent.hotshotengine.resource.ENG_PassResource;
import headwayent.hotshotengine.resource.ENG_ResourceLoader;
import headwayent.hotshotengine.resource.ENG_ResourceLoaderTask;
import headwayent.hotshotengine.resource.ENG_Shader;
import headwayent.hotshotengine.resource.ENG_ShaderNamedParams;
import headwayent.hotshotengine.resource.ENG_ShaderParam;
import headwayent.hotshotengine.resource.ENG_TechniqueResource;
import headwayent.hotshotengine.resource.ENG_TextureUnitResource;
import headwayent.hotshotengine.resource.ENG_TextureUnitResource.ColourOpType;
import headwayent.hotshotengine.resource.ENG_TextureUnitResource.CubeType;
import headwayent.hotshotengine.resource.ENG_TextureUnitResource.FilterType;
import headwayent.hotshotengine.resource.shadertype.ENG_ShaderTypeBool;
import headwayent.hotshotengine.resource.shadertype.ENG_ShaderTypeFloat;
import headwayent.hotshotengine.resource.shadertype.ENG_ShaderTypeInt;
import headwayent.hotshotengine.resource.shadertype.ENG_ShaderTypeStandard;
import headwayent.hotshotengine.scriptcompiler.ENG_CompilerUtil;
import headwayent.hotshotengine.scriptcompiler.ENG_MaterialCompiler;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeSet;
import java.util.concurrent.locks.ReentrantLock;

public class ENG_MaterialLoader {

    private static final TreeSet<String> materialsLoadedList = new TreeSet<>();
    private static final HashMap<String, ENG_Shader> shaderList = new HashMap<>();
    private static final HashMap<String, ENG_CompiledResource> compiledResourceList = new HashMap<>();
    private static final ReentrantLock shaderListLock = new ReentrantLock();
    private static final ReentrantLock compiledResourceListLock = new ReentrantLock();

    public static void clear() {
        clearMaterialList();
        clearShaderList();
        clearCompiledResourceList();
    }

    public static void clearMaterialList() {
        materialsLoadedList.clear();
    }

    public static void clearShaderList() {
        shaderListLock.lock();
        try {
            shaderList.clear();
        } finally {
            shaderListLock.unlock();
        }
    }

    public static HashMap<String, ENG_Shader> getShaderList() {
        shaderListLock.lock();
        try {
            return new HashMap<>(shaderList);
        } finally {
            shaderListLock.unlock();
        }
    }

    private static void addToShaderList(HashMap<String, ENG_Shader> shaderList) {
        shaderListLock.lock();
        try {
            for (Map.Entry<String, ENG_Shader> shader : shaderList.entrySet()) {
                ENG_Shader put = ENG_MaterialLoader.shaderList.put(shader.getKey(), shader.getValue());
//                if (put != null) {
//                    throw new IllegalArgumentException(put.name + " already loaded");
//                }
            }

        } finally {
            shaderListLock.unlock();
        }
    }

    public static void clearCompiledResourceList() {
        compiledResourceListLock.lock();
        try {
            compiledResourceList.clear();
        } finally {
            compiledResourceListLock.unlock();
        }
    }

    private static void addToCompiledResourceList(String filename, ENG_CompiledResource compiledResource) {
        compiledResourceListLock.lock();
        try {
            // Shitty hack. We need to unload every compiled resource that contains a skybox material when loading a new skybox material.
            if (filename.startsWith(APP_Game.FOLDER_RAW + File.separator + "mat_skybox")) {
                ArrayList<String> skyboxMats = new ArrayList<>();
                for (Map.Entry<String, ENG_CompiledResource> f : compiledResourceList.entrySet()) {
                    if (f.getKey().startsWith(APP_Game.FOLDER_RAW + File.separator + "mat_skybox")) {
                        skyboxMats.add(f.getKey());
                    }
                }
                for (String skyboxFilename : skyboxMats) {
                    compiledResourceList.remove(skyboxFilename);
                }


            }
            ENG_CompiledResource put = compiledResourceList.put(filename, compiledResource);
//            if (put != null) {
//                throw new IllegalArgumentException(filename + " material already loaded");
//            }
        } finally {
            compiledResourceListLock.unlock();
        }
    }

    private static void createPrograms(ArrayList<ENG_Shader> programList) {
        for (ENG_Shader shader : programList) {
            GpuProgramType type;
            switch (shader.type) {
                case ENG_Shader.VERTEX_PROGRAM:
                    type = GpuProgramType.GPT_VERTEX_PROGRAM;
                    break;
                case ENG_Shader.FRAGMENT_PROGRAM:
                    type = GpuProgramType.GPT_FRAGMENT_PROGRAM;
                    break;
                default:
                    throw new IllegalArgumentException("Invalid shader type." +
                            " Geometry programs not yet supported!");
            }
            ENG_HighLevelGpuProgram prg =
                    ENG_HighLevelGpuProgramManager.getSingleton().createProgram(
                            shader.name, MainApp.getMainThread().getApplicationSettings().applicationMode == MainApp.Mode.CLIENT ?
                                    GLSLProgramFactory.LANGUAGE_NAME : NullProgramFactory.LANGUAGE_NAME, type);
            prg.setSourceFile(shader.fileName, shader.path);
            prg.setMorphAnimationIncluded(shader.morphAnimation);
            prg.setPoseAnimationIncluded((short) shader.poseAnimationNum);
            prg.setSkeletalAnimationIncluded(shader.skeletalAnimation);
            prg.setVertexTextureFetchRequired(shader.useVertexTextureFetch);
            prg.setAdjacencyInfoRequired(shader.useAdjacencyInfo);

            if (prg.isSupported()) {
                ENG_GpuProgramParameters params = prg.getDefaultParameters();
                translateProgramParameters(shader, params);
            }
        }
    }

    private static void translateProgramParameters(ENG_Shader shader,
                                                   ENG_GpuProgramParameters params) {
        
        //Get the default params
        ENG_ShaderNamedParams namedParam = shader.defaultParam.namedParam;
        readParams(shader, params, namedParam);

        extractStandardTypes(shader, params);

		
	/*	for (ENG_ShaderParam pList : shader.paramList) {
			readParams(shader, params, pList.namedParam);
		}*/
    }

    private static void readParams(ENG_Shader shader, ENG_GpuProgramParameters params, ENG_ShaderNamedParams namedParam) {
        if (namedParam == null) {
            return;
        }
        //First get the named params
        //The glsl does support boolean but convert to int for now...
        for (ENG_ShaderTypeBool b : namedParam.BoolList) {
            ENG_Integer[] arr = ENG_Utility.convertArrayFromBooleanPrimitiveToIntObj(b.value, 0, 1);
            params.setNamedConstant(b.name, arr, 1, 1);
        }
        for (ENG_ShaderTypeBool b : namedParam.BVec2List) {
            ENG_Integer[] arr = ENG_Utility.convertArrayFromBooleanPrimitiveToIntObj(b.value, 0, 2);
            params.setNamedConstant(b.name, arr, 1, 2);
        }
        for (ENG_ShaderTypeBool b : namedParam.BVec3List) {
            ENG_Integer[] arr = ENG_Utility.convertArrayFromBooleanPrimitiveToIntObj(b.value, 0, 3);
            params.setNamedConstant(b.name, arr, 1, 3);
        }
        for (ENG_ShaderTypeBool b : namedParam.BVec4List) {
            ENG_Integer[] arr = ENG_Utility.convertArrayFromBooleanPrimitiveToIntObj(b.value, 0, 4);
            params.setNamedConstant(b.name, arr, 1, 4);
        }

        //Fill the integers
        for (ENG_ShaderTypeInt i : namedParam.IntList) {
            params.setNamedConstant(i.name, i.value, 1, 1);
        }
        for (ENG_ShaderTypeInt i : namedParam.IVec2List) {
            params.setNamedConstant(i.name, i.value, 1, 2);
        }
        for (ENG_ShaderTypeInt i : namedParam.IVec3List) {
            params.setNamedConstant(i.name, i.value, 1, 3);
        }
        for (ENG_ShaderTypeInt i : namedParam.IVec4List) {
            params.setNamedConstant(i.name, i.value, 1, 4);
        }

        //Fill the matrices
        for (ENG_ShaderTypeFloat f : namedParam.Mat2List) {
            throw new UnsupportedOperationException("Add support for 2x2 matrices later");
        }
        for (ENG_ShaderTypeFloat f : namedParam.Mat3List) {
            params.setNamedConstant(f.name, f.value, 1, 9);
        }
        for (ENG_ShaderTypeFloat f : namedParam.Mat4List) {
            params.setNamedConstant(f.name, f.value, 1, 16);
        }

        //Fill the floats
        for (ENG_ShaderTypeFloat f : namedParam.FloatList) {
            params.setNamedConstant(f.name, f.value, 1, 1);
        }
        for (ENG_ShaderTypeFloat f : namedParam.Vec2List) {
            params.setNamedConstant(f.name, f.value, 1, 2);
        }
        for (ENG_ShaderTypeFloat f : namedParam.Vec3List) {
            params.setNamedConstant(f.name, f.value, 1, 3);
        }
        for (ENG_ShaderTypeFloat f : namedParam.Vec4List) {
            params.setNamedConstant(f.name, f.value, 1, 4);
        }


    }

    private static void extractStandardTypes(ENG_Shader shader, ENG_GpuProgramParameters params) {
        if (shader.defaultParam.standardType == null) {
            return;
        }
        //Then do the named auto params
        for (ENG_ShaderTypeStandard s : shader.defaultParam.standardType) {
            AutoConstantType type;
            switch (s.type) {
                case ENG_ShaderTypeStandard.ACT_WORLD_MATRIX:
                    type = AutoConstantType.ACT_WORLD_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_WORLD_MATRIX:
                    type = AutoConstantType.ACT_INVERSE_WORLD_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_TRANSPOSE_WORLD_MATRIX:
                    type = AutoConstantType.ACT_TRANSPOSE_WORLD_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_WORLD_MATRIX:
                    type = AutoConstantType.ACT_INVERSE_TRANSPOSE_WORLD_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_WORLD_MATRIX_ARRAY_3X4:
                    type = AutoConstantType.ACT_WORLD_MATRIX_ARRAY_3x4;
                    break;
                case ENG_ShaderTypeStandard.ACT_VIEW_MATRIX:
                    type = AutoConstantType.ACT_VIEW_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_VIEW_MATRIX:
                    type = AutoConstantType.ACT_INVERSE_VIEW_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_TRANSPOSE_VIEW_MATRIX:
                    type = AutoConstantType.ACT_TRANSPOSE_VIEW_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_VIEW_MATRIX:
                    type = AutoConstantType.ACT_INVERSE_TRANSPOSE_VIEW_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_PROJECTION_MATRIX:
                    type = AutoConstantType.ACT_PROJECTION_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_PROJECTION_MATRIX:
                    type = AutoConstantType.ACT_INVERSE_PROJECTION_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_TRANSPOSE_PROJECTION_MATRIX:
                    type = AutoConstantType.ACT_TRANSPOSE_PROJECTION_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_PROJECTION_MATRIX:
                    type = AutoConstantType.ACT_INVERSE_TRANSPOSE_PROJECTION_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_WORLDVIEW_MATRIX:
                    type = AutoConstantType.ACT_WORLDVIEW_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_WORLDVIEW_MATRIX:
                    type = AutoConstantType.ACT_INVERSE_WORLDVIEW_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_TRANSPOSE_WORLDVIEW_MATRIX:
                    type = AutoConstantType.ACT_TRANSPOSE_WORLDVIEW_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_WORLDVIEW_MATRIX:
                    type = AutoConstantType.ACT_INVERSE_TRANSPOSE_WORLDVIEW_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_VIEWPROJ_MATRIX:
                    type = AutoConstantType.ACT_VIEWPROJ_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_VIEWPROJ_MATRIX:
                    type = AutoConstantType.ACT_INVERSE_VIEWPROJ_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_TRANSPOSE_VIEWPROJ_MATRIX:
                    type = AutoConstantType.ACT_TRANSPOSE_VIEWPROJ_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_VIEWPROJ_MATRIX:
                    type = AutoConstantType.ACT_INVERSE_TRANSPOSE_VIEWPROJ_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_WORLDVIEWPROJ_MATRIX:
                    type = AutoConstantType.ACT_WORLDVIEWPROJ_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_WORLDVIEWPROJ_MATRIX:
                    type = AutoConstantType.ACT_INVERSE_WORLDVIEWPROJ_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_TRANSPOSE_WORLDVIEWPROJ_MATRIX:
                    type = AutoConstantType.ACT_TRANSPOSE_WORLDVIEWPROJ_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_TRANSPOSE_WORLDVIEWPROJ_MATRIX:
                    type = AutoConstantType.ACT_INVERSE_TRANSPOSE_WORLDVIEWPROJ_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_TEXTURE_MATRIX:
                    type = AutoConstantType.ACT_TEXTURE_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_RENDER_TARGET_FLIPPING:
                    type = AutoConstantType.ACT_RENDER_TARGET_FLIPPING;
                    break;
                case ENG_ShaderTypeStandard.ACT_VERTEX_WINDING:
                    type = AutoConstantType.ACT_VERTEX_WINDING;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_DIFFUSE_COLOUR:
                    type = AutoConstantType.ACT_LIGHT_DIFFUSE_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_SPECULAR_COLOUR:
                    type = AutoConstantType.ACT_LIGHT_SPECULAR_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_ATTENUATION:
                    type = AutoConstantType.ACT_LIGHT_ATTENUATION;
                    break;
                case ENG_ShaderTypeStandard.ACT_SPOTLIGHT_PARAMS:
                    type = AutoConstantType.ACT_SPOTLIGHT_PARAMS;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_POSITION:
                    type = AutoConstantType.ACT_LIGHT_POSITION;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION:
                    type = AutoConstantType.ACT_LIGHT_DIRECTION;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_OBJECT_SPACE:
                    type = AutoConstantType.ACT_LIGHT_POSITION_OBJECT_SPACE;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_OBJECT_SPACE:
                    type = AutoConstantType.ACT_LIGHT_DIRECTION_OBJECT_SPACE;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_DISTANCE_OBJECT_SPACE:
                    type = AutoConstantType.ACT_LIGHT_DISTANCE_OBJECT_SPACE;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_VIEW_SPACE:
                    type = AutoConstantType.ACT_LIGHT_POSITION_VIEW_SPACE;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_VIEW_SPACE:
                    type = AutoConstantType.ACT_LIGHT_DIRECTION_VIEW_SPACE;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_POWER:
                    throw new UnsupportedOperationException("Check the power extra param info");
                    //break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_DIFFUSE_COLOUR_POWER_SCALED:
                    type = AutoConstantType.ACT_LIGHT_DIFFUSE_COLOUR_POWER_SCALED;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_SPECULAR_COLOUR_POWER_SCALED:
                    type = AutoConstantType.ACT_LIGHT_SPECULAR_COLOUR_POWER_SCALED;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_NUMBER:
                    type = AutoConstantType.ACT_LIGHT_NUMBER;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_DIFFUSE_COLOUR_ARRAY:
                    type = AutoConstantType.ACT_LIGHT_DIFFUSE_COLOUR_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_SPECULAR_COLOUR_ARRAY:
                    type = AutoConstantType.ACT_LIGHT_SPECULAR_COLOUR_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_DIFFUSE_COLOUR_POWER_SCALED_ARRAY:
                    type = AutoConstantType.ACT_LIGHT_DIFFUSE_COLOUR_POWER_SCALED_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_SPECULAR_COLOUR_POWER_SCALED_ARRAY:
                    type = AutoConstantType.ACT_LIGHT_SPECULAR_COLOUR_POWER_SCALED_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_ATTENUATION_ARRAY:
                    type = AutoConstantType.ACT_LIGHT_ATTENUATION_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_SPOTLIGHT_PARAMS_ARRAY:
                    type = AutoConstantType.ACT_SPOTLIGHT_PARAMS_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_ARRAY:
                    type = AutoConstantType.ACT_LIGHT_POSITION_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_ARRAY:
                    type = AutoConstantType.ACT_LIGHT_DIRECTION_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_OBJECT_SPACE_ARRAY:
                    type = AutoConstantType.ACT_LIGHT_POSITION_OBJECT_SPACE_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_OBJECT_SPACE_ARRAY:
                    type = AutoConstantType.ACT_LIGHT_DIRECTION_OBJECT_SPACE_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_DISTANCE_OBJECT_SPACE_ARRAY:
                    type = AutoConstantType.ACT_LIGHT_DISTANCE_OBJECT_SPACE_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_POSITION_VIEW_SPACE_ARRAY:
                    type = AutoConstantType.ACT_LIGHT_POSITION_VIEW_SPACE_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_DIRECTION_VIEW_SPACE_ARRAY:
                    type = AutoConstantType.ACT_LIGHT_DIRECTION_VIEW_SPACE_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_POWER_ARRAY:
                    throw new UnsupportedOperationException("Check the power extra param info");
                    //break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_COUNT:
                    type = AutoConstantType.ACT_LIGHT_COUNT;
                    break;
                case ENG_ShaderTypeStandard.ACT_LIGHT_CASTS_SHADOWS:
                    type = AutoConstantType.ACT_LIGHT_CASTS_SHADOWS;
                    break;
                case ENG_ShaderTypeStandard.ACT_AMBIENT_LIGHT_COLOUR:
                    type = AutoConstantType.ACT_AMBIENT_LIGHT_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_SURFACE_AMBIENT_COLOUR:
                    type = AutoConstantType.ACT_SURFACE_AMBIENT_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_SURFACE_DIFFUSE_COLOUR:
                    type = AutoConstantType.ACT_SURFACE_DIFFUSE_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_SURFACE_SPECULAR_COLOUR:
                    type = AutoConstantType.ACT_SURFACE_SPECULAR_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_SURFACE_EMISSIVE_COLOUR:
                    type = AutoConstantType.ACT_SURFACE_EMISSIVE_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_SURFACE_SHININESS:
                    type = AutoConstantType.ACT_SURFACE_SHININESS;
                    break;
                case ENG_ShaderTypeStandard.ACT_DERIVED_AMBIENT_LIGHT_COLOUR:
                    type = AutoConstantType.ACT_DERIVED_AMBIENT_LIGHT_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_DERIVED_SCENE_COLOUR:
                    type = AutoConstantType.ACT_DERIVED_SCENE_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_DERIVED_LIGHT_DIFFUSE_COLOUR:
                    type = AutoConstantType.ACT_DERIVED_LIGHT_DIFFUSE_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_DERIVED_LIGHT_SPECULAR_COLOUR:
                    type = AutoConstantType.ACT_DERIVED_LIGHT_SPECULAR_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_DERIVED_LIGHT_DIFFUSE_COLOUR_ARRAY:
                    type = AutoConstantType.ACT_DERIVED_LIGHT_DIFFUSE_COLOUR_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_DERIVED_LIGHT_SPECULAR_COLOUR_ARRAY:
                    type = AutoConstantType.ACT_DERIVED_LIGHT_SPECULAR_COLOUR_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_FOG_COLOUR:
                    type = AutoConstantType.ACT_FOG_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_FOG_PARAMS:
                    type = AutoConstantType.ACT_FOG_PARAMS;
                    break;
                case ENG_ShaderTypeStandard.ACT_CAMERA_POSITION:
                    type = AutoConstantType.ACT_CAMERA_POSITION;
                    break;
                case ENG_ShaderTypeStandard.ACT_CAMERA_POSITION_OBJECT_SPACE:
                    type = AutoConstantType.ACT_CAMERA_POSITION_OBJECT_SPACE;
                    break;
                case ENG_ShaderTypeStandard.ACT_LOD_CAMERA_POSITION:
                    type = AutoConstantType.ACT_LOD_CAMERA_POSITION;
                    break;
                case ENG_ShaderTypeStandard.ACT_LOD_CAMERA_POSITION_OBJECT_SPACE:
                    type = AutoConstantType.ACT_LOD_CAMERA_POSITION_OBJECT_SPACE;
                    break;
                case ENG_ShaderTypeStandard.ACT_TIME:
                    type = AutoConstantType.ACT_TIME;
                    break;
                case ENG_ShaderTypeStandard.ACT_TIME_0_X:
                    type = AutoConstantType.ACT_TIME_0_X;
                    break;
                case ENG_ShaderTypeStandard.ACT_COSTIME_0_X:
                    type = AutoConstantType.ACT_COSTIME_0_X;
                    break;
                case ENG_ShaderTypeStandard.ACT_SINTIME_0_X:
                    type = AutoConstantType.ACT_SINTIME_0_X;
                    break;
                case ENG_ShaderTypeStandard.ACT_TANTIME_0_X:
                    type = AutoConstantType.ACT_TANTIME_0_X;
                    break;
                case ENG_ShaderTypeStandard.ACT_TIME_0_X_PACKED:
                    type = AutoConstantType.ACT_TIME_0_X_PACKED;
                    break;
                case ENG_ShaderTypeStandard.ACT_TIME_0_1:
                    type = AutoConstantType.ACT_TIME_0_1;
                    break;
                case ENG_ShaderTypeStandard.ACT_COSTIME_0_1:
                    type = AutoConstantType.ACT_COSTIME_0_1;
                    break;
                case ENG_ShaderTypeStandard.ACT_SINTIME_0_1:
                    type = AutoConstantType.ACT_SINTIME_0_1;
                    break;
                case ENG_ShaderTypeStandard.ACT_TANTIME_0_1:
                    type = AutoConstantType.ACT_TANTIME_0_1;
                    break;
                case ENG_ShaderTypeStandard.ACT_TIME_0_1_PACKED:
                    type = AutoConstantType.ACT_TIME_0_1_PACKED;
                    break;
                case ENG_ShaderTypeStandard.ACT_TIME_0_2PI:
                    type = AutoConstantType.ACT_TIME_0_2PI;
                    break;
                case ENG_ShaderTypeStandard.ACT_COSTIME_0_2PI:
                    type = AutoConstantType.ACT_COSTIME_0_2PI;
                    break;
                case ENG_ShaderTypeStandard.ACT_SINTIME_0_2PI:
                    type = AutoConstantType.ACT_SINTIME_0_2PI;
                    break;
                case ENG_ShaderTypeStandard.ACT_TANTIME_0_2PI:
                    type = AutoConstantType.ACT_TANTIME_0_2PI;
                    break;
                case ENG_ShaderTypeStandard.ACT_TIME_0_2PI_PACKED:
                    type = AutoConstantType.ACT_TIME_0_2PI_PACKED;
                    break;
                case ENG_ShaderTypeStandard.ACT_FRAME_TIME:
                    type = AutoConstantType.ACT_FRAME_TIME;
                    break;
                case ENG_ShaderTypeStandard.ACT_FPS:
                    type = AutoConstantType.ACT_FPS;
                    break;
                case ENG_ShaderTypeStandard.ACT_VIEWPORT_WIDTH:
                    type = AutoConstantType.ACT_VIEWPORT_WIDTH;
                    break;
                case ENG_ShaderTypeStandard.ACT_VIEWPORT_HEIGHT:
                    type = AutoConstantType.ACT_VIEWPORT_HEIGHT;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_VIEWPORT_WIDTH:
                    type = AutoConstantType.ACT_INVERSE_VIEWPORT_WIDTH;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_VIEWPORT_HEIGHT:
                    type = AutoConstantType.ACT_INVERSE_VIEWPORT_HEIGHT;
                    break;
                case ENG_ShaderTypeStandard.ACT_VIEWPORT_SIZE:
                    type = AutoConstantType.ACT_VIEWPORT_SIZE;
                    break;
                case ENG_ShaderTypeStandard.ACT_TEXEL_OFFSETS:
                    type = AutoConstantType.ACT_TEXEL_OFFSETS;
                    break;
                case ENG_ShaderTypeStandard.ACT_VIEW_DIRECTION:
                    type = AutoConstantType.ACT_VIEW_DIRECTION;
                    break;
                case ENG_ShaderTypeStandard.ACT_VIEW_SIDE_VECTOR:
                    type = AutoConstantType.ACT_VIEW_SIDE_VECTOR;
                    break;
                case ENG_ShaderTypeStandard.ACT_VIEW_UP_VECTOR:
                    type = AutoConstantType.ACT_VIEW_UP_VECTOR;
                    break;
                case ENG_ShaderTypeStandard.ACT_FOV:
                    type = AutoConstantType.ACT_FOV;
                    break;
                case ENG_ShaderTypeStandard.ACT_NEAR_CLIP_DISTANCE:
                    type = AutoConstantType.ACT_NEAR_CLIP_DISTANCE;
                    break;
                case ENG_ShaderTypeStandard.ACT_FAR_CLIP_DISTANCE:
                    type = AutoConstantType.ACT_FAR_CLIP_DISTANCE;
                    break;
                case ENG_ShaderTypeStandard.ACT_TEXTURE_VIEWPROJ_MATRIX:
                    type = AutoConstantType.ACT_TEXTURE_VIEWPROJ_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_TEXTURE_VIEWPROJ_MATRIX_ARRAY:
                    type = AutoConstantType.ACT_TEXTURE_VIEWPROJ_MATRIX_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_TEXTURE_WORLDVIEWPROJ_MATRIX:
                    type = AutoConstantType.ACT_TEXTURE_WORLDVIEWPROJ_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_TEXTURE_WORLDVIEWPROJ_MATRIX_ARRAY:
                    type = AutoConstantType.ACT_TEXTURE_WORLDVIEWPROJ_MATRIX_ARRAY;
                    break;
                case ENG_ShaderTypeStandard.ACT_SPOTLIGHT_VIEWPROJ_MATRIX:
                    type = AutoConstantType.ACT_SPOTLIGHT_VIEWPROJ_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_SPOTLIGHT_WORLDVIEWPROJ_MATRIX:
                    type = AutoConstantType.ACT_SPOTLIGHT_WORLDVIEWPROJ_MATRIX;
                    break;
                case ENG_ShaderTypeStandard.ACT_SCENE_DEPTH_RANGE:
                    type = AutoConstantType.ACT_SCENE_DEPTH_RANGE;
                    break;
                case ENG_ShaderTypeStandard.ACT_SHADOW_SCENE_DEPTH_RANGE:
                    type = AutoConstantType.ACT_SHADOW_SCENE_DEPTH_RANGE;
                    break;
                case ENG_ShaderTypeStandard.ACT_SHADOW_COLOUR:
                    type = AutoConstantType.ACT_SHADOW_COLOUR;
                    break;
                case ENG_ShaderTypeStandard.ACT_SHADOW_EXTRUSION_DISTANCE:
                    type = AutoConstantType.ACT_SHADOW_EXTRUSION_DISTANCE;
                    break;
                case ENG_ShaderTypeStandard.ACT_TEXTURE_SIZE:
                    type = AutoConstantType.ACT_TEXTURE_SIZE;
                    break;
                case ENG_ShaderTypeStandard.ACT_INVERSE_TEXTURE_SIZE:
                    type = AutoConstantType.ACT_INVERSE_TEXTURE_SIZE;
                    break;
                case ENG_ShaderTypeStandard.ACT_PACKED_TEXTURE_SIZE:
                    type = AutoConstantType.ACT_PACKED_TEXTURE_SIZE;
                    break;
                case ENG_ShaderTypeStandard.ACT_PASS_NUMBER:
                    type = AutoConstantType.ACT_PASS_NUMBER;
                    break;
                case ENG_ShaderTypeStandard.ACT_PASS_ITERATION_NUMBER:
                    type = AutoConstantType.ACT_PASS_ITERATION_NUMBER;
                    break;
                case ENG_ShaderTypeStandard.ACT_ANIMATION_PARAMETRIC:
                    type = AutoConstantType.ACT_ANIMATION_PARAMETRIC;
                    break;
                case ENG_ShaderTypeStandard.ACT_CUSTOM:
                    type = AutoConstantType.ACT_CUSTOM;
                    break;
                default:
                    throw new IllegalArgumentException("Wrong act type " + s.type);

            }
            switch (s.extraParamType) {
                case FLOAT:
                    params.setNamedAutoConstant(s.name, type, s.extraParamFloat);
                    break;
                case INT:
                    params.setNamedAutoConstant(s.name, type, s.extraParamInt);
                    break;
                case NONE:
                    params.setNamedAutoConstant(s.name, type, 0);
                    break;
                default:
                    // Should never get here
                    throw new IllegalArgumentException(s.extraParamType + " invalid argument extra param type");
            }

        }
    }

    public static void loadMaterialList(String fileName, String path) {
        loadMaterialList(fileName, path, true, false, true);
    }

    public static void loadMaterialList(String fileName, String path, boolean clearMaterialList, boolean clearShaderList, boolean fromSDCard) {
        ArrayList<String> materialList = ENG_CompilerUtil.loadListFromFile(fileName, path);

        ENG_ResourceLoader resourceLoader = MainApp.getGame().getResourceLoaderBlock().createResourceLoader();

        for (String mat : materialList) {
            String[] pathAndFileName = ENG_CompilerUtil.getPathAndFileName(mat);
            loadMaterial(pathAndFileName[1], pathAndFileName[0], fromSDCard);
            if (clearMaterialList) {
                clearMaterialList();
                //	clearShaderList();
            }
            if (clearShaderList) {
                clearShaderList();
            }
        }
    }

    private static void createMaterial(ENG_CompiledResource res, boolean fromSDCard) {
        for (ENG_MaterialResource matRes : res.materialResourceList) {
            ENG_Material mat = ENG_MaterialManager.getSingleton().create(matRes.name);
            for (ENG_TechniqueResource tech : matRes.techniqueList) {
                ENG_Technique technique = mat.createTechnique();
                technique.setName(tech.name);
                int texPos = 0;
                for (ENG_PassResource p : tech.passList) {
                    ENG_Pass pass = technique.createPass();
                    pass.setName(p.name);
                    ENG_Shader vs = shaderList.get(p.vertexProgramID);
                    //res.vertexProgramList.get(p.vertexProgramID);
                    ENG_Shader fs = shaderList.get(p.fragmentProgramID);
                    //res.fragmentProgramList.get(p.fragmentProgramID);
                    pass.setVertexProgram(vs.name);
                    pass.setFragmentProgram(fs.name);
                    // We must create the "specialized" parameters
                    // for each shader to overwrite
                    // either the defaults or set them in the first hand
                    // Get the referenced params here
                    if (pass.getVertexProgram().isSupported()) {
                        ENG_GpuProgramParameters parameters = pass.getVertexProgramParameters();
                        if (p.vertexProgramParamListID != -1) {
                            ENG_ShaderParam param = vs.paramList.get(p.vertexProgramParamListID);
                            if (param.namedParam != null) {
                                readParams(vs, parameters, param.namedParam);
                            }
                        }
                    }
                    if (pass.getFragmentProgram().isSupported()) {
                        ENG_GpuProgramParameters parameters = pass.getFragmentProgramParameters();
                        if (p.fragmentProgramParamListID != -1) {
                            ENG_ShaderParam param = fs.paramList.get(p.fragmentProgramParamListID);
                            if (param.namedParam != null) {
                                readParams(fs, parameters, param.namedParam);
                            }
                        }

                        parameters.setNamedConstant("texture" + ((texPos == 0) ? "" : String.valueOf(texPos)), texPos++);
                    }

                    // create the texture units
                    for (ENG_TextureUnitResource tex : p.textureUnits) {
                        ENG_TextureUnitState texState = pass.createTextureUnitState();

                        if (!tex.name.isEmpty()) {
                            texState.setName(tex.name);
                        }
                        if (!tex.textureAlias.isEmpty()) {
                            texState.setTextureNameAlias(tex.textureAlias);
                        }
                        if (!tex.textureName.isEmpty()) {
                            for (int i = 0; i < tex.textureName.size(); ++i) {
                                String texName = tex.textureName.get(i);
                                String[] pathAndFileName;
                                try {
                                    pathAndFileName = ENG_CompilerUtil.getPathAndFileName(texName, fromSDCard);
                                } catch (ENG_InvalidPathException e) {
                                    //This isn't a valid path. Skip loading
                                    pathAndFileName = null;
                                }
                                String textureName = null;
                                if (pathAndFileName != null) {
                                    TextureUsage usage = getTextureUsage(tex);
                                    textureName = ENG_TextureLoader.loadTexture(
                                            pathAndFileName[1],
                                            pathAndFileName[0],
                                            tex.useShadowBuffer,
                                            usage,
                                            fromSDCard);
                                }
                                // First time set everything.
                                // For the rest of textures just add them as frame layers
                                if (i == 0) {
                                    //No extensions here!!!
                                    texState.setTextureName(textureName != null ? textureName : texName);
                                    texState.setDesiredFormat(tex.pf);
                                    texState.setNumMipmaps(tex.numMipmaps > 0 ? tex.numMipmaps : TextureMipmap.MIP_UNLIMITED.getMipmap());
                                    texState.setHardwareGammaEnabled(tex.gamma);
                                } else {
                                    texState.addFrameTextureName(textureName != null ? textureName : texName);
                                }
                            }
                        }
                        if (!tex.animTexture.isEmpty()) {
                            //Format pic_0 pic_1
                            //But we also have extensions and dir change so...
                            String[] pathAndFileName;
                            try {
                                pathAndFileName = ENG_CompilerUtil.getPathAndFileName(tex.animTexture);
                            } catch (ENG_InvalidPathException e) {
                                //This isn't a valid path. Skip loading
                                pathAndFileName = null;
                                System.out.println("Invalid pathAndFileName in texture " + tex.animTexture);
                            }
                            String textureName = null;
                            if (pathAndFileName != null) {
                                String file = ENG_CompilerUtil.trimFile(pathAndFileName[1]);
                                String extension = ENG_CompilerUtil.getExtension(pathAndFileName[1]);
                                // Since we can end up using hundreds of textures
                                // for an animation and 3ds max exports the anim
                                // in multiple pics with format _0039 we have
                                // to add 0 in front to get the correct texture name
                                for (int i = 0; i < tex.numFrames; ++i) {
                                    String aheadZeros = ENG_Utility.getAheadZeros(i);
                                    textureName = ENG_TextureLoader.loadTexture(file + "_" + aheadZeros + i + "." + extension, pathAndFileName[0], fromSDCard);
                                }
                                texState.setAnimatedTextureName(file, tex.numFrames, tex.duration);
                            }

                        }
                        if (!tex.cubeName.isEmpty()) {
                            texState.setCubicTextureName(tex.cubeName, tex.cubeType == CubeType.combinedUVW);
                        }
                        texState.setTextureCoordSet(tex.texCoordSet);
                        UVWAddressingMode addr = new UVWAddressingMode();
                        addr.u = tex.addresingMode;
                        addr.v = tex.addresingMode;
                        addr.w = tex.addresingMode;
                        texState.setTextureAddressingMode(addr);
                        texState.setTextureBorderColour(tex.textureBorder);
                        texState.setTextureFiltering(
                                translateFilterOptions(tex.filterTypeMin),
                                translateFilterOptions(tex.filterTypeMag),
                                translateFilterOptions(tex.filterTypeMip));
                        texState.setTextureAnisotropy(tex.maxAnisotropy);
                        texState.setTextureMipmapBias(tex.mipmapBias);
                        texState.setColourOperation(translateColourOp(tex.colourOpType));

                        setEnvMap(tex);

                        setBindingType(tex, texState);

                        setContentType(tex, texState);
                    }
                    pass.setDepthCheckEnabled(p.depthCheck);
                    pass.setSceneBlending(p.sourceBlendFactor, p.destBlendFactor);
                    pass.setSceneBlendingOperation(p.blendOperation);
                    pass.setDepthWriteEnabled(p.depthWrite);
                }
                mat.compile();
            }
        }
    }

    private static TextureUsage getTextureUsage(ENG_TextureUnitResource tex) {
        TextureUsage usage;
        switch (tex.usage) {
            case TU_STATIC:
                usage = TextureUsage.TU_STATIC;
                break;
            case TU_DYNAMIC:
                usage = TextureUsage.TU_DYNAMIC;
                break;
            case TU_STATIC_WRITE_ONLY:
                usage = TextureUsage.TU_STATIC_WRITE_ONLY;
                break;
            case TU_DYNAMIC_WRITE_ONLY:
                usage = TextureUsage.TU_DYNAMIC_WRITE_ONLY;
                break;
            case TU_DYNAMIC_WRITE_ONLY_DISCARDABLE:
                usage = TextureUsage.TU_DYNAMIC_WRITE_ONLY_DISCARDABLE;
                break;
            case TU_WRITE_ONLY:
                usage = TextureUsage.TU_WRITE_ONLY;
                break;
            case TU_AUTOMIPMAP:
                usage = TextureUsage.TU_AUTOMIPMAP;
                break;
            case TU_RENDERTARGET:
                usage = TextureUsage.TU_RENDERTARGET;
                break;
            case TU_DEFAULT:
                usage = TextureUsage.TU_DEFAULT;
                break;
            default:
                throw new ENG_InvalidFieldStateException();
        }
        return usage;
    }


    private static void setContentType(ENG_TextureUnitResource tex, ENG_TextureUnitState texState) {
        switch (tex.contentType) {
            case named:
                texState.setContentType(ContentType.CONTENT_NAMED);
                break;
            case shadow:
                texState.setContentType(ContentType.CONTENT_SHADOW);
                break;
            case compositor:
                texState.setContentType(ContentType.CONTENT_COMPOSITOR);
                texState.setCompositorReference(
                        tex.referencedCompositorName,
                        tex.referencedTextureName,
                        tex.mrtTextureIndex);
                break;
            default:
                throw new IllegalArgumentException("ContentType is " + tex.contentType);
        }
    }

    private static void setBindingType(ENG_TextureUnitResource tex, ENG_TextureUnitState texState) {
        switch (tex.bindingType) {
            case vertex:
                //GLES 2.0 does not support binding in vertex!!!
                //Reserved for future use
                texState.setBindingType(BindingType.BT_VERTEX);
                break;
            case fragment:
                texState.setBindingType(BindingType.BT_FRAGMENT);
                break;
            default:
                throw new IllegalArgumentException("BindingType is " + tex.bindingType);
        }
    }

    private static void setEnvMap(ENG_TextureUnitResource tex) {
        // NOT IMPLEMENTED YET!!!!!
        switch (tex.envMapType) {
            case off:
                break;
            case spherical:
                break;
            case planar:
                break;
            case cubic_reflection:
                break;
            case cubic_normal:
                break;
            default:
                throw new IllegalArgumentException("EnvMapType is " + tex.envMapType);
        }
    }

    private static LayerBlendOperation translateColourOp(ColourOpType op) {
        LayerBlendOperation lbo;
        switch (op) {
            case replace:
                lbo = LayerBlendOperation.LBO_REPLACE;
                break;
            case add:
                lbo = LayerBlendOperation.LBO_ADD;
                break;
            case modulate:
                lbo = LayerBlendOperation.LBO_MODULATE;
                break;
            case alpha_blend:
                lbo = LayerBlendOperation.LBO_ALPHA_BLEND;
                break;
            default:
                //Should never get here
                throw new IllegalArgumentException("colourOpType is " + op);
        }
        return lbo;
    }

    private static FilterOptions translateFilterOptions(FilterType op) {
        FilterOptions fo;
        switch (op) {
            case none:
                fo = FilterOptions.FO_NONE;
                break;
            case point:
                fo = FilterOptions.FO_POINT;
                break;
            case linear:
                fo = FilterOptions.FO_LINEAR;
                break;
            case anisotropic:
                fo = FilterOptions.FO_ANISOTROPIC;
                break;
            default:
                //Should never get here
                throw new IllegalArgumentException("op is " + op);
        }
        return fo;
    }

    public static void loadMaterial(final String fileName, final String path, final boolean fromSDCard) {
        loadMaterial(fileName, path, fromSDCard, null);
    }

    private static class LoadMaterialRet {
        public final ENG_MaterialCompiler materialCompiler;
        public final ENG_CompiledResource compiledResource;

        public LoadMaterialRet(ENG_MaterialCompiler materialCompiler, ENG_CompiledResource compiledResource) {
            this.materialCompiler = materialCompiler;
            this.compiledResource = compiledResource;
        }
    }

    public static void reloadMaterials(final boolean fromSDCard) {
        for (ENG_CompiledResource compiledResource : compiledResourceList.values()) {

            // We have already loaded the panel programs in order to be able to show the loading screen. No need to reload those.
            boolean load = true;
            for (ENG_Shader vs : compiledResource.vertexProgramList) {
                if (vs.name.equals("raw/panel_vs.txt")) {
                    load = false;
                    break;
                }
            }
            for (ENG_Shader fs : compiledResource.fragmentProgramList) {
                if (fs.name.equals("raw/panel_fs.txt")) {
                    load = false;
                    break;
                }
            }

            for (ENG_MaterialResource mat : compiledResource.materialResourceList) {
                if (mat.name.equals("loading_screen_mat")) {
                    load = false;
                    break;
                }
            }


            if (!load) {
                continue;
            }

            HashMap<String, ENG_Shader> vShaderMap = new HashMap<>();
            for (ENG_Shader vs : compiledResource.vertexProgramList) {
                vShaderMap.put(vs.name, vs);
            }
            addToShaderList(vShaderMap);
            HashMap<String, ENG_Shader> fShaderMap = new HashMap<>();
            for (ENG_Shader fs : compiledResource.fragmentProgramList) {
                fShaderMap.put(fs.name, fs);
            }
            addToShaderList(fShaderMap);

            loadCompiledResource(compiledResource, fromSDCard);



        }

    }

    public static void loadMaterial(final String fileName, final String path, final boolean fromSDCard, ENG_ResourceLoader resourceLoader) {
        if (!materialsLoadedList.contains(fileName)) {
            materialsLoadedList.add(fileName);
            if (resourceLoader != null) {
                resourceLoader.addResourceLoadTask(new ENG_ResourceLoaderTask<Void, LoadMaterialRet>(resourceLoader) {
                    @Override
                    public LoadMaterialRet _executeInBackground(Void aVoid) {
                        ENG_MaterialCompiler materialCompiler = new ENG_MaterialCompiler();
                        materialCompiler.addToShaderList(getShaderList());
                        ENG_CompiledResource compiledResource = materialCompiler.compile(fileName, path, fromSDCard);
                        return new LoadMaterialRet(materialCompiler, compiledResource);
                    }

                    @Override
                    public void _executeOnRenderThread(LoadMaterialRet ret) {
                        addToCompiledResourceList(fileName, ret.compiledResource);
                        addToShaderList(ret.materialCompiler.getShaderList());
                        loadCompiledResource(ret.compiledResource, fromSDCard);
                    }
                });
            } else {
                ENG_MaterialCompiler materialCompiler = new ENG_MaterialCompiler();
                materialCompiler.addToShaderList(getShaderList());
                ENG_CompiledResource compiledResource = materialCompiler.compile(fileName, path, fromSDCard);
                addToCompiledResourceList(fileName, compiledResource);
                addToShaderList(materialCompiler.getShaderList());
                loadCompiledResource(compiledResource, fromSDCard);
            }

            // Must always set the shader list before calling compile!!!
//            materialCompiler.setShaderList(shaderList);
//			ENG_MaterialCompiler.MaterialResult result = materialCompiler.compile(fileName, path, fromSDCard);
//			for (Map.Entry<String, ENG_Shader> entry : result.shaderList.entrySet()) {
//				ENG_Shader put = shaderList.put(entry.getKey(), entry.getValue());
//				if (put != null) {
//					throw new IllegalArgumentException(entry.getValue().name + " already loaded");
//				}
//			}


        }
    }

    private static void loadCompiledResource(ENG_CompiledResource res, boolean fromSDCard) {
        createPrograms(res.vertexProgramList);
        createPrograms(res.fragmentProgramList);
        //Has to be last one since we first need to setup the programs
        // UPDATE No it doesn't. we can have all program files with no material!!!
        createMaterial(res, fromSDCard);
    }
}
