/*
 * Created by Sebastian Bugiu on 4/9/23, 10:11 PM
 * sebastian.bugiu@headwayentertainment.net
 * Last modified 11/18/21, 7:57 PM
 * Copyright (c) 2023.
 * All rights reserved.
 */

package headwayent.blackholedarksun.systems;

import headwayent.blackholedarksun.entitydata.ShipData;
import headwayent.blackholedarksun.entitydata.ShipData.ShipTeam;
import headwayent.blackholedarksun.entitydata.WeaponData;
import headwayent.blackholedarksun.entitydata.WeaponData.WeaponType;
import headwayent.blackholedarksun.entitydata.WeaponData.WeaponComparator;
import headwayent.blackholedarksun.components.AIProperties;
import headwayent.blackholedarksun.components.AIProperties.AIState;
import headwayent.blackholedarksun.components.EntityProperties;
import headwayent.blackholedarksun.components.ShipProperties;
import headwayent.blackholedarksun.components.WeaponProperties;
import headwayent.blackholedarksun.gamestatedebugger.FrameInterval;
import headwayent.blackholedarksun.world.WorldManagerBase;
import headwayent.hotshotengine.*;
import headwayent.hotshotengine.renderer.ENG_SceneNode;

import java.util.ArrayList;
import java.util.Collections;

import com.artemis.Aspect;
import com.artemis.ComponentMapper;
import com.artemis.Entity;
import com.artemis.managers.GroupManager;
import com.artemis.systems.IntervalEntityProcessingSystem;
import com.artemis.utils.ImmutableBag;
import com.badlogic.gdx.math.Vector3;

import headwayent.blackholedarksun.*;
import headwayent.hotshotengine.exception.ENG_InvalidFieldStateException;

public class AISystem extends IntervalEntityProcessingSystem {
    private static final float ESCAPING_LIMITS_ACCELERATION_RATE = 0.1f;
    private static final float EVADING_COLLISION_VELOCITY_CHANGE_RATE = 0.2f;
    private static final int MAX_CHASING_PROJECTILES_NUM = 5;
    private static final int MAX_PROJECTILES_LAUNCHED_DELAY = 2000;

    private static final float MIN_DISTANCE_SLOW_DOWN_RATE = 0.2f;
    private static final float ENEMY_MIN_DISTANCE = 100000.0f;
    private static final float REACHING_DESTINATION_VELOCITY_CHANGE_STEP = 0.1f;
    private static final int VELOCITY_CHANGE_NAME_EVADING_COLLISION = 11;
    private static final int VELOCITY_CHANGE_NAME_DESTINATION_REACHED = 10;
    private static final int VELOCITY_CHANGE_NAME_REACHING_DESTINATION = 9;
    private static final int VELOCITY_CHANGE_NAME_ESCAPING_LIMITS = 8;
    private static final int VELOCITY_CHANGE_NAME_ESCAPING_LIMITS_FULL_STOP = 13;
    private static final int VELOCITY_CHANGE_NAME_MIN_DISTANCE = 7;
    private static final int VELOCITY_CHANGE_NAME_FOLLOWED_SHIP_VELOCITY = 6;
    private static final int VELOCITY_CHANGE_NAME_EVADE_HIT = 5;
    private static final int VELOCITY_CHANGE_NAME_EVADE_MISSILE = 12;
    private static final int VELOCITY_CHANGE_NAME_COLLISION_RESPONSE = 4;
    private static final int VELOCITY_CHANGE_NAME_ACCELERATE_TO_MAX_SPEED = 3;
    private static final int VELOCITY_CHANGE_NAME_PATROL_SLOW_DOWN = 2;
    private static final int VELOCITY_CHANGE_NAME_RELOADER_TOWARDS_SHIP = 1;
    private static final int VELOCITY_CHANGE_NAME_RELOADER_STOP = 0;
    private static final float RELOADER_AWAY_ANGLE = 160.0f * ENG_Math.DEGREES_TO_RADIANS;
    private static final int RELOADER_TIME_BETWEEN_RELOADING_UNITS = 1000;
    private static final float RELOADER_SPEED_CHANGE_RATE = 0.5f;
    private static final float FOLLOWING_SHIP_ACCELERATION_RATE = ESCAPING_LIMITS_ACCELERATION_RATE;
    private static final int PATROLING_ROTATION_TIME = 3000;
    private static final float PATROL_SPEED_COEFICIENT = 0.25f;
    private static final int NO_ENEMY_DIRECTION_CHANGE_CHANCE = 10;
    private static final float COLLISION_RESPONSE_ACCELERATION_STEP = 0.3f;
    private static final int COLLISION_RESPONSE_ACCELERATION_TIME = 4000;
    private static final float COLLISION_RESPONSE_ACCELERATION_ANGLE = 45.0f * ENG_Math.DEGREES_TO_RADIANS;
    private static final float EVASION_HIT_SPEED_CHANGE_STEP = 0.3f;
    private static final float EVASION_MISSILE_SPEED_CHANGE_STEP = 0.3f;
    private static final float NO_TARGET_SPEED_CHANGE_STEP = ESCAPING_LIMITS_ACCELERATION_RATE;
    private static final int SPECIAL_WEAPON_CHANCE = 10;
    private static final int CHECK_TARGETED_CHANCE = 200;
    private static final int FOLLOW_PLAYER_TO_SEEK_PLAYER_RAND = 20;
    private static final boolean DEBUG = false;
    private static final int ENEMY_SEEK_RAND = 1; // 100;
    private static final float MIN_FOLLOW_DIST = 40000000.0f;
    private static final float TARGET_MIN_DISTANCE = 4000000.0f;
    private static final float SPEED_CHANGE_RATE = 1.0f / 10.0f;
    private static final float COLLISION_EVASION_ANGLE = 135.0f * ENG_Math.DEGREES_TO_RADIANS;
    private static final float EVASION_ACCELERATION = 1.0f;
    private static final long EVADE_COLLISION_TIME = 3000;
    private static final float TARGETING_ANGLE = 1.0f * ENG_Math.DEGREES_TO_RADIANS;
    private static final float ESCAPE_LEVEL_LIMITS_ANGLE = 20.0f * ENG_Math.DEGREES_TO_RADIANS;
    private static final float MIN_COUNTERMEASURES_DISTANCE = ENG_Math.sqr(800.0f);
    private static final float MIN_EVASION_DISTANCE = ENG_Math.sqr(500.0f);
    private static final long EVASION_HIT_TIME = 5000;
    private ComponentMapper<EntityProperties> entityPropertiesMapper;
    private ComponentMapper<ShipProperties> shipPropertiesMapper;
    private ComponentMapper<AIProperties> aIPropertiesMapper;
    private ComponentMapper<WeaponProperties> weaponPropertiesMapper;
    private final ENG_Vector4D currentPos = new ENG_Vector4D(true);
    private final ENG_Vector4D otherPos = new ENG_Vector4D(true);
    private final ENG_Vector4D otherShipVelocity = new ENG_Vector4D();
    private float currentMinLen;
    private final ENG_Vector4D distVec = new ENG_Vector4D(true);
    private final ENG_Vector4D currentFrontVec = new ENG_Vector4D();
    private final ENG_Vector4D otherFrontVec = new ENG_Vector4D();
    private final ENG_Quaternion rotation = new ENG_Quaternion();
    private final headwayent.blackholedarksun.entitydata.WeaponData.WeaponComparator weaponComparator = new WeaponComparator();
    private final ArrayList<WeaponType> currentWeaponsList = new ArrayList<>();
    private final ENG_Vector4D perpendicularVec = new ENG_Vector4D();
    private boolean projectileCreated;
    private final ENG_Vector4D currentUpVec = new ENG_Vector4D();
    private boolean horzRot;
    private final ENG_Vector4D crossPosition = new ENG_Vector4D(true);
    private final ENG_Vector4D transformedCrossPosition = new ENG_Vector4D(true);
    private final ENG_Vector4D collisionResponseAxis = new ENG_Vector4D();
    private final ENG_Vector4D patrolingRotationAxis = new ENG_Vector4D();
    private final ENG_ClosestObjectData data = new ENG_ClosestObjectData();
    private final ENG_Vector4D levelLimits = new ENG_Vector4D();
    private final ENG_Vector4D currentLevelLimits = new ENG_Vector4D();
    private final ENG_Vector4D awayFromLimitsPos = new ENG_Vector4D();
    private final ENG_Vector4D destination = new ENG_Vector4D(true);
    private final float updateInterval;
    private boolean rotationAwayFromLimitsCompleted = true;
    private boolean awayFromLimitsPosSet;
//    private float beginTime;

    public AISystem(double interval) {
        super(Aspect.all(EntityProperties.class, ShipProperties.class), (float) interval);
        
        updateInterval = (float) interval;
    }

    @Override
    protected void begin() {
        super.begin();
//        beginTime = ENG_Utility.currentTimeMillis();
    }

    @Override
    protected void end() {
//        System.out.println("AISystem time: " + (ENG_Utility.currentTimeMillis() - beginTime));
        super.end();
    }

    private void rotateTowardPosition(EntityProperties entityProperties,
                                      ShipProperties shipProperties, ENG_Vector4D pos) {
        entityProperties.getNode().getPosition(currentPos);
        entityProperties.getNode().getLocalInverseZAxis(currentFrontVec);
        entityProperties.getNode().getLocalYAxis(currentUpVec);
//        ENG_Math.rotateTowardPositionDeg(pos, currentPos, currentFrontVec, currentUpVec, rotation, getRotationAngle(shipProperties));
//        entityProperties.rotate(rotation, true, TransformSpace.TS_WORLD);
        Utility.rotateToPosition(currentFrontVec, pos, updateInterval, entityProperties,
                shipProperties.getShipData().maxAngularVelocity);
    }

    @Override
    protected void process(Entity e) {


        AIProperties aiProperties = aIPropertiesMapper.getSafe(e);
        if (aiProperties != null/* && !aiProperties.isIgnoreAI()*/) {

            EntityProperties entityProperties = entityPropertiesMapper.get(e);
//            System.out.println("Ship " + entityProperties.getName() + " starting AI state: " + aiProperties.getState());
            ShipProperties shipProperties = shipPropertiesMapper.get(e);
            if (!shipProperties.isAiEnabled()) {
                return;
            }
            WeaponProperties weaponProperties = weaponPropertiesMapper.get(e);

            aiProperties.setEntityName(entityProperties.getNode().getName());
//			System.out.println("AI SHIP NAME: " + shipProperties.getName());

            switch (aiProperties.getState()) {
                case REACH_DESTINATION: {
                    reachDestination(aiProperties, entityProperties, shipProperties);
                }
                break;
                case EVADE_LEVEL_LIMITS: {
                    evadeLevelLimits(aiProperties, entityProperties, shipProperties);
                }
                break;
                case FOLLOW_PLAYER_SHIP: {
                    followPlayerShip(aiProperties, entityProperties, shipProperties);
                }
                break;
                case SEEK_CLOSEST_PLAYER: {
                    if (aiProperties.isReachDestination()) {
                        aiProperties.setState(AIState.REACH_DESTINATION);
                        showAIStateChange(shipProperties, aiProperties);
                        break;
                    }
                    if (shipProperties.isChased()) {
                        aiProperties.setState(AIState.EVADE_MISSILE);
                        showAIStateChange(shipProperties, aiProperties);
                        break;
                    }
                    seekClosestPlayer(aiProperties, entityProperties,
                            shipProperties);
                }
                break;
                case FOLLOW_PLAYER: {
                    Entity followedShip = MainApp.getGame().getWorldManager().getEntityByItemId(aiProperties.getFollowedShip());
                    if (followedShip == null
                            || ENG_Utility
                            .hasRandomChanceHit(FrameInterval.FOLLOW_PLAYER_TO_SEEK_PLAYER_RAND + entityProperties.getNode().getName(),
                                    FOLLOW_PLAYER_TO_SEEK_PLAYER_RAND)) {
                        aiProperties.resetFollowedShip();
                        aiProperties.setState(AIState.SEEK_CLOSEST_PLAYER);
                        showAIStateChange(shipProperties, aiProperties);
                    } else {
                        if (aiProperties.getCurrentHealth() > entityProperties.getHealth() && !aiProperties.isCollided()) {
                            aiProperties.setCurrentHealth(entityProperties.getHealth());
                            aiProperties.setState(AIState.EVADE_HIT);
                            showAIStateChange(shipProperties, aiProperties);
                            break;
                        }
                        if (aiProperties.isCollided()) {
                            aiProperties.setState(AIState.COLLISION_RESPONSE);
                            showAIStateChange(shipProperties, aiProperties);
                            break;
                        }
                        if (entityProperties.isLimitsReached()) {
                            aiProperties.setState(AIState.EVADE_LEVEL_LIMITS);
                            showAIStateChange(shipProperties, aiProperties);
                            break;
                        }
                        EntityProperties otherShipEntityProperties = entityPropertiesMapper.get(followedShip);

                        entityProperties.getNode().getPosition(currentPos);
                        otherShipEntityProperties.getNode().getPosition(otherPos);
                        otherPos.sub(currentPos, distVec);
                        float squaredDist = distVec.squaredLength();
                        if (!aiProperties.isLockedIn()) {
                            if (squaredDist < MIN_FOLLOW_DIST) {
                                if (!aiProperties.isFollowCountTimeStarted()) {
                                    aiProperties.setFollowBeginTime();
                                    aiProperties.setFollowCountTimeStarted(true);
                                } else {
                                    if (ENG_Utility.hasTimePassed(aiProperties.getFollowBeginTime(), WeaponType.getWeaponEnemySelectionTime(weaponProperties.getCurrentWeaponType()))) {
                                        aiProperties.setLockedIn(true);
                                    }
                                }
                            } else {
                                aiProperties.setFollowCountTimeStarted(false);

                                aiProperties.setState(AIState.SEEK_CLOSEST_PLAYER);
                                showAIStateChange(shipProperties, aiProperties);
                            }
                        }
                        // MIGHT NEED TO THINK THIS AGAIN!!!
                        if (squaredDist > ENEMY_MIN_DISTANCE + otherShipEntityProperties.getItem().getWorldAABB().getHalfSize().length()) {
                            boolean matchingSpeed = false;
                            if (squaredDist < TARGET_MIN_DISTANCE) {
                                // Try and match the followed ship speed
                                float otherShipVelocity = otherShipEntityProperties.getVelocity();
                                float velocity = entityProperties.getVelocity();
                                if (otherShipVelocity > ENG_Math.FLOAT_EPSILON) {
                                    if (otherShipVelocity < shipProperties.getShipData().maxSpeed) {
                                        if (otherShipVelocity != velocity) {
                                            changeVelocity(entityProperties, shipProperties, aiProperties,
                                                    new VelocityChange(
                                                            VELOCITY_CHANGE_NAME_FOLLOWED_SHIP_VELOCITY,
                                                            otherShipVelocity,
                                                            SPEED_CHANGE_RATE));
                                        }
                                    }
                                    matchingSpeed = true;
                                }
                            }

                            if (!matchingSpeed) {
                                changeVelocity(entityProperties, shipProperties, aiProperties,
                                        new VelocityChange(
                                                VELOCITY_CHANGE_NAME_ACCELERATE_TO_MAX_SPEED,
                                                shipProperties.getShipData().maxSpeed,
                                                FOLLOWING_SHIP_ACCELERATION_RATE));
                            }
                        } else {
                            changeVelocity(entityProperties, shipProperties, aiProperties, new VelocityChange(
                                    VELOCITY_CHANGE_NAME_MIN_DISTANCE,
                                    0.0f, MIN_DISTANCE_SLOW_DOWN_RATE));
                        }

                        // Make sure we're not on collision course
                        boolean evadeCollision = false;
                        if (squaredDist < TARGET_MIN_DISTANCE && otherShipEntityProperties.getVelocity() > 0.0f &&
                                !aiProperties.isEvadingCollision()) {
                            entityProperties.getNode().getLocalInverseZAxis(currentFrontVec);
                            otherShipEntityProperties.getNode().getLocalInverseZAxis(otherFrontVec);

                            float angleBetween = currentFrontVec.angleBetween(otherFrontVec);

                            if (DEBUG) {
                                System.out.println("entity: " + entityProperties.getName() + " collision evasion angle: " + (angleBetween * ENG_Math.RADIANS_TO_DEGREES) + " currentTimeMillis: " + ENG_Utility.currentTimeMillis());
                            }

                            if (angleBetween > COLLISION_EVASION_ANGLE) {
                                if (DEBUG) {
                                    System.out.println("entity: " + entityProperties.getName() + " Evading collision start " + ENG_Utility.currentTimeMillis());
                                }
                                aiProperties.setEvadingCollision(true);
                                aiProperties.setEvadingCollisionTimeStarted();
                            }
                        }

                        if (aiProperties.isEvadingCollision()) {
                            if (ENG_Utility.hasTimePassed(FrameInterval.EVADE_COLLISION_TIME + entityProperties.getNode().getName(),
                                    aiProperties.getEvadingCollisionTimeStarted(),
                                    EVADE_COLLISION_TIME)) {
                                if (DEBUG) {
                                    System.out.println("entity: " + entityProperties.getName() + " Evading collision stopped " + ENG_Utility.currentTimeMillis());
                                }
                                aiProperties.setEvadingCollision(false);
                            } else {
                                // Accelerate to max speed to evade the followed
                                // ship
                                changeVelocity(entityProperties, shipProperties, aiProperties,
                                        new VelocityChange(
                                                VELOCITY_CHANGE_NAME_ACCELERATE_TO_MAX_SPEED,
                                                shipProperties.getShipData().maxSpeed,
                                                EVADING_COLLISION_VELOCITY_CHANGE_RATE));

                                // Rotate to evade collision with followed ship
                                entityProperties.getNode().getLocalInverseZAxis(currentFrontVec);
//                                ENG_Math.rotateAwayFromPositionDeg(otherPos, currentPos, currentFrontVec, getRotationAngle(shipProperties), rotation);
//                                entityProperties.rotate(rotation, true, TransformSpace.TS_WORLD);
                                if (DEBUG) {
                                    System.out.println("entity: " + entityProperties.getName() + " Evading collision " + ENG_Utility.currentTimeMillis());
                                }
                                Utility.rotateAwayFromPosition(currentFrontVec, otherPos, updateInterval, entityProperties,
                                        shipProperties.getShipData().maxAngularVelocity);
                                break;
                            }
                        }

                        // Find if some enemy is targeting us
                        if (ENG_Utility.hasRandomChanceHit(FrameInterval.CHECK_TARGETED_CHANCE + entityProperties.getNode().getName(), CHECK_TARGETED_CHANCE)) {
                            boolean targeted = checkTargeted(aiProperties, shipProperties);
                            if (targeted) {
                                break;
                            }
                        }

                        // Check if we are not followed by a projectile
                        if (shipProperties.isChased()) {
                            aiProperties.setState(AIState.EVADE_MISSILE);
                            showAIStateChange(shipProperties, aiProperties);
                            break;
                        }

                        // Rotate toward the followed ship to take it in the
                        // crosshair
                        // entityProperties.getNode().getLocalXAxis(currentFrontVec);
                        // entityProperties.getNode().getLocalYAxis(currentFrontVec);
                        entityProperties.getNode().getLocalInverseZAxis(currentFrontVec);
                        entityProperties.getNode().getLocalYAxis(currentUpVec);

                        ENG_Quaternion currentOrientation = new ENG_Quaternion();
                        entityProperties.getNode().getOrientation(currentOrientation);
                        ENG_Vector4D orientedUpVec = currentOrientation.mul(currentUpVec);
                        orientedUpVec.mul(0.3f);
                        ENG_Vector4D underShipPos = currentPos.subAsVec(orientedUpVec);

                        otherShipEntityProperties.getNode().getPosition(otherPos);
                        // System.out.println("playerShip pos: " + otherPos);
                        otherPos.sub(underShipPos, distVec);
                        distVec.normalize();
                        float angleBetween = currentFrontVec.angleBetween(distVec);
                        // System.out.println("angleBetween: " + angleBetween *
                        // ENG_Math.RADIANS_TO_DEGREES);
                        // System.out.println("ai system front vec: " +
                        // currentFrontVec);
                        if (angleBetween < TARGETING_ANGLE) {
                            entityProperties.getRigidBody().setAngularVelocity(new Vector3());
                            aiProperties.setState(AIState.SHOOT_PLAYER);
                            showAIStateChange(shipProperties, aiProperties);
                        } else {
                            Utility.rotateToPosition(currentFrontVec, distVec, updateInterval, entityProperties,
                                    shipProperties.getShipData().maxAngularVelocity);
                        }
                    }
                }
                break;
                case SHOOT_PLAYER: {
                    // Find the most powerful weapon we have in order to shoot
                    Entity followedShip = MainApp.getGame().getWorldManager().getEntityByItemId(aiProperties.getFollowedShip());
                    if (followedShip == null) {
                        aiProperties.setEnemySelected(false);
                        aiProperties.setState(AIState.SEEK_CLOSEST_PLAYER);
                        break;
                    }

                    ShipProperties followedShipShipProperties = shipPropertiesMapper.getSafe(followedShip);
                    if (followedShipShipProperties == null) {
                        throw new ENG_InvalidFieldStateException(entityPropertiesMapper.get(followedShip).getItem().getName() + " is not a ship");
                    }
                    if (followedShipShipProperties.getChasingProjectilesNum() > MAX_CHASING_PROJECTILES_NUM) {
                        aiProperties.setEnemySelected(false);
                        aiProperties.setState(AIState.SEEK_CLOSEST_PLAYER);
                        showAIStateChange(shipProperties, aiProperties);
                        break;
                    }
                    if (aiProperties.getCurrentShootingAtShip() != -1 && aiProperties.getCurrentShootingAtShip() == aiProperties.getFollowedShip()) {
                        if (aiProperties.isLimitProjectilesLaunched()) {
                            if (ENG_Utility.hasTimePassed(FrameInterval.MAX_PROJECTILES_LAUNCHED_DELAY + entityProperties.getNode().getName(),
                                    aiProperties.getLimitProjectilesLaunchedStartTime(),
                                    MAX_PROJECTILES_LAUNCHED_DELAY)) {
                                aiProperties.setLimitProjectilesLaunched(false);
                                aiProperties.setCurrentLaunchedProjectiles(0);
                                // System.out.println(shipProperties.getName() +
                                // " max projectiles delay expired");
                            } else {
                                aiProperties.setState(AIState.FOLLOW_PLAYER);
                                showAIStateChange(shipProperties, aiProperties);
                                break;
                            }
                        }
                        if (aiProperties.getCurrentLaunchedProjectiles() < MAX_CHASING_PROJECTILES_NUM) {
                            // May be decremented when we realize we are too far
                            // away and don't shoot
                            aiProperties.incrementCurrentLaunchedProjectiles();
                            // System.out.println(shipProperties.getName() +
                            // " incrementing projectiles to " +
                            // aiProperties.getCurrentLaunchedProjectiles());
                        } else {
                            aiProperties.setLimitProjectilesLaunched(true);
                            aiProperties.setLimitProjectilesLaunchedStartTime();
                            // System.out.println(shipProperties.getName() +
                            // " max projectiles limit reached");
                        }

                    } else {
                        aiProperties.setCurrentShootingAtShip(aiProperties.getFollowedShip());
                        aiProperties.setCurrentLaunchedProjectiles(1);
                        aiProperties.setLimitProjectilesLaunched(false);
                    }
                    currentWeaponsList.clear();
                    currentWeaponsList.addAll(shipProperties.getShipData().weaponTypeList);
                    Collections.sort(currentWeaponsList, weaponComparator);
                    if (!currentWeaponsList.isEmpty()) {
                        boolean launch = false;
                        for (WeaponType weaponType : currentWeaponsList) {
                            // WeaponType weaponType = currentWeaponsList.get(0);
                            boolean specialWeapon = WeaponType.getSpecialWeapon(weaponType);

                            if (specialWeapon) {
                                weaponProperties.setCurrentWeaponType(weaponType);
                                if (weaponProperties.hasCurrentWeaponAmmo()
                                        && ENG_Utility
                                        .hasRandomChanceHit(FrameInterval.SPECIAL_WEAPON_CHANCE + entityProperties.getNode().getName() + "_" + weaponType, SPECIAL_WEAPON_CHANCE)) {
                                    launch = true;
                                    break;
                                }
                            }
                        }

                        if (!launch) { // No specials for now
                            // weaponProperties.setCurrentWeaponType(currentWeaponsList.get(0));

                            for (WeaponType wpn : currentWeaponsList) {
                                if (!WeaponType.getSpecialWeapon(wpn)) {
                                    weaponProperties.setCurrentWeaponType(wpn);
                                    if (weaponProperties.hasCurrentWeaponAmmo()) {
                                        launch = true;
                                        break;
                                    }
                                }
                            }
                        }
                        if (launch) {
                            EntityProperties followedShipEntityComponent = entityPropertiesMapper.get(followedShip);
                            if (followedShipEntityComponent.getNode().getPosition().distance(entityProperties.getNode().getPosition()) >
                                    WeaponData.getWeaponData(weaponProperties.getCurrentWeaponType()).maxDistance) {
                                aiProperties.decrementCurrentLaunchedProjectiles();
                                // aiProperties.setEnemySelected(false);
                                aiProperties.setState(AIState.FOLLOW_PLAYER);
                                break;
                            }

                            if (!aiProperties.isEnemySelected()) {
                                aiProperties.setEnemySelected(true);
                                aiProperties.setEnemySelectionTimeStarted();
                            } else if (ENG_Utility.hasTimePassed(
                                    FrameInterval.WEAPON_ENEMY_SELECTION_TIME + entityProperties.getNode().getName(),
                                    aiProperties.getEnemySelectionTimeStarted(),
                                    headwayent.blackholedarksun.entitydata.WeaponData.WeaponType.getWeaponEnemySelectionTime(weaponProperties.getCurrentWeaponType()))) {
                                if (ENG_Utility
                                        .hasTimePassed(
                                                FrameInterval.WEAPON_COOLDOWN_TIME + entityProperties.getNode().getName(),
                                                aiProperties.getWeaponCooldownTimeStarted(),
                                                WeaponType.getWeaponCooldownTime(weaponProperties.getCurrentWeaponType()))) {
                                    System.out.println("setCurrentSelectedEnemy from AIManager for ship: " + entityProperties.getName() + " followed ship: " + aiProperties.getFollowedShip());
                                    shipProperties.setCurrentSelectedEnemy(aiProperties.getFollowedShip());

                                    // If we're not on homing missiles we must shoot
                                    // ahead
                                    shootAhead(aiProperties, entityProperties, shipProperties, weaponProperties, followedShip);

                                    // if (!projectileCreated) {
                                    MainApp.getGame().getWorldManager().createProjectile(e);
                                    // projectileCreated = true;
                                    // }
                                    shipProperties.resetCurrentSelectedEnemy();
                                    aiProperties.setWeaponCooldownTimeStarted();
                                    weaponProperties.decrementCurrentWeaponAmmo();
                                    // System.out.println("Projectile " +
                                    // weaponProperties.getCurrentWeaponType() +
                                    // " launched by " + shipProperties.getName());
                                }
                            }
                        }
                    }
                    boolean targeted = checkTargeted(aiProperties, shipProperties);
                    if (targeted) {
                        break;
                    }
                    aiProperties.setState(AIState.FOLLOW_PLAYER);
                }
                break;
                case EVADE_MISSILE: {
                    // Seek the optimal way to escape the closest chasing projectile
                    if (!shipProperties.isChased()) {
                        aiProperties.setState(AIState.SEEK_CLOSEST_PLAYER);
                        showAIStateChange(shipProperties, aiProperties);
                        break;
                    }
                    entityProperties.getNode().getPosition(currentPos);
                    MainApp.getGame().getWorldManager().getClosestProjectileSquaredDistance(currentPos, shipProperties, data, otherPos);
                    float minSquaredDist = data.minDist;
                    Entity minDistProjectile = MainApp.getGame().getWorldManager().getEntityByGameEntityId(data.objectId);

                    if (minDistProjectile != null) {
                        // Launch countermeasures if close enough
                        if (minSquaredDist < MIN_COUNTERMEASURES_DISTANCE) {
                            if (ENG_Utility.hasTimePassed(
                                    FrameInterval.AI_COUNTERMEASURE_TIME + entityProperties.getNode().getName(),
                                    shipProperties.getCountermeasuresLastLaunchTime(),
                                    headwayent.blackholedarksun.entitydata.ShipData.COUNTERMEASURE_TIME)) {
                                MainApp.getGame().getWorldManager().createCountermeasures(e);
                                shipProperties.setAfterburnerActive(true);
                                shipProperties.setCountermeasuresLastLaunchTime();
//                                System.out.println("Creating countermeasure for entity: " + entityProperties.getName());
                            }
                        }
                        EntityProperties projectileEntityProperties = entityPropertiesMapper.get(minDistProjectile);
                        entityProperties.getNode().getLocalZAxis(currentFrontVec);
                        entityProperties.getNode().getPosition(currentPos);
                        projectileEntityProperties.getNode().getLocalZAxis(otherFrontVec);
                        projectileEntityProperties.getNode().getPosition(otherPos);
                        // Try to evade by moving erratically random and away from
                        // incoming projectile
                        if (minSquaredDist < MIN_EVASION_DISTANCE) {
                            // Try to create a 90 degrees angle to evade projectile
                            if (ENG_Utility.hasRandomChanceHit(FrameInterval.MIN_EVASION_DISTANCE + entityProperties.getNode().getName(), 2)) {
                                entityProperties.getNode().getLocalXAxis(perpendicularVec);
                            } else {
                                entityProperties.getNode().getLocalYAxis(perpendicularVec);
                            }
                            if (ENG_Utility.hasRandomChanceHit(FrameInterval.MIN_EVASION_DISTANCE_INVERT + entityProperties.getNode().getName(), 2)) {
                                perpendicularVec.invertInPlace();
                            }
//                            ENG_Math.rotateToDirectionDeg(perpendicularVec, currentFrontVec, getRotationAngle(shipProperties), rotation);
                            Utility.rotateToPosition(currentFrontVec, perpendicularVec, updateInterval, entityProperties,
                                    shipProperties.getShipData().maxAngularVelocity);

                        } else {

                            // Try to outrun the projectile

//                            ENG_Math.rotateAwayFromPositionDeg(otherPos, currentPos, currentFrontVec, getRotationAngle(shipProperties), rotation);
                            Utility.rotateAwayFromPosition(currentFrontVec, otherPos, updateInterval, entityProperties,
                                    shipProperties.getShipData().maxAngularVelocity);

                        }
//                        entityProperties.rotate(rotation, true, TransformSpace.TS_WORLD);

                        float maxSpeed = shipProperties.getShipData().maxSpeed;
                        if (maxSpeed > entityProperties.getVelocity()) {
                            changeVelocity(entityProperties, shipProperties, aiProperties,
                                    new VelocityChange(
                                            VELOCITY_CHANGE_NAME_EVADE_MISSILE,
                                            maxSpeed, EVASION_MISSILE_SPEED_CHANGE_STEP));
                        }
                    }
                }
                break;
                case COLLISION_RESPONSE: {
                    if (aiProperties.isCollided()) {
                        if (!aiProperties.isAxisAndAngleSelected()) {
                            aiProperties.setAxisAndAngleSelected(true);
                            // Take either the local x or y axis to be the destination towards we rotate the ship.
                            int nextInt = ENG_Utility.getRandom().nextInt(FrameInterval.COLLISION_RESPONSE_AXIS + entityProperties.getNode().getName(), 2);
                            switch (nextInt) {
                                case 0:
                                    entityProperties.getNode().getLocalXAxis(collisionResponseAxis);
                                    break;
                                case 1:
                                    entityProperties.getNode().getLocalYAxis(collisionResponseAxis);
                                    break;
                                default:
                                    throw new IllegalArgumentException();
                            }
                            aiProperties.setCollisionEvasionDestination(collisionResponseAxis);
                            aiProperties.setCollisionAngleDirection(
                                    ENG_Utility.hasRandomChanceHit(FrameInterval.COLLISION_RESPONSE_DIR + entityProperties.getNode().getName(), 2) ? -1.0f : 1.0f);
                            entityProperties.setVelocity(0.0f);
                        }
                    }
                    if (aiProperties.getCurrentCollisionAngle() < COLLISION_RESPONSE_ACCELERATION_ANGLE) {
//                        float rotationAngle = getRotationAngle(shipProperties);
                        entityProperties.getNode().getLocalInverseZAxis(currentFrontVec);
                        if (!aiProperties.isInitialFrontVecSet()) {
                            aiProperties.setInitialFrontVec(currentFrontVec);
                            aiProperties.setInitialFrontVecSet(true);
                        }
                        ENG_Vector4D axis = aiProperties.getCollisionAngleDirection() == 1 ?
                                aiProperties.getCollisionEvasionDestination().invert() :
                                aiProperties.getCollisionEvasionDestination();
                        Utility.rotateToPosition(currentFrontVec,
                                axis,
                                updateInterval, entityProperties,
                                shipProperties.getShipData().maxAngularVelocity);
//                        entityProperties.rotate(ENG_Quaternion.fromAngleAxisDegRet(
//                                        rotationAngle * aiProperties.getCollisionAngleDirection(),
//                                        aiProperties.getCollisionEvasionDestination()));
                        aiProperties.setCurrentCollisionAngle(currentFrontVec.angleBetween(aiProperties.getInitialFrontVec()));
                        if (DEBUG) {
                            System.out.println("COLLISION_RESPONSE: entity: " + entityProperties.getName() + " angleBetweenDeg: " + (currentFrontVec.angleBetween(aiProperties.getInitialFrontVec()) * ENG_Math.RADIANS_TO_DEGREES));
                        }
                    } else {
                        if (!aiProperties.isCollisionResponseMovement()) {
                            aiProperties.setCollisionResponseMovement(true);
                            aiProperties.setCollisionResponseMovementTime();
                        }
                        if (ENG_Utility.hasTimePassed(
                                FrameInterval.COLLISION_RESPONSE_ACCELERATION_TIME + entityProperties.getNode().getName(),
                                aiProperties.getCollisionResponseMovementTime(),
                                COLLISION_RESPONSE_ACCELERATION_TIME)) {
                            aiProperties.setCollisionResponseMovement(false);
                            aiProperties.setAxisAndAngleSelected(false);
                            aiProperties.setCollided(false);
                            aiProperties.setState(AIState.SEEK_CLOSEST_PLAYER);
                            aiProperties.setCurrentCollisionAngle(0.0f);
                            aiProperties.setInitialFrontVec(ENG_Math.VEC4_ZERO);
                            aiProperties.setInitialFrontVecSet(false);
                        } else {
                            if (DEBUG) {
                                System.out.println("COLLISION_RESPONSE: entity: " + entityProperties.getName() + " velocityChange: " + shipProperties.getShipData().maxSpeed);
                            }
                            changeVelocity(entityProperties, shipProperties, aiProperties,
                                    new VelocityChange(
                                            VELOCITY_CHANGE_NAME_COLLISION_RESPONSE,
                                            shipProperties.getShipData().maxSpeed,
                                            COLLISION_RESPONSE_ACCELERATION_STEP));
                        }
                    }
                }
                break;
                case EVADE_HIT: {
                    if (!aiProperties.isEvadingHit()) {
                        aiProperties.setEvadingHit(true);
                        aiProperties.setHitEvasionTime();
                    }
                    if (ENG_Utility.hasTimePassed(
                            FrameInterval.EVASION_HIT_TIME + entityProperties.getNode().getName(),
                            aiProperties.getHitEvasionTime(),
                            EVASION_HIT_TIME)) {
                        aiProperties.setEvadingHit(false);
                        aiProperties.setEvadeHitDirectionSet(false);
                        aiProperties.setState(AIState.SEEK_CLOSEST_PLAYER);
                        showAIStateChange(shipProperties, aiProperties);
                    } else {
                        float maxSpeed = shipProperties.getShipData().maxSpeed;
                        if (maxSpeed > entityProperties.getVelocity()) {
                            changeVelocity(entityProperties, shipProperties, aiProperties,
                                    new VelocityChange(
                                            VELOCITY_CHANGE_NAME_EVADE_HIT,
                                            maxSpeed, EVASION_HIT_SPEED_CHANGE_STEP));
                        }
                        int nextInt;
                        float dir;
                        if (aiProperties.isEvadeHitDirectionSet()) {
                            nextInt = aiProperties.getEvadeHitAxis();
                            dir = aiProperties.getEvadeHitDirection();
                        } else {
                            nextInt = ENG_Utility.getRandom().nextInt(FrameInterval.EVADE_HIT_AXIS + entityProperties.getNode().getName(), 3);
                            dir = ENG_Utility.hasRandomChanceHit(FrameInterval.EVADE_HIT_DIR + entityProperties.getNode().getName(), 2) ? -1.0f : 1.0f;
                            aiProperties.setEvadeHitAxisAndDirection(nextInt, dir);
                            aiProperties.setEvadeHitDirectionSet(true);
                        }
//					System.out.println(shipProperties.getName() +
//							" is evading in dir: " + nextInt);
                        entityProperties.getNode().getLocalInverseZAxis(currentFrontVec);
//                        int nextInt = ENG_Utility.getRandom().nextInt(FrameInterval.COLLISION_RESPONSE_AXIS + entityProperties.getNode().getName(), 3);
                        switch (nextInt) {
                            case 0:
                                entityProperties.getNode().getLocalXAxis(collisionResponseAxis);
                                break;
                            case 1:
                                entityProperties.getNode().getLocalYAxis(collisionResponseAxis);
                                break;
                            case 2:
                                entityProperties.getNode().getLocalZAxis(collisionResponseAxis);
                                break;
                            default:
                                throw new IllegalArgumentException();
                        }
                        Utility.rotateToPosition(currentFrontVec, collisionResponseAxis, updateInterval, entityProperties,
                                shipProperties.getShipData().maxAngularVelocity);
                    }
                }
                break;
                default:
                    throw new IllegalArgumentException(aiProperties.getState() + " is not a handled state");
            }
//            System.out.println("Ship " + entityProperties.getName() + " ending AI state: " + aiProperties.getState());
        }
    }

    private void seekClosestPlayer(AIProperties aiProperties, EntityProperties entityProperties, ShipProperties shipProperties) {
        if (ENG_Utility.hasRandomChanceHit(FrameInterval.ENEMY_SEEK_RAND + entityProperties.getNode().getName(), ENEMY_SEEK_RAND)) {
            ImmutableBag<Entity> entities = GameWorld.getWorld().getManager(GroupManager.class).getEntities(
                            ShipTeam.getOtherTeamAsString(shipProperties.getShipData().team));
            entityProperties.getNode().getPosition(currentPos);
            // Find smallest distance
            long closestEnemy = -1;
            int enemyTeamSize = entities.size();
            currentMinLen = Float.MAX_VALUE;
            for (int i = 0; i < enemyTeamSize; ++i) {
                Entity entity = entities.get(i);
                EntityProperties otherShipEntityProperties = entityPropertiesMapper.get(entity);
                ShipProperties otherShipShipProperties = shipPropertiesMapper.getSafe(entity);
                if (otherShipShipProperties != null && otherShipShipProperties.getShipData().shipType == ShipData.ShipType.CARGO) {
                    // Try to prioritize the enemy cargo ship.
                    if (i == 0) {
                        currentMinLen = distVec.squaredLength();
                        closestEnemy = otherShipEntityProperties.getItemId();
                    } else {
                        if (distVec.squaredLength() < currentMinLen) {
                            currentMinLen = distVec.squaredLength();
                            closestEnemy = otherShipEntityProperties.getItemId();
                        }
                    }
                }
            }
            if ((closestEnemy == -1) ||
                    (closestEnemy != -1 &&
                            ENG_Utility.getRandom().nextInt(FrameInterval.CARGO_SHIP_SELECT_CLOSEST + entityProperties.getNode().getName(), 3) != 0)) {
                for (int i = 0; i < enemyTeamSize; ++i) {
                    Entity entity = entities.get(i);
                    EntityProperties otherShipEntityProperties = entityPropertiesMapper.get(entity);
//                    ShipProperties otherShipShipProperties = shipPropertiesMapper.get(entity);

                    otherShipEntityProperties.getNode().getPosition(otherPos);
                    otherPos.sub(currentPos, distVec);

                    if (i == 0) {
                        currentMinLen = distVec.squaredLength();
                        closestEnemy = otherShipEntityProperties.getItemId();
                    } else {
                        if (distVec.squaredLength() < currentMinLen) {
                            currentMinLen = distVec.squaredLength();
                            closestEnemy = otherShipEntityProperties.getItemId();
                        }
                    }
                }
            }
            if (closestEnemy != -1) {
                // Disable all patroling data
                aiProperties.setPatroling(false);
                aiProperties.setPatrolingRotationStarted(false);
                aiProperties.setFollowedShip(closestEnemy);
                aiProperties.setState(AIState.FOLLOW_PLAYER);
                if (DEBUG) {
                    System.out.println(entityProperties.getName() + " changed state to FOLLOW_PLAYER. Following: " + closestEnemy);
                }
            } else {
                // Slow down and patrol the area
                aiProperties.setPatroling(true);
                changeVelocity(entityProperties, shipProperties, aiProperties, new VelocityChange(
                                VELOCITY_CHANGE_NAME_PATROL_SLOW_DOWN,
                                shipProperties.getShipData().maxSpeed/* * PATROL_SPEED_COEFICIENT*/,
                                NO_TARGET_SPEED_CHANGE_STEP));
                if (!aiProperties.isPatrolingRotationStarted()) {
                    if (ENG_Utility.hasRandomChanceHit(FrameInterval.NO_ENEMY_DIRECTION_CHANGE_CHANCE + entityProperties.getNode().getName(),
                                    NO_ENEMY_DIRECTION_CHANGE_CHANCE)) {
                        float dir = ENG_Utility.hasRandomChanceHit(
                                FrameInterval.NO_ENEMY_DIRECTION_CHANGE_CHANCE_DIR + entityProperties.getNode().getName(), 2) ? -1.0f : 1.0f;
                        aiProperties.setPatrolAngleDirection(dir);
                        int nextInt = ENG_Utility.getRandom().nextInt(FrameInterval.PATROL_AXIS + entityProperties.getNode().getName(), 3);
                        switch (nextInt) {
                            case 0:
                                entityProperties.getNode().getLocalXAxis(patrolingRotationAxis);
                                break;
                            case 1:
                                entityProperties.getNode().getLocalYAxis(patrolingRotationAxis);
                                break;
                            case 2:
                                entityProperties.getNode().getLocalZAxis(patrolingRotationAxis);
                                break;
                            default:
                                throw new IllegalArgumentException();
                        }
                        aiProperties.setPatrolingRotationAxis(patrolingRotationAxis);
                        aiProperties.setPatrolingRotationTimeStarted();
                        aiProperties.setPatrolingRotationStarted(true);
                    }
                }

                if (aiProperties.isPatrolingRotationStarted()) {
                    if (ENG_Utility.hasTimePassed(FrameInterval.PATROLING_ROTATION_TIME + entityProperties.getNode().getName(),
                            aiProperties.getPatrolingRotationTimeStarted(),
                            PATROLING_ROTATION_TIME)) {

                        aiProperties.setPatrolingRotationStarted(false);
                        aiProperties.setCurrentCollisionAngle(0.0f);
                    } else {
                        ENG_Vector4D torque = new ENG_Vector4D();

                        aiProperties.getPatrolingRotationAxis().mulRet(aiProperties.getPatrolAngleDirection() * shipProperties.getShipData().maxAngularVelocity, torque);
                        entityProperties.getRigidBody().applyTorqueImpulse(new Vector3(torque.x, torque.y, torque.z));
//                        float rotationAngle = getRotationAngle(shipProperties);
//                        ENG_Quaternion neworientation = ENG_Quaternion.fromAngleAxisDegRet(
//                                rotationAngle * aiProperties.getPatrolAngleDirection(),
//                                aiProperties.getPatrolingRotationAxis());
//                        ENG_Vector4D test = new ENG_Vector4D();
//                        float angleDeg = neworientation.toAngleAxisDeg(test);
//                        entityProperties.rotate(neworientation);
//                        aiProperties.setCurrentPatrolingRotationAngle(aiProperties.getCurrentPatrolingRotationAngle() + rotationAngle);
                    }
                }
            }
        }
    }

    private void followPlayerShip(AIProperties aiProperties,
                                  EntityProperties entityProperties, ShipProperties shipProperties) {
        Entity playerShip = MainApp.getGame().getWorldManager().getPlayerShip();
        if (playerShip != null) {
            EntityProperties playerShipEntityProperties = entityPropertiesMapper.get(playerShip);
            if (!playerShipEntityProperties.isDestroyed()) {
                float scanRadius = shipProperties.getScanRadius();
                float distance = playerShipEntityProperties.getNode().getPosition().distance(entityProperties.getNode().getPosition());
                if (!aiProperties.isReloaderShouldLeaveWorld()) {
                    if (distance < scanRadius) {
                        changeVelocity(entityProperties, shipProperties, aiProperties,
                                new VelocityChange(VELOCITY_CHANGE_NAME_RELOADER_STOP, 0.0f, RELOADER_SPEED_CHANGE_RATE));
                        if (entityProperties.getVelocity() < ENG_Math.FLOAT_EPSILON && playerShipEntityProperties.getVelocity() < ENG_Math.FLOAT_EPSILON) {
                            // We can begin to reload the player ship
                            ShipProperties playerShipShipProperties = shipPropertiesMapper.get(playerShip);
                            WeaponProperties playerShipWeaponProperties = weaponPropertiesMapper.get(playerShip);
                            if (playerShipWeaponProperties.hasWeapons() && ENG_Utility.hasTimePassed(FrameInterval.RELOADER_TIME_BETWEEN_RELOADING_UNITS,
                                            aiProperties.getReloaderShipIncrementWeaponNumTime(),
                                            RELOADER_TIME_BETWEEN_RELOADING_UNITS)) {
                                aiProperties.setReloaderShipIncrementWeaponNumTime();
                                boolean weaponIncremented = false; // Check if at least a weapon has been incremented.
                                for (headwayent.blackholedarksun.entitydata.WeaponData.WeaponType wpn : playerShipShipProperties.getShipData().weaponTypeList) {
                                    if (!headwayent.blackholedarksun.entitydata.WeaponData.WeaponType.hasInfiniteAmmo(wpn)) {
                                        if (playerShipWeaponProperties.getWeaponAmmo(wpn) < headwayent.blackholedarksun.entitydata.WeaponData.WeaponType.getDefaultMissileNumber(wpn)) {
                                            playerShipWeaponProperties.incrementWeaponAmmo(wpn, 1);
                                            weaponIncremented = true;
                                        }
                                    }
                                }
                                if (!weaponIncremented) {
                                    // All weapons full so leave world
                                    aiProperties.setReloaderShouldLeaveWorld(true);
                                }
                            }
                        }
                    } else {
                        ENG_Quaternion rotation = new ENG_Quaternion();
//                        ENG_Math.rotateTowardPositionDeg(
//                                playerShipEntityProperties.getNode().getPosition(),
//                                entityProperties.getNode().getPosition(),
//                                entityProperties.getNode().getLocalInverseZAxis(),
//                                entityProperties.getNode().getLocalYAxis(),
//                                rotation,
//                                getRotationAngle(shipProperties));
//                        entityProperties.rotate(rotation, true, TransformSpace.TS_WORLD);
                        Utility.rotateToPosition(entityProperties.getNode().getLocalInverseZAxis(),
                                playerShipEntityProperties.getNode().getPosition().subAsVec(entityProperties.getNode().getPosition()),
                                updateInterval, entityProperties,
                                shipProperties.getShipData().maxAngularVelocity);
//                        ENG_Vector4D axis = new ENG_Vector4D();
//                        float angle = rotation.toAngleAxisDeg(axis);
//                        System.out.println("rotation axis: " + axis + " angle: " + angle);
                        changeVelocity(entityProperties, shipProperties, aiProperties,
                                new VelocityChange(
                                        VELOCITY_CHANGE_NAME_RELOADER_TOWARDS_SHIP,
                                        shipProperties.getShipData().maxSpeed,
                                        RELOADER_SPEED_CHANGE_RATE));
                    }
                } else { // Should leave the world
//                    ENG_Quaternion rotation = ENG_Math.rotateAwayFromPositionDeg(
//                                    playerShipEntityProperties.getNode().getPosition(),
//                                    entityProperties.getNode().getPosition(),
//                                    entityProperties.getNode().getLocalInverseZAxis(),
//                                    getRotationAngle(shipProperties));
//                    ENG_Vector4D axis = new ENG_Vector4D();
//                    // We don't care about the axis
//                    float angle = ENG_Quaternion.toAngleAxisDeg(rotation, axis);
//                    aiProperties.setReloaderCurrentAwayAngle(aiProperties.getReloaderCurrentAwayAngle() + angle);
//                    entityProperties.rotate(rotation);
                    ENG_Vector4D frontVec = entityProperties.getNode().getLocalInverseZAxis();
                    ENG_Vector4D targetVec = playerShipEntityProperties.getNode().getPosition().subAsVec(entityProperties.getNode().getPosition());
                    Utility.rotateAwayFromPosition(frontVec,
                            targetVec,
                            updateInterval, entityProperties,
                            shipProperties.getShipData().maxAngularVelocity);
                    aiProperties.setReloaderCurrentAwayAngle(frontVec.angleBetween(targetVec));
                    if (aiProperties.getReloaderCurrentAwayAngle() > RELOADER_AWAY_ANGLE) {
                        Utility.clearAngularVelocity(entityProperties.getRigidBody());
                        MainApp.getGame().getWorldManager().startAnimation(entityProperties.getEntityId(), shipProperties.getExitedWorldAnimation());
                    }
                }
            }
        }
    }

    private void evadeLevelLimits(AIProperties aiProperties, EntityProperties entityProperties, ShipProperties shipProperties) {
        entityProperties.getLimitsReached(levelLimits);
        // Even if we are no longer touching the level limits we must still complete
        // the turn around.
        if (levelLimits.equals(ENG_Math.VEC4_ZERO) && rotationAwayFromLimitsCompleted) {
            aiProperties.setState(AIState.SEEK_CLOSEST_PLAYER);
            showAIStateChange(shipProperties, aiProperties);
        } else {
            // We must make sure that we reset the level limits for each frame or we
            // will end up with previous limits never going away since they are
            // only additive in EntityContactListener.
            entityProperties.resetLimitsReached();
            // If we have new level limits should we check and recreate awayFromLimitsPos?
            // Using what we have here we might become stuck switching between 2 levels limits
            // forever.
            if  (awayFromLimitsPosSet && !levelLimits.equals(currentLevelLimits) && !levelLimits.equals(ENG_Math.VEC4_ZERO)) {
                System.out.println("levelLimits: " + levelLimits + " currentLevelLimits: " + currentLevelLimits);
                awayFromLimitsPosSet = false;
            }
            if (!awayFromLimitsPosSet) {
                levelLimits.invert(awayFromLimitsPos);
                awayFromLimitsPos.normalize();
                currentLevelLimits.set(levelLimits);
                // First slow down to 0 then rotate away.
                entityProperties.setVelocity(0.0f);
                awayFromLimitsPosSet = true;
                rotationAwayFromLimitsCompleted = false;
            }

//            awayFromLimitsPos.addInPlace(entityProperties.getNode().getPosition());
            entityProperties.getNode().getPosition(currentPos);
            entityProperties.getNode().getLocalInverseZAxis(currentFrontVec);
//            entityProperties.getNode().getLocalYAxis(currentUpVec);

            float angleBetween = currentFrontVec.angleBetween(awayFromLimitsPos);


//            ENG_Math.rotateTowardPositionDeg(awayFromLimitsPos, currentPos, currentFrontVec, currentUpVec, rotation, getRotationAngle(shipProperties));
//            entityProperties.rotate(rotation, true, TransformSpace.TS_WORLD);
            rotateTowardPosition(entityProperties, shipProperties, awayFromLimitsPos);
//            Utility.rotateToPosition(currentFrontVec,
//                    awayFromLimitsPos,
//                    updateInterval, entityProperties,
//                    shipProperties.getShipData().maxAngularVelocity);



            if (angleBetween < ESCAPE_LEVEL_LIMITS_ANGLE) {
                rotationAwayFromLimitsCompleted = true;
                awayFromLimitsPosSet = false;
                // Also get to max speed
                changeVelocity(entityProperties, shipProperties, aiProperties, new VelocityChange(
                        VELOCITY_CHANGE_NAME_ESCAPING_LIMITS,
                        shipProperties.getShipData().maxSpeed,
                        ESCAPING_LIMITS_ACCELERATION_RATE));
            }
            if (DEBUG) {
                System.out.println("ship " + entityProperties.getUniqueName()
                        + " has reached the limits " + levelLimits
                        + " and moved towards " + awayFromLimitsPos
                        + " . Current pos: " + currentPos);
            }
        }
    }

    private void reachDestination(AIProperties aiProperties, EntityProperties entityProperties, ShipProperties shipProperties) {
        aiProperties.getDestination(destination);
        entityProperties.getNode().getPosition(currentPos);
        if (entityProperties.getRadius() < destination.distance(currentPos)) {
            rotateTowardPosition(entityProperties, shipProperties, destination);
            float maxSpeed = shipProperties.getShipData().maxSpeed;
            if (maxSpeed > entityProperties.getVelocity()) {
                changeVelocity(entityProperties, shipProperties, aiProperties,
                        new VelocityChange(VELOCITY_CHANGE_NAME_REACHING_DESTINATION, maxSpeed, REACHING_DESTINATION_VELOCITY_CHANGE_STEP));
            }

        } else {
            aiProperties.setDestinationReached(true);
            changeVelocity(entityProperties, shipProperties, aiProperties, new VelocityChange(
                            VELOCITY_CHANGE_NAME_DESTINATION_REACHED,
                            0.0f,
                            REACHING_DESTINATION_VELOCITY_CHANGE_STEP));
        }
    }

    private float getRotationAngle(ShipProperties shipProperties) {
        return shipProperties.getShipData().turnAngle //* 3//* 10
//				* (float) GameWorld.getWorld().getDelta();
                * updateInterval;
    }

    private void showAIStateChange(ShipProperties shipProperties, AIProperties aiProperties) {
        if (DEBUG) {
            System.out.println(shipProperties.getName() + " state changed to " + aiProperties.getState());
        }
    }

    private void shootAhead(AIProperties aiProperties, EntityProperties entityProperties, ShipProperties shipProperties, WeaponProperties weaponProperties, Entity followedShip) {
        if (!WeaponType.isHomingMissileType(weaponProperties.getCurrentWeaponType())) {
            Entity ship = MainApp.getGame().getWorldManager().getEntityByItemId(aiProperties.getFollowedShip());
            WorldManagerBase worldManager = WorldManagerBase.getSingleton();
            EntityProperties followedShipEntityProperties = worldManager.getEntityPropertiesComponentMapper().get(ship);

            WeaponData weaponData = WeaponData.getWeaponData(weaponProperties.getCurrentWeaponType());
            boolean valid = ENG_Utility.calculateCollisionPosition(
                    weaponData.maxSpeed,
                    entityProperties.getNode()._getFullTransformNative().invertAffineRet(),
                    crossPosition,
                    followedShipEntityProperties.getVelocityAsVec(),
                    followedShipEntityProperties.getNode().getOrientation(),
                    followedShipEntityProperties.getNode().getPosition());
            if (valid) {

                entityProperties.getNode().getLocalInverseZAxis(currentFrontVec);
                entityProperties.getNode().getLocalYAxis(currentUpVec);
                entityProperties.getNode().getPosition(currentPos);
                EntityProperties otherShipEntityProperties = worldManager.getEntityPropertiesComponentMapper().get(followedShip);
//                System.out.println("crossPosition: " + crossPosition);
                crossPosition.normalize();
//                entityProperties.getNode()._getFullTransformNative().transform(crossPosition, transformedCrossPosition);
                Utility.rotateToPosition(currentFrontVec, crossPosition, updateInterval, entityProperties,
                        180.0f);
            }
        }
    }

    // private VelocityChange currentVelocityChange;
    private void changeVelocity(EntityProperties entityProperties, ShipProperties shipProperties, AIProperties aiProperties, VelocityChange velocityChange) {

        VelocityChange currentVelocityChange = aiProperties.getCurrentVelocityChange();

        if (currentVelocityChange == null ||
                velocityChange.id != currentVelocityChange.id ||
                currentVelocityChange.currentChange - 1.0f > ENG_Math.FLOAT_EPSILON) {
            currentVelocityChange = velocityChange;
            currentVelocityChange.initialSpeed = entityProperties.getVelocity();
            aiProperties.setCurrentVelocityChange(velocityChange);
        }

        float velocity = entityProperties.getVelocity();
        if (Math.abs(velocity - currentVelocityChange.newSpeed) > ENG_Math.FLOAT_EPSILON && currentVelocityChange.currentChange < 1.0f) {
            currentVelocityChange.currentChange += currentVelocityChange.changeStep;
            entityProperties.setVelocity(currentVelocityChange.initialSpeed
                            + (currentVelocityChange.newSpeed - currentVelocityChange.initialSpeed)
                            * currentVelocityChange.currentChange);

        }
    }

    private boolean checkTargeted(AIProperties aiProperties, ShipProperties shipProperties) {
        ImmutableBag<Entity> enemyEntities = GameWorld.getWorld().getManager(GroupManager.class).getEntities(ShipTeam.getOtherTeamAsString(
                shipProperties.getShipData().team));
        int size = enemyEntities.size();
        boolean targeted = false;
        for (int i = 0; i < size; ++i) {
            ENG_SceneNode node = entityPropertiesMapper.get(enemyEntities.get(i)).getNode();
            node.getPosition(otherPos);
            node.getLocalInverseZAxis(otherFrontVec);

            otherPos.sub(currentPos, distVec);
            distVec.normalize();
            float angleBetween = otherFrontVec.angleBetween(distVec);
            if (angleBetween < TARGETING_ANGLE) {
                // We are being targeted try to evade
                aiProperties.setState(AIState.EVADE_MISSILE);
                targeted = true;
                break;
            }
        }
        return targeted;
    }

    public static class VelocityChange {

        public final int id;
        public float newSpeed;
        public float changeStep;
        public float currentChange;
        public float initialSpeed;

        public VelocityChange(int id) {
            this.id = id;
        }

        public VelocityChange(int id, float newSpeed, float changeStep) {
            this.id = id;
            this.newSpeed = newSpeed;
            this.changeStep = changeStep;
        }
    }
}
