/*
 * Created by Sebastian Bugiu on 4/9/23, 10:11 PM
 * sebastian.bugiu@headwayentertainment.net
 * Last modified 10/31/21, 9:59 AM
 * Copyright (c) 2023.
 * All rights reserved.
 */

package headwayent.blackholedarksun.physics;

import com.artemis.Entity;
import com.badlogic.gdx.math.Matrix3;
import com.badlogic.gdx.math.Matrix4;
import com.badlogic.gdx.math.Vector2;
import com.badlogic.gdx.math.Vector3;
import com.badlogic.gdx.physics.bullet.collision.btBoxShape;
import com.badlogic.gdx.physics.bullet.collision.btCollisionShape;
import com.badlogic.gdx.physics.bullet.collision.btGhostObject;
import com.badlogic.gdx.physics.bullet.dynamics.btDynamicsWorld;
import com.badlogic.gdx.physics.bullet.dynamics.btRigidBody;
import com.badlogic.gdx.physics.bullet.linearmath.btMotionState;
import com.badlogic.gdx.utils.Disposable;

import headwayent.blackholedarksun.MainApp;
import headwayent.blackholedarksun.components.EntityProperties;
import headwayent.hotshotengine.ENG_Matrix3;
import headwayent.hotshotengine.ENG_Matrix4;
import headwayent.hotshotengine.ENG_Vector2D;
import headwayent.hotshotengine.ENG_Vector3D;
import headwayent.hotshotengine.ENG_Vector4D;
import headwayent.hotshotengine.basictypes.ENG_Float;
import headwayent.hotshotengine.renderer.ENG_RenderRoot;

/**
 * Created by sebas on 11-Sep-17.
 */

public class PhysicsUtility {

    public static btGhostObject createGhostObject() {
        return new btGhostObject();
    }

    public static EntityRigidBody createEntityRigidBody(btRigidBody.btRigidBodyConstructionInfo constructionInfo, Entity gameEntity) {
        return new EntityRigidBody(constructionInfo, gameEntity);
    }

    public static btCollisionShape createBoxCollisionShape(ENG_Vector3D halfExtents) {
        return new btBoxShape(new Vector3(halfExtents.x, halfExtents.y, halfExtents.z));
    }

    public static void setLinearVelocity(btRigidBody rigidBody, ENG_Vector3D velocity) {
        rigidBody.setLinearVelocity(new Vector3(velocity.x, velocity.y, velocity.z));
    }

    public static void setAngularVelocity(btRigidBody rigidBody, Vector3 velocity) {
        rigidBody.setAngularVelocity(velocity);
    }

    public static void setAngularVelocity(btRigidBody rigidBody, ENG_Vector3D velocity) {
        rigidBody.setAngularVelocity(new Vector3(velocity.x, velocity.y, velocity.z));
    }

    public static void setLinearFactor(btRigidBody rigidBody, ENG_Vector3D velocity) {
        rigidBody.setLinearFactor(new Vector3(velocity.x, velocity.y, velocity.z));
    }

    public static void setAngularFactor(btRigidBody rigidBody, ENG_Vector3D velocity) {
        rigidBody.setAngularFactor(new Vector3(velocity.x, velocity.y, velocity.z));
    }

    public static ENG_Vector3D getAngularVelocity(btRigidBody rigidBody) {
        Vector3 angularVelocity = rigidBody.getAngularVelocity();
        return new ENG_Vector3D(angularVelocity.x, angularVelocity.y, angularVelocity.z);
    }

    public static ENG_Vector3D getLinearVelocity(btRigidBody rigidBody) {
        Vector3 linearVelocity = rigidBody.getLinearVelocity();
        return new ENG_Vector3D(linearVelocity.x, linearVelocity.y, linearVelocity.z);
    }

    public static void stepSimulation(btDynamicsWorld dynamicWorld, float timeStep) {
        dynamicWorld.stepSimulation(timeStep);
    }

    public static void stepSimulation(btDynamicsWorld dynamicWorld, float timeStep, int maxSteps) {
        dynamicWorld.stepSimulation(timeStep, maxSteps);
    }

    public static void stepSimulation(btDynamicsWorld dynamicWorld, float timeStep, int maxSteps, float fixedStep) {
        dynamicWorld.stepSimulation(timeStep, maxSteps, fixedStep);
    }

    public static void addRigidBody(btDynamicsWorld dynamicWorld, btRigidBody body, short collisionGroup, short collisionMask) {
        dynamicWorld.addRigidBody(body, collisionGroup, collisionMask);
    }

    public static void addGhostObject(btDynamicsWorld dynamicWorld, btGhostObject ghostObject, short collisionGroup, short collisionMask) {
        dynamicWorld.addCollisionObject(ghostObject, collisionGroup, collisionMask);
    }

    public static void addRigidBody(btDynamicsWorld dynamicWorld, btRigidBody body) {
        dynamicWorld.addRigidBody(body);
    }

    public static void removeRigidBody(btDynamicsWorld dynamicWorld, btRigidBody body) {
        dynamicWorld.removeRigidBody(body);
    }

    public static btRigidBody.btRigidBodyConstructionInfo createConstructionInfo(
            float mass, btMotionState motionState, btCollisionShape shape) {
        ENG_Vector3D fallInertia = calculateLocalInertia(shape, mass);
        Vector3 vec = new Vector3(fallInertia.x, fallInertia.y, fallInertia.z);
        return new btRigidBody.btRigidBodyConstructionInfo(mass, motionState, shape, vec);

    }

    public static btRigidBody.btRigidBodyConstructionInfo createConstructionInfo(
            float mass, btMotionState motionState, btCollisionShape shape, ENG_Vector3D fallInertia) {
        if (fallInertia == null) {
            fallInertia = calculateLocalInertia(shape, mass);
        }
        Vector3 vec = new Vector3(fallInertia.x, fallInertia.y, fallInertia.z);
        return new btRigidBody.btRigidBodyConstructionInfo(mass, motionState, shape, vec);

    }

    public static ENG_Vector3D calculateLocalInertia(btCollisionShape collisionShape, float mass) {
        Vector3 vec = new Vector3();
        collisionShape.calculateLocalInertia(mass, vec);
        return new ENG_Vector3D(vec.x, vec.y, vec.z);
    }

    public static void calculateLocalInertia(btCollisionShape collisionShape, float mass, ENG_Vector3D ret) {
        Vector3 vec = new Vector3();
        collisionShape.calculateLocalInertia(mass, vec);
        ret.set(vec.x, vec.y, vec.z);
    }

    public static void printRigidBody(btRigidBody rigidBody) {
        Matrix3 invInertiaTensorWorld = new Matrix3();
        Vector3 linearVelocity = new Vector3();
        Vector3 angularVelocity = new Vector3();
        ENG_Float invMass = new ENG_Float();
        Vector3 angularFactor = new Vector3();
        Vector3 linearFactor = new Vector3();
        Vector3 gravity = new Vector3();
        Vector3 invInertiaLocal = new Vector3();
        Vector3 totalForce = new Vector3();
        Vector3 totalTorque = new Vector3();
        ENG_Float linearDamping = new ENG_Float();
        ENG_Float angularDamping = new ENG_Float();
        ENG_Float linearSleepingThreshold = new ENG_Float();
        ENG_Float angularSleepingThreshold = new ENG_Float();
        serializeRigidBody(rigidBody, invInertiaTensorWorld, linearVelocity, angularVelocity,
                invMass, angularFactor, linearFactor, gravity, invInertiaLocal, totalForce,
                totalTorque, linearDamping, angularDamping, linearSleepingThreshold,
                angularSleepingThreshold);

        System.out.println("invInertiaTensorWorld: " + invInertiaTensorWorld +
                "\nlinearVelocity: " + linearVelocity +
                "\nangularVelocity: " + angularVelocity +
                "\ninvMass: " + invMass +
                "\nangularFactor: " + angularFactor +
                "\nlinearFactor: " + linearFactor +
                "\ngravity: " + gravity +
                "\ninvInertiaLocal: " + invInertiaLocal +
                "\ntotalForce: " + totalForce +
                "\ntotalTorque: " + totalTorque +
                "\nlinearDamping: " + linearDamping +
                "\nangularDamping: " + angularDamping +
                "\nlinearSleepingThreshold: " + linearSleepingThreshold +
                "\nangularSleepingThreshold: " + angularSleepingThreshold);
        if (!totalForce.isZero()) {
            System.out.println("totalForce not zero");
        }
        if (!totalTorque.isZero()) {
            System.out.println("totalTorque not zero");
        }
    }

    public static void serializeRigidBody(btRigidBody rigidBody,
                                          Matrix3 invInertiaTensorWorld,
                                          Vector3 linearVelocity,
                                          Vector3 angularVelocity,
                                          ENG_Float invMass,
                                          Vector3 angularFactor,
                                          Vector3 linearFactor,
                                          Vector3 gravity,
                                          Vector3 invInertiaLocal,
                                          Vector3 totalForce,
                                          Vector3 totalTorque,
                                          ENG_Float linearDamping,
                                          ENG_Float angularDamping,
                                          ENG_Float linearSleepingThreshold,
                                          ENG_Float angularSleepingThreshold) {
        invInertiaTensorWorld.set(rigidBody.getInvInertiaTensorWorld());
        linearVelocity.set(rigidBody.getLinearVelocity());
        angularVelocity.set(rigidBody.getAngularVelocity());
        invMass.setValue(rigidBody.getInvMass());
        angularFactor.set(rigidBody.getAngularFactor());
        linearFactor.set(rigidBody.getLinearFactor());
        gravity.set(rigidBody.getGravity());
        invInertiaLocal.set(rigidBody.getInvInertiaDiagLocal());
        totalForce.set(rigidBody.getTotalForce());
        totalTorque.set(rigidBody.getTotalTorque());
        linearDamping.setValue(rigidBody.getLinearDamping());
        angularDamping.setValue(rigidBody.getAngularDamping());
        linearSleepingThreshold.setValue(rigidBody.getLinearSleepingThreshold());
        angularSleepingThreshold.setValue(rigidBody.getAngularSleepingThreshold());
    }

    public static void deserializeRigidBody(btRigidBody rigidBody,
                                            Matrix3 invInertiaTensorWorld,
                                            Vector3 linearVelocity,
                                            Vector3 angularVelocity,
                                            ENG_Float mass,
                                            Vector3 angularFactor,
                                            Vector3 linearFactor,
                                            Vector3 gravity,
                                            Vector3 inertiaLocal,
                                            Vector3 totalForce,
                                            Vector3 totalTorque,
                                            ENG_Float linearDamping,
                                            ENG_Float angularDamping,
                                            ENG_Float linearSleepingThreshold,
                                            ENG_Float angularSleepingThreshold) {
        rigidBody.setLinearVelocity(linearVelocity);
        rigidBody.setAngularVelocity(angularVelocity);
        rigidBody.setMassProps(mass.getValue(), inertiaLocal);
        rigidBody.setLinearFactor(linearFactor);
        rigidBody.setAngularFactor(angularFactor);
        rigidBody.setGravity(gravity);
        rigidBody.setDamping(linearDamping.getValue(), angularDamping.getValue());
        rigidBody.setSleepingThresholds(linearSleepingThreshold.getValue(), angularSleepingThreshold.getValue());
    }

    public static Vector3 invertVector3Safe(Vector3 v) {
        Vector3 ret = new Vector3();
        invertVector3Safe(v, ret);
        return ret;
    }

    /**
     * Does 1.0f / v checking if v isn't 0.
     * @param v
     */
    public static void invertVector3Safe(Vector3 v, Vector3 ret) {
        ret.x = v.x != 0.0f ? (1.0f / v.x) : 0.0f;
        ret.y = v.y != 0.0f ? (1.0f / v.y) : 0.0f;
        ret.z = v.z != 0.0f ? (1.0f / v.z) : 0.0f;
    }

    public static void convertVector(Vector2 v, ENG_Vector2D ret) {
        ret.set(v.x, v.y);
    }

    public static ENG_Vector2D convertVector(Vector2 v) {
        ENG_Vector2D ret = new ENG_Vector2D();
        convertVector(v, ret);
        return ret;
    }

    public static void convertVector(Vector3 v, ENG_Vector3D ret) {
        ret.set(v.x, v.y, v.z);
    }

    public static ENG_Vector3D convertVector3(Vector3 v) {
        ENG_Vector3D ret = new ENG_Vector3D();
        convertVector(v, ret);
        return ret;
    }

    public static void convertVector(Vector3 v, ENG_Vector4D ret) {
        ret.set(v.x, v.y, v.z);
    }

    public static ENG_Vector4D convertVector4(Vector3 v) {
        ENG_Vector4D ret = new ENG_Vector4D();
        convertVector(v, ret);
        return ret;
    }

    public static void convertVector(ENG_Vector2D v, Vector2 ret) {
        ret.set(v.x, v.y);
    }

    public static Vector2 convertVector(ENG_Vector2D v) {
        Vector2 ret = new Vector2();
        convertVector(v, ret);
        return ret;
    }

    public static void convertVector(ENG_Vector3D v, Vector3 ret) {
        ret.set(v.x, v.y, v.z);
    }

    public static Vector3 convertVector(ENG_Vector3D v) {
        Vector3 ret = new Vector3();
        convertVector(v, ret);
        return ret;
    }

    public static void convertVector(ENG_Vector4D v, Vector3 ret) {
        ret.set(v.x, v.y, v.z);
    }

    public static Vector3 convertVector(ENG_Vector4D v) {
        Vector3 ret = new Vector3();
        convertVector(v, ret);
        return ret;
    }

    public static void convertMatrix(Matrix3 m, ENG_Matrix3 ret) {
        System.arraycopy(m.val, 0, ret.get(), 0, 9);
    }

    public static ENG_Matrix3 convertMatrix(Matrix3 m) {
        ENG_Matrix3 ret = new ENG_Matrix3();
        convertMatrix(m, ret);
        return ret;
    }

    public static void convertMatrix(Matrix4 m, ENG_Matrix4 ret) {
        System.arraycopy(m.val, 0, ret.get(), 0, 16);
    }

    public static ENG_Matrix4 convertMatrix(Matrix4 m) {
        ENG_Matrix4 ret = new ENG_Matrix4();
        convertMatrix(m, ret);
        return ret;
    }

    public static void convertMatrix(ENG_Matrix3 m, Matrix3 ret) {
        System.arraycopy(m.get(), 0, ret.val, 0, 9);
    }

    public static Matrix3 convertMatrix(ENG_Matrix3 m) {
        Matrix3 ret = new Matrix3();
        convertMatrix(m, ret);
        return ret;
    }

    public static void convertMatrix(ENG_Matrix4 m, Matrix4 ret) {
        System.arraycopy(m.get(), 0, ret.val, 0, 16);
    }

    public static Matrix4 convertMatrix(ENG_Matrix4 m) {
        Matrix4 ret = new Matrix4();
        convertMatrix(m, ret);
        return ret;
    }

    public static void getAabb(String meshName, String groupName, ENG_Vector3D centre, ENG_Vector3D halfSize) {
        getAabb(ENG_RenderRoot.getRenderRoot().getPointer(), meshName, groupName, centre, halfSize);
    }

    public static void setGravity(btRigidBody rigidBody, ENG_Vector3D gravity) {
        rigidBody.setGravity(new Vector3(gravity.x, gravity.y, gravity.z));
    }

    public static void disposePhysicsObject(EntityProperties entityProperties) {
//        System.out.println("Disposing physics object: " + entityProperties.getUniqueName());
        removeRigidBody(MainApp.getGame().getBtDiscreteDynamicsWorld(), entityProperties.getRigidBody());
        dispose(entityProperties.getRigidBody());
        dispose(entityProperties.getMotionState());
        dispose(entityProperties.getContructionInfo());
        dispose(entityProperties.getCollisionShape());
    }

    public static void dispose(Disposable disposable) {
        disposable.dispose();
    }

    public static native void getAabb(long root, String meshName, String groupName, ENG_Vector3D centre, ENG_Vector3D halfSize);

}
