﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using MikuMikuPlugin;
using MikuMikuMoving.Plugin;
using System.Diagnostics;
using DxMath;

namespace MMM_GraphEditor
{
    public partial class GraphPort : UserControl
    {
        public GraphPort()
        {
            InitializeComponent();
        }

        private Point graph_move_start_pos = new Point(0, 0);
        private bool graph_move_mode = false;
        private bool key_move_mode = false;
        private Point key_move_start_pos = new Point(0, 0);
        public GraphPoint key_move_target;
        private GraphPanel _panel_inner;
        private GraphPanel _panel_outer;
        private GraphEditorControl _main_control;
        public int center_marker_padding = 36;
        private Quaternion key_move_orig_rot;

        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);
            float zoom_ratio = ((e.Delta / 120)*0.2f)+1.0f;
            int old_panel_y = -PanelInner.AutoScrollPosition.Y;
            this.Zoom(zoom_ratio);
            PanelInner.AutoScrollPosition = new Point(PanelInner.AutoScrollPosition.X, (int) (old_panel_y * zoom_ratio));
        }

        public void Zoom(float i) {
            int min = 3;
            int max = 800;
            this.graph_scale *= i;
            if (this.graph_scale > max)
                this.graph_scale = max;
            if (this.graph_scale < min)
                this.graph_scale = min;
            this.Refresh();
        }

        protected GraphPanel PanelInner
        {
            get
            {
                if (this._panel_inner != null)
                    return _panel_inner;
                return this._panel_inner = (GraphPanel) Parent;
            }
        }

        protected GraphPanel PanelOuter
        {
            get
            {
                if (this._panel_outer != null)
                    return _panel_outer;
                if (PanelInner == null)
                    return null;
                return this._panel_outer = (GraphPanel)PanelInner.Parent;
            }
        }

        protected GraphEditorControl MainControl
        {
            get
            {
                if (this._main_control != null)
                    return _main_control;
                if (PanelOuter == null)
                    return null;
                return this._main_control = (GraphEditorControl)PanelOuter.Parent;
            }
        }


        protected FrameRuler frameRuler()
        {
            if (MainControl == null) return null;
            return (FrameRuler)MainControl.frameRuler1;
        }

        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);
            fitSize();
        }

        public void fitSize()
        {
            try
            {
                PanelInner.Width = Width + System.Windows.Forms.SystemInformation.VerticalScrollBarWidth;
                PanelInner.Height = PanelOuter.Height - 30 - System.Windows.Forms.SystemInformation.HorizontalScrollBarHeight;
                frameRuler().Width = Width;
            }
            catch (System.Exception)
            {
                // ignore
            }
        }

        private void mouse_down(object sender, MouseEventArgs e)
        {
            if (e.Button == System.Windows.Forms.MouseButtons.Right)
            {
                this.graph_move_mode = true;
                this.graph_move_start_pos = e.Location;
            }
            if (e.Button == System.Windows.Forms.MouseButtons.Left)
            {
                this.key_move_mode = true;
                this.key_move_start_pos = e.Location;
                this.key_move_target = key_pointed;
                if (key_move_target != null)
                    this.key_move_orig_rot = key_pointed.Frame.Quaternion;
            }
        }

        private void mouse_up(object sender, MouseEventArgs e)
        {
            if (e.Button == System.Windows.Forms.MouseButtons.Right)
            {
                this.graph_move_mode = false;
            }
            if (e.Button == System.Windows.Forms.MouseButtons.Left)
            {
                this.key_move_mode = false;
                this.key_move_target = null;
            }
        }

        private void mouse_move(object sender, MouseEventArgs e)
        {
            this.cur_mouse_pos = e.Location;
            if (graph_move_mode)
            {
                PanelInner.AutoScrollPosition = new Point(
                    -PanelInner.AutoScrollPosition.X,
                    -PanelInner.AutoScrollPosition.Y - e.Location.Y + graph_move_start_pos.Y);
                PanelOuter.AutoScrollPosition = new Point(
                    -PanelOuter.AutoScrollPosition.X - e.Location.X + graph_move_start_pos.X,
                    -PanelOuter.AutoScrollPosition.Y);
            }
            if (key_move_mode && key_move_target != null)
            {
                CMotionFrameData o_mfd = key_move_target.Frame;
                Vector3 pos = o_mfd.Position;
                Quaternion rot = o_mfd.Quaternion;
                bool is_rot_modified = false;
                Vector3 rot_axis = new Vector3(0, 0, 0);

                if (scene.MarkerPosition == o_mfd.FrameNumber)
                {
                    pos = target_layer.CurrentLocalMotion.Move;
                    rot = target_layer.CurrentLocalMotion.Rotation;
                }

                switch (key_move_target.Type)
                {
                    case IDX_POS_X:
                        pos.X = pos_val_from_graph_Y(e.Location.Y);
                        break;
                    case IDX_POS_Y:
                        pos.Y = pos_val_from_graph_Y(e.Location.Y);
                        break;
                    case IDX_POS_Z:
                        pos.Z = pos_val_from_graph_Y(e.Location.Y);
                        break;
                    case IDX_ROT_R:
                        is_rot_modified = true;
                        rot_axis = target_bone.LocalAxisX;
                        break;
                    case IDX_ROT_P:
                        is_rot_modified = true;
                        rot_axis = target_bone.LocalAxisY;
                        break;
                    case IDX_ROT_Y:
                        is_rot_modified = true;
                        rot_axis = target_bone.LocalAxisZ;
                        break;
                    case IDX_POS_IXA:
                        o_mfd.InterpolXA = ip_val(e.Location, 0, o_mfd, key_move_target.PrevFrame);
                        break;
                    case IDX_POS_IXB:
                        o_mfd.InterpolXB = ip_val(e.Location, 0, o_mfd, key_move_target.PrevFrame);
                        break;
                    case IDX_POS_IYA:
                        o_mfd.InterpolYA = ip_val(e.Location, 1, o_mfd, key_move_target.PrevFrame);
                        break;
                    case IDX_POS_IYB:
                        o_mfd.InterpolYB = ip_val(e.Location, 1, o_mfd, key_move_target.PrevFrame);
                        break;
                    case IDX_POS_IZA:
                        o_mfd.InterpolZA = ip_val(e.Location, 2, o_mfd, key_move_target.PrevFrame);
                        break;
                    case IDX_POS_IZB:
                        o_mfd.InterpolZB = ip_val(e.Location, 2, o_mfd, key_move_target.PrevFrame);
                        break;
                }

                if (is_rot_modified)
                {
                    float angle = (rot_val_from_graph_Y(key_move_start_pos.Y) - rot_val_from_graph_Y(e.Location.Y));
                    rot = Quaternion.Multiply(key_move_orig_rot, Quaternion.Invert(Quaternion.Normalize(Quaternion.RotationAxis(rot_axis, angle))));
                }

                MotionFrameData mfd = new MotionFrameData(o_mfd.FrameNumber, pos, rot);
                mfd.InterpolRA = o_mfd.InterpolRA;
                mfd.InterpolRB = o_mfd.InterpolRB;
                mfd.InterpolXA = o_mfd.InterpolXA;
                mfd.InterpolXB = o_mfd.InterpolXB;
                mfd.InterpolYA = o_mfd.InterpolYA;
                mfd.InterpolYB = o_mfd.InterpolYB;
                mfd.InterpolZA = o_mfd.InterpolZA;
                mfd.InterpolZB = o_mfd.InterpolZB;
                target_layer.Frames.AddKeyFrame(mfd);

                // TODO: reset current motion always!!!
                if (scene.MarkerPosition == o_mfd.FrameNumber)
                {
                    target_layer.CurrentLocalMotion = new MotionData(pos, rot);
                }
            }
        }


        public void key_down(object sender, PreviewKeyDownEventArgs e)
        {
            if (scene == null) return;
            switch (e.KeyCode)
            {
                case Keys.Right:
                    scene.MarkerPosition += 1;
                    centerMarker();
                    break;
                case Keys.Left:
                    scene.MarkerPosition -= 1;
                    centerMarker();
                    break;
                case Keys.C:
                    centerMarker();
                    break;
                case Keys.Enter:
                    if (target_bone == null || target_layer == null)  break;
                    MotionData cur_motion = target_layer.CurrentLocalMotion;
                    MotionFrameData fd = new MotionFrameData(scene.MarkerPosition, cur_motion.Move, cur_motion.Rotation);
                    foreach (MotionFrameData orig_fd in target_layer.Frames)
                    {
                        if (orig_fd.FrameNumber != scene.MarkerPosition) continue;
                        fd.InterpolRA = orig_fd.InterpolRA;
                        fd.InterpolRB = orig_fd.InterpolRB;
                        fd.InterpolXA = orig_fd.InterpolXA;
                        fd.InterpolXB = orig_fd.InterpolXB;
                        fd.InterpolYA = orig_fd.InterpolYA;
                        fd.InterpolYB = orig_fd.InterpolYB;
                        fd.InterpolZA = orig_fd.InterpolZA;
                        fd.InterpolZB = orig_fd.InterpolZB;
                        break;
                    }
                    target_layer.Frames.AddKeyFrame(fd);
                    break;
                default:
                    System.Diagnostics.Trace.WriteLine("Unknown key event: " + e.KeyCode);
                    break;
            }
        }

        public void centerMarkerForce()
        {
            centerMarkerHorizonal();
            centerMarkerVirtical();
        }

        public void centerMarker()
        {
            if (MainControl == null) return;
            if (config.track_marker)
                centerMarkerHorizonal();
            if (config.track_yzero)
                centerMarkerVirtical();
        }

        public void centerMarkerHorizonal()
        {
            if (scene == null) return;
            if (PanelOuter == null) return;

            int cur = -PanelOuter.AutoScrollPosition.X;
            if (scene.MarkerPosition * frame_width - center_marker_padding < cur)
            {
                PanelOuter.AutoScrollPosition = new Point((int)(scene.MarkerPosition * frame_width - center_marker_padding), 0);
                return;
            }
            if (scene.MarkerPosition * frame_width + center_marker_padding > cur + PanelOuter.Width)
            {
                PanelOuter.AutoScrollPosition = new Point((int)(scene.MarkerPosition * frame_width - PanelOuter.Width + center_marker_padding), 0);
                return;
            }
        }

        public void centerMarkerVirtical()
        {
            if (scene == null) return;
            if (PanelInner == null) return;
            PanelInner.AutoScrollPosition = new Point(0, (int)(graph_Y(0) - PanelInner.Height / 2));
        }

        private void key_down(object sender, KeyEventArgs e)
        {

        }
    }
}
