﻿using System.Drawing;
using System.Diagnostics;
using MikuMikuMoving;
using MikuMikuMoving.Plugin;
using DxMath;
using System.Collections.Generic;
using System.Windows.Forms;
using System;

namespace MMM_GraphEditor
{
    partial class GraphPort
    {
        /// <summary> 
        /// 必要なデザイナー変数です。
        /// </summary>
        private System.ComponentModel.IContainer components = null;

        private MikuMikuPlugin.Scene scene;
        public MikuMikuPlugin.Scene Scene {
            get { return this.scene; }
            set { this.scene = value; updateTarget(); }
        }
        public int graph_padding = 30;
        public  int frame_width = 12;
        public float graph_height = 20f;
        public float graph_max = 0f;
        public float graph_min = 0f;
        public float graph_scale = 10.0f;
        public float rot_scale = 1f;
        public long max_frame = 10;
        public static float Graph_PenSize = 2f;
        public static Pen Ip_Pen = Pens.Black;
        public float ip_mark_size = 4f;
        public int default_rot_type = 0;
        public static Color[] Graph_Colors = { Color.Red, Color.Green, Color.Blue, Color.DeepPink, Color.YellowGreen, Color.DodgerBlue };
        public static Brush[] Graph_Brushes = { Brushes.Red, Brushes.Green, Brushes.Blue,
                                                   Brushes.DeepPink, Brushes.YellowGreen, Brushes.DodgerBlue };
        public static Pen[] Graph_Pens = {
                                            new Pen(Graph_Brushes[0], Graph_PenSize),
                                            new Pen(Graph_Brushes[1], Graph_PenSize),
                                            new Pen(Graph_Brushes[2], Graph_PenSize),
                                            new Pen(Graph_Brushes[3], Graph_PenSize),
                                            new Pen(Graph_Brushes[4], Graph_PenSize),
                                            new Pen(Graph_Brushes[5], Graph_PenSize),
                                        };
        public static IEnumerable<int> XYZ = new int[3] { 0, 1, 2 };
        public static IEnumerable<int> RPY = new int[3] { 3, 4, 5 };
        public static IEnumerable<int> XYZRPY = new int[6] { 0, 1, 2, 3, 4, 5 };
        public MotionLayer target_layer = null;
        public Bone target_bone = null;
        public bool[] render_pos = { true, true, true };
        public bool[] render_rot = { true, true, true };

        /// <summary> 
        /// 使用中のリソースをすべてクリーンアップします。
        /// </summary>
        /// <param name="disposing">マネージ リソースが破棄される場合 true、破棄されない場合は false です。</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing && (components != null))
            {
                components.Dispose();
                this.scene = null;
                this.target_layer = null;
                this.target_bone = null;
            }
            base.Dispose(disposing);
        }

        protected override void OnPaint(System.Windows.Forms.PaintEventArgs e)
        {
            Trace.WriteLine("GraphPane: OnPaint");

            base.OnPaint(e);
            Graphics g = e.Graphics;
            renderGrid(g);
            renderGraph(g);
        }

        private void renderGrid(Graphics g) {
            MotionLayer layer = target_layer;
            if (layer != null && layer.Frames.Count > 0)
            {
                long fnum = layer.Frames[layer.Frames.Count - 1].FrameNumber;
                Trace.WriteLine("Max frame number: " + fnum.ToString());
                if (fnum > this.max_frame)
                    this.max_frame = fnum;
            }

            autoSetHeight();

            int width = (int) (this.max_frame +1) * frame_width -1;
            this.Width = width;

            g.FillRectangle(Brushes.LightGray, 0, 0, width, Height);
            
            int cur;
            bool has_label = false;
            bool is_em_tic = false;
            for (cur = 0; cur * frame_width < width; cur++) {
                has_label = (cur > 0 && cur % 10 == 0);
                is_em_tic = (cur % 5 == 0);
                g.DrawLine(scene != null && cur == scene.MarkerPosition ? Pens.YellowGreen :
                    has_label ? Pens.White : 
                    is_em_tic ? Pens.WhiteSmoke : Pens.Gainsboro,
                    cur * frame_width, 0, cur * frame_width, Height);
            }
            g.DrawLine(Pens.WhiteSmoke, 0, graph_Y(0f), width, graph_Y(0f));
        }

        private void renderGraph(Graphics g)
        {
            if (target_layer == null) return;
            
            CMotionFrameData prev = null;
            foreach (CMotionFrameData fd in target_layer.Frames)
            {
                Trace.WriteLine("Frame: " + fd.FrameNumber.ToString() +
                    " X = " + fd.Position.X.ToString() +
                    " Y = " + fd.Position.Y.ToString() +
                    " Z = " + fd.Position.Z.ToString());
                Trace.WriteLine("Frame: " + fd.FrameNumber.ToString() +
                    " RX = " + fd.Quaternion.X.ToString() +
                    " RY = " + fd.Quaternion.Y.ToString() +
                    " RZ = " + fd.Quaternion.Z.ToString() +
                    " RAngle = " + fd.Quaternion.Angle.ToString() 
                    );
                plotKeyFrame(g, fd, prev);
                prev = fd;
            }
        }

        private float graph_Y(float y) {
            return graph_height + graph_padding - (y - graph_min) * graph_scale;
        }

        private float graph_R(float y)
        {
            return graph_Y(y*rot_scale);
        }

        private void plotKeyFrame(Graphics g, CMotionFrameData fd, CMotionFrameData prev)
        {
            try
            {
                GraphPanel panel_outer = (GraphPanel)this.Parent.Parent;
                int cur = -panel_outer.AutoScrollPosition.X;
                if (fd.FrameNumber * frame_width < cur || prev != null && prev.FrameNumber * frame_width > cur + panel_outer.Width)
                    return; // skip rotation plotting outside view port
            }
            catch (System.Exception)
            {
                // ignore
            }

            int size = 8;
            float cur_x = fd.FrameNumber * frame_width;

            if (target_bone.BoneFlags.HasFlag(MikuMikuPlugin.BoneType.XYZ) &&
                (render_pos[0] || render_pos[1] || render_pos[2]))
            {
                foreach (int i in XYZ)
                {
                    if (!render_pos[i]) continue;
                    Vector3 pos = scene.MarkerPosition == fd.FrameNumber ? target_bone.CurrentLocalMotion.Move : fd.Position;
                    g.FillRectangle(Graph_Brushes[i], cur_x - (size / 2), graph_Y(pos[i]) - (size / 2), size, size);
                }
                drawInterpolateLine_Pos(g, prev, fd);
            }


            if (target_bone.BoneFlags.HasFlag(MikuMikuPlugin.BoneType.Rotate) &&
                (render_rot[0] || render_rot[1] || render_rot[2]))
            {
                plotRot(g, fd, prev);
            }
        }

        private void plotRot(Graphics g, CMotionFrameData fd, CMotionFrameData prev)
        {
            int size = 8;
            float cur_x = fd.FrameNumber * frame_width;

            Quaternion rot = scene.MarkerPosition == fd.FrameNumber ? target_bone.CurrentLocalMotion.Rotation : fd.Quaternion;
            Quaternion? prev_rot = null;
            if (prev != null) prev_rot = prev.Quaternion;
            int rot_type = getRotType(prev_rot, rot);
            Vector3 erot = Quaternion2Euler(rot, rot_type);
            Trace.WriteLine("Euler Rot: " + erot.ToString());
            foreach (int i in RPY)
            {
                int a = i - 3;
                if (!render_rot[a]) continue;
                g.FillEllipse(Graph_Brushes[i], cur_x - (size / 2), graph_R(erot[a]) - (size / 2), size, size);
            }
            drawInterpolateLine_Rot(g, prev, fd, rot_type);
        }

        private void drawInterpolateLine_Pos(Graphics g, CMotionFrameData from, CMotionFrameData to)
        {
            if (from == null) return;
            if (!(target_bone.BoneFlags.HasFlag(MikuMikuPlugin.BoneType.XYZ) &&
                (render_pos[0] || render_pos[1] || render_pos[2])))
                return;
            MikuMikuPlugin.InterpolatePoint[] interpolations = { to.InterpolXA, to.InterpolXB, to.InterpolYA, to.InterpolYB, to.InterpolZA, to.InterpolZB };

            float span_x = (to.FrameNumber - from.FrameNumber) * frame_width;
            float from_x = from.FrameNumber * frame_width;
            float to_x = to.FrameNumber * frame_width;

            foreach (int i in XYZ)
            {
                if (!render_pos[i]) continue;
                Vector3 pos_from = scene.MarkerPosition == from.FrameNumber ? target_bone.CurrentLocalMotion.Move : from.Position;
                Vector3 pos_to = scene.MarkerPosition == to.FrameNumber ? target_bone.CurrentLocalMotion.Move : to.Position;
                float span_y = graph_Y(pos_to[i]) - graph_Y(pos_from[i]);
                float from_y = graph_Y(pos_from[i]);
                float ip_ax  = from_x + (interpolations[i*2].X / 128f * span_x);
                float ip_ay  = from_y + (interpolations[i * 2].Y / 128f * span_y);
                float ip_bx  = from_x + (interpolations[i*2+1].X / 128f * span_x);
                float ip_by  = from_y + (interpolations[i*2+1].Y / 128f * span_y);
                float to_y   = graph_Y(pos_to[i]);
                g.DrawLine(Ip_Pen, from_x, from_y, ip_ax, ip_ay);
                g.DrawLine(Ip_Pen, ip_bx, ip_by, to_x, to_y);
                g.DrawRectangle(Ip_Pen, ip_ax - 1f, ip_ay - ip_mark_size/2, ip_mark_size, ip_mark_size);
                g.DrawRectangle(Ip_Pen, ip_bx - 1, ip_by - ip_mark_size/2, ip_mark_size, ip_mark_size);
                g.DrawBezier(Graph_Pens[i], from_x, from_y, ip_ax, ip_ay, ip_bx, ip_by, to_x, to_y);
            }
        }

        private void drawInterpolateLine_Rot(Graphics g, CMotionFrameData from, CMotionFrameData to, int rot_type)
        {
            if (from == null) return;
            if (!(target_bone.BoneFlags.HasFlag(MikuMikuPlugin.BoneType.Rotate) &&
                (render_rot[0] || render_rot[1] || render_rot[2])))
                return;

            Quaternion rot_from = scene.MarkerPosition == from.FrameNumber ? target_bone.CurrentLocalMotion.Rotation : from.Quaternion;
            Quaternion rot_to = scene.MarkerPosition == to.FrameNumber ? target_bone.CurrentLocalMotion.Rotation : to.Quaternion;
            Vector3 erot_from = Quaternion2Euler(rot_from, rot_type);
            float from_x = from.FrameNumber * frame_width;
            float span_x = (to.FrameNumber - from.FrameNumber) * frame_width;
            float to_x = to.FrameNumber * frame_width;

            g.DrawLine(Pens.Gray, from_x, graph_R((float)Math.PI), to_x, graph_R((float)Math.PI));
            g.DrawLine(Pens.Gray, from_x, graph_R((float)-Math.PI), to_x, graph_R((float)-Math.PI));
            g.DrawString(this.rotTypeName(rot_type), this.Font, Brushes.Gray, from_x, graph_R(0.2f));


            float t = 0;
            float tx = 0;
            float amount = 0;
            for (float i = 0; i < span_x; i++)
            {
                while (tx < i && t < 1)
                {
                    t += 0.01f;
                    tx = (t * t * t + 3 * t * t * (1 - t) * to.InterpolRB.X / 128f + 3 * t * (1 - t) * (1 - t) * to.InterpolRA.X / 128f) * span_x;
                    amount = (t * t * t + 3 * t * t * (1 - t) * to.InterpolRB.Y / 128f + 3 * t * (1 - t) * (1 - t) * to.InterpolRA.Y / 128f);
                }
                Quaternion mid_rq = DxMath.Quaternion.Slerp(rot_from, rot_to, amount);  // TODO: slow! move to outside of axis loop.
                Vector3 mid_erot = Quaternion2Euler(mid_rq, rot_type);

                foreach (int ax in RPY)
                {
                    int a = ax - 3;
                    if (!render_rot[a]) continue;
                    g.FillRectangle(Graph_Brushes[ax], from_x + i, graph_R(mid_erot[a]), 1, 2);
                }
            }
        }


        private void updateTarget()
        {
            MotionLayer layer = null;
            Bone bone = null;
            if (this.scene == null || this.scene.ActiveModel == null)
            {
                this.target_bone = null;
                this.target_layer = null;
                return;
            }

            foreach (Bone b in scene.ActiveModel.Bones)
            {
                foreach (MotionLayer l in b.SelectedLayers)
                {
                    bone = b;
                    layer = l;
                }
            }
            if (layer != null)
            {
                this.target_layer = layer;
                this.target_bone = bone;
            }
            else
            {
                this.target_bone = null;
                this.target_layer = null;
            }
        }

        private void autoSetGraphRange() {
            MotionLayer layer = target_layer;
            if (layer == null) return;
            float max =  3.2f * rot_scale;
            float min = -3.2f * rot_scale;
            bool check_cur_move = false;
            foreach (CMotionFrameData fd in layer.Frames)
            {
                if (fd.FrameNumber == this.scene.MarkerPosition)
                    check_cur_move = true;
                foreach (int i in XYZ)
                {
                    if (max < fd.Position[i])
                        max = fd.Position[i];
                    if (min > fd.Position[i])
                        min = fd.Position[i];
                }
            }
            if (check_cur_move)
            {
                foreach (int i in XYZ)
                {
                    float cur = this.target_bone.CurrentLocalMotion.Move[i];
                    if (max < cur)
                        max = cur;
                    if (min > cur)
                        min = cur;
                }
            }
            this.graph_height = (max - min) * graph_scale;
            this.graph_min = min;
            this.graph_max = max;
        }

        private void autoSetHeight()
        {
            autoSetGraphRange();
            this.Height = (int) graph_height + graph_padding * 2;
        }

        private int getMinDeltaAxis(Quaternion from, Quaternion to)
        {
            float span_x = Math.Abs(from.X - to.X);
            float span_y = Math.Abs(from.Y - to.Y);
            float span_z = Math.Abs(from.Z - to.Z);

            if (span_x <= span_y && span_x <= span_z)
                return 0;
            if (span_y <= span_x && span_y <= span_z)
                return 1;
            if (span_z <= span_x && span_z <= span_y)
                return 2;
            return 0;
        }

        private Vector3 Quaternion2Euler(Quaternion q, int min_axis)
        {
            Matrix rm = Quaternion2RotMatrix(q);
            double r, p, y;
            if (min_axis == 0) // Z-X-Y
            {
                r = -Math.Asin(rm.M32);
                p = Math.Atan2(rm.M31, rm.M33);
                y = Math.Atan2(rm.M12, rm.M22);
            }
            else if (min_axis == 1) // Z-Y-X
            {
                r = Math.Atan2(-rm.M32, rm.M33);
                p = Math.Asin(rm.M31);
                y = Math.Atan2(-rm.M21, rm.M11);
            }
            else // X-Z-Y
            {
                r = Math.Atan2(-rm.M32, rm.M22);
                p = Math.Atan2(-rm.M13, rm.M11);
                y = Math.Asin(rm.M12);
            }
            return new Vector3((float)r, (float)p, (float)y);
        }

        private Matrix Quaternion2RotMatrix(Quaternion q)
        {
            Matrix m = new Matrix();
            m.M11 = (float) (1 - 2 * Math.Pow(q.Y, 2) - 2 * Math.Pow(q.Z, 2));
            m.M12 = 2 * q.X * q.Y - 2 * q.W * q.Z;
            m.M13 = 2 * q.X * q.Z + 2 * q.W * q.Y;
            m.M14 = 0;
            m.M21 = 2 * q.X * q.Y + 2 * q.W * q.Z;
            m.M22 = (float) (1 - 2 * Math.Pow(q.X, 2) - 2 * Math.Pow(q.Z, 2));
            m.M23 = 2 * q.Y * q.Z - 2 * q.W * q.X;
            m.M24 = 0;
            m.M31 = 2 * q.X * q.Z - 2 * q.W * q.Y;
            m.M32 = 2 * q.Y * q.Z + 2 * q.W * q.X;
            m.M33 = (float) (1 - 2 * Math.Pow(q.X, 2) - 2 * Math.Pow(q.Y, 2));
            m.M34 = 0;
            m.M41 = 0;
            m.M42 = 0;
            m.M43 = 0;
            m.M44 = 1;
            return m;
        }

        public int getRotType(Quaternion? from, Quaternion to)
        {
            if (from == null && default_rot_type == -1)
                return 0;
            if (from == null)
                return default_rot_type;
            if (default_rot_type != -1)
                return default_rot_type;
            return getMinDeltaAxis((Quaternion)from, to);
        }

        public String rotTypeName(int rot_type)
        {
            switch (rot_type)
            {
                case 0:
                    return "Z-X-Y";
                case 1:
                    return "Z-Y-X";
                case 2:
                    return "X-Z-Y";
            }
            return "Unknown";
        }

        #region コンポーネント デザイナーで生成されたコード

        /// <summary> 
        /// デザイナー サポートに必要なメソッドです。このメソッドの内容を 
        /// コード エディターで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.SuspendLayout();
            // 
            // GraphPort
            // 
            this.AutoScaleDimensions = new System.Drawing.SizeF(6F, 12F);
            this.AutoScaleMode = System.Windows.Forms.AutoScaleMode.Font;
            this.CausesValidation = false;
            this.DoubleBuffered = true;
            this.Name = "GraphPort";
            this.Size = new System.Drawing.Size(100, 100);
            this.MouseDown += new System.Windows.Forms.MouseEventHandler(this.mouse_down);
            this.MouseMove += new System.Windows.Forms.MouseEventHandler(this.mouse_move);
            this.MouseUp += new System.Windows.Forms.MouseEventHandler(this.mouse_up);
            this.PreviewKeyDown += new System.Windows.Forms.PreviewKeyDownEventHandler(this.key_down);
            this.ResumeLayout(false);

        }

        #endregion

    }
}
