/*
 * Key parser
 *
 * Copyright (C) 2009 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 */

#include <stdio.h>
#include <stdlib.h>        /* strtol */
#include <string.h>
#include <ctype.h>

#include "dlib/dlib.h"
#include "keys.hh"
#include "utf8.hh"
#include "msg.h"

/*
 *  Local data types
 */
typedef struct {
     const char *name;
     KeysCommand_t cmd;
     int modifier, key;
} KeyBinding_t;

typedef struct {
     const char *name;
     const int value;
} Mapping_t;


/*
 *  Local data
 */
static const Mapping_t keyNames[] = {
     { "Backspace",   WXK_BACK     },
     { "Delete",      WXK_DELETE   },
     { "Down",        WXK_DOWN     },
     { "End",         WXK_END      },
     { "Esc",         WXK_ESCAPE   },
     { "F1",          WXK_F1       },
     { "F2",          WXK_F2       },
     { "F3",          WXK_F3       },
     { "F4",          WXK_F4       },
     { "F5",          WXK_F5       },
     { "F6",          WXK_F6       },
     { "F7",          WXK_F7       },
     { "F8",          WXK_F8       },
     { "F9",          WXK_F9       },
     { "F10",         WXK_F10      },
     { "F11",         WXK_F11      },
     { "F12",         WXK_F12      },
     { "Home",        WXK_HOME     },
     { "Insert",      WXK_INSERT   },
     { "Left",        WXK_LEFT     },
     { "PageDown",    WXK_PAGEDOWN },
     { "PageUp",      WXK_PAGEUP   },
     { "Print",       WXK_PRINT    },
     { "Return",      WXK_RETURN   },
     { "Right",       WXK_RIGHT    },
     { "Space",       WXK_SPACE    },
     { "Tab",         WXK_TAB      },
     { "Up",          WXK_UP       },
     /* multimedia keys */
     { "Back",        WXK_SPECIAL1  },
     { "Favorites",   WXK_SPECIAL2  },
     { "Forward",     WXK_SPECIAL3  },
     { "HomePage",    WXK_SPECIAL4  },
     { "Mail",        WXK_SPECIAL5  },
     { "MediaNext",   WXK_SPECIAL6  },
     { "MediaPlay",   WXK_SPECIAL7  },
     { "MediaPrev",   WXK_SPECIAL8  },
     { "MediaStop",   WXK_SPECIAL9  },
     { "Refresh",     WXK_SPECIAL10 },
     { "Search",      WXK_SPECIAL11 },
     { "Sleep",       WXK_SPECIAL12 },
     { "Stop",        WXK_SPECIAL13 },
     { "VolumeDown",  WXK_SPECIAL14 },
     { "VolumeMute",  WXK_SPECIAL15 },
     { "VolumeUp",    WXK_SPECIAL16 },
};

static const Mapping_t modifierNames[] = {
     { "Shift",   WXK_SHIFT  },
     { "Ctrl",    WXK_CONTROL},
     { "Alt",     WXK_ALT    },
     { "Meta",    WXK_ESCAPE },
     { "Button1", WXK_LBUTTON},
     { "Button2", WXK_RBUTTON},
     { "Button3", WXK_MBUTTON}
};

static const KeyBinding_t default_keys[] = {
     { "nop"          , KEYS_NOP          , 0            , 0               },
     { "open"         , KEYS_OPEN         , WXK_CONTROL  , 'o'             },
     { "new-window"   , KEYS_NEW_WINDOW   , WXK_CONTROL  , 'n'             },
     { "new-tab"      , KEYS_NEW_TAB      , WXK_CONTROL  , 't'             },
     { "left-tab"     , KEYS_LEFT_TAB     , WXK_CONTROL|
       WXK_SHIFT    , WXK_TAB         },
     { "left-tab"     , KEYS_LEFT_TAB     , WXK_CONTROL  , WXK_PAGEUP      },
     { "right-tab"    , KEYS_RIGHT_TAB    , WXK_CONTROL  , WXK_TAB         },
     { "right-tab"    , KEYS_RIGHT_TAB    , WXK_CONTROL  , WXK_PAGEDOWN    },
     { "close-tab"    , KEYS_CLOSE_TAB    , WXK_CONTROL  , 'w'             },
     { "find"         , KEYS_FIND         , WXK_CONTROL  , 'f'             },
     { "websearch"    , KEYS_WEBSEARCH    , WXK_CONTROL  , 's'             },
     { "bookmarks"    , KEYS_BOOKMARKS    , WXK_CONTROL  , 'b'             },
     { "reload"       , KEYS_RELOAD       , WXK_CONTROL  , 'r'             },
     { "stop"         , KEYS_STOP         , 0            , 0               },
     { "save"         , KEYS_SAVE         , 0            , 0               },
     { "hide-panels"  , KEYS_HIDE_PANELS  , 0            , WXK_ESCAPE      },
     { "file-menu"    , KEYS_FILE_MENU    , WXK_ALT      , 'f'             },
     { "close-all"    , KEYS_CLOSE_ALL    , WXK_CONTROL  , 'q'             },
     { "back"         , KEYS_BACK         , 0            , WXK_BACK        },
     { "back"         , KEYS_BACK         , 0            , ','             },
     { "forward"      , KEYS_FORWARD      , WXK_SHIFT    , WXK_BACK        },
     { "forward"      , KEYS_FORWARD      , 0            , '.'             },
     { "goto"         , KEYS_GOTO         , WXK_CONTROL  , 'l'             },
     { "home"         , KEYS_HOME         , WXK_CONTROL  , 'h'             },
     { "view-source"  , KEYS_VIEW_SOURCE  , WXK_CONTROL  , 'u'             },
     { "screen-up"    , KEYS_SCREEN_UP    , 0            , WXK_PAGEUP      },
     { "screen-up"    , KEYS_SCREEN_UP    , 0            , 'b'             },
     { "screen-down"  , KEYS_SCREEN_DOWN  , 0            , WXK_PAGEDOWN    },
     { "screen-down"  , KEYS_SCREEN_DOWN  , 0            , ' '             },
     { "screen-left"  , KEYS_SCREEN_LEFT  , 0            , 0               },
     { "screen-right" , KEYS_SCREEN_RIGHT , 0            , 0               },
     { "line-up"      , KEYS_LINE_UP      , 0            , WXK_UP          },
     { "line-down"    , KEYS_LINE_DOWN    , 0            , WXK_DOWN        },
     { "left"         , KEYS_LEFT         , 0            , WXK_LEFT        },
     { "right"        , KEYS_RIGHT        , 0            , WXK_RIGHT       },
     { "top"          , KEYS_TOP          , 0            , WXK_HOME        },
     { "bottom"       , KEYS_BOTTOM       , 0            , WXK_END         },
};

static Dlist *bindings;



/*
 * Initialize the bindings list
 */
void Keys::init()
{
     KeyBinding_t *node;

     // Fill our key bindings list
     bindings = dList_new(32);
     for (uint_t i = 0; i < sizeof(default_keys) / sizeof(default_keys[0]); i++) {
	  if (default_keys[i].key) {
	       node = dNew(KeyBinding_t, 1);
	       node->name = dStrdup(default_keys[i].name);
	       node->cmd = default_keys[i].cmd;
	       node->modifier = default_keys[i].modifier;
	       node->key = default_keys[i].key;
	       dList_insert_sorted(bindings, node, nodeByKeyCmp);
	  }
     }
}

/*
 * Free data
 */
void Keys::free()
{
     KeyBinding_t *node;

     while ((node = (KeyBinding_t*)dList_nth_data(bindings, 0))) {
	  dFree((char*)node->name);
	  dList_remove_fast(bindings, node);
	  dFree(node);
     }
     dList_free(bindings);
}

/*
 * Compare function by {key,modifier} pairs.
 */
int Keys::nodeByKeyCmp(const void *node, const void *key)
{
     KeyBinding_t *n = (KeyBinding_t*)node, *k = (KeyBinding_t*)key;
     _MSG("Keys::nodeByKeyCmp modifier=%d\n", k->modifier);
     return (n->key != k->key) ? (n->key - k->key) : (n->modifier - k->modifier);
}

/*
 * Look if the just pressed key is bound to a command.
 * Return value: The command if found, KEYS_NOP otherwise.
 */
KeysCommand_t Keys::getKeyCmd(wxFLKeyEvent& event)
{
     KeysCommand_t ret = KEYS_NOP;
     KeyBinding_t keyNode;

     keyNode.modifier = event.GetKeyCode() & (WXK_SHIFT | WXK_CONTROL | WXK_ALT | WXK_ESCAPE);

     if (iscntrl(event.event_text()[0])) {
	  keyNode.key = event.GetKeyCode();
     } else {
	  const char *beyond = event.event_text() + event.event_length();
	  keyNode.key = a_Utf8_decode(event.event_text(), beyond, NULL);

	  /* BUG: The idea is to drop the modifiers if their use results in a
	   * different character (e.g., if shift-8 gives '*', drop the shift,
	   * but if ctrl-6 gives '6', keep the ctrl), but we have to compare a
	   * keysym with a Unicode codepoint, which only works for characters
	   * below U+0100 (those known to latin-1).
	   */
	  if (keyNode.key != event.GetKeyCode())
	       keyNode.modifier = 0;
     }
     _MSG("getKeyCmd: evkey=0x%x evtext=\'%s\' key=0x%x, mod=0x%x\n",
	  event.GetKeyCode(), event.event_text(), keyNode.key, keyNode.modifier);
     void *data = dList_find_sorted(bindings, &keyNode, nodeByKeyCmp);
     if (data)
	  ret = ((KeyBinding_t*)data)->cmd;
     return ret;
}

/*
 * Remove a key binding from the table.
 */
void Keys::delKeyCmd(int key, int mod)
{
     KeyBinding_t keyNode, *node;
     keyNode.key = key;
     keyNode.modifier = mod;

     node = (KeyBinding_t*) dList_find_sorted(bindings, &keyNode, nodeByKeyCmp);
     if (node) {
	  dList_remove(bindings, node);
	  dFree((char*)node->name);
	  dFree(node);
     }
}

/*
 * Takes a key name and looks it up in the mapping table. If
 * found, its key code is returned. Otherwise -1 is given
 * back.
 */
int Keys::getKeyCode(char *keyName)
{
     uint_t i;
     for (i = 0; i < sizeof(keyNames) / sizeof(keyNames[0]); i++) {
	  if (!dStrAsciiCasecmp(keyNames[i].name, keyName)) {
	       return keyNames[i].value;
	  }
     }

     return -1;
}

/*
 * Takes a command name and searches it in the mapping table.
 * Return value: command code if found, -1 otherwise
 */
KeysCommand_t Keys::getCmdCode(const char *commandName)
{
     uint_t i;

     for (i = 0; i < sizeof(default_keys) / sizeof(KeyBinding_t); i++) {
	  if (!dStrAsciiCasecmp(default_keys[i].name, commandName))
	       return default_keys[i].cmd;
     }
     return KEYS_INVALID;
}

/*
 * Takes a modifier name and looks it up in the mapping table. If
 * found, its key code is returned. Otherwise -1 is given back.
 */
int Keys::getModifier(char *modifierName)
{
     uint_t i;
     for (i = 0; i < sizeof(modifierNames) / sizeof(modifierNames[0]); i++) {
	  if (!dStrAsciiCasecmp(modifierNames[i].name, modifierName)) {
	       return modifierNames[i].value;
	  }
     }

     return -1;
}

/*
 * Given a keys command, return a shortcut for it, or 0 if there is none
 * (e.g., for KEYS_NEW_WINDOW, return CTRL+'n').
 */
int Keys::getShortcut(KeysCommand_t cmd)
{
     int len = dList_length(bindings);

     for (int i = 0; i < len; i++) {
	  KeyBinding_t *node = (KeyBinding_t*)dList_nth_data(bindings, i);
	  if (cmd == node->cmd)
	       return node->modifier + node->key;
     }
     return 0;
}

/*
 * Parse a key-combination/command-name pair, and
 * insert it into the bindings list.
 */
void Keys::parseKey(char *key, char *commandName)
{
     char *p, *modstr, *keystr;
     KeysCommand_t symcode;
     int st, keymod = 0, keycode = 0;

     _MSG("Keys::parseKey key='%s' commandName='%s'\n", key, commandName);

     // Get command code
     if ((symcode = getCmdCode(commandName)) == KEYS_INVALID) {
	  MSG("Keys::parseKey: Invalid command name: '%s'\n", commandName);
	  return;
     }

     // Skip space
     for (  ; isspace(*key); ++key) ;
     // Get modifiers
     while(*key == '<' && (p = strchr(key, '>'))) {
	  ++key;
	  modstr = dStrndup(key, p - key);
	  if ((st = getModifier(modstr)) == -1) {
	       MSG("Keys::parseKey unknown modifier: %s\n", modstr);
	  } else {
	       keymod |= st;
	  }
	  dFree(modstr);
	  key = p + 1;
     }
     // Allow trailing space after keyname
     keystr = (*key && (p = strchr(key + 1, ' '))) ? dStrndup(key, p - key - 1) :
	  dStrdup(key);
     // Get key code
     if (!key[1]) {
	  keycode = *key;
     } else if (a_Utf8_char_count(keystr, strlen(keystr)) == 1) {
	  const char *beyond = keystr + strlen(keystr);
	  keycode = a_Utf8_decode(keystr, beyond, NULL);
     } else if (key[0] == '0' && key[1] == 'x') {
	  /* keysym */
	  keycode = strtol(key, NULL, 0x10);
     } else if ((st = getKeyCode(keystr)) == -1) {
	  MSG("Keys::parseKey unknown keyname: %s\n", keystr);
     } else {
	  keycode = st;
     }
     dFree(keystr);

     // Set binding
     if (keycode) {
	  delKeyCmd(keycode, keymod);
	  if (symcode != KEYS_NOP) {
	       KeyBinding_t *node = dNew(KeyBinding_t, 1);
	       node->name = dStrdup(commandName);
	       node->cmd = symcode;
	       node->modifier = keymod;
	       node->key = keycode;
	       dList_insert_sorted(bindings, node, nodeByKeyCmp);
	       _MSG("parseKey: Adding key=%d, mod=%d\n", node->key, node->modifier);
	  }
     }
}

/*
 * Parse the keysrc.
 */
void Keys::parse(FILE *fp)
{
     char *line, *keycomb, *command;
     int st, lineno = 1;

     // scan the file line by line
     while ((line = dGetline(fp)) != NULL) {
	  st = dParser_parse_rc_line(&line, &keycomb, &command);

	  if (st == 0) {
	       _MSG("Keys::parse: keycomb=%s, command=%s\n", keycomb, command);
	       parseKey(keycomb, command);
	  } else if (st < 0) {
	       MSG("Keys::parse: Syntax error in keysrc line %d: "
		   "keycomb=\"%s\" command=\"%s\"\n", lineno, keycomb, command);
	  }

	  dFree(line);
	  ++lineno;
     }
     fclose(fp);
}
