/*
 * Dillo web browser
 *
 * Copyright 1999-2007 Jorge Arellano Cid <jcid@dillo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>
#include <signal.h>
#include <locale.h>

#include "msg.h"
#include "paths.hh"
#include "uicmd.hh"

#include "prefs.h"
#include "prefsparser.hh"
#include "keys.hh"
#include "bw.h"
#include "misc.h"
#include "history.h"

#include "dns.h"
#include "web.hh"
#include "IO/Url.h"
#include "IO/mime.h"
#include "capi.h"
#include "dicache.h"
#include "cookies.h"
#include "domain.h"
#include "auth.h"

#include <wx/wx.h>
#include <wx/cmdline.h>
#include <wx/log.h>

/*
 * Command line options structure
 */
typedef enum {
   DILLO_CLI_NONE          = 0,
   DILLO_CLI_XID           = 1 << 0,
   DILLO_CLI_FULLWINDOW    = 1 << 1,
   DILLO_CLI_HELP          = 1 << 2,
   DILLO_CLI_VERSION       = 1 << 3,
   DILLO_CLI_LOCAL         = 1 << 4,
   DILLO_CLI_GEOMETRY      = 1 << 5,
   DILLO_CLI_ERROR         = 1 << 15,
} OptID;

static const wxCmdLineEntryDesc gCmdLineDesc[] =
{
     // fullwindow
     { wxCMD_LINE_SWITCH, wxT("p"), wxT("fullwindow"),
       wxT("  -f, --fullwindow       Start in full window mode: hide address bar,\n"
	   "                                                 navigation buttons, menu, and status bar."),
       wxCMD_LINE_VAL_NONE, wxCMD_LINE_PARAM_OPTIONAL},
     // geometry
     { wxCMD_LINE_OPTION, wxT("g"), wxT("geometry"),
       wxT("  -g, --geometry GEO     Set initial window position where GEO is\n"
	   "                                                 WxH[{+-}X{+-}Y]"),
       wxCMD_LINE_VAL_STRING, wxCMD_LINE_PARAM_OPTIONAL},
     // help
     { wxCMD_LINE_SWITCH, wxT("h"), wxT("help"),
       wxT("  -h, --help             Display this help text and exit."),
       wxCMD_LINE_VAL_NONE, wxCMD_LINE_OPTION_HELP},
     // local
     { wxCMD_LINE_SWITCH, wxT("l"), wxT("local"),
       wxT("  -l, --local            Don't load images or stylesheets for these "
	   "URL(s)."),
       wxCMD_LINE_VAL_NONE, wxCMD_LINE_PARAM_OPTIONAL},
     // version
     { wxCMD_LINE_SWITCH, wxT("v"), wxT("version"),
       wxT("  -v, --version          Display version info and exit."),
       wxCMD_LINE_VAL_NONE, wxCMD_LINE_PARAM_OPTIONAL},
     // xid
     { wxCMD_LINE_OPTION, wxT("x"), wxT("xid"),
       wxT("  -x, --xid XID	         Open first Dillo window in an existing\n"
	   "			                         window whose window ID is XID."),
       wxCMD_LINE_VAL_STRING, wxCMD_LINE_PARAM_OPTIONAL},
     // URL/FILE
     { wxCMD_LINE_PARAM, wxT("url_file"), wxEmptyString,
       wxT("[URL|FILE]\n\n"
	   "  URL                    URL to browse.\n"
	   "  FILE                   Local FILE to view.\n\n"),
       wxCMD_LINE_VAL_STRING, wxCMD_LINE_OPTION_MANDATORY},
     // NONE
     { wxCMD_LINE_NONE }
};

/*
 * Declare wxDilloMain
 */
class wxDilloMain : public wxApp {

     // for internationalize
     wxLocale m_Locale;

public:
     wxDilloMain() : m_Locale(wxLANGUAGE_DEFAULT){}
     virtual bool OnInit();
     virtual int  OnExit();
     virtual int  OnRun();
     virtual void OnInitCmdLine(wxCmdLineParser& parser);
     virtual bool OnCmdLineParsed(wxCmdLineParser& parser);

private:
     wxString m_url_file;
     long     m_xid;

     uint_t   opt_id;
     uint_t   options_got = 0;
     uint32_t xid = 0;
     int      idx = 1;
     int      xpos = PREFS_GEOMETRY_DEFAULT_XPOS, 
              ypos = PREFS_GEOMETRY_DEFAULT_YPOS,
	      width = PREFS_GEOMETRY_DEFAULT_WIDTH,
	      height = PREFS_GEOMETRY_DEFAULT_HEIGHT;
     
};

IMPLEMENT_APP(wxDilloMain)

bool wxDilloMain::OnInit() 
{
     if (!wxApp::OnInit())
	  return false;

     FILE *fp;

     srand((uint_t)(time(0) ^ getpid()));

     // Some OSes exit dillo without this (not GNU/Linux).
     signal(SIGPIPE, SIG_IGN);
 
     // set the default values for the preferences
     a_Prefs_init();

     // create ~/.dillo if not present
     Paths::init();

     // initialize default key bindings
     Keys::init();

     // parse dillorc
     if ((fp = Paths::getPrefsFP(PATHS_RC_PREFS))) {
	  PrefsParser::parse(fp);
     }
     // parse keysrc
     if ((fp = Paths::getPrefsFP(PATHS_RC_KEYS))) {
	  Keys::parse(fp);
     }
     // parse domainrc
     if ((fp = Paths::getPrefsFP(PATHS_RC_DOMAIN))) {
	  a_Domain_parse(fp);
	  fclose(fp);
     }
     dLib_show_messages(prefs.show_msg);

     // initialize internal modules
     a_Dpi_init();
     a_Dns_init();
     // a_Web_init();
     // a_Http_init();
     // a_Mime_init();
     // a_Capi_init();
     // a_Dicache_init();
     // a_Bw_init();
     // a_Cookies_init();
     // a_Auth_init();
     // a_UIcmd_init();

     return true;
}

int wxDilloMain::OnExit() 
{
     return 0;
}

int wxDilloMain::OnRun()
{
    int exitcode = wxApp::OnRun();
    if (exitcode!=0)
        return exitcode;
}

void wxDilloMain::OnInitCmdLine(wxCmdLineParser& parser)
{
     // init command line setting
     parser.SetDesc(gCmdLineDesc);
     parser.SetSwitchChars(wxT("-"));

     // init logging setting
     wxLog::SetLogLevel(wxLOG_Max);
     wxLog::SetVerbose();

     // use stdout always
     wxLog *logger_cout = new wxLogStream(&std::cout);
     wxLog::SetActiveTarget(logger_cout);
}

bool wxDilloMain::OnCmdLineParsed(wxCmdLineParser& parser)
{
     /* Handle command line options */
     wxString xid, geo;

     if ((m_url_file = parser.GetParam(0)) == wxEmptyString) {
	  parser.Usage();
	  return false;
     }

     if (parser.Found(wxT("x"), &xid)) {
     	  if(!xid.ToLong(&m_xid)) {
     	       wxLogError(wxString::Format(wxT("XID argument \"%s\" not valid.\n"), xid.c_str()));
     	       return false;
     	  }
     }

     if (parser.Found(wxT("g"), &geo)) {
     	  if(!a_Misc_parse_geometry(geo.mb_str(), &xpos, &ypos, &width, &height)) {
     	       wxLogError(wxString::Format(
			       wxT("geometry argument \"%s\" not valid. Must be of "
				   "the form WxH[{+-}X{+-}Y].\n"), geo.c_str()));
     	       return false;
     	  }
     }

     if (parser.Found(wxT("v"))) {
	  puts("Dillo version " VERSION);
	  return false;
     }
	  
     return true;
}
