/******************************************************************************
 *
 *  Copyright 2017 Google, Inc.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at:
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 ******************************************************************************/
#pragma once
#include <cstdint>
#include <vector>
namespace {

// Complete SDUs that should always assemble.
std::vector<std::vector<uint8_t> > good_sdu = {
    {0x0c, 0x00, 0x47, 0x00, 0x02, 0x41, 0x12, 0x00, 0x00, 0x01, 0x02, 0x03,
     0x04, 0x05, 0xeb, 0x0f},

    {0x0a, 0x00, 0x47, 0x00, 0x04, 0xc1, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
     0x5a, 0x8a},

    {0x0a, 0x00, 0x47, 0x00, 0x06, 0x81, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11,
     0x68, 0x1e}};

// The complete L2CAP packet that should be created by calling assemble on the
// good_sdu vector.
std::vector<uint8_t> good_l2cap_packet = {
    0x12, 0x00, 0x47, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
    0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11};

// SDUs that are missing a payload.
std::vector<std::vector<uint8_t> > empty_sdu_payload = {
    {0x06, 0x00, 0x47, 0x00, 0x02, 0x41, 0x00, 0x00, 0xde, 0xf1},

    {0x04, 0x00, 0x47, 0x00, 0x04, 0x81, 0xd7, 0x90}};

// The l2cap packet that should be created by calling assemble on the
// empty_sdu_payload vector.
std::vector<uint8_t> empty_l2cap_payload = {0x00, 0x00, 0x47, 0x00};

// SDUs that have all of the control bytes set to be the start of the SDU.
std::vector<std::vector<uint8_t> > all_first_packet = {
    {0x0c, 0x00, 0x47, 0x00, 0x02, 0x41, 0x18, 0x00, 0x00, 0x01, 0x02, 0x03,
     0x04, 0x05, 0x6b, 0x70},

    {0x0c, 0x00, 0x47, 0x00, 0x02, 0x41, 0x18, 0x00, 0x00, 0x01, 0x02, 0x03,
     0x04, 0x05, 0x6b, 0x70},

    {0x0c, 0x00, 0x47, 0x00, 0x02, 0x41, 0x18, 0x00, 0x00, 0x01, 0x02, 0x03,
     0x04, 0x05, 0x6b, 0x70}};

// A complete L2CAP packet that has not been segmented.
std::vector<std::vector<uint8_t> > one_sdu = {{0x0b, 0x00, 0x47, 0x00, 0x02,
                                               0x00, 0x01, 0x02, 0x03, 0x04,
                                               0x05, 0x06, 0x07, 0x4a, 0x2f}};

// l2cap_test_packet_[1-9] are SDUs that were extracted from snoop logs.
std::vector<uint8_t> l2cap_test_packet_1 = {
    0xf8, 0x03, 0x47, 0x00, 0x02, 0x41, 0x95, 0x1f, 0x02, 0x1f, 0x95, 0xcb,
    0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x33, 0x00, 0x49, 0x00, 0x4d, 0x00,
    0x47, 0x00, 0x5f, 0x00, 0x32, 0x00, 0x30, 0x00, 0x31, 0x00, 0x37, 0x00,
    0x30, 0x00, 0x36, 0x00, 0x30, 0x00, 0x36, 0x00, 0x5f, 0x00, 0x31, 0x00,
    0x35, 0x00, 0x34, 0x00, 0x33, 0x00, 0x33, 0x00, 0x38, 0x00, 0x2e, 0x00,
    0x6a, 0x00, 0x70, 0x00, 0x67, 0x00, 0x00, 0x42, 0x00, 0x0e, 0x69, 0x6d,
    0x61, 0x67, 0x65, 0x2f, 0x6a, 0x70, 0x65, 0x67, 0x00, 0xc3, 0x00, 0x12,
    0x30, 0x03, 0x97, 0x01, 0x48, 0x1f, 0x45, 0xff, 0xd8, 0xff, 0xe0, 0x00,
    0x10, 0x4a, 0x46, 0x49, 0x46, 0x00, 0x01, 0x01, 0x00, 0x00, 0x01, 0x00,
    0x01, 0x00, 0x00, 0xff, 0xe1, 0x2b, 0x18, 0x45, 0x78, 0x69, 0x66, 0x00,
    0x00, 0x4d, 0x4d, 0x00, 0x2a, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0c, 0x01,
    0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x0c, 0xc0, 0x01,
    0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x09, 0x90, 0x01,
    0x0f, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x4c, 0x47, 0x45, 0x00, 0x01,
    0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x9e, 0x01,
    0x12, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x01,
    0x1a, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xa6, 0x01,
    0x1b, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xae, 0x01,
    0x28, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x01,
    0x31, 0x00, 0x02, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0xb6, 0x01,
    0x32, 0x00, 0x02, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0xbd, 0x02,
    0x13, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x87,
    0x69, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xd2, 0x00,
    0x00, 0x02, 0x78, 0x4e, 0x65, 0x78, 0x75, 0x73, 0x20, 0x35, 0x00, 0x00,
    0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x48, 0x00,
    0x00, 0x00, 0x01, 0x50, 0x69, 0x63, 0x61, 0x73, 0x61, 0x00, 0x32, 0x30,
    0x31, 0x37, 0x3a, 0x30, 0x36, 0x3a, 0x30, 0x36, 0x20, 0x31, 0x35, 0x3a,
    0x34, 0x33, 0x3a, 0x33, 0x38, 0x00, 0x00, 0x00, 0x15, 0x82, 0x9a, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0xd4, 0x82, 0x9d, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01, 0xdc, 0x88, 0x27, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x02, 0x1f, 0x00, 0x00, 0x90, 0x00, 0x00,
    0x07, 0x00, 0x00, 0x00, 0x04, 0x30, 0x32, 0x32, 0x30, 0x90, 0x03, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x01, 0xe4, 0x90, 0x04, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x01, 0xf8, 0x91, 0x01, 0x00,
    0x07, 0x00, 0x00, 0x00, 0x04, 0x01, 0x02, 0x03, 0x00, 0x92, 0x01, 0x00,
    0x0a, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0x0c, 0x92, 0x02, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0x14, 0x92, 0x09, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x92, 0x0a, 0x00,
    0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0x1c, 0x92, 0x90, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x02, 0x24, 0x92, 0x91, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x02, 0x2b, 0x92, 0x92, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x02, 0x32, 0xa0, 0x00, 0x00,
    0x07, 0x00, 0x00, 0x00, 0x04, 0x30, 0x31, 0x30, 0x30, 0xa0, 0x01, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0xa0, 0x02, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x0a, 0x0a, 0xa0, 0x03, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x07, 0x77, 0xa0, 0x05, 0x00,
    0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0x5a, 0xa4, 0x03, 0x00,
    0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x20, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x21, 0x00, 0x00, 0x02, 0x39, 0x00, 0x00, 0x00,
    0x00, 0x01, 0xfc, 0x8c, 0xb2, 0x3b, 0x9a, 0xca, 0x00, 0x00, 0x00, 0x00,
    0xf0, 0x00, 0x00, 0x00, 0x64, 0x32, 0x30, 0x31, 0x37, 0x3a, 0x30, 0x36,
    0x3a, 0x30, 0x36, 0x20, 0x31, 0x35, 0x3a, 0x34, 0x33, 0x3a, 0x33, 0x38,
    0x00, 0x32, 0x30, 0x31, 0x37, 0x3a, 0x30, 0x36, 0x3a, 0x30, 0x36, 0x20,
    0x31, 0x35, 0x3a, 0x34, 0x33, 0x3a, 0x33, 0x38, 0x00, 0xff, 0xff, 0xfe,
    0x16, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x00, 0xfc, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x0f, 0x82, 0x00, 0x00, 0x03, 0xe8, 0x39, 0x39, 0x39,
    0x37, 0x32, 0x39, 0x00, 0x39, 0x39, 0x39, 0x37, 0x32, 0x39, 0x00, 0x39,
    0x39, 0x39, 0x37, 0x32, 0x39, 0x00, 0x31, 0x30, 0x61, 0x66, 0x39, 0x37,
    0x31, 0x37, 0x63, 0x31, 0x30, 0x36, 0x36, 0x33, 0x65, 0x62, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x00, 0x00, 0x02, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x04, 0x52, 0x39, 0x38, 0x00, 0x00, 0x02, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x04, 0x30, 0x31, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x01,
    0x03, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x06, 0x00, 0x00, 0x01,
    0x1a, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0xc6, 0x01,
    0x1b, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0xce, 0x01,
    0x28, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x02,
    0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0xd6, 0x02,
    0x02, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x28, 0x39, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x01, 0x00,
    0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x01, 0xff, 0xd8, 0xff, 0xe0, 0x00,
    0x10, 0x4a, 0x46, 0x49, 0x46, 0x00, 0x01, 0x01, 0x00, 0x00, 0x01, 0x00,
    0x01, 0x00, 0x00, 0xff, 0xdb, 0x00, 0x43, 0x00, 0x05, 0x03, 0x04, 0x04,
    0x04, 0x03, 0x05, 0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0x06, 0x07, 0x0c,
    0x08, 0x07, 0x07, 0x07, 0x07, 0x0f, 0x0b, 0x0b, 0x09, 0x0c, 0x11, 0x0f,
    0x12, 0x12, 0x11, 0x0f, 0x11, 0x11, 0x13, 0x16, 0x1c, 0x17, 0x13, 0x14,
    0x1a, 0x15, 0x11, 0x11, 0x18, 0x21, 0x18, 0x1a, 0x1d, 0x1d, 0x1f, 0x1f,
    0x1f, 0x13, 0x17, 0x22, 0x24, 0x22, 0x1e, 0x24, 0x1c, 0x1e, 0x1f, 0x1e,
    0xff, 0xdb, 0x00, 0x43, 0x01, 0x05, 0x05, 0x05, 0x07, 0x06, 0x07, 0x0e,
    0x08, 0x08, 0x0e, 0x1e, 0x14, 0x11, 0x14, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e,
    0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e,
    0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e,
    0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e,
    0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0xff, 0xc0, 0x00,
    0x11, 0x08, 0x00, 0x78, 0x00, 0xa0, 0x03, 0x01, 0x22, 0x00, 0xaa, 0x72};

std::vector<uint8_t> l2cap_test_packet_2 = {
    0xf6, 0x03, 0x47, 0x00, 0x04, 0xc1, 0x02, 0x11, 0x01, 0x03, 0x11, 0x01,
    0xff, 0xc4, 0x00, 0x1c, 0x00, 0x00, 0x03, 0x00, 0x03, 0x01, 0x01, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x06, 0x07,
    0x00, 0x04, 0x08, 0x03, 0x02, 0x01, 0xff, 0xc4, 0x00, 0x40, 0x10, 0x00,
    0x01, 0x03, 0x02, 0x05, 0x02, 0x04, 0x04, 0x04, 0x03, 0x06, 0x04, 0x07,
    0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x11, 0x00, 0x06, 0x12,
    0x21, 0x31, 0x07, 0x41, 0x13, 0x22, 0x51, 0x61, 0x14, 0x32, 0x71, 0x81,
    0x08, 0x42, 0x91, 0xa1, 0x15, 0x16, 0xc1, 0x23, 0x52, 0x62, 0x82, 0xb1,
    0xe1, 0x24, 0x43, 0x72, 0xf0, 0x17, 0x25, 0x33, 0x92, 0xb2, 0xd1, 0xd2,
    0xff, 0xc4, 0x00, 0x1c, 0x01, 0x00, 0x03, 0x01, 0x01, 0x00, 0x03, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x06, 0x07,
    0x04, 0x03, 0x00, 0x02, 0x08, 0x01, 0xff, 0xc4, 0x00, 0x37, 0x11, 0x00,
    0x01, 0x02, 0x04, 0x04, 0x03, 0x06, 0x04, 0x05, 0x04, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x00, 0x04, 0x05, 0x11, 0x06, 0x21,
    0x31, 0x41, 0x12, 0x51, 0x71, 0x13, 0x22, 0x61, 0x81, 0x91, 0xa1, 0x07,
    0xb1, 0xc1, 0xf0, 0x14, 0x42, 0xd1, 0xe1, 0xf1, 0x15, 0x24, 0x32, 0x82,
    0x33, 0x52, 0x72, 0xff, 0xda, 0x00, 0x0c, 0x03, 0x01, 0x00, 0x02, 0x11,
    0x03, 0x11, 0x00, 0x3f, 0x00, 0xb1, 0x31, 0x2c, 0x2d, 0xd8, 0xd2, 0xda,
    0x43, 0x26, 0x4a, 0x99, 0x1e, 0x15, 0x98, 0x2e, 0x2f, 0x51, 0x49, 0x0b,
    0x03, 0x70, 0x90, 0x6c, 0x9f, 0x4d, 0xb7, 0xf5, 0xc3, 0x0b, 0x4f, 0x09,
    0x1f, 0x08, 0xa5, 0x95, 0xb6, 0xb7, 0x50, 0x4e, 0x95, 0xab, 0xe4, 0xbf,
    0x2a, 0xb1, 0x17, 0xb8, 0xb1, 0xde, 0xdd, 0xf1, 0x34, 0xa1, 0x3b, 0x25,
    0xb6, 0x5d, 0x67, 0xe2, 0xda, 0x5b, 0xeb, 0x49, 0xf0, 0xd2, 0x83, 0xe1,
    0xa9, 0x37, 0xbe, 0x91, 0x72, 0xab, 0x5f, 0x73, 0x7f, 0xb6, 0x1d, 0x28,
    0xf0, 0xdf, 0xf1, 0x9b, 0x05, 0x2e, 0x20, 0x32, 0xd8, 0x64, 0x23, 0x52,
    0x4a, 0x9c, 0xb7, 0x2b, 0x52, 0x8d, 0xd5, 0xe9, 0xe5, 0xdb, 0x83, 0xea,
    0x30, 0x09, 0x75, 0x86, 0x25, 0xec, 0x1e, 0x58, 0x4d, 0xf9, 0x90, 0x20,
    0x87, 0x60, 0x4e, 0x91, 0xb2, 0xc3, 0x72, 0x63, 0xcd, 0x40, 0x76, 0x63,
    0xd2, 0x5b, 0x4a, 0x75, 0x04, 0x20, 0x06, 0xf9, 0xdb, 0xcd, 0xdc, 0x83,
    0xe9, 0x6b, 0xec, 0x0d, 0xf1, 0xba, 0x5f, 0x76, 0xe9, 0x50, 0x43, 0x65,
    0xd9, 0x05, 0x21, 0x08, 0xf2, 0x92, 0x94, 0x05, 0x73, 0x72, 0x48, 0x20,
    0x5c, 0xaa, 0xf8, 0xf8, 0xa9, 0x34, 0x97, 0xe4, 0x3e, 0xc7, 0xc4, 0x80,
    0x95, 0x90, 0x90, 0x9d, 0x1a, 0x94, 0x8b, 0xd8, 0x6d, 0xb5, 0x80, 0xef,
    0xf7, 0xc7, 0xeb, 0x14, 0xb7, 0xdb, 0x79, 0x45, 0x2f, 0xb6, 0x50, 0x55,
    0x7b, 0x96, 0x86, 0xb4, 0xf1, 0xc2, 0xaf, 0x7e, 0xc3, 0x7c, 0x65, 0x77,
    0x15, 0x52, 0x98, 0x36, 0x53, 0xc3, 0xca, 0xe7, 0xe5, 0x78, 0xf0, 0xb0,
    0xb3, 0x99, 0x11, 0xae, 0xa6, 0xd8, 0x91, 0x2c, 0x38, 0x3c, 0x37, 0xb4,
    0x36, 0xa6, 0x99, 0xd2, 0xb2, 0xb0, 0xda, 0x01, 0x04, 0xea, 0x24, 0x8d,
    0xcd, 0xb6, 0xfa, 0x63, 0x62, 0x92, 0xdb, 0x2c, 0xb8, 0xcb, 0x29, 0x7d,
    0x97, 0x94, 0x5b, 0x53, 0xa0, 0x22, 0xfa, 0x02, 0x52, 0x52, 0x35, 0x12,
    0x2f, 0x71, 0x7b, 0x72, 0x77, 0xe0, 0x63, 0xc2, 0xb9, 0x43, 0xa9, 0x39,
    0x09, 0xc8, 0xd0, 0x1f, 0x6d, 0x95, 0xbb, 0xa5, 0x7f, 0x10, 0xfa, 0x3c,
    0x5d, 0xf5, 0x58, 0x24, 0x20, 0x5b, 0x51, 0x3c, 0x58, 0x9b, 0x5a, 0xf7,
    0xbd, 0xf1, 0x2f, 0xcd, 0x94, 0x9c, 0xff, 0x00, 0x4a, 0x94, 0xe4, 0xba,
    0x8e, 0x60, 0xa8, 0xba, 0xc0, 0x05, 0x1a, 0xe2, 0x47, 0x09, 0x09, 0x4f,
    0xa6, 0x94, 0x8b, 0xa7, 0xd8, 0x0f, 0xb6, 0x0a, 0xd3, 0xaa, 0x54, 0xe9,
    0xe5, 0xa1, 0x29, 0x98, 0x4a, 0x4a, 0xb4, 0x0a, 0xba, 0x6f, 0x9d, 0xb7,
    0x16, 0xbf, 0x21, 0x7b, 0x9d, 0x84, 0x0d, 0xa8, 0x4c, 0xb9, 0x26, 0xd9,
    0x73, 0xb2, 0x52, 0xc0, 0xff, 0x00, 0xa8, 0x07, 0xda, 0xf7, 0xeb, 0x95,
    0x84, 0x55, 0xdb, 0x4c, 0x26, 0xa4, 0x3e, 0xdb, 0x0b, 0x2e, 0xcb, 0x91,
    0x24, 0xbe, 0xa2, 0x2c, 0x08, 0x52, 0x86, 0xca, 0xb7, 0x17, 0x00, 0x58,
    0xed, 0x72, 0x0e, 0xf8, 0xf8, 0x7e, 0x6c, 0x28, 0xce, 0x3a, 0xc3, 0x25,
    0x90, 0x88, 0xc9, 0xd4, 0xe9, 0x71, 0xc0, 0x09, 0x52, 0x85, 0xed, 0xa8,
    0xf7, 0xbd, 0xff, 0x00, 0x4d, 0x86, 0x39, 0x96, 0xa7, 0xd4, 0x6a, 0x63,
    0x28, 0x5b, 0x1f, 0xce, 0x33, 0xf5, 0x36, 0x0a, 0x5c, 0x49, 0x75, 0x6b,
    0x70, 0xab, 0x8b, 0x8b, 0x91, 0x63, 0x6b, 0x6d, 0x6e, 0xd8, 0x00, 0xee,
    0x76, 0xca, 0xce, 0x27, 0xc6, 0x6a, 0x8b, 0x32, 0x78, 0x69, 0x3f, 0xda,
    0x3a, 0xa6, 0xca, 0x89, 0xb7, 0x2a, 0x3b, 0x9f, 0x7c, 0x37, 0xff, 0x00,
    0x41, 0x69, 0x1f, 0xf2, 0xbe, 0x07, 0xa7, 0xd4, 0x88, 0x54, 0x18, 0xaa,
    0x61, 0xcf, 0xf8, 0xa4, 0xd6, 0x7a, 0xe5, 0xf3, 0x1f, 0xac, 0x75, 0x8c,
    0x0c, 0xc3, 0x02, 0xa3, 0x50, 0x6a, 0x9c, 0xc5, 0x62, 0x92, 0xe4, 0xa7,
    0x10, 0xb6, 0xbe, 0x1d, 0xb9, 0x2d, 0xad, 0x00, 0x80, 0x3f, 0x2f, 0xcc,
    0xb5, 0x0b, 0x1e, 0x2c, 0x07, 0xb9, 0xc7, 0xae, 0x68, 0x96, 0xa4, 0xd2,
    0xcb, 0x31, 0x25, 0x29, 0x72, 0x00, 0x0d, 0x96, 0xda, 0xec, 0x9b, 0xf7,
    0xb0, 0xd8, 0xf7, 0xda, 0xc7, 0xb6, 0x21, 0x79, 0x22, 0x3d, 0x1e, 0xbf,
    0x45, 0x8b, 0x3c, 0xd3, 0x95, 0x11, 0xb9, 0x26, 0xf1, 0x5c, 0x69, 0x65,
    0xa7, 0x10, 0xa0, 0xe6, 0x84, 0x9d, 0xb7, 0x1e, 0x71, 0x89, 0xac, 0x0a,
    0x9d, 0x56, 0x8f, 0x38, 0x48, 0x8d, 0x52, 0x9d, 0x0d, 0x65, 0xc2, 0x54,
    0xeb, 0x2e, 0xa9, 0x0a, 0x55, 0x8d, 0x89, 0xe7, 0x7e, 0xfc, 0xe0, 0x0d,
    0x79, 0xb6, 0xe9, 0x8a, 0x42, 0x50, 0x4a, 0x82, 0xc6, 0xbf, 0xa4, 0x51,
    0xbe, 0x1b, 0xc8, 0x2b, 0x18, 0x35, 0x32, 0xe3, 0x87, 0xb2, 0x53, 0x2a,
    0x09, 0xe1, 0xd6, 0xe4, 0x8b, 0xe6, 0x76, 0xf4, 0x31, 0xd6, 0x31, 0x67,
    0x66, 0x00, 0xda, 0x90, 0x22, 0xc6, 0x79, 0xb4, 0x20, 0xa1, 0x6f, 0xbc,
    0x0a, 0x41, 0x09, 0xf3, 0x6b, 0x09, 0x37, 0xb0, 0x36, 0xdc, 0x12, 0x4e,
    0xd8, 0x11, 0x51, 0x7f, 0xf8, 0xed, 0x5a, 0x1c, 0x88, 0x35, 0x04, 0xa5,
    0xc8, 0x4d, 0x2d, 0x4d, 0xa9, 0xc8, 0xd6, 0x6b, 0x52, 0x87, 0x99, 0x4a,
    0xb8, 0xdc, 0x93, 0xc0, 0x06, 0xdc, 0xe2, 0x21, 0x13, 0xa9, 0xd9, 0xb0,
    0xb1, 0xe1, 0x4a, 0xaf, 0x3a, 0xe9, 0x4a, 0xb5, 0x21, 0x6e, 0xb0, 0x85,
    0x9f, 0x6b, 0xa8, 0x6f, 0x7f, 0xb6, 0x35, 0xe2, 0xf5, 0x37, 0x37, 0xca,
    0x7c, 0x35, 0x51, 0xaf, 0xae, 0x13, 0x00, 0xfc, 0xed, 0x46, 0xd7, 0x71,
    0xfe, 0x22, 0x0d, 0xc7, 0xe8, 0x70, 0x10, 0xcf, 0xb0, 0x53, 0x61, 0x73,
    0xe9, 0xfa, 0xc3, 0xcc, 0xc6, 0x09, 0x7a, 0x50, 0x15, 0x29, 0x1c, 0x5f,
    0xf9, 0xb9, 0xfa, 0x67, 0xe5, 0x68, 0xe8, 0x7a, 0x34, 0x17, 0x60, 0x34,
    0xa0, 0x66, 0x53, 0xdb, 0x53, 0x84, 0xea, 0x5b, 0x67, 0x4d, 0xfd, 0xfc,
    0xbe, 0xfb, 0xf3, 0xbf, 0xd3, 0x6c, 0x7d, 0x66, 0x57, 0x5b};

std::vector<uint8_t> l2cap_test_packet_3 = {
    0xf6, 0x03, 0x47, 0x00, 0x06, 0xc1, 0x87, 0xeb, 0x74, 0xf8, 0xb1, 0xe1,
    0xd2, 0xc4, 0x35, 0x49, 0x90, 0xcb, 0xae, 0xc3, 0x73, 0x7d, 0x3a, 0x90,
    0x9b, 0x90, 0x45, 0xed, 0xbd, 0xf6, 0x3c, 0x0e, 0x6d, 0x89, 0xa6, 0x46,
    0x89, 0x56, 0xac, 0x4b, 0x81, 0x35, 0xac, 0xdc, 0xfd, 0x56, 0x31, 0x7d,
    0x2a, 0x29, 0x60, 0x0b, 0x25, 0x49, 0x58, 0xd9, 0xc0, 0x07, 0x96, 0xe3,
    0xb1, 0xb6, 0xd8, 0xac, 0xe7, 0xb4, 0x45, 0x6a, 0xaf, 0x4b, 0x43, 0x4a,
    0x6d, 0x25, 0xa9, 0x32, 0x11, 0xa1, 0x24, 0x5d, 0x29, 0x53, 0x5b, 0x6d,
    0xd8, 0x5c, 0x60, 0xad, 0x29, 0x4d, 0xcc, 0x38, 0x3b, 0xb6, 0x00, 0x8d,
    0x77, 0x85, 0xb9, 0x96, 0x19, 0x49, 0x52, 0x10, 0x9c, 0xc2, 0x54, 0x73,
    0x1a, 0x10, 0x92, 0x47, 0xbd, 0xa2, 0x1c, 0x9f, 0xfc, 0x40, 0x7a, 0x32,
    0x5f, 0x97, 0x99, 0x1d, 0x52, 0xdb, 0x36, 0x11, 0x92, 0xe2, 0x90, 0xd3,
    0x62, 0xdc, 0x0d, 0x24, 0x11, 0xb1, 0xc2, 0xfd, 0x5a, 0x5e, 0x60, 0x42,
    0x2c, 0xfc, 0x07, 0x18, 0x75, 0x2a, 0xba, 0x26, 0x21, 0xe5, 0x3f, 0x63,
    0xfd, 0xe4, 0xa9, 0x47, 0xca, 0x46, 0xe3, 0x71, 0xdf, 0x19, 0xd6, 0x3a,
    0xd5, 0x5e, 0x91, 0x0d, 0x2f, 0xc1, 0x21, 0x94, 0x29, 0xd7, 0x2e, 0x15,
    0x65, 0x05, 0x5b, 0x4e, 0xe4, 0x7d, 0xf8, 0x38, 0x0b, 0x42, 0xea, 0x69,
    0x30, 0x9b, 0x6a, 0xb9, 0x4b, 0x43, 0xe1, 0xb0, 0x12, 0x1d, 0x8e, 0x7c,
    0x32, 0x8f, 0xb7, 0x6f, 0xb6, 0x28, 0xcb, 0x99, 0x92, 0x95, 0x74, 0x36,
    0xa0, 0x94, 0xa8, 0x8b, 0x8c, 0xac, 0x39, 0x74, 0x88, 0xa6, 0x1f, 0xc4,
    0x18, 0xbd, 0x12, 0x9f, 0x8e, 0x67, 0x85, 0xe6, 0xd4, 0x4d, 0xd2, 0x6d,
    0x71, 0x6d, 0x6d, 0x98, 0x3e, 0x40, 0xf9, 0x40, 0xf9, 0x19, 0x9a, 0xad,
    0xe3, 0x3b, 0x11, 0x6e, 0xca, 0xa9, 0x6b, 0x56, 0x8d, 0x72, 0x5c, 0x4a,
    0xad, 0x7b, 0x0b, 0x24, 0xe9, 0x3a, 0x71, 0xf7, 0x97, 0xaa, 0x19, 0x86,
    0x65, 0x52, 0x24, 0x66, 0xe9, 0x0d, 0xb7, 0xe3, 0x38, 0x1b, 0x49, 0x44,
    0x50, 0xab, 0x82, 0xab, 0x5e, 0xc4, 0x6f, 0x6c, 0x35, 0xb3, 0x4b, 0xcb,
    0x79, 0x89, 0xe4, 0xce, 0xa6, 0x26, 0x6c, 0x69, 0x5a, 0x82, 0x95, 0x29,
    0x2b, 0x52, 0x1d, 0x41, 0x3b, 0x8b, 0x91, 0xda, 0xc3, 0x6d, 0x88, 0xd8,
    0xe1, 0x93, 0x2f, 0xcf, 0xab, 0xe5, 0x7c, 0xc7, 0x4d, 0x46, 0x60, 0x97,
    0x0e, 0x75, 0x31, 0xe9, 0x08, 0x69, 0x12, 0x8c, 0x60, 0x87, 0xda, 0xf4,
    0xbe, 0x8b, 0x05, 0x0f, 0xb0, 0x3d, 0xf1, 0x8a, 0x64, 0xd5, 0x5a, 0x1c,
    0x52, 0xae, 0x24, 0xa0, 0x67, 0x6b, 0x01, 0xfc, 0xfa, 0xc3, 0x75, 0x33,
    0xe2, 0x63, 0x4f, 0xbc, 0x99, 0x57, 0xd2, 0x59, 0x73, 0x4e, 0x12, 0x32,
    0xbf, 0xcf, 0xd6, 0x06, 0xf5, 0x01, 0xd9, 0x74, 0xac, 0xba, 0xdb, 0x2f,
    0x33, 0x1d, 0xba, 0x85, 0x36, 0xae, 0xe4, 0x77, 0x5d, 0x61, 0xad, 0x96,
    0x0b, 0x63, 0x70, 0x07, 0xfd, 0x37, 0xb6, 0x1f, 0xba, 0x4b, 0x5b, 0x95,
    0x3f, 0x2b, 0xc1, 0x44, 0xc7, 0x1e, 0x96, 0x02, 0x8d, 0xca, 0x9b, 0xb2,
    0x94, 0x41, 0x24, 0x12, 0x2f, 0xcd, 0xec, 0x37, 0xe7, 0xdf, 0x13, 0xbe,
    0xb2, 0x54, 0x4c, 0xba, 0xce, 0x60, 0x5c, 0x25, 0x87, 0xe9, 0xee, 0x4a,
    0x6e, 0x49, 0xf0, 0xd3, 0x72, 0x95, 0x29, 0x24, 0x03, 0xa8, 0x1d, 0xb9,
    0x3b, 0x7b, 0x7b, 0x60, 0xef, 0x43, 0x5d, 0x29, 0x88, 0xdb, 0x8b, 0x72,
    0x33, 0x65, 0x5e, 0x25, 0xee, 0x17, 0x7b, 0x10, 0x05, 0xac, 0x0d, 0x89,
    0xdb, 0x83, 0x85, 0x8c, 0x5a, 0x78, 0x66, 0x9a, 0x58, 0xdd, 0x20, 0x9f,
    0x3b, 0x9f, 0xac, 0x3b, 0xd5, 0xdb, 0xb9, 0x4a, 0xd5, 0xa9, 0x00, 0xfb,
    0x40, 0xac, 0xbf, 0x98, 0xbe, 0x31, 0x9a, 0xab, 0x73, 0x42, 0xde, 0x80,
    0xc7, 0x87, 0xa9, 0x2b, 0x6c, 0x10, 0xe0, 0x28, 0xbe, 0xe0, 0xec, 0x47,
    0xa7, 0xfb, 0x62, 0x9b, 0x94, 0x9a, 0xa9, 0xd6, 0x72, 0xa8, 0x95, 0x90,
    0x33, 0x73, 0x72, 0xd6, 0x86, 0x2e, 0x69, 0x92, 0xbc, 0xcb, 0x61, 0x76,
    0xf9, 0x52, 0x54, 0x6f, 0xa7, 0xd8, 0x92, 0x3d, 0x14, 0x31, 0x03, 0x76,
    0x3b, 0x4d, 0xe4, 0xcc, 0xcd, 0x23, 0x58, 0xbb, 0x8d, 0xa2, 0xe9, 0x04,
    0xd9, 0x20, 0x04, 0x8d, 0xbd, 0x76, 0xc4, 0xf3, 0x2a, 0x55, 0x33, 0x44,
    0x4c, 0xf6, 0x22, 0xe5, 0x27, 0xa6, 0xae, 0xa2, 0xe4, 0xa0, 0x96, 0x44,
    0x72, 0x6e, 0x0d, 0x85, 0xcd, 0x87, 0x6b, 0x5e, 0xff, 0x00, 0xbe, 0x3d,
    0xa6, 0x9a, 0x94, 0x98, 0xa7, 0xcb, 0x4a, 0x4c, 0xb4, 0x16, 0x14, 0x9b,
    0xe6, 0x01, 0xcf, 0x2d, 0xb5, 0xdf, 0x50, 0x41, 0x89, 0x54, 0x8c, 0xa3,
    0xc2, 0xa1, 0x3d, 0x50, 0x65, 0xee, 0x12, 0x85, 0x81, 0x62, 0x6c, 0x2d,
    0x62, 0x49, 0xdc, 0x65, 0x6d, 0xd2, 0x44, 0x5c, 0x1e, 0xcf, 0x3d, 0x5a,
    0x62, 0xa4, 0xe5, 0x3e, 0xa5, 0x50, 0xa7, 0xd2, 0xdd, 0x49, 0xd2, 0x84,
    0xbc, 0xd6, 0x95, 0x39, 0xcd, 0x82, 0x6f, 0x7b, 0xf1, 0xce, 0xe0, 0x7a,
    0xe1, 0x83, 0x20, 0xbf, 0xd4, 0x0a, 0xad, 0x42, 0x4b, 0xd5, 0x6c, 0xe0,
    0xf1, 0x66, 0x24, 0xe6, 0xee, 0xcb, 0x6d, 0x20, 0x02, 0x9b, 0x82, 0x52,
    0xab, 0x0f, 0x4d, 0x88, 0xf7, 0xc1, 0x8c, 0xd5, 0x2e, 0x8a, 0x7a, 0x5d,
    0x4e, 0x8b, 0x5b, 0x8e, 0xd2, 0xab, 0x06, 0xea, 0x50, 0x6d, 0x3b, 0xa1,
    0xb2, 0xa3, 0xa0, 0x58, 0x7f, 0xcc, 0x50, 0xb5, 0x80, 0xdc, 0x83, 0xbe,
    0x24, 0xbd, 0x36, 0xcc, 0x33, 0xb2, 0xcd, 0x54, 0x25, 0x42, 0x4a, 0x5a,
    0x2e, 0x02, 0x96, 0xdc, 0x90, 0x4a, 0x55, 0x7d, 0x89, 0xb7, 0x04, 0x81,
    0xff, 0x00, 0xd6, 0xf8, 0x54, 0xad, 0xe1, 0x4a, 0x6d, 0x38, 0x07, 0x19,
    0x69, 0x20, 0x1b, 0xe4, 0x52, 0x2f, 0x96, 0xe0, 0xf2, 0x3b, 0x7d, 0x61,
    0xaf, 0x0d, 0x62, 0x79, 0x8a, 0xb0, 0x52, 0x1c, 0x02, 0xe9, 0x09, 0xcc,
    0x68, 0x4a, 0x85, 0xed, 0xa7, 0xf9, 0x0d, 0xc7, 0x8e, 0xd1, 0xd8, 0x64,
    0xff, 0x00, 0xc5, 0xb4, 0xd3, 0x68, 0x05, 0x28, 0x0a, 0x5a, 0x92, 0x4f,
    0xe5, 0x4b, 0x97, 0xb7, 0xe8, 0x71, 0x1a, 0xab, 0xfe, 0x21, 0x29, 0xd4,
    0x7e, 0xa5, 0xce, 0xca, 0x35, 0xea, 0x64, 0x91, 0x09, 0x01, 0x05, 0x12,
    0xda, 0x40, 0x50, 0x4e, 0xa1, 0xdd, 0x3c, 0xd8, 0x7a, 0xef, 0x8a, 0xc5,
    0x36, 0xa9, 0x0e, 0x44, 0x84, 0xe8, 0x73, 0xfb, 0x59, 0x2c, 0x29, 0x4d,
    0xa7, 0xbf, 0x65, 0x7f, 0xa1, 0xc7, 0x13, 0xf5, 0x06, 0x9b, 0x52, 0xac,
    0xfe, 0x20, 0x6a, 0x90, 0x20, 0x45, 0x91, 0x2a, 0x5b, 0x8b, 0x6d, 0xb6,
    0x19, 0x6c, 0x5d, 0x4b, 0xb2, 0x78, 0x1f, 0xbe, 0xfc, 0x0c, 0x21, 0x61,
    0xb9, 0x66, 0xa7, 0xdb, 0xb3, 0xa7, 0x44, 0x2b, 0x43, 0x6b, 0x10, 0xb4,
    0x81, 0xec, 0x77, 0xb8, 0xcc, 0xc1, 0xa7, 0xf8, 0x99, 0x48, 0x36, 0xce,
    0xe0, 0x67, 0xc8, 0x85, 0x1f, 0xa6, 0xa2, 0x3a, 0x3b, 0x33, 0xe4, 0x8c,
    0x85, 0xd4, 0x46, 0xda, 0xae, 0x52, 0xc4, 0x07, 0x64, 0x2e, 0xc0, 0x49,
    0x61, 0x00, 0x2b, 0xe8, 0xbb, 0x6f, 0xbe, 0x14, 0xab, 0xb9, 0x30, 0x50,
    0x22, 0xff, 0x00, 0x0f, 0xfe, 0x13, 0x18, 0x7c, 0x72, 0x96, 0xc0, 0x90,
    0x41, 0x29, 0x41, 0x50, 0xb0, 0xbf, 0x1e, 0x5e, 0x76, 0xdb, 0xdc, 0xfa,
    0xb5, 0xf4, 0xeb, 0x24, 0x53, 0x7a, 0x71, 0x45, 0x44, 0xe6};

std::vector<uint8_t> l2cap_test_packet_4 = {
    0xf6, 0x03, 0x47, 0x00, 0x08, 0xc1, 0x7a, 0xa1, 0x5b, 0xa8, 0x32, 0x89,
    0xd3, 0x96, 0xda, 0xa5, 0x25, 0xb5, 0xf9, 0x1b, 0x08, 0x3a, 0x92, 0x80,
    0x7f, 0x32, 0xb5, 0x5a, 0xea, 0xf6, 0xb0, 0xbe, 0x06, 0x75, 0x3e, 0xa9,
    0x5a, 0xce, 0x51, 0xa1, 0xc3, 0xa7, 0xb0, 0x61, 0xc5, 0x5c, 0x80, 0x59,
    0x52, 0x8d, 0x9c, 0x58, 0x48, 0x24, 0x1b, 0x7e, 0x54, 0xef, 0xdf, 0x72,
    0x71, 0xda, 0x98, 0xf4, 0xe8, 0x9f, 0xec, 0xa5, 0x9d, 0x2e, 0x32, 0x35,
    0xb8, 0xc8, 0x78, 0x03, 0xe9, 0xa6, 0xa7, 0x6b, 0x46, 0x87, 0xc3, 0x45,
    0xae, 0x35, 0xa4, 0x5f, 0x9e, 0xff, 0x00, 0xbf, 0xa6, 0x9b, 0xc2, 0xfc,
    0xa9, 0xf1, 0x13, 0x97, 0xbc, 0x4a, 0x32, 0x2e, 0xd4, 0x0b, 0xb4, 0xa0,
    0x11, 0xa7, 0xcc, 0xd3, 0x89, 0x52, 0x94, 0x94, 0x8b, 0x00, 0x2f, 0x72,
    0x3d, 0xb1, 0x36, 0xeb, 0x1c, 0x07, 0x28, 0xd3, 0x6b, 0xb4, 0xc9, 0x4e,
    0x15, 0x25, 0x82, 0xb5, 0x31, 0xa4, 0xdd, 0x29, 0x4a, 0x9c, 0x43, 0xa9,
    0x23, 0xfc, 0xaa, 0xfd, 0xf1, 0xbb, 0x2e, 0x55, 0x6a, 0x87, 0x1a, 0xbc,
    0x89, 0x20, 0x29, 0x69, 0x63, 0x5a, 0x52, 0xb0, 0x6c, 0x51, 0xa1, 0x40,
    0xf1, 0xea, 0x06, 0x06, 0x67, 0xc9, 0x92, 0x33, 0x14, 0x36, 0xe4, 0xcb,
    0x5f, 0x8a, 0xec, 0x9a, 0x6c, 0x7b, 0xab, 0x48, 0x04, 0xff, 0x00, 0x62,
    0x11, 0xfb, 0x14, 0xe2, 0xb3, 0x88, 0x78, 0x7f, 0x01, 0x26, 0xb3, 0xa8,
    0x03, 0xe9, 0xfa, 0x46, 0x4f, 0x85, 0x0c, 0x3c, 0xa9, 0xba, 0xd4, 0xba,
    0x45, 0xc2, 0xc5, 0xc5, 0xb5, 0xe2, 0xb1, 0xb7, 0xce, 0xde, 0x71, 0x36,
    0xa6, 0x4b, 0xaa, 0xbf, 0x20, 0x98, 0xaf, 0x68, 0x69, 0xb4, 0xdf, 0x4d,
    0xb6, 0xb0, 0xf6, 0xe4, 0x93, 0x87, 0x95, 0xaa, 0x38, 0xa5, 0x44, 0x4a,
    0x81, 0xf8, 0xd4, 0xeb, 0x32, 0x1d, 0x1b, 0x21, 0x49, 0x36, 0x29, 0xb0,
    0xf5, 0x1b, 0xdc, 0xfd, 0x30, 0x0f, 0x2d, 0xd1, 0xd3, 0x0d, 0xa6, 0xde,
    0xa8, 0xb8, 0x12, 0xb0, 0x2f, 0x60, 0x0a, 0x83, 0x7b, 0x6f, 0x6b, 0x72,
    0xac, 0x36, 0xd2, 0xa8, 0x46, 0xbb, 0x1e, 0x4c, 0xa8, 0x92, 0x22, 0xa0,
    0x34, 0xc9, 0x71, 0xa6, 0x0b, 0xa1, 0x4a, 0x3d, 0xbc, 0xd6, 0x3b, 0x2b,
    0x93, 0x7e, 0x01, 0xdb, 0x9c, 0x28, 0x4c, 0xad, 0x2e, 0x2a, 0xc8, 0x02,
    0xdc, 0xe2, 0xc1, 0x46, 0x2f, 0x50, 0x25, 0x7f, 0x11, 0x50, 0x75, 0x5c,
    0x4a, 0xff, 0x00, 0x16, 0xef, 0xa7, 0x5b, 0xe9, 0xd3, 0x6d, 0xf3, 0xc8,
    0x6e, 0xf4, 0xff, 0x00, 0x3d, 0x4c, 0xca, 0xf9, 0x88, 0x7c, 0x2c, 0x1d,
    0x46, 0x63, 0x29, 0x6c, 0xa4, 0x27, 0x40, 0xb2, 0x38, 0x59, 0xf5, 0x27,
    0x7f, 0xd7, 0x1d, 0x08, 0xce, 0x65, 0x6e, 0xa7, 0x54, 0xa7, 0x4f, 0xac,
    0xa6, 0x3c, 0x79, 0x29, 0x7e, 0x13, 0xc8, 0x49, 0x50, 0x05, 0x45, 0xd0,
    0xa4, 0xad, 0x36, 0xfc, 0xdc, 0x03, 0x8e, 0x7d, 0x32, 0xa0, 0x57, 0xa9,
    0x52, 0x6a, 0xeb, 0x52, 0x19, 0xaa, 0x46, 0x8a, 0x52, 0x96, 0xb4, 0x24,
    0x2d, 0xc2, 0x15, 0xdf, 0xe8, 0x07, 0x1c, 0x91, 0xfa, 0x63, 0xa0, 0xf3,
    0x25, 0x39, 0x0e, 0x51, 0x29, 0x12, 0xc2, 0x59, 0xb3, 0x52, 0x60, 0x91,
    0x74, 0x0b, 0xdb, 0xc3, 0x3d, 0xff, 0x00, 0x4c, 0x14, 0xa5, 0x29, 0x69,
    0x70, 0x24, 0x69, 0xcb, 0xce, 0x13, 0xa6, 0xa7, 0x45, 0x49, 0xf7, 0x5d,
    0x75, 0x20, 0x29, 0x79, 0x65, 0xb0, 0xd2, 0xde, 0x62, 0x21, 0xff, 0x00,
    0x88, 0xf6, 0x9b, 0xa9, 0x52, 0x2a, 0xb2, 0xa1, 0x21, 0x6d, 0x21, 0x55,
    0x37, 0x96, 0xce, 0xa1, 0xa4, 0xe8, 0x52, 0xc1, 0xbe, 0x9e, 0xc2, 0xd7,
    0xdb, 0x12, 0x9e, 0x9c, 0xd2, 0xe6, 0xd5, 0x5f, 0x31, 0xe1, 0x34, 0xeb,
    0xce, 0xa9, 0x77, 0x5a, 0x95, 0xe6, 0x48, 0x48, 0x16, 0x25, 0x46, 0xf6,
    0x03, 0x91, 0x8b, 0xa6, 0x64, 0xa0, 0x53, 0xb3, 0x35, 0x3e, 0x43, 0x55,
    0x17, 0x8b, 0x4d, 0x25, 0xe2, 0xe2, 0x8a, 0x57, 0xa7, 0x6b, 0x9b, 0x8b,
    0x8c, 0x20, 0x55, 0xf3, 0xad, 0x03, 0x2c, 0x43, 0x5e, 0x5a, 0xca, 0x8c,
    0xc6, 0x6e, 0xc7, 0x77, 0xd1, 0x65, 0x24, 0x2b, 0xde, 0xff, 0x00, 0x3e,
    0x29, 0x53, 0xb4, 0xa6, 0x9e, 0x71, 0x2f, 0xbc, 0xab, 0x21, 0x09, 0x00,
    0xff, 0x00, 0x3f, 0x66, 0x3e, 0x6d, 0xa0, 0xe2, 0x09, 0xa6, 0xd8, 0x5c,
    0x8c, 0x8b, 0x65, 0x4f, 0x95, 0xa8, 0xdf, 0xf2, 0xa4, 0x13, 0xaf, 0xcf,
    0x28, 0x6c, 0x8e, 0xec, 0x2c, 0xa7, 0x0d, 0x98, 0xad, 0x3d, 0xfc, 0x46,
    0xa5, 0x21, 0x21, 0xa4, 0x45, 0x8e, 0x92, 0x55, 0xb7, 0x01, 0x29, 0xf4,
    0xdc, 0xf9, 0x8f, 0xfa, 0x63, 0xe2, 0xa1, 0x42, 0xcc, 0xf5, 0x67, 0x04,
    0xfa, 0xac, 0x96, 0xd8, 0x75, 0x2b, 0x40, 0x6a, 0x1b, 0x67, 0x59, 0x66,
    0xff, 0x00, 0x4f, 0x99, 0x7f, 0xb0, 0xf7, 0xc0, 0x8c, 0x9b, 0x51, 0xa7,
    0x46, 0xa6, 0xaa, 0x5c, 0xe9, 0xaf, 0x3d, 0x52, 0x99, 0xaf, 0xc5, 0x74,
    0x2e, 0xca, 0x52, 0x38, 0xb1, 0x3f, 0x30, 0x16, 0xbe, 0xc2, 0xc3, 0x7c,
    0x51, 0x1e, 0xcd, 0xd4, 0xc5, 0x25, 0xa8, 0xca, 0x9d, 0x09, 0x98, 0xcd,
    0xa5, 0xb6, 0x8a, 0x84, 0x93, 0xad, 0x56, 0x1c, 0x9b, 0x26, 0xfe, 0xbb,
    0x8e, 0x6f, 0x85, 0xaa, 0x96, 0x21, 0x2a, 0x4f, 0xe1, 0xa4, 0xc7, 0x0a,
    0x34, 0xbe, 0xe7, 0xa7, 0x21, 0xef, 0x0e, 0x94, 0x6c, 0x22, 0xdc, 0xab,
    0x9f, 0x8d, 0x9d, 0x3d, 0xac, 0xc1, 0xcc, 0x93, 0x98, 0x1d, 0x07, 0x84,
    0x44, 0x73, 0x33, 0xd3, 0xe9, 0x30, 0x26, 0x95, 0xbb, 0xe3, 0x04, 0xb6,
    0x94, 0x3e, 0x82, 0x2d, 0x72, 0x97, 0x08, 0xbe, 0xde, 0x84, 0x1c, 0x51,
    0xba, 0x0d, 0x3d, 0x72, 0x29, 0x25, 0x6c, 0x48, 0x5b, 0x5a, 0x96, 0x97,
    0x2e, 0xa2, 0x12, 0x1b, 0x00, 0x8b, 0xf3, 0x7b, 0xdf, 0xb6, 0x17, 0xb3,
    0x4c, 0x5a, 0x6d, 0x43, 0x2d, 0x66, 0x37, 0x20, 0xba, 0x99, 0x29, 0x44,
    0x67, 0x9c, 0x2e, 0x24, 0x6c, 0x56, 0x97, 0x8a, 0xbf, 0xf8, 0xd8, 0xfd,
    0xf1, 0xf5, 0xf8, 0x7d, 0x6c, 0x38, 0x66, 0x43, 0x4a, 0x1d, 0xf0, 0x8b,
    0xc1, 0xd0, 0x7c, 0x3b, 0xa5, 0x28, 0xd3, 0x7b, 0x0b, 0x5f, 0xbe, 0xd6,
    0x27, 0x1e, 0x63, 0x04, 0x36, 0x4b, 0x2e, 0xb7, 0xba, 0x47, 0xb4, 0x3e,
    0xcf, 0xa5, 0x60, 0xa0, 0xab, 0x74, 0x88, 0x1f, 0x4d, 0x6a, 0x75, 0x5a,
    0x85, 0x98, 0xe0, 0xb0, 0x94, 0x95, 0x49, 0x75, 0xb6, 0xc0, 0xe6, 0xc0,
    0x5a, 0xdb, 0xfa, 0x5b, 0x9c, 0x13, 0xc9, 0xf5, 0xea, 0x5f, 0x4b, 0xa2,
    0x4c, 0xa8, 0xc4, 0x9e, 0xc2, 0x6b, 0x13, 0x9b, 0x51, 0x54, 0x94, 0xb4,
    0x16, 0xea, 0x49, 0x04, 0x14, 0xb6, 0x54, 0x3c, 0xa9, 0x17, 0xed, 0xb9,
    0x3e, 0x98, 0x0f, 0x42, 0x62, 0xa9, 0x22, 0x74, 0xb8, 0x14, 0xe9, 0x11,
    0x23, 0xc7, 0x92, 0xe0, 0x52, 0xca, 0x95, 0xa0, 0x71, 0x62, 0x9d, 0x86,
    0xff, 0x00, 0x41, 0x6b, 0xf0, 0x4e, 0x08, 0x37, 0xd1, 0x25, 0x57, 0x67,
    0x99, 0x55, 0x4c, 0xd8, 0xb5, 0xb8, 0xad, 0x92, 0x94, 0x37, 0xa7, 0x48,
    0xec, 0x07, 0x60, 0x31, 0x96, 0x6b, 0x15, 0x52, 0x29, 0xf2, 0xad, 0x05,
    0xaa, 0xee, 0x84, 0x81, 0xa1, 0x36, 0xc8, 0x5e, 0xff, 0x00, 0x7e, 0xd1,
    0x3c, 0x6b, 0x0b, 0x55, 0xa7, 0x66, 0x5e, 0xba, 0xb8, 0x18, 0x52, 0xf8,
    0xb2, 0x20, 0x15, 0x1d, 0xb3, 0xfb, 0xf3, 0x3a, 0xb0, 0x21};

std::vector<uint8_t> l2cap_test_packet_5 = {
    0xf6, 0x03, 0x47, 0x00, 0x0a, 0xc1, 0x0a, 0x8d, 0x2e, 0xa1, 0x99, 0xd9,
    0x12, 0x98, 0xab, 0x13, 0x50, 0x64, 0x87, 0x9b, 0xf1, 0x1c, 0xb0, 0x48,
    0xd4, 0x76, 0x1f, 0xdd, 0x1e, 0xa7, 0xe8, 0x37, 0xbe, 0x1b, 0xe9, 0x35,
    0x68, 0x53, 0x99, 0x55, 0x32, 0x7c, 0x66, 0x51, 0x53, 0x41, 0x0e, 0x3c,
    0xeb, 0x29, 0x1e, 0x1b, 0x8d, 0x95, 0x69, 0x0b, 0xd8, 0xec, 0x49, 0x3f,
    0x4d, 0xb0, 0x73, 0x2e, 0xf4, 0x5f, 0x24, 0xd0, 0x90, 0xbf, 0x8e, 0xcd,
    0x32, 0x8b, 0x65, 0x43, 0xc4, 0x02, 0x62, 0x12, 0x92, 0x47, 0xaf, 0x9a,
    0xe7, 0xe9, 0x86, 0x58, 0x39, 0x6b, 0xa3, 0x14, 0x79, 0xc9, 0x9a, 0xe5,
    0x4a, 0x13, 0xb2, 0xda, 0x01, 0x21, 0xc7, 0x65, 0xeb, 0xb0, 0x1c, 0x6c,
    0x4e, 0x9b, 0x62, 0x73, 0x52, 0xc5, 0xcd, 0x4d, 0x3a, 0x57, 0xc2, 0xb5,
    0x93, 0xc9, 0x3f, 0xc6, 0x50, 0xf7, 0x23, 0x4e, 0x44, 0xab, 0x61, 0xb6,
    0x80, 0x00, 0x6c, 0x9f, 0xd8, 0x6b, 0xce, 0x1b, 0x24, 0x43, 0x99, 0x1a,
    0x65, 0x00, 0x38, 0xb5, 0x29, 0x25, 0xa5, 0xb4, 0xbb, 0x7a, 0xea, 0x40,
    0xfe, 0x98, 0xf0, 0xae, 0xcd, 0xc9, 0x3d, 0x32, 0x7a, 0xa7, 0x99, 0x24,
    0x31, 0x09, 0x35, 0x49, 0x48, 0x4a, 0x65, 0xc9, 0x2b, 0x25, 0xd2, 0x81,
    0xc2, 0x4a, 0xbf, 0x28, 0x3f, 0xdc, 0x4e, 0xe4, 0xf3, 0x7c, 0x22, 0x75,
    0xaf, 0xad, 0x59, 0x7e, 0x9b, 0x0e, 0x9f, 0x13, 0x2c, 0x4c, 0x6e, 0x6c,
    0xf2, 0xfa, 0x34, 0xe8, 0x21, 0xd0, 0x11, 0xbe, 0xa2, 0x42, 0x4d, 0xf7,
    0xd8, 0x0e, 0x09, 0x3c, 0x71, 0x88, 0xd6, 0x70, 0xcd, 0x31, 0xf3, 0x25,
    0x41, 0x2e, 0xd5, 0x68, 0x95, 0x29, 0xe9, 0x4a, 0x8a, 0xc3, 0x0b, 0x49,
    0x4a, 0x50, 0xa3, 0xed, 0xdc, 0xf6, 0xb9, 0xdf, 0x18, 0xf0, 0xa6, 0x0d,
    0xa8, 0x55, 0x07, 0x6a, 0xb5, 0x76, 0x4d, 0xd8, 0x85, 0x0c, 0xee, 0x41,
    0x37, 0xb5, 0xad, 0xa1, 0xf1, 0xcb, 0x2d, 0xe3, 0xde, 0xa9, 0x5b, 0x6a,
    0x59, 0x09, 0x41, 0x69, 0x4e, 0x2f, 0x50, 0x00, 0x16, 0xbe, 0xd7, 0x37,
    0xca, 0xc3, 0x61, 0x99, 0xbe, 0xd0, 0xe3, 0x37, 0xac, 0xc9, 0xcd, 0xb9,
    0x8d, 0x72, 0x65, 0xca, 0x8d, 0x1a, 0x9f, 0x1d, 0xc0, 0x98, 0xa8, 0x20,
    0x0d, 0x42, 0xe4, 0x03, 0x63, 0xb8, 0x16, 0x37, 0xb7, 0x3f, 0xd0, 0x3d,
    0x2b, 0xa9, 0x52, 0xa8, 0xb5, 0x1b, 0xc2, 0x9e, 0xa9, 0x3e, 0x0a, 0xac,
    0xea, 0x96, 0x92, 0xef, 0x8e, 0x02, 0x48, 0xb6, 0xc3, 0x6d, 0xcf, 0x3e,
    0xc3, 0x1e, 0x39, 0x2a, 0x3c, 0x19, 0xad, 0x95, 0x46, 0xc8, 0xae, 0xb0,
    0x11, 0x75, 0x17, 0x9d, 0x6d, 0x94, 0x22, 0xd7, 0xb5, 0xb5, 0x38, 0x40,
    0xf6, 0xe4, 0x9f, 0x6c, 0x13, 0x55, 0x4e, 0xbe, 0x89, 0x0e, 0xb1, 0x0b,
    0x25, 0x25, 0xf0, 0xcb, 0xab, 0x6c, 0x3a, 0xd8, 0x6d, 0x68, 0x5d, 0xb7,
    0xd8, 0xa4, 0xd8, 0xd8, 0x11, 0x72, 0x0d, 0xb1, 0x5d, 0x63, 0x05, 0xca,
    0xa1, 0x29, 0x42, 0x5c, 0xb0, 0x1a, 0x00, 0x3f, 0x7b, 0xf9, 0x98, 0x4a,
    0xfe, 0xb9, 0x5b, 0x71, 0xc2, 0x96, 0xa4, 0x14, 0xa2, 0x79, 0xe5, 0x97,
    0x90, 0xb5, 0xba, 0x46, 0xf3, 0x75, 0x37, 0x73, 0x65, 0x1e, 0xb1, 0x53,
    0x96, 0xd2, 0x51, 0x25, 0xe8, 0x2a, 0x43, 0x89, 0x4b, 0x6a, 0x4e, 0xe1,
    0xb2, 0x01, 0x37, 0xee, 0x7f, 0x7b, 0x13, 0x8d, 0x2a, 0xad, 0x06, 0xac,
    0xed, 0x0f, 0x2f, 0xcd, 0x85, 0x49, 0x95, 0x22, 0x3b, 0xb4, 0xb6, 0x42,
    0x5c, 0x61, 0xa5, 0x2c, 0x13, 0xa4, 0x13, 0x7b, 0x0d, 0x8d, 0xcf, 0x7c,
    0x7d, 0x48, 0x77, 0xa8, 0xd2, 0xe9, 0xef, 0xf8, 0x39, 0x6d, 0x2c, 0xc7,
    0x28, 0x50, 0x71, 0x20, 0x04, 0xdd, 0x36, 0xdc, 0x12, 0x2e, 0x78, 0xf4,
    0xdf, 0x1e, 0x1d, 0x31, 0xce, 0x99, 0xaa, 0x9b, 0x96, 0xc5, 0x39, 0xfa,
    0xb5, 0x3a, 0x0e, 0x95, 0xa9, 0xa6, 0x93, 0xf1, 0x0a, 0x42, 0x96, 0x80,
    0x4d, 0xb5, 0x24, 0x0b, 0x0b, 0x5e, 0xc0, 0x92, 0x36, 0x03, 0x1a, 0xeb,
    0xf4, 0xd0, 0xe4, 0xa3, 0x52, 0xa8, 0x24, 0xf0, 0xee, 0x6c, 0x32, 0x1c,
    0xc9, 0xb0, 0x83, 0x38, 0x0e, 0x7a, 0xa5, 0x86, 0x2a, 0x53, 0x35, 0x2a,
    0x8b, 0x25, 0x01, 0xdd, 0x12, 0x02, 0x95, 0x9f, 0x44, 0x82, 0x72, 0x19,
    0xe9, 0x68, 0x13, 0x23, 0x20, 0x67, 0x9a, 0xa2, 0xd2, 0x5a, 0xa5, 0xaa,
    0x24, 0x72, 0x6c, 0x16, 0xf1, 0xd3, 0x72, 0x76, 0x16, 0xc3, 0x36, 0x42,
    0xe9, 0x8e, 0x68, 0xa5, 0x3a, 0xec, 0x31, 0x5e, 0xa2, 0x35, 0x2a, 0x45,
    0x99, 0x0d, 0xf8, 0xda, 0x9c, 0x4a, 0xb7, 0x16, 0x00, 0x7e, 0x6d, 0xef,
    0xf5, 0x03, 0x04, 0x17, 0x54, 0xcc, 0x09, 0x74, 0x3e, 0xac, 0xcd, 0x1d,
    0x85, 0x83, 0xa8, 0x29, 0x0b, 0x2a, 0x50, 0xf7, 0x1e, 0x71, 0x8d, 0x79,
    0xf9, 0x8e, 0xb7, 0xf0, 0x4e, 0xd3, 0x0e, 0x64, 0x68, 0xb6, 0xe5, 0xd0,
    0xa9, 0xa1, 0xd8, 0xed, 0xad, 0x77, 0x37, 0x2a, 0xb6, 0x92, 0xb2, 0x77,
    0x3c, 0xaa, 0xfe, 0xf8, 0x0a, 0xde, 0x16, 0x9c, 0x29, 0xe0, 0x20, 0x01,
    0xd4, 0x7d, 0x2f, 0x05, 0x2a, 0xbf, 0x13, 0xe8, 0x73, 0x93, 0x05, 0xd2,
    0x54, 0xe2, 0x8f, 0x81, 0x00, 0x0f, 0x00, 0x48, 0x19, 0x72, 0xd4, 0xc1,
    0x8c, 0xbf, 0xd0, 0xc9, 0x10, 0xa5, 0x2e, 0x54, 0x9c, 0xe1, 0x19, 0x84,
    0xa9, 0x65, 0x4a, 0x07, 0x65, 0x13, 0xde, 0xfa, 0xad, 0x7d, 0xfd, 0x71,
    0x4a, 0xcf, 0x79, 0x96, 0x83, 0x46, 0xca, 0xb1, 0x62, 0xbb, 0x5c, 0x8b,
    0x29, 0xf8, 0xcf, 0x32, 0xe2, 0xfc, 0x35, 0x02, 0x43, 0x6d, 0x24, 0xdd,
    0x4a, 0xb6, 0xc3, 0xf2, 0x8d, 0xf9, 0x27, 0x10, 0x77, 0x17, 0x45, 0xf1,
    0x94, 0x26, 0x66, 0xb9, 0x4f, 0xac, 0x9e, 0x4a, 0x91, 0x73, 0xf7, 0xb1,
    0xc7, 0xb5, 0x7b, 0x2a, 0x51, 0x6a, 0x79, 0x3e, 0x5c, 0xc8, 0x95, 0x19,
    0xcf, 0xa9, 0xa4, 0xeb, 0xb2, 0x9f, 0x51, 0x06, 0xde, 0xa3, 0x83, 0x82,
    0xac, 0xe1, 0xa9, 0x86, 0x7b, 0xe9, 0x29, 0xe2, 0xea, 0x4f, 0xd2, 0x03,
    0x2f, 0xe2, 0x75, 0x19, 0xa7, 0x12, 0x7b, 0x27, 0x2d, 0x71, 0x6e, 0xe8,
    0x02, 0xfb, 0x5c, 0xdc, 0x9b, 0x40, 0x2c, 0xcf, 0x57, 0xa5, 0xc9, 0x6e,
    0x75, 0x35, 0x55, 0xa5, 0xb1, 0x19, 0xc7, 0x54, 0x7c, 0x26, 0x93, 0x7b,
    0x79, 0x89, 0xb0, 0x3b, 0x6d, 0x85, 0x08, 0xd4, 0x6c, 0xa4, 0xf4, 0x84,
    0x80, 0x9a, 0x9c, 0x85, 0x13, 0x62, 0xb2, 0xda, 0x95, 0xfa, 0x01, 0x8e,
    0x87, 0x43, 0xf1, 0x29, 0x61, 0x8a, 0x76, 0x5c, 0xc9, 0xf4, 0x09, 0x50,
    0x62, 0x46, 0x6c, 0x27, 0xc6, 0x6c, 0xba, 0xe3, 0xde, 0x51, 0xa9, 0x45,
    0x77, 0xe4, 0x9b, 0xf0, 0x0f, 0xdf, 0x18, 0xff, 0x00, 0x54, 0x59, 0xa4,
    0x7f, 0x66, 0x3a, 0x74, 0xc4, 0x37, 0x00, 0xf9, 0xda, 0x4a, 0x54, 0x01,
    0xf6, 0xf2, 0x7f, 0xa8, 0x18, 0x2d, 0x3f, 0x3d, 0x34, 0x5b, 0x4a, 0x9e,
    0x65, 0x27, 0xc8, 0x9b, 0x75, 0x17, 0xfa, 0x43, 0xf4, 0xae, 0x1a, 0x69,
    0x94, 0x87, 0x99, 0x96, 0x48, 0x0b, 0xef, 0x1b, 0x01, 0xbe, 0xe7, 0x33,
    0x6f, 0x41, 0x12, 0x68, 0xb9, 0x4d, 0x86, 0x23, 0x7c, 0x44, 0x6a, 0x0d,
    0x5a, 0x53, 0x62, 0xca, 0xf2, 0xc5, 0x74, 0x5c, 0x7d, 0x54, 0x9b, 0x71,
    0xef, 0x82, 0xf0, 0x32, 0x95, 0x69, 0x6f, 0x29, 0x11, 0xb2, 0x24, 0x97,
    0xd2, 0x4d, 0xd2, 0x5f, 0x6c, 0xa4, 0xe9, 0x3c, 0x96, 0xae};

std::vector<uint8_t> l2cap_test_packet_6 = {
    0xf6, 0x03, 0x47, 0x00, 0x0c, 0xc1, 0x5c, 0x0b, 0x8c, 0x33, 0x2b, 0xac,
    0x14, 0x79, 0x95, 0x37, 0x6a, 0x2b, 0xca, 0x2e, 0xbb, 0x31, 0x0b, 0x4b,
    0x8a, 0x2a, 0x71, 0xd2, 0x02, 0xbb, 0x10, 0x92, 0x42, 0x47, 0x1c, 0x01,
    0x82, 0x8b, 0xeb, 0x4e, 0x66, 0x76, 0x58, 0x71, 0x14, 0xe9, 0x11, 0x12,
    0xf0, 0xd4, 0x35, 0x35, 0x74, 0xda, 0xc3, 0x72, 0x49, 0x36, 0xda, 0xdc,
    0xe1, 0x69, 0x58, 0x8d, 0x6d, 0x9e, 0xe8, 0x48, 0xe8, 0x91, 0x1d, 0x84,
    0xc2, 0x19, 0x3c, 0x29, 0x09, 0x1e, 0x9f, 0x40, 0x60, 0x2b, 0x99, 0x1f,
    0xa9, 0x12, 0x68, 0x52, 0xa0, 0xc4, 0xa0, 0xc3, 0xa7, 0x46, 0x76, 0x3a,
    0xdb, 0x5a, 0x02, 0x14, 0x14, 0xa4, 0x11, 0xba, 0x45, 0xc5, 0x85, 0xfd,
    0x70, 0x13, 0xa5, 0xd9, 0x39, 0xd8, 0xcb, 0xf0, 0x5a, 0x9d, 0x32, 0x9a,
    0xea, 0xd6, 0x59, 0x90, 0x1a, 0x59, 0x2a, 0x4a, 0xfd, 0x2c, 0x38, 0xb5,
    0xb1, 0xd1, 0x7d, 0x1c, 0xcd, 0x33, 0xb3, 0x35, 0x0d, 0xea, 0x9c, 0x97,
    0x94, 0xb2, 0xd4, 0xa2, 0xc3, 0x8d, 0xad, 0x20, 0x8b, 0x69, 0x06, 0xe0,
    0x81, 0xf5, 0x18, 0x8b, 0xe7, 0x5f, 0x8a, 0xa4, 0x75, 0x36, 0xac, 0x23,
    0xad, 0x49, 0x40, 0x9e, 0x56, 0x50, 0xa2, 0x40, 0x20, 0x80, 0x47, 0x18,
    0x15, 0x58, 0x9f, 0x76, 0x71, 0xa4, 0xba, 0xb5, 0x5c, 0x79, 0x0f, 0xbd,
    0x20, 0x7c, 0xea, 0xbf, 0x11, 0x72, 0x75, 0x1f, 0x28, 0x97, 0x3a, 0xc5,
    0x52, 0xb5, 0x29, 0x70, 0x68, 0xb2, 0xdc, 0x8f, 0x31, 0x0d, 0x97, 0x95,
    0xe1, 0x91, 0x7d, 0xd4, 0x76, 0xbd, 0xf6, 0xb5, 0xb9, 0xde, 0xfb, 0x61,
    0x36, 0xb9, 0xfc, 0xcf, 0x47, 0x7c, 0xb5, 0x57, 0x7e, 0xb3, 0xc6, 0xea,
    0x4c, 0x85, 0x14, 0x1f, 0xdf, 0x0d, 0x9d, 0x3b, 0x8f, 0x36, 0x99, 0x9b,
    0xeb, 0x4b, 0x1a, 0x89, 0x6a, 0x31, 0xb1, 0x4f, 0x3a, 0x49, 0x2b, 0xb8,
    0x27, 0xd8, 0x7e, 0xa7, 0x1e, 0x39, 0x6b, 0xa9, 0x5f, 0x1c, 0xfb, 0xb1,
    0x6b, 0xd1, 0x14, 0xe2, 0x5c, 0x73, 0x50, 0x79, 0xb4, 0xa5, 0x6a, 0x48,
    0x1c, 0x02, 0x95, 0x6c, 0xa0, 0x2f, 0xc6, 0x1b, 0xa9, 0x32, 0xf2, 0x32,
    0xf2, 0x4d, 0xad, 0x49, 0x4f, 0x12, 0xaf, 0x9a, 0x80, 0x39, 0xf9, 0xe9,
    0x12, 0x49, 0xfa, 0xa5, 0x4c, 0x4f, 0xbc, 0x86, 0x87, 0x1b, 0x68, 0xb6,
    0x40, 0xd8, 0x8b, 0xf2, 0x84, 0x96, 0xaa, 0xce, 0xc8, 0x64, 0xc4, 0x65,
    0x87, 0x54, 0xca, 0xc5, 0x94, 0x84, 0xb2, 0x92, 0x55, 0xbd, 0xcf, 0x98,
    0xdc, 0xfe, 0xf8, 0xd9, 0x4b, 0xae, 0x34, 0x14, 0xf2, 0xe8, 0x0c, 0x21,
    0x29, 0xdc, 0xf8, 0x82, 0xc7, 0xf4, 0xbe, 0x2a, 0xa7, 0x28, 0xb1, 0x2d,
    0x46, 0xab, 0x95, 0xaa, 0x10, 0x52, 0xf2, 0x95, 0xe2, 0xa5, 0xa4, 0x37,
    0x76, 0xaf, 0x6e, 0x34, 0x9d, 0xc7, 0xaf, 0xd7, 0x08, 0x35, 0x18, 0xf2,
    0x20, 0xd4, 0x16, 0xc5, 0x5d, 0x2a, 0x65, 0xf0, 0x8b, 0x8f, 0x35, 0xc1,
    0xed, 0xa8, 0x11, 0xee, 0x38, 0xc7, 0x5a, 0x85, 0x4e, 0xa9, 0x4f, 0x1c,
    0x69, 0x42, 0x78, 0x79, 0x81, 0xf7, 0x6f, 0x94, 0x68, 0xa3, 0xcc, 0x53,
    0x6a, 0xaa, 0x2d, 0xa5, 0x44, 0x2f, 0x74, 0x9b, 0x83, 0xe3, 0xd6, 0x19,
    0xfa, 0x3f, 0x06, 0x8b, 0x5b, 0xcf, 0xf4, 0x09, 0x4f, 0xc0, 0x43, 0x5e,
    0x12, 0x1d, 0x79, 0x2d, 0xa1, 0x45, 0x20, 0xba, 0x84, 0x9d, 0x17, 0x23,
    0x7b, 0x03, 0x63, 0xf6, 0xb6, 0x3a, 0x0e, 0xbb, 0xd3, 0xa6, 0x44, 0x24,
    0x0c, 0xbd, 0x3d, 0xba, 0x6c, 0xb2, 0xd8, 0x21, 0xb5, 0x30, 0x85, 0xa1,
    0x47, 0xd4, 0x02, 0x2e, 0x2f, 0xed, 0x8e, 0x7e, 0xfc, 0x3c, 0x32, 0xb7,
    0xb3, 0xe5, 0x27, 0x72, 0xa4, 0x21, 0x52, 0x75, 0x90, 0x3e, 0x54, 0xe8,
    0x51, 0xfe, 0x98, 0x6c, 0xfc, 0x47, 0xe7, 0xdc, 0xd7, 0x94, 0xb3, 0xed,
    0x1d, 0xda, 0x05, 0x5d, 0x71, 0xdb, 0x11, 0x0a, 0x8b, 0x7a, 0x02, 0x9b,
    0x76, 0xe4, 0x6c, 0xb4, 0x91, 0xb8, 0xed, 0x89, 0xee, 0x39, 0x99, 0xaa,
    0xb9, 0x88, 0xd8, 0x44, 0x8b, 0xe5, 0xb2, 0x5a, 0x0a, 0xf0, 0xdf, 0x23,
    0xcf, 0x6d, 0x61, 0xef, 0x0d, 0x38, 0xdc, 0x9d, 0x35, 0xc5, 0x9f, 0xca,
    0xb2, 0x0e, 0xe6, 0xd7, 0x00, 0x7c, 0xe0, 0x1f, 0x50, 0x8f, 0x55, 0xf2,
    0xa3, 0xaa, 0x45, 0x4a, 0x04, 0x79, 0xb0, 0x55, 0xb8, 0x91, 0x19, 0x8d,
    0x4d, 0xaa, 0xde, 0xa3, 0x80, 0x47, 0xd3, 0x09, 0xea, 0xae, 0xf5, 0x11,
    0x6d, 0xa5, 0xc7, 0x1c, 0x99, 0x15, 0xbd, 0x65, 0x01, 0x49, 0x8b, 0xa0,
    0x6a, 0xb5, 0xf4, 0x8b, 0x27, 0x9d, 0x8f, 0xe9, 0x8b, 0x87, 0x4c, 0xba,
    0xd9, 0x4b, 0xce, 0x4d, 0xff, 0x00, 0x07, 0xcc, 0x94, 0x71, 0x16, 0x72,
    0x80, 0x05, 0x6d, 0x5d, 0x71, 0xdd, 0x27, 0x61, 0x71, 0xc8, 0x37, 0x3d,
    0xef, 0xcf, 0x38, 0x79, 0x9f, 0x93, 0x8c, 0x58, 0x4e, 0xd4, 0x68, 0xf2,
    0x4b, 0x4c, 0xdb, 0x53, 0x8c, 0x2c, 0x6a, 0x09, 0x23, 0x6f, 0x29, 0x3b,
    0x8f, 0x4f, 0xbe, 0x04, 0x9c, 0x77, 0x52, 0x92, 0x58, 0x97, 0xa9, 0xdd,
    0x2a, 0xe6, 0x0e, 0x47, 0xd2, 0x0e, 0x25, 0x2e, 0x4c, 0x27, 0x8d, 0xa7,
    0x8d, 0x8e, 0xd7, 0xb7, 0x95, 0xf5, 0xf5, 0xf5, 0x89, 0x1f, 0x47, 0xa4,
    0x67, 0x06, 0x27, 0x4a, 0x35, 0xc7, 0x25, 0x86, 0xd4, 0xd1, 0x08, 0xf1,
    0x52, 0x52, 0x1c, 0xba, 0x55, 0x71, 0xbf, 0xd0, 0x1f, 0xbe, 0x00, 0xaf,
    0x5c, 0xa8, 0x5a, 0x94, 0x10, 0x5a, 0x84, 0xcb, 0x81, 0x5c, 0x27, 0x48,
    0xf1, 0xd4, 0x3e, 0xe4, 0x95, 0x7b, 0x9f, 0xd3, 0x14, 0x2f, 0xe6, 0xaa,
    0x6d, 0x4d, 0x6c, 0xd3, 0xdb, 0x42, 0x5a, 0x79, 0x32, 0x1d, 0x2a, 0x4a,
    0x2e, 0x92, 0x2c, 0x40, 0xb6, 0xae, 0x49, 0xb0, 0xb9, 0x3e, 0xf8, 0x47,
    0xcc, 0xb4, 0xc3, 0x4a, 0x6d, 0x6b, 0x2e, 0x25, 0xc2, 0xe4, 0xa7, 0xc1,
    0x08, 0xb8, 0x03, 0x4b, 0xa4, 0xff, 0x00, 0xa1, 0x18, 0x6f, 0xa9, 0xce,
    0x2e, 0x6a, 0x8a, 0xda, 0x9c, 0x37, 0x25, 0x47, 0xdb, 0xf9, 0x87, 0x1c,
    0x20, 0xa3, 0xda, 0x25, 0xc3, 0x9a, 0x80, 0x56, 0xb9, 0x9d, 0xb7, 0x85,
    0x77, 0xe9, 0xb4, 0xfa, 0xab, 0x28, 0x11, 0xe7, 0x2e, 0x39, 0x24, 0x85,
    0xad, 0x06, 0xc1, 0x06, 0xfe, 0xd7, 0xfb, 0xed, 0x82, 0x39, 0x17, 0x2c,
    0xd0, 0xd9, 0x9e, 0xf4, 0x3c, 0xcf, 0x54, 0x5f, 0x88, 0xbb, 0x18, 0x8b,
    0x5b, 0xc4, 0xb6, 0xbd, 0xf9, 0x0a, 0x1d, 0xfd, 0xb6, 0xf4, 0xb6, 0x25,
    0x94, 0xb9, 0x92, 0x62, 0x49, 0x75, 0xd8, 0xce, 0xa9, 0xb5, 0x29, 0xc5,
    0x6d, 0x7d, 0xb9, 0x3b, 0x1c, 0x51, 0x28, 0x4e, 0x48, 0xaa, 0xd2, 0x5e,
    0x79, 0xf8, 0xbe, 0x46, 0x52, 0x95, 0x2c, 0x84, 0xdd, 0x04, 0x15, 0x69,
    0xe7, 0xd6, 0xfd, 0xb0, 0x15, 0xd3, 0x31, 0x2c, 0x9b, 0x71, 0x92, 0x9e,
    0xb0, 0x2a, 0x5d, 0x9a, 0x26, 0x2b, 0x50, 0x2a, 0x68, 0x31, 0x30, 0xad,
    0xc0, 0x04, 0x28, 0xfb, 0x67, 0xd7, 0x3f, 0x13, 0x15, 0xe8, 0x99, 0x07,
    0x2d, 0x48, 0x6d, 0x0d, 0xc5, 0x61, 0x12, 0xdc, 0x5b, 0x6e, 0x5d, 0xa5,
    0x2f, 0xec, 0x13, 0xb8, 0x3c, 0x1b, 0x0d, 0xf7, 0xbd, 0xce, 0x05, 0xc1,
    0xa4, 0x39, 0x43, 0xca, 0x15, 0x0a, 0x3b, 0xad, 0x84, 0x2e, 0x3b, 0x6f,
    0xb4, 0x42, 0x56, 0x5c, 0x06, 0xc5, 0x44, 0x79, 0xac, 0x2e, 0x34, 0xdb,
    0x7b, 0x0c, 0x2b, 0x65, 0x9a, 0xcd, 0x47, 0x2d, 0x54, 0x92};

std::vector<uint8_t> l2cap_test_packet_7 = {
    0xf6, 0x03, 0x47, 0x00, 0x0e, 0xc1, 0xa0, 0xaa, 0x33, 0xea, 0x72, 0x1b,
    0xab, 0xb2, 0xa3, 0xea, 0x1a, 0xae, 0x3c, 0xd7, 0x07, 0xe6, 0xb5, 0xff,
    0x00, 0xec, 0x61, 0xd2, 0x7e, 0x67, 0x6b, 0x30, 0x49, 0x9a, 0xb0, 0x9f,
    0x04, 0x2c, 0x36, 0x1f, 0xb6, 0xc0, 0x21, 0x68, 0x20, 0x8d, 0xb8, 0xb5,
    0xb0, 0xc7, 0x84, 0x66, 0x82, 0xa7, 0x14, 0x39, 0xa4, 0xfd, 0x22, 0x3f,
    0xf1, 0x5f, 0x0e, 0xcc, 0x52, 0x24, 0xd0, 0xdb, 0x80, 0x1b, 0x2d, 0x26,
    0xe3, 0x96, 0x63, 0xe7, 0x1a, 0xf4, 0xf9, 0xb2, 0xe3, 0xe5, 0xb9, 0x2f,
    0xb6, 0x99, 0x09, 0x53, 0x71, 0xf4, 0x25, 0xc4, 0x70, 0x8b, 0x2b, 0x7b,
    0xff, 0x00, 0x97, 0xd3, 0x08, 0x59, 0x5f, 0xab, 0x71, 0xd2, 0xa5, 0xc5,
    0xae, 0x53, 0x64, 0x25, 0xbd, 0x56, 0xf8, 0x86, 0x5e, 0x2a, 0x5a, 0x2c,
    0x6f, 0x7b, 0x1e, 0x37, 0xdf, 0x6c, 0x58, 0xd2, 0xdd, 0x2e, 0xa5, 0xd2,
    0x9a, 0x1f, 0xf0, 0xc2, 0xcf, 0xc5, 0xae, 0x0b, 0xc2, 0x62, 0x19, 0xf9,
    0x8d, 0xd3, 0xb2, 0x96, 0x3e, 0xa9, 0xef, 0x8e, 0x46, 0x7d, 0x6b, 0x8b,
    0x52, 0x93, 0x09, 0xc4, 0x95, 0xa1, 0x0f, 0xad, 0x36, 0x1c, 0xde, 0xe4,
    0x6d, 0x83, 0x75, 0x4a, 0x93, 0xe8, 0x29, 0x71, 0xbc, 0xb6, 0x3e, 0x36,
    0x8a, 0x83, 0xd5, 0xe9, 0x96, 0x29, 0xb2, 0x4e, 0xb2, 0x78, 0x41, 0x4d,
    0x88, 0x23, 0x5b, 0x00, 0x05, 0xf7, 0xd8, 0xe9, 0x1d, 0x04, 0xec, 0xac,
    0xbb, 0x9a, 0xa0, 0xae, 0x45, 0x3a, 0xb4, 0x15, 0x2d, 0xb2, 0x16, 0xd1,
    0x69, 0x94, 0xb6, 0xf2, 0x8d, 0xff, 0x00, 0x38, 0xb5, 0x96, 0x07, 0x3c,
    0x5f, 0x1f, 0x12, 0x8e, 0x69, 0x0d, 0x33, 0x26, 0xa0, 0x83, 0x52, 0x84,
    0x96, 0xf4, 0x78, 0x91, 0xd0, 0x94, 0x92, 0x9b, 0x8f, 0x99, 0x36, 0xb1,
    0xe3, 0x8d, 0xb0, 0x1f, 0xa4, 0x19, 0x3e, 0x7c, 0x69, 0x10, 0xf3, 0x0b,
    0x8c, 0x7c, 0x38, 0x68, 0x97, 0x63, 0xb0, 0xf1, 0xf3, 0xba, 0x46, 0xe0,
    0x1e, 0x39, 0xf4, 0xed, 0x83, 0x15, 0xac, 0xf6, 0xed, 0x26, 0x24, 0x88,
    0x34, 0x94, 0xaf, 0xc6, 0xd4, 0x4b, 0xcb, 0x6d, 0x37, 0xd1, 0x7d, 0xac,
    0x3d, 0xf9, 0xdb, 0x9c, 0x63, 0x5c, 0x8c, 0xac, 0xcc, 0xa9, 0x76, 0x75,
    0x01, 0xbe, 0x56, 0xca, 0xfe, 0x50, 0xc0, 0xd2, 0x18, 0x9a, 0x94, 0x33,
    0x33, 0xcd, 0x04, 0x11, 0xbd, 0xed, 0x97, 0x80, 0xd7, 0xa0, 0xb1, 0x31,
    0x7b, 0xe9, 0x12, 0xe8, 0xec, 0xe5, 0x1b, 0xd3, 0x65, 0x25, 0x2b, 0x7e,
    0x38, 0x94, 0xb2, 0xa5, 0x15, 0x0b, 0x8d, 0x8d, 0xc6, 0xd6, 0x20, 0x1d,
    0xc0, 0xe0, 0x8c, 0x47, 0xfa, 0xd7, 0x35, 0x11, 0x33, 0xed, 0x55, 0xe7,
    0x8b, 0x6a, 0xd4, 0x84, 0xb8, 0x46, 0xab, 0x0b, 0x80, 0x47, 0xef, 0x6c,
    0x6c, 0xf4, 0x0a, 0x53, 0xf3, 0xe4, 0xd4, 0x59, 0xbc, 0x86, 0x5b, 0x4c,
    0x72, 0xd8, 0x4b, 0xbb, 0x06, 0xf5, 0x21, 0x57, 0x1f, 0x5b, 0x81, 0x7c,
    0x2a, 0x7e, 0x26, 0x9b, 0x7e, 0x36, 0x64, 0x8b, 0xe6, 0x4b, 0xa8, 0x98,
    0xd2, 0x3c, 0xac, 0x82, 0xa2, 0xa0, 0x9b, 0x77, 0x23, 0x6d, 0xf7, 0xc2,
    0x83, 0xbc, 0x0b, 0x68, 0x34, 0x9d, 0x02, 0xb2, 0xe9, 0x09, 0xce, 0x80,
    0x86, 0x94, 0xb4, 0x1b, 0x8b, 0x9d, 0x75, 0xb5, 0xf2, 0x8d, 0x18, 0x68,
    0x47, 0xf3, 0x58, 0x60, 0x04, 0x24, 0x96, 0x46, 0xab, 0xaa, 0xdb, 0x59,
    0x69, 0x23, 0x7e, 0xe4, 0xf6, 0x18, 0x8c, 0x65, 0x88, 0x92, 0xe5, 0xd7,
    0xfe, 0x12, 0x23, 0x0b, 0x79, 0xd5, 0x28, 0xa7, 0x4a, 0x47, 0xbf, 0xed,
    0x8b, 0x58, 0x8b, 0x1d, 0x59, 0xe9, 0xb6, 0xd6, 0xa0, 0xd2, 0x1c, 0x80,
    0x95, 0x6c, 0x6c, 0x51, 0xe6, 0x26, 0xff, 0x00, 0x6e, 0x70, 0x0e, 0xa7,
    0x99, 0x72, 0xbe, 0x49, 0x79, 0xf6, 0x29, 0x30, 0x99, 0x9c, 0xe3, 0xaa,
    0xd6, 0xe0, 0x4a, 0x8a, 0x75, 0x2b, 0x7b, 0x82, 0x46, 0xf6, 0x3d, 0xc6,
    0x1e, 0xa4, 0xe4, 0x91, 0x35, 0x4c, 0x61, 0x4e, 0x2a, 0xc9, 0x48, 0x37,
    0xf5, 0x88, 0xd4, 0xe4, 0xeb, 0xb2, 0xb5, 0x59, 0x86, 0xd8, 0x41, 0x5a,
    0xd4, 0x13, 0x6e, 0x43, 0x5c, 0xc9, 0xf3, 0x86, 0x4c, 0xb5, 0x16, 0x9f,
    0x93, 0x61, 0x8a, 0x85, 0x46, 0xa4, 0xc7, 0x8e, 0xa4, 0xa8, 0x2d, 0x09,
    0x57, 0x97, 0x56, 0xdb, 0x7b, 0x9d, 0xb0, 0xbd, 0x9d, 0x69, 0xd5, 0x4c,
    0xc7, 0x25, 0xea, 0xab, 0x2c, 0x06, 0xda, 0x6e, 0x3a, 0xec, 0x93, 0xf3,
    0x2a, 0xf6, 0x55, 0xcd, 0xb8, 0xe0, 0x58, 0x5c, 0xfb, 0xde, 0xf8, 0x9b,
    0xe6, 0x0c, 0xd3, 0x2b, 0x30, 0xd5, 0x05, 0x4a, 0xa4, 0xab, 0xbf, 0xa9,
    0x21, 0x0d, 0x36, 0x90, 0x96, 0x9a, 0x48, 0xb5, 0x92, 0x00, 0x1e, 0x82,
    0xd8, 0x7d, 0x9d, 0xd4, 0x5a, 0x62, 0xe8, 0x72, 0x22, 0xc7, 0x6d, 0xe5,
    0x3a, 0xe4, 0x52, 0xd0, 0x48, 0x05, 0x37, 0x51, 0xb0, 0xd4, 0x3d, 0x3b,
    0xfe, 0xd8, 0xc7, 0x51, 0x9f, 0x79, 0xf6, 0x44, 0xac, 0x93, 0x47, 0x83,
    0xa1, 0x37, 0xb7, 0xdf, 0x58, 0xdf, 0x45, 0xa1, 0xa1, 0x89, 0x83, 0x3f,
    0x38, 0xbe, 0x27, 0x8f, 0x90, 0x17, 0xe5, 0x19, 0xf8, 0x7e, 0xa9, 0x7f,
    0x0c, 0xcf, 0x74, 0xe6, 0x41, 0x4a, 0xd2, 0xeb, 0xce, 0xb3, 0xa8, 0xde,
    0xe0, 0x68, 0x5e, 0xff, 0x00, 0xed, 0x86, 0xef, 0xc4, 0xae, 0x57, 0xad,
    0x66, 0x9c, 0xf5, 0x96, 0xa9, 0x99, 0x7a, 0x9d, 0x22, 0xa3, 0x2d, 0xe8,
    0xaa, 0x40, 0x08, 0x48, 0x09, 0x16, 0x37, 0x25, 0x44, 0xec, 0x91, 0x6d,
    0xee, 0x48, 0x18, 0x40, 0xe9, 0x72, 0x97, 0x1f, 0x38, 0xd0, 0xe4, 0x4c,
    0x01, 0x82, 0xed, 0x49, 0xc4, 0x5d, 0xc5, 0x00, 0x2e, 0xa4, 0xe9, 0x1b,
    0xf1, 0xc9, 0xb6, 0x3a, 0x7f, 0x38, 0x67, 0x5a, 0x2e, 0x59, 0xa6, 0xa5,
    0xe0, 0x17, 0x2e, 0xa6, 0xb8, 0xfa, 0x52, 0xdb, 0x00, 0xd8, 0x26, 0xfb,
    0x6b, 0x20, 0x5c, 0xef, 0xc2, 0x45, 0xf0, 0x93, 0x8e, 0x17, 0x35, 0x2b,
    0x5d, 0x93, 0x71, 0x96, 0x8a, 0xd6, 0x59, 0x02, 0xde, 0x3d, 0xe1, 0xed,
    0xbc, 0x50, 0xb0, 0xeb, 0x0d, 0xcd, 0x53, 0xdf, 0x6c, 0x1d, 0x56, 0x6f,
    0xe5, 0x63, 0xf4, 0x85, 0xbe, 0x94, 0xf4, 0x72, 0x85, 0xd3, 0xd8, 0x08,
    0xcc, 0xb9, 0x96, 0x4b, 0x33, 0xab, 0x2c, 0xd8, 0x87, 0x41, 0xbc, 0x68,
    0x8b, 0xec, 0x1b, 0x07, 0xff, 0x00, 0x51, 0x63, 0xfb, 0xdd, 0xbb, 0x0e,
    0xf8, 0xdb, 0xcc, 0xd9, 0xc6, 0x7d, 0x6e, 0x97, 0x22, 0x99, 0x97, 0xdc,
    0x30, 0xd2, 0xa6, 0xd4, 0x18, 0x23, 0xe7, 0x70, 0xff, 0x00, 0x7d, 0x44,
    0xf0, 0x2f, 0xe9, 0xbd, 0xed, 0xef, 0x88, 0x96, 0x6e, 0xcd, 0xdd, 0x45,
    0xce, 0x15, 0x44, 0x29, 0xea, 0x7c, 0xe4, 0x42, 0x6a, 0xe1, 0xa6, 0x49,
    0x08, 0xda, 0xe7, 0x7b, 0x5f, 0xcb, 0x7f, 0x41, 0xe9, 0xdf, 0x07, 0xa9,
    0xaf, 0xe7, 0x01, 0x14, 0xb6, 0xb8, 0x4d, 0x0f, 0x11, 0x21, 0x09, 0x40,
    0x76, 0xe5, 0x24, 0xef, 0x60, 0x07, 0xdf, 0x6c, 0x0f, 0x6b, 0x07, 0xd4,
    0xa6, 0xdd, 0x13, 0x53, 0xa8, 0x2b, 0x5e, 0xc2, 0xdd, 0xd4, 0xf4, 0x82,
    0xcd, 0x87, 0x52, 0x9e, 0xce, 0x55, 0xb2, 0x40, 0xde, 0xc7, 0xda, 0xff,
    0x00, 0x3f, 0xe6, 0x14, 0xb2, 0x2c, 0x09, 0x51, 0x33, 0x8c, 0x66, 0x96,
    0xe1, 0x7d, 0xc5, 0x2d, 0xc4, 0x94, 0x29, 0xcb, 0xf9, 0x86, 0xc6, 0xe4,
    0x5e, 0xdd, 0xf0, 0x56, 0x6c, 0xc9, 0x93, 0x95, 0xfa, 0xf6};

std::vector<uint8_t> l2cap_test_packet_8 = {
    0xf6, 0x03, 0x47, 0x00, 0x10, 0xc1, 0x53, 0x69, 0x65, 0xc7, 0x52, 0xcc,
    0xa7, 0x94, 0x02, 0xd5, 0xa8, 0xa0, 0x6b, 0xde, 0xde, 0xbd, 0xbf, 0x4c,
    0x39, 0xe4, 0xec, 0x9f, 0x57, 0x77, 0x33, 0x35, 0x50, 0x9f, 0x4e, 0x44,
    0x25, 0x34, 0xab, 0xa5, 0xd0, 0x8d, 0x3a, 0x81, 0xd8, 0x85, 0x71, 0xc7,
    0x37, 0xe4, 0x93, 0x84, 0x99, 0x2f, 0xb0, 0xd5, 0x72, 0xae, 0x88, 0xe8,
    0x7a, 0x4a, 0xdb, 0x98, 0xf0, 0x75, 0x09, 0x68, 0x9d, 0x0a, 0xd5, 0xb0,
    0xbf, 0x1b, 0xdb, 0x0e, 0x75, 0x19, 0x29, 0x84, 0x51, 0xd0, 0x85, 0xa0,
    0xf1, 0xf1, 0xde, 0xdb, 0xd8, 0x8f, 0xda, 0x1b, 0xb0, 0xc3, 0x8c, 0xc8,
    0x9f, 0xee, 0x94, 0x1b, 0x04, 0x2b, 0xfc, 0x8d, 0xb6, 0x1b, 0x9e, 0x86,
    0x13, 0x32, 0xee, 0x51, 0x95, 0x26, 0x62, 0x97, 0x2b, 0xca, 0x85, 0x38,
    0x74, 0x36, 0x8d, 0xd4, 0xab, 0x9d, 0xbe, 0x98, 0x69, 0xa8, 0xfc, 0x0d,
    0x1c, 0x22, 0x1c, 0x55, 0x32, 0x08, 0x1b, 0xa5, 0x0e, 0x6a, 0x08, 0xf5,
    0x04, 0x8e, 0x4f, 0x7d, 0xb1, 0xf9, 0x3a, 0x8f, 0x99, 0xea, 0x68, 0x4a,
    0x63, 0xc8, 0x85, 0x01, 0xbe, 0x74, 0x97, 0x81, 0xd3, 0x7f, 0x5b, 0x72,
    0x71, 0xa3, 0x1b, 0xa7, 0x33, 0x18, 0x92, 0xdb, 0x95, 0x3a, 0xd3, 0x84,
    0xba, 0x82, 0xa1, 0xe0, 0x46, 0x71, 0x6a, 0x29, 0xe3, 0xb0, 0x36, 0xc6,
    0x54, 0xe1, 0xea, 0x8c, 0xd0, 0x0b, 0x75, 0x36, 0x1c, 0xb2, 0x10, 0xa0,
    0xf7, 0xc4, 0x6c, 0x37, 0x86, 0xc1, 0x96, 0xa4, 0x1e, 0x35, 0x0c, 0x8a,
    0xc8, 0xb9, 0x3d, 0x32, 0xf6, 0xc8, 0x43, 0x2e, 0x4d, 0x99, 0x97, 0xd4,
    0x5c, 0x4d, 0x69, 0x97, 0x3c, 0x55, 0x0b, 0xa6, 0x42, 0xaf, 0x66, 0xc6,
    0x92, 0x6c, 0x12, 0x0d, 0xbe, 0xfe, 0xbd, 0xf0, 0xcf, 0x05, 0xa8, 0x4e,
    0xd6, 0x27, 0x1a, 0x5b, 0xc8, 0x2c, 0x08, 0x8d, 0x82, 0xb6, 0x55, 0x70,
    0xa5, 0x00, 0xaf, 0x5e, 0xdb, 0xe1, 0x61, 0x8e, 0x9b, 0x44, 0x0a, 0x68,
    0xbd, 0x5a, 0x7d, 0x69, 0x28, 0x05, 0x2a, 0x5a, 0xc8, 0xf2, 0xfa, 0x7b,
    0x7d, 0x30, 0xdd, 0x95, 0xa9, 0x54, 0xbc, 0xb6, 0xc4, 0x8b, 0xcf, 0x69,
    0x48, 0x5f, 0x9c, 0xf9, 0x49, 0x3b, 0x0b, 0x6e, 0x7b, 0xe1, 0x86, 0x83,
    0x40, 0x98, 0x91, 0x9b, 0x0f, 0x39, 0x60, 0x90, 0x0e, 0xf1, 0x1b, 0xc6,
    0xf8, 0xe6, 0x56, 0xbb, 0x28, 0xa4, 0xa5, 0x4a, 0x53, 0x84, 0x8d, 0x41,
    0xd8, 0xc6, 0x8e, 0x49, 0x2b, 0xfe, 0x58, 0xa4, 0x2c, 0x3a, 0xb4, 0x07,
    0x1a, 0x2d, 0x2e, 0xcb, 0x29, 0xbe, 0x95, 0x91, 0xcf, 0xf9, 0x86, 0x34,
    0x1b, 0xc8, 0xf4, 0x8c, 0xaf, 0x5e, 0x7b, 0x30, 0xd7, 0x5a, 0x69, 0xe4,
    0x15, 0xf8, 0x89, 0x2e, 0x3a, 0x34, 0x81, 0xea, 0x07, 0xae, 0x0e, 0xf4,
    0xc1, 0xba, 0x5d, 0x63, 0xa7, 0xae, 0xc4, 0x55, 0x72, 0x1c, 0x19, 0xf0,
    0x66, 0x38, 0x02, 0x5f, 0x41, 0xf9, 0x54, 0x41, 0x1c, 0x6f, 0x6f, 0x70,
    0x0e, 0xe2, 0xc7, 0x1e, 0x95, 0x1c, 0xa7, 0x95, 0x6a, 0x8f, 0x78, 0x99,
    0xa7, 0x38, 0xbb, 0x31, 0x60, 0xec, 0xdc, 0x55, 0x04, 0xb2, 0x91, 0xed,
    0xab, 0x73, 0xf5, 0x23, 0x1e, 0xf3, 0x75, 0x49, 0x44, 0x23, 0x3e, 0xf2,
    0x92, 0x4d, 0x87, 0x99, 0xce, 0x3e, 0x82, 0xa0, 0x54, 0x25, 0x95, 0x47,
    0x96, 0xed, 0x52, 0x14, 0xb4, 0x0c, 0xae, 0x0e, 0x47, 0x63, 0x7d, 0xad,
    0x7f, 0x13, 0xf3, 0x08, 0x59, 0xa7, 0xa9, 0x90, 0xe7, 0xd4, 0x3f, 0x87,
    0xd3, 0xd6, 0x44, 0x73, 0xb1, 0x7d, 0x46, 0xca, 0xb7, 0x64, 0x8b, 0x7c,
    0xa3, 0x8d, 0x86, 0xe7, 0xb9, 0xed, 0x80, 0xd4, 0x1a, 0xac, 0x04, 0xb8,
    0xe1, 0x71, 0x0b, 0xd2, 0xb7, 0x09, 0xdc, 0xdf, 0x49, 0xdf, 0x6b, 0x1f,
    0xad, 0xcf, 0xed, 0x8a, 0xe2, 0x32, 0x5e, 0x40, 0x85, 0x1d, 0x89, 0x14,
    0xa8, 0x54, 0xea, 0xb8, 0x52, 0x94, 0x1c, 0x0f, 0x4f, 0x53, 0x3e, 0x1f,
    0xf7, 0x46, 0xc8, 0x37, 0x36, 0xbd, 0xed, 0xc6, 0x18, 0xa9, 0xf4, 0xfe,
    0x9a, 0xa6, 0x12, 0x14, 0xfd, 0x0c, 0x31, 0x25, 0x0a, 0x21, 0xe6, 0x58,
    0x43, 0xce, 0x83, 0xb6, 0xc5, 0x2e, 0x5b, 0x71, 0xfa, 0x61, 0x36, 0xa1,
    0x3e, 0x26, 0x94, 0x4b, 0xae, 0x0f, 0x33, 0x18, 0x26, 0x3b, 0x79, 0xb7,
    0x4a, 0x9e, 0x70, 0x78, 0x00, 0x32, 0x1e, 0xa7, 0xde, 0x37, 0x7a, 0x17,
    0xe0, 0x3b, 0x48, 0x42, 0x53, 0xe0, 0x3a, 0xfb, 0xa9, 0x5b, 0xee, 0x29,
    0xa1, 0xc9, 0x06, 0xc7, 0x9e, 0xc0, 0x5b, 0x02, 0x7f, 0x12, 0x50, 0xd2,
    0xdd, 0x7e, 0x93, 0x25, 0x20, 0x05, 0x29, 0x82, 0x90, 0x37, 0xb1, 0xb1,
    0x1e, 0x9f, 0x43, 0x86, 0xaa, 0x5d, 0x7f, 0x25, 0x51, 0x5d, 0x0b, 0xa7,
    0x50, 0xaa, 0x37, 0x00, 0x68, 0x09, 0x69, 0x66, 0xc3, 0xef, 0x84, 0xbe,
    0xa7, 0xe6, 0x98, 0xb9, 0xb2, 0xbf, 0x16, 0x33, 0x50, 0x64, 0xc2, 0xf8,
    0x26, 0xff, 0x00, 0xe7, 0x24, 0xa5, 0x4a, 0x4f, 0xae, 0xe3, 0x8d, 0xf9,
    0xe3, 0x6c, 0x60, 0x4b, 0x8c, 0xf6, 0x25, 0x01, 0x60, 0xaa, 0xf7, 0xc8,
    0xc6, 0x57, 0x18, 0xec, 0x90, 0x45, 0xf2, 0xf1, 0xb7, 0xca, 0xe6, 0x24,
    0x3d, 0x41, 0x97, 0x21, 0x1f, 0x0d, 0xf0, 0x4e, 0xa2, 0x3c, 0xc7, 0x96,
    0x4e, 0xa5, 0x10, 0x0f, 0x86, 0x05, 0x92, 0x8b, 0x9f, 0xca, 0x76, 0xfd,
    0x31, 0x3c, 0x10, 0xaa, 0xae, 0xbe, 0xaf, 0xfc, 0xad, 0x97, 0x97, 0x7d,
    0xce, 0x92, 0xa3, 0xfa, 0xe2, 0xae, 0x8a, 0x4c, 0x1a, 0xef, 0x50, 0x69,
    0x54, 0x9a, 0x83, 0xc9, 0x62, 0x3f, 0xc0, 0xba, 0xf1, 0x71, 0x36, 0x3a,
    0x52, 0x3e, 0x5d, 0xcf, 0xa0, 0x37, 0xbe, 0x07, 0xe6, 0x8c, 0x85, 0x9a,
    0xe3, 0xcf, 0x28, 0xc9, 0xd5, 0x29, 0xb5, 0x66, 0x42, 0x43, 0x88, 0x44,
    0x72, 0xa2, 0xbd, 0x2a, 0xbd, 0xb9, 0xdd, 0x5c, 0x1b, 0x73, 0xf5, 0xc5,
    0x33, 0x0d, 0xb6, 0xf3, 0x34, 0xd0, 0xe7, 0x19, 0xb1, 0xb9, 0x00, 0x01,
    0x90, 0xb9, 0x1a, 0xab, 0x2f, 0x28, 0x57, 0x55, 0x29, 0xb7, 0xc2, 0xa6,
    0x0a, 0x49, 0xd8, 0xdb, 0x5c, 0xa0, 0x05, 0x1f, 0x29, 0x54, 0xdd, 0xa7,
    0xa9, 0xe7, 0xe3, 0x53, 0xa2, 0x10, 0x9b, 0xb7, 0xe2, 0x4f, 0x09, 0x2e,
    0x0b, 0x5e, 0xe0, 0x04, 0x2a, 0xff, 0x00, 0xb6, 0x3c, 0x19, 0xcb, 0xd9,
    0xd9, 0x45, 0x3f, 0x0d, 0x47, 0x8e, 0x9d, 0x48, 0x4a, 0xbc, 0xa5, 0x2a,
    0xb5, 0xc6, 0xd7, 0xb9, 0x36, 0x3e, 0xd8, 0x0e, 0x89, 0x59, 0xb5, 0x89,
    0x4b, 0x82, 0xa9, 0x72, 0xe3, 0xb8, 0x83, 0x65, 0x24, 0xf9, 0x74, 0x9f,
    0x43, 0xb6, 0xc7, 0xeb, 0x82, 0xb4, 0xea, 0x6e, 0x75, 0x9e, 0xca, 0xa5,
    0x36, 0xed, 0x49, 0xc4, 0xa1, 0x5a, 0x54, 0x52, 0x6f, 0xc7, 0x3c, 0x63,
    0x43, 0xb8, 0xa9, 0xa6, 0xcf, 0x0a, 0x8a, 0xc5, 0xbf, 0xf3, 0xf7, 0xe8,
    0x04, 0x71, 0x6e, 0x9f, 0x4a, 0x51, 0xcd, 0x0b, 0x3d, 0x48, 0xfa, 0xde,
    0x33, 0x30, 0xe5, 0xec, 0xe9, 0x12, 0x9a, 0x89, 0xd5, 0x86, 0x99, 0x6a,
    0x34, 0x75, 0x78, 0x88, 0xd4, 0x05, 0x8a, 0x86, 0xe1, 0x36, 0xb7, 0x7b,
    0x7d, 0x31, 0xeb, 0x53, 0xa8, 0x67, 0xc8, 0x49, 0x65, 0x6e, 0x39, 0x22,
    0x13, 0x2a, 0x17, 0x65, 0x41, 0xb5, 0xb4, 0x9b, 0x13, 0xdb, 0x55, 0x88,
    0x1b, 0xf7, 0xc3, 0xfd, 0x0e, 0x44, 0xc7, 0xa8, 0x2c, 0xc1, 0xac, 0x34,
    0x92, 0xb6, 0x27, 0xc7, 0x6d, 0x68, 0x51, 0xb8, 0xa4, 0x1d};

std::vector<uint8_t> l2cap_test_packet_9 = {0x09, 0x00, 0x47, 0x00, 0x12,
                                            0x81, 0x3c, 0x1d, 0xc7, 0x62,
                                            0x45, 0x1a, 0x78};

// complete_l2cap_packet is the l2cap packet that should be created when
// assemble is called on a vector<L2capSdu*> that contains the previous 9 test
// packets.
std::vector<uint8_t> complete_l2cap_packet = {
    0x95, 0x1f, 0x47, 0x00, 0x02, 0x1f, 0x95, 0xcb, 0x00, 0x00, 0x00, 0x01,
    0x01, 0x00, 0x33, 0x00, 0x49, 0x00, 0x4d, 0x00, 0x47, 0x00, 0x5f, 0x00,
    0x32, 0x00, 0x30, 0x00, 0x31, 0x00, 0x37, 0x00, 0x30, 0x00, 0x36, 0x00,
    0x30, 0x00, 0x36, 0x00, 0x5f, 0x00, 0x31, 0x00, 0x35, 0x00, 0x34, 0x00,
    0x33, 0x00, 0x33, 0x00, 0x38, 0x00, 0x2e, 0x00, 0x6a, 0x00, 0x70, 0x00,
    0x67, 0x00, 0x00, 0x42, 0x00, 0x0e, 0x69, 0x6d, 0x61, 0x67, 0x65, 0x2f,
    0x6a, 0x70, 0x65, 0x67, 0x00, 0xc3, 0x00, 0x12, 0x30, 0x03, 0x97, 0x01,
    0x48, 0x1f, 0x45, 0xff, 0xd8, 0xff, 0xe0, 0x00, 0x10, 0x4a, 0x46, 0x49,
    0x46, 0x00, 0x01, 0x01, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0xff,
    0xe1, 0x2b, 0x18, 0x45, 0x78, 0x69, 0x66, 0x00, 0x00, 0x4d, 0x4d, 0x00,
    0x2a, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0c, 0x01, 0x00, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x0c, 0xc0, 0x01, 0x01, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x09, 0x90, 0x01, 0x0f, 0x00, 0x02, 0x00,
    0x00, 0x00, 0x04, 0x4c, 0x47, 0x45, 0x00, 0x01, 0x10, 0x00, 0x02, 0x00,
    0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x9e, 0x01, 0x12, 0x00, 0x03, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x01, 0x1a, 0x00, 0x05, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xa6, 0x01, 0x1b, 0x00, 0x05, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xae, 0x01, 0x28, 0x00, 0x03, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x01, 0x31, 0x00, 0x02, 0x00,
    0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0xb6, 0x01, 0x32, 0x00, 0x02, 0x00,
    0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0xbd, 0x02, 0x13, 0x00, 0x03, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x87, 0x69, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xd2, 0x00, 0x00, 0x02, 0x78, 0x4e,
    0x65, 0x78, 0x75, 0x73, 0x20, 0x35, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x01, 0x50,
    0x69, 0x63, 0x61, 0x73, 0x61, 0x00, 0x32, 0x30, 0x31, 0x37, 0x3a, 0x30,
    0x36, 0x3a, 0x30, 0x36, 0x20, 0x31, 0x35, 0x3a, 0x34, 0x33, 0x3a, 0x33,
    0x38, 0x00, 0x00, 0x00, 0x15, 0x82, 0x9a, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x01, 0xd4, 0x82, 0x9d, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x01, 0xdc, 0x88, 0x27, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x02, 0x1f, 0x00, 0x00, 0x90, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x04, 0x30, 0x32, 0x32, 0x30, 0x90, 0x03, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x01, 0xe4, 0x90, 0x04, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x14, 0x00, 0x00, 0x01, 0xf8, 0x91, 0x01, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x04, 0x01, 0x02, 0x03, 0x00, 0x92, 0x01, 0x00, 0x0a, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x02, 0x0c, 0x92, 0x02, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x02, 0x14, 0x92, 0x09, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x92, 0x0a, 0x00, 0x05, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x02, 0x1c, 0x92, 0x90, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x07, 0x00, 0x00, 0x02, 0x24, 0x92, 0x91, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x07, 0x00, 0x00, 0x02, 0x2b, 0x92, 0x92, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x07, 0x00, 0x00, 0x02, 0x32, 0xa0, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
    0x04, 0x30, 0x31, 0x30, 0x30, 0xa0, 0x01, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x01, 0x00, 0x00, 0xa0, 0x02, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x0a, 0x0a, 0xa0, 0x03, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x07, 0x77, 0xa0, 0x05, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x02, 0x5a, 0xa4, 0x03, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0xa4, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00,
    0x21, 0x00, 0x00, 0x02, 0x39, 0x00, 0x00, 0x00, 0x00, 0x01, 0xfc, 0x8c,
    0xb2, 0x3b, 0x9a, 0xca, 0x00, 0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00,
    0x64, 0x32, 0x30, 0x31, 0x37, 0x3a, 0x30, 0x36, 0x3a, 0x30, 0x36, 0x20,
    0x31, 0x35, 0x3a, 0x34, 0x33, 0x3a, 0x33, 0x38, 0x00, 0x32, 0x30, 0x31,
    0x37, 0x3a, 0x30, 0x36, 0x3a, 0x30, 0x36, 0x20, 0x31, 0x35, 0x3a, 0x34,
    0x33, 0x3a, 0x33, 0x38, 0x00, 0xff, 0xff, 0xfe, 0x16, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0xfc, 0x00, 0x00, 0x00, 0x64, 0x00, 0x00, 0x0f,
    0x82, 0x00, 0x00, 0x03, 0xe8, 0x39, 0x39, 0x39, 0x37, 0x32, 0x39, 0x00,
    0x39, 0x39, 0x39, 0x37, 0x32, 0x39, 0x00, 0x39, 0x39, 0x39, 0x37, 0x32,
    0x39, 0x00, 0x31, 0x30, 0x61, 0x66, 0x39, 0x37, 0x31, 0x37, 0x63, 0x31,
    0x30, 0x36, 0x36, 0x33, 0x65, 0x62, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30,
    0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x00, 0x00,
    0x02, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x52, 0x39, 0x38,
    0x00, 0x00, 0x02, 0x00, 0x07, 0x00, 0x00, 0x00, 0x04, 0x30, 0x31, 0x30,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x01, 0x03, 0x00, 0x03, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x06, 0x00, 0x00, 0x01, 0x1a, 0x00, 0x05, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0xc6, 0x01, 0x1b, 0x00, 0x05, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0xce, 0x01, 0x28, 0x00, 0x03, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x02, 0x01, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x02, 0xd6, 0x02, 0x02, 0x00, 0x04, 0x00,
    0x00, 0x00, 0x01, 0x00, 0x00, 0x28, 0x39, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x48, 0x00,
    0x00, 0x00, 0x01, 0xff, 0xd8, 0xff, 0xe0, 0x00, 0x10, 0x4a, 0x46, 0x49,
    0x46, 0x00, 0x01, 0x01, 0x00, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0xff,
    0xdb, 0x00, 0x43, 0x00, 0x05, 0x03, 0x04, 0x04, 0x04, 0x03, 0x05, 0x04,
    0x04, 0x04, 0x05, 0x05, 0x05, 0x06, 0x07, 0x0c, 0x08, 0x07, 0x07, 0x07,
    0x07, 0x0f, 0x0b, 0x0b, 0x09, 0x0c, 0x11, 0x0f, 0x12, 0x12, 0x11, 0x0f,
    0x11, 0x11, 0x13, 0x16, 0x1c, 0x17, 0x13, 0x14, 0x1a, 0x15, 0x11, 0x11,
    0x18, 0x21, 0x18, 0x1a, 0x1d, 0x1d, 0x1f, 0x1f, 0x1f, 0x13, 0x17, 0x22,
    0x24, 0x22, 0x1e, 0x24, 0x1c, 0x1e, 0x1f, 0x1e, 0xff, 0xdb, 0x00, 0x43,
    0x01, 0x05, 0x05, 0x05, 0x07, 0x06, 0x07, 0x0e, 0x08, 0x08, 0x0e, 0x1e,
    0x14, 0x11, 0x14, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e,
    0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e,
    0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e,
    0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0x1e,
    0x1e, 0x1e, 0x1e, 0x1e, 0x1e, 0xff, 0xc0, 0x00, 0x11, 0x08, 0x00, 0x78,
    0x00, 0xa0, 0x03, 0x01, 0x22, 0x00, 0x02, 0x11, 0x01, 0x03, 0x11, 0x01,
    0xff, 0xc4, 0x00, 0x1c, 0x00, 0x00, 0x03, 0x00, 0x03, 0x01, 0x01, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x06, 0x07,
    0x00, 0x04, 0x08, 0x03, 0x02, 0x01, 0xff, 0xc4, 0x00, 0x40, 0x10, 0x00,
    0x01, 0x03, 0x02, 0x05, 0x02, 0x04, 0x04, 0x04, 0x03, 0x06, 0x04, 0x07,
    0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x11, 0x00, 0x06, 0x12,
    0x21, 0x31, 0x07, 0x41, 0x13, 0x22, 0x51, 0x61, 0x14, 0x32, 0x71, 0x81,
    0x08, 0x42, 0x91, 0xa1, 0x15, 0x16, 0xc1, 0x23, 0x52, 0x62, 0x82, 0xb1,
    0xe1, 0x24, 0x43, 0x72, 0xf0, 0x17, 0x25, 0x33, 0x92, 0xb2, 0xd1, 0xd2,
    0xff, 0xc4, 0x00, 0x1c, 0x01, 0x00, 0x03, 0x01, 0x01, 0x00, 0x03, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x06, 0x07,
    0x04, 0x03, 0x00, 0x02, 0x08, 0x01, 0xff, 0xc4, 0x00, 0x37, 0x11, 0x00,
    0x01, 0x02, 0x04, 0x04, 0x03, 0x06, 0x04, 0x05, 0x04, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x00, 0x04, 0x05, 0x11, 0x06, 0x21,
    0x31, 0x41, 0x12, 0x51, 0x71, 0x13, 0x22, 0x61, 0x81, 0x91, 0xa1, 0x07,
    0xb1, 0xc1, 0xf0, 0x14, 0x42, 0xd1, 0xe1, 0xf1, 0x15, 0x24, 0x32, 0x82,
    0x33, 0x52, 0x72, 0xff, 0xda, 0x00, 0x0c, 0x03, 0x01, 0x00, 0x02, 0x11,
    0x03, 0x11, 0x00, 0x3f, 0x00, 0xb1, 0x31, 0x2c, 0x2d, 0xd8, 0xd2, 0xda,
    0x43, 0x26, 0x4a, 0x99, 0x1e, 0x15, 0x98, 0x2e, 0x2f, 0x51, 0x49, 0x0b,
    0x03, 0x70, 0x90, 0x6c, 0x9f, 0x4d, 0xb7, 0xf5, 0xc3, 0x0b, 0x4f, 0x09,
    0x1f, 0x08, 0xa5, 0x95, 0xb6, 0xb7, 0x50, 0x4e, 0x95, 0xab, 0xe4, 0xbf,
    0x2a, 0xb1, 0x17, 0xb8, 0xb1, 0xde, 0xdd, 0xf1, 0x34, 0xa1, 0x3b, 0x25,
    0xb6, 0x5d, 0x67, 0xe2, 0xda, 0x5b, 0xeb, 0x49, 0xf0, 0xd2, 0x83, 0xe1,
    0xa9, 0x37, 0xbe, 0x91, 0x72, 0xab, 0x5f, 0x73, 0x7f, 0xb6, 0x1d, 0x28,
    0xf0, 0xdf, 0xf1, 0x9b, 0x05, 0x2e, 0x20, 0x32, 0xd8, 0x64, 0x23, 0x52,
    0x4a, 0x9c, 0xb7, 0x2b, 0x52, 0x8d, 0xd5, 0xe9, 0xe5, 0xdb, 0x83, 0xea,
    0x30, 0x09, 0x75, 0x86, 0x25, 0xec, 0x1e, 0x58, 0x4d, 0xf9, 0x90, 0x20,
    0x87, 0x60, 0x4e, 0x91, 0xb2, 0xc3, 0x72, 0x63, 0xcd, 0x40, 0x76, 0x63,
    0xd2, 0x5b, 0x4a, 0x75, 0x04, 0x20, 0x06, 0xf9, 0xdb, 0xcd, 0xdc, 0x83,
    0xe9, 0x6b, 0xec, 0x0d, 0xf1, 0xba, 0x5f, 0x76, 0xe9, 0x50, 0x43, 0x65,
    0xd9, 0x05, 0x21, 0x08, 0xf2, 0x92, 0x94, 0x05, 0x73, 0x72, 0x48, 0x20,
    0x5c, 0xaa, 0xf8, 0xf8, 0xa9, 0x34, 0x97, 0xe4, 0x3e, 0xc7, 0xc4, 0x80,
    0x95, 0x90, 0x90, 0x9d, 0x1a, 0x94, 0x8b, 0xd8, 0x6d, 0xb5, 0x80, 0xef,
    0xf7, 0xc7, 0xeb, 0x14, 0xb7, 0xdb, 0x79, 0x45, 0x2f, 0xb6, 0x50, 0x55,
    0x7b, 0x96, 0x86, 0xb4, 0xf1, 0xc2, 0xaf, 0x7e, 0xc3, 0x7c, 0x65, 0x77,
    0x15, 0x52, 0x98, 0x36, 0x53, 0xc3, 0xca, 0xe7, 0xe5, 0x78, 0xf0, 0xb0,
    0xb3, 0x99, 0x11, 0xae, 0xa6, 0xd8, 0x91, 0x2c, 0x38, 0x3c, 0x37, 0xb4,
    0x36, 0xa6, 0x99, 0xd2, 0xb2, 0xb0, 0xda, 0x01, 0x04, 0xea, 0x24, 0x8d,
    0xcd, 0xb6, 0xfa, 0x63, 0x62, 0x92, 0xdb, 0x2c, 0xb8, 0xcb, 0x29, 0x7d,
    0x97, 0x94, 0x5b, 0x53, 0xa0, 0x22, 0xfa, 0x02, 0x52, 0x52, 0x35, 0x12,
    0x2f, 0x71, 0x7b, 0x72, 0x77, 0xe0, 0x63, 0xc2, 0xb9, 0x43, 0xa9, 0x39,
    0x09, 0xc8, 0xd0, 0x1f, 0x6d, 0x95, 0xbb, 0xa5, 0x7f, 0x10, 0xfa, 0x3c,
    0x5d, 0xf5, 0x58, 0x24, 0x20, 0x5b, 0x51, 0x3c, 0x58, 0x9b, 0x5a, 0xf7,
    0xbd, 0xf1, 0x2f, 0xcd, 0x94, 0x9c, 0xff, 0x00, 0x4a, 0x94, 0xe4, 0xba,
    0x8e, 0x60, 0xa8, 0xba, 0xc0, 0x05, 0x1a, 0xe2, 0x47, 0x09, 0x09, 0x4f,
    0xa6, 0x94, 0x8b, 0xa7, 0xd8, 0x0f, 0xb6, 0x0a, 0xd3, 0xaa, 0x54, 0xe9,
    0xe5, 0xa1, 0x29, 0x98, 0x4a, 0x4a, 0xb4, 0x0a, 0xba, 0x6f, 0x9d, 0xb7,
    0x16, 0xbf, 0x21, 0x7b, 0x9d, 0x84, 0x0d, 0xa8, 0x4c, 0xb9, 0x26, 0xd9,
    0x73, 0xb2, 0x52, 0xc0, 0xff, 0x00, 0xa8, 0x07, 0xda, 0xf7, 0xeb, 0x95,
    0x84, 0x55, 0xdb, 0x4c, 0x26, 0xa4, 0x3e, 0xdb, 0x0b, 0x2e, 0xcb, 0x91,
    0x24, 0xbe, 0xa2, 0x2c, 0x08, 0x52, 0x86, 0xca, 0xb7, 0x17, 0x00, 0x58,
    0xed, 0x72, 0x0e, 0xf8, 0xf8, 0x7e, 0x6c, 0x28, 0xce, 0x3a, 0xc3, 0x25,
    0x90, 0x88, 0xc9, 0xd4, 0xe9, 0x71, 0xc0, 0x09, 0x52, 0x85, 0xed, 0xa8,
    0xf7, 0xbd, 0xff, 0x00, 0x4d, 0x86, 0x39, 0x96, 0xa7, 0xd4, 0x6a, 0x63,
    0x28, 0x5b, 0x1f, 0xce, 0x33, 0xf5, 0x36, 0x0a, 0x5c, 0x49, 0x75, 0x6b,
    0x70, 0xab, 0x8b, 0x8b, 0x91, 0x63, 0x6b, 0x6d, 0x6e, 0xd8, 0x00, 0xee,
    0x76, 0xca, 0xce, 0x27, 0xc6, 0x6a, 0x8b, 0x32, 0x78, 0x69, 0x3f, 0xda,
    0x3a, 0xa6, 0xca, 0x89, 0xb7, 0x2a, 0x3b, 0x9f, 0x7c, 0x37, 0xff, 0x00,
    0x41, 0x69, 0x1f, 0xf2, 0xbe, 0x07, 0xa7, 0xd4, 0x88, 0x54, 0x18, 0xaa,
    0x61, 0xcf, 0xf8, 0xa4, 0xd6, 0x7a, 0xe5, 0xf3, 0x1f, 0xac, 0x75, 0x8c,
    0x0c, 0xc3, 0x02, 0xa3, 0x50, 0x6a, 0x9c, 0xc5, 0x62, 0x92, 0xe4, 0xa7,
    0x10, 0xb6, 0xbe, 0x1d, 0xb9, 0x2d, 0xad, 0x00, 0x80, 0x3f, 0x2f, 0xcc,
    0xb5, 0x0b, 0x1e, 0x2c, 0x07, 0xb9, 0xc7, 0xae, 0x68, 0x96, 0xa4, 0xd2,
    0xcb, 0x31, 0x25, 0x29, 0x72, 0x00, 0x0d, 0x96, 0xda, 0xec, 0x9b, 0xf7,
    0xb0, 0xd8, 0xf7, 0xda, 0xc7, 0xb6, 0x21, 0x79, 0x22, 0x3d, 0x1e, 0xbf,
    0x45, 0x8b, 0x3c, 0xd3, 0x95, 0x11, 0xb9, 0x26, 0xf1, 0x5c, 0x69, 0x65,
    0xa7, 0x10, 0xa0, 0xe6, 0x84, 0x9d, 0xb7, 0x1e, 0x71, 0x89, 0xac, 0x0a,
    0x9d, 0x56, 0x8f, 0x38, 0x48, 0x8d, 0x52, 0x9d, 0x0d, 0x65, 0xc2, 0x54,
    0xeb, 0x2e, 0xa9, 0x0a, 0x55, 0x8d, 0x89, 0xe7, 0x7e, 0xfc, 0xe0, 0x0d,
    0x79, 0xb6, 0xe9, 0x8a, 0x42, 0x50, 0x4a, 0x82, 0xc6, 0xbf, 0xa4, 0x51,
    0xbe, 0x1b, 0xc8, 0x2b, 0x18, 0x35, 0x32, 0xe3, 0x87, 0xb2, 0x53, 0x2a,
    0x09, 0xe1, 0xd6, 0xe4, 0x8b, 0xe6, 0x76, 0xf4, 0x31, 0xd6, 0x31, 0x67,
    0x66, 0x00, 0xda, 0x90, 0x22, 0xc6, 0x79, 0xb4, 0x20, 0xa1, 0x6f, 0xbc,
    0x0a, 0x41, 0x09, 0xf3, 0x6b, 0x09, 0x37, 0xb0, 0x36, 0xdc, 0x12, 0x4e,
    0xd8, 0x11, 0x51, 0x7f, 0xf8, 0xed, 0x5a, 0x1c, 0x88, 0x35, 0x04, 0xa5,
    0xc8, 0x4d, 0x2d, 0x4d, 0xa9, 0xc8, 0xd6, 0x6b, 0x52, 0x87, 0x99, 0x4a,
    0xb8, 0xdc, 0x93, 0xc0, 0x06, 0xdc, 0xe2, 0x21, 0x13, 0xa9, 0xd9, 0xb0,
    0xb1, 0xe1, 0x4a, 0xaf, 0x3a, 0xe9, 0x4a, 0xb5, 0x21, 0x6e, 0xb0, 0x85,
    0x9f, 0x6b, 0xa8, 0x6f, 0x7f, 0xb6, 0x35, 0xe2, 0xf5, 0x37, 0x37, 0xca,
    0x7c, 0x35, 0x51, 0xaf, 0xae, 0x13, 0x00, 0xfc, 0xed, 0x46, 0xd7, 0x71,
    0xfe, 0x22, 0x0d, 0xc7, 0xe8, 0x70, 0x10, 0xcf, 0xb0, 0x53, 0x61, 0x73,
    0xe9, 0xfa, 0xc3, 0xcc, 0xc6, 0x09, 0x7a, 0x50, 0x15, 0x29, 0x1c, 0x5f,
    0xf9, 0xb9, 0xfa, 0x67, 0xe5, 0x68, 0xe8, 0x7a, 0x34, 0x17, 0x60, 0x34,
    0xa0, 0x66, 0x53, 0xdb, 0x53, 0x84, 0xea, 0x5b, 0x67, 0x4d, 0xfd, 0xfc,
    0xbe, 0xfb, 0xf3, 0xbf, 0xd3, 0x6c, 0x7d, 0x66, 0x87, 0xeb, 0x74, 0xf8,
    0xb1, 0xe1, 0xd2, 0xc4, 0x35, 0x49, 0x90, 0xcb, 0xae, 0xc3, 0x73, 0x7d,
    0x3a, 0x90, 0x9b, 0x90, 0x45, 0xed, 0xbd, 0xf6, 0x3c, 0x0e, 0x6d, 0x89,
    0xa6, 0x46, 0x89, 0x56, 0xac, 0x4b, 0x81, 0x35, 0xac, 0xdc, 0xfd, 0x56,
    0x31, 0x7d, 0x2a, 0x29, 0x60, 0x0b, 0x25, 0x49, 0x58, 0xd9, 0xc0, 0x07,
    0x96, 0xe3, 0xb1, 0xb6, 0xd8, 0xac, 0xe7, 0xb4, 0x45, 0x6a, 0xaf, 0x4b,
    0x43, 0x4a, 0x6d, 0x25, 0xa9, 0x32, 0x11, 0xa1, 0x24, 0x5d, 0x29, 0x53,
    0x5b, 0x6d, 0xd8, 0x5c, 0x60, 0xad, 0x29, 0x4d, 0xcc, 0x38, 0x3b, 0xb6,
    0x00, 0x8d, 0x77, 0x85, 0xb9, 0x96, 0x19, 0x49, 0x52, 0x10, 0x9c, 0xc2,
    0x54, 0x73, 0x1a, 0x10, 0x92, 0x47, 0xbd, 0xa2, 0x1c, 0x9f, 0xfc, 0x40,
    0x7a, 0x32, 0x5f, 0x97, 0x99, 0x1d, 0x52, 0xdb, 0x36, 0x11, 0x92, 0xe2,
    0x90, 0xd3, 0x62, 0xdc, 0x0d, 0x24, 0x11, 0xb1, 0xc2, 0xfd, 0x5a, 0x5e,
    0x60, 0x42, 0x2c, 0xfc, 0x07, 0x18, 0x75, 0x2a, 0xba, 0x26, 0x21, 0xe5,
    0x3f, 0x63, 0xfd, 0xe4, 0xa9, 0x47, 0xca, 0x46, 0xe3, 0x71, 0xdf, 0x19,
    0xd6, 0x3a, 0xd5, 0x5e, 0x91, 0x0d, 0x2f, 0xc1, 0x21, 0x94, 0x29, 0xd7,
    0x2e, 0x15, 0x65, 0x05, 0x5b, 0x4e, 0xe4, 0x7d, 0xf8, 0x38, 0x0b, 0x42,
    0xea, 0x69, 0x30, 0x9b, 0x6a, 0xb9, 0x4b, 0x43, 0xe1, 0xb0, 0x12, 0x1d,
    0x8e, 0x7c, 0x32, 0x8f, 0xb7, 0x6f, 0xb6, 0x28, 0xcb, 0x99, 0x92, 0x95,
    0x74, 0x36, 0xa0, 0x94, 0xa8, 0x8b, 0x8c, 0xac, 0x39, 0x74, 0x88, 0xa6,
    0x1f, 0xc4, 0x18, 0xbd, 0x12, 0x9f, 0x8e, 0x67, 0x85, 0xe6, 0xd4, 0x4d,
    0xd2, 0x6d, 0x71, 0x6d, 0x6d, 0x98, 0x3e, 0x40, 0xf9, 0x40, 0xf9, 0x19,
    0x9a, 0xad, 0xe3, 0x3b, 0x11, 0x6e, 0xca, 0xa9, 0x6b, 0x56, 0x8d, 0x72,
    0x5c, 0x4a, 0xad, 0x7b, 0x0b, 0x24, 0xe9, 0x3a, 0x71, 0xf7, 0x97, 0xaa,
    0x19, 0x86, 0x65, 0x52, 0x24, 0x66, 0xe9, 0x0d, 0xb7, 0xe3, 0x38, 0x1b,
    0x49, 0x44, 0x50, 0xab, 0x82, 0xab, 0x5e, 0xc4, 0x6f, 0x6c, 0x35, 0xb3,
    0x4b, 0xcb, 0x79, 0x89, 0xe4, 0xce, 0xa6, 0x26, 0x6c, 0x69, 0x5a, 0x82,
    0x95, 0x29, 0x2b, 0x52, 0x1d, 0x41, 0x3b, 0x8b, 0x91, 0xda, 0xc3, 0x6d,
    0x88, 0xd8, 0xe1, 0x93, 0x2f, 0xcf, 0xab, 0xe5, 0x7c, 0xc7, 0x4d, 0x46,
    0x60, 0x97, 0x0e, 0x75, 0x31, 0xe9, 0x08, 0x69, 0x12, 0x8c, 0x60, 0x87,
    0xda, 0xf4, 0xbe, 0x8b, 0x05, 0x0f, 0xb0, 0x3d, 0xf1, 0x8a, 0x64, 0xd5,
    0x5a, 0x1c, 0x52, 0xae, 0x24, 0xa0, 0x67, 0x6b, 0x01, 0xfc, 0xfa, 0xc3,
    0x75, 0x33, 0xe2, 0x63, 0x4f, 0xbc, 0x99, 0x57, 0xd2, 0x59, 0x73, 0x4e,
    0x12, 0x32, 0xbf, 0xcf, 0xd6, 0x06, 0xf5, 0x01, 0xd9, 0x74, 0xac, 0xba,
    0xdb, 0x2f, 0x33, 0x1d, 0xba, 0x85, 0x36, 0xae, 0xe4, 0x77, 0x5d, 0x61,
    0xad, 0x96, 0x0b, 0x63, 0x70, 0x07, 0xfd, 0x37, 0xb6, 0x1f, 0xba, 0x4b,
    0x5b, 0x95, 0x3f, 0x2b, 0xc1, 0x44, 0xc7, 0x1e, 0x96, 0x02, 0x8d, 0xca,
    0x9b, 0xb2, 0x94, 0x41, 0x24, 0x12, 0x2f, 0xcd, 0xec, 0x37, 0xe7, 0xdf,
    0x13, 0xbe, 0xb2, 0x54, 0x4c, 0xba, 0xce, 0x60, 0x5c, 0x25, 0x87, 0xe9,
    0xee, 0x4a, 0x6e, 0x49, 0xf0, 0xd3, 0x72, 0x95, 0x29, 0x24, 0x03, 0xa8,
    0x1d, 0xb9, 0x3b, 0x7b, 0x7b, 0x60, 0xef, 0x43, 0x5d, 0x29, 0x88, 0xdb,
    0x8b, 0x72, 0x33, 0x65, 0x5e, 0x25, 0xee, 0x17, 0x7b, 0x10, 0x05, 0xac,
    0x0d, 0x89, 0xdb, 0x83, 0x85, 0x8c, 0x5a, 0x78, 0x66, 0x9a, 0x58, 0xdd,
    0x20, 0x9f, 0x3b, 0x9f, 0xac, 0x3b, 0xd5, 0xdb, 0xb9, 0x4a, 0xd5, 0xa9,
    0x00, 0xfb, 0x40, 0xac, 0xbf, 0x98, 0xbe, 0x31, 0x9a, 0xab, 0x73, 0x42,
    0xde, 0x80, 0xc7, 0x87, 0xa9, 0x2b, 0x6c, 0x10, 0xe0, 0x28, 0xbe, 0xe0,
    0xec, 0x47, 0xa7, 0xfb, 0x62, 0x9b, 0x94, 0x9a, 0xa9, 0xd6, 0x72, 0xa8,
    0x95, 0x90, 0x33, 0x73, 0x72, 0xd6, 0x86, 0x2e, 0x69, 0x92, 0xbc, 0xcb,
    0x61, 0x76, 0xf9, 0x52, 0x54, 0x6f, 0xa7, 0xd8, 0x92, 0x3d, 0x14, 0x31,
    0x03, 0x76, 0x3b, 0x4d, 0xe4, 0xcc, 0xcd, 0x23, 0x58, 0xbb, 0x8d, 0xa2,
    0xe9, 0x04, 0xd9, 0x20, 0x04, 0x8d, 0xbd, 0x76, 0xc4, 0xf3, 0x2a, 0x55,
    0x33, 0x44, 0x4c, 0xf6, 0x22, 0xe5, 0x27, 0xa6, 0xae, 0xa2, 0xe4, 0xa0,
    0x96, 0x44, 0x72, 0x6e, 0x0d, 0x85, 0xcd, 0x87, 0x6b, 0x5e, 0xff, 0x00,
    0xbe, 0x3d, 0xa6, 0x9a, 0x94, 0x98, 0xa7, 0xcb, 0x4a, 0x4c, 0xb4, 0x16,
    0x14, 0x9b, 0xe6, 0x01, 0xcf, 0x2d, 0xb5, 0xdf, 0x50, 0x41, 0x89, 0x54,
    0x8c, 0xa3, 0xc2, 0xa1, 0x3d, 0x50, 0x65, 0xee, 0x12, 0x85, 0x81, 0x62,
    0x6c, 0x2d, 0x62, 0x49, 0xdc, 0x65, 0x6d, 0xd2, 0x44, 0x5c, 0x1e, 0xcf,
    0x3d, 0x5a, 0x62, 0xa4, 0xe5, 0x3e, 0xa5, 0x50, 0xa7, 0xd2, 0xdd, 0x49,
    0xd2, 0x84, 0xbc, 0xd6, 0x95, 0x39, 0xcd, 0x82, 0x6f, 0x7b, 0xf1, 0xce,
    0xe0, 0x7a, 0xe1, 0x83, 0x20, 0xbf, 0xd4, 0x0a, 0xad, 0x42, 0x4b, 0xd5,
    0x6c, 0xe0, 0xf1, 0x66, 0x24, 0xe6, 0xee, 0xcb, 0x6d, 0x20, 0x02, 0x9b,
    0x82, 0x52, 0xab, 0x0f, 0x4d, 0x88, 0xf7, 0xc1, 0x8c, 0xd5, 0x2e, 0x8a,
    0x7a, 0x5d, 0x4e, 0x8b, 0x5b, 0x8e, 0xd2, 0xab, 0x06, 0xea, 0x50, 0x6d,
    0x3b, 0xa1, 0xb2, 0xa3, 0xa0, 0x58, 0x7f, 0xcc, 0x50, 0xb5, 0x80, 0xdc,
    0x83, 0xbe, 0x24, 0xbd, 0x36, 0xcc, 0x33, 0xb2, 0xcd, 0x54, 0x25, 0x42,
    0x4a, 0x5a, 0x2e, 0x02, 0x96, 0xdc, 0x90, 0x4a, 0x55, 0x7d, 0x89, 0xb7,
    0x04, 0x81, 0xff, 0x00, 0xd6, 0xf8, 0x54, 0xad, 0xe1, 0x4a, 0x6d, 0x38,
    0x07, 0x19, 0x69, 0x20, 0x1b, 0xe4, 0x52, 0x2f, 0x96, 0xe0, 0xf2, 0x3b,
    0x7d, 0x61, 0xaf, 0x0d, 0x62, 0x79, 0x8a, 0xb0, 0x52, 0x1c, 0x02, 0xe9,
    0x09, 0xcc, 0x68, 0x4a, 0x85, 0xed, 0xa7, 0xf9, 0x0d, 0xc7, 0x8e, 0xd1,
    0xd8, 0x64, 0xff, 0x00, 0xc5, 0xb4, 0xd3, 0x68, 0x05, 0x28, 0x0a, 0x5a,
    0x92, 0x4f, 0xe5, 0x4b, 0x97, 0xb7, 0xe8, 0x71, 0x1a, 0xab, 0xfe, 0x21,
    0x29, 0xd4, 0x7e, 0xa5, 0xce, 0xca, 0x35, 0xea, 0x64, 0x91, 0x09, 0x01,
    0x05, 0x12, 0xda, 0x40, 0x50, 0x4e, 0xa1, 0xdd, 0x3c, 0xd8, 0x7a, 0xef,
    0x8a, 0xc5, 0x36, 0xa9, 0x0e, 0x44, 0x84, 0xe8, 0x73, 0xfb, 0x59, 0x2c,
    0x29, 0x4d, 0xa7, 0xbf, 0x65, 0x7f, 0xa1, 0xc7, 0x13, 0xf5, 0x06, 0x9b,
    0x52, 0xac, 0xfe, 0x20, 0x6a, 0x90, 0x20, 0x45, 0x91, 0x2a, 0x5b, 0x8b,
    0x6d, 0xb6, 0x19, 0x6c, 0x5d, 0x4b, 0xb2, 0x78, 0x1f, 0xbe, 0xfc, 0x0c,
    0x21, 0x61, 0xb9, 0x66, 0xa7, 0xdb, 0xb3, 0xa7, 0x44, 0x2b, 0x43, 0x6b,
    0x10, 0xb4, 0x81, 0xec, 0x77, 0xb8, 0xcc, 0xc1, 0xa7, 0xf8, 0x99, 0x48,
    0x36, 0xce, 0xe0, 0x67, 0xc8, 0x85, 0x1f, 0xa6, 0xa2, 0x3a, 0x3b, 0x33,
    0xe4, 0x8c, 0x85, 0xd4, 0x46, 0xda, 0xae, 0x52, 0xc4, 0x07, 0x64, 0x2e,
    0xc0, 0x49, 0x61, 0x00, 0x2b, 0xe8, 0xbb, 0x6f, 0xbe, 0x14, 0xab, 0xb9,
    0x30, 0x50, 0x22, 0xff, 0x00, 0x0f, 0xfe, 0x13, 0x18, 0x7c, 0x72, 0x96,
    0xc0, 0x90, 0x41, 0x29, 0x41, 0x50, 0xb0, 0xbf, 0x1e, 0x5e, 0x76, 0xdb,
    0xdc, 0xfa, 0xb5, 0xf4, 0xeb, 0x24, 0x53, 0x7a, 0x71, 0x45, 0x7a, 0xa1,
    0x5b, 0xa8, 0x32, 0x89, 0xd3, 0x96, 0xda, 0xa5, 0x25, 0xb5, 0xf9, 0x1b,
    0x08, 0x3a, 0x92, 0x80, 0x7f, 0x32, 0xb5, 0x5a, 0xea, 0xf6, 0xb0, 0xbe,
    0x06, 0x75, 0x3e, 0xa9, 0x5a, 0xce, 0x51, 0xa1, 0xc3, 0xa7, 0xb0, 0x61,
    0xc5, 0x5c, 0x80, 0x59, 0x52, 0x8d, 0x9c, 0x58, 0x48, 0x24, 0x1b, 0x7e,
    0x54, 0xef, 0xdf, 0x72, 0x71, 0xda, 0x98, 0xf4, 0xe8, 0x9f, 0xec, 0xa5,
    0x9d, 0x2e, 0x32, 0x35, 0xb8, 0xc8, 0x78, 0x03, 0xe9, 0xa6, 0xa7, 0x6b,
    0x46, 0x87, 0xc3, 0x45, 0xae, 0x35, 0xa4, 0x5f, 0x9e, 0xff, 0x00, 0xbf,
    0xa6, 0x9b, 0xc2, 0xfc, 0xa9, 0xf1, 0x13, 0x97, 0xbc, 0x4a, 0x32, 0x2e,
    0xd4, 0x0b, 0xb4, 0xa0, 0x11, 0xa7, 0xcc, 0xd3, 0x89, 0x52, 0x94, 0x94,
    0x8b, 0x00, 0x2f, 0x72, 0x3d, 0xb1, 0x36, 0xeb, 0x1c, 0x07, 0x28, 0xd3,
    0x6b, 0xb4, 0xc9, 0x4e, 0x15, 0x25, 0x82, 0xb5, 0x31, 0xa4, 0xdd, 0x29,
    0x4a, 0x9c, 0x43, 0xa9, 0x23, 0xfc, 0xaa, 0xfd, 0xf1, 0xbb, 0x2e, 0x55,
    0x6a, 0x87, 0x1a, 0xbc, 0x89, 0x20, 0x29, 0x69, 0x63, 0x5a, 0x52, 0xb0,
    0x6c, 0x51, 0xa1, 0x40, 0xf1, 0xea, 0x06, 0x06, 0x67, 0xc9, 0x92, 0x33,
    0x14, 0x36, 0xe4, 0xcb, 0x5f, 0x8a, 0xec, 0x9a, 0x6c, 0x7b, 0xab, 0x48,
    0x04, 0xff, 0x00, 0x62, 0x11, 0xfb, 0x14, 0xe2, 0xb3, 0x88, 0x78, 0x7f,
    0x01, 0x26, 0xb3, 0xa8, 0x03, 0xe9, 0xfa, 0x46, 0x4f, 0x85, 0x0c, 0x3c,
    0xa9, 0xba, 0xd4, 0xba, 0x45, 0xc2, 0xc5, 0xc5, 0xb5, 0xe2, 0xb1, 0xb7,
    0xce, 0xde, 0x71, 0x36, 0xa6, 0x4b, 0xaa, 0xbf, 0x20, 0x98, 0xaf, 0x68,
    0x69, 0xb4, 0xdf, 0x4d, 0xb6, 0xb0, 0xf6, 0xe4, 0x93, 0x87, 0x95, 0xaa,
    0x38, 0xa5, 0x44, 0x4a, 0x81, 0xf8, 0xd4, 0xeb, 0x32, 0x1d, 0x1b, 0x21,
    0x49, 0x36, 0x29, 0xb0, 0xf5, 0x1b, 0xdc, 0xfd, 0x30, 0x0f, 0x2d, 0xd1,
    0xd3, 0x0d, 0xa6, 0xde, 0xa8, 0xb8, 0x12, 0xb0, 0x2f, 0x60, 0x0a, 0x83,
    0x7b, 0x6f, 0x6b, 0x72, 0xac, 0x36, 0xd2, 0xa8, 0x46, 0xbb, 0x1e, 0x4c,
    0xa8, 0x92, 0x22, 0xa0, 0x34, 0xc9, 0x71, 0xa6, 0x0b, 0xa1, 0x4a, 0x3d,
    0xbc, 0xd6, 0x3b, 0x2b, 0x93, 0x7e, 0x01, 0xdb, 0x9c, 0x28, 0x4c, 0xad,
    0x2e, 0x2a, 0xc8, 0x02, 0xdc, 0xe2, 0xc1, 0x46, 0x2f, 0x50, 0x25, 0x7f,
    0x11, 0x50, 0x75, 0x5c, 0x4a, 0xff, 0x00, 0x16, 0xef, 0xa7, 0x5b, 0xe9,
    0xd3, 0x6d, 0xf3, 0xc8, 0x6e, 0xf4, 0xff, 0x00, 0x3d, 0x4c, 0xca, 0xf9,
    0x88, 0x7c, 0x2c, 0x1d, 0x46, 0x63, 0x29, 0x6c, 0xa4, 0x27, 0x40, 0xb2,
    0x38, 0x59, 0xf5, 0x27, 0x7f, 0xd7, 0x1d, 0x08, 0xce, 0x65, 0x6e, 0xa7,
    0x54, 0xa7, 0x4f, 0xac, 0xa6, 0x3c, 0x79, 0x29, 0x7e, 0x13, 0xc8, 0x49,
    0x50, 0x05, 0x45, 0xd0, 0xa4, 0xad, 0x36, 0xfc, 0xdc, 0x03, 0x8e, 0x7d,
    0x32, 0xa0, 0x57, 0xa9, 0x52, 0x6a, 0xeb, 0x52, 0x19, 0xaa, 0x46, 0x8a,
    0x52, 0x96, 0xb4, 0x24, 0x2d, 0xc2, 0x15, 0xdf, 0xe8, 0x07, 0x1c, 0x91,
    0xfa, 0x63, 0xa0, 0xf3, 0x25, 0x39, 0x0e, 0x51, 0x29, 0x12, 0xc2, 0x59,
    0xb3, 0x52, 0x60, 0x91, 0x74, 0x0b, 0xdb, 0xc3, 0x3d, 0xff, 0x00, 0x4c,
    0x14, 0xa5, 0x29, 0x69, 0x70, 0x24, 0x69, 0xcb, 0xce, 0x13, 0xa6, 0xa7,
    0x45, 0x49, 0xf7, 0x5d, 0x75, 0x20, 0x29, 0x79, 0x65, 0xb0, 0xd2, 0xde,
    0x62, 0x21, 0xff, 0x00, 0x88, 0xf6, 0x9b, 0xa9, 0x52, 0x2a, 0xb2, 0xa1,
    0x21, 0x6d, 0x21, 0x55, 0x37, 0x96, 0xce, 0xa1, 0xa4, 0xe8, 0x52, 0xc1,
    0xbe, 0x9e, 0xc2, 0xd7, 0xdb, 0x12, 0x9e, 0x9c, 0xd2, 0xe6, 0xd5, 0x5f,
    0x31, 0xe1, 0x34, 0xeb, 0xce, 0xa9, 0x77, 0x5a, 0x95, 0xe6, 0x48, 0x48,
    0x16, 0x25, 0x46, 0xf6, 0x03, 0x91, 0x8b, 0xa6, 0x64, 0xa0, 0x53, 0xb3,
    0x35, 0x3e, 0x43, 0x55, 0x17, 0x8b, 0x4d, 0x25, 0xe2, 0xe2, 0x8a, 0x57,
    0xa7, 0x6b, 0x9b, 0x8b, 0x8c, 0x20, 0x55, 0xf3, 0xad, 0x03, 0x2c, 0x43,
    0x5e, 0x5a, 0xca, 0x8c, 0xc6, 0x6e, 0xc7, 0x77, 0xd1, 0x65, 0x24, 0x2b,
    0xde, 0xff, 0x00, 0x3e, 0x29, 0x53, 0xb4, 0xa6, 0x9e, 0x71, 0x2f, 0xbc,
    0xab, 0x21, 0x09, 0x00, 0xff, 0x00, 0x3f, 0x66, 0x3e, 0x6d, 0xa0, 0xe2,
    0x09, 0xa6, 0xd8, 0x5c, 0x8c, 0x8b, 0x65, 0x4f, 0x95, 0xa8, 0xdf, 0xf2,
    0xa4, 0x13, 0xaf, 0xcf, 0x28, 0x6c, 0x8e, 0xec, 0x2c, 0xa7, 0x0d, 0x98,
    0xad, 0x3d, 0xfc, 0x46, 0xa5, 0x21, 0x21, 0xa4, 0x45, 0x8e, 0x92, 0x55,
    0xb7, 0x01, 0x29, 0xf4, 0xdc, 0xf9, 0x8f, 0xfa, 0x63, 0xe2, 0xa1, 0x42,
    0xcc, 0xf5, 0x67, 0x04, 0xfa, 0xac, 0x96, 0xd8, 0x75, 0x2b, 0x40, 0x6a,
    0x1b, 0x67, 0x59, 0x66, 0xff, 0x00, 0x4f, 0x99, 0x7f, 0xb0, 0xf7, 0xc0,
    0x8c, 0x9b, 0x51, 0xa7, 0x46, 0xa6, 0xaa, 0x5c, 0xe9, 0xaf, 0x3d, 0x52,
    0x99, 0xaf, 0xc5, 0x74, 0x2e, 0xca, 0x52, 0x38, 0xb1, 0x3f, 0x30, 0x16,
    0xbe, 0xc2, 0xc3, 0x7c, 0x51, 0x1e, 0xcd, 0xd4, 0xc5, 0x25, 0xa8, 0xca,
    0x9d, 0x09, 0x98, 0xcd, 0xa5, 0xb6, 0x8a, 0x84, 0x93, 0xad, 0x56, 0x1c,
    0x9b, 0x26, 0xfe, 0xbb, 0x8e, 0x6f, 0x85, 0xaa, 0x96, 0x21, 0x2a, 0x4f,
    0xe1, 0xa4, 0xc7, 0x0a, 0x34, 0xbe, 0xe7, 0xa7, 0x21, 0xef, 0x0e, 0x94,
    0x6c, 0x22, 0xdc, 0xab, 0x9f, 0x8d, 0x9d, 0x3d, 0xac, 0xc1, 0xcc, 0x93,
    0x98, 0x1d, 0x07, 0x84, 0x44, 0x73, 0x33, 0xd3, 0xe9, 0x30, 0x26, 0x95,
    0xbb, 0xe3, 0x04, 0xb6, 0x94, 0x3e, 0x82, 0x2d, 0x72, 0x97, 0x08, 0xbe,
    0xde, 0x84, 0x1c, 0x51, 0xba, 0x0d, 0x3d, 0x72, 0x29, 0x25, 0x6c, 0x48,
    0x5b, 0x5a, 0x96, 0x97, 0x2e, 0xa2, 0x12, 0x1b, 0x00, 0x8b, 0xf3, 0x7b,
    0xdf, 0xb6, 0x17, 0xb3, 0x4c, 0x5a, 0x6d, 0x43, 0x2d, 0x66, 0x37, 0x20,
    0xba, 0x99, 0x29, 0x44, 0x67, 0x9c, 0x2e, 0x24, 0x6c, 0x56, 0x97, 0x8a,
    0xbf, 0xf8, 0xd8, 0xfd, 0xf1, 0xf5, 0xf8, 0x7d, 0x6c, 0x38, 0x66, 0x43,
    0x4a, 0x1d, 0xf0, 0x8b, 0xc1, 0xd0, 0x7c, 0x3b, 0xa5, 0x28, 0xd3, 0x7b,
    0x0b, 0x5f, 0xbe, 0xd6, 0x27, 0x1e, 0x63, 0x04, 0x36, 0x4b, 0x2e, 0xb7,
    0xba, 0x47, 0xb4, 0x3e, 0xcf, 0xa5, 0x60, 0xa0, 0xab, 0x74, 0x88, 0x1f,
    0x4d, 0x6a, 0x75, 0x5a, 0x85, 0x98, 0xe0, 0xb0, 0x94, 0x95, 0x49, 0x75,
    0xb6, 0xc0, 0xe6, 0xc0, 0x5a, 0xdb, 0xfa, 0x5b, 0x9c, 0x13, 0xc9, 0xf5,
    0xea, 0x5f, 0x4b, 0xa2, 0x4c, 0xa8, 0xc4, 0x9e, 0xc2, 0x6b, 0x13, 0x9b,
    0x51, 0x54, 0x94, 0xb4, 0x16, 0xea, 0x49, 0x04, 0x14, 0xb6, 0x54, 0x3c,
    0xa9, 0x17, 0xed, 0xb9, 0x3e, 0x98, 0x0f, 0x42, 0x62, 0xa9, 0x22, 0x74,
    0xb8, 0x14, 0xe9, 0x11, 0x23, 0xc7, 0x92, 0xe0, 0x52, 0xca, 0x95, 0xa0,
    0x71, 0x62, 0x9d, 0x86, 0xff, 0x00, 0x41, 0x6b, 0xf0, 0x4e, 0x08, 0x37,
    0xd1, 0x25, 0x57, 0x67, 0x99, 0x55, 0x4c, 0xd8, 0xb5, 0xb8, 0xad, 0x92,
    0x94, 0x37, 0xa7, 0x48, 0xec, 0x07, 0x60, 0x31, 0x96, 0x6b, 0x15, 0x52,
    0x29, 0xf2, 0xad, 0x05, 0xaa, 0xee, 0x84, 0x81, 0xa1, 0x36, 0xc8, 0x5e,
    0xff, 0x00, 0x7e, 0xd1, 0x3c, 0x6b, 0x0b, 0x55, 0xa7, 0x66, 0x5e, 0xba,
    0xb8, 0x18, 0x52, 0xf8, 0xb2, 0x20, 0x15, 0x1d, 0xb3, 0xfb, 0xf3, 0x3a,
    0x0a, 0x8d, 0x2e, 0xa1, 0x99, 0xd9, 0x12, 0x98, 0xab, 0x13, 0x50, 0x64,
    0x87, 0x9b, 0xf1, 0x1c, 0xb0, 0x48, 0xd4, 0x76, 0x1f, 0xdd, 0x1e, 0xa7,
    0xe8, 0x37, 0xbe, 0x1b, 0xe9, 0x35, 0x68, 0x53, 0x99, 0x55, 0x32, 0x7c,
    0x66, 0x51, 0x53, 0x41, 0x0e, 0x3c, 0xeb, 0x29, 0x1e, 0x1b, 0x8d, 0x95,
    0x69, 0x0b, 0xd8, 0xec, 0x49, 0x3f, 0x4d, 0xb0, 0x73, 0x2e, 0xf4, 0x5f,
    0x24, 0xd0, 0x90, 0xbf, 0x8e, 0xcd, 0x32, 0x8b, 0x65, 0x43, 0xc4, 0x02,
    0x62, 0x12, 0x92, 0x47, 0xaf, 0x9a, 0xe7, 0xe9, 0x86, 0x58, 0x39, 0x6b,
    0xa3, 0x14, 0x79, 0xc9, 0x9a, 0xe5, 0x4a, 0x13, 0xb2, 0xda, 0x01, 0x21,
    0xc7, 0x65, 0xeb, 0xb0, 0x1c, 0x6c, 0x4e, 0x9b, 0x62, 0x73, 0x52, 0xc5,
    0xcd, 0x4d, 0x3a, 0x57, 0xc2, 0xb5, 0x93, 0xc9, 0x3f, 0xc6, 0x50, 0xf7,
    0x23, 0x4e, 0x44, 0xab, 0x61, 0xb6, 0x80, 0x00, 0x6c, 0x9f, 0xd8, 0x6b,
    0xce, 0x1b, 0x24, 0x43, 0x99, 0x1a, 0x65, 0x00, 0x38, 0xb5, 0x29, 0x25,
    0xa5, 0xb4, 0xbb, 0x7a, 0xea, 0x40, 0xfe, 0x98, 0xf0, 0xae, 0xcd, 0xc9,
    0x3d, 0x32, 0x7a, 0xa7, 0x99, 0x24, 0x31, 0x09, 0x35, 0x49, 0x48, 0x4a,
    0x65, 0xc9, 0x2b, 0x25, 0xd2, 0x81, 0xc2, 0x4a, 0xbf, 0x28, 0x3f, 0xdc,
    0x4e, 0xe4, 0xf3, 0x7c, 0x22, 0x75, 0xaf, 0xad, 0x59, 0x7e, 0x9b, 0x0e,
    0x9f, 0x13, 0x2c, 0x4c, 0x6e, 0x6c, 0xf2, 0xfa, 0x34, 0xe8, 0x21, 0xd0,
    0x11, 0xbe, 0xa2, 0x42, 0x4d, 0xf7, 0xd8, 0x0e, 0x09, 0x3c, 0x71, 0x88,
    0xd6, 0x70, 0xcd, 0x31, 0xf3, 0x25, 0x41, 0x2e, 0xd5, 0x68, 0x95, 0x29,
    0xe9, 0x4a, 0x8a, 0xc3, 0x0b, 0x49, 0x4a, 0x50, 0xa3, 0xed, 0xdc, 0xf6,
    0xb9, 0xdf, 0x18, 0xf0, 0xa6, 0x0d, 0xa8, 0x55, 0x07, 0x6a, 0xb5, 0x76,
    0x4d, 0xd8, 0x85, 0x0c, 0xee, 0x41, 0x37, 0xb5, 0xad, 0xa1, 0xf1, 0xcb,
    0x2d, 0xe3, 0xde, 0xa9, 0x5b, 0x6a, 0x59, 0x09, 0x41, 0x69, 0x4e, 0x2f,
    0x50, 0x00, 0x16, 0xbe, 0xd7, 0x37, 0xca, 0xc3, 0x61, 0x99, 0xbe, 0xd0,
    0xe3, 0x37, 0xac, 0xc9, 0xcd, 0xb9, 0x8d, 0x72, 0x65, 0xca, 0x8d, 0x1a,
    0x9f, 0x1d, 0xc0, 0x98, 0xa8, 0x20, 0x0d, 0x42, 0xe4, 0x03, 0x63, 0xb8,
    0x16, 0x37, 0xb7, 0x3f, 0xd0, 0x3d, 0x2b, 0xa9, 0x52, 0xa8, 0xb5, 0x1b,
    0xc2, 0x9e, 0xa9, 0x3e, 0x0a, 0xac, 0xea, 0x96, 0x92, 0xef, 0x8e, 0x02,
    0x48, 0xb6, 0xc3, 0x6d, 0xcf, 0x3e, 0xc3, 0x1e, 0x39, 0x2a, 0x3c, 0x19,
    0xad, 0x95, 0x46, 0xc8, 0xae, 0xb0, 0x11, 0x75, 0x17, 0x9d, 0x6d, 0x94,
    0x22, 0xd7, 0xb5, 0xb5, 0x38, 0x40, 0xf6, 0xe4, 0x9f, 0x6c, 0x13, 0x55,
    0x4e, 0xbe, 0x89, 0x0e, 0xb1, 0x0b, 0x25, 0x25, 0xf0, 0xcb, 0xab, 0x6c,
    0x3a, 0xd8, 0x6d, 0x68, 0x5d, 0xb7, 0xd8, 0xa4, 0xd8, 0xd8, 0x11, 0x72,
    0x0d, 0xb1, 0x5d, 0x63, 0x05, 0xca, 0xa1, 0x29, 0x42, 0x5c, 0xb0, 0x1a,
    0x00, 0x3f, 0x7b, 0xf9, 0x98, 0x4a, 0xfe, 0xb9, 0x5b, 0x71, 0xc2, 0x96,
    0xa4, 0x14, 0xa2, 0x79, 0xe5, 0x97, 0x90, 0xb5, 0xba, 0x46, 0xf3, 0x75,
    0x37, 0x73, 0x65, 0x1e, 0xb1, 0x53, 0x96, 0xd2, 0x51, 0x25, 0xe8, 0x2a,
    0x43, 0x89, 0x4b, 0x6a, 0x4e, 0xe1, 0xb2, 0x01, 0x37, 0xee, 0x7f, 0x7b,
    0x13, 0x8d, 0x2a, 0xad, 0x06, 0xac, 0xed, 0x0f, 0x2f, 0xcd, 0x85, 0x49,
    0x95, 0x22, 0x3b, 0xb4, 0xb6, 0x42, 0x5c, 0x61, 0xa5, 0x2c, 0x13, 0xa4,
    0x13, 0x7b, 0x0d, 0x8d, 0xcf, 0x7c, 0x7d, 0x48, 0x77, 0xa8, 0xd2, 0xe9,
    0xef, 0xf8, 0x39, 0x6d, 0x2c, 0xc7, 0x28, 0x50, 0x71, 0x20, 0x04, 0xdd,
    0x36, 0xdc, 0x12, 0x2e, 0x78, 0xf4, 0xdf, 0x1e, 0x1d, 0x31, 0xce, 0x99,
    0xaa, 0x9b, 0x96, 0xc5, 0x39, 0xfa, 0xb5, 0x3a, 0x0e, 0x95, 0xa9, 0xa6,
    0x93, 0xf1, 0x0a, 0x42, 0x96, 0x80, 0x4d, 0xb5, 0x24, 0x0b, 0x0b, 0x5e,
    0xc0, 0x92, 0x36, 0x03, 0x1a, 0xeb, 0xf4, 0xd0, 0xe4, 0xa3, 0x52, 0xa8,
    0x24, 0xf0, 0xee, 0x6c, 0x32, 0x1c, 0xc9, 0xb0, 0x83, 0x38, 0x0e, 0x7a,
    0xa5, 0x86, 0x2a, 0x53, 0x35, 0x2a, 0x8b, 0x25, 0x01, 0xdd, 0x12, 0x02,
    0x95, 0x9f, 0x44, 0x82, 0x72, 0x19, 0xe9, 0x68, 0x13, 0x23, 0x20, 0x67,
    0x9a, 0xa2, 0xd2, 0x5a, 0xa5, 0xaa, 0x24, 0x72, 0x6c, 0x16, 0xf1, 0xd3,
    0x72, 0x76, 0x16, 0xc3, 0x36, 0x42, 0xe9, 0x8e, 0x68, 0xa5, 0x3a, 0xec,
    0x31, 0x5e, 0xa2, 0x35, 0x2a, 0x45, 0x99, 0x0d, 0xf8, 0xda, 0x9c, 0x4a,
    0xb7, 0x16, 0x00, 0x7e, 0x6d, 0xef, 0xf5, 0x03, 0x04, 0x17, 0x54, 0xcc,
    0x09, 0x74, 0x3e, 0xac, 0xcd, 0x1d, 0x85, 0x83, 0xa8, 0x29, 0x0b, 0x2a,
    0x50, 0xf7, 0x1e, 0x71, 0x8d, 0x79, 0xf9, 0x8e, 0xb7, 0xf0, 0x4e, 0xd3,
    0x0e, 0x64, 0x68, 0xb6, 0xe5, 0xd0, 0xa9, 0xa1, 0xd8, 0xed, 0xad, 0x77,
    0x37, 0x2a, 0xb6, 0x92, 0xb2, 0x77, 0x3c, 0xaa, 0xfe, 0xf8, 0x0a, 0xde,
    0x16, 0x9c, 0x29, 0xe0, 0x20, 0x01, 0xd4, 0x7d, 0x2f, 0x05, 0x2a, 0xbf,
    0x13, 0xe8, 0x73, 0x93, 0x05, 0xd2, 0x54, 0xe2, 0x8f, 0x81, 0x00, 0x0f,
    0x00, 0x48, 0x19, 0x72, 0xd4, 0xc1, 0x8c, 0xbf, 0xd0, 0xc9, 0x10, 0xa5,
    0x2e, 0x54, 0x9c, 0xe1, 0x19, 0x84, 0xa9, 0x65, 0x4a, 0x07, 0x65, 0x13,
    0xde, 0xfa, 0xad, 0x7d, 0xfd, 0x71, 0x4a, 0xcf, 0x79, 0x96, 0x83, 0x46,
    0xca, 0xb1, 0x62, 0xbb, 0x5c, 0x8b, 0x29, 0xf8, 0xcf, 0x32, 0xe2, 0xfc,
    0x35, 0x02, 0x43, 0x6d, 0x24, 0xdd, 0x4a, 0xb6, 0xc3, 0xf2, 0x8d, 0xf9,
    0x27, 0x10, 0x77, 0x17, 0x45, 0xf1, 0x94, 0x26, 0x66, 0xb9, 0x4f, 0xac,
    0x9e, 0x4a, 0x91, 0x73, 0xf7, 0xb1, 0xc7, 0xb5, 0x7b, 0x2a, 0x51, 0x6a,
    0x79, 0x3e, 0x5c, 0xc8, 0x95, 0x19, 0xcf, 0xa9, 0xa4, 0xeb, 0xb2, 0x9f,
    0x51, 0x06, 0xde, 0xa3, 0x83, 0x82, 0xac, 0xe1, 0xa9, 0x86, 0x7b, 0xe9,
    0x29, 0xe2, 0xea, 0x4f, 0xd2, 0x03, 0x2f, 0xe2, 0x75, 0x19, 0xa7, 0x12,
    0x7b, 0x27, 0x2d, 0x71, 0x6e, 0xe8, 0x02, 0xfb, 0x5c, 0xdc, 0x9b, 0x40,
    0x2c, 0xcf, 0x57, 0xa5, 0xc9, 0x6e, 0x75, 0x35, 0x55, 0xa5, 0xb1, 0x19,
    0xc7, 0x54, 0x7c, 0x26, 0x93, 0x7b, 0x79, 0x89, 0xb0, 0x3b, 0x6d, 0x85,
    0x08, 0xd4, 0x6c, 0xa4, 0xf4, 0x84, 0x80, 0x9a, 0x9c, 0x85, 0x13, 0x62,
    0xb2, 0xda, 0x95, 0xfa, 0x01, 0x8e, 0x87, 0x43, 0xf1, 0x29, 0x61, 0x8a,
    0x76, 0x5c, 0xc9, 0xf4, 0x09, 0x50, 0x62, 0x46, 0x6c, 0x27, 0xc6, 0x6c,
    0xba, 0xe3, 0xde, 0x51, 0xa9, 0x45, 0x77, 0xe4, 0x9b, 0xf0, 0x0f, 0xdf,
    0x18, 0xff, 0x00, 0x54, 0x59, 0xa4, 0x7f, 0x66, 0x3a, 0x74, 0xc4, 0x37,
    0x00, 0xf9, 0xda, 0x4a, 0x54, 0x01, 0xf6, 0xf2, 0x7f, 0xa8, 0x18, 0x2d,
    0x3f, 0x3d, 0x34, 0x5b, 0x4a, 0x9e, 0x65, 0x27, 0xc8, 0x9b, 0x75, 0x17,
    0xfa, 0x43, 0xf4, 0xae, 0x1a, 0x69, 0x94, 0x87, 0x99, 0x96, 0x48, 0x0b,
    0xef, 0x1b, 0x01, 0xbe, 0xe7, 0x33, 0x6f, 0x41, 0x12, 0x68, 0xb9, 0x4d,
    0x86, 0x23, 0x7c, 0x44, 0x6a, 0x0d, 0x5a, 0x53, 0x62, 0xca, 0xf2, 0xc5,
    0x74, 0x5c, 0x7d, 0x54, 0x9b, 0x71, 0xef, 0x82, 0xf0, 0x32, 0x95, 0x69,
    0x6f, 0x29, 0x11, 0xb2, 0x24, 0x97, 0xd2, 0x4d, 0xd2, 0x5f, 0x6c, 0xa4,
    0xe9, 0x3c, 0x5c, 0x0b, 0x8c, 0x33, 0x2b, 0xac, 0x14, 0x79, 0x95, 0x37,
    0x6a, 0x2b, 0xca, 0x2e, 0xbb, 0x31, 0x0b, 0x4b, 0x8a, 0x2a, 0x71, 0xd2,
    0x02, 0xbb, 0x10, 0x92, 0x42, 0x47, 0x1c, 0x01, 0x82, 0x8b, 0xeb, 0x4e,
    0x66, 0x76, 0x58, 0x71, 0x14, 0xe9, 0x11, 0x12, 0xf0, 0xd4, 0x35, 0x35,
    0x74, 0xda, 0xc3, 0x72, 0x49, 0x36, 0xda, 0xdc, 0xe1, 0x69, 0x58, 0x8d,
    0x6d, 0x9e, 0xe8, 0x48, 0xe8, 0x91, 0x1d, 0x84, 0xc2, 0x19, 0x3c, 0x29,
    0x09, 0x1e, 0x9f, 0x40, 0x60, 0x2b, 0x99, 0x1f, 0xa9, 0x12, 0x68, 0x52,
    0xa0, 0xc4, 0xa0, 0xc3, 0xa7, 0x46, 0x76, 0x3a, 0xdb, 0x5a, 0x02, 0x14,
    0x14, 0xa4, 0x11, 0xba, 0x45, 0xc5, 0x85, 0xfd, 0x70, 0x13, 0xa5, 0xd9,
    0x39, 0xd8, 0xcb, 0xf0, 0x5a, 0x9d, 0x32, 0x9a, 0xea, 0xd6, 0x59, 0x90,
    0x1a, 0x59, 0x2a, 0x4a, 0xfd, 0x2c, 0x38, 0xb5, 0xb1, 0xd1, 0x7d, 0x1c,
    0xcd, 0x33, 0xb3, 0x35, 0x0d, 0xea, 0x9c, 0x97, 0x94, 0xb2, 0xd4, 0xa2,
    0xc3, 0x8d, 0xad, 0x20, 0x8b, 0x69, 0x06, 0xe0, 0x81, 0xf5, 0x18, 0x8b,
    0xe7, 0x5f, 0x8a, 0xa4, 0x75, 0x36, 0xac, 0x23, 0xad, 0x49, 0x40, 0x9e,
    0x56, 0x50, 0xa2, 0x40, 0x20, 0x80, 0x47, 0x18, 0x15, 0x58, 0x9f, 0x76,
    0x71, 0xa4, 0xba, 0xb5, 0x5c, 0x79, 0x0f, 0xbd, 0x20, 0x7c, 0xea, 0xbf,
    0x11, 0x72, 0x75, 0x1f, 0x28, 0x97, 0x3a, 0xc5, 0x52, 0xb5, 0x29, 0x70,
    0x68, 0xb2, 0xdc, 0x8f, 0x31, 0x0d, 0x97, 0x95, 0xe1, 0x91, 0x7d, 0xd4,
    0x76, 0xbd, 0xf6, 0xb5, 0xb9, 0xde, 0xfb, 0x61, 0x36, 0xb9, 0xfc, 0xcf,
    0x47, 0x7c, 0xb5, 0x57, 0x7e, 0xb3, 0xc6, 0xea, 0x4c, 0x85, 0x14, 0x1f,
    0xdf, 0x0d, 0x9d, 0x3b, 0x8f, 0x36, 0x99, 0x9b, 0xeb, 0x4b, 0x1a, 0x89,
    0x6a, 0x31, 0xb1, 0x4f, 0x3a, 0x49, 0x2b, 0xb8, 0x27, 0xd8, 0x7e, 0xa7,
    0x1e, 0x39, 0x6b, 0xa9, 0x5f, 0x1c, 0xfb, 0xb1, 0x6b, 0xd1, 0x14, 0xe2,
    0x5c, 0x73, 0x50, 0x79, 0xb4, 0xa5, 0x6a, 0x48, 0x1c, 0x02, 0x95, 0x6c,
    0xa0, 0x2f, 0xc6, 0x1b, 0xa9, 0x32, 0xf2, 0x32, 0xf2, 0x4d, 0xad, 0x49,
    0x4f, 0x12, 0xaf, 0x9a, 0x80, 0x39, 0xf9, 0xe9, 0x12, 0x49, 0xfa, 0xa5,
    0x4c, 0x4f, 0xbc, 0x86, 0x87, 0x1b, 0x68, 0xb6, 0x40, 0xd8, 0x8b, 0xf2,
    0x84, 0x96, 0xaa, 0xce, 0xc8, 0x64, 0xc4, 0x65, 0x87, 0x54, 0xca, 0xc5,
    0x94, 0x84, 0xb2, 0x92, 0x55, 0xbd, 0xcf, 0x98, 0xdc, 0xfe, 0xf8, 0xd9,
    0x4b, 0xae, 0x34, 0x14, 0xf2, 0xe8, 0x0c, 0x21, 0x29, 0xdc, 0xf8, 0x82,
    0xc7, 0xf4, 0xbe, 0x2a, 0xa7, 0x28, 0xb1, 0x2d, 0x46, 0xab, 0x95, 0xaa,
    0x10, 0x52, 0xf2, 0x95, 0xe2, 0xa5, 0xa4, 0x37, 0x76, 0xaf, 0x6e, 0x34,
    0x9d, 0xc7, 0xaf, 0xd7, 0x08, 0x35, 0x18, 0xf2, 0x20, 0xd4, 0x16, 0xc5,
    0x5d, 0x2a, 0x65, 0xf0, 0x8b, 0x8f, 0x35, 0xc1, 0xed, 0xa8, 0x11, 0xee,
    0x38, 0xc7, 0x5a, 0x85, 0x4e, 0xa9, 0x4f, 0x1c, 0x69, 0x42, 0x78, 0x79,
    0x81, 0xf7, 0x6f, 0x94, 0x68, 0xa3, 0xcc, 0x53, 0x6a, 0xaa, 0x2d, 0xa5,
    0x44, 0x2f, 0x74, 0x9b, 0x83, 0xe3, 0xd6, 0x19, 0xfa, 0x3f, 0x06, 0x8b,
    0x5b, 0xcf, 0xf4, 0x09, 0x4f, 0xc0, 0x43, 0x5e, 0x12, 0x1d, 0x79, 0x2d,
    0xa1, 0x45, 0x20, 0xba, 0x84, 0x9d, 0x17, 0x23, 0x7b, 0x03, 0x63, 0xf6,
    0xb6, 0x3a, 0x0e, 0xbb, 0xd3, 0xa6, 0x44, 0x24, 0x0c, 0xbd, 0x3d, 0xba,
    0x6c, 0xb2, 0xd8, 0x21, 0xb5, 0x30, 0x85, 0xa1, 0x47, 0xd4, 0x02, 0x2e,
    0x2f, 0xed, 0x8e, 0x7e, 0xfc, 0x3c, 0x32, 0xb7, 0xb3, 0xe5, 0x27, 0x72,
    0xa4, 0x21, 0x52, 0x75, 0x90, 0x3e, 0x54, 0xe8, 0x51, 0xfe, 0x98, 0x6c,
    0xfc, 0x47, 0xe7, 0xdc, 0xd7, 0x94, 0xb3, 0xed, 0x1d, 0xda, 0x05, 0x5d,
    0x71, 0xdb, 0x11, 0x0a, 0x8b, 0x7a, 0x02, 0x9b, 0x76, 0xe4, 0x6c, 0xb4,
    0x91, 0xb8, 0xed, 0x89, 0xee, 0x39, 0x99, 0xaa, 0xb9, 0x88, 0xd8, 0x44,
    0x8b, 0xe5, 0xb2, 0x5a, 0x0a, 0xf0, 0xdf, 0x23, 0xcf, 0x6d, 0x61, 0xef,
    0x0d, 0x38, 0xdc, 0x9d, 0x35, 0xc5, 0x9f, 0xca, 0xb2, 0x0e, 0xe6, 0xd7,
    0x00, 0x7c, 0xe0, 0x1f, 0x50, 0x8f, 0x55, 0xf2, 0xa3, 0xaa, 0x45, 0x4a,
    0x04, 0x79, 0xb0, 0x55, 0xb8, 0x91, 0x19, 0x8d, 0x4d, 0xaa, 0xde, 0xa3,
    0x80, 0x47, 0xd3, 0x09, 0xea, 0xae, 0xf5, 0x11, 0x6d, 0xa5, 0xc7, 0x1c,
    0x99, 0x15, 0xbd, 0x65, 0x01, 0x49, 0x8b, 0xa0, 0x6a, 0xb5, 0xf4, 0x8b,
    0x27, 0x9d, 0x8f, 0xe9, 0x8b, 0x87, 0x4c, 0xba, 0xd9, 0x4b, 0xce, 0x4d,
    0xff, 0x00, 0x07, 0xcc, 0x94, 0x71, 0x16, 0x72, 0x80, 0x05, 0x6d, 0x5d,
    0x71, 0xdd, 0x27, 0x61, 0x71, 0xc8, 0x37, 0x3d, 0xef, 0xcf, 0x38, 0x79,
    0x9f, 0x93, 0x8c, 0x58, 0x4e, 0xd4, 0x68, 0xf2, 0x4b, 0x4c, 0xdb, 0x53,
    0x8c, 0x2c, 0x6a, 0x09, 0x23, 0x6f, 0x29, 0x3b, 0x8f, 0x4f, 0xbe, 0x04,
    0x9c, 0x77, 0x52, 0x92, 0x58, 0x97, 0xa9, 0xdd, 0x2a, 0xe6, 0x0e, 0x47,
    0xd2, 0x0e, 0x25, 0x2e, 0x4c, 0x27, 0x8d, 0xa7, 0x8d, 0x8e, 0xd7, 0xb7,
    0x95, 0xf5, 0xf5, 0xf5, 0x89, 0x1f, 0x47, 0xa4, 0x67, 0x06, 0x27, 0x4a,
    0x35, 0xc7, 0x25, 0x86, 0xd4, 0xd1, 0x08, 0xf1, 0x52, 0x52, 0x1c, 0xba,
    0x55, 0x71, 0xbf, 0xd0, 0x1f, 0xbe, 0x00, 0xaf, 0x5c, 0xa8, 0x5a, 0x94,
    0x10, 0x5a, 0x84, 0xcb, 0x81, 0x5c, 0x27, 0x48, 0xf1, 0xd4, 0x3e, 0xe4,
    0x95, 0x7b, 0x9f, 0xd3, 0x14, 0x2f, 0xe6, 0xaa, 0x6d, 0x4d, 0x6c, 0xd3,
    0xdb, 0x42, 0x5a, 0x79, 0x32, 0x1d, 0x2a, 0x4a, 0x2e, 0x92, 0x2c, 0x40,
    0xb6, 0xae, 0x49, 0xb0, 0xb9, 0x3e, 0xf8, 0x47, 0xcc, 0xb4, 0xc3, 0x4a,
    0x6d, 0x6b, 0x2e, 0x25, 0xc2, 0xe4, 0xa7, 0xc1, 0x08, 0xb8, 0x03, 0x4b,
    0xa4, 0xff, 0x00, 0xa1, 0x18, 0x6f, 0xa9, 0xce, 0x2e, 0x6a, 0x8a, 0xda,
    0x9c, 0x37, 0x25, 0x47, 0xdb, 0xf9, 0x87, 0x1c, 0x20, 0xa3, 0xda, 0x25,
    0xc3, 0x9a, 0x80, 0x56, 0xb9, 0x9d, 0xb7, 0x85, 0x77, 0xe9, 0xb4, 0xfa,
    0xab, 0x28, 0x11, 0xe7, 0x2e, 0x39, 0x24, 0x85, 0xad, 0x06, 0xc1, 0x06,
    0xfe, 0xd7, 0xfb, 0xed, 0x82, 0x39, 0x17, 0x2c, 0xd0, 0xd9, 0x9e, 0xf4,
    0x3c, 0xcf, 0x54, 0x5f, 0x88, 0xbb, 0x18, 0x8b, 0x5b, 0xc4, 0xb6, 0xbd,
    0xf9, 0x0a, 0x1d, 0xfd, 0xb6, 0xf4, 0xb6, 0x25, 0x94, 0xb9, 0x92, 0x62,
    0x49, 0x75, 0xd8, 0xce, 0xa9, 0xb5, 0x29, 0xc5, 0x6d, 0x7d, 0xb9, 0x3b,
    0x1c, 0x51, 0x28, 0x4e, 0x48, 0xaa, 0xd2, 0x5e, 0x79, 0xf8, 0xbe, 0x46,
    0x52, 0x95, 0x2c, 0x84, 0xdd, 0x04, 0x15, 0x69, 0xe7, 0xd6, 0xfd, 0xb0,
    0x15, 0xd3, 0x31, 0x2c, 0x9b, 0x71, 0x92, 0x9e, 0xb0, 0x2a, 0x5d, 0x9a,
    0x26, 0x2b, 0x50, 0x2a, 0x68, 0x31, 0x30, 0xad, 0xc0, 0x04, 0x28, 0xfb,
    0x67, 0xd7, 0x3f, 0x13, 0x15, 0xe8, 0x99, 0x07, 0x2d, 0x48, 0x6d, 0x0d,
    0xc5, 0x61, 0x12, 0xdc, 0x5b, 0x6e, 0x5d, 0xa5, 0x2f, 0xec, 0x13, 0xb8,
    0x3c, 0x1b, 0x0d, 0xf7, 0xbd, 0xce, 0x05, 0xc1, 0xa4, 0x39, 0x43, 0xca,
    0x15, 0x0a, 0x3b, 0xad, 0x84, 0x2e, 0x3b, 0x6f, 0xb4, 0x42, 0x56, 0x5c,
    0x06, 0xc5, 0x44, 0x79, 0xac, 0x2e, 0x34, 0xdb, 0x7b, 0x0c, 0x2b, 0x65,
    0x9a, 0xcd, 0x47, 0x2d, 0xa0, 0xaa, 0x33, 0xea, 0x72, 0x1b, 0xab, 0xb2,
    0xa3, 0xea, 0x1a, 0xae, 0x3c, 0xd7, 0x07, 0xe6, 0xb5, 0xff, 0x00, 0xec,
    0x61, 0xd2, 0x7e, 0x67, 0x6b, 0x30, 0x49, 0x9a, 0xb0, 0x9f, 0x04, 0x2c,
    0x36, 0x1f, 0xb6, 0xc0, 0x21, 0x68, 0x20, 0x8d, 0xb8, 0xb5, 0xb0, 0xc7,
    0x84, 0x66, 0x82, 0xa7, 0x14, 0x39, 0xa4, 0xfd, 0x22, 0x3f, 0xf1, 0x5f,
    0x0e, 0xcc, 0x52, 0x24, 0xd0, 0xdb, 0x80, 0x1b, 0x2d, 0x26, 0xe3, 0x96,
    0x63, 0xe7, 0x1a, 0xf4, 0xf9, 0xb2, 0xe3, 0xe5, 0xb9, 0x2f, 0xb6, 0x99,
    0x09, 0x53, 0x71, 0xf4, 0x25, 0xc4, 0x70, 0x8b, 0x2b, 0x7b, 0xff, 0x00,
    0x97, 0xd3, 0x08, 0x59, 0x5f, 0xab, 0x71, 0xd2, 0xa5, 0xc5, 0xae, 0x53,
    0x64, 0x25, 0xbd, 0x56, 0xf8, 0x86, 0x5e, 0x2a, 0x5a, 0x2c, 0x6f, 0x7b,
    0x1e, 0x37, 0xdf, 0x6c, 0x58, 0xd2, 0xdd, 0x2e, 0xa5, 0xd2, 0x9a, 0x1f,
    0xf0, 0xc2, 0xcf, 0xc5, 0xae, 0x0b, 0xc2, 0x62, 0x19, 0xf9, 0x8d, 0xd3,
    0xb2, 0x96, 0x3e, 0xa9, 0xef, 0x8e, 0x46, 0x7d, 0x6b, 0x8b, 0x52, 0x93,
    0x09, 0xc4, 0x95, 0xa1, 0x0f, 0xad, 0x36, 0x1c, 0xde, 0xe4, 0x6d, 0x83,
    0x75, 0x4a, 0x93, 0xe8, 0x29, 0x71, 0xbc, 0xb6, 0x3e, 0x36, 0x8a, 0x83,
    0xd5, 0xe9, 0x96, 0x29, 0xb2, 0x4e, 0xb2, 0x78, 0x41, 0x4d, 0x88, 0x23,
    0x5b, 0x00, 0x05, 0xf7, 0xd8, 0xe9, 0x1d, 0x04, 0xec, 0xac, 0xbb, 0x9a,
    0xa0, 0xae, 0x45, 0x3a, 0xb4, 0x15, 0x2d, 0xb2, 0x16, 0xd1, 0x69, 0x94,
    0xb6, 0xf2, 0x8d, 0xff, 0x00, 0x38, 0xb5, 0x96, 0x07, 0x3c, 0x5f, 0x1f,
    0x12, 0x8e, 0x69, 0x0d, 0x33, 0x26, 0xa0, 0x83, 0x52, 0x84, 0x96, 0xf4,
    0x78, 0x91, 0xd0, 0x94, 0x92, 0x9b, 0x8f, 0x99, 0x36, 0xb1, 0xe3, 0x8d,
    0xb0, 0x1f, 0xa4, 0x19, 0x3e, 0x7c, 0x69, 0x10, 0xf3, 0x0b, 0x8c, 0x7c,
    0x38, 0x68, 0x97, 0x63, 0xb0, 0xf1, 0xf3, 0xba, 0x46, 0xe0, 0x1e, 0x39,
    0xf4, 0xed, 0x83, 0x15, 0xac, 0xf6, 0xed, 0x26, 0x24, 0x88, 0x34, 0x94,
    0xaf, 0xc6, 0xd4, 0x4b, 0xcb, 0x6d, 0x37, 0xd1, 0x7d, 0xac, 0x3d, 0xf9,
    0xdb, 0x9c, 0x63, 0x5c, 0x8c, 0xac, 0xcc, 0xa9, 0x76, 0x75, 0x01, 0xbe,
    0x56, 0xca, 0xfe, 0x50, 0xc0, 0xd2, 0x18, 0x9a, 0x94, 0x33, 0x33, 0xcd,
    0x04, 0x11, 0xbd, 0xed, 0x97, 0x80, 0xd7, 0xa0, 0xb1, 0x31, 0x7b, 0xe9,
    0x12, 0xe8, 0xec, 0xe5, 0x1b, 0xd3, 0x65, 0x25, 0x2b, 0x7e, 0x38, 0x94,
    0xb2, 0xa5, 0x15, 0x0b, 0x8d, 0x8d, 0xc6, 0xd6, 0x20, 0x1d, 0xc0, 0xe0,
    0x8c, 0x47, 0xfa, 0xd7, 0x35, 0x11, 0x33, 0xed, 0x55, 0xe7, 0x8b, 0x6a,
    0xd4, 0x84, 0xb8, 0x46, 0xab, 0x0b, 0x80, 0x47, 0xef, 0x6c, 0x6c, 0xf4,
    0x0a, 0x53, 0xf3, 0xe4, 0xd4, 0x59, 0xbc, 0x86, 0x5b, 0x4c, 0x72, 0xd8,
    0x4b, 0xbb, 0x06, 0xf5, 0x21, 0x57, 0x1f, 0x5b, 0x81, 0x7c, 0x2a, 0x7e,
    0x26, 0x9b, 0x7e, 0x36, 0x64, 0x8b, 0xe6, 0x4b, 0xa8, 0x98, 0xd2, 0x3c,
    0xac, 0x82, 0xa2, 0xa0, 0x9b, 0x77, 0x23, 0x6d, 0xf7, 0xc2, 0x83, 0xbc,
    0x0b, 0x68, 0x34, 0x9d, 0x02, 0xb2, 0xe9, 0x09, 0xce, 0x80, 0x86, 0x94,
    0xb4, 0x1b, 0x8b, 0x9d, 0x75, 0xb5, 0xf2, 0x8d, 0x18, 0x68, 0x47, 0xf3,
    0x58, 0x60, 0x04, 0x24, 0x96, 0x46, 0xab, 0xaa, 0xdb, 0x59, 0x69, 0x23,
    0x7e, 0xe4, 0xf6, 0x18, 0x8c, 0x65, 0x88, 0x92, 0xe5, 0xd7, 0xfe, 0x12,
    0x23, 0x0b, 0x79, 0xd5, 0x28, 0xa7, 0x4a, 0x47, 0xbf, 0xed, 0x8b, 0x58,
    0x8b, 0x1d, 0x59, 0xe9, 0xb6, 0xd6, 0xa0, 0xd2, 0x1c, 0x80, 0x95, 0x6c,
    0x6c, 0x51, 0xe6, 0x26, 0xff, 0x00, 0x6e, 0x70, 0x0e, 0xa7, 0x99, 0x72,
    0xbe, 0x49, 0x79, 0xf6, 0x29, 0x30, 0x99, 0x9c, 0xe3, 0xaa, 0xd6, 0xe0,
    0x4a, 0x8a, 0x75, 0x2b, 0x7b, 0x82, 0x46, 0xf6, 0x3d, 0xc6, 0x1e, 0xa4,
    0xe4, 0x91, 0x35, 0x4c, 0x61, 0x4e, 0x2a, 0xc9, 0x48, 0x37, 0xf5, 0x88,
    0xd4, 0xe4, 0xeb, 0xb2, 0xb5, 0x59, 0x86, 0xd8, 0x41, 0x5a, 0xd4, 0x13,
    0x6e, 0x43, 0x5c, 0xc9, 0xf3, 0x86, 0x4c, 0xb5, 0x16, 0x9f, 0x93, 0x61,
    0x8a, 0x85, 0x46, 0xa4, 0xc7, 0x8e, 0xa4, 0xa8, 0x2d, 0x09, 0x57, 0x97,
    0x56, 0xdb, 0x7b, 0x9d, 0xb0, 0xbd, 0x9d, 0x69, 0xd5, 0x4c, 0xc7, 0x25,
    0xea, 0xab, 0x2c, 0x06, 0xda, 0x6e, 0x3a, 0xec, 0x93, 0xf3, 0x2a, 0xf6,
    0x55, 0xcd, 0xb8, 0xe0, 0x58, 0x5c, 0xfb, 0xde, 0xf8, 0x9b, 0xe6, 0x0c,
    0xd3, 0x2b, 0x30, 0xd5, 0x05, 0x4a, 0xa4, 0xab, 0xbf, 0xa9, 0x21, 0x0d,
    0x36, 0x90, 0x96, 0x9a, 0x48, 0xb5, 0x92, 0x00, 0x1e, 0x82, 0xd8, 0x7d,
    0x9d, 0xd4, 0x5a, 0x62, 0xe8, 0x72, 0x22, 0xc7, 0x6d, 0xe5, 0x3a, 0xe4,
    0x52, 0xd0, 0x48, 0x05, 0x37, 0x51, 0xb0, 0xd4, 0x3d, 0x3b, 0xfe, 0xd8,
    0xc7, 0x51, 0x9f, 0x79, 0xf6, 0x44, 0xac, 0x93, 0x47, 0x83, 0xa1, 0x37,
    0xb7, 0xdf, 0x58, 0xdf, 0x45, 0xa1, 0xa1, 0x89, 0x83, 0x3f, 0x38, 0xbe,
    0x27, 0x8f, 0x90, 0x17, 0xe5, 0x19, 0xf8, 0x7e, 0xa9, 0x7f, 0x0c, 0xcf,
    0x74, 0xe6, 0x41, 0x4a, 0xd2, 0xeb, 0xce, 0xb3, 0xa8, 0xde, 0xe0, 0x68,
    0x5e, 0xff, 0x00, 0xed, 0x86, 0xef, 0xc4, 0xae, 0x57, 0xad, 0x66, 0x9c,
    0xf5, 0x96, 0xa9, 0x99, 0x7a, 0x9d, 0x22, 0xa3, 0x2d, 0xe8, 0xaa, 0x40,
    0x08, 0x48, 0x09, 0x16, 0x37, 0x25, 0x44, 0xec, 0x91, 0x6d, 0xee, 0x48,
    0x18, 0x40, 0xe9, 0x72, 0x97, 0x1f, 0x38, 0xd0, 0xe4, 0x4c, 0x01, 0x82,
    0xed, 0x49, 0xc4, 0x5d, 0xc5, 0x00, 0x2e, 0xa4, 0xe9, 0x1b, 0xf1, 0xc9,
    0xb6, 0x3a, 0x7f, 0x38, 0x67, 0x5a, 0x2e, 0x59, 0xa6, 0xa5, 0xe0, 0x17,
    0x2e, 0xa6, 0xb8, 0xfa, 0x52, 0xdb, 0x00, 0xd8, 0x26, 0xfb, 0x6b, 0x20,
    0x5c, 0xef, 0xc2, 0x45, 0xf0, 0x93, 0x8e, 0x17, 0x35, 0x2b, 0x5d, 0x93,
    0x71, 0x96, 0x8a, 0xd6, 0x59, 0x02, 0xde, 0x3d, 0xe1, 0xed, 0xbc, 0x50,
    0xb0, 0xeb, 0x0d, 0xcd, 0x53, 0xdf, 0x6c, 0x1d, 0x56, 0x6f, 0xe5, 0x63,
    0xf4, 0x85, 0xbe, 0x94, 0xf4, 0x72, 0x85, 0xd3, 0xd8, 0x08, 0xcc, 0xb9,
    0x96, 0x4b, 0x33, 0xab, 0x2c, 0xd8, 0x87, 0x41, 0xbc, 0x68, 0x8b, 0xec,
    0x1b, 0x07, 0xff, 0x00, 0x51, 0x63, 0xfb, 0xdd, 0xbb, 0x0e, 0xf8, 0xdb,
    0xcc, 0xd9, 0xc6, 0x7d, 0x6e, 0x97, 0x22, 0x99, 0x97, 0xdc, 0x30, 0xd2,
    0xa6, 0xd4, 0x18, 0x23, 0xe7, 0x70, 0xff, 0x00, 0x7d, 0x44, 0xf0, 0x2f,
    0xe9, 0xbd, 0xed, 0xef, 0x88, 0x96, 0x6e, 0xcd, 0xdd, 0x45, 0xce, 0x15,
    0x44, 0x29, 0xea, 0x7c, 0xe4, 0x42, 0x6a, 0xe1, 0xa6, 0x49, 0x08, 0xda,
    0xe7, 0x7b, 0x5f, 0xcb, 0x7f, 0x41, 0xe9, 0xdf, 0x07, 0xa9, 0xaf, 0xe7,
    0x01, 0x14, 0xb6, 0xb8, 0x4d, 0x0f, 0x11, 0x21, 0x09, 0x40, 0x76, 0xe5,
    0x24, 0xef, 0x60, 0x07, 0xdf, 0x6c, 0x0f, 0x6b, 0x07, 0xd4, 0xa6, 0xdd,
    0x13, 0x53, 0xa8, 0x2b, 0x5e, 0xc2, 0xdd, 0xd4, 0xf4, 0x82, 0xcd, 0x87,
    0x52, 0x9e, 0xce, 0x55, 0xb2, 0x40, 0xde, 0xc7, 0xda, 0xff, 0x00, 0x3f,
    0xe6, 0x14, 0xb2, 0x2c, 0x09, 0x51, 0x33, 0x8c, 0x66, 0x96, 0xe1, 0x7d,
    0xc5, 0x2d, 0xc4, 0x94, 0x29, 0xcb, 0xf9, 0x86, 0xc6, 0xe4, 0x5e, 0xdd,
    0xf0, 0x56, 0x6c, 0xc9, 0x93, 0x95, 0x53, 0x69, 0x65, 0xc7, 0x52, 0xcc,
    0xa7, 0x94, 0x02, 0xd5, 0xa8, 0xa0, 0x6b, 0xde, 0xde, 0xbd, 0xbf, 0x4c,
    0x39, 0xe4, 0xec, 0x9f, 0x57, 0x77, 0x33, 0x35, 0x50, 0x9f, 0x4e, 0x44,
    0x25, 0x34, 0xab, 0xa5, 0xd0, 0x8d, 0x3a, 0x81, 0xd8, 0x85, 0x71, 0xc7,
    0x37, 0xe4, 0x93, 0x84, 0x99, 0x2f, 0xb0, 0xd5, 0x72, 0xae, 0x88, 0xe8,
    0x7a, 0x4a, 0xdb, 0x98, 0xf0, 0x75, 0x09, 0x68, 0x9d, 0x0a, 0xd5, 0xb0,
    0xbf, 0x1b, 0xdb, 0x0e, 0x75, 0x19, 0x29, 0x84, 0x51, 0xd0, 0x85, 0xa0,
    0xf1, 0xf1, 0xde, 0xdb, 0xd8, 0x8f, 0xda, 0x1b, 0xb0, 0xc3, 0x8c, 0xc8,
    0x9f, 0xee, 0x94, 0x1b, 0x04, 0x2b, 0xfc, 0x8d, 0xb6, 0x1b, 0x9e, 0x86,
    0x13, 0x32, 0xee, 0x51, 0x95, 0x26, 0x62, 0x97, 0x2b, 0xca, 0x85, 0x38,
    0x74, 0x36, 0x8d, 0xd4, 0xab, 0x9d, 0xbe, 0x98, 0x69, 0xa8, 0xfc, 0x0d,
    0x1c, 0x22, 0x1c, 0x55, 0x32, 0x08, 0x1b, 0xa5, 0x0e, 0x6a, 0x08, 0xf5,
    0x04, 0x8e, 0x4f, 0x7d, 0xb1, 0xf9, 0x3a, 0x8f, 0x99, 0xea, 0x68, 0x4a,
    0x63, 0xc8, 0x85, 0x01, 0xbe, 0x74, 0x97, 0x81, 0xd3, 0x7f, 0x5b, 0x72,
    0x71, 0xa3, 0x1b, 0xa7, 0x33, 0x18, 0x92, 0xdb, 0x95, 0x3a, 0xd3, 0x84,
    0xba, 0x82, 0xa1, 0xe0, 0x46, 0x71, 0x6a, 0x29, 0xe3, 0xb0, 0x36, 0xc6,
    0x54, 0xe1, 0xea, 0x8c, 0xd0, 0x0b, 0x75, 0x36, 0x1c, 0xb2, 0x10, 0xa0,
    0xf7, 0xc4, 0x6c, 0x37, 0x86, 0xc1, 0x96, 0xa4, 0x1e, 0x35, 0x0c, 0x8a,
    0xc8, 0xb9, 0x3d, 0x32, 0xf6, 0xc8, 0x43, 0x2e, 0x4d, 0x99, 0x97, 0xd4,
    0x5c, 0x4d, 0x69, 0x97, 0x3c, 0x55, 0x0b, 0xa6, 0x42, 0xaf, 0x66, 0xc6,
    0x92, 0x6c, 0x12, 0x0d, 0xbe, 0xfe, 0xbd, 0xf0, 0xcf, 0x05, 0xa8, 0x4e,
    0xd6, 0x27, 0x1a, 0x5b, 0xc8, 0x2c, 0x08, 0x8d, 0x82, 0xb6, 0x55, 0x70,
    0xa5, 0x00, 0xaf, 0x5e, 0xdb, 0xe1, 0x61, 0x8e, 0x9b, 0x44, 0x0a, 0x68,
    0xbd, 0x5a, 0x7d, 0x69, 0x28, 0x05, 0x2a, 0x5a, 0xc8, 0xf2, 0xfa, 0x7b,
    0x7d, 0x30, 0xdd, 0x95, 0xa9, 0x54, 0xbc, 0xb6, 0xc4, 0x8b, 0xcf, 0x69,
    0x48, 0x5f, 0x9c, 0xf9, 0x49, 0x3b, 0x0b, 0x6e, 0x7b, 0xe1, 0x86, 0x83,
    0x40, 0x98, 0x91, 0x9b, 0x0f, 0x39, 0x60, 0x90, 0x0e, 0xf1, 0x1b, 0xc6,
    0xf8, 0xe6, 0x56, 0xbb, 0x28, 0xa4, 0xa5, 0x4a, 0x53, 0x84, 0x8d, 0x41,
    0xd8, 0xc6, 0x8e, 0x49, 0x2b, 0xfe, 0x58, 0xa4, 0x2c, 0x3a, 0xb4, 0x07,
    0x1a, 0x2d, 0x2e, 0xcb, 0x29, 0xbe, 0x95, 0x91, 0xcf, 0xf9, 0x86, 0x34,
    0x1b, 0xc8, 0xf4, 0x8c, 0xaf, 0x5e, 0x7b, 0x30, 0xd7, 0x5a, 0x69, 0xe4,
    0x15, 0xf8, 0x89, 0x2e, 0x3a, 0x34, 0x81, 0xea, 0x07, 0xae, 0x0e, 0xf4,
    0xc1, 0xba, 0x5d, 0x63, 0xa7, 0xae, 0xc4, 0x55, 0x72, 0x1c, 0x19, 0xf0,
    0x66, 0x38, 0x02, 0x5f, 0x41, 0xf9, 0x54, 0x41, 0x1c, 0x6f, 0x6f, 0x70,
    0x0e, 0xe2, 0xc7, 0x1e, 0x95, 0x1c, 0xa7, 0x95, 0x6a, 0x8f, 0x78, 0x99,
    0xa7, 0x38, 0xbb, 0x31, 0x60, 0xec, 0xdc, 0x55, 0x04, 0xb2, 0x91, 0xed,
    0xab, 0x73, 0xf5, 0x23, 0x1e, 0xf3, 0x75, 0x49, 0x44, 0x23, 0x3e, 0xf2,
    0x92, 0x4d, 0x87, 0x99, 0xce, 0x3e, 0x82, 0xa0, 0x54, 0x25, 0x95, 0x47,
    0x96, 0xed, 0x52, 0x14, 0xb4, 0x0c, 0xae, 0x0e, 0x47, 0x63, 0x7d, 0xad,
    0x7f, 0x13, 0xf3, 0x08, 0x59, 0xa7, 0xa9, 0x90, 0xe7, 0xd4, 0x3f, 0x87,
    0xd3, 0xd6, 0x44, 0x73, 0xb1, 0x7d, 0x46, 0xca, 0xb7, 0x64, 0x8b, 0x7c,
    0xa3, 0x8d, 0x86, 0xe7, 0xb9, 0xed, 0x80, 0xd4, 0x1a, 0xac, 0x04, 0xb8,
    0xe1, 0x71, 0x0b, 0xd2, 0xb7, 0x09, 0xdc, 0xdf, 0x49, 0xdf, 0x6b, 0x1f,
    0xad, 0xcf, 0xed, 0x8a, 0xe2, 0x32, 0x5e, 0x40, 0x85, 0x1d, 0x89, 0x14,
    0xa8, 0x54, 0xea, 0xb8, 0x52, 0x94, 0x1c, 0x0f, 0x4f, 0x53, 0x3e, 0x1f,
    0xf7, 0x46, 0xc8, 0x37, 0x36, 0xbd, 0xed, 0xc6, 0x18, 0xa9, 0xf4, 0xfe,
    0x9a, 0xa6, 0x12, 0x14, 0xfd, 0x0c, 0x31, 0x25, 0x0a, 0x21, 0xe6, 0x58,
    0x43, 0xce, 0x83, 0xb6, 0xc5, 0x2e, 0x5b, 0x71, 0xfa, 0x61, 0x36, 0xa1,
    0x3e, 0x26, 0x94, 0x4b, 0xae, 0x0f, 0x33, 0x18, 0x26, 0x3b, 0x79, 0xb7,
    0x4a, 0x9e, 0x70, 0x78, 0x00, 0x32, 0x1e, 0xa7, 0xde, 0x37, 0x7a, 0x17,
    0xe0, 0x3b, 0x48, 0x42, 0x53, 0xe0, 0x3a, 0xfb, 0xa9, 0x5b, 0xee, 0x29,
    0xa1, 0xc9, 0x06, 0xc7, 0x9e, 0xc0, 0x5b, 0x02, 0x7f, 0x12, 0x50, 0xd2,
    0xdd, 0x7e, 0x93, 0x25, 0x20, 0x05, 0x29, 0x82, 0x90, 0x37, 0xb1, 0xb1,
    0x1e, 0x9f, 0x43, 0x86, 0xaa, 0x5d, 0x7f, 0x25, 0x51, 0x5d, 0x0b, 0xa7,
    0x50, 0xaa, 0x37, 0x00, 0x68, 0x09, 0x69, 0x66, 0xc3, 0xef, 0x84, 0xbe,
    0xa7, 0xe6, 0x98, 0xb9, 0xb2, 0xbf, 0x16, 0x33, 0x50, 0x64, 0xc2, 0xf8,
    0x26, 0xff, 0x00, 0xe7, 0x24, 0xa5, 0x4a, 0x4f, 0xae, 0xe3, 0x8d, 0xf9,
    0xe3, 0x6c, 0x60, 0x4b, 0x8c, 0xf6, 0x25, 0x01, 0x60, 0xaa, 0xf7, 0xc8,
    0xc6, 0x57, 0x18, 0xec, 0x90, 0x45, 0xf2, 0xf1, 0xb7, 0xca, 0xe6, 0x24,
    0x3d, 0x41, 0x97, 0x21, 0x1f, 0x0d, 0xf0, 0x4e, 0xa2, 0x3c, 0xc7, 0x96,
    0x4e, 0xa5, 0x10, 0x0f, 0x86, 0x05, 0x92, 0x8b, 0x9f, 0xca, 0x76, 0xfd,
    0x31, 0x3c, 0x10, 0xaa, 0xae, 0xbe, 0xaf, 0xfc, 0xad, 0x97, 0x97, 0x7d,
    0xce, 0x92, 0xa3, 0xfa, 0xe2, 0xae, 0x8a, 0x4c, 0x1a, 0xef, 0x50, 0x69,
    0x54, 0x9a, 0x83, 0xc9, 0x62, 0x3f, 0xc0, 0xba, 0xf1, 0x71, 0x36, 0x3a,
    0x52, 0x3e, 0x5d, 0xcf, 0xa0, 0x37, 0xbe, 0x07, 0xe6, 0x8c, 0x85, 0x9a,
    0xe3, 0xcf, 0x28, 0xc9, 0xd5, 0x29, 0xb5, 0x66, 0x42, 0x43, 0x88, 0x44,
    0x72, 0xa2, 0xbd, 0x2a, 0xbd, 0xb9, 0xdd, 0x5c, 0x1b, 0x73, 0xf5, 0xc5,
    0x33, 0x0d, 0xb6, 0xf3, 0x34, 0xd0, 0xe7, 0x19, 0xb1, 0xb9, 0x00, 0x01,
    0x90, 0xb9, 0x1a, 0xab, 0x2f, 0x28, 0x57, 0x55, 0x29, 0xb7, 0xc2, 0xa6,
    0x0a, 0x49, 0xd8, 0xdb, 0x5c, 0xa0, 0x05, 0x1f, 0x29, 0x54, 0xdd, 0xa7,
    0xa9, 0xe7, 0xe3, 0x53, 0xa2, 0x10, 0x9b, 0xb7, 0xe2, 0x4f, 0x09, 0x2e,
    0x0b, 0x5e, 0xe0, 0x04, 0x2a, 0xff, 0x00, 0xb6, 0x3c, 0x19, 0xcb, 0xd9,
    0xd9, 0x45, 0x3f, 0x0d, 0x47, 0x8e, 0x9d, 0x48, 0x4a, 0xbc, 0xa5, 0x2a,
    0xb5, 0xc6, 0xd7, 0xb9, 0x36, 0x3e, 0xd8, 0x0e, 0x89, 0x59, 0xb5, 0x89,
    0x4b, 0x82, 0xa9, 0x72, 0xe3, 0xb8, 0x83, 0x65, 0x24, 0xf9, 0x74, 0x9f,
    0x43, 0xb6, 0xc7, 0xeb, 0x82, 0xb4, 0xea, 0x6e, 0x75, 0x9e, 0xca, 0xa5,
    0x36, 0xed, 0x49, 0xc4, 0xa1, 0x5a, 0x54, 0x52, 0x6f, 0xc7, 0x3c, 0x63,
    0x43, 0xb8, 0xa9, 0xa6, 0xcf, 0x0a, 0x8a, 0xc5, 0xbf, 0xf3, 0xf7, 0xe8,
    0x04, 0x71, 0x6e, 0x9f, 0x4a, 0x51, 0xcd, 0x0b, 0x3d, 0x48, 0xfa, 0xde,
    0x33, 0x30, 0xe5, 0xec, 0xe9, 0x12, 0x9a, 0x89, 0xd5, 0x86, 0x99, 0x6a,
    0x34, 0x75, 0x78, 0x88, 0xd4, 0x05, 0x8a, 0x86, 0xe1, 0x36, 0xb7, 0x7b,
    0x7d, 0x31, 0xeb, 0x53, 0xa8, 0x67, 0xc8, 0x49, 0x65, 0x6e, 0x39, 0x22,
    0x13, 0x2a, 0x17, 0x65, 0x41, 0xb5, 0xb4, 0x9b, 0x13, 0xdb, 0x55, 0x88,
    0x1b, 0xf7, 0xc3, 0xfd, 0x0e, 0x44, 0xc7, 0xa8, 0x2c, 0xc1, 0xac, 0x34,
    0x92, 0xb6, 0x27, 0xc7, 0x6d, 0x68, 0x51, 0xb8, 0x3c, 0x1d, 0xc7, 0x62,
    0x45};
}  // namespace
