package jp.operation.distance.vector;

import org.apache.commons.math.DimensionMismatchException;
import org.apache.commons.math.linear.Array2DRowRealMatrix;
import org.apache.commons.math.linear.DecompositionSolver;
import org.apache.commons.math.linear.RealMatrix;
import org.apache.commons.math.linear.RealMatrixImpl;
import org.apache.commons.math.linear.SingularValueDecompositionImpl;
import org.apache.commons.math.stat.descriptive.moment.VectorialCovariance;
import org.apache.commons.math.stat.descriptive.moment.VectorialMean;

/**
 * マハラノビス距離(マハラノビス汎距離、正規化ユークリッド距離とも呼ぶ)
 * 相関のある方向に平行な距離を相対的に短く、相関のある方向に垂直な距離を相対的に長くなるようにする
 * 共分散行列が単位行列ならば、ユークリッド距離に一致する
 * 
 * @author yasuda_masahiro
 *
 */
public class Mahalanobis {
 
	private VectorialMean mean;
	private VectorialCovariance covariance;
	private RealMatrix covarianceMatrix;
	
    public Mahalanobis(int dimentions) {
        mean = new VectorialMean(dimentions);
        // use unbiased variance
        covariance = new VectorialCovariance(dimentions, false);
    }

    public void add(double values[]) throws DimensionMismatchException {
    	mean.increment(values);
    	covariance.increment(values);
    }
	
    public double distance(double[] p) throws DimensionMismatchException {

    	if(covarianceMatrix == null) {
        	covarianceMatrix = covariance.getResult();
    	}
        RealMatrix pMatrix = new Array2DRowRealMatrix(p);
        RealMatrix meanMatrix = new Array2DRowRealMatrix(mean.getResult());
        RealMatrix pMinusMean = pMatrix.subtract(meanMatrix);

//        System.out.println(pMatrix);
//        System.out.println(meanMatrix);
//        System.out.println(pMinusMean);
    	DecompositionSolver covSolver = new SingularValueDecompositionImpl(covarianceMatrix).getSolver();
    	RealMatrix covInverse = covSolver.getInverse();
    	RealMatrix result = pMinusMean.transpose().multiply(covInverse).multiply(pMinusMean);
    	
        return Math.sqrt(result.getEntry(0, 0));
    }

}
