﻿using System;
using System.Linq;
using System.Text.RegularExpressions;
using System.Collections.Generic;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using FooEditEngine;
using FooEditEngine.WPF;
using Prop = FooEditor.Properties;

namespace FooEditor
{
    /// <summary>
    /// FindReplaceWindow.xaml の相互作用ロジック
    /// </summary>
    public partial class FindReplaceWindow : UserControl
    {
        private DocumentWindow ActiveDocument;
        private DocumentsFinder Finder = new DocumentsFinder();
        private MainWindow Main;

        public FindReplaceWindow()
        {
            InitializeComponent();
            this.Title = "検索と置き換え";

            this.CommandBindings.Add(new CommandBinding(FindReplaceCommands.FindStart, FindStartCommand));
            this.CommandBindings.Add(new CommandBinding(FindReplaceCommands.ReplaceStart, ReplaceStartCommand));
            this.CommandBindings.Add(new CommandBinding(FindReplaceCommands.ReplaceAllStart, ReplaceAlStartlCommand));

            this.FindText.TextChanged += new TextChangedEventHandler(FindText_TextChanged);

            this.AllDocuments.Checked += AllDocuments_Checked;
        }

        void AllDocuments_Checked(object sender, RoutedEventArgs e)
        {
            this.Finder.Reset();
        }

        public FindReplaceWindow(MainWindow main) : this()
        {
            this.Main = main;
            this.Main.ActiveDocumentChanged += Main_ActiveDocumentChanged;
            this.Main.Documents.CollectionChanged += Documents_CollectionChanged;
            this.Finder.ChangedTargetDocument += Finder_ChangedTargetDocument;
        }

        void Finder_ChangedTargetDocument(object sender, DocumentEventArgs e)
        {
            this.Main.ActivateDocument(e.Document);
        }

        void Documents_CollectionChanged(object sender, System.Collections.Specialized.NotifyCollectionChangedEventArgs e)
        {
            this.Finder.Reset();
        }

        void Main_ActiveDocumentChanged(object sender, EventArgs e)
        {
            if (this.Main.ActiveDocument != null && this.Finder.IsMultipleDocument == false)
            {
                this.Finder.Reset();
                this.ActiveDocument = this.Main.ActiveDocument;
            }
        }

        public string Title
        {
            get;
            set;
        }

        void FindText_TextChanged(object sender, TextChangedEventArgs e)
        {
            this.Finder.Reset();
        }

        private RegexOptions DecideRegexOpt()
        {
            RegexOptions opt = RegexOptions.None;
            if (this.RestrictSearch.IsChecked == true)
                opt |= RegexOptions.IgnoreCase;
            return opt;
        }

        void FindStartCommand(object sender, ExecutedRoutedEventArgs e)
        {
            if (this.Finder.Documents == null)
            {
                if ((bool)this.AllDocuments.IsChecked)
                    this.Finder.Documents = this.Main.Documents;
                else
                    this.Finder.Documents = new DocumentWindow[] { this.ActiveDocument };
            }

            if (this.Finder.IsCanNotFindOrReplace())
            {
                MessageBox.Show(Prop.Resources.MustBeDisableRectSelect);
                return;
            }

            try
            {
                if (this.Finder.FindAndSelect(this.FindText.Text,(bool)this.UseRegEx.IsChecked,DecideRegexOpt()))
                {
                    this.Finder.Reset();
                    MessageBox.Show(Prop.Resources.FindDialogNotFound);
                    return;
                }
            }
            catch (ArgumentException ex)
            {
                MessageBox.Show(ex.Message);
                return;
            }

            this.ReplaceButton.IsEnabled = true;
        }

        void ReplaceStartCommand(object sender, ExecutedRoutedEventArgs e)
        {
            this.Finder.Replace(this.ReplaceText.Text, (bool)this.UseGroup.IsChecked);
            this.FindStartCommand(sender, e);
        }
        
        void ReplaceAlStartlCommand(object sender, ExecutedRoutedEventArgs e)
        {
            if (this.Finder.Documents == null)
            {
                if ((bool)this.AllDocuments.IsChecked)
                    this.Finder.Documents = this.Main.Documents;
                else
                    this.Finder.Documents = new DocumentWindow[] { this.ActiveDocument };
            }

            if (this.Finder.IsCanNotFindOrReplace())
            {
                MessageBox.Show(Prop.Resources.MustBeDisableRectSelect);
                return;
            }

            this.Finder.ReplaceAll(this.FindText.Text, this.ReplaceText.Text, (bool)this.UseGroup.IsChecked, (bool)this.UseRegEx.IsChecked, DecideRegexOpt());

            MessageBox.Show(Prop.Resources.ReplaceAllComplete);
        }
    }

    public static class FindReplaceCommands
    {
        public static RoutedCommand FindStart = new RoutedCommand("FindStart", typeof(FindReplaceWindow));
        public static RoutedCommand ReplaceStart = new RoutedCommand("ReplaceStart", typeof(FindReplaceWindow));
        public static RoutedCommand ReplaceAllStart = new RoutedCommand("ReplaceAllStart", typeof(FindReplaceWindow));
    }

    class DocumentsFinder
    {
        private IEnumerator<SearchResult> sr;
        private DocumentWindow document;
        private int index;

        public IEnumerable<DocumentWindow> Documents
        {
            get;
            set;
        }

        public bool IsMultipleDocument
        {
            get
            {
                if (this.Documents == null)
                    return false;
                return this.Documents.Count() > 1;
            }
        }

        public event EventHandler<DocumentEventArgs> ChangedTargetDocument;

        public bool FindAndSelect(string pattern, bool useRegEx, RegexOptions option)
        {
            bool result = true;
            if (this.sr == null)
            {
                this.index = 0;
                this.document = this.Documents.First();
                if (this.IsMultipleDocument && this.ChangedTargetDocument != null)
                    this.ChangedTargetDocument(this, new DocumentEventArgs(this.document));
                result = this.FindFirst(this.document, pattern, useRegEx, option);
            }
            else
            {
                result = this.FindNext();
            }

            while (result == false)
            {
                index++;
                if (index >= this.Documents.Count())
                    return true;
                this.document = this.Documents.ElementAt(index);
                if (this.ChangedTargetDocument != null)
                    this.ChangedTargetDocument(this, new DocumentEventArgs(this.document));
                result = this.FindFirst(this.document, pattern, useRegEx, option);
            }

            SearchResult current = this.sr.Current;
            this.document.TextBox.JumpCaret(current.Start);
            this.document.TextBox.Select(current.Start, current.End - current.Start + 1);
            this.document.TextBox.Refresh();

            return false;
        }

        public void Replace(string replacement, bool isGroup)
        {
            if (isGroup == true)
                this.document.TextBox.SelectedText = this.sr.Current.Result(replacement);
            else
                this.document.TextBox.SelectedText = replacement;
            this.document.TextBox.DeSelectAll();
            this.document.TextBox.Refresh();
        }

        public void ReplaceAll(string pattern, string replacement, bool isGroup, bool useRegEx, RegexOptions option)
        {
            this.index = 0;
            while (this.index < this.Documents.Count())
            {
                this.document = this.Documents.ElementAt(this.index);
                this.document.TextBox.DeSelectAll();
                this.document.TextBox.Document.SetFindParam(pattern, useRegEx, option);
                this.document.TextBox.Document.ReplaceAll(replacement, isGroup);
                this.document.TextBox.Refresh();
                index++;
            }
            this.index = 0;
        }

        public void Reset()
        {
            this.sr = null;
            this.Documents = null;
        }

        bool FindFirst(DocumentWindow Document,string pattern, bool useRegEx, RegexOptions option)
        {
            Document.TextBox.Document.SetFindParam(pattern, useRegEx, option);
            this.sr = Document.TextBox.Document.Find();
            return this.sr.MoveNext();
        }

        bool FindNext()
        {
            return sr.MoveNext();
        }

        public bool IsCanNotFindOrReplace()
        {
            foreach (DocumentWindow docment in this.Documents)
                if (docment.TextBox.RectSelectMode)
                    return true;
            return false;
        }
    }
}
