﻿using System;
using System.IO;
using System.Runtime.Serialization;
using System.Text.RegularExpressions;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Media;
using EncodeDetect;
using Prop = FooEditor.Properties;
using FooEditEngine;
using FooEditEngine.WPF;
using System.Runtime.CompilerServices;

namespace FooEditor
{
    /// <summary>
    /// 文章タイプが変化している時に送られてくるイベント
    /// </summary>
    public class DocumentTypeChangeingEventArgs : EventArgs
    {
        /// <summary>
        /// 変更後の文章タイプ
        /// </summary>
        public string Type;
        /// <summary>
        /// イベント処理後にハイライター、フォールディングを割り当てる必要がないなら真を設定する
        /// </summary>
        public bool Handled = false;
        public DocumentTypeChangeingEventArgs(string type)
        {
            this.Type = type;
        }
    }
    /// <summary>
    /// DocumentWindow.xaml の相互作用ロジック
    /// </summary>
    [Serializable]
    public partial class DocumentWindow : UserControl,INotifyPropertyChanged,ISerializable,IDisposable
    {
        int updateCount;
        string _Title, _FilePath, _DocumentType;
        Encoding _Encoding;
        LineFeedType _LineFeed;
        AutoIndent autoIndent;

        public DocumentWindow()
        {
            InitializeComponent();
            this.PropertyChanged +=new PropertyChangedEventHandler((s,e)=>{});
            this.DocumentTypeChanged += new EventHandler((s, e) => { });
            this.DocumentTypeChangeing += new EventHandler<DocumentTypeChangeingEventArgs>((s, e) => { });
            
            this.Encoding = Encoding.Default;
            this.LineFeed = LineFeedType.CRLF;

            Config config = Config.GetInstance();
            this.TextBox.Foreground = config.Fore;
            this.TextBox.Background = config.Back;
            this.TextBox.Comment = config.Comment;
            this.TextBox.Keyword1 = config.Keyword1;
            this.TextBox.Keyword2 = config.Keyword2;
            this.TextBox.Literal = config.Literal;
            this.TextBox.URL = config.URL;
            this.TextBox.InsertCaret = config.InsetCaret;
            this.TextBox.OverwriteCaret = config.OverwriteCaret;
            this.TextBox.LineMarker = config.LineMarker;
            this.TextBox.ControlChar = config.Control;
            this.TextBox.Hilight = config.Hilight;
            this.TextBox.MarkURL = config.UrlMark;
            this.TextBox.WordRap = config.WordRap;
            this.TextBox.DrawCaretLine = config.DrawLine;
            this.TextBox.DrawLineNumber = config.DrawLineNumber;
            this.TextBox.FontFamily = new FontFamily(config.FontName);
            this.TextBox.FontSize = config.FontSize;
            this.TextBox.TabChars = config.TabStops;
            this.TextBox.TextAntialiasMode = config.TextAntialiasMode;
            this.TextBox.ShowFullSpace = true;
            this.TextBox.ShowTab = true;
            this.TextBox.Document.Update += new DocumentUpdateEventHandler(Document_Update);
            this.TextBox.MouseDoubleClick += new MouseButtonEventHandler(TextBox_MouseDoubleClick);

            this.autoIndent = new AutoIndent(this);
            this.autoIndent.Enable = config.AutoIndent;
        }

        public DocumentWindow(int number) : this()
        {
            this.Title = string.Format(Prop.Resources.NewDocumentTitle,number);
        }

        public string FilePath
        {
            get { return this._FilePath; }
            set
            {
                this.Title = Path.GetFileName(value);               
                this._FilePath = value;
                this.OnPropertyChanged();
            }
        }

        public string Title
        {
            get { return this._Title; }
            set
            {
                this._Title = value;
                this.OnPropertyChanged();
            }
        }

        public Encoding Encoding
        {
            get { return this._Encoding; }
            set
            {
                this._Encoding = value;
                this.OnPropertyChanged();
            }
        }

        public LineFeedType LineFeed
        {
            get { return this._LineFeed; }
            set
            {
                this._LineFeed = value;
                this.OnPropertyChanged();
            }
        }

        public string DocumentType
        {
            get { return this._DocumentType; }
            set
            {
                this._DocumentType = value;
                
                DocumentTypeChangeingEventArgs e = new DocumentTypeChangeingEventArgs(value);
                this.DocumentTypeChangeing(this, e);
                
                if (!e.Handled)
                {
                    if (AttachHeilighter(value))
                        return;
                }

                if (value == null || value == Prop.Resources.DocumetTypeNone || value == string.Empty)
                {
                    this.TextBox.LayoutLineCollection.ClearHilight();
                    this.TextBox.LayoutLineCollection.ClearFolding();
                }
                else
                {
                    this.TextBox.LayoutLineCollection.HilightAll();
                    this.TextBox.LayoutLineCollection.ClearFolding();
                    this.TextBox.LayoutLineCollection.GenerateFolding();
                }
                this.TextBox.Refresh();
                this.OnPropertyChanged();
                this.DocumentTypeChanged(this, null);
            }
        }

        public bool Dirty
        {
            get { return (bool)GetValue(DirtyProperty); }
            set { SetValue(DirtyProperty, value); }
        }

        // Using a DependencyProperty as the backing store for Dirty.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty DirtyProperty =
            DependencyProperty.Register("Dirty", typeof(bool), typeof(DocumentWindow), new UIPropertyMetadata(false));
        
        public DockPanel LayoutRoot
        {
            get { return this.DockPanel; }
        }

        public FooTextBox TextBox
        {
            get { return this.FooTextBox; }
        }

        public SyntaxDefnition SynataxDefnition
        {
            get;
            private set;
        }

        public bool EnableAutoIndent
        {
            get { return this.autoIndent.Enable; }
            set { this.autoIndent.Enable = value; }
        }

        public AutocompleteBox CompleteBox
        {
            get;
            set;
        }

        public void OnPropertyChanged([CallerMemberName] string name = "")
        {
            if (this.PropertyChanged != null)
                this.PropertyChanged(this, new PropertyChangedEventArgs(name));
        }

        public event EventHandler DocumentTypeChanged;

        public event EventHandler<DocumentTypeChangeingEventArgs> DocumentTypeChangeing;

        public void Load(string filepath, Encoding enc)
        {
            try
            {
                this.FilePath = filepath;

                if (enc == null)
                    enc = DectingEncode.GetCode2(filepath);
                this.Encoding = enc;
                
                this.LineFeed = LineFeedHelper.GetLineFeed(filepath, enc);
                
                this.TextBox.Document.Load(filepath, enc);

                this.DocumentType = this.DeciedKeywordFileName(filepath);

                this.TextBox.JumpCaret(0);
                this.TextBox.Refresh();
            }
            catch (UnauthorizedAccessException ex)
            {
                MessageBox.Show(ex.Message);
            }
            catch (FileNotFoundException ex)
            {
                MessageBox.Show(ex.Message);
            }
            catch (IOException ex)
            {
                MessageBox.Show(ex.Message);
            }
            Config config = Config.GetInstance();
            config.RecentFile.InsertAtFirst(filepath);
            config.SyntaxDefinitions.Select(this._DocumentType);
            this.Dirty = false;
            this.updateCount = 0;
        }

        public void Save(string filepath, Encoding enc)
        {
            Config confing = Config.GetInstance();

            if (confing.MaxBackupCount > 0 && File.Exists(filepath))
            {
                string oldFilename = GetBackupFileName(filepath);
                if (oldFilename != null)
                    File.Move(filepath, oldFilename);
            }

            try
            {
                this.TextBox.Document.Save(filepath, enc, LineFeedHelper.ToString(this.LineFeed));
            }
            catch (UnauthorizedAccessException)
            {
                MessageBoxResult result = MessageBox.Show(Prop.Resources.UACSaveConfirm, Prop.Resources.UACSaveConfirmDialogTitle, MessageBoxButton.YesNo);
                if (result == MessageBoxResult.Yes)
                {
                    string tempfile = Path.GetTempPath() + Path.GetRandomFileName();
                    this.TextBox.Document.Save(tempfile, this.Encoding, LineFeedHelper.ToString(this.LineFeed));
                    this.DoAdminOperation(tempfile, filepath, false);
                }
            }
            finally
            {
                this.updateCount = 0;
                this.Dirty = false;
            }
        }

        public void Dispose()
        {
            this.FooTextBox.Dispose();
        }

        public event PropertyChangedEventHandler PropertyChanged;

        void DoAdminOperation(string tempfile, string filename, bool backup)
        {
            AdminiOperation operation = new AdminiOperation();

            if (backup && File.Exists(filename))
            {
                string oldFilename = GetBackupFileName(filename);
                if (oldFilename != null)
                    operation.WriteCode(string.Format("move\t{0}\t{1}", filename, oldFilename));
            }

            operation.WriteCode(string.Format("copy\t{0}\t{1}", tempfile, filename));

            bool result = false;
            try
            {
                result = operation.Execute();
            }
            catch (Win32Exception ex)  //UACのキャンセル時に発生するので
            {
                if (ex.NativeErrorCode != 1223)
                    throw;
            }
            if (result)
            {
                this.FilePath = filename;
                Config cfg = Config.GetInstance();
                cfg.RecentFile.InsertAtFirst(filename);
            }
        }

        void Document_Update(object sender, FooEditEngine.DocumentUpdateEventArgs e)
        {
            Config cfg = Config.GetInstance();
            if (e.type == UpdateType.Replace && cfg.AutoSaveCount != 0)
            {
                if (this.updateCount > cfg.AutoSaveCount)
                    this.SaveBackupFolder(this.FilePath);
                else
                    this.updateCount++;
            }
            if(this.Dirty == false)
                this.Dirty = true;
        }

        void SaveBackupFolder(string filename)
        {
            if (filename == string.Empty || filename == null)
                filename = this.Title;
            Task task = new Task(new Action(()=>{
                string BackupFolderPath = Path.Combine(Config.ApplicationFolder, "Backup");
                if (Directory.Exists(BackupFolderPath) == false)
                    Directory.CreateDirectory(BackupFolderPath);
                string BackupFilename = filename.Replace(Path.DirectorySeparatorChar, '_');
                BackupFilename = BackupFilename.Replace(Path.VolumeSeparatorChar, '_');
                this.TextBox.Document.Save(Path.Combine(BackupFolderPath, BackupFilename), this.Encoding, LineFeedHelper.ToString(this.LineFeed));
                this.updateCount = 0;
            }));
            task.Start();
        }

        string GetBackupFileName(string filename)
        {
            string directoryPart = Path.GetDirectoryName(filename);
            string filePart = Path.GetFileName(filename);
            IEnumerable<string> files = Directory.EnumerateFiles(directoryPart, filePart + "*");

            int newCount = files.Count();

            Config cfg = Config.GetInstance();
            if (newCount > cfg.MaxBackupCount)
                return null;

            return filename + "." + newCount;
        }

        private string DeciedKeywordFileName(string EditingFile)
        {
            Config cfg = Config.GetInstance();

            if (EditingFile == null || EditingFile == string.Empty)
                return null;

            foreach (DocumentType item in cfg.SyntaxDefinitions)
            {
                string type = item.Name;
                string targetExt = item.Extension;
                if (type == Prop.Resources.DocumetTypeNone || targetExt == null || targetExt == string.Empty)
                    continue;
                Regex regex = new Regex(targetExt, RegexOptions.IgnoreCase);
                Match m = regex.Match(Path.GetFileName(EditingFile));
                if (m.Success == true)
                    return type;
            }
            return null;
        }

        private bool AttachHeilighter(string name)
        {
            if (name == null || name == Prop.Resources.DocumetTypeNone || name == string.Empty)
            {
                this.TextBox.Hilighter = null;
                this.TextBox.FoldingStrategy = null;
                return false;
            }

            string filepath = GetKeywordFilePath(name);

            if (filepath == null)
                return true;

            this.SynataxDefnition = new SyntaxDefnition();
            this.SynataxDefnition.generateKeywordList(filepath);

            if (this.SynataxDefnition.Hilighter == FooEditor.SyntaxDefnition.XmlHilighter)
            {
                this.TextBox.Hilighter = new XmlHilighter();
            }
            else
            {
                GenericHilighter Hilighter = new GenericHilighter();
                Hilighter.KeywordManager = this.SynataxDefnition;
                this.TextBox.Hilighter = Hilighter;
            }

            if (!string.IsNullOrEmpty(this.SynataxDefnition.FoldingBegin) && !string.IsNullOrEmpty(this.SynataxDefnition.FoldingEnd))
            {
                if (this.SynataxDefnition.FoldingMethod == FooEditor.SyntaxDefnition.CLangFolding)
                    this.TextBox.FoldingStrategy = new CLangFoldingGenerator(this.SynataxDefnition.FoldingBegin, this.SynataxDefnition.FoldingEnd, '{', '}');
                else
                    this.TextBox.FoldingStrategy = new RegexFoldingGenerator(this.SynataxDefnition.FoldingBegin, this.SynataxDefnition.FoldingEnd);
            }
            else
            {
                this.TextBox.FoldingStrategy = null;
            }

            return false;
        }

        string GetKeywordFilePath(string name)
        {
            Config cfg = Config.GetInstance();

            string KeywordFolderName = "Keywords";

            string filepath = Path.Combine(Config.ApplicationFolder, KeywordFolderName, name);

            if (File.Exists(filepath))
                return filepath;

            filepath = Path.Combine(Config.ExecutablePath, KeywordFolderName, name);

            if (File.Exists(filepath))
                return filepath;

            return null;
        }

        void TextBox_MouseDoubleClick(object sender, MouseButtonEventArgs e)
        {
            FooMouseButtonEventArgs fe = (FooMouseButtonEventArgs)e;
            foreach (Marker m in this.TextBox.Document.GetMarkers(fe.Index))
            {
                if (m.hilight == HilightType.Url)
                {
                    ProcessStartInfo info = new ProcessStartInfo();
                    info.Arguments = "";
                    info.FileName = this.TextBox.Document.ToString(m.start, m.length);
                    info.Verb = "open";
                    info.UseShellExecute = true;
                    Process process = Process.Start(info);

                    fe.Handled = true;
                }
            }
        }

        public DocumentWindow(SerializationInfo info, StreamingContext context) : this()
        {
            this.FilePath = info.GetString("FilePath");
            
            this.Title = info.GetString("Title");
            
            this.DocumentType = info.GetString("DocumentType");
            
            this.Encoding = Encoding.GetEncoding(info.GetInt32("Encoding"));
            
            this.TextBox.InsertMode = info.GetBoolean("InsertMode");
            
            this.TextBox.RectSelectMode = info.GetBoolean("RectSelectMode");

            int maxCount = info.GetInt32("LineCount");
            this.TextBox.Document.FireUpdateEvent = false;
            this.TextBox.Document.UndoManager.BeginLock();
            for (int i = 0; i < maxCount; i++)
                this.TextBox.Document.Append(info.GetString(i.ToString()));
            this.TextBox.Document.UndoManager.EndLock();
            this.TextBox.Document.FireUpdateEvent = true;
            
            TextPoint tp = new TextPoint();
            tp.row = info.GetInt32("row");
            tp.col = info.GetInt32("col");
            this.TextBox.Loaded += new RoutedEventHandler((s,e)=>{
                this.TextBox.JumpCaret(tp.row, tp.col);
                this.TextBox.Refresh();
            });
        }

        public void GetObjectData(SerializationInfo info, StreamingContext context)
        {
            info.AddValue("row", this.TextBox.CaretPostion.row);
            
            info.AddValue("col", this.TextBox.CaretPostion.col);
            
            info.AddValue("RectSelectMode", this.TextBox.RectSelectMode);
            
            info.AddValue("Title", this.Title);
            
            info.AddValue("FilePath", this.FilePath);
            
            info.AddValue("DocumentType", this.DocumentType);
            
            info.AddValue("Encoding", this.Encoding.CodePage);
            
            info.AddValue("InsertMode", this.TextBox.InsertMode);
            
            info.AddValue("LineCount", this.TextBox.LayoutLineCollection.Count);
            for (int i = 0; i < this.TextBox.LayoutLineCollection.Count; i++)
                info.AddValue(i.ToString(), this.TextBox.LayoutLineCollection[i]);
        }
    }
}
