﻿using System;
using AvalonDock;
using AvalonDock.Layout;
using System.IO;
using System.ComponentModel;
using System.Collections.ObjectModel;
using System.Windows.Threading;
using System.Diagnostics;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Media;
using System.Runtime.Serialization.Formatters.Binary;
using Microsoft.WindowsAPICodePack.Dialogs;
using Prop = FooEditor.Properties;
using FooEditEngine.WPF;
using FooEditor.Plugin;
using System.Runtime.CompilerServices;

namespace FooEditor
{
    /// <summary>
    /// MainWindow.xaml の相互作用ロジック
    /// </summary>
    public partial class MainWindow : Window,INotifyPropertyChanged
    {
        ExplorerBar<FindReplaceWindow> findWindow;
        IPCServer Server;
        PluginManager<IPlugin> plugins;

        public MainWindow()
        {
            InitializeComponent();

            this.CreatedDocument += new EventHandler<CreatedDocumentEventArgs>((s, e) => { });
            this.ActiveDocumentChanged += new EventHandler((s, e) => { });

            this.DataContext = this;
            this.Documents = new ObservableCollection<DocumentWindow>();

            InputMethod.Current.StateChanged +=Current_StateChanged;

            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.New, NewCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Open, OpenCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Save, SaveCommand, CanExecute));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.SaveAs, SaveAsCommand, CanExecute));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Print, PrintCommand, CanExecute));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Find, FindCommand, CanExecute));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Replace, FindCommand, CanExecute));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Properties, PropertiesCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Close, PropertiesCommand));
            this.CommandBindings.Add(new CommandBinding(ApplicationCommands.Help, HelpCommand));
            this.CommandBindings.Add(new CommandBinding(FooEditorCommands.SelectDocumentType, SelectDocumentTypeCommand, CanExecute));
            this.CommandBindings.Add(new CommandBinding(FooEditorCommands.OpenRecentFile, OpenRecentFileCommand));
            this.CommandBindings.Add(new CommandBinding(FooEditorCommands.Grep, GrepCommand));
            this.CommandBindings.Add(new CommandBinding(FooEditorCommands.SaveWorkSpace, SaveWorkSpace, CanExecute));
            this.CommandBindings.Add(new CommandBinding(FooEditorCommands.Quit, QuitCommand));
            this.CommandBindings.Add(new CommandBinding(FooEditorCommands.About, AboutCommand));
            this.CommandBindings.Add(new CommandBinding(FooEditorCommands.TileHorizontal, TileHorizontalCommand));
            this.CommandBindings.Add(new CommandBinding(FooEditorCommands.TileVertical, TileVerticalCommand));
            this.CommandBindings.Add(new CommandBinding(FooEditorCommands.Casecade, CasecadeCommand));
            this.CommandBindings.Add(new CommandBinding(FooEditorCommands.LineJump, LineJumpCommand, CanExecute));

            this.DockManager.ActiveContentChanged += new EventHandler(DockManager_ActiveContentChanged);
            this.DockManager.DocumentClosed += new EventHandler<AvalonDock.DocumentClosedEventArgs>(DockManager_DocumentClosed);
            this.DockManager.DocumentClosing += new EventHandler<DocumentClosingEventArgs>(DockManager_DocumentClosing);
            this.Loaded += new RoutedEventHandler(MainWindow_Loaded);
            this.Closing += new CancelEventHandler(MainWindow_Closing);
            this.Closed += new EventHandler(MainWindow_Closed);
            this.Activated += new EventHandler(MainWindow_Activated);
            this.PreviewDragOver += new DragEventHandler(MainWindow_PreviewDragOver);
            this.PreviewDrop += new DragEventHandler(MainWindow_PreviewDrop);

            this.findWindow = new ExplorerBar<FindReplaceWindow>();
            this.findWindow.Content = new FindReplaceWindow();
            this.RegisterExploereBar(this.findWindow);
            this.findWindow.IsVisible = false;

            Config config = Config.GetInstance();
            this.Width = config.Width;
            this.Height = config.Height;

            this.plugins = new PluginManager<IPlugin>(config.DontLoadPlugins);
            foreach (IPlugin plugin in this.plugins)
                plugin.Initalize(this);

            Application.Current.DispatcherUnhandledException += Current_DispatcherUnhandledException;
        }

        void Current_DispatcherUnhandledException(object sender, DispatcherUnhandledExceptionEventArgs e)
        {
            ExceptionDialog dialog = new ExceptionDialog();
            dialog.Exception = e.Exception;
            dialog.Owner = this;
            if (dialog.ShowDialog() == true)
                this.SaveWorkSpace(this, null);
            else
                e.Handled = true;
        }

        public event PropertyChangedEventHandler PropertyChanged;

        public void OnPropertyChanged([CallerMemberName] string name = "")
        {
            if (this.PropertyChanged != null)
                this.PropertyChanged(this, new PropertyChangedEventArgs(name));
        }

        private void Current_StateChanged(object sender, InputMethodStateChangedEventArgs e)
        {
            if (e.IsImeConversionModeChanged)
            {
                this.OnPropertyChanged("ImeConversionMode");
                System.Diagnostics.Debug.WriteLine(this.ImeConversionMode.ToString());
            }
            if (e.IsImeStateChanged)
            {
                this.OnPropertyChanged("ImeState");
            }
        }

        public InputMethodState ImeState
        {
            get
            {
                return InputMethod.Current.ImeState;
            }
        }

        public ImeConversionModeValues ImeConversionMode
        {
            get
            {
                return InputMethod.Current.ImeConversionMode;
            }
        }

        public DocumentWindow ActiveDocument
        {
            get { return (DocumentWindow)GetValue(ActiveDocumentProperty); }
            private set { SetValue(ActiveDocumentProperty, value); }
        }

        public static readonly DependencyProperty ActiveDocumentProperty =
            DependencyProperty.Register("ActiveDocument", typeof(DocumentWindow), typeof(MainWindow), new FrameworkPropertyMetadata(null));

        public EventHandler ActiveDocumentChanged;

        public EventHandler<CreatedDocumentEventArgs> CreatedDocument;

        public ObservableCollection<DocumentWindow> Documents
        {
            get;
            private set;
        }

        public RecentFileCollection RecentFiles
        {
            get
            {
                Config config = Config.GetInstance();
                return config.RecentFile;
            }
        }

        public DocumentTypeCollection DocumentTypes
        {
            get
            {
                Config config = Config.GetInstance();
                return config.SyntaxDefinitions;
            }
        }

        public static string PipeServerName
        {
            get
            {
                return Prop.Resources.IPCServerName + "." + Process.GetCurrentProcess().SessionId;
            }
        }

        public void RegisterExploereBar<T>(ExplorerBar<T> bar)
        {
            this.ToolsPanel.Children.Add(bar.Anchor);
        }

        public DocumentWindow CreateDocument()
        {
            DocumentWindow newWindow = new DocumentWindow(this.Documents.Count);
            this.CreatedDocument(this, new CreatedDocumentEventArgs(newWindow));
            this.Documents.Add(newWindow);
            return newWindow;
        }

        void MainWindow_PreviewDragOver(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
                e.Effects = DragDropEffects.Copy;
        }

        void MainWindow_PreviewDrop(object sender, DragEventArgs e)
        {
            string[] filepaths = (string[])e.Data.GetData(DataFormats.FileDrop);
            if (filepaths == null)
                return;
            foreach (string filepath in filepaths)
            {
                DocumentWindow document = this.CreateDocument();
                document.Load(filepath, null);
                this.DockManager.ActiveContent = document;
            }
        }

        void MainWindow_Closing(object sender, CancelEventArgs e)
        {
            bool hasDirtyDoc = false;
            foreach (DocumentWindow doc in this.Documents)
                if (doc.Dirty)
                    hasDirtyDoc = true;
            if (hasDirtyDoc == false)
                return;
            if (MessageBox.Show(Prop.Resources.ConfirmDocumentClose, string.Empty, MessageBoxButton.YesNo) == MessageBoxResult.No)
                return;
            foreach (DocumentWindow doc in this.Documents)
                if (doc.Dirty)
                    this.Save(doc);
        }

        void MainWindow_Activated(object sender, EventArgs e)
        {
            if (this.ActiveDocument != null)
                this.ActiveDocument.TextBox.Focus();
        }

        void MainWindow_Closed(object sender, System.EventArgs e)
        {
            if (this.plugins != null)
            {
                foreach (IPlugin plugin in this.plugins)
                    plugin.ClosedApp();
            }
            Config config = Config.GetInstance();
            config.Width = this.Width;
            config.Height = this.Height;
            config.Save();
            if(this.Server != null)
                this.Server.Dispose();
        }

        void MainWindow_Loaded(object sender, RoutedEventArgs e)
        {
            DocumentWindow document = null;
            if(this.LoadWorkSpace())
                document = this.CreateDocument();

            Server = new IPCServer(MainWindow.PipeServerName);
            Server.Recive += new ServerReciveEventHandler(Server_Recive);

            App.ParseArgs(Environment.GetCommandLineArgs());

            if (document != null)
                document.TextBox.Document.Load(Console.In);
        }

        void Server_Recive(object sender, ServerReciveEventArgs e)
        {
            string data = e.StreamReader.ReadLine();

            PipeCommandListener listener = new PipeCommandListener(this);

            listener.Execute(data);
        }

        void DockManager_DocumentClosing(object sender, DocumentClosingEventArgs e)
        {
            DocumentWindow document = (DocumentWindow)e.Document.Content;
            if (document.Dirty == false)
                return;
            MessageBoxResult result = MessageBox.Show(Prop.Resources.ConfirmDocumentClose, document.Title, MessageBoxButton.YesNo);
            if (result == MessageBoxResult.Yes)
            {
                if (document.FilePath == null || document.FilePath == string.Empty)
                    this.SaveAs(document);
                else
                    document.Save(document.FilePath, document.Encoding);
            }
        }

        void DockManager_DocumentClosed(object sender, DocumentClosedEventArgs e)
        {
            DocumentWindow document = (DocumentWindow)e.Document.Content;
            this.Documents.Remove(document);
            document.Dispose();
        }

        void DockManager_ActiveContentChanged(object sender, EventArgs e)
        {
            DocumentWindow document = this.DockManager.ActiveContent as DocumentWindow;
            if (document == null)
            {
                this.ActiveDocument = null;
                this.ActiveDocumentChanged(this, null);
                
                return;
            }

            if (this.findWindow != null)
                ((FindReplaceWindow)this.findWindow.Content).TextBox = document.TextBox;

            this.ActiveDocument = document;
            this.ActiveDocumentChanged(this, null);
        }

        bool LoadWorkSpace()
        {
            string recoveryStateFilePattern = string.Format(Prop.Resources.RecoveryState, "*", "");
            if (Directory.Exists(Config.ApplicationFolder) == false)
                return true;
            string[] files = Directory.GetFiles(Config.ApplicationFolder, recoveryStateFilePattern);
            if (files.Length > 0 && MessageBox.Show(Prop.Resources.RecoveryConfirm, "", MessageBoxButton.YesNo) == MessageBoxResult.No)
            {
                for (int i = 0; i < files.Length; i++)
                    File.Delete(files[i]);
                return true;
            }
            else if(files.Length == 0)
            {
                return true;
            }
            for (int i = 0; i < files.Length; i++)
            {
                BinaryFormatter formatter = new BinaryFormatter();
                FileStream fs = new FileStream(files[i], FileMode.Open, FileAccess.Read);
                DocumentWindow document = (DocumentWindow)formatter.Deserialize(fs);
                fs.Close();
                File.Delete(files[i]);
                this.CreatedDocument(this, new CreatedDocumentEventArgs(document));
                this.Documents.Add(document);
            }
            return false;
        }

        #region Command
        void CanExecute(object sender, CanExecuteRoutedEventArgs e)
        {
            DocumentWindow document = this.DockManager.ActiveContent as DocumentWindow;
            if (document == null)
                e.CanExecute = false;
            else
                e.CanExecute = true;
        }

        void LineJumpCommand(object sender, ExecutedRoutedEventArgs e)
        {
            DocumentWindow document = this.DockManager.ActiveContent as DocumentWindow;
            if (document == null)
                return;
            LineJumpDialog dialog = new LineJumpDialog();
            dialog.TextBox = document.TextBox;
            dialog.Owner = this;
            dialog.ShowDialog();
        }

        void PrintCommand(object sender, ExecutedRoutedEventArgs e)
        {
            DocumentWindow document = this.DockManager.ActiveContent as DocumentWindow;
            if (document == null)
                return;
            PrintDialog pd = new PrintDialog();
            pd.PageRangeSelection = PageRangeSelection.AllPages;
            pd.UserPageRangeEnabled = true;
            if (pd.ShowDialog() == false)
                return;
            Config config = Config.GetInstance();
            FooPrintText printtext = new FooPrintText();
            PrintHFParser Parser = new PrintHFParser(document);
            printtext.Document = document.TextBox.Document;
            printtext.Font = document.TextBox.FontFamily;
            printtext.FontSize = document.TextBox.FontSize;
            printtext.DrawLineNumber = document.TextBox.DrawLineNumber;
            printtext.Header = config.Header;
            printtext.Footer = config.Footer;
            printtext.ParseHF = Parser.Parse;
            printtext.isWordRap = true;
            printtext.MarkURL = document.TextBox.MarkURL;
            printtext.Hilighter = document.TextBox.Hilighter;
            printtext.Foreground = new SolidColorBrush(document.TextBox.Foreground);
            printtext.URL = new SolidColorBrush(document.TextBox.URL);
            printtext.Comment = new SolidColorBrush(document.TextBox.Comment);
            printtext.Keyword1 = new SolidColorBrush(document.TextBox.Keyword1);
            printtext.Keyword2 = new SolidColorBrush(document.TextBox.Keyword2);
            printtext.Litral = new SolidColorBrush(document.TextBox.Literal);
            if (pd.PageRangeSelection == PageRangeSelection.AllPages)
            {
                printtext.StartPage = -1;
                printtext.EndPage = -1;
            }
            else
            {
                printtext.StartPage = pd.PageRange.PageFrom;
                printtext.EndPage = pd.PageRange.PageTo;
            }
            printtext.PageRect = new Rect(config.LeftSpace,
                config.TopSpace,
                pd.PrintableAreaWidth - config.RightSpace,
                pd.PrintableAreaHeight - config.BottomSpace);
            printtext.Print(pd);
        }

        void SelectDocumentTypeCommand(object sender, ExecutedRoutedEventArgs e)
        {
            DocumentWindow document = this.DockManager.ActiveContent as DocumentWindow;
            if (document == null)
                return;
            string type = (string)e.Parameter;
            document.DocumentType = (string)type;
            this.DocumentTypes.Select(type);
        }

        void PropertiesCommand(object sender, ExecutedRoutedEventArgs e)
        {
            ConfigDialog cd = new ConfigDialog(this.plugins);
            cd.Owner = this;
            if (cd.ShowDialog() == false)
                return;
            Config config = Config.GetInstance();
            foreach (DocumentWindow document in this.Documents)
            {
                document.TextBox.Foreground = config.Fore;
                document.TextBox.Background = config.Back;
                document.TextBox.Comment = config.Comment;
                document.TextBox.ControlChar = config.Control;
                document.TextBox.Keyword1 = config.Keyword1;
                document.TextBox.Keyword2 = config.Keyword2;
                document.TextBox.Literal = config.Literal;
                document.TextBox.Hilight = config.Hilight;
                document.TextBox.URL = config.URL;
                document.TextBox.InsertCaret = config.InsetCaret;
                document.TextBox.OverwriteCaret = config.OverwriteCaret;
                document.TextBox.LineMarker = config.LineMarker;
                document.TextBox.FontFamily = new FontFamily(config.FontName);
                document.TextBox.FontSize = config.FontSize;
                document.TextBox.DrawCaretLine = config.DrawLine;
                document.TextBox.DrawLineNumber = config.DrawLineNumber;
                document.TextBox.WordRap = config.WordRap;
                document.TextBox.MarkURL = config.UrlMark;
                document.TextBox.TabChars = config.TabStops;
                document.TextBox.TextAntialiasMode = config.TextAntialiasMode;
                document.EnableAutoIndent = config.AutoIndent;
                document.TextBox.Refresh();
            }
        }

        void QuitCommand(object sender, ExecutedRoutedEventArgs e)
        {
            this.Close();
        }

        void GrepCommand(object sender, ExecutedRoutedEventArgs e)
        {
            Config config = Config.GetInstance();
            if (!File.Exists(config.GrepPath))
                return;
            ProcessStartInfo info = new ProcessStartInfo();
            info.FileName = config.GrepPath;
            info.Verb = "open";
            info.UseShellExecute = true;
            Process process = Process.Start(info);
        }

        void NewCommand(object sender, ExecutedRoutedEventArgs e)
        {
            this.CreateDocument();
        }

        void OpenRecentFileCommand(object sender, ExecutedRoutedEventArgs e)
        {
            DocumentWindow document = this.DockManager.ActiveContent as DocumentWindow;
            string filepath = (string)e.Parameter;
            if (document == null || document.FilePath != null || document.Dirty)
                document = this.CreateDocument();
            document.Load(filepath, null);
        }

        void OpenCommand(object sender, ExecutedRoutedEventArgs e)
        {
            DocumentWindow document = this.DockManager.ActiveContent as DocumentWindow;
            CustomOpenFileDialog ofd = new CustomOpenFileDialog();
            ofd.addFilter(Prop.Resources.AllFileLable, "*.*");
            if (ofd.ShowDialog() == CommonFileDialogResult.Ok)
            {
                if (document == null || document.FilePath != null || document.Dirty)
                    document = this.CreateDocument();
                document.Load(ofd.FileName, ofd.FileEncoding);
            }
            ofd.Dispose();
        }

        void SaveCommand(object sender, ExecutedRoutedEventArgs e)
        {
            DocumentWindow document = this.DockManager.ActiveContent as DocumentWindow;
            this.Save(document);
        }
        
        void SaveAsCommand(object sender, ExecutedRoutedEventArgs e)
        {
            DocumentWindow document = this.DockManager.ActiveContent as DocumentWindow;
            this.SaveAs(document);
        }

        void Save(DocumentWindow document)
        {
            if (document == null)
                return;
            if (document.FilePath == string.Empty || document.FilePath == null)
                this.SaveAs(document);
            else
                document.Save(document.FilePath, document.Encoding);
        }

        void SaveAs(DocumentWindow document)
        {
            if (document == null)
                return;
            CustomSaveFileDialog sfd = new CustomSaveFileDialog();
            sfd.addFilter(Prop.Resources.AllFileLable, "*.*");
            sfd.FileEncoding = document.Encoding;
            sfd.LineFeed = document.LineFeed;
            if (sfd.ShowDialog() == CommonFileDialogResult.Ok)
            {
                document.Save(sfd.FileName, sfd.FileEncoding);
                document.FilePath = sfd.FileName;
            }
            sfd.Dispose();
        }

        void FindCommand(object sender, ExecutedRoutedEventArgs e)
        {
            DocumentWindow document = this.DockManager.ActiveContent as DocumentWindow;
            if (document == null)
                return;
            this.findWindow.Content.TextBox = document.TextBox;
            this.findWindow.IsVisible = true;
        }

        void SaveWorkSpace(object sender, ExecutedRoutedEventArgs e)
        {
            foreach (DocumentWindow document in this.Documents)
            {
                string stateFilePath = string.Format(Config.ApplicationFolder + "\\" + Prop.Resources.RecoveryState, Process.GetCurrentProcess().Id, document.Title);

                if (Directory.Exists(Config.ApplicationFolder) == false)
                    Directory.CreateDirectory(Config.ApplicationFolder);

                BinaryFormatter formatter = new BinaryFormatter();
                FileStream fs = new FileStream(stateFilePath, FileMode.Create, FileAccess.Write);
                formatter.Serialize(fs, document);
                fs.Close();
            }
        }

        void HelpCommand(object sender, ExecutedRoutedEventArgs e)
        {
            Process.Start(Path.Combine(Config.ExecutablePath, Prop.Resources.HelpFileName));
        }

        void AboutCommand(object sender, ExecutedRoutedEventArgs e)
        {
            FileVersionInfo versionInfo = FileVersionInfo.GetVersionInfo(Environment.GetCommandLineArgs()[0]);
            string str = string.Format("{0} version{1}\n{2}", versionInfo.ProductName, versionInfo.ProductVersion, versionInfo.CompanyName);
            MessageBox.Show(str, "FooEditorについて");
            throw new NotImplementedException();
        }
        void TileVerticalCommand(object sender, ExecutedRoutedEventArgs e)
        {
            this.TileDocument(LayoutFlag.Vertical);
        }
        void TileHorizontalCommand(object sender, ExecutedRoutedEventArgs e)
        {
            this.TileDocument(LayoutFlag.Horizontal);
        }
        void CasecadeCommand(object sender, ExecutedRoutedEventArgs e)
        {
            this.CascadeDocument();
        }
        #endregion

        enum LayoutFlag
        {
            Vertical,
            Horizontal,
        }

        void TileDocument(LayoutFlag flag)
        {
            if (this.Documents.Count <= 1)
                return;

            this.DocumentPanels.Children.Clear();
            if(flag == LayoutFlag.Horizontal)
                this.DocumentPanels.Orientation = Orientation.Horizontal;
            else
                this.DocumentPanels.Orientation = Orientation.Vertical;
            for (int i = 0; i < this.Documents.Count; i++)
            {
                DocumentWindow document = this.Documents[i];
                LayoutDocument layoutDocument = new LayoutDocument();
                layoutDocument.Content = document;
                LayoutDocumentPane panel = new LayoutDocumentPane();
                panel.Children.Add(layoutDocument);
                panel.DockWidth = new GridLength(1, GridUnitType.Star);
                panel.DockHeight = new GridLength(1,GridUnitType.Star);
                this.DocumentPanels.Children.Add(panel);
            }
        }

        void CascadeDocument()
        {
            if (this.Documents.Count <= 1)
                return;

            this.DocumentPanels.Children.Clear();
            LayoutDocumentPane panel = new LayoutDocumentPane();
            for (int i = 0; i < this.Documents.Count; i++)
            {
                DocumentWindow document = this.Documents[i];
                LayoutDocument layoutDocument = new LayoutDocument();
                layoutDocument.Content = document;
                panel.Children.Add(layoutDocument);
            }
            this.DocumentPanels.Children.Add(panel);
        }
    }

    public static class FooEditorMenuItemName
    {
        public static string FileMenuName = "FileMenuItem";
        public static string EditMenuName = "EditMenuItem";
        public static string LookMenuName = "LookMenuItem";
        public static string ToolMenuName = "ToolMenuItem";
    }

    public static class FooEditorResourceName
    {
        public static string ContextMenuName = "ContextMenu";
    }

    public class CreatedDocumentEventArgs : EventArgs
    {
        public DocumentWindow newDocument;
        public CreatedDocumentEventArgs(DocumentWindow newdoc)
        {
            this.newDocument = newdoc;
        }
    }
}
