#!/usr/bin/perl

# Copyright (c) 2017 dyknon
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. The name of the author may not be used to endorse or promote products
#    derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

use warnings;
use strict;
use utf8;
use IO::Pipe;
use IO::Socket::UNIX;
use IO::Select;

my $sockpath = $ENV{HOME}."/socks/media_server.sock";
our $buf_len = 5000;

$| = 1;
$SIG{PIPE} = "IGNORE";

our %ioev_handler;
our @alarm_handler;
our $readsel;
our $writesel;
our $header = "";
our %clusters;
our $cluster_num = 0;
our $stream_end = 0;

package StreamInHandler;
sub new{
    my $self = bless({}, shift);
    $self->{pipe} = shift;
    $self->{stat} = 0;
    $self->{buf} = "";
    $self->{cur} = \&proc_first;
    $self->{byteindex} = 0;
    $self->{pipe}->blocking(0);
    $self->{tcscale} = 1000000;
    $self->{tracks} = {};
    $readsel->add($self->{pipe});
    $ioev_handler{$self->{pipe}} = $self;
    return $self;
}

sub snapshoot{
    my $self = shift;
    $self->{bufbup} = $self->{buf};
    $self->{indexbup} = $self->{byteindex};
}

sub restore{
    my $self = shift;
    $self->{buf} = $self->{bufbup};
    $self->{byteindex} = $self->{indexbup};
}

sub gets{
    my $self = shift;
    my $size = shift;
    if($size > length($self->{buf})){
        return undef;
    }
    my $ret = substr($self->{buf}, 0, $size);
    $self->{buf} = substr($self->{buf}, $size);
    $self->{byteindex} += $size;
    return $ret;
}


sub read{
    my $self = shift;
    my $rbuf;
    my $read = $self->{pipe}->read($rbuf, 8192);
    if(!$read){
        print("video pipe closed\n");
        $stream_end = 1;
        for(@alarm_handler){
            $_->alarm();
        }
        $readsel->remove($self->{pipe});
    }else{
        $self->{buf} .= $rbuf;
        $self->proc_dat();
    }
}

sub read_ebml_value{
    my $self = shift;
    my $slim = shift;
    my $undflg = 1;
    my $buf;
    my $dat = "";
    defined($buf = $self->gets(1)) || return undef;
    $dat .= $buf;
    my @byte = unpack("C", $buf);
    my $size = -1;
    for(my $i = 0; $i < $slim; $i++){
        if(($byte[0] << $i) & 0x80){
            if($byte[0] != 0xff >> $i){
                $undflg = 0;
            }
            $size = $i;
            $byte[0] &= 0x7f >> $i;
            last;
        }
    }
    if($size < 0){
        die;
    }
    defined($buf = $self->gets($size)) || return undef;
    $dat .= $buf;
    push(@byte, unpack("C*", $buf));
    for(@byte[1 .. $#byte]){
        if($_ != 0xff){
            $undflg = 0;
        }
    }
    my $val = 0;
    for(@byte){
        $val <<= 8;
        $val |= $_;
    }
    if($undflg){
        return (undef, $size, $dat);
    }else{
        if(!(0x80 << ($size * 7))){
            die "data overflow";
        }
        return ($val, $size, $dat);
    }
}

sub read_ebml_value_string{
    my $str = shift;
    if(ref($str)){
        $str = shift;
    }
    my $slim = shift || 8;
    if(!length($str)){
        return undef;
    }
    my @strbin = unpack("C*", $str);
    my $class;
    for(0 .. $slim-1){
        if($strbin[0] & (0x80 >> $_)){
            $class = $_;
            $strbin[0] &= ~(0x80 >> $_);
            last;
        }
    }
    if(!defined($class)){
        return undef;
    }
    if($class >= @strbin){
        return (undef, $class);
    }
    my $val = 0;
    for(0 .. $class){
        $val <<= 8;
        $val |= $strbin[$_];
    }
    return ($val, $class);
}

sub string_to_uint{
    my $str = shift;
    if(ref($str)){
        $str = shift;
    }
    if(length($str) < 1 || 8 < length($str)){
        return undef;
    }
    my $val = 0;
    for(unpack("C*", $str)){
        $val <<= 8;
        $val |= $_;
    }
    return $val;
}

sub read_ebml_tag{
    my $self = shift;
    my @ret = $self->read_ebml_value(4);
    if(defined($ret[1])){
        if(!defined($ret[0])){
            die;
        }
        return ($ret[0] | (0x80 << $ret[1] * 7), $ret[2]);
    }else{
        return undef;
    }
}

sub read_ebml_len{
    my $self = shift;
    my @ret = $self->read_ebml_value(8);
    if(defined($ret[1])){
        return ($ret[0], $ret[2]);
    }else{
        return undef;
    }
}

sub proc_dat{
    my $self = shift;
    while($self->{cur}){
        $self->snapshoot();
        my $ret = $self->{cur}->($self);
        if($ret){
            if(ref($ret)){
                $self->{cur} = $ret;
            }else{
                undef($self->{cur});
            }
        }else{
            $self->restore();
            last;
        }
    }
}

sub proc_first{
    my $self = shift;
    (my ($tag, $rawtag) = $self->read_ebml_tag())[1] || return undef;
    (my ($len, $rawlen) = $self->read_ebml_len())[1] || return undef;
    if($tag == 0x18538067){      #Segment
        $header .= $rawtag.$rawlen;
        return \&proc_segm_head;
    }else{
        if(!defined($len)){
            die "unknown tag with undefined length";
        }
        (my $body = $self->gets($len)) || return undef;
        $header .= $rawtag.$rawlen.$body;
    }
    return \&proc_first;
}

sub proc_segm_head{
    my $self = shift;
    (my ($tag, $rawtag) = $self->read_ebml_tag())[1] || return undef;
    (my ($len, $rawlen) = $self->read_ebml_len())[1] || return undef;
    if($tag == 0x1549a966){             #Info
        if(!defined($len)){
            die "info's length needed";
        }
        $self->{tlt_end} = $self->{byteindex} + $len;
        $header .= $rawtag.$rawlen;
        return \&proc_info;
    }elsif($tag == 0x1654ae6b){         #Tracks
        if(!defined($len)){
            die "tracks's length needed";
        }
        $self->{tlt_end} = $self->{byteindex} + $len;
        $header .= $rawtag.$rawlen;
        return \&proc_tracks;
    }elsif($tag == 0x1f43b675){         #Cluster
        $self->restore();
        return \&proc_cluster;
    }else{
        if(!defined($len)){
            die "undefined length";
        }
        (my $body = $self->gets($len)) || return undef;
        $header .= $rawtag.$rawlen.$body;
        return \&proc_segm_head;
    }
}

sub proc_info{
    my $self = shift;
    if($self->{tlt_end} == $self->{byteindex}){
        return \&proc_segm_head;
    }elsif($self->{tlt_end} < $self->{byteindex}){
        die "info section bloken";
    }
    (my ($tag, $rawtag) = $self->read_ebml_tag())[1] || return undef;
    (my ($len, $rawlen) = $self->read_ebml_len())[1] || return undef;
    if(!defined($len)){
        die "info's length needed";
    }
    (my $body = $self->gets($len)) || return undef;
    $header .= $rawtag.$rawlen.$body;
    if($tag == 0x2ad7b1){
        ($self->{tcscale}) = string_to_uint($body);
        if(!$self->{tcscale}){      #undef or zero
            die "TimecodeScale error";
        }
    }
    return \&proc_info;
}

sub proc_tracks{
    my $self = shift;
    if($self->{tlt_end} == $self->{byteindex}){
        return \&proc_segm_head;
    }elsif($self->{tlt_end} < $self->{byteindex}){
        die "tracks section bloken";
    }
    (my ($tag, $rawtag) = $self->read_ebml_tag())[1] || return undef;
    (my ($len, $rawlen) = $self->read_ebml_len())[1] || return undef;
    if(!defined($len)){
        die "tracks's length needed";
    }
    $header .= $rawtag.$rawlen;
    if($tag == 0xae){                   #TrackEntry
        $self->{slt_end} = $self->{byteindex} + $len;
        $self->{new_track} = {};
        return \&proc_trackentry;
    }else{
        (my $body = $self->gets($len)) || return undef;
        $header .= $body;
    }
    return \&proc_tracks;
}

sub proc_trackentry{
    my $self = shift;
    if($self->{slt_end} == $self->{byteindex}){
        $self->{tracks}{$self->{new_track}{num}} = $self->{new_track}{type};
        print $self->{new_track}{num}.":".$self->{new_track}{type}."\n";
        return \&proc_tracks;
    }elsif($self->{slt_end} < $self->{byteindex}){
        die "trackentry bloken";
    }
    (my ($tag, $rawtag) = $self->read_ebml_tag())[1] || return undef;
    (my ($len, $rawlen) = $self->read_ebml_len())[1] || return undef;
    if(!defined($len)){
        die "trackentry's length needed";
    }
    (my $body = $self->gets($len)) || return undef;
    $header .= $rawtag.$rawlen.$body;
    if($tag == 0xd7){
        $self->{new_track}{num} = string_to_uint($body);
        if(!$self->{new_track}{num}){
            die "invalid track number";
        }
    }elsif($tag == 0x83){
        $self->{new_track}{type} = string_to_uint($body);
        if(!$self->{new_track}{type} || $self->{new_track}{type} > 254){
            die "invalid track type";
        }
    }
    return \&proc_trackentry;
}

sub proc_cluster{
    my $self = shift;
    (my ($tag, $rawtag) = $self->read_ebml_tag())[1] || return undef;
    (my ($len, $rawlen) = $self->read_ebml_len())[1] || return undef;
    if($tag == 0x1f43b675){         #Cluster
        if(!defined($len)){
            die "cluster length needed";
        }
        $self->{cluster_end} = $self->{byteindex} + $len;
        $self->{cluster_buf} = $rawtag.$rawlen;
        $self->{cluster_meta} = [];
        $self->{cluster_tc} = -1;
        undef($self->{cluster_key});
        return \&proc_clusters_children;
    }else{
        print "stream end\n";
        $stream_end = 1;
        for(@alarm_handler){
            $_->alarm();
        }
        return "end";
    }
}

sub proc_clusters_children{
    my $self = shift;
    if($self->{cluster_end} == $self->{byteindex}){
        $clusters{$cluster_num} = {dat=>$self->{cluster_buf}};
        my $nodep_cluster = 1;
        my %uns_tracks = %{$self->{tracks}};
        for(@{$self->{cluster_meta}}){
            if(defined($uns_tracks{$_->{num}})){
                $nodep_cluster = $nodep_cluster && $_->{nodep};
                delete($uns_tracks{$_->{num}});
            }
        }
        $nodep_cluster = $nodep_cluster && !%uns_tracks;
        $clusters{$cluster_num}{nodep} = $nodep_cluster;
        if($self->{cluster_tc} != -1){
            $clusters{$cluster_num}{time} = $self->{tcscale}/1000000000*$self->{cluster_tc};
        }
        if(defined($clusters{$cluster_num-32})){
            delete($clusters{$cluster_num-32});
        }
        if($nodep_cluster){
            printf("cluster%04d(at %07.2fsec, keyframe) bufferd\n",
                ${cluster_num}, $clusters{$cluster_num}{time});
        }else{
            printf("cluster%04d(at %07.2fsec) bufferd\n",
                ${cluster_num}, $clusters{$cluster_num}{time});
        }
        $cluster_num++;
        for(@alarm_handler){
            $_->alarm();
        }
        return \&proc_cluster;
    }elsif($self->{cluster_end} < $self->{byteindex}){
        die "bloken cluster";
    }
    (my ($tag, $rawtag) = $self->read_ebml_tag())[1] || return undef;
    (my ($len, $rawlen) = $self->read_ebml_len())[1] || return undef;
    if(!defined($len)){
        die "cluster's child's length needed";
    }
    (my $body = $self->gets($len)) || return undef;
    if($tag == 0xa3){                       #SimpleBlock
        my ($stream_num, $stnum_class) = read_ebml_value_string($body);
        if(!defined($stnum_class)){
            die "EBML SimpleBlock Bloken";
        }
        my $ifkey = unpack("C", substr($body, $stnum_class + 3, 1)) & 0x80;
        push(@{$self->{cluster_meta}}, {num=>$stream_num, nodep=>!!$ifkey});
    }elsif($tag == 0xe7){                   #Timecode
        $self->{cluster_tc} = string_to_uint($body);
    }
    $self->{cluster_buf} .= $rawtag.$rawlen.$body;
    return \&proc_clusters_children;
}

package AccepterHandler;
sub new{
    my $self = bless({}, shift);
    $self->{sock} = shift;
    $readsel->add($self->{sock});
    $ioev_handler{$self->{sock}} = $self;
    return $self;
}

sub read{
    my $self = shift;
    my $con = $self->{sock}->accept();
    if($con){
        ConnectionHandler->new($con);
    }
}

package ConnectionHandler;
sub new{
    my $self = bless({}, shift);
    $self->{sock} = shift;
    binmode($self->{sock}, ":raw");
    $readsel->add($self->{sock});
    $ioev_handler{$self->{sock}} = $self;
}

sub reset{
    my $self = shift;
    delete($self->{rtype});
    delete($self->{rval});
    $readsel->add($self->{sock});
    $writesel->remove($self->{sock});
    @alarm_handler = grep{$_!=$self}(@alarm_handler);
}

sub close{
    my $self = shift;
    $readsel->remove($self->{sock});
    $writesel->remove($self->{sock});
    delete($ioev_handler{$self->{sock}});
    @alarm_handler = grep{$_!=$self}(@alarm_handler);
    $self->{sock}->close();
}

sub read{
    my $self = shift;
    my $rbuf;
    if(!defined($self->{rtype})){
        $self->{sock}->recv($rbuf, 1, 0);
        if(!length($rbuf)){
            $self->close();
            return;
        }
        $self->{rtype} = $rbuf;
        if($self->{rtype} eq "h"){
            $readsel->remove($self->{sock});
            $writesel->add($self->{sock});
            $self->{wbuf} = pack("Q", length($header));
            $self->{pos} = 0;
        }elsif($self->{rtype} eq "k"){
            $readsel->remove($self->{sock});
            if($cluster_num && $clusters{$cluster_num-1}{nodep}){
                $writesel->add($self->{sock});
                $self->{wbuf} = pack("Q", $cluster_num-1);
            }elsif($stream_end){
                $writesel->add($self->{sock});
                $self->{wbuf} = pack("Q", 0);
            }else{
                push(@alarm_handler, $self);
            }
        }elsif($self->{rtype} eq "c" || $self->{rtype} eq "i"
                || $self->{rtype} eq "w"){
            $self->{rbuf} = "";
        }else{
            $self->close();
        }
    }elsif(!defined($self->{rval})){
        $self->{sock}->recv($rbuf, 8-length($self->{rbuf}), 0);
        if(!length($rbuf)){
            $self->close();
            return;
        }
        $self->{rbuf} .= $rbuf;
        if(length($self->{rbuf}) >= 8){
            $self->{rval} = unpack("Q", $self->{rbuf});
            $readsel->remove($self->{sock});
            $writesel->add($self->{sock});
            if($self->{rtype} eq "c"){
                if(defined($clusters{$self->{rval}})){
                    $self->{wbuf} = pack("Q",
                            length($clusters{$self->{rval}}->{dat}));
                    $self->{pos} = 0;
                }else{
                    $self->{wbuf} = pack("Q", 0);
                }
            }elsif($self->{rtype} eq "w"){
                if($self->{rval} < $cluster_num){
                    $self->{wbuf} = "*";
                }elsif($stream_end){
                    $writesel->add($self->{sock});
                    $self->{wbuf} = pack("C", 0);
                }else{
                    $writesel->remove($self->{sock});
                    push(@alarm_handler, $self);
                }
            }elsif($self->{rtype} eq "i"){
                my $sum = 0;
                for(my $i = $self->{rval}; $i < $cluster_num; $i++){
                    if(defined($clusters{$i})){
                        $sum += length($clusters{$i}->{dat});
                    }else{
                        $sum = 0;
                        last;
                    }
                }
                my $flags = 0;
                my $pos = 0;
                if($clusters{$self->{rval}}){
                    if($clusters{$self->{rval}}{nodep}){
                        $flags |= 1;
                    }
                    $pos = int($clusters{$self->{rval}}{time} * 1000);
                }
                if($stream_end){
                    $flags |= 2;
                }
                my $cpos = 0;
                if($cluster_num){
                    $cpos = int($clusters{$cluster_num-1}{time} * 1000);
                }
                $self->{wbuf} = pack("QQQQLC",
                        $cluster_num, $sum, $pos, $cpos, $buf_len, $flags);
            }
        }
    }else{
        die;
    }
}

sub write{
    my $self = shift;
    if(length($self->{wbuf})){
        my $written = $self->{sock}->send($self->{wbuf}, 0);
        if(!defined($written)){
            $self->close();
            return;
        }
        $self->{wbuf} = substr($self->{wbuf}, $written);
    }else{
        if($self->{rtype} eq "h"){
            my $written = $self->{sock}->send(
                    substr($header, $self->{pos}), 0);
            if(!defined($written)){
                $self->close();
                return;
            }
            $self->{pos} += $written;
            if($self->{pos} == length($header)){
                $self->reset();
            }elsif($self->{pos} > length($header)){
                $self->close();
            }
        }elsif($self->{rtype} eq "c"){
            if(defined($clusters{$self->{rval}})){
                my $written = $self->{sock}->send(
                        substr($clusters{$self->{rval}}->{dat}, $self->{pos}), 0);
                if(!defined($written)){
                    $self->close();
                    return;
                }
                $self->{pos} += $written;
                if($self->{pos} == length($clusters{$self->{rval}}->{dat})){
                    $self->reset();
                }elsif($self->{pos} > length($clusters{$self->{rval}}->{dat})){
                    $self->close();
                }
            }
        }elsif($self->{rtype} eq "i" || $self->{rtype} eq "w"
                || $self->{rtype} eq "k"){
            $self->reset();
        }
    }
}

sub alarm{
    my $self = shift;
    if($self->{rtype} eq "k"){
        if($cluster_num && $clusters{$cluster_num-1}{nodep}){
            $self->{wbuf} = pack("Q", $cluster_num-1);
            $writesel->add($self->{sock});
            @alarm_handler = grep{$_!=$self}(@alarm_handler);
        }elsif($stream_end){
            $self->{wbuf} = pack("Q", 0);
            $writesel->add($self->{sock});
            @alarm_handler = grep{$_!=$self}(@alarm_handler);
        }
    }elsif($self->{rtype} eq "w"){
        if($self->{rval} < $cluster_num){
            $self->{wbuf} = "*";
            $writesel->add($self->{sock});
            @alarm_handler = grep{$_!=$self}(@alarm_handler);
        }elsif($stream_end){
            $self->{wbuf} = pack("C", 0);
            $writesel->add($self->{sock});
            @alarm_handler = grep{$_!=$self}(@alarm_handler);
        }
    }
}

package main;

$readsel = IO::Select->new();
$writesel = IO::Select->new();

my $in_handle = new IO::Pipe;
#$in_handle->reader(qw{ffmpeg -loglevel panic -i video2stream -map 0:v:0 -map 0:a:0 -c:v vp8 -deadline:v realtime -cpu-used:v 8 -g:v 20 -q:v 30 -qmax:v 32 -live 1 -vf realtime -c:a libvorbis -b:a 64k -cluster_size_limit 10000000 -cluster_time_limit 1000000 -s wvga -f webm -});
#$in_handle->reader(qw{ffmpeg -loglevel panic -f video4linux2 -r:v 10 -s vga -i /dev/video0 -map 0:v -c:v vp8 -deadline:v good -cpu-used:v 16 -g:v 10 -q:v 25 -qmax:v 28 -pix_fmt yuv420p -live 1 -vf realtime -cluster_size_limit 10000000 -cluster_time_limit 1000000 -f webm -});
#$in_handle->reader(qw{ffmpeg -loglevel panic -s 1366x768 -r:v 20 -f x11grab -i :0+1080,880 -map 0:v -c:v vp8 -deadline:v good -cpu-used:v 8 -g:v 10 -q:v 25 -qmax:v 28 -pix_fmt yuv420p -live 1 -cluster_size_limit 10000000 -cluster_time_limit 1000000 -s wvga -vf realtime -f webm -});
defined($ARGV[0]) || die;
$in_handle->reader(qw{ffmpeg -loglevel panic -i}, $ARGV[0], qw{-map 0:v:0 -map 0:a:0 -c:v vp8 -deadline:v realtime -cpu-used:v 8 -g:v 20 -q:v 24 -qmax:v 28 -live 1 -vf realtime -c:a libvorbis -ac 2 -ar 44.1k -b:a 64k -cluster_size_limit 10000000 -cluster_time_limit 1000000 -s wvga -f webm -});
binmode($in_handle, ":raw");
StreamInHandler->new($in_handle);

if(-e $sockpath){
    unlink($sockpath) || die;
}

my $listen = IO::Socket::UNIX->new(Type=>SOCK_STREAM,
                                   Local=>$sockpath,
                                   Listen=>1) || die;
chmod(0777, $sockpath);
AccepterHandler->new($listen);
print("Sock: $sockpath\n");

while(1){
    my ($read, $write, $except) = 
            IO::Select->select($readsel, $writesel, undef);
    for(@$read){
        $ioev_handler{$_}->read();
    }
    for(@$write){
        $ioev_handler{$_}->write();
    }
}

$listen->close();
