#!/usr/bin/perl

# Copyright (c) 2017 dyknon
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. The name of the author may not be used to endorse or promote products
#    derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

use warnings;
use strict;
use utf8;
use IO::Socket::UNIX;
use JSON;
use Encode;
use Cwd;

$| = 1;

my $home = Cwd::getcwd();
$home =~ s!/public_html/.*!/!;
my $sockpath = $home."socks/media_server.sock";

if(!$ENV{QUERY_STRING}){
    binmode(STDOUT, ":utf8");
    print("Content-type: text/html; charset=UTF-8\n\n");
    print(<<EOT
<!DOCTYPE html>
<html>
  <head>
    <meta charset="UTF-8">
    <title>なんかLiveストリーム</title>
    <script>
        function live_streamer(videoelm, controlbox, statbox, url){
            this.velm = videoelm;
            var defvol = this.velm.volume;
            this.cont = controlbox;
            this.statbox = statbox;
            this.serv = url;
            this.media_mime = "video/webm; codecs=\\\"vp8, vorbis\\\"";
            this.started = false;
            this.cache_len = 5;
            this.skip_delay = 10;

            if(!MediaSource
                        || !MediaSource.isTypeSupported(this.media_mime)){
              alert("キミの環境では再生するのむり");
              return;
            }
            this.ms = new MediaSource();
            this.velm.src = URL.createObjectURL(this.ms);
            this.velm.volume = defvol;
            this.nexcb = null;
            var lsobj = this;
            this.ms.addEventListener("sourceopen", function(){
                lsobj.sb = lsobj.ms.addSourceBuffer(lsobj.media_mime);
                lsobj.sb.mode = "sequence";
                lsobj.nexcb = lsobj.keycb;
                lsobj.request("key", null);
                lsobj.statbox.textContent = "waiting for key frame";
            });

            this.vol_bar = document.createElement("input");
            this.vol_bar.type = "range";
            this.vol_bar.max = 1;
            this.vol_bar.min = 0;
            this.vol_bar.step = 0.01;
            this.vol_bar.value = defvol;
            this.velm.volume = this.vol_bar.value;
            this.vol_bar.addEventListener("change", this);
            this.cont.innerHTML = "volume: ";
            this.cont.appendChild(this.vol_bar);

            this.velm.addEventListener("fullscreenchange", this);
            this.fs_button = document.createElement("input");
            this.fs_button.type = "button";
            this.fs_button.value = "フルスクリーン(可能なら)";
            this.fs_button.addEventListener("click", this);
            this.cont.appendChild(this.fs_button);
        }
        live_streamer.prototype.keycb = function(resp){
            if(resp === null){
                this.statbox.textContent = "通信エラー(再読込してね)";
                return;
            }else if(resp.status == "nostreams"){
                this.statbox.textContent = "配信なしだよ。";
                window.setTimeout(function(t){t.keyretrycb()}, 300, this);
                return;
            }else if(resp.flags & 2){
                this.statbox.textContent = "配信終了だよ";
                window.setTimeout(function(t){t.keyretrycb()}, 300, this);
                return;
            }
            this.sindex = resp.key;
            this.cindex = resp.key;
            this.cache_len = resp.buflen / 1000;
            this.skip_delay = this.cache_len*2;
            this.nexcb = this.datacb;
            this.request("head", null);
            this.statbox.textContent = "loading video header";
            //this.cindex++;
        };
        live_streamer.prototype.keyretrycb = function(){
            this.nexcb = this.keycb;
            this.request("key", null);
        };
        live_streamer.prototype.datacb = function(resp){
            if(resp === null){
                this.statbox.textContent = "放送終了かな。";
                this.ms.endOfStream();
                window.setTimeout(function(t){
                    new live_streamer(t.velm, t.cont, t.statbox, t.serv);
                }, (1000*this.bufremain()+500)|0, this);
                return;
            }else if(resp.status == "nostreams"){
                this.statbox.textContent = "配信なしだよ。";
                return;
            }
            this.nexcb = this.updatecb;
            this.sb.addEventListener("updateend", this);
            this.sb.appendBuffer(resp);
        };
        live_streamer.prototype.updatecb = function(ev){
            this.nexcb = this.datacb;
            this.request("dat", this.cindex);
            if(!this.started && this.buftail() >= this.cache_len){
                this.started = true;
                this.velm.play();
            }
            var stat = "cluster "+this.cindex+" requesting<br>";
            stat += "time:"+this.velm.currentTime+"<br>";
            stat += "buffered:"+this.bufremain()+"<br>";
            this.statbox.innerHTML = stat;
            this.cindex++;
            if(this.bufremain() > this.skip_delay){
                this.velm.currentTime += this.bufremain() - this.cache_len;
            }else if(this.bufremain() > this.cache_len + 1.5){
                this.velm.playbackRate = 1
                        + (this.bufremain() - this.cache_len) / 10;
            }else if(this.bufremain() < this.cache_len - 1.5){
                this.velm.playbackRate = 1
                        - (this.cache_len - this.bufremain())
                        / (this.cache_len * 2);
            }else if(this.velm.playbackRate > 1 &&
                                this.bufremain() <= this.cache_len){
                this.velm.playbackRate = 1;
            }else if(this.velm.playbackRate < 1 &&
                                this.bufremain() >= this.cache_len){
                this.velm.playbackRate = 1;
            }
        };
        live_streamer.prototype.request = function(com, n){
            var req = new XMLHttpRequest;
            if(n === null){
                req.open("get", this.serv+"?req="+com);
            }else{
                req.open("get", this.serv+"?req="+com+"&n="+n);
            }
            req.responseType = "arraybuffer";
            req.addEventListener("load", this);
            req.send();
        };
        live_streamer.prototype.buftail = function(){
            var ret = 0;
            for(var i = 0; i < this.sb.buffered.length; i++){
                var kh = this.sb.buffered.end(i);
                if(ret < kh){
                    ret = kh;
                }
            }
            return ret;
        };
        live_streamer.prototype.bufremain = function(){
            return this.buftail() - this.velm.currentTime;
        };
        live_streamer.prototype.handleEvent = function(ev){
            ev.target.removeEventListener(ev.type, this);
            if(ev.type == "load" && ev.target instanceof XMLHttpRequest){
                if(ev.target.status != 200){
                    this.nexcb.call(this, null);
                    return;
                }
                var ct = ev.target.getResponseHeader("Content-Type");
                if(ct == "application/json"){
                    var str = String.fromCharCode.apply(window,
                                new Uint8Array(ev.target.response));
                    this.nexcb.call(this, JSON.parse(str));
                }else{
                    this.nexcb.call(this, ev.target.response);
                }
            }else if(ev.type == "change" && ev.target == this.vol_bar){
                this.velm.volume = this.vol_bar.value;
                this.vol_bar.addEventListener("change", this);
            }else if(ev.type == "click" && ev.target == this.fs_button){
                if(this.velm.requestFullscreen){
                    this.velm.requestFullscreen();
                }else if(this.velm.mozRequestFullScreen){
                    this.velm.mozRequestFullScreen();
                }else if(this.velm.webkitRequestFullScreen){
                    this.velm.webkitRequestFullScreen();
                }else if(this.velm.msRequestFullScreen){
                    this.velm.msRequestFullScreen();
                }
                this.fs_button.addEventListener("click", this);
            }else if(ev.type == "fullscreenchange"
                    && ev.target == this.velm){
                if(this.velm.fullscreenEnabled){
                    this.velm.style.width = "100%";
                    this.velm.style.height = "100%";
                }else{
                    this.velm.style.width = "";
                    this.velm.style.height = "";
                }
                this.velm.addEventListener("fullscreenchange", this);
            }else{
                this.nexcb.call(this, ev);
            }
        }
        function init_player(e){
            var videoelm = document.getElementById("mainvideo");
            var statbox = document.getElementById("statbox");
            var controlbox = document.getElementById("controls");
            var wl = window.location;
            var url = wl.protocol+"//"+wl.host+wl.pathname;
            new live_streamer(videoelm, controlbox, statbox, url);
        }
        window.addEventListener("load", init_player);
    </script>
  </head>
  <body>
    <video id="mainvideo" style="max-width:100%;max-height:100%"></video>
    <div id="controls"></div>
    <div id="statbox"></div>
  </body>
</html>
EOT
);
    exit;
}

my $con = IO::Socket::UNIX->new(Type=>SOCK_STREAM, Peer=>$sockpath);

if($con){
    my %args;
    for(split(/&/, $ENV{QUERY_STRING})){
        my ($key, $val) = split(/=/, $_, 2);
        $key =~ s/%([0-9A-Fa-f]{2})/chr(hex($1))/ge;
        if(defined($val)){
            $val =~ s/%([0-9A-Fa-f]{2})/chr(hex($1))/ge;
            $args{Encode::decode("utf8", $key)}
                    = Encode::decode("utf8", $val);
        }else{
            $args{Encode::decode("utf8", $key)} = 1;
        }
    }

    my $buf;
    if(!defined($args{req})){
        goto IF_BAD_REQ;
    }elsif($args{req} eq "info"){
        my $num = 0;
        if($args{n} && $args{n} =~ /^[0-9]+$/){
            $num = $args{n};
        }
        $con->send("i".pack("Q", $num), 0);
        $con->recv($buf, 8*4+4+1, 0);
        my ($cnum, $size, $pos, $cpos, $buf_len, $flags) = unpack("QQQQLC", $buf);
        send_json({status=>"ok", num=>$cnum, size=>$size,
                pos=>$pos, cpos=>$cpos, buflen=>$buf_len, flags=>$flags});
    }elsif($args{req} eq "key"){
        $con->send("k", 0);
        $con->recv($buf, 8, 0);
        my $num = unpack("Q", $buf);
        $con->send("i".pack("Q", $num), 0);
        $con->recv($buf, 8*4+4+1, 0);
        my ($cnum, $size, $pos, $cpos, $buf_len, $flags) = unpack("QQQQLC", $buf);
        send_json({status=>"ok", key=>$num, num=>$cnum, size=>$size,
                pos=>$pos, cpos=>$cpos, buflen=>$buf_len, flags=>$flags});
    }elsif($args{req} eq "head"){
        if(defined($args{n})){
            if($args{n} =~ /^[0-9]+$/){
                my $num = $args{n};
                binmode(STDOUT, ":utf8");
                $con->send("h", 0);
                $con->recv($buf, 8, 0);
                my $size = unpack("Q", $buf);
                if(!$size){
                    goto IF_ERROR;
                }
                print("Content-type: video/webm\n\n");
                send_dat($size);
                $con->send("c".pack("Q", $num), 0);
                $con->recv($buf, 8, 0);
                $size = unpack("Q", $buf);
                if(!$size){
                    $con->close();
                    exit;
                }
                send_dat($size);
                $con->close();
                exit;
            }else{
                goto IF_BAD_REQ;
            }
        }else{
            $con->send("h", 0);
            goto RESP_DATA;
        }
    }elsif($args{req} eq "dat"){
        if(!defined($args{n}) || $args{n} !~ /^[0-9]+$/){
            goto IF_BAD_REQ;
        }
        my $num = $args{n};
        $con->send("w".pack("Q", $num), 0);
        $con->recv($buf, 1, 0);
        $con->send("c".pack("Q", $num), 0);
        goto RESP_DATA;
    }elsif($args{req} eq "stream"){
        my $num;
        if(!defined($args{n})){
            $con->send("k", 0);
            $con->recv($buf, 8, 0);
            $num = unpack("Q", $buf);
        }elsif($args{n} =~ /^[0-9]+$/){
            $num = $args{n};
            $con->send("i".pack("Q", $num), 0);
            $con->recv($buf, 8*4+4+1, 0);
            my ($cnum, $size, $pos, $cpos, $buf_len, $flags) = unpack("QQQQLC", $buf);
            if(!($flags & 0x01)){
                goto IF_BAD_REQ;
            }
        }else{
            goto IF_BAD_REQ;
        }

        binmode(STDOUT, ":utf8");
        $con->send("h", 0);
        $con->recv($buf, 8, 0);
        my $size = unpack("Q", $buf);
        if(!$size){
            goto IF_ERROR;
        }
        print("Content-type: video/webm\n\n");
        send_dat($size);

        my $stnum = $num;
        while(1){
            $con->send("c".pack("Q", $num), 0);
            $con->recv($buf, 8, 0);
            my $size = unpack("Q", $buf);
            if(!$size){
                $con->close();
                exit;
            }
            send_dat($size);
            $num++;
            $con->send("w".pack("Q", $num), 0);
            $con->recv($buf, 1, 0);
        }
    }

IF_BAD_REQ:
    binmode(STDOUT, ":utf8");
    print("Content-type: text/html; charset=UTF-8\n\n");
    print("Status: 400\n\n");
    $con->close();
    exit;
IF_ERROR:
    binmode(STDOUT, ":utf8");
    print("Content-type: text/html; charset=UTF-8\n\n");
    print("Status: 503\n\n");
    $con->close();
    exit;
RESP_DATA:
    $con->recv($buf, 8, 0);
    my $size = unpack("Q", $buf);
    binmode(STDOUT, ":utf8");
    if(!$size){
        print("Status: 404\n\n");
        $con->close();
        exit;
    }
    print("Content-type: video/x-webm-flagment\n");
    print("Content-length: $size\n\n");
    send_dat($size);
    $con->close();
    exit;
}else{
    send_json({status=>"nostreams"});
}

sub send_dat{
    my $size = shift;
    my $buf;
    binmode(STDOUT, ":raw");
    while($size > 0){
        $con->recv($buf, $size<1024?$size:1024, 0);
        if(!length($buf)){
            $con->close();
            exit;
        }
        $size -= length($buf);
        print($buf);
    }
}

sub send_json{
    binmode(STDOUT, ":utf8");
    print("Content-type: application/json\n\n");
    binmode(STDOUT, ":raw");
    print(to_json(shift, {ascii=>1}));
    if($con){
        $con->close();
    }
    exit;
}
