/*
 * JHPdf Free PDF Library : HPdfTrueTypeFontDef.java
 *
 * URL:
 *
 * Copyright (c) 2012- Toshiaki Yoshida <toshi@doju-m.jp>
 * {
 * Based on 'Haru Free PDF Library' (http://libharu.org)
 * Copyright (c) 1999-2006 Takeshi Kanno <takeshi_kanno@est.hi-ho.ne.jp>
 * Copyright (c) 2007-2009 Antony Dovgal <tony@daylessday.org>
 * }
 *
 * Permission to use, copy, modify, distribute and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation.
 * It is provided "as is" without express or implied warranty.
 *
 */

package net.sf.jhpdf.pdfobject.font;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.jhpdf.ByteUtil;
import net.sf.jhpdf.HPdfConst;
import net.sf.jhpdf.HPdfErrorCode;
import net.sf.jhpdf.HPdfException;
import net.sf.jhpdf.graphics.HPdfRect;
import net.sf.jhpdf.io.HPdfFileWriteStream;
import net.sf.jhpdf.io.HPdfMemStream;
import net.sf.jhpdf.io.HPdfReadStream;
import net.sf.jhpdf.io.HPdfReadStream.SeekMode;
import net.sf.jhpdf.io.HPdfStream.HPdfFilterFlag;
import net.sf.jhpdf.io.HPdfWriteStream;

/**
 * Class represents PDF Type1 Font definition.
 * @author Toshiaki Yoshida
 * @version 0.1
 *
 */
public class HPdfTrueTypeFontDef extends HPdfFontDef {

    private static final Logger logger = LoggerFactory.getLogger(HPdfTrueTypeFontDef.class);
    
    private static final String[] REQUIRED_TAGS = {
        "OS/2",
        "cmap",
        "cvt ",
        "fpgm",
        "glyf",
        "head",
        "hhea",
        "hmtx",
        "loca",
        "maxp",
        "name",
        "post",
        "prep"
    };
    
    private static final boolean IS_DUMP_FONTDATA =
            (System.getProperty("HPDF_DUMP_FONTDATA") != null);
    
    private static final int HPDF_TTF_MAX_MEM_SIZ = 10000;
    
    private static final int HPDF_TTF_FONT_TAG_LEN = 6;
    
    public HPdfTrueTypeFontDef() {
        super();
        
        logger.trace("HPdfTrueTypeFontDef#ctor");
        
        this.setFontDefType(HPdfFontDefType.TRUETYPE);
        this.setFlag(HPDF_FONT_STD_CHARSET);
    }

    @Override
    public void init() {
        // MEMO nothing to do.
    }

    @Override
    public void clean() {
        HPdfTTFGlyphOffsets glyphTable = fontDefAttr.getGlyphTable();
        if (glyphTable.flgs == null) {
            glyphTable.flgs = new byte[fontDefAttr.getNumGlyphs()];
        }
        Arrays.fill(glyphTable.flgs, (byte) 0);
        glyphTable.flgs[0] = 1;
    }

    @Override
    public void dispose() {
        logger.trace("HPdfTrueTypeFontDef#dispose");
        
        fontDefAttr.dispose();
    }
    
    public static HPdfTrueTypeFontDef load(HPdfReadStream stream, boolean embedding) {
        logger.trace("HPdfTrueTypeFontDef.load");
        
        HPdfTrueTypeFontDef fontdef = new HPdfTrueTypeFontDef();
        fontdef.loadFontData(stream, embedding, 0);
        return fontdef;
    }
    
    public static HPdfTrueTypeFontDef load2(HPdfReadStream stream, int index, boolean embedding) {
        logger.trace("HPdfTrueTypeFontDef.load2");
        
        HPdfTrueTypeFontDef fontdef = new HPdfTrueTypeFontDef();
        fontdef.loadFontData2(stream, index, embedding);
        return fontdef;
    }
    
    private final void dumpTable() {
        for (final String tag : REQUIRED_TAGS) {
            HPdfTTFTable tbl = findTable(tag);
            
            if (tbl == null) {
                logger.trace(" ERR: cannot seek " + tag);
                return;
            }
            
            String fname = tag + ".dat";
            logger.trace(" " + fname + " open");
            if (tag.equals("OS/2")) {
                fname = fname.replace('/', '_');
            }
            
            final HPdfReadStream attrStream = fontDefAttr.getStream();
            HPdfFileWriteStream stream = null;
            try {
                stream = new HPdfFileWriteStream(fname);
                
                attrStream.seek(tbl.offset, SeekMode.SET);
                
                long tblLen = tbl.length;
                for (;;) {
                    byte[] buf = new byte[HPdfConst.HPDF_STREAM_BUF_SIZ];
                    int len = buf.length;
                    if (len > tblLen) { 
                        len = (int) tblLen;
                    }
                    
                    len = attrStream.read(buf, len);
                    if (len <= 0) {
                        break;
                    }
                    
                    stream.write(buf, len);
                    
                    tblLen -= len;
                    if (tblLen <= 0) {
                        break;
                    }
                }
                
            } finally {
                if (stream != null) {
                    stream.close();
                }
            }
            
        }
    }

    private final void loadFontData(HPdfReadStream stream, boolean embedding, long offset) {
        logger.trace("HPdfTrueTypeFontDef#loadFontData");
        
        fontDefAttr.setStream(stream);
        fontDefAttr.setEmbedding(embedding);
        
        stream.seek(offset, SeekMode.SET);
        
        loadTTFTable();
        
        if (IS_DUMP_FONTDATA) {
            dumpTable();
        }
        
        parseHead();
        parseMaxp();
        parseHhea();
        parseCMap();
        parseHmtx();
        parseLoca();
        parseName();
        parseOS2();
        
        HPdfTTFTable tbl = findTable("glyf");
        if (tbl == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_MISSING_TABLE, 4);
        }
        
        fontDefAttr.getGlyphTable().baseOffset = tbl.offset;
        this.setCapHeight((int) this.getCharBBox('H').getTop());
        this.setXHeight((int) this.getCharBBox('x').getTop());
        this.setMissingWidth(
                fontDefAttr.getHMetric()[0].advanceWidth * 1000 /
                fontDefAttr.getHeader().unitsPerEm);
        
        if (logger.isTraceEnabled()) {
            logger.trace(String.format(" getCapHeight()=%d", this.getCapHeight()));
            logger.trace(String.format(" getXHeight()=%d", this.getXHeight()));
            logger.trace(String.format(" getMissingWidth()=%d", this.getMissingWidth()));
        }
        
        if (!embedding) {
            fontDefAttr.setStream(null);
            stream.close();
        }
    }

    private final void loadFontData2(HPdfReadStream stream, int index, boolean embedding) {
        logger.trace("HPdfTrueTypeFontDef#loadFontData2");
        
        fontDefAttr.setStream(stream);
        fontDefAttr.setEmbedding(embedding);
        
        stream.seek(0, SeekMode.SET);
        
        byte[] tag = new byte[4];
        int size = tag.length;
        size = stream.read(tag, size);
        
        if (ByteUtil.indexOf(tag, "ttcf", 4) != 0) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_TTC_FILE, 0);
        }
        
        stream.seek(8, SeekMode.SET);
        
        long numFonts = getUint32(stream);
        if (logger.isTraceEnabled()) {
            logger.trace(String.format("HPdfTrueTypeFontDef#loadFontData2 num_fonts=%d", numFonts));
        }
        
        if (index >= numFonts) {
            throw new HPdfException(HPdfErrorCode.HPDF_INVALID_TTC_INDEX, 0);
        }
        
        /* read offset table for target font and set stream positioning to offset
         * value.
         */
        stream.seek(12 + index * 4, SeekMode.SET);
        long offset = getUint32(stream);
        loadFontData(stream, embedding, offset);
    }

    private HPdfRect getCharBBox(char unicode) {
        int gid = this.getGlyphId(unicode);
        HPdfRect bbox = new HPdfRect(0, 0, 0, 0);
        
        if (gid == 0) {
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(" GetCharHeight cannot get gid char=0x%04x", (int) unicode));
            }
            return bbox;
        }
        
        int m;
        if (fontDefAttr.getHeader().indexToLocFormat == 0) {
            m = 2;
        } else {
            m = 1;
        }
        
        fontDefAttr.getStream().seek(fontDefAttr.getGlyphTable().baseOffset +
                fontDefAttr.getGlyphTable().offsets[gid] * m + 2, SeekMode.SET);
        
        int i = getInt16(fontDefAttr.getStream());
        bbox.setLeft(i * 1000 / fontDefAttr.getHeader().unitsPerEm);
        
        i = getInt16(fontDefAttr.getStream());
        bbox.setBottom(i * 1000 / fontDefAttr.getHeader().unitsPerEm);
        
        i = getInt16(fontDefAttr.getStream());
        bbox.setRight(i * 1000 / fontDefAttr.getHeader().unitsPerEm);
        
        i = getInt16(fontDefAttr.getStream());
        bbox.setRight(i * 1000 / fontDefAttr.getHeader().unitsPerEm);
        
        if (logger.isTraceEnabled()) {
            logger.trace(String.format(
                    " PdfTTFontDef_GetCharBBox char=0x%04X, box=[%f,%f,%f,%f]",
                    (int) unicode, bbox.getLeft(), bbox.getBottom(), bbox.getRight(),
                    bbox.getTop()));
        }
        
        return bbox;
    }
    
    // for binary conversion
    private byte[] binaryBuf = new byte[8];
    private ByteBuffer binaryWrapper = ByteBuffer.wrap(binaryBuf);
    
    private long getUint32(HPdfReadStream stream) {
        stream.read(binaryBuf, 4);
        return 0xFFFFFFFFL & binaryWrapper.getInt(0);
    }
    
    private int getUint16(HPdfReadStream stream) {
        return (0xFFFF & getInt16(stream));
    }
    
    private short getInt16(HPdfReadStream stream) {
        stream.read(binaryBuf, 2);
        return binaryWrapper.getShort(0);
    }
    
    private void writeUint32(HPdfWriteStream stream, long val) {
        binaryWrapper.putLong(0, val);
        stream.write(binaryBuf, 4, 4);
    }
    
    private void writeUint16(HPdfWriteStream stream, int val) {
        binaryWrapper.putInt(0, val);
        stream.write(binaryBuf, 2, 2);
    }
    
    private void writeInt16(HPdfWriteStream stream, short val) {
        binaryWrapper.putShort(0, val);
        stream.write(binaryBuf, 2);
    }
    
    private void loadTTFTable() {
        logger.trace("HPdfTrueTypeFontDef#loadTTFTable");
        
        HPdfTTFOffsetTbl offsetTable = fontDefAttr.getOffsetTable();
        offsetTable.sfntVersion = (int) getUint32(fontDefAttr.getStream());
        offsetTable.numTables = getUint16(fontDefAttr.getStream());
        offsetTable.searchRange = getUint16(fontDefAttr.getStream());
        offsetTable.entrySelector = getUint16(fontDefAttr.getStream());
        offsetTable.rangeShift = getUint16(fontDefAttr.getStream());
        
        if (offsetTable.numTables * HPdfTTFTable.STRUCT_SIZE >
                HPDF_TTF_MAX_MEM_SIZ) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_INVALID_FOMAT, 0);
        }
        offsetTable.createTTFTables(offsetTable.numTables);
        
        int i = 0;
        for (HPdfTTFTable tbl : offsetTable.table) {
            offsetTable.table[i] = tbl;
            fontDefAttr.getStream().read(tbl.tag, tbl.tag.length);
            tbl.checkSum = getUint32(fontDefAttr.getStream());
            tbl.offset = getUint32(fontDefAttr.getStream());
            tbl.length = getUint32(fontDefAttr.getStream());
            
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " [%d] tag=[%c%c%c%c] check_sum=%d offset=%d length=%d",
                        i, tbl.tag[0], tbl.tag[1], tbl.tag[2], tbl.tag[3],
                        tbl.checkSum, tbl.offset,
                        tbl.length));
            }
            ++i;
        }
    }
    
    private void parseHead() {
        logger.trace("HPdfTrueTypeFontDef#parseHead");
        
        HPdfTTFTable tbl = this.findTable("head");
        if (tbl == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_MISSING_TABLE, 5);
        }
        
        final HPdfReadStream attrStream = fontDefAttr.getStream();
        attrStream.seek(tbl.offset, SeekMode.SET);
        
        HPdfTTFFontHeader header = fontDefAttr.getHeader();
        attrStream.read(header.versionNumber, 4);
        header.fontRevision = getUint32(attrStream);
        header.chekSumAdjustment = getUint32(attrStream);
        header.magicNumber = getUint32(attrStream);
        header.flags = getUint16(attrStream);
        header.unitsPerEm = getUint16(attrStream);
        
        attrStream.read(header.created, 8);
        attrStream.read(header.modified, 8);
        
        header.xMin = getInt16(attrStream);
        header.yMin = getInt16(attrStream);
        header.xMax = getInt16(attrStream);
        header.yMax = getInt16(attrStream);
        header.macStyle = getUint16(attrStream);
        header.lowestRecPpem = getUint16(attrStream);
        header.fontDirectionHint = getInt16(attrStream);
        header.indexToLocFormat = getInt16(attrStream);
        header.glyphDataFormat = getInt16(attrStream);
        
        this.getFontBbox().setLeft(
                header.xMin * 1000 / header.unitsPerEm);
        this.getFontBbox().setBottom(
                header.yMin * 1000 / header.unitsPerEm);
        this.getFontBbox().setRight(
                header.xMax * 1000 / header.unitsPerEm);
        this.getFontBbox().setTop(
                header.yMax * 1000 / header.unitsPerEm);
    }
    
    private void parseMaxp() {
        logger.trace("HPdfTrueTypeFontDef#parseMaxp");
        
        HPdfTTFTable tbl = this.findTable("maxp");
        if (tbl == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_MISSING_TABLE, 9);
        }
        
        fontDefAttr.getStream().seek(tbl.offset + 4, SeekMode.SET);
        
        fontDefAttr.setNumGlyphs(getUint16(fontDefAttr.getStream()));
        
        if (logger.isTraceEnabled()) {
            logger.trace(String.format(" HPDF_TTFontDef_ParseMaxp num_glyphs=%d",
                    fontDefAttr.getNumGlyphs()));
        }
    }
    
    private void parseHhea() {
        logger.trace("HPdfTrueTypeFontDef#parseHhea");
        
        HPdfTTFTable tbl = this.findTable("hhea");
        if (tbl == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_MISSING_TABLE, 6);
        }
        
        final HPdfReadStream attrStream = fontDefAttr.getStream();
        attrStream.seek(tbl.offset + 4, SeekMode.SET);
        
        int temp = getInt16(attrStream);
        this.setAscent(temp * 1000 / fontDefAttr.getHeader().unitsPerEm);
        temp = getInt16(attrStream);
        this.setDescent(temp * 1000 / fontDefAttr.getHeader().unitsPerEm);
        
        attrStream.seek(tbl.offset + 34, SeekMode.SET);
        
        fontDefAttr.setNumHMetric(getUint16(attrStream));
        
        if (logger.isTraceEnabled()) {
            logger.trace(String.format(" HPDF_TTFontDef_ParseHhea num_h_metric=%d",
                    fontDefAttr.getNumHMetric()));
        }
    }
    
    private void parseCMap() {
        logger.trace("HPdfTrueTypeFontDef#parseCMap");
        
        HPdfTTFTable tbl = this.findTable("cmap");
        if (tbl == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_MISSING_TABLE, 1);
        }
        
        final HPdfReadStream attrStream = fontDefAttr.getStream();
        attrStream.seek(tbl.offset, SeekMode.SET);
        
        int version = getUint16(attrStream);
        
        if (version != 0) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_INVALID_FOMAT, 0);
        }
        
        int numCmap = getUint16(attrStream);
        
        long msUnicodeEncodingOffset = 0;
        long byteEncodingOffset = 0;
        for (int i = 0; i < numCmap; ++i) {
            int platformID = getUint16(attrStream);
            int encodingID = getUint16(attrStream);
            long offset = getUint32(attrStream);
            
            long saveOffset = attrStream.tell();
            // MEMO: original code checks if saveOffset < 0.
            
            attrStream.seek(tbl.offset + offset, SeekMode.SET);
            
            int format = getUint16(attrStream);
            
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " HPDF_TTFontDef_ParseCMap tables[%d] platformID=%d " +
                        "encodingID=%d format=%d offset=%d",
                        i, platformID,
                        encodingID, format, offset));
            }
            
            /* MS-Unicode-CMAP is used for priority */
            if (platformID == 3 && encodingID == 1 && format == 4) {
                msUnicodeEncodingOffset = offset;
                break;
            }
            /* Byte-Encoding-CMAP will be used if MS-Unicode-CMAP is not found */
            if (platformID == 1 && encodingID == 0 && format == 1) {
                byteEncodingOffset = offset;
            }
            
            attrStream.seek(saveOffset, SeekMode.SET);
        }
        
        if (msUnicodeEncodingOffset != 0) {
            logger.trace(" found microsoft unicode cmap.");
            parseCMapFormat4(msUnicodeEncodingOffset + tbl.offset);
        } else if (byteEncodingOffset != 0) {
            logger.trace(" found byte encoding cmap.");
            parseCMapFormat0(byteEncodingOffset + tbl.offset);
        } else {
            logger.trace(" cannot found target cmap.");
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_INVALID_FOMAT, 0);
        }
        
    }
    
    private void parseCMapFormat0(long offset) {
        logger.trace(" parseCMapFormat0");
        
        final HPdfReadStream attrStream = fontDefAttr.getStream();
        attrStream.seek(offset, SeekMode.SET);
        
        HPdfTTFCMapRange cmap = fontDefAttr.getCmap();
        cmap.format = getUint16(attrStream);
        cmap.length = getUint16(attrStream);
        cmap.language = getUint16(attrStream);
        
        if (cmap.format != 0) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_INVALID_FOMAT, 0);
        }
        
        byte[] array = new byte[256];
        attrStream.read(array, array.length);
        
        cmap.glyphIdArray = new int[256];
        // MEMO: original code is below.
        // it copies value between same elements... what means?
        // probably required code is coping array to cmap.glyph_id_array...
        /*
        HPDF_UINT16 *parray;
        ......
        parray = attr->cmap.glyph_id_array;
        for (i = 0; i < 256; i++) {
            *parray = attr->cmap.glyph_id_array[i];
            HPDF_PTRACE((" ParseCMAP_format0 glyph_id_array[%d]=%d\n",
                        i, *parray));
            parray++;
        }
        */
        for (int i = 0; i < array.length; ++i) {
            cmap.glyphIdArray[i] = (0xFF & array[i]);
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " ParseCMAP_format0 glyph_id_array[%d]=%d",
                        i, cmap.glyphIdArray[i]));
            }
        }
    }
    
    private void parseCMapFormat4(long offset) {
        logger.trace(" parseCMapFormat4");
        
        final HPdfReadStream attrStream = fontDefAttr.getStream();
        attrStream.seek(offset, SeekMode.SET);
        
        HPdfTTFCMapRange cmap = fontDefAttr.getCmap();
        cmap.format = getUint16(attrStream);
        cmap.length = getUint16(attrStream);
        cmap.language = getUint16(attrStream);
        
        if (cmap.format != 4) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_INVALID_FOMAT, 0);
        }
        
        cmap.segCountX2 = getUint16(attrStream);
        cmap.searchRange = getUint16(attrStream);
        cmap.entrySelector = getUint16(attrStream);
        cmap.rangeShift = getUint16(attrStream);
        
        /* end_count */
        cmap.endCount = new int[cmap.segCountX2 / 2];
        for (int i = 0; i < cmap.endCount.length; ++i) {
            cmap.endCount[i] = getUint16(attrStream);
        }
        cmap.reservedPad = getUint16(attrStream);
        
        /* start_count */
        cmap.startCount = new int[cmap.segCountX2 / 2];
        for (int i = 0; i < cmap.startCount.length; ++i) {
            cmap.startCount[i] = getUint16(attrStream);
        }
        
        /* id_delta */
        cmap.idDelta = new int[cmap.segCountX2 / 2];
        for (int i = 0; i < cmap.idDelta.length; ++i) {
            cmap.idDelta[i] = getInt16(attrStream);
        }
        
        /* id_range_offset */
        cmap.idRangeOffset = new int[cmap.segCountX2 / 2];
        for (int i = 0; i < cmap.idRangeOffset.length; ++i) {
            cmap.idRangeOffset[i] = getUint16(attrStream);
        }
        
        long numRead = attrStream.tell() - offset;
        // MEMO: original code checks if numRead < 0.
        
        int glyphIdArrayCount = (int) ((cmap.length - numRead) / 2);
        if (glyphIdArrayCount > 0) {
            /* glyph_id_array */
            cmap.glyphIdArray = new int[glyphIdArrayCount];
            for (int i = 0; i < cmap.glyphIdArray.length; ++i) {
                cmap.glyphIdArray[i] = getUint16(attrStream);
            }
        } else {
            cmap.glyphIdArray = new int[0];
        }
        
        // MEMO: original code #ifdef LIBHPDF_DEBUG
        if (IS_DUMP_FONTDATA) {
            /* print all elements of cmap table */
            if (logger.isTraceEnabled()) {
                for (int i = 0; i < (cmap.segCountX2 / 2); ++i){
                    logger.trace(String.format(
                            " ParseCMAP_format4[%d] start_count=0x%04X, " +
                            "end_count=0x%04X, id_delta=%d, id_range_offset=%d", i,
                            cmap.startCount[i], cmap.endCount[i],
                            cmap.idDelta[i], cmap.idRangeOffset[i]));
                }
            }
        }
    }
    
    private int getGlyphId(char unicode) {
        logger.trace("HPdfTrueTypeFontDef#getGlyphId");
        final HPdfTTFCMapRange cmap = fontDefAttr.getCmap();
        int segCount = cmap.segCountX2 / 2;
        
        /* format 0 */
        if (cmap.format == 0) {
            unicode &= 0xFF;
            return cmap.glyphIdArray[unicode];
        }
        
        /* format 4 */
        if (cmap.segCountX2 == 0) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_INVALID_CMAP, 0);
        }
        
        int i;
        for (i = 0; i < segCount; ++i) {
            if (unicode <= cmap.endCount[i]) {
                break;
            }
        }
        
        if (cmap.startCount[i] > unicode) {
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " HPDF_TTFontDef_GetGlyphid undefined char(0x%04X)",
                        unicode));
            }
            return 0;
        }
        
        if (cmap.idRangeOffset[i] == 0) {
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " HPDF_TTFontDef_GetGlyphid  idx=%d code=%d ret=%d",
                        i, (int) unicode, unicode + cmap.idDelta[i]));
            }
            return unicode + cmap.idDelta[i];
        } else {
            int idx = cmap.idRangeOffset[i] / 2 +
                    (unicode - cmap.startCount[i]) - (segCount - i);
            if (idx > cmap.getGlyphIdArrayCount()) {
                if (logger.isTraceEnabled()) {
                    logger.trace(String.format(
                            " HPDF_TTFontDef_GetGlyphid[%d] %d > %d",
                            i, idx, cmap.getGlyphIdArrayCount()));
                }
                return 0;
            } else {
                int gid = cmap.glyphIdArray[idx] +
                        cmap.idDelta[i];
                if (logger.isTraceEnabled()) {
                    logger.trace(String.format(
                            " HPDF_TTFontDef_GetGlyphid idx=%d unicode=0x%04X id=%d",
                            idx, (int) unicode, gid));
                }
                return gid;
            }
        }
    }

    @Override
    public int getWidth(char code) {
        logger.trace("HPdfTrueTypeFontDef#getWidth");
        // MEMO: original code is HPDF_TTFontDef_GetCharWidth() in hpdf_fontdef_tt.c
        int gid = getGlyphId(code);
        
        if (gid >= fontDefAttr.getNumGlyphs()) {
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " HPDF_TTFontDef_GetCharWidth WARNING gid > num_glyphs %d > %d",
                        gid, fontDefAttr.getNumGlyphs()));
            }
            return this.getMissingWidth();
        }
        
        HPdfTTFLongHorMetric hmetrics = fontDefAttr.getHMetric()[gid];
        if (fontDefAttr.getGlyphTable().flgs[gid] == 0) {
            fontDefAttr.getGlyphTable().flgs[gid] = 1;
            if (fontDefAttr.isEmbedding()) {
                checkCompositGlyph(gid);
            }
        }
        int advanceWidth = hmetrics.advanceWidth * 1000 /
                fontDefAttr.getHeader().unitsPerEm;
        
        return advanceWidth;
    }
    
    private void checkCompositGlyph(int gid) {
        logger.trace(" checkCompositGlyph");
        
        long offset = fontDefAttr.getGlyphTable().offsets[gid];
        if (fontDefAttr.getHeader().indexToLocFormat == 0) {
            offset *= 2;
        }
        
        offset += fontDefAttr.getGlyphTable().baseOffset;
        
        final HPdfReadStream attrStream = fontDefAttr.getStream();
        attrStream.seek(offset, SeekMode.SET);
        
        int numOfContours = getInt16(attrStream);
        if (numOfContours != -1) {
            return;
        }
        if (logger.isTraceEnabled()) {
            logger.trace(String.format(
                    " CheckCompositGryph composit font gid=%d", gid));
        }
        
        int flags;
        final int ARG_1_AND_2_ARE_WORDS = 1;
        final int WE_HAVE_A_SCALE  = 8;
        final int MORE_COMPONENTS = 32;
        final int WE_HAVE_AN_X_AND_Y_SCALE = 64;
        final int WE_HAVE_A_TWO_BY_TWO = 128;
        
        attrStream.seek(8, SeekMode.CUR);
        do {
            flags = getInt16(attrStream);
            int glyphIndex = getInt16(attrStream);
            if ((flags & ARG_1_AND_2_ARE_WORDS) != 0) {
                attrStream.seek(4, SeekMode.CUR);
            } else {
                attrStream.seek(2, SeekMode.CUR);
            }
            
            if ((flags & WE_HAVE_A_SCALE) != 0) {
                attrStream.seek(2, SeekMode.CUR);
            } else if ((flags & WE_HAVE_AN_X_AND_Y_SCALE) != 0) {
                attrStream.seek(4, SeekMode.CUR);
            } else if ((flags & WE_HAVE_A_TWO_BY_TWO) != 0) {
                attrStream.seek(8, SeekMode.CUR);
            }
            
            if (glyphIndex > 0 && glyphIndex < fontDefAttr.getNumGlyphs()) {
                fontDefAttr.getGlyphTable().flgs[glyphIndex] = 1;
            }
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " gid=%d, num_of_contours=%d, flags=%d, glyph_index=%d",
                        gid, numOfContours, flags, glyphIndex));
            }
        } while ((flags & MORE_COMPONENTS) != 0);
    }

    private int getGidWidth(int gid) {
        logger.trace("HPdfTrueTypeFontDef#getGidWidth");
        
        if (gid >= fontDefAttr.getNumGlyphs()) {
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " HPDF_TTFontDef_GetGidWidth WARNING gid > num_glyphs %d > %d",
                        gid, fontDefAttr.getNumGlyphs()));
            }
            return this.getMissingWidth();
        }
        HPdfTTFLongHorMetric hmetrics = fontDefAttr.hMetric[gid];
        int advanceWidth = hmetrics.advanceWidth * 1000 / 
                fontDefAttr.getHeader().unitsPerEm;
        if (logger.isTraceEnabled()) {
            logger.trace(String.format(
                    " HPDF_TTFontDef_GetGidWidth gid=%d, width=%d",
                    gid, advanceWidth));
        }
        return advanceWidth;
    }

    private void parseHmtx() {
        logger.trace("HPdfTrueTypeFontDef#parseHmtx");
        
        HPdfTTFTable tbl = this.findTable("hmtx");
        if (tbl == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_MISSING_TABLE, 7);
        }
        
        final HPdfReadStream attrStream = fontDefAttr.getStream();
        attrStream.seek(tbl.offset, SeekMode.SET);
        
        /* allocate memory for a table of holizontal matrix.
         * the count of metric records is same as the number of glyphs
         */
        fontDefAttr.createHMetrics(fontDefAttr.getNumGlyphs());
        
        int saveAw = 0;
        int i;
        for (i = 0; i < fontDefAttr.getNumHMetric(); ++i) {
            HPdfTTFLongHorMetric metric = fontDefAttr.getHMetric()[i];
            metric.advanceWidth = getUint16(attrStream);
            metric.lsb = getInt16(attrStream);
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " ParseHmtx metric[%d] aw=%d lsb=%d", i,
                        metric.advanceWidth, metric.lsb));
            }
            saveAw = metric.advanceWidth;
        }
        
        /* pad the advance_width of remaining metrics with the value of last metric */
        while (i < fontDefAttr.getNumGlyphs()) {
            HPdfTTFLongHorMetric metric = fontDefAttr.getHMetric()[i];
            metric.advanceWidth = saveAw;
            metric.lsb = getInt16(attrStream);
            ++i;
        }
    }
    
    private void parseLoca() {
        logger.trace("HPdfTrueTypeFontDef#parseLoca");
        
        HPdfTTFTable tbl = this.findTable("loca");
        if (tbl == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_MISSING_TABLE, 8);
        }
        
        final HPdfReadStream attrStream = fontDefAttr.getStream();
        attrStream.seek(tbl.offset, SeekMode.SET);
        
        final HPdfTTFGlyphOffsets glyphTable = fontDefAttr.getGlyphTable();
        /* allocate glyph-offset-table. */
        glyphTable.offsets =
                new long[fontDefAttr.getNumGlyphs() + 1];
        
        /* allocate glyph-flg-table.
         * this flgs are used to judge whether glyphs should be embedded.
         */
        glyphTable.flgs =
                new byte[fontDefAttr.getNumGlyphs()];
        glyphTable.flgs[0] = 1;
        
        if (fontDefAttr.getHeader().indexToLocFormat == 0) {
            /* short version */
            for (int i = 0; i <= fontDefAttr.getNumGlyphs(); ++i) {
                glyphTable.offsets[i] = getUint16(attrStream);
            }
        } else {
            /* long version */
            for (int i = 0; i <= fontDefAttr.getNumGlyphs(); ++i) {
                glyphTable.offsets[i] = getUint32(attrStream);
            }
        }
        
        // MEMO: original code is #ifdef LIBHPDF_DEBUG
        if (IS_DUMP_FONTDATA) {
            if (logger.isTraceEnabled()) {
                for (int i = 0; i <= fontDefAttr.getNumGlyphs(); ++i) {
                    logger.trace(String.format(
                            " ParseLOCA offset[%d]=%d", i, glyphTable.offsets[i]));
                }
            }
        }
    }
    
    private String loadUnicodeName(HPdfReadStream stream, long offset, int len) {
        byte[] tmp = new byte[len];
        
        stream.seek(offset, SeekMode.SET);
        len = stream.read(tmp, len);
        return new String(tmp, 0, len, StandardCharsets.UTF_16BE);
    }
    
    private void parseName() {
        logger.trace("HPdfTrueTypeFontDef#parseName");
        
        HPdfTTFTable tbl = this.findTable("name");
        if (tbl == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_MISSING_TABLE, 10);
        }
        
        final HPdfReadStream attrStream = fontDefAttr.getStream();
        attrStream.seek(tbl.offset, SeekMode.SET);
        
        final HPdfTTFNamingTable nameTbl = fontDefAttr.getNameTable();
        nameTbl.format = getUint16(attrStream);
        nameTbl.count = getUint16(attrStream);
        nameTbl.stringOffset = getUint16(attrStream);
        
        if (logger.isTraceEnabled()) {
            logger.trace(String.format(
                    " ParseName() format=%d, count=%d, string_offset=%d",
                    nameTbl.format, nameTbl.count, nameTbl.stringOffset));
        }
        
        nameTbl.createNameRecords(nameTbl.count);
        
        long offsetId1 = 0;
        long offsetId2 = 0;
        long offsetId1u = 0;
        long offsetId2u = 0;
        long lenId1 = 0;
        long lenId2 = 0;
        long lenId1u = 0;
        long lenId2u = 0;
        for (final HPdfTTFNameRecord nameRec : nameTbl.nameRecords) {
            nameRec.platformId = getUint16(attrStream);
            nameRec.encodingId = getUint16(attrStream);
            nameRec.languageId = getUint16(attrStream);
            nameRec.nameId = getUint16(attrStream);
            nameRec.length = getUint16(attrStream);
            nameRec.offset = getUint16(attrStream);
            
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " ParseName() platformID=%d, encodingID=%d, nameID=%d",
                        nameRec.platformId, nameRec.encodingId, nameRec.nameId));
            }
            
            if (nameRec.platformId == 1 && nameRec.encodingId == 0 &&
                    nameRec.nameId == 6) {
                offsetId1 = tbl.offset + nameRec.offset +
                        nameTbl.stringOffset;
                lenId1 = nameRec.length;
            }
            
            if (nameRec.platformId == 1 && nameRec.encodingId == 0 &&
                    nameRec.nameId == 2) {
                offsetId2 = tbl.offset + nameRec.offset +
                        nameTbl.stringOffset;
                lenId2 = nameRec.length;
            }
            
            if (nameRec.platformId == 3 && nameRec.encodingId == 1 &&
                    nameRec.nameId == 6 && nameRec.languageId == 0x0409) {
                offsetId1u = tbl.offset + nameRec.offset +
                        nameTbl.stringOffset;
                lenId1u = nameRec.length;
            }
            
            if (nameRec.platformId == 3 && nameRec.encodingId == 1 &&
                    nameRec.nameId == 2 && nameRec.languageId == 0x0409) {
                offsetId2u = tbl.offset + nameRec.offset +
                        nameTbl.stringOffset;
                lenId2u = nameRec.length;
            }
        }
        
        if ((offsetId1 == 0 && offsetId1u == 0) ||
                (offsetId2 == 0 && offsetId2u == 0)) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_INVALID_FOMAT, 0);
        }
        
        if (lenId1 == 0 && lenId1u > 0) {
            lenId1 = lenId1u / 2 + lenId1u % 2;
        }
        
        if (lenId2 == 0 && lenId2u > 0) {
            lenId2 = lenId2u / 2 + lenId2u % 2;
        }
        
        if (lenId1 + lenId2 + 8 > 127) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_INVALID_FOMAT, 0);
        }
        
        if (offsetId1 != 0) {
            byte[] buf = new byte[(int) lenId1];
            attrStream.seek(offsetId1, SeekMode.SET);
            attrStream.read(buf, buf.length);
            fontDefAttr.setBaseFont(
                    new String(buf, StandardCharsets.ISO_8859_1));
        } else {
            fontDefAttr.setBaseFont(
                    loadUnicodeName(attrStream, offsetId1u, (int) lenId1u));
        }
        
        String subfamilyName;
        if (offsetId2 != 0) {
            byte[] buf = new byte[(int) lenId2];
            attrStream.seek(offsetId2, SeekMode.SET);
            attrStream.read(buf, buf.length);
            subfamilyName = new String(buf, StandardCharsets.ISO_8859_1);
        } else {
            subfamilyName = loadUnicodeName(attrStream, offsetId2u, (int) lenId2u);
        }
        
        /*
         * get "postscript name" of from a "name" table as BaseName.
         * if subfamily name is not "Regular", add subfamily name to BaseName.
         * if subfamily name includes the blank character, remove it.
         * if subfamily name is "Bold" or "Italic" or "BoldItalic", set flags
         * attribute.
         */
        if (!subfamilyName.startsWith("Regular")) {
            StringBuilder appendName = new StringBuilder();
            appendName.append(fontDefAttr.getBaseFont());
            appendName.append(',');
            
            appendName.append(subfamilyName.replace(" ", ""));
            if (appendName.length() > HPdfConst.HPDF_LIMIT_MAX_NAME_LEN) {
                appendName.setLength(HPdfConst.HPDF_LIMIT_MAX_NAME_LEN);
            }
            fontDefAttr.setBaseFont(appendName.toString());
            
            if (subfamilyName.indexOf("Bold") > 0) {
                this.setFlag(HPDF_FONT_FOURCE_BOLD);
            }
            if (subfamilyName.indexOf("Italic") > 0) {
                this.setFlag(HPDF_FONT_ITALIC);
            }
        }
        this.setBaseFont(fontDefAttr.getBaseFont());
        
        if (logger.isTraceEnabled()) {
            logger.trace(" ParseName() base_font=" + fontDefAttr.getBaseFont());
        }
    }
    
    private void parseOS2() {
        logger.trace("HPdfTrueTypeFontDef#parseOS2");
        
        HPdfTTFTable tbl = this.findTable("OS/2");
        if (tbl == null) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_MISSING_TABLE, 0);
        }
        
        final HPdfReadStream attrStream = fontDefAttr.getStream();
        /* get the number version. */
        attrStream.seek(tbl.offset, SeekMode.SET);
        
        int version = getUint16(attrStream);
        
        /* check whether the font is allowed to be embedded. */
        attrStream.seek(tbl.offset + 8, SeekMode.SET);
        
        fontDefAttr.setFsType(getUint16(attrStream));
        
        if ((fontDefAttr.getFsType() & (0x0002 | 0x0100 | 0x0200)) != 0
                && fontDefAttr.isEmbedding()) {
            throw new HPdfException(HPdfErrorCode.HPDF_TTF_CANNOT_EMBEDDING_FONT, 0);
        }
        
        /* get fields sfamilyclass and panose. */
        attrStream.seek(tbl.offset + 30, SeekMode.SET);
        
        byte[] sFamilyClass = fontDefAttr.getSFamilyClass();
        attrStream.read(sFamilyClass, 2);
        
        byte[] panose = fontDefAttr.getPanose();
        attrStream.read(panose, 10);
        
        if (logger.isTraceEnabled()) {
            logger.trace(String.format(
                    " ParseOS2 sFamilyClass=%d-%d " +
                    "Panose=%02X-%02X-%02X-%02X-%02X-%02X-%02X-%02X-%02X-%02X",
                    sFamilyClass[0], sFamilyClass[1],
                    panose[0], panose[1], panose[2], panose[3],
                    panose[4], panose[5], panose[6], panose[7],
                    panose[8], panose[9]));
        }
        /* Class ID = 1   Oldstyle Serifs
           Class ID = 2   Transitional Serifs
           Class ID = 3   Modern Serifs
           Class ID = 4   Clarendon Serifs
           Class ID = 5   Slab Serifs
           Class ID = 6   (reserved for future use)
           Class ID = 7   Freeform Serifs
           Class ID = 8   Sans Serif
           Class ID = 9   Ornamentals
           Class ID = 10  Scripts
           Class ID = 11  (reserved for future use)
           Class ID = 12  Symbolic */
        if ((sFamilyClass[0] > 0 && sFamilyClass[0] < 6)
                || (sFamilyClass[0] == 7)) {
            this.setFlag(HPDF_FONT_SERIF);
        }
        if (sFamilyClass[0] == 10) {
            this.setFlag(HPDF_FONT_SCRIPT);
        }
        if (sFamilyClass[0] == 12) {
            this.setFlag(HPDF_FONT_SYMBOLIC);
        }
        
        /* get fields ulCodePageRange1 and ulCodePageRange2 */
        if (version > 0) {
            attrStream.seek(36, SeekMode.CUR);
            fontDefAttr.setCodePageRange1(getUint32(attrStream));
            fontDefAttr.setCodePageRange2(getUint32(attrStream));
        }
        if (logger.isTraceEnabled()) {
            logger.trace(String.format(
                    " ParseOS2 CodePageRange1=%08X CodePageRange2=%08X",
                    fontDefAttr.getCodePageRange1(),
                    fontDefAttr.getCodePageRange2()));
        }
    }
    
    private void recreateGLYF(long[] newOffsets, HPdfWriteStream stream) {
        logger.trace(" recreateGLYF");
        
        long saveOffset = 0;
        long startOffset = stream.getSize();
        
        final HPdfTTFGlyphOffsets glyphTable = fontDefAttr.getGlyphTable();
        for (int i = 0; i < fontDefAttr.getNumGlyphs(); ++i) {
            if (glyphTable.flgs[i] == 1) {
                long offset = glyphTable.offsets[i];
                long len = glyphTable.offsets[i + 1] - offset;
                
                newOffsets[i] = stream.getSize() - startOffset;
                if (fontDefAttr.getHeader().indexToLocFormat == 0) {
                    newOffsets[i] /= 2;
                    len *= 2;
                }
                if (logger.isTraceEnabled()) {
                    logger.trace(String.format(
                            " RecreateGLYF[%d] move from [%d] to [%d]", i,
                            glyphTable.baseOffset + offset,
                            newOffsets[i]));
                }
                if (fontDefAttr.getHeader().indexToLocFormat == 0) {
                    offset *= 2;
                }
                offset += glyphTable.baseOffset;
                
                fontDefAttr.getStream().seek(offset, SeekMode.SET);
                
                byte[] buf = new byte[HPdfConst.HPDF_STREAM_BUF_SIZ];
                while (len > 0) {
                    long tmpLen = (len > HPdfConst.HPDF_STREAM_BUF_SIZ) ?
                            HPdfConst.HPDF_STREAM_BUF_SIZ : len;
                    tmpLen = fontDefAttr.getStream().read(buf, (int) tmpLen);
                    stream.write(buf, (int) tmpLen);
                    len -= tmpLen;
                }
                
                saveOffset = stream.getSize() - startOffset;
                if (fontDefAttr.getHeader().indexToLocFormat == 0) {
                    saveOffset /= 2;
                }
            } else {
                newOffsets[i] = saveOffset;
            }
        }
        
        newOffsets[fontDefAttr.getNumGlyphs()] = saveOffset;
        
        // MEMO: original code is #ifdef DEBUG.
        if (IS_DUMP_FONTDATA) {
            if (logger.isTraceEnabled()) {
                for (int i = 0; i < fontDefAttr.getNumGlyphs(); ++i) {
                    logger.trace(String.format(
                            " RecreateGLYF[%d] offset=%d", i,
                            newOffsets[i]));
                }
            }
        }
    }
    
    private void recreateName(HPdfWriteStream stream) {
        logger.trace(" recreateName");
        
        HPdfTTFTable tbl = this.findTable("name");
        
        HPdfMemStream tmpStream = null;
        try {
            tmpStream = new HPdfMemStream(HPdfConst.HPDF_STREAM_BUF_SIZ);
            
            HPdfTTFNamingTable nameTbl = fontDefAttr.getNameTable();
            writeUint16(stream, nameTbl.format);
            writeUint16(stream, nameTbl.count);
            writeUint16(stream, nameTbl.stringOffset);
            
            for (int i = 0; i < nameTbl.count; ++i) {
                HPdfTTFNameRecord nameRec = nameTbl.nameRecords[i];
                long nameLen = nameRec.length;
                long tmpLen = nameLen;
                long offset = tbl.offset + nameTbl.stringOffset + 
                        nameRec.offset;
                long recOffset = tmpStream.getSize();
                
                /* add suffix to font-name. */
                if (nameRec.nameId == 1 || nameRec.nameId == 4) {
                    if (nameRec.platformId == 0 || nameRec.platformId == 3) {
                        byte[] tmp = fontDefAttr.getTagName2AsBytes();
                        tmpStream.write(tmp, tmp.length);
                        nameLen += tmp.length;
                    } else {
                        byte[] tmp = fontDefAttr.getTagNameAsBytes();
                        tmpStream.write(tmp, tmp.length);
                        nameLen += tmp.length;
                    }
                }
                writeUint16(stream, nameRec.platformId);
                writeUint16(stream, nameRec.encodingId);
                writeUint16(stream, nameRec.languageId);
                writeUint16(stream, nameRec.nameId);
                writeUint16(stream, (int) nameLen);
                writeUint16(stream, (int) recOffset);
                
                fontDefAttr.getStream().seek(offset, SeekMode.SET);
                
                byte[] buf = new byte[HPdfConst.HPDF_STREAM_BUF_SIZ];
                while (tmpLen > 0) {
                    int len = (int) ((tmpLen > HPdfConst.HPDF_STREAM_BUF_SIZ) ?
                            HPdfConst.HPDF_STREAM_BUF_SIZ : tmpLen);
                    
                    len = fontDefAttr.getStream().read(buf, len);
                    tmpStream.write(buf, len);
                    tmpLen -= len;
                }
                if (logger.isTraceEnabled()) {
                    logger.trace(String.format(
                            " RecreateNAME name_rec[%d] platform_id=%d " +
                            "encoding_id=%d language_id=%d name_rec->name_id=%d " +
                            "length=%d offset=%d", i, nameRec.platformId,
                            nameRec.encodingId, nameRec.languageId,
                            nameRec.nameId, nameLen, recOffset));
                }
            }
            tmpStream.writeToStream(stream, HPdfFilterFlag.NONE, null);
        } finally {
            if (tmpStream != null) {
                tmpStream.close();
            }
        }
    }
    
    private long writeHeader(HPdfWriteStream stream) {
        logger.trace(" writeHeader");
        
        HPdfTTFFontHeader header = fontDefAttr.getHeader();
        stream.write(header.versionNumber, 4);
        writeUint32(stream, header.fontRevision);
        
        /* save the address of checkSumAdjustment.
         * the value is rewrite to computed value after new check-sum-value is
         * generated.
         */
        long checkSumPtr = stream.getSize();
        
        writeUint32(stream, 0);
        writeUint32(stream, header.magicNumber);
        writeUint16(stream, header.flags);
        writeUint16(stream, header.unitsPerEm);
        stream.write(header.created, 8);
        stream.write(header.modified, 8);
        writeInt16(stream, header.xMin);
        writeInt16(stream, header.yMin);
        writeInt16(stream, header.xMax);
        writeInt16(stream, header.yMax);
        writeUint16(stream, header.macStyle);
        writeUint16(stream, header.lowestRecPpem);
        writeInt16(stream, header.fontDirectionHint);
        writeInt16(stream, header.indexToLocFormat);
        writeInt16(stream, header.glyphDataFormat);
        
        return checkSumPtr;
    }
    
    void saveFontData(HPdfWriteStream stream) {
        logger.trace(" saveFontData");
        
        HPdfMemStream tmpStream = null;
        
        try {
            tmpStream = new HPdfMemStream(HPdfConst.HPDF_STREAM_BUF_SIZ);
            // MEMO: assist method to emulate 'goto' logic in original code.
            saveFontDataImpl(stream, tmpStream);
        } finally {
            if (tmpStream != null) {
                tmpStream.close();
            }
        }
    }
    
    private void saveFontDataImpl(HPdfWriteStream stream, HPdfMemStream tmpStream) {
        
        final HPdfTTFOffsetTbl offsetTbl = fontDefAttr.getOffsetTable();
        writeUint32(stream, offsetTbl.sfntVersion);
        writeUint16(stream, REQUIRED_TAGS.length);
        writeUint16(stream, offsetTbl.searchRange);
        writeUint16(stream, offsetTbl.entrySelector);
        writeUint16(stream, offsetTbl.rangeShift);
        
        HPdfTTFTable[] tmpTbl = new HPdfTTFTable[REQUIRED_TAGS.length];
        int offsetBase = 12 + 16 * REQUIRED_TAGS.length;
        long[] newOffsets = new long[fontDefAttr.getNumGlyphs() + 1];
        long checkSumPtr = 0;
        long tmpCheckSum = 0xB1B0AFBAL;
        byte[] buf = new byte[8];
        
        for (int i = 0; i < REQUIRED_TAGS.length; ++i) {
            HPdfTTFTable tbl = findTable(REQUIRED_TAGS[i]);
            if (tbl == null) {
                throw new HPdfException(HPdfErrorCode.HPDF_TTF_MISSING_TABLE, i);
            }
            fontDefAttr.getStream().seek(tbl.offset, SeekMode.SET);
            
            long length = tbl.length;
            long newOffset = tmpStream.getSize();
            
            if (ByteUtil.indexOf(tbl.tag, "head", 4) == 0) {
                checkSumPtr = writeHeader(tmpStream);
            } else if (ByteUtil.indexOf(tbl.tag, "glyf", 4) == 0) {
                recreateGLYF(newOffsets, tmpStream);
            } else if (ByteUtil.indexOf(tbl.tag, "loca", 4) == 0) {
                if (fontDefAttr.getHeader().indexToLocFormat == 0) {
                    for (int j = 0; j <= fontDefAttr.getNumGlyphs(); ++j) {
                        writeUint16(tmpStream, (int) newOffsets[j]);
                    }
                } else {
                    for (int j = 0; j <= fontDefAttr.getNumGlyphs(); ++j) {
                        writeUint32(tmpStream, newOffsets[j]);
                    }
                }
            }  else if (ByteUtil.indexOf(tbl.tag, "name", 4) == 0) {
                recreateName(tmpStream);
            } else {
                int size = 4;
                
                while (length > 4) {
                    size = fontDefAttr.getStream().read(buf, 4);
                    tmpStream.write(buf, size);
                    length -= 4;
                }
                
                size = fontDefAttr.getStream().read(buf, (int) length);
                tmpStream.write(buf, size);
            }
            tmpTbl[i].offset = newOffset;
            tmpTbl[i].length = tmpStream.getSize() - newOffset;
        }
        
        /* recalcurate checksum */
        ByteBuffer bufWrapper = ByteBuffer.wrap(buf);
        for (int i = 0; i < REQUIRED_TAGS.length; ++i) {
            HPdfTTFTable tbl = tmpTbl[i];
            long length = tbl.length;
            
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " SaveFontData() tag[%s] length=%d",
                        REQUIRED_TAGS[i], length));
            }
            
            tmpStream.seek(tbl.offset, SeekMode.SET);
            
            tbl.checkSum = 0;
            while (length > 0) {
                long rlen = (length > 4) ? 4 : length;
                bufWrapper.putInt(0, 0);
                rlen = tmpStream.read(buf, (int) rlen);
                tbl.checkSum += (0xFFFFFFFFL & bufWrapper.getInt(0));
                length -= rlen;
            }
            tbl.checkSum &= 0xFFFFFFFFL;
            
            if (logger.isTraceEnabled()) {
                logger.trace(String.format(
                        " SaveFontData() tag[%s] check-sum=%d offset=%d",
                        REQUIRED_TAGS[i], tbl.checkSum,
                        tbl.offset));
            }
            stream.writeAsBytes(REQUIRED_TAGS[i]);
            writeUint32(stream, tbl.checkSum);
            tbl.offset += offsetBase;
            writeUint32(stream, tbl.offset);
            writeUint32(stream, tbl.length);
        }
        
        /* calucurate checkSumAdjustment.*/
        tmpStream.seek(0, SeekMode.SET);
        
        for (;;) {
            // MEMO: zero-clearing against buf not needed?
            // (original code does nothing.)
            int siz = tmpStream.read(buf, 4);
            if (siz <= 0) {
                break;
            }
            tmpCheckSum -= (0xFFFFFFFFL & bufWrapper.getInt(0));
        }
        if (logger.isTraceEnabled()) {
            logger.trace(String.format(
                    " SaveFontData() new checkSumAdjustment=%d",
                    tmpCheckSum));
        }
        tmpStream.seek(checkSumPtr, SeekMode.SET);
        bufWrapper.putLong(0, tmpCheckSum);
        tmpStream.rewrite(buf, 4, 4);
        
        fontDefAttr.setLength1(tmpStream.getSize() + offsetBase);
        tmpStream.writeToStream(stream, HPdfFilterFlag.NONE, null);
    }
    
    private void setTagName(String tag) {
        logger.trace("HPdfTrueTypeFontDef#setTagName");
        
        if (tag.length() != HPDF_TTF_FONT_TAG_LEN) {
            return;
        }
        
        fontDefAttr.setTagName(tag + "+");
        // MEMO: tagName2 is tagname encoded with UTF-16BE
        // merely set same value here.
        fontDefAttr.setTagName2(fontDefAttr.getTagName());
        
        StringBuilder buf = new StringBuilder();
        buf.append(fontDefAttr.getTagName());
        buf.append(this.getBaseFont());
        if (buf.length() > HPdfConst.HPDF_LIMIT_MAX_NAME_LEN) {
            buf.setLength(HPdfConst.HPDF_LIMIT_MAX_NAME_LEN);
        }
        fontDefAttr.setBaseFont(buf.toString());
    }
    
    private HPdfTTFTable findTable(String tag) {

        for (final HPdfTTFTable tbl : fontDefAttr.getOffsetTable().table) {
            if (ByteUtil.indexOf(tbl.tag, tag, 4) == 0) {
                if (logger.isTraceEnabled()) {
                    logger.trace(String.format(
                            " FindTable find table[%c%c%c%c]",
                            tbl.tag[0], tbl.tag[1], tbl.tag[2], tbl.tag[3]));
                }
                return tbl;
            }
        }
        
        return null;
    }
    
    private final class HPdfTTFFontHeader {
        // TODO should make accessor?
        private byte[] versionNumber = new byte[4];
        private long fontRevision;
        private long chekSumAdjustment;
        private long magicNumber;
        private int flags;
        private int unitsPerEm;
        private byte[] created = new byte[8];
        private byte[] modified = new byte[8];
        private short xMin;
        private short yMin;
        private short xMax;
        private short yMax;
        private int macStyle;
        private int lowestRecPpem;
        private short fontDirectionHint;
        private short indexToLocFormat;
        private short glyphDataFormat;
    }
    
    private final class HPdfTTFGlyphOffsets {
        // TODO should make accessor?
        private long baseOffset;
        private long[] offsets;
        private byte[] flgs = new byte[0];   /* 0: unused, 1: used */
    }
    
    private final class HPdfTTFNameRecord {
        // TODO should make accessor?
        private int platformId;
        private int encodingId;
        private int languageId;
        private int nameId;
        private long length;
        private long offset;
    }
    
    private final class HPdfTTFNamingTable {
        // TODO should make accessor?
        private int format;
        private int count;
        private int stringOffset;
        private HPdfTTFNameRecord[] nameRecords;
        
        private void createNameRecords(int size) {
            nameRecords = new HPdfTTFNameRecord[size];
            for (int i = 0; i < nameRecords.length; ++i) {
                nameRecords[i] = new HPdfTTFNameRecord();
            }
        }
        private void clearNameRecords() {
            nameRecords = null;
        }
    }
    
    private final class HPdfTTFLongHorMetric {
        // TODO should make accessor?
        private int advanceWidth;
        private int lsb;
    }
    
    private final class HPdfTTFTable {
        // TODO should make accessor?
        private byte[] tag = new byte[4];
        private long checkSum;
        private long offset;
        private long length;
        
        // MEMO: 4-bytes alignment C struct size.
        private static final int STRUCT_SIZE = 16;
    }
    
    private final class HPdfTTFOffsetTbl {
        // TODO should make accessor?
        private int sfntVersion;
        private int numTables;
        private int searchRange;
        private int entrySelector;
        private int rangeShift;
        private HPdfTTFTable[] table;
        
        private void createTTFTables(int size) {
            table = new HPdfTTFTable[size];
            for (int i = 0; i < table.length; ++i) {
                table[i] = new HPdfTTFTable();
            }
        }
        
        private void clearTables() {
            table = null;
        }
    }
    
    private final class HPdfTTFCMapRange {
        // TODO should make accessor?
        private int format;
        private int length;
        private int language;
        private int segCountX2;
        private int searchRange;
        private int entrySelector;
        private int rangeShift;
        private int[] endCount;
        private int reservedPad;
        private int[] startCount;
        private int[] idDelta;
        private int[] idRangeOffset;
        private int[] glyphIdArray;
        private int getGlyphIdArrayCount() {
            if (glyphIdArray == null) {
                return 0;
            } else {
                return glyphIdArray.length;
            }
        }
    }
    
    String getCharset() {
        return fontDefAttr.getCharSet();
    }
    
    long getLength1() {
        return fontDefAttr.getLength1();
    }
    
    boolean isEmbedding() {
        return fontDefAttr.isEmbedding();
    }
    
    private HPdfTrueTypeFontDefAttr fontDefAttr = new HPdfTrueTypeFontDefAttr();
    
    private final class HPdfTrueTypeFontDefAttr {
    
        private String baseFont;
        
        private String getBaseFont() {
            return this.baseFont;
        }
        
        private void setBaseFont(String baseFont) {
            this.baseFont = baseFont;
        }
        
        private char firstChar;
        
        private char getFirstChar() {
            return this.firstChar;
        }
        
        private void setFirstChar(char firstChar) {
            this.firstChar = firstChar;
        }
        
        private char lastChar;
        
        private char getLastChar() {
            return this.lastChar;
        }
        
        private void setLastChar(char lastChar) {
            this.lastChar = lastChar;
        }
        
        private String charSet;
        
        private String getCharSet() {
            return this.charSet;
        }
        
        private void setCharSet(String charSet) {
            this.charSet = charSet;
        }
        
        private String tagName;
        
        private String getTagName() {
            return this.tagName;
        }
        
        private byte[] getTagNameAsBytes() {
            return this.tagName.getBytes(StandardCharsets.ISO_8859_1);
        }
        
        private void setTagName(String tagName) {
            this.tagName = tagName;
        }
        
        private String tagName2;
        
        private String getTagName2() {
            return this.tagName2;
        }
        
        private byte[] getTagName2AsBytes() {
            // MEMO: tagName2 is treated as UTF-16BE on byte-level.
            return this.tagName2.getBytes(StandardCharsets.UTF_16BE);
        }
        
        private void setTagName2(String tagName) {
            this.tagName2 = tagName;
        }
        
        private HPdfTTFFontHeader header = new HPdfTTFFontHeader();
        
        private HPdfTTFFontHeader getHeader() {
            return this.header;
        }
        
        private HPdfTTFGlyphOffsets glyphTbl = new HPdfTTFGlyphOffsets();
        
        private HPdfTTFGlyphOffsets getGlyphTable() {
            return this.glyphTbl;
        }
        
        private int numGlyphs;
        
        private int getNumGlyphs() {
            return this.numGlyphs;
        }
        
        private void setNumGlyphs(int numGlyphs) {
            this.numGlyphs = numGlyphs;
        }
        
        private HPdfTTFNamingTable nameTbl = new HPdfTTFNamingTable();
        
        private HPdfTTFNamingTable getNameTable() {
            return this.nameTbl;
        }
        
        private HPdfTTFLongHorMetric[] hMetric;
        
        private HPdfTTFLongHorMetric[] getHMetric() {
            return this.hMetric;
        }
        
        private void createHMetrics(int size) {
            this.hMetric = new HPdfTTFLongHorMetric[size];
            for (int i = 0; i < this.hMetric.length; ++i) {
                this.hMetric[i] = new HPdfTTFLongHorMetric();
            }
        }
        
        private void clearHMetrics() {
            this.hMetric = null;
        }
        
        int numHMetric;
        
        private int getNumHMetric() {
            return this.numHMetric;
        }
        
        private void setNumHMetric(int num) {
            this.numHMetric = num;
        }
        
        private HPdfTTFOffsetTbl offsetTbl = new HPdfTTFOffsetTbl();
        
        private HPdfTTFOffsetTbl getOffsetTable() {
            return this.offsetTbl;
        }
        
        private HPdfTTFCMapRange cmap = new HPdfTTFCMapRange();
        
        private HPdfTTFCMapRange getCmap() {
            return this.cmap;
        }
        
        private int fsType;
        
        private int getFsType() {
            return this.fsType;
        }
        
        private void setFsType(int fsType) {
            this.fsType = fsType;
        }
        
        private byte[] sFamilyClass = new byte[2];
        
        private byte[] getSFamilyClass() {
            return this.sFamilyClass;
        }
        
        private byte[] panose = new byte[10];
        
        private byte[] getPanose() {
            return this.panose;
        }
        
        private long codePageRange1;
        
        private long getCodePageRange1() {
            return this.codePageRange1;
        }
        
        private void setCodePageRange1(long codePageRange) {
            this.codePageRange1 = codePageRange;
        }
        
        private long codePageRange2;
        
        private long getCodePageRange2() {
            return this.codePageRange2;
        }
        
        private void setCodePageRange2(long codePageRange) {
            this.codePageRange2 = codePageRange;
        }
        
        private long length1;
        
        private long getLength1() {
            return this.length1;
        }
        
        private void setLength1(long length1) {
            this.length1 = length1;
        }
        
        private boolean flgEmbedding;
        
        private boolean isEmbedding() {
            return this.flgEmbedding;
        }
        
        private void setEmbedding(boolean flg) {
            this.flgEmbedding = flg;
        }
        
        private boolean flgCIDFont;
        
        private boolean isCIDFont() {
            return this.flgCIDFont;
        }
        
        private void setIsCIDFont(boolean flg) {
            this.flgCIDFont = flg;
        }
        
        private HPdfReadStream stream;
        
        private HPdfReadStream getStream() {
            return this.stream;
        }
        
        private void setStream(HPdfReadStream stream) {
            this.stream = stream;
        }
        
        private void disposeStream() {
            if (this.stream != null) {
                this.stream.close();
                this.stream = null;
            }
        }
        
        private void dispose() {
            // memo: corresponds original code InitAttr.
            this.setCharSet(null);
            this.clearHMetrics();
            this.getNameTable().clearNameRecords();
            HPdfTTFCMapRange cmap = this.getCmap();
            cmap.endCount = null;
            cmap.startCount = null;
            cmap.idDelta = null;
            cmap.idRangeOffset = null;
            cmap.glyphIdArray = null;
            this.getOffsetTable().clearTables();
            HPdfTTFGlyphOffsets glyphTable = this.getGlyphTable();
            glyphTable.flgs = null;
            glyphTable.offsets = null;
            this.disposeStream();
        }
    }
}
